"""In-process Python function tools for InnerLoop.

This module provides a decorator-based system for exposing Python functions
as tools that Big Pickle can call directly.

Usage:
    from innerloop import Loop, tool

    @tool
    def add(a: int, b: int) -> int:
        '''Add two numbers.'''
        return a + b

    loop = Loop(model="opencode/big-pickle")
    response = loop.run("What's 15 + 27?")
    # Big Pickle will call the 'add' tool and get 42

Design:
- @tool decorator registers functions in a registry
- Loop discovers tools and injects schemas into prompt
- When Big Pickle calls a tool, we parse the call and execute the function
- Results are injected back into the conversation

This uses "prompt injection" rather than MCP to avoid IPC complexity while
still providing a clean decorator-based API.
"""

from __future__ import annotations

import inspect
from collections.abc import Callable
from typing import Any, TypeVar, get_type_hints

# ============================================================================
# Global tool registry
# ============================================================================

_TOOL_REGISTRY: dict[str, ToolDefinition] = {}


class ToolDefinition:
    """Metadata about a registered tool."""

    def __init__(
        self,
        func: Callable[..., Any],
        name: str | None = None,
        description: str | None = None,
    ):
        self.func = func
        self.name = name or func.__name__
        self.description = description or (func.__doc__ or "").strip()

        # Extract type hints for schema generation
        self.signature = inspect.signature(func)
        self.type_hints = get_type_hints(func)

    def to_prompt_schema(self) -> str:
        """Generate human-readable schema for prompt injection."""
        params = []
        for param_name, param in self.signature.parameters.items():
            param_type = self.type_hints.get(param_name, Any)
            type_name = getattr(param_type, "__name__", str(param_type))

            if param.default == inspect.Parameter.empty:
                params.append(f"{param_name}: {type_name}")
            else:
                params.append(f"{param_name}: {type_name} = {param.default}")

        return_type = self.type_hints.get("return", Any)
        return_type_name = getattr(return_type, "__name__", str(return_type))

        schema = f"**{self.name}**({', '.join(params)}) -> {return_type_name}"
        if self.description:
            schema += f"\n  {self.description}"

        return schema

    def call(self, **kwargs: Any) -> Any:
        """Call the underlying function with kwargs."""
        try:
            return self.func(**kwargs)
        except Exception as e:
            return f"Error calling {self.name}: {e}"


# ============================================================================
# @tool decorator
# ============================================================================

F = TypeVar("F", bound=Callable[..., Any])


def tool(
    func: F | None = None,
    *,
    name: str | None = None,
    description: str | None = None,
) -> F:
    """Decorator to register a Python function as an InnerLoop tool.

    The decorated function becomes available to Big Pickle as a callable tool.
    Tools are automatically discovered when creating a Loop instance.

    Args:
        func: The function to decorate (when used without arguments)
        name: Optional custom name for the tool (defaults to function name)
        description: Optional description (defaults to docstring)

    Returns:
        The original function, unmodified

    Example:
        @tool
        def add(a: int, b: int) -> int:
            '''Add two numbers together.'''
            return a + b

        @tool(name="multiply", description="Multiply numbers")
        def mult(a: int, b: int) -> int:
            return a * b

        loop = Loop(model="opencode/big-pickle")
        response = loop.run("What's 5 + 3?")
        # Big Pickle will call add(5, 3) and get 8
    """

    def decorator(f: F) -> F:
        tool_def = ToolDefinition(f, name=name, description=description)
        _TOOL_REGISTRY[tool_def.name] = tool_def
        return f

    if func is None:
        # Called with arguments: @tool(name="...", description="...")
        return decorator  # type: ignore
    else:
        # Called without arguments: @tool
        return decorator(func)


# ============================================================================
# Tool registry access
# ============================================================================


def list_tools() -> list[str]:
    """List all registered tool names."""
    return list(_TOOL_REGISTRY.keys())


def get_tool(name: str) -> ToolDefinition | None:
    """Get a tool definition by name."""
    return _TOOL_REGISTRY.get(name)


def get_all_tools() -> dict[str, ToolDefinition]:
    """Get all registered tools."""
    return dict(_TOOL_REGISTRY)


def clear_tools() -> None:
    """Clear all registered tools (useful for testing)."""
    _TOOL_REGISTRY.clear()


def generate_tool_prompt(
    tools: dict[str, ToolDefinition] | None = None,
) -> str:
    """Generate prompt text describing all registered tools.

    This is injected into the system prompt so Big Pickle knows about
    the available Python tools.

    Args:
        tools: Optional dict of tools to use. If None, uses global registry.
    """
    tool_dict = tools if tools is not None else _TOOL_REGISTRY

    if not tool_dict:
        return ""

    lines = [
        "## Available Python Tools",
        "",
        "You have access to the following Python functions via the innerloop-tool CLI:",
        "",
    ]

    for tool_def in tool_dict.values():
        lines.append(f"- {tool_def.to_prompt_schema()}")

    lines.extend(
        [
            "",
            "To call a tool, use bash with innerloop-tool:",
            "```bash",
            "innerloop-tool <tool_name> <arguments>",
            "```",
            "",
            "Examples:",
            "```bash",
            "# Simple arguments (key=value)",
            "innerloop-tool add a=5 b=3",
            "",
            "# String arguments",
            "innerloop-tool greet name=Alice greeting=Hello",
            "",
            "# JSON for complex objects",
            'innerloop-tool analyze_data \'{"filepath": "data.csv", "options": {}}\'',
            "```",
            "",
            "The tool result will be printed to stdout.",
        ]
    )

    return "\n".join(lines)


__all__ = [
    "tool",
    "list_tools",
    "get_tool",
    "get_all_tools",
    "clear_tools",
    "generate_tool_prompt",
    "ToolDefinition",
]
