"""
Agent Loop

Core tool execution loop: send messages -> process tool calls -> execute -> repeat.
Stateless function design - state passed in/out explicitly.
"""

from __future__ import annotations

import asyncio
import json
import time
from collections.abc import AsyncIterator, Callable
from typing import TYPE_CHECKING, Any

from .types import (
    AssistantMessage,
    Config,
    DoneEvent,
    ErrorEvent,
    Event,
    Message,
    Response,
    TextEvent,
    TextPart,
    ThinkingEvent,
    ThinkingPart,
    Tool,
    ToolCallEvent,
    ToolContext,
    ToolResult,
    ToolResultEvent,
    ToolResultMessage,
    ToolUsePart,
    TurnStartEvent,
    Usage,
    UsageEvent,
    UserMessage,
)

if TYPE_CHECKING:
    from .providers.base import Provider
    from .tooling.todo import TodoState


async def execute(
    provider: Provider,
    messages: list[Message],
    tools: list[Tool] | None = None,
    config: Config | None = None,
    tool_choice: dict[str, str] | None = None,
    on_event: Callable[[Event], None] | None = None,
    context: ToolContext | None = None,
    todo_state: TodoState | None = None,
) -> tuple[list[Message], Response]:
    """
    Execute the agent loop.

    Streams from provider, executes tools, repeats until done.

    Args:
        provider: LLM provider to use
        messages: Conversation history (modified in place)
        tools: Available tools (optional)
        config: Execution configuration (optional)
        on_event: Callback for streaming events (optional)
        context: Tool execution context (optional)
        todo_state: Optional TodoState for exit-with-pending-todos prompting

    Returns:
        Tuple of (updated messages, Response object)

    Note: This is a pure function - messages list is copied internally.
    """
    config = config or Config()
    tool_map = {t.name: t for t in (tools or [])}
    all_messages = list(messages)  # Copy to avoid mutating input
    all_tool_results: list[ToolResult] = []
    total_usage = Usage()
    turn = 0
    final_text_parts: list[str] = []
    final_thinking_parts: list[str] = []
    stop_reason = "end_turn"
    start_time = time.monotonic()

    while turn < config.max_turns:
        # Check timeout
        elapsed = time.monotonic() - start_time
        if elapsed >= config.timeout:
            stop_reason = "timeout"
            break

        turn += 1

        # Emit turn start event
        if on_event:
            on_event(TurnStartEvent(turn=turn))

        # Stream one turn from provider
        round_result = await _stream_round(
            provider=provider,
            messages=all_messages,
            tools=tools,
            config=config,
            tool_choice=tool_choice,
            on_event=on_event,
        )

        # Clear tool_choice after first turn
        # This allows the model to decide whether to call tools or return text
        # on subsequent turns, preventing infinite loops when tool_choice is forced
        if tool_choice is not None:
            tool_choice = None

        # Unpack round result
        text_parts = round_result["text_parts"]
        thinking_parts = round_result["thinking_parts"]
        tool_calls = round_result["tool_calls"]
        usage = round_result["usage"]
        stop_reason = round_result["stop_reason"]
        error = round_result["error"]

        # Aggregate
        final_text_parts.extend(text_parts)
        final_thinking_parts.extend(thinking_parts)
        total_usage = total_usage.add(usage)

        # Build assistant message content
        content: list[Any] = []
        if thinking_parts:
            content.append(ThinkingPart(text="".join(thinking_parts)))
        if text_parts:
            content.append(TextPart(text="".join(text_parts)))
        for tc in tool_calls:
            try:
                tool_input = json.loads(tc["input"])
            except json.JSONDecodeError:
                tool_input = {}
            content.append(ToolUsePart(id=tc["id"], name=tc["name"], input=tool_input))

        # Add assistant message
        assistant_msg = AssistantMessage(
            content=content,
            model=f"{provider.name}/{provider.model_id}",
        )
        all_messages.append(assistant_msg)

        # Handle error
        if error:
            return all_messages, Response(
                text="".join(final_text_parts),
                thinking=(
                    "".join(final_thinking_parts) if final_thinking_parts else None
                ),
                model=f"{provider.name}/{provider.model_id}",
                session_id="",  # Caller sets this
                usage=total_usage,
                tool_results=all_tool_results,
                stop_reason="error",
            )

        # Check timeout after turn completes (catches slow single-turn responses)
        elapsed = time.monotonic() - start_time
        if elapsed >= config.timeout:
            stop_reason = "timeout"
            break

        # No tool calls = done (unless pending todos)
        if not tool_calls:
            # Check for pending todos before allowing exit
            if todo_state:
                pending = todo_state.active()
                if pending:
                    prompt = (
                        f"You have {len(pending)} pending todo(s):\n"
                        f"{todo_state.summary()}\n\n"
                        "Please complete them, mark them done, or skip with a reason."
                    )
                    # Inject as a user message to prompt continuation
                    all_messages.append(UserMessage(content=prompt))
                    # Continue the loop (don't break)
                    continue

            # No pending todos - actually done
            break

        # Execute tools in parallel
        tool_results = await _execute_tools(
            tool_calls=tool_calls,
            tool_map=tool_map,
            on_event=on_event,
            context=context,
        )

        # Add tool result messages and track results
        for tc, (result_content, is_error) in zip(
            tool_calls, tool_results, strict=True
        ):
            tool_msg = ToolResultMessage(
                tool_use_id=tc["id"],
                tool_name=tc["name"],
                content=result_content,
                is_error=is_error,
            )
            all_messages.append(tool_msg)

            # Track for response
            try:
                tool_input = json.loads(tc["input"])
            except json.JSONDecodeError:
                tool_input = {}

            all_tool_results.append(
                ToolResult(
                    tool_use_id=tc["id"],
                    tool_name=tc["name"],
                    input=tool_input,
                    output=result_content,
                    is_error=is_error,
                )
            )

    else:
        # Max turns exceeded
        stop_reason = "max_turns"

    return all_messages, Response(
        text="".join(final_text_parts),
        thinking=("".join(final_thinking_parts) if final_thinking_parts else None),
        model=f"{provider.name}/{provider.model_id}",
        session_id="",  # Caller sets this
        usage=total_usage,
        tool_results=all_tool_results,
        stop_reason=stop_reason,
    )


async def stream(
    provider: Provider,
    messages: list[Message],
    tools: list[Tool] | None = None,
    config: Config | None = None,
    tool_choice: dict[str, str] | None = None,
    context: ToolContext | None = None,
    todo_state: TodoState | None = None,
) -> AsyncIterator[Event]:
    """
    Stream events from the agent loop.

    Yields events as they arrive, including tool results.

    Args:
        provider: LLM provider to use
        messages: Conversation history
        tools: Available tools (optional)
        config: Execution configuration (optional)
        context: Tool execution context (optional)
        todo_state: Optional TodoState for exit-with-pending-todos prompting

    Yields:
        Event objects
    """
    config = config or Config()
    tool_map = {t.name: t for t in (tools or [])}
    all_messages = list(messages)
    turn = 0
    start_time = time.monotonic()

    while turn < config.max_turns:
        # Check timeout
        elapsed = time.monotonic() - start_time
        if elapsed >= config.timeout:
            yield ErrorEvent(error=f"Timeout ({config.timeout}s) exceeded")
            yield DoneEvent(stop_reason="timeout")
            return

        turn += 1

        # Emit turn start event
        yield TurnStartEvent(turn=turn)

        tool_calls: list[dict[str, str]] = []
        text_parts: list[str] = []
        thinking_parts: list[str] = []

        # Stream from provider
        async for event in provider.stream(all_messages, tools, config, tool_choice):
            yield event

            if isinstance(event, TextEvent):
                text_parts.append(event.text)
            elif isinstance(event, ThinkingEvent):
                thinking_parts.append(event.text)
            elif isinstance(event, ToolCallEvent):
                # Clear tool_choice after first turn to avoid forcing same tool
                if tool_choice is not None:
                    tool_choice = None
                tool_calls.append(
                    {
                        "id": event.id,
                        "name": event.name,
                        "input": event.input,
                    }
                )
            elif isinstance(event, DoneEvent):
                # Check stop reason
                if event.stop_reason != "tool_use":
                    # Check timeout after turn completes (catches slow single-turn)
                    elapsed = time.monotonic() - start_time
                    if elapsed >= config.timeout:
                        yield ErrorEvent(error=f"Timeout ({config.timeout}s) exceeded")
                        yield DoneEvent(stop_reason="timeout")
                    return  # Done

        # Check timeout after turn completes
        elapsed = time.monotonic() - start_time
        if elapsed >= config.timeout:
            yield ErrorEvent(error=f"Timeout ({config.timeout}s) exceeded")
            yield DoneEvent(stop_reason="timeout")
            return

        # Build assistant message content (needed for both tool and text responses)
        content: list[Any] = []
        if thinking_parts:
            content.append(ThinkingPart(text="".join(thinking_parts)))
        if text_parts:
            content.append(TextPart(text="".join(text_parts)))
        for tc in tool_calls:
            try:
                tool_input = json.loads(tc["input"])
            except json.JSONDecodeError:
                tool_input = {}
            content.append(ToolUsePart(id=tc["id"], name=tc["name"], input=tool_input))

        # Add assistant message to history (before todo check so context is preserved)
        if content:
            assistant_msg = AssistantMessage(
                content=content,
                model=f"{provider.name}/{provider.model_id}",
            )
            all_messages.append(assistant_msg)

        # No tool calls = done (unless pending todos)
        if not tool_calls:
            # Check for pending todos before allowing exit
            if todo_state:
                pending = todo_state.active()
                if pending:
                    prompt = (
                        f"You have {len(pending)} pending todo(s):\n"
                        f"{todo_state.summary()}\n\n"
                        "Please complete them, mark them done, or skip with a reason."
                    )
                    # Inject as a user message to prompt continuation
                    all_messages.append(UserMessage(content=prompt))
                    # Yield a text event to notify the user about the prompt
                    yield TextEvent(
                        text=f"\n[System: Prompting for {len(pending)} pending todo(s)]\n"
                    )
                    # Continue the loop (don't return)
                    continue

            # No pending todos - actually done
            return

        # Execute tools in parallel
        results = await _execute_tools(tool_calls, tool_map, context=context)

        # Add results and yield events
        for tc, (result_content, is_error) in zip(tool_calls, results, strict=True):
            # Yield result event
            yield ToolResultEvent(
                tool_use_id=tc["id"],
                tool_name=tc["name"],
                content=result_content,
                is_error=is_error,
            )

            # Add to messages
            tool_msg = ToolResultMessage(
                tool_use_id=tc["id"],
                tool_name=tc["name"],
                content=result_content,
                is_error=is_error,
            )
            all_messages.append(tool_msg)

    # Max turns exceeded
    yield ErrorEvent(error=f"Max turns ({config.max_turns}) exceeded")
    yield DoneEvent(stop_reason="max_turns")


async def _stream_round(
    provider: Provider,
    messages: list[Message],
    tools: list[Tool] | None,
    config: Config,
    tool_choice: dict[str, str] | None,
    on_event: Callable[[Event], None] | None,
) -> dict[str, Any]:
    """Stream one round from the provider and collect results."""
    text_parts: list[str] = []
    thinking_parts: list[str] = []
    tool_calls: list[dict[str, str]] = []
    usage = Usage()
    stop_reason = "end_turn"
    error: str | None = None

    async for event in provider.stream(
        messages, tools, config, tool_choice=tool_choice
    ):
        if on_event:
            on_event(event)

        if isinstance(event, TextEvent):
            text_parts.append(event.text)
        elif isinstance(event, ThinkingEvent):
            thinking_parts.append(event.text)
        elif isinstance(event, ToolCallEvent):
            tool_calls.append(
                {
                    "id": event.id,
                    "name": event.name,
                    "input": event.input,
                }
            )
        elif isinstance(event, UsageEvent):
            usage = usage.add(event)
        elif isinstance(event, ErrorEvent):
            error = event.error
        elif isinstance(event, DoneEvent):
            stop_reason = event.stop_reason

    return {
        "text_parts": text_parts,
        "thinking_parts": thinking_parts,
        "tool_calls": tool_calls,
        "usage": usage,
        "stop_reason": stop_reason,
        "error": error,
    }


async def _execute_tools(
    tool_calls: list[dict[str, str]],
    tool_map: dict[str, Tool],
    on_event: Callable[[Event], None] | None = None,
    context: ToolContext | None = None,
) -> list[tuple[str, bool]]:
    """Execute tools in parallel.

    Args:
        tool_calls: List of tool calls from LLM
        tool_map: Map of tool name to Tool instance
        on_event: Optional callback for events
        context: Optional tool execution context

    Returns:
        List of (result_content, is_error) tuples
    """

    async def run_one(tc: dict[str, str]) -> tuple[str, bool]:
        tool = tool_map.get(tc["name"])
        if tool is None:
            return f"Unknown tool: {tc['name']}", True

        try:
            tool_input = json.loads(tc["input"])
        except json.JSONDecodeError as e:
            return f"Invalid JSON input: {e}", True

        result, is_error = await tool.execute(tool_input, context)

        # Emit result event if callback provided
        if on_event:
            on_event(
                ToolResultEvent(
                    tool_use_id=tc["id"],
                    tool_name=tc["name"],
                    content=result,
                    is_error=is_error,
                )
            )

        return result, is_error

    # Execute all tools in parallel
    results = await asyncio.gather(*[run_one(tc) for tc in tool_calls])
    return list(results)


__all__ = ["execute", "stream"]
