# iotery.io Python SDK

The python iotery.io SDK is intended to be used on your server or in your data processing apps to interact with the itoery.io IoT Platform. The SDK is a fully featured wrapper for the [REST API](https://somelink_to_swagger_docs).

## Getting Started

Setup your free account on [iotery.io]() and go to your [settings dashboard]() to get your server API Key.

After you get your key, install the SDK:

```bash
pip install iotery-python-server-sdk
```

> Note: Make sure you are using Python 3.5+!

And finally, some simple example usage:

```python
from Iotery import Iotery


iotery = Iotery("my-key")

# find the exact `data` specification at https://iotery.io/v1/docs#createDeviceType
device_type = iotery.createDeviceType(
    data={"name": "My Device Type", "enum": "MY_DEVICE_TYPE", ...})

device_type_by_uuid = iotery.getDeviceTypeByUuid(
    deviceTypeUuid=device_type["uuid"], opts={"limit": 1})
```

> The above code connects you to the iotary.io platform, creates a device type and then gets that device type.

Next, you might want to create a data type for the the device type you created...here's an example snippet:

```python
temperature_data_type = iotery.createDataType(
  deviceTypeUuid=device_type_by_uuid["uuid"],
  data = {
    name: "Temperature",
    enum: "TEMPERATURE",
    units: "C",
    isNumber: true
  }
)
```

> To have a "thing" (like a Raspberry Pi) create data, you will want to check out the [iotery.io thing client](link).

For a tutorial on setting up a full stack system in 15 minutes using iotery.io, check out [this link](medium_article).

## API

This SDK simply wraps the [REST API](https://somelink_to_swagger_docs), so more information and specifics can be found there. Since the API is a wrapper around the REST API, the syntax is standard for each of the Create, Read, Update, and Delete operations on iotery.io resources. All methods return a dictonary containing the API response. If there is an error, the method will `raise` an expection.

### Creating Resources

The generalized syntax for creating resources in iotery.io python sdk looks like:

```python
iotery.methodName(inputParameter="parameter", data={ "data": "variables" })
```

For example, to create a device, the javascript would look like

```python
createDevice(
  deviceTypeUuid="a-valid-device-type-uuid",
  data={ "name": "My Device", "other": "parameter" }
)
```

where `createDevice` maps to `methodName`, `deviceTypeUuid` maps to `inputParameter`, and `name` and `other` map to the dictonary `{data : "variables"}` in the generalized form given above.

The available resource creation methods are

|            `methodName`             |                                                                                  `input`                                                                                  |                                                                link                                                                |              `description`              |
| :---------------------------------: | :-----------------------------------------------------------------------------------------------------------------------------------------------------------------------: | :--------------------------------------------------------------------------------------------------------------------------------: | :-------------------------------------: |
|         getDeviceTokenBasic         |          `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1basic~1get-token/POST) | Get token for device via key, serial, and secret.           |
|    reportAlreadyExecutedCommands    | `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1command-instances~1executed/POST) | Report an already executed set of commands to the server. |
| createDeviceCommandInstanceEmbedded |                                                                      `deviceUuid`,`commandTypeUuid`                                                                       | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1command-types~1:commandTypeUuid/POST) | Create a command instance via a device. |
|              postData               |                                                                               `deviceUuid`                                                                                |              [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1data/POST)               |         Post data to the cloud.         |
|        upsertDeviceSettings         |                                                                               `deviceUuid`                                                                                |            [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1settings/POST)             |      Upsert settings for a device.      |
|          uploadDeviceLogs           |                                                                               `deviceUuid`                                                                                |         [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1upload-log-file/POST)         |     Upload zip file of device logs.     |
|      getDeviceTokenAsymmetric       |          `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1token~1asymmetric/POST) | Get device token based on encrypted credentials.           |

### Reading Resources

The generalized syntax for reading (getting) resources in iotery.io python sdk looks like:

```python
iotery.methodName(inputParameter="parameter", opts={"query":"parameter"})
```

For example, to get a device by it's unique identifier `uuid`, the python would look like

```python
getDeviceByUuid(
  deviceUuid="a-valid-device-uuid",
  opts={ "limit": 1 }
)
```

where `getDeviceByUuid` maps to `methodName`, `deviceUuid` maps to `inputParameter`, and `{ "limit": 1 }` maps to the dictonary `{"query" : "parameters"}` in the generalized form given above.

> The `limit` option is for instructive purposes only. By definition, a `uuid` is unique and so there will never be more than one device for a given `uuid`.

The available resource creation methods are

|              `methodName`              |                                                                           `input`                                                                           |                                                         link                                                          |                            `description`                            |
| :------------------------------------: | :---------------------------------------------------------------------------------------------------------------------------------------------------------: | :-------------------------------------------------------------------------------------------------------------------: | :-----------------------------------------------------------------: |
|            getBrokerAddress            |               `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1broker-address/GET) | Get the MQTT broker address.                |
|       getCommandTypeListEmbedded       |           `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1command-types/GET) | Get a list of available command types.           |
|          getCurrentTimestamp           |   `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1current-timestamp/GET) | Get the current server time in epoch (unix) time.    |
|        getDataTypeListEmbedded         |              `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1data-types/GET) | Get a list of available data types.              |
| getDeviceUnexecutedCommandInstanceList |                                                                        `deviceUuid`                                                                         | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1unexecuted-commands/GET) |      Get a list of unexecuted command instances for a device.       |
|      getDeviceTypeFirmwareRecord       |                                                                 `deviceTypeUuid`,`version`                                                                  |    [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1firmware~1:deviceTypeUuid~1:version/GET)    | Get a desired version of a firmware record for a given device type. |
|           getPublicCloudKey            | `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1key/GET) | Get base64 encoded public key from the server for use in RSA scheme. |
|                 getMe                  |                   `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1me/GET) | Get information about the device.                   |
|      getNotificationListEmbedded       |           `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1notifications/GET) | Get a list of available notifications.           |
|       getSettingTypeListEmbedded       |           `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1setting-types/GET) | Get a list of available setting types.           |

### Updating Resources

The generalized syntax for updating resources in iotery.io python sdk looks like:

```python
iotery.methodName(inputParameter="parameter", data={ "data": "variables" })
```

For example, to update a device type, the javascript would look like

```python
updateDevice(
  deviceTypeUuid="a-valid-device-type-uuid",
  data={ "name": "My New Name" }
)
```

where `updateDevice` maps to `methodName`, `deviceTypeUuid` maps to `inputParameter`, and `{ "name": "My New Name" }` maps to the dictonary `{data : "variables"}` in the generalized form given above.

The available resource creation methods are

|             `methodName`             |                                                                                                `input`                                                                                                 | link | `description` |
| :----------------------------------: | :----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------: | :--: | :-----------: |
| setBatchedCommandInstancesAsExecuted | `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1batch-command-instances~1:batchReferenceUuid~1executed/PATCH) | Set a collection of batched command instances as executed. |
|     setCommandInstanceAsExecuted     |     `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1command-instances~1:commandInstanceUuid~1executed/PATCH) | Report that a command has been executed to the server.      |
|         updateDeviceChannel          |                        `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1devices~1:deviceUuid~1channelId~1:channelId/PATCH) | Update device channel.                         |
|   setNotificationInstanceInactive    |          `` | [link](https://iotery.io/docs/embedded/#tag/Embedded/paths/~1embedded~1notification-instances~1:notificationInstanceUuid~1inactive/PATCH) | Set notification instance inactive.          |

### Deleting Resources

The generalized syntax for reading (getting) resources in iotery.io python sdk looks like:

```python
iotery.methodName(inputParameter="parameter", opts={"query":"parameter"})
```

For example, to get a device by it's unique identifier `uuid`, the python would look like

```python
deleteDevice(
  deviceUuid="a-valid-device-uuid",
  opts={ "some": "option" }
)
```

where `deleteDevice` maps to `methodName`, `deviceUuid` maps to `inputParameter`, and `{ "some": "option" }` maps to the dictonary `{"query" : "parameters"}` in the generalized form given above.

The available resource creation methods are

| `methodName` | `input` | link | `description` |
| :----------: | :-----: | :--: | :-----------: |


## Contributing

We welcome contributors and PRs! Let us know if you are interested.
