import inspect
from types import ModuleType
from typing import List, Tuple, Type, TypeVar

T = TypeVar("T")


def get_class_full_path(cls) -> str:
    """
    获取类的全路径表示（模块路径.类名）

    参数:
        cls: 目标类（需传入类本身，而非实例）

    返回:
        str: 类的全路径，格式为 "模块路径.类名"
    """
    # __qualname__ 获取类的限定名（含嵌套关系，如 "OuterClass.InnerClass"）
    return f"{cls.__module__}.{cls.__qualname__}"


def collect_subclass_of(
    base_class: Type[T], _from: ModuleType, include_base_class: bool = False
):
    """
    收集指定模块中所有继承自 base_class 的子类

    参数:
        base_class: 基类类型
        _from: 要搜索的模块对象
        include_base_class: 是否包含基类本身，默认为 False

    返回:
        List[Tuple[str, T]]: 包含子类名和子类对象的元组列表
    """

    def predicate(m):
        if not include_base_class and m is base_class:
            return False
        return inspect.isclass(m) and issubclass(m, base_class)

    name_class_map: List[Tuple[str, T]] = inspect.getmembers(_from, predicate=predicate)
    return name_class_map
