#!/usr/bin/env python3
"""
IPIP 命令行接口 - 通过 pyproject.toml 管理依赖
"""

import argparse
import os
import sys
import subprocess
import toml
from typing import List, Dict, Any
from .dptree import DependencyTree, analyze_dependencies


class Colors:
    """终端颜色常量"""
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'


class Icons:
    """终端图标常量"""
    INFO = "ℹ️ "
    SUCCESS = "✅ "
    WARNING = "⚠️ "
    ERROR = "❌ "
    INSTALL = "📦 "
    UNINSTALL = "🗑️ "
    VENV = "🐍 "
    CONFIG = "⚙️ "
    CLEAN = "🧹 "
    ANALYZE = "🔍 "


class IPIP:
    """IPIP 主类 - 通过 pyproject.toml 管理依赖"""
    
    def __init__(self):
        self.tsinghua_index = "https://pypi.tuna.tsinghua.edu.cn/simple/"
        self.trusted_host = "pypi.tuna.tsinghua.edu.cn"
        self.pyproject_file = "pyproject.toml"
    
    def print_info(self, message: str) -> None:
        """打印信息消息"""
        print(f"{Colors.OKBLUE}{Icons.INFO}{message}{Colors.ENDC}")
    
    def print_success(self, message: str) -> None:
        """打印成功消息"""
        print(f"{Colors.OKGREEN}{Icons.SUCCESS}{message}{Colors.ENDC}")
    
    def print_warning(self, message: str) -> None:
        """打印警告消息"""
        print(f"{Colors.WARNING}{Icons.WARNING}{message}{Colors.ENDC}")
    
    def print_error(self, message: str) -> None:
        """打印错误消息"""
        print(f"{Colors.FAIL}{Icons.ERROR}{message}{Colors.ENDC}", file=sys.stderr)
    
    def print_header(self, message: str) -> None:
        """打印标题消息"""
        print(f"{Colors.HEADER}{Colors.BOLD}{message}{Colors.ENDC}")
    
    def print_step(self, message: str) -> None:
        """打印步骤消息"""
        print(f"{Colors.OKCYAN}→ {message}{Colors.ENDC}")
    
    def is_venv(self) -> bool:
        """检测是否在虚拟环境中"""
        # 检查是否在 .venv 虚拟环境中
        if hasattr(sys, 'real_prefix') or (hasattr(sys, 'base_prefix') and sys.base_prefix != sys.prefix):
            return True
        
        # 检查当前 Python 可执行文件是否在 .venv 目录中
        python_path = sys.executable
        if os.name == 'nt':  # Windows
            venv_python = os.path.join('.venv', 'Scripts', 'python.exe')
        else:  # Unix/Linux/macOS
            venv_python = os.path.join('.venv', 'bin', 'python')
        
        return os.path.abspath(python_path) == os.path.abspath(venv_python)
    
    def create_and_activate_venv(self) -> bool:
        """创建并激活虚拟环境"""
        try:
            # 检查 .venv 目录是否已存在
            if os.path.exists('.venv'):
                self.print_info("虚拟环境 .venv 已存在")
                return True
            
            # 创建虚拟环境
            self.print_step("正在创建虚拟环境 .venv...")
            result = subprocess.run([sys.executable, "-m", "venv", ".venv"], check=True)
            
            # 激活虚拟环境
            if os.name == 'nt':  # Windows
                activate_script = os.path.join('.venv', 'Scripts', 'activate')
                if os.path.exists(activate_script):
                    self.print_success("虚拟环境创建成功")
                    return True
                else:
                    self.print_warning("无法找到虚拟环境激活脚本")
                    return False
            else:  # Unix/Linux/macOS
                activate_script = os.path.join('.venv', 'bin', 'activate')
                if os.path.exists(activate_script):
                    self.print_success("虚拟环境创建成功")
                    return True
                else:
                    self.print_warning("无法找到虚拟环境激活脚本")
                    return False
                    
        except subprocess.CalledProcessError as e:
            self.print_error(f"创建虚拟环境失败: {e}")
            return False
        except Exception as e:
            self.print_error(f"创建虚拟环境时发生错误: {e}")
            return False
    
    def get_python_executable(self) -> str:
        """获取虚拟环境中的 Python 可执行文件路径"""
        if os.name == 'nt':  # Windows
            return os.path.join('.venv', 'Scripts', 'python.exe')
        else:  # Unix/Linux/macOS
            return os.path.join('.venv', 'bin', 'python')
    
    def get_pip_executable(self) -> str:
        """获取虚拟环境中的 pip 可执行文件路径"""
        if os.name == 'nt':  # Windows
            return os.path.join('.venv', 'Scripts', 'pip.exe')
        else:  # Unix/Linux/macOS
            return os.path.join('.venv', 'bin', 'pip')
    
    def clean_egg_info(self) -> None:
        """清除 project.egg-info 目录"""
        import shutil
        import glob
        
        # 查找所有 egg-info 目录
        egg_info_patterns = [
            "*.egg-info",
            "project.egg-info",
            "src/*.egg-info"
        ]
        
        cleaned_count = 0
        for pattern in egg_info_patterns:
            for egg_info_path in glob.glob(pattern):
                if os.path.isdir(egg_info_path):
                    try:
                        shutil.rmtree(egg_info_path)
                        self.print_step(f"已清除 {egg_info_path}")
                        cleaned_count += 1
                    except Exception as e:
                        self.print_warning(f"清除 {egg_info_path} 失败: {e}")
        
        if cleaned_count > 0:
            self.print_success(f"清理完成，共清除 {cleaned_count} 个 egg-info 目录")
    
    def get_installed_versions(self, packages: List[str]) -> Dict[str, str]:
        """获取已安装包的精确版本"""
        python_exe = self.get_python_executable()
        if not os.path.exists(python_exe):
            return {}
        
        try:
            # 使用 pip freeze 获取所有已安装包的版本
            result = subprocess.run(
                [python_exe, "-m", "pip", "freeze"],
                capture_output=True, text=True, check=True
            )
            
            installed_packages = {}
            for line in result.stdout.strip().split('\n'):
                if '==' in line:
                    name, version = line.split('==', 1)
                    installed_packages[name.lower()] = line  # 保存完整的包名和版本
            
            # 只返回请求的包的版本
            requested_versions = {}
            for package in packages:
                package_lower = package.lower()
                if package_lower in installed_packages:
                    requested_versions[package] = installed_packages[package_lower]
            
            return requested_versions
        except subprocess.CalledProcessError:
            return {}
    
    def load_pyproject(self) -> Dict[str, Any]:
        """加载 pyproject.toml 文件"""
        if not os.path.exists(self.pyproject_file):
            # 如果文件不存在，创建默认的 pyproject.toml
            default_config = {
                "build-system": {
                    "requires": ["setuptools>=61.0", "wheel"],
                    "build-backend": "setuptools.build_meta"
                },
                "project": {
                    "name": "project",
                    "version": "0.0.0",
                    "description": "",
                    "dependencies": []
                }
            }
            with open(self.pyproject_file, 'w', encoding='utf-8') as f:
                toml.dump(default_config, f)
            self.print_success(f"已创建默认的 {self.pyproject_file} 文件")
            return default_config
        
        try:
            with open(self.pyproject_file, 'r', encoding='utf-8') as f:
                return toml.load(f)
        except Exception as e:
            self.print_error(f"读取 {self.pyproject_file} 失败: {e}")
            return {}
    
    def save_pyproject(self, config: Dict[str, Any]) -> bool:
        """保存 pyproject.toml 文件"""
        try:
            with open(self.pyproject_file, 'w', encoding='utf-8') as f:
                toml.dump(config, f)
            return True
        except Exception as e:
            self.print_error(f"保存 {self.pyproject_file} 失败: {e}")
            return False
    
    def install_project(self) -> int:
        """安装当前项目（相当于 pip install -e .）"""
        self.print_header(f"{Icons.INSTALL} 安装当前项目")
        
        # 检查是否在虚拟环境中
        if not self.is_venv():
            self.print_info("检测到不在虚拟环境中，正在创建虚拟环境...")
            if not self.create_and_activate_venv():
                self.print_error("无法创建或激活虚拟环境")
                return 1
        
        # 检查是否存在 pyproject.toml 或 setup.py
        if not os.path.exists(self.pyproject_file) and not os.path.exists("setup.py"):
            self.print_error("当前目录下没有找到 pyproject.toml 或 setup.py 文件")
            return 1
        
        # 安装当前项目
        python_exe = self.get_python_executable()
        if not os.path.exists(python_exe):
            self.print_error("虚拟环境中的 Python 可执行文件不存在")
            return 1
        
        pip_args = [
            python_exe, "-m", "pip", "install", "-e", ".", 
            "--index-url", self.tsinghua_index,
            "--trusted-host", self.trusted_host
        ]
        
        try:
            self.print_step("正在安装当前项目...")
            result = subprocess.run(pip_args, check=True, env=dict(os.environ))
            self.print_success("当前项目安装成功")
            
            # 清除 project.egg-info 目录
            self.clean_egg_info()
            
            return result.returncode
        except subprocess.CalledProcessError as e:
            self.print_error(f"安装失败: {e}")
            return e.returncode
        except FileNotFoundError:
            self.print_error("找不到 pip 命令")
            return 1

    def install(self, packages: List[str]) -> int:
        """安装包并更新 pyproject.toml，支持版本锁定"""
        self.print_header(f"{Icons.INSTALL} 安装包")
        
        # 检查是否在虚拟环境中
        if not self.is_venv():
            self.print_info("检测到不在虚拟环境中，正在创建虚拟环境...")
            if not self.create_and_activate_venv():
                self.print_error("无法创建或激活虚拟环境")
                return 1
        
        # 加载 pyproject.toml
        config = self.load_pyproject()
        if not config:
            return 1
        
        # 获取当前依赖列表
        dependencies = config.get("project", {}).get("dependencies", [])
        
        # 使用 pip 安装包
        python_exe = self.get_python_executable()
        if not os.path.exists(python_exe):
            self.print_error("虚拟环境中的 Python 可执行文件不存在")
            return 1
        
        pip_args = [
            python_exe, "-m", "pip", "install",
            "--index-url", self.tsinghua_index,
            "--trusted-host", self.trusted_host
        ] + packages
        
        try:
            self.print_step(f"正在安装包: {', '.join(packages)}")
            result = subprocess.run(pip_args, check=True, env=dict(os.environ))
            self.print_success("包安装成功")
            
            # 安装成功后，获取已安装包的精确版本
            installed_versions = self.get_installed_versions(packages)
            
            # 更新 pyproject.toml 中的依赖，使用版本锁定
            updated = False
            for package in packages:
                if package in installed_versions:
                    versioned_package = installed_versions[package]
                    # 检查是否已经存在（可能没有版本信息）
                    existing_package = None
                    for i, dep in enumerate(dependencies):
                        if dep.split('==')[0].lower() == package.lower():
                            existing_package = i
                            break
                    
                    if existing_package is not None:
                        if dependencies[existing_package] != versioned_package:
                            dependencies[existing_package] = versioned_package
                            updated = True
                            self.print_info(f"已更新 {package} 版本锁定: {versioned_package}")
                    else:
                        dependencies.append(versioned_package)
                        updated = True
                        self.print_info(f"已添加 {package} 版本锁定: {versioned_package}")
                else:
                    # 如果没有找到版本信息，添加不带版本的包名
                    if package not in dependencies:
                        dependencies.append(package)
                        updated = True
                        self.print_info(f"已添加 {package} 到 {self.pyproject_file}")
            
            if updated:
                # 更新 pyproject.toml
                config["project"]["dependencies"] = dependencies
                if not self.save_pyproject(config):
                    self.print_error("无法保存 pyproject.toml 文件")
                    return 1
                self.print_success(f"已更新 {self.pyproject_file} 文件，包含版本锁定")
            else:
                self.print_info("所有指定的包都已存在于 pyproject.toml 中")
            
            # 清除 project.egg-info 目录
            self.clean_egg_info()
            
            return result.returncode
        except subprocess.CalledProcessError as e:
            self.print_error(f"安装失败: {e}")
            return e.returncode
        except FileNotFoundError:
            self.print_error("找不到 pip 命令")
            return 1

    def uninstall(self, packages: List[str]) -> int:
        """卸载包并更新 pyproject.toml，然后重新安装项目"""
        self.print_header(f"{Icons.UNINSTALL} 卸载包")
        
        # 检查是否在虚拟环境中
        if not self.is_venv():
            if not self.create_and_activate_venv():
                self.print_error("无法创建或激活虚拟环境")
                return 1
        
        # 加载 pyproject.toml
        config = self.load_pyproject()
        if not config:
            return 1
        
        # 获取当前依赖列表
        dependencies = config.get("project", {}).get("dependencies", [])
        
        # 从依赖列表中移除要卸载的包
        updated = False
        for package in packages:
            # 查找并移除包（可能带版本信息）
            for i, dep in enumerate(dependencies):
                if dep.split('==')[0].lower() == package.lower():
                    dependencies.pop(i)
                    updated = True
                    self.print_info(f"已从 {self.pyproject_file} 中移除 {package}")
                    break
        
        if not updated:
            self.print_warning("指定的包在 pyproject.toml 中不存在")
            return 0
        
        # 更新 pyproject.toml
        config["project"]["dependencies"] = dependencies
        if not self.save_pyproject(config):
            self.print_error("无法保存 pyproject.toml 文件")
            return 1
        
        self.print_success(f"已更新 {self.pyproject_file} 文件")
        
        # 使用依赖树分析要卸载的包及其依赖
        python_exe = self.get_python_executable()
        if not os.path.exists(python_exe):
            self.print_error("虚拟环境中的 Python 可执行文件不存在")
            return 1
        
        try:
            # 先分析依赖关系，确定要卸载的包
            self.print_step("正在分析依赖关系...")
            tree = analyze_dependencies(python_exe=python_exe)
            
            # 获取要卸载的包及其所有依赖
            packages_to_remove = []
            for package in packages:
                if package in tree.nodes:
                    # 获取包的依赖链
                    deps_chain = tree.get_dependency_chain(package)
                    for dep_node, _ in deps_chain:
                        if dep_node.name not in packages_to_remove:
                            packages_to_remove.append(dep_node.name)
            
            # 基于 pyproject.toml 中的依赖关系过滤包
            final_packages_to_remove = []
            remaining_dependencies = config.get("project", {}).get("dependencies", [])
            
            for pkg in packages_to_remove:
                if pkg in tree.nodes:
                    node = tree.nodes[pkg]
                    # 检查是否还有其他 pyproject.toml 中的包依赖这个包
                    remaining_dependents = [dep for dep in node.dependents if dep.name in remaining_dependencies]
                    if not remaining_dependents:
                        final_packages_to_remove.append(pkg)
            
            # 一次性卸载所有相关包
            if final_packages_to_remove:
                pip_uninstall_args = [
                    python_exe, "-m", "pip", "uninstall", "-y"
                ] + final_packages_to_remove
                
                try:
                    self.print_step(f"正在卸载包: {', '.join(final_packages_to_remove)}")
                    result = subprocess.run(pip_uninstall_args, check=True, env=dict(os.environ))
                    self.print_success("包卸载成功")
                except subprocess.CalledProcessError as e:
                    self.print_error(f"卸载失败: {e}")
                    return e.returncode
            else:
                self.print_info("没有发现需要卸载的包")
            
        except Exception as e:
            self.print_error(f"依赖分析失败: {e}")
            # 如果依赖分析失败，回退到简单的卸载方式
            if packages:
                pip_uninstall_args = [
                    python_exe, "-m", "pip", "uninstall", "-y", "--autoremove"
                ] + packages
                
                try:
                    self.print_step(f"正在卸载包: {', '.join(packages)}")
                    result = subprocess.run(pip_uninstall_args, check=True, env=dict(os.environ))
                    self.print_success("包卸载成功")
                except subprocess.CalledProcessError as e:
                    self.print_error(f"卸载失败: {e}")
                    return e.returncode
        
        return 0


def create_parser() -> argparse.ArgumentParser:
    """创建命令行参数解析器"""
    parser = argparse.ArgumentParser(
        prog="ipip",
        description="轻量化的 pip 包装器，通过 pyproject.toml 管理依赖，自动使用清华镜像源",
        epilog="示例: ipip install requests 或 ipip install"
    )
    
    parser.add_argument(
        "--version",
        action="version",
        version="ipip 0.0.0"
    )
    
    # 添加子命令
    subparsers = parser.add_subparsers(dest="command", help="可用命令")
    
    # install 命令
    install_parser = subparsers.add_parser("install", help="安装包或当前项目")
    install_parser.add_argument("packages", nargs="*", help="要安装的包名（不指定则安装当前项目）")
    
    # i 命令（install 的简写）
    i_parser = subparsers.add_parser("i", help="安装包或当前项目（install 的简写）")
    i_parser.add_argument("packages", nargs="*", help="要安装的包名（不指定则安装当前项目）")
    
    # uninstall 命令
    uninstall_parser = subparsers.add_parser("uninstall", help="卸载包并更新 pyproject.toml")
    uninstall_parser.add_argument("packages", nargs="+", help="要卸载的包名")
    
    # uni 命令（uninstall 的简写）
    uni_parser = subparsers.add_parser("uni", help="卸载包并更新 pyproject.toml（uninstall 的简写）")
    uni_parser.add_argument("packages", nargs="+", help="要卸载的包名")
    
    return parser


def main():
    """主函数"""
    parser = create_parser()
    args = parser.parse_args()
    
    # 如果没有指定命令，显示帮助信息
    if not hasattr(args, 'command') or args.command is None:
        parser.print_help()
        return 1
    
    ipip = IPIP()
    
    try:
        if args.command == "install":
            if not args.packages:
                # 没有指定包名，安装当前项目
                return ipip.install_project()
            else:
                # 指定了包名，安装包
                return ipip.install(args.packages)
        elif args.command == "i":
            if not args.packages:
                # 没有指定包名，安装当前项目
                return ipip.install_project()
            else:
                # 指定了包名，安装包
                return ipip.install(args.packages)
        elif args.command == "uninstall":
            return ipip.uninstall(args.packages)
        elif args.command == "uni":
            return ipip.uninstall(args.packages)
        else:
            ipip.print_error(f"未知命令: {args.command}")
            return 1
    except KeyboardInterrupt:
        print(f"\n{Colors.WARNING}{Icons.WARNING}操作被用户中断{Colors.ENDC}", file=sys.stderr)
        return 1
    except Exception as e:
        print(f"{Colors.FAIL}{Icons.ERROR}发生错误: {e}{Colors.ENDC}", file=sys.stderr)
        return 1


if __name__ == "__main__":
    sys.exit(main()) 