# Publishing janus-python-sdk to PyPI

## Prerequisites

1. **PyPI Account**: You need a PyPI account at https://pypi.org
2. **API Token**: Create an API token at https://pypi.org/manage/account/token/
   - For test PyPI: https://test.pypi.org/manage/account/token/
3. **Build Tools**: Install build and twine:
   ```bash
   pip install build twine
   ```

## Current Package Info

- **Package Name**: `janus-python-sdk`
- **Current Version**: `0.3.0` (in `pyproject.toml`)
- **Build Backend**: `hatchling`
- **Package Location**: `janus_sdk/`

## Step-by-Step Publishing Process

### 1. Update Version (if needed)

If you've made changes, bump the version in `pyproject.toml`:

```toml
version = "0.3.1"  # or 0.4.0 for major changes
```

**Version Guidelines:**
- `0.3.0` → `0.3.1`: Bug fixes (patch)
- `0.3.0` → `0.4.0`: New features (minor)
- `0.3.0` → `1.0.0`: Breaking changes (major)

### 2. Clean Previous Builds

```bash
cd janus-python-sdk
rm -rf dist/ build/ *.egg-info
```

### 3. Build the Package

```bash
python -m build
```

This creates:
- `dist/janus_python_sdk-<version>-py3-none-any.whl` (wheel)
- `dist/janus_python_sdk-<version>.tar.gz` (source distribution)

### 4. Verify the Build

Check the built package contents:

```bash
# List contents of wheel
unzip -l dist/janus_python_sdk-*.whl

# Verify package structure
python -m twine check dist/*
```

### 5. Test on Test PyPI (Recommended)

First, test upload to Test PyPI:

```bash
# Upload to Test PyPI
python -m twine upload --repository testpypi dist/*

# You'll be prompted for:
# - Username: __token__
# - Password: <your-test-pypi-api-token>
```

Then test installation:

```bash
pip install --index-url https://test.pypi.org/simple/ janus-python-sdk
```

### 6. Publish to Production PyPI

Once tested, publish to production PyPI:

```bash
python -m twine upload dist/*

# You'll be prompted for:
# - Username: __token__
# - Password: <your-pypi-api-token>
```

### 7. Verify Installation

After publishing, verify it's available:

```bash
pip install janus-python-sdk
```

## Using Environment Variables for Credentials

Instead of entering credentials interactively, you can use environment variables:

```bash
export TWINE_USERNAME=__token__
export TWINE_PASSWORD=<your-api-token>

python -m twine upload dist/*
```

Or create a `.pypirc` file in your home directory:

```ini
[pypi]
username = __token__
password = <your-api-token>

[testpypi]
username = __token__
password = <your-test-pypi-api-token>
```

## Quick Publish Script

Create a script `publish.sh`:

```bash
#!/bin/bash
set -e

echo "🧹 Cleaning previous builds..."
rm -rf dist/ build/ *.egg-info

echo "📦 Building package..."
python -m build

echo "✅ Checking package..."
python -m twine check dist/*

echo "🚀 Uploading to PyPI..."
python -m twine upload dist/*

echo "✅ Published successfully!"
```

Make it executable:
```bash
chmod +x publish.sh
```

## Troubleshooting

### "File already exists" Error

If you try to upload the same version twice, PyPI will reject it. You must:
1. Bump the version in `pyproject.toml`
2. Rebuild the package
3. Upload again

### "Invalid package name"

Ensure `pyproject.toml` has:
- `name = "janus-python-sdk"` (matches PyPI package name)
- Valid version format (e.g., `0.3.1`)

### Missing Files

Check `MANIFEST.in` includes all necessary files:
- `include README.md`
- `include LICENSE`
- `recursive-include janus_sdk *.py`

## Current Status

- ✅ Package structure: Configured
- ✅ Build system: hatchling (configured in pyproject.toml)
- ✅ Manifest: Includes required files
- ✅ Version: 0.3.0 (update if needed before publishing)
- ⚠️  Uncommitted changes: `janus_sdk/__init__.py` has fixes that need to be published

## Next Steps

1. **Commit your changes** (the judge fix in `janus_sdk/__init__.py`)
2. **Bump version** to `0.3.1` in `pyproject.toml` (bug fix release)
3. **Build and test** on Test PyPI first
4. **Publish** to production PyPI

