"""Customer Support Demo: Example usage of the Salesforce Agentforce workflow test."""

import asyncio
import os
from dotenv import load_dotenv
load_dotenv()

from test.customer_support_tracing_test import CustomerSupportAgent

async def demo_customer_support_scenarios():
    """Demonstrate different customer support scenarios."""
    
    print("🚀 Salesforce Agentforce Customer Support Demo")
    print("=" * 50)
    
    agent = CustomerSupportAgent()
    
    # Scenario 1: Perfect Path - Billing Issue with Account ID
    print("\n📋 Scenario 1: Billing Dispute (Perfect Path)")
    print("-" * 40)
    prompt1 = "Hi, I'm Sarah from TechCorp (SFDC-12345). We were charged twice for our January license - $20,833 showed up twice on our statement."
    
    print(f"Customer: {prompt1}")
    print("\n🤖 Agent Response:")
    response1 = await agent.runner(prompt1)
    print(response1)
    
    # Scenario 2: Technical Issue
    print("\n📋 Scenario 2: Technical Issue")
    print("-" * 40)
    prompt2 = "Hello, I'm John from StartupCo (SFDC-67890). Our dashboard is showing errors and we can't access our reports. This is urgent as we have a client meeting tomorrow."
    
    print(f"Customer: {prompt2}")
    print("\n🤖 Agent Response:")
    response2 = await agent.runner(prompt2)
    print(response2)
    
    # Scenario 3: Escalation Request
    print("\n📋 Scenario 3: Escalation Request")
    print("-" * 40)
    prompt3 = "This billing issue has been open for 3 weeks, I need to speak to a manager immediately. We're evaluating renewal next week. Account SFDC-12345."
    
    print(f"Customer: {prompt3}")
    print("\n🤖 Agent Response:")
    response3 = await agent.runner(prompt3)
    print(response3)
    
    # Scenario 4: General Inquiry (No Account ID)
    print("\n📋 Scenario 4: General Inquiry")
    print("-" * 40)
    prompt4 = "What's your refund policy for enterprise accounts?"
    
    print(f"Customer: {prompt4}")
    print("\n🤖 Agent Response:")
    response4 = await agent.runner(prompt4)
    print(response4)
    
    # Scenario 5: Hallucination Test (Agent might invent policies)
    print("\n📋 Scenario 5: Policy Inquiry (Potential Hallucination)")
    print("-" * 40)
    prompt5 = "Do you offer a 30-day money-back guarantee for enterprise contracts?"
    
    print(f"Customer: {prompt5}")
    print("\n🤖 Agent Response:")
    response5 = await agent.runner(prompt5)
    print(response5)
    
    # Scenario 6: Invalid Account ID (Database Failure)
    print("\n📋 Scenario 6: Invalid Account ID (Database Failure)")
    print("-" * 40)
    prompt6 = "Hi, I'm calling about account SFDC-00000. We have a billing issue that needs immediate attention."
    
    print(f"Customer: {prompt6}")
    print("\n🤖 Agent Response:")
    response6 = await agent.runner(prompt6)
    print(response6)
    
    # Scenario 7: Refund Policy Query (Knowledge Base Timeout)
    print("\n📋 Scenario 7: Refund Policy Query (Knowledge Base Timeout)")
    print("-" * 40)
    prompt7 = "What's your refund policy for enterprise accounts? I need to know about money back guarantees."
    
    print(f"Customer: {prompt7}")
    print("\n🤖 Agent Response:")
    response7 = await agent.runner(prompt7)
    print(response7)
    
    # Scenario 8: System Down Issue (Ticket System Overload)
    print("\n📋 Scenario 8: System Down Issue (Ticket System Overload)")
    print("-" * 40)
    prompt8 = "Hi, I'm from TechCorp (SFDC-12345). Our system is down and not working properly. We need a ticket created for this outage."
    
    print(f"Customer: {prompt8}")
    print("\n🤖 Agent Response:")
    response8 = await agent.runner(prompt8)
    print(response8)
    
    # Scenario 9: After Hours Escalation (Manager Unavailable)
    print("\n📋 Scenario 9: After Hours Escalation (Manager Unavailable)")
    print("-" * 40)
    prompt9 = "This is urgent! I need to speak to a manager immediately. It's after hours but this can't wait until tomorrow. Account SFDC-12345."
    
    print(f"Customer: {prompt9}")
    print("\n🤖 Agent Response:")
    response9 = await agent.runner(prompt9)
    print(response9)
    
    # Scenario 10: SLA Query (Corrupted Knowledge Base)
    print("\n📋 Scenario 10: SLA Query (Corrupted Knowledge Base)")
    print("-" * 40)
    prompt10 = "What's your response time for billing issues? How long will it take to resolve this?"
    
    print(f"Customer: {prompt10}")
    print("\n🤖 Agent Response:")
    response10 = await agent.runner(prompt10)
    print(response10)
    
    # Scenario 11: Minor Issue (Escalation Denied)
    print("\n📋 Scenario 11: Minor Issue (Escalation Denied)")
    print("-" * 40)
    prompt11 = "Hi, I have a small question about my account. It's not important but I'd like to speak to a manager about this minor issue."
    
    print(f"Customer: {prompt11}")
    print("\n🤖 Agent Response:")
    response11 = await agent.runner(prompt11)
    print(response11)
    
    print("\n✅ Demo completed! Check Janus dashboard for tracing data.")
    print("\n💡 Key Features Demonstrated:")
    print("   • Automatic tracing with @track decorator")
    print("   • Manual tracing with start_tool_event/finish_tool_event")
    print("   • One-shot tracing with record_tool_event")
    print("   • CRM data lookup and knowledge base search")
    print("   • Support ticket creation and email drafting")
    print("   • Escalation and follow-up scheduling")
    print("   • Error handling and fallback responses")
    print("   • REALISTIC FAILURE SCENARIOS:")
    print("     - Invalid account IDs (SFDC-00000, SFDC-99999)")
    print("     - Refund policy queries (knowledge base timeout)")
    print("     - System down issues (ticket system overload)")
    print("     - After hours escalation (manager unavailable)")
    print("     - SLA queries (corrupted knowledge base)")
    print("     - Minor issues (escalation denied)")
    print("     - Billing issues (wrong manager assignment)")

if __name__ == "__main__":
    asyncio.run(demo_customer_support_scenarios())
