"""Customer Support Test: Demonstrating manual tracing with Salesforce Agentforce CRM workflow."""

import asyncio
import os
import time
import json
from openai import AsyncOpenAI
from dotenv import load_dotenv
load_dotenv()

import janus_sdk as janus
from janus_sdk import start_tool_event, finish_tool_event, record_tool_event, track

class CustomerSupportAgent:
    """Salesforce Agentforce customer support agent with manual tracing for CRM tool calls."""

    def __init__(self):
        self.client = AsyncOpenAI(
            api_key=os.getenv("OPENAI_API_KEY"),
        )
        self._turn = 0
        self.system_prompt = """You are a professional Salesforce Agentforce customer support agent. 
        You have access to CRM records, knowledge base, and support tools. Always provide accurate, 
        helpful responses based on the data you retrieve. Keep responses under 200 tokens and 
        maintain a professional, empathetic tone."""

    @track
    def extract_customer_info(self, message: str) -> dict:
        """Extract key customer information from their message.
        
        This function is decorated with @track to demonstrate automatic tracing
        alongside manual tracing in the same application.
        """
        # Simple keyword extraction for demo purposes
        account_indicators = ["account", "company", "corp", "inc", "ltd", "techcorp", "enterprise"]
        issue_indicators = ["billing", "charge", "payment", "invoice", "technical", "bug", "error", "problem"]
        urgency_indicators = ["urgent", "immediately", "asap", "critical", "emergency", "renewal"]
        
        # Extract potential account ID (look for SFDC- pattern)
        import re
        account_id_match = re.search(r'SFDC-\d+', message)
        account_id = account_id_match.group() if account_id_match else None
        
        # Determine issue type
        issue_type = "general"
        if any(word in message.lower() for word in ["billing", "charge", "payment", "invoice"]):
            issue_type = "billing"
        elif any(word in message.lower() for word in ["technical", "bug", "error", "problem", "not working"]):
            issue_type = "technical"
        
        # Determine urgency
        urgency = "normal"
        if any(word in message.lower() for word in urgency_indicators):
            urgency = "high"
        
        return {
            "account_id": account_id,
            "issue_type": issue_type,
            "urgency": urgency,
            "has_account_info": account_id is not None,
            "message_length": len(message)
        }

    async def get_crm_record(self, account_id: str) -> dict:
        """Simulate getting CRM record from Salesforce database."""
        # Start manual tracing for CRM lookup
        handle = start_tool_event("crm_database", f"accountId: {account_id}")
        
        try:
            # Simulate database query delay
            await asyncio.sleep(0.4)
            
            # HARDCODED FAILURES - Simulate real-world issues
            if account_id == "SFDC-00000" or account_id == "SFDC-99999":
                # Simulate database connection error for invalid account IDs
                raise Exception("Database connection timeout - unable to retrieve account data")
            
            if account_id == "SFDC-11111" or account_id == "SFDC-22222":
                # Simulate expired account
                result = {
                    "accountId": account_id,
                    "companyName": "Expired Corp",
                    "tier": "Expired",
                    "contractValue": "$0/year",
                    "renewalDate": "2024-01-01",
                    "recentTickets": [],
                    "billingHistory": [],
                    "supportContact": "No Contact",
                    "phoneNumber": "N/A",
                    "status": "EXPIRED",
                    "error": "Account expired - no active contract"
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            # Mock CRM data based on account ID
            if account_id == "SFDC-12345":
                result = {
                    "accountId": "SFDC-12345",
                    "companyName": "TechCorp Industries", 
                    "tier": "Enterprise",
                    "contractValue": "$250,000/year",
                    "renewalDate": "2025-03-15",
                    "recentTickets": [
                        {"id": "T-9876", "type": "Billing", "status": "Open", "created": "2025-01-10"},
                        {"id": "T-9875", "type": "Technical", "status": "Resolved", "created": "2025-01-05"}
                    ],
                    "billingHistory": [
                        {"date": "2024-12-01", "amount": "$20,833", "status": "Paid"},
                        {"date": "2025-01-01", "amount": "$20,833", "status": "Disputed"}
                    ],
                    "supportContact": "Sarah Chen, IT Director",
                    "phoneNumber": "+1-555-0123"
                }
            elif account_id == "SFDC-67890":
                result = {
                    "accountId": "SFDC-67890",
                    "companyName": "StartupCo", 
                    "tier": "Professional",
                    "contractValue": "$12,000/year",
                    "renewalDate": "2025-06-20",
                    "recentTickets": [],
                    "billingHistory": [
                        {"date": "2025-01-01", "amount": "$1,000", "status": "Paid"}
                    ],
                    "supportContact": "John Smith, CEO",
                    "phoneNumber": "+1-555-0456"
                }
            elif account_id == "SFDC-CORRUPT":
                # Simulate corrupted data - wrong information
                result = {
                    "accountId": "SFDC-CORRUPT",
                    "companyName": "Wrong Company Name",  # Wrong company name
                    "tier": "Enterprise",
                    "contractValue": "$999,999/year",  # Wrong contract value
                    "renewalDate": "2026-12-31",  # Wrong renewal date
                    "recentTickets": [
                        {"id": "T-FAKE", "type": "Billing", "status": "Open", "created": "2025-01-01"}
                    ],
                    "billingHistory": [
                        {"date": "2024-12-01", "amount": "$50,000", "status": "Paid"}  # Wrong amount
                    ],
                    "supportContact": "Fake Person",  # Wrong contact
                    "phoneNumber": "+1-555-FAKE"  # Wrong phone
                }
            else:
                result = {
                    "accountId": account_id,
                    "companyName": "Unknown Company",
                    "tier": "Unknown",
                    "contractValue": "Unknown",
                    "renewalDate": "Unknown",
                    "recentTickets": [],
                    "billingHistory": [],
                    "supportContact": "Unknown",
                    "phoneNumber": "Unknown"
                }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error retrieving CRM record: {str(e)}"}

    async def search_knowledge(self, query: str) -> dict:
        """Simulate searching knowledge base for policies and procedures."""
        # Start manual tracing for knowledge search
        handle = start_tool_event("knowledge_base", f"query: {query}")
        
        try:
            # Simulate search delay
            await asyncio.sleep(0.3)
            
            # HARDCODED FAILURES - Simulate knowledge base issues
            if any(word in query.lower() for word in ["refund policy", "money back", "guarantee", "return policy"]):
                # Simulate knowledge base timeout for refund queries (common customer question)
                raise Exception("Knowledge base search timeout - service unavailable")
            
            if any(word in query.lower() for word in ["sla", "response time", "how long", "when will"]):
                # Simulate corrupted knowledge base response for SLA queries
                result = {
                    "results": [
                        {
                            "title": "CORRUPTED SLA DATA",
                            "content": "This SLA information is corrupted and should not be trusted.",
                            "policyId": "POL-CORRUPT-001"
                        }
                    ],
                    "error": "Knowledge base returned corrupted data"
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            if any(word in query.lower() for word in ["escalation", "manager", "supervisor", "urgent"]):
                # Simulate outdated policy information for escalation queries
                result = {
                    "results": [
                        {
                            "title": "OUTDATED Escalation Process",
                            "content": "Escalate to manager within 2 hours. (NOTE: This policy is outdated - current escalation time is 4 hours)",
                            "policyId": "POL-ESC-OLD-001",
                            "lastUpdated": "2023-01-01",
                            "status": "OUTDATED"
                        }
                    ]
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            # Mock knowledge base responses
            query_lower = query.lower()
            
            if "billing dispute" in query_lower or "billing" in query_lower:
                result = {
                    "results": [
                        {
                            "title": "Enterprise Billing Dispute Process",
                            "content": "Enterprise accounts: Resolve within 48 hours. Requires manager approval for adjustments over $5,000.",
                            "policyId": "POL-BIL-001"
                        },
                        {
                            "title": "Billing Refund Policy",
                            "content": "No 30-day money-back guarantee for enterprise contracts. Refunds only for billing errors.",
                            "policyId": "POL-BIL-002"
                        }
                    ]
                }
            elif "technical" in query_lower or "bug" in query_lower:
                result = {
                    "results": [
                        {
                            "title": "Technical Issue Resolution",
                            "content": "Standard SLA: 24 hours for critical issues, 72 hours for normal issues.",
                            "policyId": "POL-TECH-001"
                        }
                    ]
                }
            elif "escalation" in query_lower or "manager" in query_lower:
                result = {
                    "results": [
                        {
                            "title": "Escalation Procedures",
                            "content": "Escalate to manager for: billing disputes >$5k, renewal risk, or customer requests.",
                            "policyId": "POL-ESC-001"
                        }
                    ]
                }
            else:
                result = {
                    "results": [
                        {
                            "title": "General Support Policy",
                            "content": "Provide helpful, accurate information. Escalate complex issues to appropriate teams.",
                            "policyId": "POL-GEN-001"
                        }
                    ]
                }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error searching knowledge base: {str(e)}"}

    async def create_ticket(self, details: str, priority: str = "normal") -> dict:
        """Simulate creating a support ticket."""
        # Start manual tracing for ticket creation
        handle = start_tool_event("ticket_system", f"details: {details[:50]}..., priority: {priority}")
        
        try:
            # Simulate ticket creation delay
            await asyncio.sleep(0.2)
            
            # HARDCODED FAILURES - Simulate ticket system issues
            if any(word in details.lower() for word in ["duplicate", "already", "same issue", "repeated"]):
                # Simulate duplicate ticket creation failure
                raise Exception("Ticket creation failed - duplicate ticket already exists for this issue")
            
            if any(word in details.lower() for word in ["system down", "not working", "broken", "outage"]):
                # Simulate system overload
                raise Exception("Ticket system overloaded - unable to create ticket at this time")
            
            if any(word in details.lower() for word in ["urgent", "emergency", "critical", "asap"]):
                # Simulate invalid ticket data for urgent requests
                result = {
                    "ticketId": "INVALID",
                    "status": "Failed",
                    "priority": priority,
                    "assignedTo": "None",
                    "eta": "Unknown",
                    "error": "Invalid ticket data provided - missing required fields"
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            # Generate mock ticket ID
            import random
            ticket_id = f"T-{random.randint(9000, 9999)}"
            
            # Mock ticket creation response
            result = {
                "ticketId": ticket_id,
                "status": "Created",
                "priority": priority,
                "assignedTo": "Billing Team" if "billing" in details.lower() else "Technical Team",
                "eta": "48 hours" if priority == "high" else "72 hours"
            }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error creating ticket: {str(e)}"}

    async def draft_email(self, recipient: str, subject: str, content: str) -> dict:
        """Simulate drafting an email response."""
        # Start manual tracing for email drafting
        handle = start_tool_event("email_system", f"to: {recipient}, subject: {subject[:30]}...")
        
        try:
            # Simulate email drafting delay
            await asyncio.sleep(0.1)
            
            # Mock email response
            result = {
                "emailId": f"EMAIL-{int(time.time())}",
                "status": "Drafted",
                "recipient": recipient,
                "subject": subject,
                "content": content,
                "sent": False
            }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error drafting email: {str(e)}"}

    async def escalate_to_manager(self, reason: str, details: str) -> dict:
        """Simulate escalating to manager."""
        # Start manual tracing for escalation
        handle = start_tool_event("escalation_system", f"reason: {reason}, details: {details[:50]}...")
        
        try:
            # Simulate escalation delay
            await asyncio.sleep(0.2)
            
            # HARDCODED FAILURES - Simulate escalation issues
            if any(word in reason.lower() for word in ["after hours", "weekend", "holiday", "night"]) or any(word in details.lower() for word in ["after hours", "weekend", "holiday", "night"]):
                # Simulate manager unavailable during off-hours
                raise Exception("Manager escalation failed - no managers available at this time")
            
            if any(word in reason.lower() for word in ["small", "minor", "low", "not important"]) or any(word in details.lower() for word in ["small", "minor", "low", "not important"]):
                # Simulate insufficient priority for escalation
                result = {
                    "escalationId": "ESC-DENIED",
                    "status": "Denied",
                    "assignedManager": "None",
                    "priority": "Low",
                    "eta": "N/A",
                    "error": "Insufficient priority for manager escalation - issue does not meet escalation criteria"
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            if any(word in reason.lower() for word in ["billing", "payment", "invoice"]) or any(word in details.lower() for word in ["billing", "payment", "invoice"]):
                # Simulate wrong manager assignment for billing issues
                result = {
                    "escalationId": f"ESC-{int(time.time())}",
                    "status": "Escalated",
                    "assignedManager": "Technical Manager, IT Department",  # Wrong manager for billing
                    "priority": "High",
                    "eta": "4 hours",
                    "error": "Escalated to wrong manager - will need to be reassigned"
                }
                finish_tool_event(handle, json.dumps(result, indent=2))
                return result
            
            # Mock escalation response
            result = {
                "escalationId": f"ESC-{int(time.time())}",
                "status": "Escalated",
                "assignedManager": "Sarah Johnson, Support Manager",
                "priority": "High",
                "eta": "4 hours"
            }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error escalating to manager: {str(e)}"}

    async def schedule_followup(self, date: str, followup_type: str) -> dict:
        """Simulate scheduling a follow-up."""
        # Start manual tracing for follow-up scheduling
        handle = start_tool_event("calendar_system", f"date: {date}, type: {followup_type}")
        
        try:
            # Simulate scheduling delay
            await asyncio.sleep(0.1)
            
            # Mock follow-up response
            result = {
                "followupId": f"FU-{int(time.time())}",
                "status": "Scheduled",
                "date": date,
                "type": followup_type,
                "assignedTo": "Support Team"
            }
            
            # Finish tracing with success
            finish_tool_event(handle, json.dumps(result, indent=2))
            return result
            
        except Exception as e:
            # Finish tracing with error
            finish_tool_event(handle, error=e)
            return {"error": f"Error scheduling follow-up: {str(e)}"}

    async def runner(self, prompt: str) -> str:
        """Main agent runner with manual tracing for customer support tool calls."""
        self._turn += 1
        
        # Use @track decorated function to extract customer info (automatic tracing)
        customer_info = self.extract_customer_info(prompt)
        
        # Check if customer provided account ID
        if customer_info["has_account_info"]:
            # Get CRM record with manual tracing
            crm_data = await self.get_crm_record(customer_info["account_id"])
            
            # Search knowledge base for relevant policies
            knowledge_query = f"{customer_info['issue_type']} {customer_info['urgency']} resolution"
            knowledge_data = await self.search_knowledge(knowledge_query)
            
            # Create support ticket
            ticket_details = f"Customer inquiry: {prompt[:100]}... Account: {customer_info['account_id']}"
            ticket_priority = "high" if customer_info["urgency"] == "high" else "normal"
            ticket_data = await self.create_ticket(ticket_details, ticket_priority)
            
            # Check if escalation is needed
            escalation_needed = False
            if customer_info["urgency"] == "high" or "manager" in prompt.lower() or "escalate" in prompt.lower():
                escalation_data = await self.escalate_to_manager(
                    f"High priority {customer_info['issue_type']} issue", 
                    f"Account: {customer_info['account_id']}, Issue: {prompt[:100]}"
                )
                escalation_needed = True
            
            # Draft email response
            email_subject = f"Re: {customer_info['issue_type'].title()} Support Request - {ticket_data.get('ticketId', 'T-XXXX')}"
            email_content = f"Thank you for contacting support. We've created ticket {ticket_data.get('ticketId', 'T-XXXX')} and will resolve this within {ticket_data.get('eta', '72 hours')}."
            
            if escalation_needed:
                email_content += f" Your case has been escalated to {escalation_data.get('assignedManager', 'our manager')}."
            
            email_data = await self.draft_email(
                crm_data.get("supportContact", "customer@company.com"),
                email_subject,
                email_content
            )
            
            # Schedule follow-up if needed
            if customer_info["issue_type"] == "billing":
                followup_data = await self.schedule_followup("2025-01-15", "billing_resolution")
            
            # Use one-shot tracing for the response generation
            record_tool_event("response_generation", f"support_query: {prompt}", "generating_response")
            
            # Generate response with all the gathered information
            resp = await self.client.chat.completions.create(
                model="gpt-4o-mini",
                messages=[
                    {"role": "system", "content": self.system_prompt},
                    {"role": "user", "content": f"""Customer inquiry: {prompt}

CRM Data: {json.dumps(crm_data, indent=2)}
Knowledge Base: {json.dumps(knowledge_data, indent=2)}
Ticket Created: {json.dumps(ticket_data, indent=2)}
Email Drafted: {json.dumps(email_data, indent=2)}
{f"Escalation: {json.dumps(escalation_data, indent=2)}" if escalation_needed else ""}

Provide a helpful, professional response incorporating this information."""}
                ],
                max_tokens=200,
            )
            
        else:
            # No account ID provided - general support
            knowledge_data = await self.search_knowledge("general support policy")
            
            # Use one-shot tracing for the response generation
            record_tool_event("response_generation", f"general_query: {prompt}", "generating_response")
            
            # Generate general response
            resp = await self.client.chat.completions.create(
                model="gpt-4o-mini",
                messages=[
                    {"role": "system", "content": self.system_prompt},
                    {"role": "user", "content": f"""Customer inquiry: {prompt}

Knowledge Base: {json.dumps(knowledge_data, indent=2)}

Provide a helpful response. If they need specific account help, ask for their account ID (SFDC-XXXXX format)."""}
                ],
                max_tokens=200,
            )
        
        if resp.choices is None or not resp.choices:
            err = getattr(resp, "error", None) or resp.model_dump().get("error")
            raise RuntimeError(f"OpenAI API error: {err}")
        
        message = resp.choices[0].message
        answer = message.content
        return answer

async def main():
    """Run customer support simulations with manual tracing."""
    
    await janus.run_simulations(
        num_simulations=5,
        max_turns=5,
        target_agent=lambda: CustomerSupportAgent().runner,
        api_key=os.getenv("JANUS_API_KEY")
    )

if __name__ == "__main__":
    asyncio.run(main())
