# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['jetblack_messagebus',
 'jetblack_messagebus.authentication',
 'jetblack_messagebus.io']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'jetblack-messagebus-python3',
    'version': '4.1.1',
    'description': 'A python3 client for jetblack-messagebus',
    'long_description': '# jetblack-messagebus-python3\n\n\n## Overview\n\nThis is a Python 3.7+ client for the [jetblack-messagebus](https://github.com/rob-blackbourn/jetblack-messagebus).\n\nIt follows the publish-subscribe pattern, and includes support for "notification" of\na subscription by another client. This allows it to provide data on-demand.\n\nSee the server documentation for more detailed information.\n\n## Example\n\nThe client below subscribes on feed "TEST" to topic "FOO" and prints out \nthe data it receives.\n\n```python\nimport asyncio\n\nfrom jetblack_messagebus import CallbackClient\n\nasync def on_data(user, host, feed, topic, data_packets, is_image):\n    print(f\'data: user="{user}",host="{host}",feed="{feed}",topic="{topic}",is_image={is_image}\')\n    if not data_packets:\n        print("no data")\n    else:\n        for packet in data_packets:\n            message = packet.data.decode(\'utf8\') if packet.data else None\n            print(f\'packet: entitlements={packet.entitlements},message={message}\')\n\nasync def main():\n    client = await CallbackClient.create(\'localhost\', 9001)\n    client.data_handlers.append(on_data)\n    await client.add_subscription(\'TEST\', \'FOO\')\n    await client.start()\n\nif __name__ == \'__main__\':\n    asyncio.run(main())\n```\n\n## SSL\n\nTo create an SSL subscriber, pass in the ssl context.\n\n```python\nimport ssl\nfrom jetblack_messagebus import CallbackClient\n\n...\n\nasync def main():\n    """Start the demo"""\n    ssl_context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)\n    client = await CallbackClient.create(\'myhost.example.com\', 9001, ssl=ssl_context)\n    await client.add_subscription(\'TEST\', \'FOO\')\n    await client.start()\n```\n\n## Authentication\n\nThe message bus currently supports the following authentication methods:\n\n- Password File\n- LDAP\n- JWT\n\n### Password File and LDAP\n\nBoth the password file and LDAP methods require a username and password. This\nis provided by the basic authenticator.\n\n```python\nimport ssl\nfrom jetblack_messagebus import CallbackClient\nfrom jetblack_messagebus.authentication import BasicAuthenticator\n\n...\n\nasync def main():\n    authenticator = BasicAuthenticator("john.doe@example.com", "pa$$word")\n    ssl_context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)\n    client = await CallbackClient.create(\n        \'myhost.example.com\',\n        9001,\n        ssl=ssl_context,\n        authenticator=authenticator\n    )\n    await client.add_subscription(\'TEST\', \'FOO\')\n    await client.start()\n```\n\n### JWT\n\nJWT authentication requires a valid token to be passed by the client. This\nis provided by the token authenticator.\n\n```python\nimport ssl\nfrom jetblack_messagebus import CallbackClient\nfrom jetblack_messagebus.authentication import TokenAuthenticator\n\n...\n\nasync def main():\n    authenticator = TokenAuthenticator(\n        "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiJqb2huLmRvZUBleGFtcGxlLmNvbSIsIm5hbWUiOiJKb2huIERvZSIsImlhdCI6MTUxNjIzOTAyMiwiZXhwIjoxNTE3MTgzNTAxfQ.wLSGBcNUT8r1DqQvaBrrGY4NHiiVOpoxrgeoPsSsJkY"\n    )\n    ssl_context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)\n    client = await CallbackClient.create(\n        \'myhost.example.com\',\n        9001,\n        ssl=ssl_context,\n        authenticator=authenticator\n    )\n    await client.add_subscription(\'TEST\', \'FOO\')\n    await client.start()\n```\n\n## Message handlers\n\nThere are three types of messages that can be received:\n\n- Data\n- Subscription notifications\n- Authorization requests.\n\n### Data\n\nA data handler looks like this:\n\n```python\n# A data handler.\nasync def on_data(\n        user: str,\n        host: str,\n        feed: str,\n        topic: str,\n        data_packets: Optional[List[DataPacket]],\n        is_image: bool\n) -> None:\n    """Called when data is received"""\n    pass\n\n# Add the handler to the client.\nclient.data_handlers.append(on_data)\n# Remove the handler\nclient.data_handlers.remove(on_data)\n```\n\nThe data packets have two fields: `entitlements` and `data`.\n\nThe `entitlements` is a optional set of ints which express the entitlements that were\nrequired for the data to have been received.\n\nThe `data` is an optional `bytes` holding the encoded data. What this represents\nis agreed by the sender and receiver. For example it might be a simple string, some\nJSON text, or a protocol buffer.\n\n### Subscription notifications\n\nA subscription notification handler looks like this:\n\n```python\n# A notification handler.\nasync def on_notification(\n        client_id: UUID,\n        user: str,\n        host: str,\n        feed: str,\n        topic: str,\n        is_add: bool\n) -> None:\n    """Called for a notification"""\n    pass\n\n# Add the handler to the client.\nclient.notification_handlers.append(on_notification)\n# Remove the handler\nclient.notification_handlers.remove(on_notification)\n```\n\n### Authorization requests\n\n```python\n# An authorization handler.\nasync def on_authorization(\n        client_id: UUID,\n        host: str,\n        user: str,\n        feed: str,\n        topic: str\n) -> None:\n    """Called when authorization is requested"""\n    pass\n\n# Add the handler to the client.\nclient._authorization_handlers.append(on_authorization)\n# Remove the handler\nclient._authorization_handlers.remove(on_authorization)\n```\n',
    'author': 'Rob Blackbourn',
    'author_email': 'rob.blackbourn@googlemail.com',
    'url': 'https://github.com/rob-blackbourn/jetblack-messagebus-python3',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
