#!/usr/bin/env python3
"""
verify_md5_manifest.py
======================

Summary
-------
Verifies file integrity within a test-case directory against the
`scripts/md5_manifest.txt` file generated by `create_md5_manifest.py`.
Each entry’s MD5 checksum is compared with the current file’s checksum.
A concise execution report is written to the output directory.

Compliance
----------
Implements:
- CLP1 : Command-line Parameters
- LOG1 / LOG2 : Logging behavior
- GG2 : Standard output behavior (absolute path to report file)
- EXC1 : Controlled exit codes
- VAL1 : Validation of test-case-XXX directory naming
- VER1 : Manifest verification logic
- REP1 : Report generation summarizing results

Usage
-----
$ python verify_md5_manifest.py --test-case-dir <path> [--outdir <dir>] [--logfile <file>]
"""

from __future__ import annotations

import getpass
import hashlib
import logging
import sys
from datetime import datetime
from pathlib import Path
from typing import List, Tuple

import typer

# ---------------------------------------------------------------------------
# Logging
# ---------------------------------------------------------------------------


def setup_logging(logfile: Path) -> logging.Logger:
    """Configure logging such that INFO+ goes to file and WARNING+ goes to stderr.

    Args:
        logfile: Path to the log file.

    Returns:
        Configured logger instance.
    """
    logger = logging.getLogger("verify_md5_manifest")
    logger.setLevel(logging.DEBUG)
    logger.handlers.clear()

    fh = logging.FileHandler(logfile, encoding="utf-8")
    fh.setLevel(logging.INFO)
    fmt = logging.Formatter("%(levelname)s : %(asctime)s : %(pathname)s : %(lineno)d : %(message)s")
    fh.setFormatter(fmt)
    logger.addHandler(fh)

    sh = logging.StreamHandler(stream=sys.stderr)
    sh.setLevel(logging.WARNING)
    sh.setFormatter(fmt)
    logger.addHandler(sh)

    return logger


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------


def validate_test_case_dir(path: Path) -> None:
    """Validate that the directory name follows the test-case-XXX pattern.

    Args:
        path: Path to the test case directory.

    Raises:
        BadParameter: If the directory name does not match required format.
    """
    if not path.is_dir():
        raise typer.BadParameter(f"Not a directory: {path}")
    name = path.name
    if not name.startswith("test-case-"):
        raise typer.BadParameter("Directory name must start with 'test-case-'.")
    suffix = name.replace("test-case-", "")
    if not suffix.isdigit() or len(suffix) != 3 or suffix == "000":
        raise typer.BadParameter("Basename must end with a three-digit integer 001..999.")


def compute_md5(file_path: Path, chunk_size: int = 1024 * 1024) -> str:
    """Compute MD5 checksum for a file.

    Args:
        file_path: Path to the file to compute MD5 for.
        chunk_size: Size of chunks to read the file in bytes.

    Returns:
        MD5 checksum as a hexadecimal string.
    """
    h = hashlib.md5()
    with file_path.open("rb") as f:
        for chunk in iter(lambda: f.read(chunk_size), b""):
            h.update(chunk)
    return h.hexdigest()


def default_outdir(progname: str) -> Path:
    """Construct default output directory under /tmp/<user>/<progname>/<timestamp>.

    Args:
        progname: Name of the running program, used as the subdirectory.

    Returns:
        Path to the newly constructed output directory.
    """
    user = getpass.getuser()
    ts = datetime.now().strftime("%Y-%m-%d-%H%M%S")
    return Path(f"/tmp/{user}/{progname}/{ts}")


# ---------------------------------------------------------------------------
# CLI
# ---------------------------------------------------------------------------

app = typer.Typer(add_completion=False, no_args_is_help=True)


@app.command()
def main(
    test_case_dir: Path = typer.Option(
        ...,
        "--test-case-dir",
        exists=True,
        file_okay=False,
        dir_okay=True,
        readable=True,
        help="Path to the test case directory (must be named test-case-XXX).",
    ),
    outdir: Path = typer.Option(
        default_outdir(Path(__file__).stem),
        "--outdir",
        help="Where to write logs/report. Defaults to /tmp/[user]/<progname>/<timestamp>.",
    ),
    logfile: Path = typer.Option(
        None,
        "--logfile",
        help="Optional logfile path. Defaults to <outdir>/<progname>.log.",
    ),
):
    """Verify MD5 manifest and generate concise report (SRS: VER1, REP1).

    Args:
        test_case_dir: Path to the test-case-XXX directory.
        outdir: Directory to write logs and report.
        logfile: Optional path to the logfile.

    Raises:
        Exit: On validation or verification failure.
    """
    validate_test_case_dir(test_case_dir)
    outdir.mkdir(parents=True, exist_ok=True)
    final_log = logfile if logfile else outdir / f"{Path(__file__).stem}.log"
    logger = setup_logging(final_log)

    logger.info("Loading manifest for verification.")
    manifest_path = test_case_dir / "scripts" / "md5_manifest.txt"
    if not manifest_path.exists():
        typer.secho(f"Manifest not found: {manifest_path}", err=True)
        raise typer.Exit(code=3)

    # Read manifest
    entries: List[Tuple[Path, str]] = []
    with manifest_path.open("r", encoding="utf-8") as mf:
        for line in mf:
            if line.startswith("##") or not line.strip():
                continue
            cols = line.strip().split("\t")
            if len(cols) < 2:
                continue
            entries.append((Path(cols[0]), cols[1]))

    passed = 0
    failures: List[Tuple[Path, str, str]] = []
    missing: List[Path] = []

    for file_path, expected_md5 in entries:
        if not file_path.exists():
            missing.append(file_path)
            continue
        actual_md5 = compute_md5(file_path)
        if actual_md5 == expected_md5:
            passed += 1
        else:
            failures.append((file_path, actual_md5, expected_md5))

    failed = len(failures)
    logger.info(
        "Verification complete: %d passed, %d failed, %d missing", passed, failed, len(missing)
    )

    # -----------------------------------------------------------------------
    # Report and Exit Handling
    # -----------------------------------------------------------------------
    report_lines = [
        f"Program: {Path(__file__).stem}",
        f"Test case dir: {test_case_dir}",
        f"Manifest: {manifest_path.resolve()}",
        f"Verified files: {passed + failed} (passed={passed}, failed={failed})",
    ]

    if failures:
        report_lines.append("Failures:")
        for p, got, exp in failures:
            report_lines.append(f"- {p} current={got} expected={exp}")
        logger.error("Detected %d file(s) with MD5 mismatches.", len(failures))
    else:
        logger.info("All files verified successfully.")

    if missing:
        report_lines.append("Missing files:")
        report_lines.extend([f"- {p}" for p in missing])
        logger.error("Detected %d missing file(s).", len(missing))

    report_path = outdir / "report.txt"
    report_path.write_text("\n".join(report_lines) + "\n", encoding="utf-8")
    logger.info("Report written: %s", report_path)

    # GG2: print absolute path to report file to stdout
    print(str(report_path.resolve()))

    # Determine exit code
    if missing:
        raise typer.Exit(code=3)  # Missing files → failure
    elif failures:
        raise typer.Exit(code=1)  # MD5 mismatches → integrity failure
    else:
        raise typer.Exit(code=0)  # All OK


if __name__ == "__main__":
    app()
