# encoding: utf-8
#
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Contact: Kyle Lahnakoski (kyle@lahnakoski.com)
#


import itertools

from jx_base.expressions import TRUE
from jx_base.expressions.variable import is_variable
from jx_base.language import is_expression
from jx_base.models.container import Container
from jx_base.models.namespace import Namespace
from jx_base.models.schema import Schema
from jx_base.models.snowflake import Snowflake
from jx_base.models.table import Table
from jx_base.utils import delist, enlist
from jx_python.convert import list2cube, list2table
from jx_python.expressions import jx_expression_to_function
from jx_python.lists.aggs import is_aggs, list_aggs
from mo_collections import UniqueIndex
from mo_dots import (
    Data,
    Null,
    is_data,
    is_list,
    from_data,
    to_data,
    coalesce,
    dict_to_data,
    last,
    startswith_field,
)
from mo_future import first, sort_using_key
from mo_imports import export, expect
from mo_json import ARRAY
from mo_logs import Log
from mo_threads import Lock

jx, get_schema_from_list, Column = expect("jx", "get_schema_from_list", "Column")


class ListContainer(Container, Namespace, Table):
    """
    A CONTAINER WITH ONLY ONE TABLE
    A PYTHON LIST PAIRED WITH SCHEMA SO QUERY EXPRESSIONS CAN BE TRANSPILED
    """

    def __init__(self, name, data, schema=None):
        # TODO: STORE THIS LIKE A CUBE FOR FASTER ACCESS AND TRANSFORMATION
        Container.__init__(self)
        self.name = name = coalesce(name, ".")
        self.data = data = list(from_data(data))
        self.schema = schema or get_schema_from_list(name, data)
        self.locker = Lock()  # JUST IN CASE YOU WANT TO DO MORE THAN ONE THING

    @property
    def nested_path(self):
        return [self.name]

    def get_facts(self, fact_name):
        return self

    def get_schema(self, query_path=None):
        if query_path is None:
            return self.schema
        snowflake = self.schema.snowflake
        if query_path not in snowflake.query_paths:
            Log.error("This container only has tables with names {names}", names=self.schema.snowflake.query_paths)

        nested_path = []
        for path in snowflake.query_paths:
            if startswith_field(query_path, path):
                nested_path.append(path)
        return Schema(list(reversed(nested_path)), snowflake)

    def get_snowflake(self, fact_name):
        return Snowflake(fact_name, self)

    def get_relations(self):
        return self.relations[:]

    def get_columns(self, table_name):
        return self.columns.find_columns(table_name)

    def get_tables(self):
        return list(sorted(self.columns.data.keys()))

    @property
    def namespace(self):
        return self

    def last(self):
        """
        :return:  Last element in the list, or Null
        """
        last(self.data)

    def query(self, q):
        q = to_data(q)
        output = self
        if is_aggs(q):
            output = list_aggs(output.data, q)
        else:
            if q.where is not TRUE:
                output = output.where(q.where)

            if q.sort:
                output = output.sort(q.sort)

            if q.select:
                output = output.select(q.select)

        # TODO: ADD EXTRA COLUMN DESCRIPTIONS TO RESULTING SCHEMA
        for param in q.window:
            output.window(param)

        if q.format:
            if q.format == "list":
                return Data(data=output.data, meta={"format": "list"})
            elif q.format == "table":
                head = [c.name for c in output.schema.columns]
                data = [[r if h == "." else r[h] for h in head] for r in output.data]
                return Data(header=head, data=data, meta={"format": "table"})
            elif q.format == "cube":
                head = [c.name for c in output.schema.columns]
                rows = [[r[h] for h in head] for r in output.data]
                data = {h: c for h, c in zip(head, zip(*rows))}
                return Data(
                    data=data,
                    meta={"format": "cube"},
                    edges=[{
                        "name": "rownum",
                        "domain": {"type": "rownum", "min": 0, "max": len(rows), "interval": 1},
                    }],
                )
            else:
                Log.error("unknown format {{format}}", format=q.format)
        else:
            return output

    def update(self, command):
        """
        EXPECTING command == {"set":term, "clear":term, "where":where}
        THE set CLAUSE IS A DICT MAPPING NAMES TO VALUES
        THE where CLAUSE IS A JSON EXPRESSION FILTER
        """
        command = to_data(command)
        command_clear = enlist(command["clear"])
        command_set = command.set.items()
        command_where = jx.get(command.where)

        for c in self.data:
            if command_where(c):
                for k in command_clear:
                    c[k] = None
                for k, v in command_set:
                    c[k] = v

    def where(self, where):
        if is_data(where) or is_expression(where):
            temp = jx_expression_to_function(where)
        else:
            temp = where

        return ListContainer("from " + self.name, filter(temp, self.data), self.schema)

    filter = where

    def sort(self, sort):
        return ListContainer("sorted " + self.name, jx.sort(self.data, sort, already_normalized=True), self.schema,)

    def get(self, select):
        """
        :param select: the variable to extract from list
        :return:  a simple list of the extraction
        """
        if is_list(select):
            return [(d[s] for s in select) for d in self.data]
        else:
            return [d[select] for d in self.data]

    def select(self, select):
        selects = enlist(select)

        if len(selects) == 1 and is_variable(selects[0].value) and selects[0].value.var == ".":
            new_schema = self.schema
            if selects[0].name == ".":
                return self
        else:
            new_schema = None

        if is_list(select):
            if all(is_variable(s.value) and s.name == s.value.var for s in select):
                names = set(s.value.var for s in select)
                new_schema = Schema(".", [c for c in self.schema.columns if c.name in names])

            push_and_pull = [(s.name, jx_expression_to_function(s.value)) for s in selects]

            def selector(d):
                output = Data()
                for n, p in push_and_pull:
                    output[n] = delist(p(to_data(d)))
                return from_data(output)

            new_data = list(map(selector, self.data))
        else:
            select_value = jx_expression_to_function(select.value)
            new_data = list(map(select_value, self.data))
            if is_variable(select.value):
                column = dict(**first(c for c in self.schema.columns if c.name == select.value.var))
                column.update({
                    "name": ".",
                    "json_type": ARRAY,
                    "es_type": "nested",
                    "multi": 1001,
                    "cardinality": 1,
                })
                new_schema = Schema("from " + self.name, [Column(**column)])

        return ListContainer("from " + self.name, data=new_data, schema=new_schema)

    def window(self, window):
        # _ = window
        jx.window(self.data, window)
        return self

    def format(self, format):
        if format == "table":
            frum = list2table(self.data, self._schema.lookup.keys())
        elif format == "cube":
            frum = list2cube(self.data, self.schema.lookup.keys())
        else:
            frum = self.__data__()

        return frum

    def groupby(self, keys, contiguous=False):
        try:
            keys = enlist(keys)
            get_key = jx_expression_to_function(keys)
            if not contiguous:
                data = sort_using_key(self.data, key=get_key)

            def _output():
                for g, v in itertools.groupby(data, get_key):
                    group = Data()
                    for k, gg in zip(keys, g):
                        group[k] = gg
                    yield (group, to_data(list(v)))

            return _output()
        except Exception as e:
            Log.error("Problem grouping", e)

    def insert(self, documents):
        self.data.extend(documents)

    def extend(self, documents):
        self.data.extend(documents)

    def __data__(self):
        if first(self.schema.columns).name == ".":
            return dict_to_data({"meta": {"format": "list"}, "data": self.data})
        else:
            return dict_to_data({
                "meta": {"format": "list"},
                "data": [{k: delist(v) for k, v in row.items()} for row in self.data],
            })

    def get_columns(self, table_name=None):
        return self.schema.values()

    def add(self, value):
        self.data.append(value)

    def __getitem__(self, item):
        if item < 0 or len(self.data) <= item:
            return Null
        return self.data[item]

    def __iter__(self):
        return (to_data(d) for d in self.data)

    def __len__(self):
        return len(self.data)

    def get_snowflake(self, name):
        if self.name != name:
            Log.error("This container only has table by name of {{name}}", name=name)
        return self

    def get_table(self, name):
        if self is name or self.name == name:
            return self
        Log.error("This container only has table by name of {{name}}", name=name)


DUAL = ListContainer(
    name="dual", data=[{}], schema=Schema(["dual"], Snowflake(None, ["dual"], columns=UniqueIndex(keys=("name",))))
)


export("jx_base.models.container", ListContainer)
