import logging
from playwright.async_api import async_playwright
from bs4 import BeautifulSoup, Comment
import asyncio
import nest_asyncio
import subprocess
import os
import json
from kaizen.helpers import general

logger = logging.getLogger(__name__)


PR_COLLAPSIBLE_TEMPLATE = """
<details>
<summary> {comment}</summary> \n\n
<strong> Potential Solution:</strong> \n\n{solution}
\n\n
<blockquote>  
    <p><code>{file_name} | {start_line} - {end_line}</code></p>
    <p>reason_for_request: {reason}</p>
    <p>level: [{confidence}] , severity: [{severity}]</p>
</blockquote>  
</details> \n

"""

DESC_COLLAPSIBLE_TEMPLATE = """
<details>
<summary>Original Description</summary>
{desc}
</details>

"""


def create_pr_description(desc, original_desc):
    markdown_output = desc
    markdown_output += (
        "\n\n> ✨ Generated with love by [Kaizen](https://cloudcode.ai) ❤️"
    )
    markdown_output += "\n\n" + DESC_COLLAPSIBLE_TEMPLATE.format(desc=original_desc)
    return markdown_output


def create_issue_description(desc, original_desc):
    pass


async def get_html(url):
    async with async_playwright() as p:
        subprocess.run(["playwright", "install", "--with-deps"], check=True)
        browser = await p.chromium.launch(headless=True)
        page = await browser.new_page()
        await page.goto(url)
        html = await page.content()
        await browser.close()
        return html


def get_web_html(url):
    nest_asyncio.apply()
    html = asyncio.run(get_html(url))
    soup = BeautifulSoup(html, "html.parser")

    # Remove SVG elements
    for svg in soup.find_all("svg"):
        svg.decompose()

    # Remove HTML comments
    for comment in soup.find_all(text=lambda text: isinstance(text, Comment)):
        comment.extract()

    # Remove <style> elements
    for style_block in soup.find_all("style"):
        style_block.decompose()

    # Remove <script> elements
    for script in soup.find_all("script"):
        script.decompose()

    # Remove <noscript> elements
    for noscript in soup.find_all("noscript"):
        noscript.decompose()

    # Remove <link> elements (typically used for stylesheets)
    for link in soup.find_all("link"):
        link.decompose()

    # Remove <meta> elements (typically used for metadata)
    for meta in soup.find_all("meta"):
        meta.decompose()

    # Remove <head> element (contains metadata, scripts, and stylesheets)
    for head in soup.find_all("head"):
        head.decompose()

    pretty_html = soup.prettify()
    return pretty_html


def get_parent_folder():
    return os.getcwd()


def create_folder(folder_path):
    if not folder_path:
        raise ValueError("Folder path cannot be empty")
    if not os.path.exists(folder_path):
        os.makedirs(folder_path)
        logger.debug(f"Folder '{folder_path}' created successfully.")
    else:
        logger.debug(f"Folder '{folder_path}' already exists.")


def create_test_files(json_tests, folder_path):
    with open(f"{folder_path}/tests.json", "w") as f:
        f.write(json.dumps(json_tests))

    for module in json_tests:
        temp_folder_path = os.path.join(folder_path, module["folder_name"])
        create_folder(temp_folder_path)

        for test in module["tests"]:
            file_path = os.path.join(
                temp_folder_path,
                "test_" + "_".join(test["test_name"].lower().split(" ")) + ".py",
            )
            with open(file_path, "w") as f:
                cleaned_code = general.clean_python_code(test["code"])
                if not cleaned_code:
                    logger.info("Failed to clean code")
                else:
                    cleaned_code = (
                        f"'''Importance: {module['importance']}\
                            \nModule Name: {module['module_title']}\
                            \nDescription: {test['test_description']}\n'''\n\n"
                        + cleaned_code
                    )
                    f.write(cleaned_code)
