from kaze.Utils.WalletFixtureTestCase import WalletFixtureTestCase
from kaze.Wallets.utils import to_aes_key
from kaze.Implementations.Wallets.peewee.UserWallet import UserWallet
from kaze.Core.Blockchain import Blockchain
from kazecore.UInt160 import UInt160
from kazecore.Fixed8 import Fixed8
from kaze.Prompt.Commands.Wallet import CreateAddress, DeleteAddress, ImportToken, ImportWatchAddr, ShowUnspentCoins, SplitUnspentCoin
import shutil


class UserWalletTestCase(WalletFixtureTestCase):

    wallet_1_script_hash = UInt160(data=b'S\xefB\xc8\xdf!^\xbeZ|z\xe8\x01\xcb\xc3\xac/\xacI)')

    wallet_1_addr = 'APRgMZHZubii29UXF9uFa6sohrsYupNAvx'

    import_watch_addr = UInt160(data=b'\xaf\x12\xa8h{\x14\x94\x8b\xc4\xa0\x08\x12\x8aU\nci[\xc1\xa5')
    watch_addr_str = 'AXjaFSP23Jkbe6Pk9pPGT6NBDs1HVdqaXK'
    _wallet1 = None

    @property
    def stream(self):
        return Blockchain.Default().SystemCoin().Hash

    @property
    def kaze(self):
        return Blockchain.Default().SystemShare().Hash

    @classmethod
    def GetWallet1(cls, recreate=False):
        if cls._wallet1 is None or recreate:
            shutil.copyfile(cls.wallet_1_path(), cls.wallet_1_dest())
            cls._wallet1 = UserWallet.Open(UserWalletTestCase.wallet_1_dest(),
                                           to_aes_key(UserWalletTestCase.wallet_1_pass()))
        return cls._wallet1

    def test_1_import_addr(self):

        wallet = self.GetWallet1()

        self.assertEqual(len(wallet.LoadWatchOnly()), 0)

        result = ImportWatchAddr(wallet, self.watch_addr_str)

        self.assertEqual(len(wallet.LoadWatchOnly()), 1)

    def test_2_import_addr(self):

        wallet = self.GetWallet1()

        self.assertEqual(len(wallet.LoadWatchOnly()), 1)

        success = DeleteAddress(None, wallet, self.watch_addr_str)

        self.assertTrue(success)

        self.assertEqual(len(wallet.LoadWatchOnly()), 0)

    def test_3_import_token(self):

        wallet = self.GetWallet1()

        self.assertEqual(len(wallet.GetTokens()), 0)

        token_hash = 'f8d448b227991cf07cb96a6f9c0322437f1599b9'

        ImportToken(wallet, token_hash)

        token = list(wallet.GetTokens().values())[0]

        self.assertEqual(token.name, 'NEP5 Standard')
        self.assertEqual(token.symbol, 'NEP5')
        self.assertEqual(token.decimals, 8)
        self.assertEqual(token.Address, 'AYhE3Svuqdfh1RtzvE8hUhNR7HSpaSDFQg')

    def test_4_get_synced_balances(self):
        wallet = self.GetWallet1()
        synced_balances = wallet.GetSyncedBalances()
        self.assertEqual(len(synced_balances), 2)

    def test_5_show_unspent(self):

        wallet = self.GetWallet1(True)
        unspents = ShowUnspentCoins(wallet, [])
        self.assertEqual(len(unspents), 2)

        unspents = ShowUnspentCoins(wallet, ['kaze'])
        self.assertEqual(len(unspents), 1)

        unspents = ShowUnspentCoins(wallet, ['stream'])
        self.assertEqual(len(unspents), 1)

        unspents = ShowUnspentCoins(wallet, ['APRgMZHZubii29UXF9uFa6sohrsYupNAvx'])
        self.assertEqual(len(unspents), 2)

        unspents = ShowUnspentCoins(wallet, ['AYhE3Svuqdfh1RtzvE8hUhNR7HSpaSDFQg'])
        self.assertEqual(len(unspents), 0)

        unspents = ShowUnspentCoins(wallet, ['--watch'])
        self.assertEqual(len(unspents), 0)

    def test_6_split_unspent(self):

        wallet = self.GetWallet1(True)

        # test bad
        tx = SplitUnspentCoin(wallet, [])
        self.assertEqual(tx, None)

        # bad inputs
        tx = SplitUnspentCoin(wallet, ['APRgMZHZubii29UXF9uFa6sohrsYupNAvx', 'kaze', 3, 2])
        self.assertEqual(tx, None)

        # should be ok
        tx = SplitUnspentCoin(wallet, ['APRgMZHZubii29UXF9uFa6sohrsYupNAvx', 'kaze', 0, 2], prompt_passwd=False)
        self.assertIsNotNone(tx)

        # rebuild wallet and try with non-even amount of kaze, should be split into integer values of kaze
        wallet = self.GetWallet1(True)
        tx = SplitUnspentCoin(wallet, ['APRgMZHZubii29UXF9uFa6sohrsYupNAvx', 'kaze', 0, 3], prompt_passwd=False)
        self.assertIsNotNone(tx)

        self.assertEqual([Fixed8.FromDecimal(34), Fixed8.FromDecimal(34), Fixed8.FromDecimal(32)], [item.Value for item in tx.outputs])

        # try with stream
        wallet = self.GetWallet1(True)
        tx = SplitUnspentCoin(wallet, ['APRgMZHZubii29UXF9uFa6sohrsYupNAvx', 'stream', 0, 3], prompt_passwd=False)
        self.assertIsNotNone(tx)

    def test_7_create_address(self):

        wallet = self.GetWallet1(True)

        # not specifying a number of addresses
        CreateAddress(None, wallet, None)
        self.assertEqual(len(wallet.Addresses), 1)

        # trying to create too many addresses
        CreateAddress(None, wallet, 5)
        self.assertEqual(len(wallet.Addresses), 1)

        # should pass
        success = CreateAddress(None, wallet, 1)
        self.assertTrue(success)

        # check the number of addresses
        self.assertEqual(len(wallet.Addresses), 2)
