from kdb_python.common.constants import InfoMessage
from kdb_python.config.settings import DriverSettings
from kdb_python.webdriver import actions
from kdb_python.webdriver.actions import video
from kdb_python.webdriver.common import log_start, report_passed_test_step, report_failed_test_step


class Video:
    """
    Class for action video
    """

    def __init__(self, driver):
        self._driver = driver

    def __new__(cls, driver):
        """
        Classic singleton in Python, we check whether an instance is already created.
        If it is created, we return it; otherwise, we create a new instance, assign it to a class attribute,
        and return it.
        """
        if not hasattr(cls, 'instance'):
            cls.instance = super(Video, cls).__new__(cls)
        else:
            cls.instance._driver = driver
        return cls.instance

    def play(self, locator, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT, extra_time=None,
             log=True):
        """
        Play a video
        """
        args_passed = locals()
        start_time = log_start(self.play.__name__, args_passed, log)
        try:
            actions.perform(video.play, self._driver, locator, timeout, in_frame_on_ios, extra_time)
            report_passed_test_step(self.play.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Play video")
        except Exception as ex:
            report_failed_test_step(self._driver, self.play.__name__, args_passed, start_time, str(ex))
        return self

    def pause(self, locator, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT, extra_time=None,
              log=True):
        """
        Pause a video
        """
        args_passed = locals()
        start_time = log_start(self.pause.__name__, args_passed, log)
        try:
            actions.perform(video.pause, self._driver, locator, timeout, in_frame_on_ios, extra_time)
            report_passed_test_step(self.pause.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Pause video")
        except Exception as ex:
            report_failed_test_step(self._driver, self.pause.__name__, args_passed, start_time, str(ex))
        return self

    def muted(self, locator, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT, extra_time=None,
              log=True):
        """
        Muted a video
        """
        args_passed = locals()
        start_time = log_start(self.muted.__name__, args_passed, log)
        try:
            actions.perform(video.muted, self._driver, locator, timeout, in_frame_on_ios, extra_time)
            report_passed_test_step(self.muted.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Muted video")
        except Exception as ex:
            report_failed_test_step(self._driver, self.muted.__name__, args_passed, start_time, str(ex))
        return self

    def unmuted(self, locator, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT, extra_time=None,
                log=True):
        """
        Unmuted a video
        """
        args_passed = locals()
        start_time = log_start(self.unmuted.__name__, args_passed, log)
        try:
            actions.perform(video.unmuted, self._driver, locator, timeout, in_frame_on_ios, extra_time)
            report_passed_test_step(self.unmuted.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Unmuted video")
        except Exception as ex:
            report_failed_test_step(self._driver, self.unmuted.__name__, args_passed, start_time, str(ex))
        return self

    def volume(self, locator, value, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT, extra_time=None, log=True):
        """
        Change volume for video
        """
        args_passed = locals()
        start_time = log_start(self.volume.__name__, args_passed, log)
        try:
            actions.perform(video.volume, self._driver, locator, timeout, False, value, extra_time)
            report_passed_test_step(self.volume.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Volume video " + str(value))
        except Exception as ex:
            report_failed_test_step(self._driver, self.volume.__name__, args_passed, start_time, str(ex))
        return self

    def track_time(self, locator, value, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                   extra_time=None, log=True):
        """
        Track time video
        """
        args_passed = locals()
        start_time = log_start(self.track_time.__name__, args_passed, log)
        try:
            actions.perform(video.track_time, self._driver, locator, timeout, in_frame_on_ios, value, extra_time)
            report_passed_test_step(self.track_time.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % ("Track time video " + str(value)))
        except Exception as ex:
            report_failed_test_step(self._driver, self.track_time.__name__, args_passed, start_time, str(ex))
        return self

    def is_playing(self, locator, reverse=False, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                   extra_time=None, log=True):
        """
         Verify video is playing
        """
        args_passed = locals()
        start_time = log_start(self.is_playing.__name__, args_passed, log)
        try:
            assert actions.perform(video.is_playing, self._driver, locator, timeout, in_frame_on_ios, reverse,
                                   extra_time)
            report_passed_test_step(self.is_playing.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Verify video is playing ")
            return True
        except AssertionError:
            report_failed_test_step(self._driver, self.is_playing.__name__, args_passed, start_time,
                                    "The video is not playing: " + locator)
        except Exception as ex:
            report_failed_test_step(self._driver, self.is_playing.__name__, args_passed, start_time, str(ex))

    def is_paused(self, locator, reverse=False, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                  extra_time=None, log=True):

        """
        Verify video was paused
        """
        args_passed = locals()
        start_time = log_start(self.is_paused.__name__, args_passed, log)
        try:
            assert actions.perform(video.is_paused, self._driver, locator, timeout, in_frame_on_ios, reverse,
                                   extra_time)
            report_passed_test_step(self.is_paused.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Verify video paused ")
            return True
        except AssertionError:
            report_failed_test_step(self._driver, self.is_paused.__name__, args_passed, start_time,
                                    "The video is not pause: " + locator)
        except Exception as ex:
            report_failed_test_step(self._driver, self.is_paused.__name__, args_passed, start_time, str(ex))

    def is_muted(self, locator, reverse=False, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                 extra_time=None, log=True):
        """
        Verify video was muted
        """
        args_passed = locals()
        start_time = log_start(self.is_muted.__name__, args_passed, log)
        try:
            assert actions.perform(video.is_muted, self._driver, locator, timeout, in_frame_on_ios, reverse, extra_time)
            report_passed_test_step(self.is_muted.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Verify video muted ")
            return True
        except AssertionError:
            report_failed_test_step(self._driver, self.is_muted.__name__, args_passed, start_time,
                                    "The video is not muted: " + locator)
        except Exception as ex:
            report_failed_test_step(self._driver, self.is_muted.__name__, args_passed, start_time, str(ex))

    def is_unmuted(self, locator, reverse=False, in_frame_on_ios=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                   extra_time=None, log=True):
        """
        Verify video was unmuted
        """
        args_passed = locals()
        start_time = log_start(self.is_unmuted.__name__, args_passed, log)
        try:
            assert actions.perform(video.is_unmuted, self._driver, locator, timeout, in_frame_on_ios, reverse,
                                   extra_time)
            report_passed_test_step(self.is_unmuted.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % "Verify video unmuted ")
            return True
        except AssertionError:
            report_failed_test_step(self._driver, self.is_unmuted.__name__, args_passed, start_time,
                                    "The video is not unmuted: " + locator)
        except Exception as ex:
            report_failed_test_step(self._driver, self.is_unmuted.__name__, args_passed, start_time, str(ex))

    def verify_volume(self, locator, value, reverse=False, timeout=DriverSettings.DRIVER_IMPLICITLY_WAIT,
                      extra_time=None, log=True):
        """
        Verify volume of video
        """
        args_passed = locals()
        start_time = log_start(self.verify_volume.__name__, args_passed, log)
        try:
            assert actions.perform(video.verify_volume, self._driver, locator, timeout, False, value, reverse,
                                   extra_time)
            report_passed_test_step(self.verify_volume.__name__, args_passed, start_time,
                                    InfoMessage.ACTION_SUCCESS % ("Verify volume is " + str(value)))
        except AssertionError:
            report_failed_test_step(self._driver, self.verify_volume.__name__, args_passed, start_time,
                                    "The video is not match with volume = %s. %s" % (value, (locator, value, reverse)))
        except Exception as ex:
            report_failed_test_step(self._driver, self.verify_volume.__name__, args_passed, start_time, str(ex))
        return self
