"""Shared pytest fixtures for CLI and module-entry tests."""

from __future__ import annotations

import re
from collections.abc import Callable, Iterator
from dataclasses import dataclass, fields

import pytest
from click.testing import CliRunner

import lib_cli_exit_tools

ANSI_ESCAPE_PATTERN = re.compile(r"\x1B\[[0-?]*[ -/]*[@-~]")
CONFIG_FIELDS: tuple[str, ...] = tuple(field.name for field in fields(type(lib_cli_exit_tools.config)))


@dataclass(frozen=True, slots=True)
class CliConfigSnapshot:
    """Typed snapshot of lib_cli_exit_tools configuration state."""

    traceback: bool
    traceback_force_color: bool

    @classmethod
    def capture(cls) -> "CliConfigSnapshot":
        """Capture current configuration state."""
        return cls(
            traceback=bool(getattr(lib_cli_exit_tools.config, "traceback", False)),
            traceback_force_color=bool(getattr(lib_cli_exit_tools.config, "traceback_force_color", False)),
        )

    def restore(self) -> None:
        """Reapply this configuration state."""
        lib_cli_exit_tools.config.traceback = self.traceback
        lib_cli_exit_tools.config.traceback_force_color = self.traceback_force_color


def _remove_ansi_codes(text: str) -> str:
    """Return ``text`` stripped of ANSI escape sequences.

    Why
        Tests compare human-readable CLI output; stripping colour codes keeps
        assertions stable across environments.

    Inputs
        text:
            Raw string captured from CLI output.

    Outputs
        str: The string without ANSI escape sequences.
    """

    return ANSI_ESCAPE_PATTERN.sub("", text)


def _snapshot_cli_config() -> CliConfigSnapshot:
    """Capture every attribute from ``lib_cli_exit_tools.config``.

    Why
        Tests toggle traceback behaviour; keeping a snapshot lets fixtures
        restore the configuration after each run.

    Outputs
        CliConfigSnapshot: Dataclass containing captured configuration values.
    """

    return CliConfigSnapshot.capture()


def _restore_cli_config(snapshot: CliConfigSnapshot) -> None:
    """Reapply the previously captured CLI configuration.

    Why
        Ensures global state looks untouched to subsequent tests.

    Inputs
        snapshot:
            Dataclass generated by :func:`_snapshot_cli_config`.
    """

    snapshot.restore()


@pytest.fixture
def cli_runner() -> CliRunner:
    """Provide a fresh :class:`CliRunner` per test."""

    return CliRunner()


@pytest.fixture
def strip_ansi() -> Callable[[str], str]:
    """Return a helper that strips ANSI escape sequences from a string."""

    def _strip(value: str) -> str:
        return _remove_ansi_codes(value)

    return _strip


@pytest.fixture
def preserve_traceback_state() -> Iterator[None]:
    """Snapshot and restore the entire ``lib_cli_exit_tools`` configuration."""

    snapshot = _snapshot_cli_config()
    try:
        yield
    finally:
        _restore_cli_config(snapshot)


@pytest.fixture
def isolated_traceback_config(monkeypatch: pytest.MonkeyPatch) -> None:
    """Reset traceback flags to a known baseline before each test."""

    lib_cli_exit_tools.reset_config()
    monkeypatch.setattr(lib_cli_exit_tools.config, "traceback", False, raising=False)
    monkeypatch.setattr(lib_cli_exit_tools.config, "traceback_force_color", False, raising=False)
