# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ki2_python_utils']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=4.12.2,<5.0.0']

setup_kwargs = {
    'name': 'ki2-python-utils',
    'version': '0.3.0',
    'description': '',
    'long_description': '# ki2 - Python Utility Elements\n\n## Installation\n\nTo install using pip:\n\n```\npip install ki2-python-utils\n```\n\nOr, if you\'re using Poetry:\n\n```\npoetry add ki2-python-utils\n```\n\n## Usage\n\n### Lists\n\n#### `UniqueList`\n\nA list that cannot contain duplicate elements.\n\n```python\nfrom ki2_python_utils import UniqueList\n\nx = UniqueList()\nx.append(1)\nx.append(2)\nx.append(1)\nprint(x)  # [1, 2]\n```\n\n`UniqueList` can be typed to enforce the type of its elements:\n\n```python\nx = UniqueList[int]()  # Restrict elements to type `int`\n```\n\n#### `CallbackList`\n\nA list of callback functions that can be invoked all at once.\n\n```python\nfrom ki2_python_utils import CallbackList\n\ndef cb1():\n    print("cb1")\n\ndef cb2():\n    print("cb2")\n\nx = CallbackList()\nx.append(cb1)\nx.append(cb2)\nx()  # Prints "cb1" then "cb2"\n```\n\nCallbacks are called sequentially, and the order of addition matters:\n\n```python\nx = CallbackList()\nx.append(cb2)\nx.append(cb1)\nx()  # Prints "cb2" then "cb1"\n```\n\nBy default, the `CallbackList` type assumes callbacks take no parameters. However, you can specify the parameter types for the callbacks in the list.\n\nExample with a single parameter:\n\n```python\ndef cb1(x: int):\n    print(f"cb1({x})")\n\ndef cb2(x: int):\n    print(f"cb2({x})")\n\nx = CallbackList[int]()\nx.append(cb1)\nx.append(cb2)\nx(1)  # Prints "cb1(1)" then "cb2(1)"\n```\n\nExample with two parameters:\n\n```python\ndef cb1(x: int, y: str):\n    print(f"cb1({x}, {y})")\n\ndef cb2(x: int, y: str):\n    print(f"cb2({x}, {y})")\n\nx = CallbackList[int, str]()\nx.append(cb1)\nx.append(cb2)\nx(1, "test")  # Prints "cb1(1, test)" then "cb2(1, test)"\n```\n\nIf the same callback function is added multiple times, only the first occurrence is kept. Subsequent additions are ignored.\n\nThe `CallbackList` class is an alias for `UniqueCallbackList`, which uses `UniqueList` to store callbacks and ensures that the same callback cannot be added more than once. If you need a callback list that allows duplicates, you can use `MultipleCallbackList` instead.\n\nThe `UniqueCallbackList` (or `CallbackList`) and `MultipleCallbackList` classes work with synchronous functions. If you need to work with asynchronous functions, you can use `AsyncUniqueCallbackList` (or `AsyncCallbackList`) and `AsyncMultipleCallbackList` instead.\n\n### JSON\n\nThe `Json` submodule provides utility functions for working with JSON data. It includes specific type definitions for JSON objects and functions to validate their types.\n\n- **`Number` Type**: Represents a JSON number, as defined in JavaScript, which can be either an `int` or a `float`. The `is_number` validator checks if a value is a valid JSON number. Unlike Python\'s behavior, `is_number` returns `False` for boolean values.\n\n- **`Json` Type**: Allows typing or casting a value as a JSON object. The `is_json` validator can be used to check whether a value is a valid JSON object.\n\nThis submodule helps ensure consistency when handling JSON data in Python, aligning behavior more closely with JSON standards.\n\n### Exist\n\nThe `exist` submodule provides utilities to validate whether an optional object exists or not—specifically, whether it is `None`.\n\n#### `exist`\n\nThe `exist` function takes an optional object as input and returns `True` if the object is not `None`, and `False` otherwise. This function supports type-checking and serves as a compact alternative to the test `x is not None`.\n\nExample usage:\n\n```python\nfrom ki2_python_utils import exist\n\n# Assume x is of type int | None\n\nif exist(x):\n    ...  # Here, x is of type int\nelse:\n    ...  # Here, x is of type None\n```\n\n#### `count_exist`\n\nCounts the number of elements in a list that are not `None`.\n\n#### `count_none`\n\nCounts the number of elements in a list that are `None`.\n\n#### `exist_all`\n\nEnsures that all elements in a list are not `None`. This function also supports type narrowing:\n\n```python\nfrom ki2_python_utils import exist_all\n\n# Assume x is of type list[int | None]\n\nif exist_all(x):\n    ...  # Here, x is of type list[int]\nelse:\n    ...  # Here, x is of type list[int | None]\n```\n\nIf `exist_all` is called with an empty list, it returns `True`.\n\n#### `exist_some`\n\nChecks whether at least one element in a list is not `None`. If the function is called with an empty list, it also returns `True`.\n\n### Filter\n\nThe `filter` submodule provides utilities for filtering elements in lists or dictionaries.\n\n#### `filter_exist`\n\nRemoves `None` values from a list or keys with `None` values from dictionaries.\n\nExample:\n\n```python\nfrom ki2_python_utils import filter_exist\n\nx = [1, None, 2, None, 3]\ny = filter_exist(x)  # y = [1, 2, 3]\n```\n\n```python\nfrom ki2_python_utils import filter_exist\n\nx = {"a": 1, "b": None, "c": 2}\ny = filter_exist(x)  # y = {"a": 1, "c": 2}\n```\n\n#### `first_exist`\n\nReturns the first non-`None` element from a list.\n\nExample:\n\n```python\nfrom ki2_python_utils import first_exist\n\nx = [None, None, 2, None, 3]\ny = first_exist(x)  # y = 2\n```\n\n#### `last_exist`\n\nReturns the last non-`None` element from a list.\n\nExample:\n\n```python\nfrom ki2_python_utils import last_exist\n\nx = [2, None, 3, None, None]\ny = last_exist(x)  # y = 3\n```\n\n### Async Utils\n\nThe `async_utils` submodule offers utilities to simplify working with asynchronous functions.\n\n#### `apply_parallel`\n\nExecutes an asynchronous function on a list of arguments in parallel using `asyncio.gather`.\n\n**Example**:\n\n```python\nfrom ki2_python_utils import apply_parallel\n\nasync def add_one(x: int) -> int:\n    return x + 1\n\nx = [1, 2, 3]\ny = await apply_parallel(add_one, x)  # y = [2, 3, 4]\n```\n\n#### `run_parallel`\n\nExecutes a list of coroutines concurrently using `asyncio.gather`. This function is ideal for running multiple asynchronous tasks that do not require arguments.\n\n**Example**:\n\n```python\nfrom ki2_python_utils import run_parallel\n\nasync def func_1():\n    print("func_1")\n\nasync def func_2():\n    print("func_2")\n\nawait run_parallel(func_1, func_2)\n```\n\n### FlowBuffer\n\nThe `FlowBuffer` submodule provides a `FlowBuffer` class designed to manage a sliding window buffer where elements are continuously added.\n\n#### `FlowBuffer`\n\nThe `FlowBuffer` class accepts an integer parameter `max_length`, which defines the maximum size of the buffer.\n\nIt offers the following methods:\n\n- **`append(item)`**: Adds an element to the buffer. If the buffer is full, the oldest element is removed to make room for the new one.\n- **`extend(items)`**: Adds multiple elements to the buffer. If the total number of elements exceeds the buffer\'s maximum size, only the most recent elements are retained.\n- **`clear()`**: Empties the buffer.\n- **`get(index)`**: Retrieves the element at the specified index in reverse order. Index 0 corresponds to the most recently added element, index 1 to the second most recent, and so on.\n- **`get_raw(index)`**: Retrieves the element at the specified index in insertion order. Index 0 corresponds to the first element added, index 1 to the second element, and so forth.\n\nIt also provides the following properties:\n\n- **`is_full`** (`bool`): Indicates whether the buffer is completely filled.\n- **`last_item`**: Returns the most recently added element in the buffer.\n\n#### `IndexedFlowBuffer`\n\nUnlike `FlowBuffer`, which retrieves elements in reverse order, `IndexedFlowBuffer` assigns an incrementing index to each element as it is added. Elements are accessed using this index.\n\n**Example**:\n\n```python\nfrom ki2_python_utils import IndexedFlowBuffer\n\nbuffer = IndexedFlowBuffer(max_length=3)\nbuffer.append(1)\nbuffer.append(2)\nbuffer.append(3)\nprint(buffer.get(0))  # Output: 1\nprint(buffer.get(1))  # Output: 2\nprint(buffer.get(2))  # Output: 3\nbuffer.append(4)\n# buffer.get(0) is no longer accessible\nprint(buffer.get(1))  # Output: 2\nprint(buffer.get(2))  # Output: 3\nprint(buffer.get(3))  # Output: 4\n```\n',
    'author': 'Adrien KERFOURN',
    'author_email': 'ak.sitecontact@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
