"""
SAM (Serverless Application Model) template generator
"""

import yaml
import os
from typing import Dict, List, Any
from pathlib import Path


class SAMTemplateGenerator:
    """Generates SAM templates from analysis results"""

    def __init__(self, config: Dict):
        """Initialize with configuration"""
        self.config = config
        self.template_config = config.get('template', {})

    def generate(self, analysis_results: List) -> str:
        """
        Generate SAM template from analysis results

        Args:
            analysis_results: List of AnalysisResult objects

        Returns:
            SAM template as YAML string
        """
        template = {
            'AWSTemplateFormatVersion': '2010-09-09',
            'Transform': 'AWS::Serverless-2016-10-31',
            'Description': 'SAM Template generated by Lambda Analyzer',
            'Parameters': {},
            'Globals': self._generate_globals(),
            'Resources': {},
            'Outputs': {}
        }

        # Collect all environment variables and services
        all_env_vars = set()
        all_services = set()

        for result in analysis_results:
            all_env_vars.update(result.environment_variables)
            all_services.update(result.services)

        # Generate parameters
        template['Parameters'] = self._generate_parameters(all_env_vars)

        # Generate Lambda functions
        for i, result in enumerate(analysis_results):
            function_resources = self._generate_function_resources(result, i, len(analysis_results))
            template['Resources'].update(function_resources)

        # Generate supporting AWS resources
        support_resources = self._generate_support_resources(all_services)
        template['Resources'].update(support_resources)

        # Generate outputs
        template['Outputs'] = self._generate_outputs(analysis_results, all_services)

        return yaml.dump(template, default_flow_style=False, sort_keys=False)

    def _generate_globals(self) -> Dict[str, Any]:
        """Generate global SAM configuration"""
        return {
            'Function': {
                'Runtime': self.template_config.get('runtime', 'python3.9'),
                'MemorySize': self.template_config.get('memory', 512),
                'Timeout': self.template_config.get('timeout', 30),
                'Environment': {
                    'Variables': {
                        'POWERTOOLS_SERVICE_NAME': 'lambda-analyzer-app'
                    }
                }
            }
        }

    def _generate_parameters(self, env_vars: set) -> Dict[str, Any]:
        """Generate CloudFormation parameters"""
        parameters = {}

        for env_var in sorted(env_vars):
            parameters[f"{env_var}Parameter"] = {
                'Type': 'String',
                'Description': f'Value for {env_var} environment variable'
            }

        # Add standard parameters
        parameters['Environment'] = {
            'Type': 'String',
            'Default': 'dev',
            'AllowedValues': ['dev', 'staging', 'prod'],
            'Description': 'Deployment environment'
        }

        return parameters

    def _generate_function_resources(self, result, index: int, total_count: int) -> Dict[str, Any]:
        """Generate Lambda function resources"""
        resources = {}

        # Generate function name
        if total_count > 1:
            function_name = f"Function{index + 1}"
            logical_name = function_name
        else:
            function_name = "Function"
            logical_name = "LambdaFunction"

        # Extract handler name from file path
        file_stem = Path(result.file_path).stem
        handler = f"{file_stem}.lambda_handler"

        # Base function configuration
        function_config = {
            'Type': 'AWS::Serverless::Function',
            'Properties': {
                'CodeUri': '.',
                'Handler': handler,
                'Environment': {
                    'Variables': {}
                },
                'Events': {},
                'Policies': []
            }
        }

        # Add environment variables
        for env_var in result.environment_variables:
            function_config['Properties']['Environment']['Variables'][env_var] = f"!Ref {env_var}Parameter"

        # Add IAM policies based on detected services
        policies = self._generate_sam_policies(result)
        if policies:
            function_config['Properties']['Policies'].extend(policies)

        # Add event sources based on detected triggers
        events = self._generate_events(result)
        if events:
            function_config['Properties']['Events'].update(events)

        # Add function description
        services_list = ', '.join(result.services) if result.services else 'None'
        function_config['Properties']['Description'] = f"Generated function using: {services_list}"

        resources[logical_name] = function_config

        return resources

    def _generate_sam_policies(self, result) -> List[Dict[str, Any]]:
        """Generate SAM-style IAM policies"""
        policies = []

        for service in result.services:
            if service == 's3':
                policies.extend([
                    {'S3ReadPolicy': {'BucketName': '!Ref S3Bucket'}},
                    {'S3WritePolicy': {'BucketName': '!Ref S3Bucket'}}
                ])
            elif service == 'dynamodb':
                policies.append({
                    'DynamoDBCrudPolicy': {'TableName': '!Ref DynamoDBTable'}
                })
            elif service == 'sns':
                policies.append({
                    'SNSPublishMessagePolicy': {'TopicName': '!GetAtt SNSTopic.TopicName'}
                })
            elif service == 'sqs':
                policies.extend([
                    {'SQSPollerPolicy': {'QueueName': '!GetAtt SQSQueue.QueueName'}},
                    {'SQSSendMessagePolicy': {'QueueName': '!GetAtt SQSQueue.QueueName'}}
                ])
            elif service == 'secretsmanager':
                policies.append({
                    'SecretsManagerReadWrite': {}
                })
            elif service == 'ssm':
                policies.append({
                    'SSMParameterReadPolicy': {'ParameterName': '*'}
                })

        return policies

    def _generate_events(self, result) -> Dict[str, Any]:
        """Generate event sources for the function"""
        events = {}

        for i, trigger in enumerate(result.triggers):
            trigger_type = trigger['type']
            event_name = f"{trigger_type.title()}Event"

            if trigger_type == 's3':
                events[event_name] = {
                    'Type': 'S3',
                    'Properties': {
                        'Bucket': '!Ref S3Bucket',
                        'Event': 's3:ObjectCreated:*'
                    }
                }
            elif trigger_type == 'api_gateway':
                events[event_name] = {
                    'Type': 'Api',
                    'Properties': {
                        'Path': '/api',
                        'Method': 'post',
                        'RestApiId': '!Ref ApiGateway'
                    }
                }
            elif trigger_type == 'sqs':
                events[event_name] = {
                    'Type': 'SQS',
                    'Properties': {
                        'Queue': '!GetAtt SQSQueue.Arn',
                        'BatchSize': 10
                    }
                }
            elif trigger_type == 'sns':
                events[event_name] = {
                    'Type': 'SNS',
                    'Properties': {
                        'Topic': '!Ref SNSTopic'
                    }
                }
            elif trigger_type == 'dynamodb':
                events[event_name] = {
                    'Type': 'DynamoDB',
                    'Properties': {
                        'Stream': '!GetAtt DynamoDBTable.StreamArn',
                        'StartingPosition': 'TRIM_HORIZON'
                    }
                }
            elif trigger_type == 'scheduled':
                events[event_name] = {
                    'Type': 'Schedule',
                    'Properties': {
                        'Schedule': 'rate(5 minutes)',
                        'Description': 'Scheduled trigger'
                    }
                }

        return events

    def _generate_support_resources(self, services: set) -> Dict[str, Any]:
        """Generate supporting AWS resources"""
        resources = {}

        if 's3' in services:
            resources['S3Bucket'] = {
                'Type': 'AWS::S3::Bucket',
                'Properties': {
                    'BucketName': '!Sub ${AWS::StackName}-lambda-analyzer-bucket',
                    'BucketEncryption': {
                        'ServerSideEncryptionConfiguration': [{
                            'ServerSideEncryptionByDefault': {
                                'SSEAlgorithm': 'AES256'
                            }
                        }]
                    },
                    'PublicAccessBlockConfiguration': {
                        'BlockPublicAcls': True,
                        'BlockPublicPolicy': True,
                        'IgnorePublicAcls': True,
                        'RestrictPublicBuckets': True
                    },
                    'NotificationConfiguration': {
                        'LambdaConfigurations': []
                    }
                }
            }

        if 'dynamodb' in services:
            resources['DynamoDBTable'] = {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'TableName': '!Sub ${AWS::StackName}-lambda-analyzer-table',
                    'BillingMode': 'PAY_PER_REQUEST',
                    'AttributeDefinitions': [{
                        'AttributeName': 'id',
                        'AttributeType': 'S'
                    }],
                    'KeySchema': [{
                        'AttributeName': 'id',
                        'KeyType': 'HASH'
                    }],
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_AND_OLD_IMAGES'
                    },
                    'SSESpecification': {
                        'SSEEnabled': True
                    }
                }
            }

        if 'sns' in services:
            resources['SNSTopic'] = {
                'Type': 'AWS::SNS::Topic',
                'Properties': {
                    'TopicName': '!Sub ${AWS::StackName}-lambda-analyzer-topic',
                    'KmsMasterKeyId': 'alias/aws/sns'
                }
            }

        if 'sqs' in services:
            resources['SQSQueue'] = {
                'Type': 'AWS::SQS::Queue',
                'Properties': {
                    'QueueName': '!Sub ${AWS::StackName}-lambda-analyzer-queue',
                    'KmsMasterKeyId': 'alias/aws/sqs',
                    'MessageRetentionPeriod': 1209600,  # 14 days
                    'VisibilityTimeoutSeconds': 180
                }
            }

            resources['SQSDeadLetterQueue'] = {
                'Type': 'AWS::SQS::Queue',
                'Properties': {
                    'QueueName': '!Sub ${AWS::StackName}-lambda-analyzer-dlq',
                    'KmsMasterKeyId': 'alias/aws/sqs'
                }
            }

        return resources

    def _generate_outputs(self, results: List, services: set) -> Dict[str, Any]:
        """Generate CloudFormation outputs"""
        outputs = {}

        # Function outputs
        for i, result in enumerate(results):
            if len(results) > 1:
                func_name = f"Function{i + 1}"
            else:
                func_name = "LambdaFunction"

            outputs[f"{func_name}Arn"] = {
                'Description': f'ARN of the {func_name}',
                'Value': {'Fn::GetAtt': [func_name, 'Arn']},
                'Export': {
                    'Name': {'Fn::Sub': f"${{{{AWS::StackName}}}}-{func_name}-Arn"}
                }
            }

        # Resource outputs
        if 's3' in services:
            outputs['S3BucketName'] = {
                'Description': 'Name of the S3 bucket',
                'Value': {'Ref': 'S3Bucket'},
                'Export': {
                    'Name': {'Fn::Sub': '${AWS::StackName}-S3Bucket'}
                }
            }

        if 'dynamodb' in services:
            outputs['DynamoDBTableName'] = {
                'Description': 'Name of the DynamoDB table',
                'Value': {'Ref': 'DynamoDBTable'},
                'Export': {
                    'Name': {'Fn::Sub': '${AWS::StackName}-DynamoDBTable'}
                }
            }

        if 'sns' in services:
            outputs['SNSTopicArn'] = {
                'Description': 'ARN of the SNS topic',
                'Value': {'Ref': 'SNSTopic'},
                'Export': {
                    'Name': {'Fn::Sub': '${AWS::StackName}-SNSTopic'}
                }
            }

        return outputs