# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lambda_cache']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'lambda-cache',
    'version': '0.7.0',
    'description': 'Python utility for simple caching in AWS Lambda functions',
    'long_description': '<h1 align="center"> simple_lambda_cache </h2>\n<h2 align="center"> Simple Caching for AWS Lambda</h2>\n\n![PackageStatus](https://img.shields.io/static/v1?label=status&message=beta&color=red?style=flat-square) \n![PythonSupport](https://img.shields.io/static/v1?label=python&message=3.6%20|%203.7|%203.8&color=blue?style=flat-square&logo=python)\n\n![Build](https://github.com/keithrozario/simple_lambda_cache/workflows/Build/badge.svg?branch=release) [![Coverage Status](https://coveralls.io/repos/github/keithrozario/simple_lambda_cache/badge.svg?branch=release)](https://coveralls.io/github/keithrozario/simple_lambda_cache?branch=release) [![Documentation Status](https://readthedocs.org/projects/simple-lambda-cache/badge/?version=latest)](https://simple-lambda-cache.readthedocs.io/en/latest/?badge=latest)\n\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/keithrozario/simple_lambda_cache.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/keithrozario/simple_lambda_cache/context:python) [![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n# Basics\n\n_simple_lambda_cache_ prioritizes simplicity over performance and flexibility. The goal of the package is to provide the **simplest** way for developers to cache api calls in their Lambda functions.\n\nCurrently only SSM Parameters and Secrets from Secrets Manager are supported.\n\n## Cache single parameter\n\nTo cache a parameter from ssm, decorate your handler function:\n\n```python\nfrom simple_lambda_cache import ssm_cache\n\n@ssm_cache(parameter=\'/production/app/var\')\ndef handler(event, context):\n    var = context.get(\'var\')\n    response = do_something(var)\n    return response\n```\nAll invocations of this function over in the next minute will reference the parameter from the function\'s internal cache, rather than making a network call to ssm. After one minute, the the next invocation will invoke `get_parameter` to refresh the cache.\n\n## Change cache settings\n\nThe default `ttl_seconds` settings is 60 seconds (1 minute), it defines how long a parameter should be kept in cache before it is refreshed from ssm. To configure longer or shorter times, modify this argument like so:\n\n```python\nfrom simple_lambda_cache import ssm_cache\n\n@ssm_cache(parameter=\'/production/app/var\', ttl_seconds=300)\ndef handler(event, context):\n    var = context.get(\'var\')\n    response = do_something(var)\n    return response\n```\n\n_Note: The caching logic runs only at invocation, regardless of how long the function runs. A 15 minute lambda function will not refresh the parameter, unless explicitly refreshed using get_cache_ssm. The library is primary interested in caching \'across\' invocation rather than \'within\' an invocation_\n\n## Change cache entry settings\n\nThe name of the parameter is simply shortened to the string after the last slash(\'/\') character of its name. This means `/production/app/var` and `test/app/var` resolve to just `var`. To over-ride this default, use `entry_name`:\n\n```python\nfrom simple_lambda_cache import ssm_cache\n\n@ssm_cache(parameter=\'/production/app/var\', entry_name=\'new_var\')\ndef handler(event, context):\n    var = context.get(\'new_var\')\n    response = do_something(var)\n    return response\n```\n\n## Cache multiple parameters\n\nTo cache multiple entries at once, pass a list of parameters to the parameter argument, and grab the parameters from `context[\'parameters\']`.\n\n```python\nfrom simple_lambda_cache import ssm_cache\n\n@ssm_cache(parameter=[\'/app/var1\', \'/app/var2\'], entry_name=\'parameters\')\ndef handler(event, context):\n    var1 = context.get(\'parameters\').get(\'var1\')\n    var2 = context.get(\'parameters\').get(\'var2\')\n    response = do_something(var)\n    return response\n```\n\nUnder the hood, we us the `get_parameters` API call for boto3, which translate to a single network call for multiple parameters. You can group all parameters types in a single call, with `StringList` parameters returned a python lists.\n\n## Decorator stacking\nIf you wish to cache multiple parameters with different expiry times, stack the decorators. In this example, `var1` will be refreshed every 30 seconds, `var2` will be refreshed after 60.\n\n```python\n@ssm_cache(parameter=\'/production/app/var1\', ttl_seconds=30)\n@ssm_cache(parameter=\'/production/app/var2\', ttl_seconds=60)\ndef handler(event, context):\n    var1 = context.get(\'var1\')\n    var2 = context.get(\'var2\')\n    response = do_something(var)\n    return response\n```\n_Note: Decorator stacking performs one API call per decorator, which might result is slower performance_\n\n## Cache invalidation\n\nIf you require a fresh value at some point of the code, you can force a refresh using the `get_ssm_cache` function, and setting the `ttl_seconds` argument to 0.\n\n```python\nfrom simple_lambda_cache import ssm_cache, get_ssm_cache\n\n@ssm_cache(parameter=\'/prod/var\')\ndef handler(event, context):\n\n    if event.get(\'refresh\'):\n        # refresh parameter\n        var = get_ssm_cache(parameter=\'/prod/var\', ttl_seconds=0)\n    else:\n        var = context.get(\'var\')\n    \n    response = do_something(var)\n    return response\n```\n\nTo disable cache, set `ttl_seconds=0`.\n\nTo only get parameter once in the lifetime of the function, set `ttl_seconds` to some arbitary large number ~36000 (10 hours).\n\n## Return Values\n\nCaching supports `String`, `SecureString` and `StringList` parameters with no change required (ensure you have `kms:Decrypt` permission for `SecureString`). For simplicity, `StringList` parameters are automatically converted into list (delimited by \'/\'), while `String` and `SecureString` both return the single string value of the parameter.\n\n# Secrets Manager\n\nSecret support is similar, but uses the `secret_cache` decorator.\n\n```python\nfrom simple_lambda_cache import secret_cache\n\n@secret_cache(name=\'/prod/db/conn_string\')\ndef handler(event, context):\n    conn_string = context.get(\'conn_string\')\n    return context\n```\n\nSecrets Managers supports all the previously mentioned features including `ttl_seconds`, `entry_name` and cache invalidation.\n\n## Cache Invalidation\n\nTo invalidate a secret, use the `get_secret_cache`, setting the `ttl_seconds=0`.\n```python\nfrom simple_lambda_cache import secret_cache, get_secret_cache\n\n@secret_cache(name=\'/prod/db/conn_string\')\ndef handler(event, context):\n\n    if event.get(\'refresh\'):\n        var = get_secret_cache(name=\'/prod/db/conn_string\', ttl_seconds=0)\n    else:\n        var = context.get(\'conn_string\')\n    response = do_something(var)\n    return response\n```\n\n## Return Values\n\nSecrets Manager supports both string and binary secrets. For simplicity we will cache the secret in the format it is stored. It is up to the calling application to process the return as Binary or Strings.\n\n# Credit\n\nProject inspired by:\n* [SSM-Cache](https://github.com/alexcasalboni/ssm-cache-python)\n* [middy](https://github.com/middyjs/middy)\n',
    'author': 'keithrozario',
    'author_email': 'keith@keithrozario.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/keithrozario/simple_lambda_cache',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
