# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lambda_cache']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'lambda-cache',
    'version': '0.8.0',
    'description': 'Python utility for simple caching in AWS Lambda functions',
    'long_description': '<h1 align="center"> lambda-cache </h1>\n<h2 align="center"> Simple Caching for AWS Lambda</h2>\n\n![PackageStatus](https://img.shields.io/static/v1?label=status&message=beta&color=blueviolet?style=flat-square) [![PyPI version](https://badge.fury.io/py/lambda-cache.svg)](https://badge.fury.io/py/lambda-cache) ![PythonSupport](https://img.shields.io/static/v1?label=python&message=3.6%20|%203.7|%203.8&color=blue?style=flat-square&logo=python) [![License: MIT](https://img.shields.io/badge/License-MIT-blue.svg)](https://opensource.org/licenses/MIT)\n\n![Test](https://github.com/keithrozario/lambda-cache/workflows/Test/badge.svg?branch=release) [![Coverage Status](https://coveralls.io/repos/github/keithrozario/lambda-cache/badge.svg?branch=release)](https://coveralls.io/github/keithrozario/lambda-cache?branch=release) [![Documentation Status](https://readthedocs.org/projects/simple-lambda-cache/badge/?version=latest)](https://simple-lambda-cache.readthedocs.io/en/latest/?badge=latest)\n\n [![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black) \n\n# Basics\n\n_lambda_cache_ prioritizes simplicity over performance and flexibility. The goal is to provide the **simplest** way for developers to cache api calls in their Lambda functions.\n\nCurrently only SSM Parameters and Secrets from Secrets Manager are supported. S3 and generic function support is coming soon.\n\n## SSM - Parameter Store\n\n### Cache single parameter\n\nTo cache a parameter from ssm, decorate your handler function:\n\n```python\nfrom lambda_cache import ssm\n\n@ssm.cache(parameter=\'/production/app/var\')\ndef handler(event, context):\n    var = getattr(context,\'var\')\n    response = do_something(var)\n    return response\n```\nAll invocations of this function over in the next minute will reference the parameter from the function\'s internal cache, rather than making a network call to ssm. After one minute, the the next invocation will invoke `get_parameter` to refresh the cache.\n\n### Change cache expiry\n\nThe default `max_age_in_seconds` settings is 60 seconds (1 minute), it defines the maximum age of a parameter that is acceptable. Cache entries older than this value will be refreshed. To set a longer cache duration (e.g 5 minutes), change the setting like so:\n\n```python\nfrom lambda_cache import ssm\n\n@ssm.cache(parameter=\'/production/app/var\', max_age_in_seconds=300)\ndef handler(event, context):\n    var = getattr(context,\'var\')\n    response = do_something(var)\n    return response\n```\n\n_Note: The caching logic runs only at invocation, regardless of how long the function runs. A 15 minute lambda function will not refresh the parameter, unless explicitly refreshed using `get_entry` method. The library is primary interested in caching \'across\' invocation rather than \'within\' an invocation_\n\n### Change cache entry settings\n\nThe name of the parameter is simply shortened to the string after the last slash(\'/\') character of its name. This means `/production/app/var` and `test/app/var` resolve to just `var`. To over-ride this default, use `entry_name` setting like so:\n\n```python\nfrom lambda_cache import ssm\n\n@ssm.cache(parameter=\'/production/app/var\', entry_name=\'new_var\')\ndef handler(event, context):\n    var = getattr(context,\'new_var\')\n    response = do_something(var)\n    return response\n```\n\n### Cache multiple parameters\n\nTo cache multiple entries at once, pass a list of parameters to the parameter argument. This method groups all the parameter value under one python dictionary, stored in the Lambda Context under the `entry_name`. When using this method, `entry_name` is a mandatory setting, otherwise a `NoEntryNameError` exception is thrown.\n\n```python\nfrom lambda_cache import ssm\n\n@ssm.cache(parameter=[\'/app/var1\', \'/app/var2\'], entry_name=\'parameters\')\ndef handler(event, context):\n    var1 = getattr(context,\'parameters\').get(\'var1\')\n    var2 = getattr(context,\'parameters\').get(\'var2\')\n    response = do_something(var)\n    return response\n```\n\nUnder the hood, we use the `get_parameters` API call for boto3, which translate to a single network call for multiple parameters. You can group all parameters types in a single call, including `String`, `StringList` and `SecureString`. `StringList` will return as a list, while all other types will return as plain-text strings. The library does not support returning `SecureString` parameters in encrypted form, and will only return plain-text strings regardless of String type.\n\n_Note: for this method to work, ensure you have the `ssm:GetParameters` (with the \'s\' at the end) in your function\'s permission policy_\n\n### Decorator stacking\n\nIf you wish to cache multiple parameters with different expiry times, stack the decorators. In this example, `var1` will be refreshed every 30 seconds, `var2` will be refreshed after 60.\n\n```python\n@ssm.cache(parameter=\'/production/app/var1\', max_age_in_seconds=30)\n@ssm.cache(parameter=\'/production/app/var2\', max_age_in_seconds=60)\ndef handler(event, context):\n    var1 = getattr(context,\'var1\')\n    var2 = getattr(context,\'var2\')\n    response = do_something(var)\n    return response\n```\n_Note: Decorator stacking performs one API call per decorator, which might result is slower performance_\n\n### Cache invalidation\n\nIf you require a fresh value at some point of the code, you can force a refresh using the `ssm.get_entry` function, and setting the `max_age_in_seconds` argument to 0.\n\n```python\nfrom lambda_cache import ssm\n\n@ssm.cache(parameter=\'/prod/var\')\ndef handler(event, context):\n\n    if event.get(\'refresh\'):\n        # refresh parameter\n        var = ssm.get_entry(parameter=\'/prod/var\', max_age_in_seconds=0)\n    else:\n        var = getattr(context,\'var\')\n    \n    response = do_something(var)\n    return response\n```\n\nYou may also use `ssm.get_entry` to get a parameter entry from anywhere in your functions code.\n\nTo only get parameter once in the lifetime of the function, set `max_age_in_seconds` to some arbitary large number ~36000 (10 hours).\n\n### Return Values\n\nCaching supports `String`, `SecureString` and `StringList` parameters with no change required (ensure you have `kms:Decrypt` permission for `SecureString`). For simplicity, `StringList` parameters are automatically converted into list (delimited by \'/\'), while `String` and `SecureString` both return the single string value of the parameter.\n\n## Secrets Manager\n\n### Cache single secret\n\nSecret support is similar, but uses the `secret.cache` decorator.\n\n```python\nfrom lambda_cache import secret\n\n@secret.cache(name=\'/prod/db/conn_string\')\ndef handler(event, context):\n    conn_string = getattr(context,\'conn_string\')\n    return context\n```\n\n\n### Change Cache expiry\n\nThe default `max_age_in_seconds` settings is 60 seconds (1 minute), it defines how long a parameter should be kept in cache before it is refreshed from ssm. To configure longer or shorter times, modify this argument like so:\n\n```python\nfrom lambda_cache import secret\n\n@secret.cache(name=\'/prod/db/conn_string\', max_age_in_seconds=300)\ndef handler(event, context):\n    var = getattr(context,\'conn_string\')\n    response = do_something(var)\n    return response\n```\n\n_Note: The caching logic runs only at invocation, regardless of how long the function runs. A 15 minute lambda function will not refresh the parameter, unless explicitly refreshed using get_cache_ssm. The library is primary interested in caching \'across\' invocation rather than \'within\' an invocation_\n\n### Change Cache entry settings\n\nThe name of the secret is simply shortened to the string after the last slash(\'/\') character of the secret\'s name. This means `/prod/db/conn_string` and `/test/db/conn_string` resolve to just `conn_string`. To over-ride this default, use `entry_name`:\n\n```python\nfrom lambda_cache import secret\n\n@secret.cache(name=\'/prod/db/conn_string\', entry_name=\'new_var\')\ndef handler(event, context):\n    var = getattr(context,\'new_var\')\n    response = do_something(var)\n    return response\n```\n\n### Decorator stacking\n\nIf you wish to cache multiple secrets, you can use decorator stacking.\n\n```python\n@secret.cache(name=\'/prod/db/conn_string\', max_age_in_seconds=30)\n@secret.cache(name=\'/prod/app/elk_username_password\', max_age_in_seconds=60)\ndef handler(event, context):\n    var1 = getattr(context,\'conn_string\')\n    var2 = getattr(context,\'elk_username_password\')\n    response = do_something(var)\n    return response\n```\n\n_Note: Decorator stacking performs one API call per decorator, which might result is slower performance._\n\n### Cache Invalidation\n\nTo invalidate a secret, use the `get_entry`, setting the `max_age_in_seconds=0`.\n```python\nfrom lambda_cache import secret\n\n@secret.cache(name=\'/prod/db/conn_string\')\ndef handler(event, context):\n\n    if event.get(\'refresh\'):\n        var = secret.get_entry(name=\'/prod/db/conn_string\', max_age_in_seconds=0)\n    else:\n        var = getattr(context,\'conn_string\')\n    response = do_something(var)\n    return response\n```\n\n### Return Values\n\nSecrets Manager supports both string and binary secrets. For simplicity we will cache the secret in the format it is stored. It is up to the calling application to process the return as Binary or Strings.\n\n## S3\n\nS3 support is considered _experimental_ for now, but withing the python community we see a lot of folks pull down files from S3 for use in AI/ML models.\n\nFiles downloaded from s3 are automatically stored in the `tmp` directory of the lambda function. This is the only writable directory within lambda, and has a 512MB of storage space. \n\nTo download a file from S3 use the the same decorator pattern:\n\n```python\nfrom lambda_cache import s3\n\n@s3.cache(s3Uri=\'s3://bucket_name/path/to/object.json\')\ndef s3_download_entry_name(event, context):\n    with open("/tmp/object.json") as file_data:\n        status = json.loads(file_data.read())[\'status\']\n\n    return status\n```\n\n### Change Cache expiry\n\nThe default `max_age_in_seconds` settings is 60 seconds (1 minute), it defines how long a file should be kept in `/tmp` before it is refreshed from S3. To configure longer or shorter times, modify this argument like so:\n\n```python\nfrom lambda_cache import s3\n\n@s3.cache(s3Uri=\'s3://bucket_name/path/to/object.json\', max_age_in_seconds=300)\ndef s3_download_entry_name(event, context):\n    with open("/tmp/object.json") as file_data:\n        status = json.loads(file_data.read())[\'status\']\n\n    return status\n```\n\n_Note: The caching logic runs only at invocation, regardless of how long the function runs. A 15 minute lambda function will not refresh the object, unless explicitly refreshed using `s3.get_entry`. The library is primary interested in caching \'across\' invocation rather than \'within\' an invocation_\n\nBy default, _lambda_cache_ will download the file once at cache has expired, however, to save on network bandwidth (and possibly time), we can set the `check_before_download` parameter to True. This will check the age of the object in S3 and download only if the object has changed since the last download. \n\n```python\nfrom lambda_cache import s3\n\n@s3.cache(s3Uri=\'s3://bucket_name/path/to/object.json\', max_age_in_seconds=300, check_before_download=True)\ndef s3_download_entry_name(event, context):\n    with open("/tmp/object.json") as file_data:\n        status = json.loads(file_data.read())[\'status\']\n\n    return status\n```\n\n_Note: we use the GetHead object call to verify the objects `last_modified_date`. This simplifies the IAM policy of the function, as it still onlyrequires only the `s3:GetObject` permission. However, this is still a GET requests, and will be charged as such, for smaller objects it might be cheaper to just download the object_\n\n# Credit\n\nProject inspired by:\n* [SSM-Cache](https://github.com/alexcasalboni/ssm-cache-python)\n* [middy](https://github.com/middyjs/middy)\n',
    'author': 'keithrozario',
    'author_email': 'keith@keithrozario.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/keithrozario/simple_lambda_cache',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
