"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_lambda_1 = require("@aws-sdk/client-lambda");
const client = new client_lambda_1.LambdaClient({});
exports.handler = async (_event, _context) => {
    // Validate RETAIN_VERSIONS environment variable
    const retainVersionStr = process.env.RETAIN_VERSIONS;
    if (retainVersionStr === undefined || !/^(0|[1-9]\d*)$/.test(retainVersionStr)) {
        throw new Error('Invalid RETAIN_VERSIONS environment variable value. It should be a string containing a positive integer.');
    }
    const retainVersions = parseInt(retainVersionStr, 10);
    try {
        // List all Lambda layers
        const listAllLayersCmd = new client_lambda_1.ListLayersCommand({});
        const layers = await client.send(listAllLayersCmd);
        if (layers.Layers === undefined) {
            console.log('No layers found.');
            return;
        }
        // Iterate over each layer
        layers.Layers.forEach(async (layer) => {
            if (layer.LayerName === undefined) {
                console.log('Layer name is undefined.');
                return;
            }
            // List versions for the current layer
            const listLayerVersionsCmd = new client_lambda_1.ListLayerVersionsCommand({ LayerName: layer.LayerName });
            const layerVersions = await client.send(listLayerVersionsCmd);
            if (layerVersions.LayerVersions === undefined || layerVersions.LayerVersions.length <= retainVersions) {
                console.log(`Layer: ${layer.LayerName} - Keeping all versions. No action needed.`);
                return;
            }
            // Sort layer versions in ascending order
            const sortedVersions = layerVersions.LayerVersions.sort((a, b) => {
                const aVersion = a.Version ?? 0;
                const bVersion = b.Version ?? 0;
                return aVersion - bVersion;
            });
            // Identify versions to delete and delete them
            const versionsToDelete = sortedVersions.slice(0, sortedVersions.length - retainVersions);
            versionsToDelete.forEach(async (version) => {
                if (version.Version === undefined) {
                    console.log('Version number is undefined.');
                    return;
                }
                const deleteLayerVersionCmd = new client_lambda_1.DeleteLayerVersionCommand({ LayerName: layer.LayerName, VersionNumber: version.Version });
                await client.send(deleteLayerVersionCmd);
                console.log(`Deleted version ${version.Version} of layer ${layer.LayerName}.`);
            });
        });
    }
    catch (error) {
        console.error('Error occurred:', error);
        throw error;
    }
};
//# sourceMappingURL=data:application/json;base64,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