"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_lambda_1 = require("@aws-sdk/client-lambda");
const client = new client_lambda_1.LambdaClient({});
exports.handler = async (_event, _context) => {
    // Validate RETAIN_VERSIONS environment variable
    const retainVersionStr = process.env.RETAIN_VERSIONS;
    if (retainVersionStr === undefined || !/^(0|[1-9]\d*)$/.test(retainVersionStr)) {
        throw new Error('Invalid RETAIN_VERSIONS environment variable value. It should be a string containing a positive integer.');
    }
    const retainVersions = parseInt(retainVersionStr, 10);
    let successfulDeletes = 0;
    let failedDeletes = 0;
    let failedLayerNames = [];
    try {
        // List all Lambda layers
        const listAllLayersCmd = new client_lambda_1.ListLayersCommand({});
        const layers = await client.send(listAllLayersCmd);
        if (layers.Layers === undefined) {
            console.log('No layers found.');
            return;
        }
        // Iterate over each layer
        for (const layer of layers.Layers) {
            if (layer.LayerName === undefined) {
                console.log('Layer name is undefined.');
                continue;
            }
            console.log(`Processing layer: ${layer.LayerName}`);
            // List versions for the current layer
            let layerVersions;
            try {
                const listLayerVersionsCmd = new client_lambda_1.ListLayerVersionsCommand({ LayerName: layer.LayerName });
                layerVersions = await client.send(listLayerVersionsCmd);
                if (layerVersions.LayerVersions === undefined || layerVersions.LayerVersions.length <= retainVersions) {
                    console.log(`Layer: ${layer.LayerName} - Keeping all versions. No action needed.`);
                    continue;
                }
            }
            catch (error) {
                console.error(`Error listing versions for layer ${layer.LayerName}:`, error);
                failedDeletes++;
                failedLayerNames.push(layer.LayerName);
                continue;
            }
            console.log(`Current total versions: ${layerVersions.LayerVersions.length}`);
            // Sort layer versions in ascending order
            const sortedVersions = layerVersions.LayerVersions.sort((a, b) => {
                const aVersion = a.Version ?? 0;
                const bVersion = b.Version ?? 0;
                return aVersion - bVersion;
            });
            // Identify versions to delete and delete them
            const versionsToDelete = sortedVersions.slice(0, sortedVersions.length - retainVersions);
            for (const version of versionsToDelete) {
                if (version.Version === undefined) {
                    console.log('Version number is undefined.');
                    continue;
                }
                try {
                    const deleteLayerVersionCmd = new client_lambda_1.DeleteLayerVersionCommand({ LayerName: layer.LayerName, VersionNumber: version.Version });
                    await client.send(deleteLayerVersionCmd);
                    console.log(`Deleted version ${version.Version} of layer ${layer.LayerName}.`);
                    successfulDeletes++;
                }
                catch (error) {
                    console.error(`Error deleting version ${version.Version} of layer ${layer.LayerName}:`, error);
                    failedDeletes++;
                    failedLayerNames.push(layer.LayerName);
                }
            }
            ;
        }
        ;
    }
    catch (error) {
        console.error('Error occurred:', error);
        throw error;
    }
    finally {
        console.log(`Summary: ${successfulDeletes} successful deletes, ${failedDeletes} failed deletes.`);
        console.log(`Failed layer names: ${failedLayerNames.join(', ')}`);
    }
};
//# sourceMappingURL=data:application/json;base64,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