#!/usr/bin/python3

"""Utilities for parsing lambda terms"""

from __future__ import annotations
from collections import deque
from collections.abc import Iterator, Sequence
from itertools import chain
from lambda_calculus.terms import Abstraction, Application, Term, Variable
from lark import Lark, Token
from lark.exceptions import UnexpectedInput, UnexpectedToken
from lark.lark import PostLex
from lark.visitors import Transformer, v_args

__all__ = (
    "PARSER",
    "WhitespacePostLex",
    "LambdaTransformer"
)


class WhitespacePostLex(PostLex):
    """PostLex removing unnecessary whitespace"""

    __slots__ = ()

    def is_unnecessary(self, before: Token | None, after: Token | None) -> bool:
        """check if whitespace is unnecessary"""
        return before in {"\\", "λ", ".", "(", None} or after in {".", ")", None}

    def process(self, stream: Iterator[Token]) -> Iterator[Token]:
        """process the tokens generated by the lexer"""
        try:
            first = next(stream)
        except StopIteration:
            return
        buffer: deque[Token | None] = deque((None,) * 2, maxlen=3)
        buffer.append(first)
        for token in chain(stream, (None,)):
            buffer.append(token)
            if buffer[1] is not None:
                if buffer[1].type == "_WHITESPACE" \
                        and self.is_unnecessary(buffer[0], buffer[2]):
                    continue
                yield buffer[1]


class LambdaTransformer(Transformer[Token, Term[str]]):
    """Transformer transforming an AST into a lambda term"""

    def transform_string(self, string: str) -> Term[str]:
        """parse a string and return the transformed lambda term"""
        # the parser sometimes return tokens directly instead of a tree
        match PARSER.parse(string):
            case Token(type="VARIABLE") as name:    # type: ignore
                return self.VARIABLE(name)          # type: ignore
            case Token() as token:                  # type: ignore
                raise UnexpectedToken(token, {"VARIABLE",})
            case tree:
                return self.transform(tree)

    def __default__(self, data: object, children: object, meta: object) -> Term[str]:
        """handle unknown nodes"""
        raise UnexpectedInput(f"unknown node: {data}")

    def VARIABLE(self, name: Token) -> Variable[str]:
        """transform a variable node"""
        return Variable(name.value)

    @v_args(inline=True)
    def abstraction(self, variable: Variable[str], body: Term[str]) -> Abstraction[str]:
        """transform an abstraction"""
        return Abstraction(variable.name, body)

    def application(self, children: Sequence[Term[str]]) -> Application[str]:
        """transform an application"""
        return Application.with_arguments(children[0], children[1:])


PARSER = Lark.open_from_package(
    __name__,
    "grammar.lark",
    start="term",
    parser="lalr",
    lexer="basic",
    postlex=WhitespacePostLex(),
    propagate_positions=True
)
