# export OPENAI_API_KEY=sk-<OPENAI API KEY>
# export WCS_DEMO_URL=https://<CLUSTER ID>.weaviate.cloud
# export WCS_DEMO_RO_KEY=<YOUR READ ONLY KEY>
# export LANGTRACE_API_KEY=<YOUR LANGTRACE API KEY>

# python main.py

# Example taken from startup guide
# https://weaviate.io/developers/weaviate/starter-guides/generative

import json
import os
from pathlib import Path

import requests
import weaviate
import weaviate.classes as wvc
from weaviate.classes.aggregate import GroupByAggregate
from weaviate.classes.query import Filter, HybridFusion, MetadataQuery
from weaviate.collections.classes.grpc import Move

import langtrace_python_sdk.langtrace as langtrace
from langtrace_python_sdk import with_langtrace_root_span

# Set these environment variables
WCS_DEMO_URL = os.environ["WCS_DEMO_URL"]
WCS_DEMO_RO_KEY = os.environ["WCS_DEMO_RO_KEY"]

# Connect to a WCS instance
client = weaviate.connect_to_wcs(
    cluster_url=WCS_DEMO_URL,
    auth_credentials=weaviate.auth.AuthApiKey(WCS_DEMO_RO_KEY),
    skip_init_checks=True,
    headers={"X-OpenAI-Api-Key": os.environ["OPENAI_API_KEY"]},
)

langtrace.init()


@with_langtrace_root_span()
def create():
    questions = client.collections.create(
        name="Question",
        # Set the vectorizer to "text2vec-openai" to use the OpenAI API for vector-related operations
        vectorizer_config=wvc.config.Configure.Vectorizer.text2vec_openai(),
        # Ensure the `generative-openai` module is used for generative queries
        generative_config=wvc.config.Configure.Generative.openai(),
    )


def insert():
    resp = requests.get(
        "https://raw.githubusercontent.com/weaviate-tutorials/quickstart/main/data/jeopardy_tiny.json"
    )
    data = json.loads(resp.text)  # Load data

    question_objs = list()
    for i, d in enumerate(data):
        question_objs.append(
            {
                "answer": d["Answer"],
                "question": d["Question"],
                "category": d["Category"],
            }
        )

    questions = client.collections.get("Question")
    questions.data.insert_many(question_objs)
    client.close()


@with_langtrace_root_span()
def query_data_bm25():
    try:
        questions = client.collections.get("Question")
        response = questions.query.bm25(query="science", limit=2)

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()  # Close client gracefully


@with_langtrace_root_span()
def query_data_near_text():
    try:
        questions = client.collections.get("Question")
        response = questions.query.near_text(
            query="biology",
            limit=2,
            distance=0.6,
            # certainty=0.7,
            return_properties=["category", "question", "answer"],
            # rerank=False,
            return_metadata=wvc.query.MetadataQuery(
                score=True, explain_score=True, distance=True
            ),
            filters=wvc.query.Filter.by_property("category").equal("SCIENCE"),
            move_to=Move(force=0.85, concepts="study"),
            # include_vector=True,
            move_away=Move(force=0.45, concepts="history"),
            # group_by=GroupByAggregate(prop="inPublication"),
        )

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()  # Close client gracefully


@with_langtrace_root_span()
def query_fetch_object_by_id():
    try:
        questions = client.collections.get("Question")
        response = questions.query.fetch_object_by_id(
            "cb9899ad-5903-4596-9465-6ab012486855"
        )

        print(response)

    finally:
        client.close()  # Close client gracefully


@with_langtrace_root_span()
def query_fetch_objects():
    try:
        questions = client.collections.get("Question")
        response = questions.query.fetch_objects(include_vector=True)

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()  # Close client gracefully


@with_langtrace_root_span()
def query_hybrid():
    try:
        questions = client.collections.get("Question")
        response = questions.query.hybrid(
            query="human body",
            target_vector="question",
            limit=3,
            filters=Filter.by_property("category").equal("SCIENCE"),
        )

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()  # Close client gracefully


@with_langtrace_root_span()
def query_near_object():
    try:
        questions = client.collections.get("Question")
        response = questions.query.near_object(
            # Provide a `WeaviateUUID` object
            near_object="cb9899ad-5903-4596-9465-6ab012486855",
            return_properties=["category", "question", "answer"],
            limit=1,
        )

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()


# @with_langtrace_root_span()


@with_langtrace_root_span()
def query_data_near_vector():
    try:
        questions = client.collections.get("Question")
        response = questions.query.near_vector(
            near_vector=[
                -0.0037347390316426754,
                0.014777889475226402,
                -0.0031122826039791107,
                -0.009939171373844147,
                0.0026794152799993753,
                0.01773078367114067,
                -0.010214326903223991,
                -0.018871674314141273,
                -0.02146216668188572,
                -0.025529107078909874,
                -0.02005283161997795,
                0.012643752619624138,
                -0.011086773127317429,
                -0.00863721314817667,
                0.01595904678106308,
                0.034119341522455215,
                0.029394712299108505,
                0.008764725178480148,
                0.027126353234052658,
                -0.003603872377425432,
                -0.018012650310993195,
                0.029394712299108505,
                0.0015108411898836493,
                -0.017529450356960297,
                0.004580340348184109,
                0.023569459095597267,
                0.008046634495258331,
                -0.012012907303869724,
                -0.021005811169743538,
                0.027126353234052658,
                -0.008261390961706638,
                0.026535775512456894,
                -0.008818414062261581,
                -0.020589720457792282,
                -0.016469093039631844,
                -0.018307939171791077,
                0.004295117687433958,
                -0.03122013807296753,
                0.02214670181274414,
                -0.025220394134521484,
                0.018925363197922707,
                0.011932373978197575,
                -0.0024562703911215067,
                -0.005607142113149166,
                -0.013918866403400898,
                -0.00790570117533207,
                -0.006097054108977318,
                -0.006016520317643881,
                -0.007999656721949577,
                0.012650463730096817,
                0.005556808784604073,
                0.004080361686646938,
                -0.028911512345075607,
                -0.014308110810816288,
                -0.015435579232871532,
                -0.00805334560573101,
                0.010831749998033047,
                -0.013932288624346256,
                -0.003432738594710827,
                -0.0011778018670156598,
                -0.014160466380417347,
                -0.004576984792947769,
                -0.02756928652524948,
                0.02912626788020134,
                -0.0099861491471529,
                -0.008174145594239235,
                -0.016791226342320442,
                -0.008039923384785652,
                0.0016685526352375746,
                -0.017180470749735832,
                0.021005811169743538,
                0.005707808770239353,
                0.027139775454998016,
                -0.01312695350497961,
                0.039541929960250854,
                -0.017368381842970848,
                -0.017086515203118324,
                -0.012677308171987534,
                0.01969042979180813,
                -0.004969585686922073,
                0.026065995916724205,
                -0.033582452684640884,
                -0.02734110876917839,
                0.013173931278288364,
                0.02479088306427002,
                0.009670726023614407,
                -0.0064829434268176556,
                0.020079676061868668,
                -0.011865262873470783,
                0.0031793939415365458,
                0.037689659744501114,
                0.009966015815734863,
                0.010120371356606483,
                0.006831921637058258,
                -0.03189124912023544,
                0.011979351751506329,
                0.0039696283638477325,
                0.009764681570231915,
                -0.004382362589240074,
                -0.036991700530052185,
                -0.015851669013500214,
                0.0013875244185328484,
                0.004295117687433958,
                -0.011053217574954033,
                -0.017368381842970848,
                -0.02277754619717598,
                0.0003485588822513819,
                0.008388902060687542,
                0.004053517244756222,
                -0.014388645067811012,
                -0.003093827050179243,
                0.013704109936952591,
                0.008543257601559162,
                -0.03887081518769264,
                0.02167692221701145,
                -0.01989176496863365,
                0.015824824571609497,
                0.02099238894879818,
                0.009086858481168747,
                -0.016603315249085426,
                0.03887081518769264,
                -0.009664014913141727,
                0.022804390639066696,
                -0.021099766716361046,
                0.019153540953993797,
                0.03500521183013916,
                -0.007838590070605278,
                0.0008992903167381883,
                -0.009999571368098259,
                -0.0006547538214363158,
                0.001299440860748291,
                0.018334783613681793,
                0.021489011123776436,
                0.01642882637679577,
                -0.019932031631469727,
                -0.002704581944271922,
                -0.001963003072887659,
                0.017328115180134773,
                0.01340210996568203,
                -0.01821398362517357,
                -0.0051004523411393166,
                0.030414802953600883,
                -0.026925019919872284,
                0.005331986118108034,
                0.003902517259120941,
                0.01181828510016203,
                0.04354175552725792,
                0.013972555287182331,
                0.01146259531378746,
                0.012019618414342403,
                -0.0003449936048127711,
                0.014898689463734627,
                0.04507189244031906,
                -0.010234460234642029,
                0.0059024314396083355,
                -0.0026089486200362444,
                -0.005929275881499052,
                0.0184287391602993,
                -0.012482685968279839,
                0.0015561413019895554,
                0.013153797946870327,
                0.006919166538864374,
                0.009670726023614407,
                0.007194322533905506,
                0.011241128668189049,
                0.01936829648911953,
                0.0016677137464284897,
                -0.006939299870282412,
                0.007368811406195164,
                0.0005993870436213911,
                -0.0048185852356255054,
                0.006744677200913429,
                -0.0012566575314849615,
                0.033743519335985184,
                0.037528593093156815,
                0.010422371327877045,
                -0.014281266368925571,
                0.0021609810646623373,
                -0.0036172945983707905,
                -0.023998970165848732,
                -0.005321919452399015,
                0.0008917402592487633,
                0.019086429849267006,
                0.01844216138124466,
                -0.027005553245544434,
                -0.0031391270458698273,
                0.00359380547888577,
                -0.0109995286911726,
                0.008576813153922558,
                0.012751131318509579,
                0.009194236248731613,
                0.008603657595813274,
                0.003033427055925131,
                0.009180814027786255,
                -0.6571530103683472,
                -0.017099937424063683,
                0.02477746084332466,
                0.0018589806277304888,
                0.030656402930617332,
                0.028562532737851143,
                0.0038186281453818083,
                -0.013851755298674107,
                -0.023045990616083145,
                0.00702654430642724,
                -0.0025888150557875633,
                0.013771221041679382,
                -0.00536554167047143,
                -0.004432695917785168,
                -0.006727899424731731,
                -0.02440163865685463,
                0.028696754947304726,
                -0.022106435149908066,
                0.017234159633517265,
                0.0027079374995082617,
                -0.026227062568068504,
                0.014026244170963764,
                0.012516241520643234,
                0.01320748683065176,
                0.016683848574757576,
                0.01828109472990036,
                0.0015980858588591218,
                -0.012590063735842705,
                0.004942741245031357,
                -0.00991232693195343,
                -0.019932031631469727,
                0.039917752146720886,
                -0.007362100295722485,
                -0.0252472385764122,
                0.034441474825143814,
                -0.00369782792404294,
                0.008892236277461052,
                0.03199862688779831,
                0.027166619896888733,
                0.02657604217529297,
                -0.01405308861285448,
                -0.010717661119997501,
                0.006593676749616861,
                -0.019328029826283455,
                -0.0025653261691331863,
                0.019932031631469727,
                0.007570145186036825,
                -0.004835363011807203,
                0.010825038887560368,
                -0.03438778594136238,
                0.026951864361763,
                0.02083132043480873,
                0.01332828775048256,
                0.012912197969853878,
                -0.0025502261705696583,
                -0.0009865348692983389,
                0.004181028809398413,
                -0.0020066252909600735,
                -0.009697570465505123,
                0.019583052024245262,
                -0.014106777496635914,
                -0.00161150807980448,
                0.008026501163840294,
                -0.01790527254343033,
                -0.013670554384589195,
                0.019032740965485573,
                -0.043488066643476486,
                -0.012751131318509579,
                0.016912026330828667,
                -0.018952207639813423,
                0.00032695746631361544,
                0.011764596216380596,
                -0.014200733043253422,
                0.00056163698900491,
                -0.00032632829970680177,
                0.034897830337285995,
                0.0006023231544531882,
                -0.0018052917439490557,
                -0.008375479839742184,
                0.013851755298674107,
                -0.01289206463843584,
                0.0007713595405220985,
                -0.004603829234838486,
                -0.019019318744540215,
                0.03379720821976662,
                -0.03836077079176903,
                -0.0009353625937364995,
                0.015341623686254025,
                0.01602615788578987,
                -0.014308110810816288,
                0.008664057590067387,
                0.014388645067811012,
                -0.01673753745853901,
                0.0032062383834272623,
                0.007482900749891996,
                -0.00710707763209939,
                0.0014462467515841126,
                -0.018643496558070183,
                0.008462724275887012,
                -0.04338068887591362,
                -0.002253259066492319,
                -0.0002050037874141708,
                0.025623062625527382,
                0.004466251470148563,
                -0.0013279631966724992,
                0.008107034489512444,
                -0.023569459095597267,
                -0.004217939916998148,
                0.01943540759384632,
                -0.019166963174939156,
                0.023045990616083145,
                0.015529535710811615,
                -0.009106991812586784,
                0.004033383913338184,
                -0.02138163335621357,
                -0.02495194971561432,
                0.028213554993271828,
                0.0037179612554609776,
                -0.012717575766146183,
                -0.0036474945954978466,
                0.019542785361409187,
                -0.0011853518662974238,
                0.01797238364815712,
                -0.01951594091951847,
                0.0016475802985951304,
                0.028562532737851143,
                -0.007503034081310034,
                -0.027998799458146095,
                -0.006690988317131996,
                -0.029394712299108505,
                0.01215384155511856,
                -0.018334783613681793,
                0.010979394428431988,
                -0.006748032756149769,
                0.012462552636861801,
                0.018482428044080734,
                0.01289206463843584,
                0.015234245918691158,
                0.017596561461687088,
                -0.021435322239995003,
                -0.030387958511710167,
                -0.006016520317643881,
                0.011422328650951385,
                -0.015046334825456142,
                -0.010321704670786858,
                -0.020804475992918015,
                -0.024213725700974464,
                0.009194236248731613,
                -0.04088415205478668,
                -0.020415231585502625,
                -0.0030585937201976776,
                -0.011066639795899391,
                -0.01829451695084572,
                0.00014890301099512726,
                0.00485885189846158,
                0.017677094787359238,
                -0.0004987202119082212,
                -0.013442376628518105,
                -0.010764638893306255,
                -0.0016257691895589232,
                0.022509101778268814,
                0.014657089486718178,
                -0.011348506435751915,
                -0.01628118008375168,
                -0.010972683317959309,
                -0.016227491199970245,
                -0.03787757083773613,
                0.030334269627928734,
                -0.006378921214491129,
                -0.014912111684679985,
                -0.0067346105352044106,
                -0.01153641752898693,
                -0.0013497743057087064,
                0.00932845938950777,
                0.014187310822308064,
                -0.02602572925388813,
                -0.028320932760834694,
                0.003560249926522374,
                -0.032106004655361176,
                -0.019341452047228813,
                -0.009026458486914635,
                0.010529750026762486,
                -0.01499264594167471,
                -0.00359380547888577,
                -0.011677351780235767,
                -0.009543214924633503,
                0.022737279534339905,
                0.011778018437325954,
                0.016912026330828667,
                0.03540787845849991,
                -0.0007352872635237873,
                0.004610540345311165,
                -0.011637085117399693,
                0.020428653806447983,
                -0.014348377473652363,
                -0.020254164934158325,
                0.006016520317643881,
                0.01440206728875637,
                -0.0023807703983038664,
                0.02068367600440979,
                0.02253594622015953,
                -0.028938356786966324,
                -0.0002110857458319515,
                -0.014616822823882103,
                0.03454885259270668,
                -0.019019318744540215,
                0.004506518132984638,
                -0.015408734790980816,
                0.003046849276870489,
                0.007388944737613201,
                0.01649593748152256,
                -0.01030157133936882,
                -0.013677265495061874,
                -0.030710091814398766,
                0.011006239801645279,
                -0.0006430093199014664,
                -0.01196592953056097,
                -0.015100023709237576,
                -0.00824796874076128,
                -0.0014940634137019515,
                -0.0008648958173580468,
                0.00020814963500015438,
                0.002600559499114752,
                0.02014678716659546,
                -0.011529706418514252,
                0.010952549986541271,
                0.009100280702114105,
                0.017690517008304596,
                0.012684019282460213,
                -0.02036154270172119,
                0.0019898475147783756,
                0.01573086902499199,
                0.0076104118488729,
                0.005174274556338787,
                0.01065055001527071,
                -0.007670811843127012,
                0.019851498305797577,
                -0.026280751451849937,
                0.020938700065016747,
                0.005056830123066902,
                0.006979566533118486,
                -0.006456098984926939,
                0.023998970165848732,
                -0.01719389297068119,
                0.044883981347084045,
                -0.0007898150943219662,
                0.01247597485780716,
                0.002894171280786395,
                -0.01595904678106308,
                0.0034528719261288643,
                -0.02269701287150383,
                -0.011113617569208145,
                0.0034629388246685266,
                0.009731126017868519,
                0.013918866403400898,
                -0.010892149992287159,
                0.0030233601573854685,
                0.021435322239995003,
                0.010449216701090336,
                0.029233645647764206,
                0.006617165636271238,
                -0.007482900749891996,
                0.014522867277264595,
                -0.008046634495258331,
                0.021515855565667152,
                0.024683505296707153,
                -0.015824824571609497,
                -0.0337703637778759,
                0.007758056744933128,
                0.004170962143689394,
                -0.02547541819512844,
                0.0021358144003897905,
                -0.0007180899847298861,
                -0.026589464396238327,
                0.021327944472432137,
                0.006744677200913429,
                -0.009664014913141727,
                0.013811487704515457,
                0.0150597570464015,
                -0.017247581854462624,
                -0.006701054982841015,
                -0.01571744680404663,
                0.020281009376049042,
                0.00282705994322896,
                -0.016925448551774025,
                0.004983007907867432,
                -0.006925877649337053,
                0.009751259349286556,
                -0.031193293631076813,
                0.04246797785162926,
                0.01873745210468769,
                0.0066809216514229774,
                -0.008308368735015392,
                0.012368597090244293,
                0.0031542270444333553,
                -0.022952035069465637,
                0.019542785361409187,
                0.002619015285745263,
                -0.0076372562907636166,
                0.01742207072675228,
                0.007878856733441353,
                -0.012100151740014553,
                0.007341966964304447,
                -0.0013489354168996215,
                0.02471034973859787,
                -0.006321876309812069,
                -0.005103807896375656,
                -0.02160981111228466,
                0.006865477189421654,
                -0.0018036139663308859,
                0.0018002584110945463,
                -0.034522008150815964,
                -0.015355045907199383,
                -0.022884923964738846,
                -0.02107292227447033,
                0.01138877309858799,
                0.00117612408939749,
                -0.002085481071844697,
                0.027596130967140198,
                -0.004228006582707167,
                -0.004550140351057053,
                -0.026616308838129044,
                -0.03068324737250805,
                -0.01703282631933689,
                0.07231904566287994,
                0.014979223720729351,
                -0.016482515260577202,
                0.021972211077809334,
                -0.00607692077755928,
                -0.026683419942855835,
                -0.03183756023645401,
                0.0030351048335433006,
                0.013194064609706402,
                -0.02006625384092331,
                -0.0017583138542249799,
                0.02748875319957733,
                0.011690774001181126,
                0.007919123396277428,
                0.0219990573823452,
                0.002231447957456112,
                -0.005905786994844675,
                -0.023985547944903374,
                -0.009623748250305653,
                -0.02748875319957733,
                0.01239544153213501,
                0.01007339358329773,
                0.0021525921765714884,
                0.029072578996419907,
                0.0008149818168021739,
                -0.023609725758433342,
                0.0234620813280344,
                0.006761454977095127,
                0.0233010146766901,
                -0.02464323863387108,
                -0.01544900145381689,
                0.0007281567086465657,
                0.0009429125930182636,
                -0.0034495163708925247,
                -0.001994880847632885,
                -0.0034562277141958475,
                0.006003098096698523,
                -0.016066424548625946,
                -0.00769094517454505,
                -0.0062547652050852776,
                0.038065481930971146,
                0.007194322533905506,
                0.02826724387705326,
                -0.02385132573544979,
                -0.005919209215790033,
                0.0014428910799324512,
                -0.0021978924050927162,
                0.015261090360581875,
                -0.011845129542052746,
                -0.03178387135267258,
                0.0022180257365107536,
                -0.0039058728143572807,
                0.012489397078752518,
                -0.018804563209414482,
                0.023327859118580818,
                -0.00035464082611724734,
                -0.012999442405998707,
                -0.004972941242158413,
                -0.026280751451849937,
                -0.023663414642214775,
                -0.04934016615152359,
                -0.030146358534693718,
                0.005583653226494789,
                -0.004587051458656788,
                -0.04834691807627678,
                -0.00631180964410305,
                -0.012187397107481956,
                0.005402452778071165,
                -0.014039666391909122,
                -0.00359380547888577,
                0.011979351751506329,
                -0.04300486668944359,
                0.0025418372824788094,
                -0.013798065483570099,
                0.03339454159140587,
                0.012173974886536598,
                0.011046506464481354,
                0.010529750026762486,
                0.020817898213863373,
                0.01774420589208603,
                0.011019662022590637,
                -0.01057001668959856,
                -0.003922650590538979,
                -0.008456013165414333,
                0.022952035069465637,
                -0.012308197095990181,
                0.001498257857747376,
                -0.00673796609044075,
                -0.007596989627927542,
                -0.0010595183121040463,
                -0.001191224087961018,
                -0.0003775005752686411,
                0.008596946485340595,
                -0.008268102072179317,
                0.017985805869102478,
                0.009684148244559765,
                0.004160895477980375,
                0.01990518718957901,
                0.0025972039438784122,
                -0.010979394428431988,
                0.003018326824530959,
                -0.0037481614854186773,
                -0.01212028507143259,
                -0.02144874446094036,
                0.0007050872081890702,
                0.009509659372270107,
                0.007617122959345579,
                0.0036542057059705257,
                -0.0020989032927900553,
                0.006056786980479956,
                0.025837818160653114,
                -0.02091185562312603,
                0.00832179095596075,
                5.91417592659127e-05,
                -0.0002946601889561862,
                -0.013569887727499008,
                0.012583352625370026,
                0.013583309948444366,
                -0.013368554413318634,
                -0.01354975439608097,
                0.006784943863749504,
                0.0006564315990544856,
                0.00897948071360588,
                0.02818671055138111,
                -0.008811702951788902,
                0.03857552632689476,
                -0.002979737939313054,
                -0.0068285660818219185,
                -0.03167649358510971,
                -0.007154055405408144,
                0.006774877198040485,
                0.025931773707270622,
                -0.01246926374733448,
                -0.03930032625794411,
                -0.004311895463615656,
                -0.016227491199970245,
                -0.02022732049226761,
                0.020428653806447983,
                -0.024280838668346405,
                -0.02671026438474655,
                -0.021878255531191826,
                -0.009080147370696068,
                -0.0050702523440122604,
                0.012965886853635311,
                0.04244113340973854,
                -0.023287592455744743,
                -0.017153626307845116,
                0.015126868151128292,
                0.004509873688220978,
                0.004546784795820713,
                -0.020482342690229416,
                -0.03556894510984421,
                0.010053260251879692,
                -0.011482728645205498,
                0.016710693016648293,
                -0.03905872628092766,
                -0.015395312570035458,
                -0.00457027368247509,
                0.010596861131489277,
                0.027622975409030914,
                0.02128767780959606,
                -0.03586423397064209,
                -0.004238073248416185,
                0.008966058492660522,
                0.0067580994218587875,
                0.015435579232871532,
                -0.011952507309615612,
                0.015435579232871532,
                -0.027703510597348213,
                0.01727442629635334,
                -0.011207573115825653,
                -0.029555778950452805,
                0.006013164762407541,
                -0.004271628800779581,
                0.001421918859705329,
                0.026133107021450996,
                -0.0150597570464015,
                -0.00991903804242611,
                -0.024495594203472137,
                -0.03382405266165733,
                -0.01081161666661501,
                -0.01681807078421116,
                -0.0001783690240699798,
                -0.007791612297296524,
                -0.025904929265379906,
                -0.021099766716361046,
                0.03446831926703453,
                0.004399140365421772,
                -0.0020972255151718855,
                -0.013066553510725498,
                0.03913925960659981,
                -0.0020234030671417713,
                0.024844571948051453,
                -0.0056977421045303345,
                -0.0017281138570979238,
                -0.034656230360269547,
                0.003164293710142374,
                -0.026777375489473343,
                0.012489397078752518,
                -0.004234717693179846,
                0.00964388158172369,
                0.03530050069093704,
                0.0053756083361804485,
                -0.012502819299697876,
                0.0011668963124975562,
                0.02501906082034111,
                -0.008972769603133202,
                0.005976253654807806,
                0.006580254528671503,
                -0.03454885259270668,
                -0.013865177519619465,
                -0.028723599389195442,
                -0.02664315328001976,
                -0.03221338242292404,
                0.035193122923374176,
                0.007556722965091467,
                0.006919166538864374,
                0.016254335641860962,
                0.0060836318880319595,
                -0.014576556161046028,
                0.023327859118580818,
                -0.003936072811484337,
                0.04273642227053642,
                0.00243110372684896,
                0.015972469002008438,
                0.028320932760834694,
                -0.008127167820930481,
                -0.03401196375489235,
                0.025676751509308815,
                0.00030074213282205164,
                0.004922607447952032,
                0.015462423674762249,
                -0.00739565584808588,
                -0.005546742118895054,
                -0.01912669651210308,
                0.0040870727971196175,
                0.010361971333622932,
                -0.023743947967886925,
                -0.05315208435058594,
                0.019153540953993797,
                -0.025690173730254173,
                0.008160723373293877,
                -0.019005896523594856,
                -0.023502347990870476,
                -0.005724586546421051,
                0.0016106691909953952,
                0.013730954378843307,
                0.011059928685426712,
                -0.0003577866591513157,
                -0.01472420059144497,
                -0.01437522191554308,
                0.008697613142430782,
                -0.020576298236846924,
                0.01634829118847847,
                0.00806005671620369,
                0.002744848607107997,
                0.006086987443268299,
                -0.014281266368925571,
                -0.01006668247282505,
                -0.00206366996280849,
                0.006231276318430901,
                0.0344146303832531,
                -0.011986062861979008,
                -0.036750100553035736,
                0.015234245918691158,
                -0.0053756083361804485,
                -0.023260748013854027,
                0.0010653905337676406,
                -0.012563219293951988,
                0.034656230360269547,
                -0.033099252730607986,
                -0.010200904682278633,
                0.00232372572645545,
                0.016173802316188812,
                0.018549539148807526,
                0.00666078832000494,
                -0.01022103801369667,
                -0.0013086687540635467,
                -0.013039709068834782,
                -0.020925277844071388,
                0.015865091234445572,
                0.006295031867921352,
                0.0068990327417850494,
                -0.0013313187519088387,
                -0.005127296783030033,
                0.002805248834192753,
                -0.021180300042033195,
                -0.006701054982841015,
                -0.0219185221940279,
                -0.016603315249085426,
                0.014818156138062477,
                -0.005358830559998751,
                -0.01834820583462715,
                0.014012821950018406,
                0.028159866109490395,
                -0.015126868151128292,
                -0.007831878960132599,
                0.03293818607926369,
                -0.017717361450195312,
                0.019341452047228813,
                0.004479673691093922,
                0.004422629252076149,
                -0.011757885105907917,
                0.005338697228580713,
                0.021193722262978554,
                -0.029099423438310623,
                0.0034629388246685266,
                -0.036052145063877106,
                -0.009596903808414936,
                0.002874037716537714,
                0.001696235965937376,
                0.011422328650951385,
                0.0018203917425125837,
                -0.011395484209060669,
                0.024186881259083748,
                0.012637041509151459,
                0.011133750900626183,
                -0.02648208476603031,
                -0.025609640404582024,
                0.03275027498602867,
                0.013200775720179081,
                -0.014240999706089497,
                0.015006068162620068,
                -0.009106991812586784,
                0.02097896672785282,
                -0.036454811692237854,
                0.010476061142981052,
                0.019770964980125427,
                -0.04571615904569626,
                -0.010019704699516296,
                -0.01371082104742527,
                0.010596861131489277,
                0.000424058991484344,
                0.002592170611023903,
                -0.011066639795899391,
                -0.0005436008214019239,
                -0.04120628535747528,
                0.015314779244363308,
                -0.00048110351781360805,
                -0.02307283505797386,
                -0.015046334825456142,
                -0.00041818676982074976,
                -0.015287934802472591,
                0.0030300712678581476,
                -0.01188539620488882,
                -0.01975754275918007,
                0.010214326903223991,
                -0.0202944315969944,
                -0.05497750639915466,
                -0.01602615788578987,
                -0.0009227792033925653,
                0.03205231577157974,
                0.014496022835373878,
                -0.02300572395324707,
                -0.028643066063523293,
                0.012838375754654408,
                -0.02301914617419243,
                -0.003046849276870489,
                -0.004181028809398413,
                0.0033958274871110916,
                0.0035166277084499598,
                0.020885011181235313,
                -0.008603657595813274,
                -0.016375137493014336,
                0.008664057590067387,
                0.0021861479617655277,
                -0.014455756172537804,
                0.0012851797509938478,
                0.006103765219449997,
                0.025059327483177185,
                0.02385132573544979,
                0.031085915863513947,
                -0.029958447441458702,
                -0.010523038916289806,
                0.01633486896753311,
                -0.016858337447047234,
                0.003506561042740941,
                -0.005429297219961882,
                -0.031327515840530396,
                -0.005996386986225843,
                -0.008019790053367615,
                -0.023878170177340508,
                0.005959475878626108,
                0.008388902060687542,
                -0.0056507643312215805,
                -0.0053756083361804485,
                -0.03269658237695694,
                0.008744591847062111,
                -0.013663843274116516,
                -0.03986406326293945,
                0.011207573115825653,
                -0.01215384155511856,
                0.013308154419064522,
                0.04222637787461281,
                -0.016254335641860962,
                -0.02402581460773945,
                -0.008617079816758633,
                -0.001504968968220055,
                0.010616994462907314,
                0.029851067811250687,
                0.004506518132984638,
                0.037528593093156815,
                -0.00046516460133716464,
                -0.015006068162620068,
                -0.015838246792554855,
                -0.0027414930518716574,
                -0.010429082438349724,
                0.028133021667599678,
                0.004187739919871092,
                0.006630588322877884,
                0.03723330423235893,
                -0.03556894510984421,
                0.018093183636665344,
                -0.027784043923020363,
                0.018549539148807526,
                -0.013737665489315987,
                -0.00039952146471478045,
                -0.01914011873304844,
                0.019623318687081337,
                0.012784686870872974,
                -0.02091185562312603,
                -0.02355603687465191,
                -0.00965059269219637,
                0.01022774912416935,
                0.00499643012881279,
                -0.02479088306427002,
                -0.009979438036680222,
                -0.028777290135622025,
                0.009549926035106182,
                0.0036575612612068653,
                -0.0036709834821522236,
                -0.04034726321697235,
                -0.006905743852257729,
                0.005130652338266373,
                -0.016589893028140068,
                0.02136821113526821,
                0.19242127239704132,
                -0.002301914617419243,
                0.014173888601362705,
                0.040857307612895966,
                0.008462724275887012,
                -0.004298473242670298,
                0.014710778370499611,
                -0.0006044203764759004,
                -0.002439492614939809,
                0.010590150021016598,
                0.02052260935306549,
                0.003530049929395318,
                0.01196592953056097,
                0.002729748608544469,
                0.01007339358329773,
                -0.02315337024629116,
                -0.005127296783030033,
                -0.00485885189846158,
                -0.011979351751506329,
                -0.008039923384785652,
                0.02719346433877945,
                -0.022334612905979156,
                0.02246883511543274,
                -0.018455583602190018,
                0.02014678716659546,
                -0.0025938483886420727,
                0.0009194236481562257,
                -0.004140762146562338,
                -0.003184427274391055,
                0.007590278517454863,
                -0.03052218072116375,
                0.0031793939415365458,
                0.0028723599389195442,
                -0.00871774647384882,
                -0.009100280702114105,
                -0.0005704453215003014,
                0.0202944315969944,
                0.01930118538439274,
                0.010898861102759838,
                0.022119857370853424,
                0.016214068979024887,
                -0.01479131169617176,
                0.03240129351615906,
                0.012294774875044823,
                -0.0036542057059705257,
                0.0028924935031682253,
                -0.0068184994161129,
                0.017220737412571907,
                0.005875586997717619,
                0.020858164876699448,
                -0.03170333802700043,
                0.040239885449409485,
                0.010939127765595913,
                0.010187482461333275,
                0.003338783048093319,
                -0.005217897240072489,
                -0.0020586366299539804,
                -0.009006325155496597,
                -0.011623662896454334,
                0.005852098111063242,
                -0.0337703637778759,
                0.025690173730254173,
                -0.010905572213232517,
                0.03385089710354805,
                -0.007422500289976597,
                0.04456184804439545,
                -0.014267844147980213,
                0.005835320334881544,
                0.015502690337598324,
                0.008939214050769806,
                -0.002206281293183565,
                0.002040180843323469,
                -0.01782473921775818,
                -0.008194279856979847,
                -0.00922108069062233,
                -0.011778018437325954,
                0.01803949475288391,
                0.012355174869298935,
                0.009422414936125278,
                0.01223437488079071,
                -0.005717875435948372,
                -0.013066553510725498,
                -0.010234460234642029,
                -0.0015376857481896877,
                0.00631516519933939,
                -0.019851498305797577,
                0.020187053829431534,
                -0.02091185562312603,
                -0.017865005880594254,
                -0.022656746208667755,
                -0.007952678948640823,
                0.027864577248692513,
                -0.02006625384092331,
                -0.017314692959189415,
                0.013811487704515457,
                0.009355303831398487,
                -0.0166435819119215,
                0.01795896142721176,
                -0.02084474265575409,
                -0.022052746266126633,
                -0.0018807918531820178,
                0.012610197067260742,
                0.007892278954386711,
                0.0054695638827979565,
                -0.009865348227322102,
                0.034360941499471664,
                -0.0049528079107403755,
                0.005281652789562941,
                0.00311731593683362,
                0.0017247581854462624,
                0.013187353499233723,
                -0.02246883511543274,
                -0.0003468811046332121,
                -0.013999399729073048,
                0.0067346105352044106,
                0.02237487956881523,
                0.03591792285442352,
                -0.015408734790980816,
                0.01951594091951847,
                -0.025757284834980965,
                0.010180771350860596,
                -0.03355560824275017,
                -0.0071808998472988605,
                0.009932460263371468,
                -0.004744763020426035,
                -0.02021389827132225,
                0.007744634058326483,
                0.019421985372900963,
                0.00015865510795265436,
                -0.04633358493447304,
                -0.0073553891852498055,
                -0.029475245624780655,
                0.019381718710064888,
                -0.011053217574954033,
                0.011375350877642632,
                -0.006248054094612598,
                -0.0037917837034910917,
                -0.01930118538439274,
                -0.031005380675196648,
                0.004523295909166336,
                -0.0005939342663623393,
                0.019717274233698845,
                0.005405808333307505,
                0.015690602362155914,
                0.014173888601362705,
                0.0010972684249281883,
                0.01464366726577282,
                -0.009113702923059464,
                -0.010100238025188446,
                0.015704024583101273,
                -0.013811487704515457,
                0.002405937062576413,
                -0.0002340503706363961,
                0.01930118538439274,
                0.009187525138258934,
                -0.018455583602190018,
                -0.013878599740564823,
                -0.02936786785721779,
                0.0010813294211402535,
                0.009127125144004822,
                -0.007341966964304447,
                0.006731254979968071,
                0.04211900010704994,
                -0.02099238894879818,
                -0.007617122959345579,
                0.00933517049998045,
                -0.17180471122264862,
                0.010523038916289806,
                0.01626775786280632,
                -0.010898861102759838,
                0.021744033321738243,
                0.01154312863945961,
                0.021005811169743538,
                -0.0040870727971196175,
                9.71539702732116e-05,
                0.004603829234838486,
                0.02571701817214489,
                0.02347550354897976,
                -0.003936072811484337,
                -0.00991232693195343,
                0.012066596187651157,
                0.005862164776772261,
                -0.02563648484647274,
                0.018952207639813423,
                -0.008086901158094406,
                0.012610197067260742,
                0.005415874999016523,
                -0.03181071579456329,
                0.0002797698834910989,
                -0.024267416447401047,
                0.000947107037063688,
                0.008167434483766556,
                0.01479131169617176,
                0.0054393638856709,
                -0.029314178973436356,
                -0.017475759610533714,
                -0.016697270795702934,
                0.004754829686135054,
                0.006597032304853201,
                -0.002078769961372018,
                0.010523038916289806,
                -0.0017935471842065454,
                -0.011220995336771011,
                -0.008919080719351768,
                -0.01960989646613598,
                0.026388129219412804,
                0.0043622287921607494,
                0.0007222844287753105,
                0.029877912253141403,
                0.0013069909764453769,
                -0.00631180964410305,
                -0.014200733043253422,
                0.014549711719155312,
                0.03154227137565613,
                0.018079761415719986,
                -0.02224065735936165,
                0.011167306452989578,
                -0.00019231557962484658,
                0.0021744032856076956,
                -0.01215384155511856,
                0.03828023746609688,
                0.017636828124523163,
                0.003377371933311224,
                0.01371082104742527,
                0.004486384801566601,
                -0.0034058941528201103,
                0.009751259349286556,
                -0.04206531122326851,
                0.00020164823217783123,
                0.014670511707663536,
                -0.02944840118288994,
                -0.023059412837028503,
                -0.010368682444095612,
                -0.004056872799992561,
                -0.01774420589208603,
                -0.010026415809988976,
                0.0009202625369653106,
                -0.023207059130072594,
                0.0021878257393836975,
                -0.017932116985321045,
                0.008113745599985123,
                0.01212028507143259,
                -0.001818713964894414,
                0.0016106691909953952,
                0.00025816846755333245,
                -0.005644053220748901,
                -0.009630459360778332,
                0.01581140235066414,
                -0.023354703560471535,
                -0.008033212274312973,
                0.03401196375489235,
                0.004271628800779581,
                0.01774420589208603,
                0.0008267262601293623,
                0.011710907332599163,
                0.004610540345311165,
                0.01720731519162655,
                -0.054574839770793915,
                -0.013489354401826859,
                -0.005063541233539581,
                0.02068367600440979,
                0.005761497654020786,
                -0.020576298236846924,
                -0.00964388158172369,
                0.017475759610533714,
                -0.011784729547798634,
                0.002919337945058942,
                0.0032028828281909227,
                -0.019878342747688293,
                0.02036154270172119,
                0.022817812860012054,
                -0.000618262100033462,
                0.0018136806320399046,
                0.0252472385764122,
                0.05296416953206062,
                -0.005754786543548107,
                -0.012227663770318031,
                0.007335255853831768,
                0.00300993793644011,
                -0.0004538396024145186,
                -0.0233010146766901,
                0.02044207602739334,
                -0.00972441490739584,
                -0.02068367600440979,
                0.013106820173561573,
                -0.019623318687081337,
                0.02797195501625538,
                0.0018824696308001876,
                0.026763953268527985,
                0.002107292180880904,
                -0.0003567380481399596,
                -0.015006068162620068,
                -0.1144648864865303,
                -0.024428483098745346,
                0.025676751509308815,
                0.035971611738204956,
                -0.034360941499471664,
                0.01096597220748663,
                0.019784387201070786,
                0.030038980767130852,
                -0.03401196375489235,
                0.01704624854028225,
                0.020549453794956207,
                -0.034441474825143814,
                0.01289206463843584,
                -0.016925448551774025,
                0.03524681180715561,
                0.0005440202658064663,
                -0.015529535710811615,
                -0.034199874848127365,
                0.007120499853044748,
                0.025676751509308815,
                -0.0036005168221890926,
                -0.00012405088637024164,
                0.0015980858588591218,
                -0.006227920763194561,
                -0.002424392616376281,
                0.009851926006376743,
                -0.014012821950018406,
                0.00875801406800747,
                0.021274255588650703,
                -0.01720731519162655,
                0.0033824052661657333,
                -0.026374706998467445,
                0.001719724852591753,
                0.009878771379590034,
                -0.027676666155457497,
                0.010187482461333275,
                -0.022804390639066696,
                -0.03607898950576782,
                0.025985462591052055,
                -0.01247597485780716,
                -0.006509787868708372,
                0.013771221041679382,
                -0.008180857636034489,
                -0.01022774912416935,
                -0.014066510833799839,
                -0.010019704699516296,
                -0.002424392616376281,
                0.012375308200716972,
                0.02114003337919712,
                -0.028320932760834694,
                -0.023730525746941566,
                -0.019784387201070786,
                0.0053990972228348255,
                -0.003483072156086564,
                0.017529450356960297,
                0.009039880707859993,
                -0.008435879833996296,
                0.007120499853044748,
                -0.02130110003054142,
                -0.005640697665512562,
                0.008355346508324146,
                0.040562018752098083,
                -0.0014798023039475083,
                0.027394797652959824,
                -0.003269993932917714,
                -0.009073436260223389,
                -0.023878170177340508,
                -0.03146173804998398,
                0.03852183744311333,
                -0.009093569591641426,
                -0.018187139183282852,
                0.013670554384589195,
                -0.008006367832422256,
                0.006295031867921352,
                -0.02323390357196331,
                0.005456141661852598,
                -0.014845000579953194,
                0.007248011417686939,
                0.010851883329451084,
                -0.039085570722818375,
                -0.00636885454878211,
                -0.02105950005352497,
                -0.011710907332599163,
                -0.006768166087567806,
                0.023180214688181877,
                -0.0005821898230351508,
                -0.002900882391259074,
                -0.016938870772719383,
                0.015408734790980816,
                -0.008033212274312973,
                -0.0064527434296905994,
                -0.017086515203118324,
                0.009710992686450481,
                -0.02301914617419243,
                -0.0058084758929908276,
                0.013422243297100067,
                0.005754786543548107,
                -0.018777718767523766,
                0.021475588902831078,
                0.005268230568617582,
                -0.01138877309858799,
                -0.015663757920265198,
                -0.05164879187941551,
                0.012187397107481956,
                -0.004768251907080412,
                -0.006435965653508902,
                0.018254250288009644,
                -0.00914054736495018,
                8.331228309543803e-05,
                -0.01196592953056097,
                0.011516284197568893,
                0.009892193600535393,
                -0.005264875013381243,
                0.025529107078909874,
                -0.02858937717974186,
                -0.015328201465308666,
                -0.006170876324176788,
                -0.0024361370597034693,
                0.026857908815145493,
                0.008274813182651997,
                0.019663585349917412,
                0.0199722982943058,
                -0.005436008330434561,
                -0.0038823839277029037,
                0.01203975174576044,
                0.029958447441458702,
                -0.018724029883742332,
                0.014804733917117119,
                -0.020549453794956207,
                0.013811487704515457,
                -0.012368597090244293,
                -0.005825253669172525,
                0.014308110810816288,
                -0.000771778984926641,
                -0.017140204086899757,
                0.02904573455452919,
                -0.006905743852257729,
                -0.02237487956881523,
                0.026669997721910477,
                0.021515855565667152,
                0.020468920469284058,
                0.03221338242292404,
                -0.025918351486325264,
                -0.02818671055138111,
                0.025582795962691307,
                -0.008013078942894936,
                -0.02556937374174595,
                0.011348506435751915,
                -0.018992474302649498,
                0.021650077775120735,
                0.03261604905128479,
                0.0011333406437188387,
                0.011690774001181126,
                0.008234546519815922,
                0.0018908585188910365,
                -0.027300842106342316,
                0.012986020185053349,
                -0.006825210526585579,
                0.0066809216514229774,
                0.016750959679484367,
                -0.013368554413318634,
                -0.011952507309615612,
                0.019583052024245262,
                0.04276326671242714,
                0.008804991841316223,
                -0.009113702923059464,
                -0.015100023709237576,
                -0.006214498542249203,
                -0.022978879511356354,
                0.012496108189225197,
                0.006348721217364073,
                -0.014697356149554253,
                -0.02105950005352497,
                -0.009013036265969276,
                0.0085633909329772,
                -0.005660830996930599,
                -0.001337190973572433,
                -0.0035568943712860346,
                -0.0073553891852498055,
                -0.004446118138730526,
                -0.027864577248692513,
                0.007341966964304447,
                0.0062782540917396545,
                0.024683505296707153,
                -0.025730440393090248,
                0.019636740908026695,
                0.0023941926192492247,
                0.01975754275918007,
                -0.036213211715221405,
                0.013811487704515457,
                0.018240828067064285,
                0.0004668423789553344,
                -0.008945925161242485,
                0.026589464396238327,
                0.014428911730647087,
                0.009791526012122631,
                -0.009831792674958706,
                0.030736936256289482,
                -0.009046591818332672,
                0.011590107344090939,
                0.03454885259270668,
                0.006140676327049732,
                0.004560207016766071,
                0.010455927811563015,
                0.01782473921775818,
                -0.018791140988469124,
                -0.022106435149908066,
                0.0024646592792123556,
                -0.0074090780690312386,
                -0.014912111684679985,
                -0.042870644479990005,
                0.00790570117533207,
                -0.00012132449774071574,
                0.00398640614002943,
                -0.016844915226101875,
                0.013462509959936142,
                -0.010939127765595913,
                -0.011301528662443161,
                0.004912540782243013,
                -0.027622975409030914,
                -0.048212695866823196,
                0.01499264594167471,
                0.021341366693377495,
                0.012771264649927616,
                0.01743549294769764,
                -0.017448915168642998,
                0.008992902934551239,
                0.024670083075761795,
                0.027864577248692513,
                -0.013247753493487835,
                0.03245498239994049,
                -0.0032767050433903933,
                0.029153112322092056,
                -0.004828651901334524,
                -0.021005811169743538,
                -0.006841988302767277,
                -0.0014034633059054613,
                -0.009073436260223389,
                0.017932116985321045,
                0.02115345560014248,
                -0.01289206463843584,
                0.06490996479988098,
                0.0003200366045348346,
                -0.005358830559998751,
                0.011455884203314781,
                -0.02555595152080059,
                0.02377079240977764,
                0.017005981877446175,
                0.009972726926207542,
                -0.03454885259270668,
                -0.015583224594593048,
                0.01107335090637207,
                -0.015086601488292217,
                -0.0018438806291669607,
                -0.016482515260577202,
                -0.03513943403959274,
                0.006187654100358486,
                -0.011348506435751915,
                0.017690517008304596,
                -0.005784987006336451,
                0.0006857927073724568,
                0.04687047377228737,
                -0.00825467985123396,
                0.037286993116140366,
                0.002941149054095149,
                -0.02006625384092331,
                -0.0027767266146838665,
                -0.0048722741194069386,
                -0.011590107344090939,
                0.0022482257336378098,
                -6.857927655801177e-05,
                0.00557694211602211,
                -0.005392386112362146,
                -0.02069709822535515,
                -0.01904616318643093,
                -0.00999286025762558,
                -0.0029830934945493937,
                0.00019598571816459298,
                -0.0171267818659544,
                0.004583695903420448,
                0.02534119412302971,
                0.012086729519069195,
                0.010697527788579464,
                -0.028830979019403458,
                -0.0014496023068204522,
                0.02224065735936165,
                0.01781131699681282,
                -0.03382405266165733,
                -0.010623705573379993,
                -0.025193549692630768,
            ],
            distance=0.7,
            limit=2,
        )

        for each_obj in response.objects:
            print(each_obj.properties)

    finally:
        client.close()  # Close client gracefully


# create()
# insert()
# query_data_bm25()
# query_fetch_object_by_id()
# query_fetch_objects()
# query_hybrid()
# query_near_object()
# query_data_near_text()
# query_data_near_vector()
