import requests as requests_http
from . import utils
from latitude.models import operations, shared
from typing import Optional

class Teams:
    _client: requests_http.Session
    _security_client: requests_http.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests_http.Session, security_client: requests_http.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version
        
    def get_team(self, security: operations.GetTeamSecurity) -> operations.GetTeamResponse:
        r"""Retrieve the Current Team
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/team'
        
        
        client = utils.configure_security_client(self._client, security)
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.GetTeamResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Teams])
                res.teams = out

        return res

    def patch_current_team(self, request: operations.PatchCurrentTeamRequest, security: operations.PatchCurrentTeamSecurity) -> operations.PatchCurrentTeamResponse:
        r"""Update a Team
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(base_url, '/team/{team_id}', request.path_params)
        
        headers = {}
        req_content_type, data, form = utils.serialize_request_body(request)
        if req_content_type not in ('multipart/form-data', 'multipart/mixed'):
            headers['content-type'] = req_content_type
        
        client = utils.configure_security_client(self._client, security)
        
        http_res = client.request('PATCH', url, data=data, files=form, headers=headers)
        content_type = http_res.headers.get('Content-Type')

        res = operations.PatchCurrentTeamResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[operations.PatchCurrentTeam200ApplicationJSON])
                res.patch_current_team_200_application_json_object = out
        elif http_res.status_code in [403, 404]:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.ErrorObject])
                res.error_object = out

        return res

    def post_team(self, request: operations.PostTeamRequest, security: operations.PostTeamSecurity) -> operations.PostTeamResponse:
        r"""Create a Team
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/team'
        
        headers = {}
        req_content_type, data, form = utils.serialize_request_body(request)
        if req_content_type not in ('multipart/form-data', 'multipart/mixed'):
            headers['content-type'] = req_content_type
        
        client = utils.configure_security_client(self._client, security)
        
        http_res = client.request('POST', url, data=data, files=form, headers=headers)
        content_type = http_res.headers.get('Content-Type')

        res = operations.PostTeamResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 201:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[operations.PostTeam201ApplicationJSON])
                res.post_team_201_application_json_object = out
        elif http_res.status_code in [406, 422]:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.ErrorObject])
                res.error_object = out

        return res

    