import requests as requests_http
from . import utils
from latitude.models import operations, shared
from typing import Optional

class Plans:
    _client: requests_http.Session
    _security_client: requests_http.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests_http.Session, security_client: requests_http.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version
        
    def get_plan(self, request: operations.GetPlanRequest, security: operations.GetPlanSecurity) -> operations.GetPlanResponse:
        r"""Retrieve a Plan
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(base_url, '/plans/{plan_id}', request.path_params)
        
        
        client = utils.configure_security_client(self._client, security)
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.GetPlanResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Plan])
                res.plan = out
        elif http_res.status_code == 404:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.ErrorObject])
                res.error_object = out

        return res

    def get_plans(self, request: operations.GetPlansRequest, security: operations.GetPlansSecurity) -> operations.GetPlansResponse:
        r"""List all Plans
        Lists all plans. Availability by region is included in the `data.attributes.available_in[*]`
        node for a given plan.
        
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/plans'
        
        query_params = utils.get_query_params(request.query_params)
        
        client = utils.configure_security_client(self._client, security)
        
        http_res = client.request('GET', url, params=query_params)
        content_type = http_res.headers.get('Content-Type')

        res = operations.GetPlansResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[operations.GetPlans200ApplicationJSON])
                res.get_plans_200_application_json_object = out

        return res

    