# lawkit-python Package Structure

This document describes the organization and structure of the lawkit-python package.

## Directory Structure

```
lawkit-python/
├── Cargo.toml                  # Rust binary configuration for maturin
├── pyproject.toml              # Project configuration and dependencies
├── README.md                   # Package documentation and usage examples
├── STRUCTURE.md               # This file - package structure documentation
├── src/                       # Source code
│   ├── lawkit/                # Python package directory
│   │   ├── __init__.py        # Package initialization and exports
│   │   ├── __main__.py        # Module entry point for python -m lawkit
│   │   └── lawkit.py          # Main wrapper implementation
│   ├── main.rs                # Rust CLI binary entry point
│   ├── lib.rs                 # Rust library configuration
│   ├── *.rs                   # Complete lawkit CLI implementation
│   └── subcommands/           # CLI subcommands implementation
├── test_manual.py             # Manual testing script
└── bin/                       # Binary output directory (maturin generated)
```

## File Descriptions

### `Cargo.toml`
- Rust binary configuration for maturin build system
- Defines lawkit CLI binary target
- Dependencies for complete CLI functionality
- Workspace isolation from main lawkit project

### `pyproject.toml`
- Project configuration using maturin build backend
- Metadata: version, description, author, license, keywords
- Dependencies and optional development dependencies
- Build system configuration (maturin with bindings = "bin")
- Tool configuration (ruff, mypy)
- Binary embedding configuration

### `README.md`
- Comprehensive documentation with installation instructions
- Usage examples for all major features
- API reference and supported platforms
- Use cases and real-world examples

### `src/lawkit/__init__.py`
- Package initialization and public API exports
- Version information
- Import statements for all public functions and classes

### `src/lawkit/lawkit.py`
- Main wrapper implementation around the lawkit CLI
- Core analysis functions for all statistical laws:
  - `analyze_benford()` - Benford's Law analysis
  - `analyze_pareto()` - Pareto principle analysis
  - `analyze_zipf()` - Zipf's Law analysis
  - `analyze_normal()` - Normal distribution analysis
  - `analyze_poisson()` - Poisson distribution analysis
  - `compare_laws()` - Multi-law comparison
- Utility functions:
  - `generate_data()` - Sample data generation
  - `analyze_string()` - Direct string analysis
  - `is_lawkit_available()` - Binary availability check
  - `get_version()` - Version information
  - `selftest()` - Self-test functionality
- Classes:
  - `LawkitOptions` - Configuration options
  - `LawkitResult` - Analysis results
  - `LawkitError` - Error handling

### `src/lawkit/__main__.py`
- Python module entry point for `python -m lawkit` usage
- Direct command forwarding to embedded binary
- Seamless CLI access from Python environment
- Cross-platform binary execution handling

### `src/*.rs` and `src/subcommands/`
- Complete Rust implementation of lawkit CLI
- All statistical law analysis functionality
- Subcommands: analyze, validate, diagnose, benf, pareto, zipf, normal, poisson
- Generate functionality for sample data creation
- Compiled into binary and embedded in Python wheel

### `src/lawkit/compat.py`
- Backward compatibility functions for legacy users
- `run_lawkit()` - Direct command execution
- `run_benford_analysis()` - Legacy Benford analysis
- `run_pareto_analysis()` - Legacy Pareto analysis
- `check_lawkit_installation()` - Installation verification
- `get_lawkit_help()` - Help text retrieval

### `bin/`
- Binary output directory (generated by maturin)
- Contains compiled lawkit executable
- Automatically embedded in Python wheel during build

## Package Features

### Multi-Platform Support
- Windows (x86_64)
- macOS (x86_64, ARM64/Apple Silicon)
- Linux (x86_64, ARM64)

### Statistical Laws
- **Benford's Law**: Fraud detection and anomaly analysis
- **Pareto Principle**: 80/20 rule analysis and concentration measurement
- **Zipf's Law**: Power-law distribution analysis
- **Normal Distribution**: Normality testing and outlier detection
- **Poisson Distribution**: Rare event analysis

### File Format Support
- CSV, JSON, YAML, TOML, XML
- Excel files (.xlsx, .xls)
- PDF documents
- Word documents (.docx)
- PowerPoint presentations (.pptx)

### Advanced Features
- International number format support
- Time series analysis
- Parallel processing
- Memory-efficient streaming
- Outlier detection algorithms
- Multi-law comparison and ranking

## API Design

### Modern API Pattern
```python
import lawkit

# Simple analysis
result = lawkit.analyze_benford('data.csv')

# Advanced analysis with options
result = lawkit.analyze_benford(
    'data.csv',
    lawkit.LawkitOptions(
        format='csv',
        output='json',
        verbose=True,
        optimize=True
    )
)

# Structured result access
print(f"Risk level: {result.risk_level}")
print(f"P-value: {result.p_value}")
```

### Legacy API Pattern
```python
from lawkit.compat import run_lawkit

# Direct command execution
result = run_lawkit(['benf', 'data.csv', '--format', 'csv'])
```

## Installation Methods

### Automatic (Recommended)
```bash
pip install lawkit-python
```

### Module Usage
```bash
# Direct CLI access via Python module
python -m lawkit benf data.csv
python -m lawkit analyze --laws all data.csv
```

### Development Installation
```bash
git clone https://github.com/kako-jun/lawkit
cd lawkit/lawkit-python
pip install -e .[dev]
```

## Testing and Verification

### Package Build Test
```bash
# Build package with maturin
pip install maturin
maturin develop
```

### Manual Functionality Test
```bash
python test_manual.py
```

### Import Test
```python
import lawkit
print(lawkit.__version__)
print(lawkit.is_lawkit_available())
```

## Distribution

The package is designed for distribution via:
- PyPI (Python Package Index)
- GitHub Releases
- Direct installation from source

## Dependencies

### Runtime Dependencies
- Python 3.8+
- No external dependencies (all included)

### Development Dependencies
- pytest >= 6.0
- pytest-cov
- pytest-asyncio
- black (code formatting)
- isort (import sorting)
- mypy (type checking)
- ruff (linting)
- types-requests (type hints)

## Version Synchronization

The package version (2.1.0) is synchronized with the main lawkit project to ensure compatibility between the Python wrapper and the underlying CLI tool.