# FastAPI Scaffold - 企业级脚手架项目

一套基于 FastAPI 的企业级脚手架项目，旨在简化开发过程，提供完整的后端开发解决方案。

## ✨ 特性

- 🚀 **路由自动注册** - 自动扫描并注册 API 路由
- 🔧 **多环境配置** - 支持开发、生产、测试环境配置
- 📝 **日志系统** - 支持控制台与文件双输出，可配置日志级别和存储
- ⚡ **统一异常处理** - 完善的异常处理机制和错误日志记录
- 📊 **统一响应格式** - 标准化的 API 响应格式，自动包装处理
- 🗄️ **多数据源支持** - 基于 SQLAlchemy，支持配置多个数据源
- 🏗️ **分层架构** - 清晰的代码分层：API 层、服务层、仓储层、模型层
- 💾 **缓存支持** - 基于 Redis 的缓存系统，可配置开启/关闭
- 🔒 **安全特性** - 密码加密、数据验证、CORS 配置

## 📁 项目结构

```\nlesscode-fastapi/\n├── app/                    # 应用核心目录\n│   ├── api/               # API 路由层（控制器）\n│   │   └── user.py        # 用户 API 示例\n│   ├── core/              # 核心功能模块\n│   │   ├── cache.py       # 缓存管理\n│   │   ├── database.py    # 数据库连接\n│   │   ├── exceptions.py  # 自定义异常\n│   │   ├── exception_handlers.py # 异常处理器\n│   │   ├── logging.py     # 日志配置\n│   │   └── router.py      # 路由注册器\n│   ├── models/            # 数据模型层\n│   │   ├── base.py        # 基础模型\n│   │   └── user.py        # 用户模型示例\n│   ├── repositories/      # 数据访问层（仓储）\n│   │   ├── base.py        # 基础仓储类\n│   │   └── user.py        # 用户仓储示例\n│   ├── schemas/           # 数据验证模式\n│   │   ├── response.py    # 响应模式\n│   │   └── user.py        # 用户模式示例\n│   ├── services/          # 业务逻辑层\n│   │   └── user.py        # 用户服务示例\n│   └── utils/             # 工具函数\n│       └── response.py    # 响应工具\n├── config/                # 配置模块\n│   └── settings.py        # 配置管理\n├── logs/                  # 日志目录\n├── tests/                 # 测试目录\n├── scripts/               # 脚本目录\n├── .env.development       # 开发环境配置\n├── .env.production        # 生产环境配置\n├── .env.testing           # 测试环境配置\n├── main.py                # 应用入口\n├── run.py                 # 启动脚本\n└── requirements.txt       # 项目依赖\n```

## 🚀 快速开始

### 1. 安装依赖\n\n```bash\npip install -r requirements.txt\n```\n\n### 2. 配置环境\n\n复制并修改环境配置文件：\n\n```bash\n# 开发环境\ncp .env.development .env\n\n# 或根据需要选择其他环境\ncp .env.production .env\ncp .env.testing .env\n```\n\n### 3. 启动应用\n\n#### 使用启动脚本（推荐）\n\n```bash\n# 开发环境启动（默认）\npython run.py\n\n# 指定环境启动\npython run.py --env development\npython run.py --env production\npython run.py --env testing\n\n# 自定义参数启动\npython run.py --host 0.0.0.0 --port 8080 --reload\n```\n\n#### 直接使用 uvicorn\n\n```bash\nuvicorn main:app --reload --host 0.0.0.0 --port 8000\n```\n\n### 4. 访问应用\n\n- API 文档：http://localhost:8000/docs\n- ReDoc 文档：http://localhost:8000/redoc\n- 健康检查：http://localhost:8000/health\n\n## ⚙️ 配置说明\n\n### 环境配置文件\n\n项目支持三种环境配置：\n\n- `.env.development` - 开发环境\n- `.env.production` - 生产环境  \n- `.env.testing` - 测试环境\n\n### 主要配置项\n\n```bash\n# 应用配置\nAPP_NAME=FastAPI Scaffold\nDEBUG=true\nENVIRONMENT=development\nHOST=0.0.0.0\nPORT=8000\n\n# 数据库配置\nDATABASE_URL=sqlite:///./dev.db\nDATABASE_POOL_SIZE=10\n\n# 多数据源配置（可选）\nDATABASE_URLS=main=sqlite:///./main.db,analytics=sqlite:///./analytics.db\n\n# Redis 缓存配置\nREDIS_URL=redis://localhost:6379/0\nREDIS_ENABLED=false\n\n# 日志配置\nLOG_LEVEL=DEBUG\nLOG_FILE=logs/app.log\nLOG_ROTATION=10 MB\nLOG_RETENTION=10 days\n\n# JWT 配置\nSECRET_KEY=your-secret-key\nACCESS_TOKEN_EXPIRE_MINUTES=30\n```\n\n## 🏗️ 架构设计\n\n### 分层架构\n\n1. **API 层 (app/api/)** - 路由和请求处理\n2. **服务层 (app/services/)** - 业务逻辑处理\n3. **仓储层 (app/repositories/)** - 数据访问抽象\n4. **模型层 (app/models/)** - 数据库模型定义\n\n### 数据流\n\n```\nAPI 层 → 服务层 → 仓储层 → 模型层 → 数据库\n  ↑        ↑        ↑        ↑\n请求处理  业务逻辑  数据访问  数据模型\n```\n\n## 💾 数据库使用\n\n### 单数据源\n\n```python\nfrom app.core.database import get_db\nfrom sqlalchemy.orm import Session\n\ndef some_function(db: Session = Depends(get_db)):\n    # 使用默认数据库\n    pass\n```\n\n### 多数据源\n\n```python\nfrom app.core.database import get_db\n\ndef some_function():\n    # 使用指定数据库\n    with get_db_context(\"analytics\") as db:\n        # 数据库操作\n        pass\n```\n\n## 🔄 缓存使用\n\n### 启用缓存\n\n在环境配置中设置：\n```bash\nREDIS_ENABLED=true\nREDIS_URL=redis://localhost:6379/0\n```\n\n### 使用缓存装饰器\n\n```python\nfrom app.core.cache import cached, cache_clear\n\n@cached(key_prefix=\"user\", expire=3600)\nasync def get_user_data(user_id: int):\n    # 数据获取逻辑\n    return user_data\n\n@cache_clear([\"user:*\"])\nasync def update_user_data(user_id: int, data):\n    # 数据更新逻辑\n    pass\n```\n\n## 📝 添加新功能\n\n### 1. 创建模型\n\n```python\n# app/models/your_model.py\nfrom sqlalchemy import Column, String\nfrom app.models.base import BaseModel\n\nclass YourModel(BaseModel):\n    __tablename__ = \"your_table\"\n    \n    name = Column(String(100), nullable=False)\n```\n\n### 2. 创建 Schema\n\n```python\n# app/schemas/your_schema.py\nfrom pydantic import BaseModel\n\nclass YourSchema(BaseModel):\n    name: str\n    \n    class Config:\n        from_attributes = True\n```\n\n### 3. 创建仓储\n\n```python\n# app/repositories/your_repository.py\nfrom app.repositories.base import BaseRepository\nfrom app.models.your_model import YourModel\n\nclass YourRepository(BaseRepository[YourModel]):\n    def __init__(self, database_name: str = \"default\"):\n        super().__init__(YourModel, database_name)\n```\n\n### 4. 创建服务\n\n```python\n# app/services/your_service.py\nfrom app.repositories.your_repository import YourRepository\n\nclass YourService:\n    def __init__(self, database_name: str = \"default\"):\n        self.repo = YourRepository(database_name)\n    \n    async def get_data(self, id: int):\n        return self.repo.get(id)\n```\n\n### 5. 创建 API\n\n```python\n# app/api/your_api.py\nfrom fastapi import APIRouter, Depends\nfrom app.services.your_service import YourService\n\nrouter = APIRouter()\nPREFIX = \"/your-resource\"\nTAGS = [\"Your Resource\"]\n\n@router.get(\"/{id}\")\nasync def get_item(id: int, service: YourService = Depends()):\n    return await service.get_data(id)\n```\n\n## 🔍 日志使用\n\n```python\nfrom app.core.logging import get_logger\n\nlogger = get_logger(__name__)\n\nlogger.info(\"信息日志\")\nlogger.error(\"错误日志\")\nlogger.warning(\"警告日志\")\nlogger.debug(\"调试日志\")\n```\n\n## 🚦 异常处理\n\n```python\nfrom app.core.exceptions import (\n    BusinessException,\n    ValidationException,\n    ResourceNotFoundException\n)\n\n# 业务异常\nraise BusinessException(\"业务逻辑错误\")\n\n# 验证异常\nraise ValidationException(\"数据验证失败\")\n\n# 资源未找到\nraise ResourceNotFoundException(\"资源不存在\")\n```\n\n## 📊 响应格式\n\n所有 API 响应都遵循统一格式：\n\n```json\n{\n  \"success\": true,\n  \"message\": \"操作成功\",\n  \"code\": 200,\n  \"data\": {},\n  \"timestamp\": 1640995200\n}\n```\n\n分页响应格式：\n\n```json\n{\n  \"success\": true,\n  \"message\": \"操作成功\",\n  \"code\": 200,\n  \"data\": [],\n  \"page_info\": {\n    \"page\": 1,\n    \"page_size\": 20,\n    \"total\": 100,\n    \"total_pages\": 5,\n    \"has_next\": true,\n    \"has_prev\": false\n  },\n  \"timestamp\": 1640995200\n}\n```\n\n## 🧪 测试\n\n```bash\n# 安装测试依赖\npip install pytest pytest-asyncio httpx\n\n# 运行测试\npytest tests/\n\n# 运行测试并查看覆盖率\npytest tests/ --cov=app\n```\n\n## 📋 开发规范\n\n### 代码组织\n\n1. **模型层** - 只定义数据结构，不包含业务逻辑\n2. **仓储层** - 只处理数据访问，不包含业务逻辑\n3. **服务层** - 包含核心业务逻辑\n4. **API 层** - 只处理请求响应，调用服务层\n\n### 命名规范\n\n- 文件名：小写下划线命名 `user_service.py`\n- 类名：大驼峰命名 `UserService`\n- 函数名：小写下划线命名 `get_user_by_id`\n- 常量：大写下划线命名 `DEFAULT_PAGE_SIZE`\n\n### 错误处理\n\n- 使用自定义异常类\n- 记录详细的错误日志\n- 返回用户友好的错误信息\n- 不在生产环境暴露系统内部信息\n\n## 🚀 部署\n\n### 生产环境部署\n\n```bash\n# 使用 gunicorn\ngunicorn main:app -w 4 -k uvicorn.workers.UvicornWorker\n\n# 或使用启动脚本\npython run.py --env production --workers 4\n```\n\n### 环境变量\n\n生产环境请确保设置以下环境变量：\n\n- `ENVIRONMENT=production`\n- `SECRET_KEY=强密钥`\n- `DATABASE_URL=生产数据库URL`\n- `REDIS_URL=生产Redis URL`\n\n## 🤝 贡献\n\n欢迎提交 Issue 和 Pull Request 来改进这个项目。\n\n## 📄 许可证\n\n[MIT License](LICENSE)\n\n---\n\n**FastAPI Scaffold** - 让后端开发更简单高效！