#
# Copyright (c) 2019. JetBrains s.r.o.
# Use of this source code is governed by the MIT license that can be found in the LICENSE file.
#
from pkgutil import extend_path
from typing import Dict, Union

# To handle the situation when the 'lets_plot' package is shared by modules in different locations.
__path__ = extend_path(__path__, __name__)

from ._version import __version__
from ._global_settings import _settings, is_production, get_global_bool
from ._global_settings import NO_JS, OFFLINE

from .plot import *
from .export import *
from .frontend_context import *
from .mapping import *
from .settings_utils import *
from .plot._global_theme import _set_global_theme

__all__ = (plot.__all__ +
           frontend_context.__all__ +
           mapping.__all__ +
           settings_utils.__all__ +
           export.__all__ +
           ['LetsPlot'])

from .frontend_context import _configuration as cfg


class LetsPlot:
    """
    Initialize the library and its options.
    """

    @classmethod
    def setup_html(cls, *,
                   isolated_frame: bool = None,
                   offline: bool = None,
                   no_js: bool = None,
                   show_status: bool = False) -> None:
        """
        Configure Lets-Plot HTML output.
        Depending on the usage, LetsPlot generates different HTML to show plots.
        In most cases LetsPlot will detect type of the environment automatically.
        Auto-detection can be overwritten using this method parameters.

        Parameters
        ----------
        isolated_frame : bool
            True - generate HTLM which can be used in iframe or in a standalone HTML document.
            False - pre-load Lets-Plot JS library. Notebook cell output will only consist
            of HTML for the plot rendering. Default: None - auto-detect.
        offline : bool
            True - full Lets-Plot JS bundle will be added to the notebook.
            Use this option if you would like to work with notebook
            without the Internet connection. False - load Lets-Plot JS library from CDN.
            Default (None): 'connected' mode in production environment
            and 'offline' mode in dev environment.
        no_js : bool, default=False
            True - do not generate HTML+JS as an output - just static SVG image.
            Note that without JS interactive maps and tooltips doesn't work!
        show_status : bool, default=False
            Whether to show status of loading of the Lets-Plot JS library.
            Only applicable when the Lets-Plot JS library is preloaded.

        Examples
        --------
        .. jupyter-execute::
            :linenos:
            :emphasize-lines: 2

            from lets_plot import *
            LetsPlot.setup_html()
            ggplot({'x': [0], 'y': [0]}, aes('x', 'y')) + geom_point()

        |

        .. jupyter-execute::
            :linenos:
            :emphasize-lines: 2-3

            from lets_plot import *
            LetsPlot.setup_html(isolated_frame=False, offline=True, \\
                                no_js=True, show_status=True)
            ggplot({'x': [0], 'y': [0]}, aes('x', 'y')) + geom_point()

        """
        if not (isinstance(isolated_frame, bool) or isolated_frame is None):
            raise ValueError("'isolated' argument is not boolean: {}".format(type(isolated_frame)))
        if not (isinstance(offline, bool) or offline is None):
            raise ValueError("'offline' argument is not boolean: {}".format(type(offline)))
        if not (isinstance(no_js, bool) or no_js is None):
            raise ValueError("'no_js' argument is not boolean: {}".format(type(no_js)))
        if not isinstance(show_status, bool):
            raise ValueError("'show_status' argument is not boolean: {}".format(type(show_status)))

        offline = offline if offline is not None else get_global_bool(OFFLINE)
        no_js = no_js if no_js is not None else get_global_bool(NO_JS)

        cfg._setup_html_context(isolated_frame=isolated_frame,
                                offline=offline,
                                no_js=no_js,
                                show_status=show_status)

    @classmethod
    def set(cls, settings: Dict):
        """
        Set up library options.
        For more info see `Configuring Globally <https://lets-plot.org/python/pages/basemap_tiles.html#configuring-globally>`__.

        Parameters
        ----------
        settings : dict
            Dictionary of settings.

        Notes
        -----
        List of possible settings:

        - html_isolated_frame : preload Lets-Plot JS library or not (bool). Do not use this parameter explicitly. Instead you should call `LetsPlot.setup_html()`.
        - offline : to work with notebook without the Internet connection (bool). Do not use this parameter explicitly. Instead you should call `LetsPlot.setup_html()`.
        - no_js : do not generate HTML+JS as an output (bool). Do not use this parameter explicitly. Instead you should call `LetsPlot.setup_html()`. Also note that without JS interactive maps and tooltips doesn't work!

        Interactive map settings could also be specified:

        - maptiles_kind : kind of the tiles, could be 'raster_zxy' or 'vector_lets_plot'. Do not use this parameter explicitly. Instead you should construct it with functions `maptiles_zxy()` and `maptiles_lets_plot()`.
        - maptiles_url : address of the tile server (str). Do not use this parameter explicitly. Instead you should construct it with functions `maptiles_zxy()` and `maptiles_lets_plot()`.
        - maptiles_theme : tiles theme, could be 'color', 'light' or 'dark'. Do not use this parameter explicitly. Instead you should construct it with function `maptiles_lets_plot()`.
        - maptiles_attribution : an attribution or a copyright notice to display on the map as required by the tile license (str, supports HTML links). Do not use this parameter explicitly. Instead you should construct it with function `maptiles_zxy()`.
        - maptiles_min_zoom : minimal zoom limit (int). Do not use this parameter explicitly. Instead you should construct it with function `maptiles_zxy()`.
        - maptiles_max_zoom : maximal zoom limit (int). Do not use this parameter explicitly. Instead you should construct it with function `maptiles_zxy()`.

        Examples
        --------
        .. jupyter-execute::
            :linenos:
            :emphasize-lines: 4

            from lets_plot import *
            from lets_plot import tilesets
            LetsPlot.setup_html()
            LetsPlot.set(tilesets.LETS_PLOT_LIGHT)
            ggplot() + geom_livemap()

        |

        .. jupyter-execute::
            :linenos:
            :emphasize-lines: 4

            from lets_plot import *
            from lets_plot import tilesets
            LetsPlot.setup_html()
            LetsPlot.set(tilesets.LETS_PLOT_BW)
            ggplot() + geom_livemap()

        """
        if is_production():
            _settings.update(settings)
        else:
            _settings.update({'dev_' + key: value for key, value in settings.items()})

    @classmethod
    def set_theme(cls, theme: Union['core.FeatureSpec', 'core.FeatureSpecArray']):
        """
        Set up global theme.

        Parameters
        ----------
        theme : spec
            Theme spec provided by `theme(...)`, `theme_xxx()`, `flavor_xxx()` functions, or their sum.

        """
        if theme is None:
            _set_global_theme(None)
            return

        if theme.kind != 'theme' and not (theme.kind == 'feature-list' and all(f.kind == 'theme' for f in theme)):
            raise ValueError("Only `theme(...)`, `theme_xxx()`, `flavor_xxx()`, or a sum of them are supported")

        _set_global_theme(theme)

    @classmethod
    def setup_show_ext(cls, *,
                       exec: str = None,
                       new: bool = False) -> None:
        """
        Configure Lets-Plot to show its HTML output in an external browser.

        When the "show externally" is set up, an invocation of `figire.show()` will
        - generate HTML output
        - save it to a temporary file
        - open the file in the default web browser or in a web browser specified by the `exec` parameter.

        Parameters
        ----------
        exec : str, optional
            Specify an app to open the generated temporary HTML file.
            If not specified, the default browser will be used.
        new : bool, default=False
            If True, the URL is opened in a new window of the web browser.
            If False, the URL is opened in the already opened web browser window.
            The `new` parameter is only applicable when the `exec` parameter is not specified.
            Please note that the `new` parameter is not supported by all web browsers and all OS-s.

        Examples
        --------
        .. code-block::
            :linenos:
            :emphasize-lines: 3

            # Show the plot in the default web browser.
            from lets_plot import *
            LetsPlot.setup_show_ext()
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        |

        .. code-block::
            :linenos:
            :emphasize-lines: 3

            # Show the plot in the new window of the default web browser if possible.
            from lets_plot import *
            LetsPlot.setup_show_ext(new=True)
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        |

        .. code-block::
            :linenos:
            :emphasize-lines: 4

            # Show the plot in the Chrome web browser for Windows.
            # This is the default setup path. Replace the file path with your own if it differs.
            from lets_plot import *
            LetsPlot.setup_show_ext(exec='C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe --app=%s')
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        |

        .. code-block::
            :linenos:
            :emphasize-lines: 3

            # Show the plot in the Safari web browser for macOS.
            from lets_plot import *
            LetsPlot.setup_show_ext(exec='open -a safari %s')
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        |

        .. code-block::
            :linenos:
            :emphasize-lines: 4

            # Show the plot in the Chrome web browser for macOS in the application mode.
            # This is the default setup path. Replace the path with your own if it differs.
            from lets_plot import *
            LetsPlot.setup_show_ext(exec='/Applications/Google\\ Chrome.app/Contents/MacOS/Google\\ Chrome --app=%s')
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        |

        .. code-block::
            :linenos:
            :emphasize-lines: 3

            # Show the plot in the Chrome web browser for Linux.
            from lets_plot import *
            LetsPlot.setup_show_ext(exec='google-chrome --app=%s')
            p = ggplot() + geom_point(x=0, y=0)
            p.show()

        """
        cfg._setup_wb_html_context(exec=exec, new=new)
