# 艾默生OIF平台运行主程序
from typing import Optional
from yangke.pytorch.mytorch import DataFitterNet, DataSetFitting

# Import library declarations here
try:
    # noinspection all
    from oif.pisystem_python import SystemLib
    # noinspection all
    from oif.pimath_python import MathLib
    # noinspection all
    from oif.pipoint_python import PointLib
    # noinspection all
    from oif.piexecution_python import ExecutionLib
    # noinspection all
    from oif.pimemory_python import MemoryLib
except:
    from fake_OIF import SystemLib, MathLib, PointLib, ExecutionLib, MemoryLib
import pandas as pd
from yangke.performance.iapws97 import H_PT
from yangke.common.config import logger, add_file_logger


class main_OIF:  # 该类名必须和文件名相同
    class Calculate:
        def __init__(self):
            add_file_logger('log-cal-tpri-cold.txt')
            self.model = DataFitterNet.load_yk(r"背压预测模型.torch")
            # 准备返回指定列的数据
            self.input_columns = [
                "汽轮发电机功率", "凝汽器热负荷", "风机运行台数", "循泵运行方式数字编码",
                "平均大气压力", "环境温度", "大气湿度", "风速"
            ]
            self.循泵运行方式映射 = {
                "000": 0,
                "001": 1,  # 1台低速小泵
                "010": 2,  # 1台高速小泵
                "100": 3,  # 1台大泵
                "020": 4,  # 2台高速小泵
                "110": 5,  # 一大一小高速
                "200": 6,  # 2台大泵
                "120": 7,  # 一大两小高速（不常用）
                "210": 8,  # 两大一小高速（不常用）
            }
            self.tags = {
                "1号锅炉高压主汽流量": {"KKS": "11LBA90CF101-SEL", "default": 296.644},
                "2号锅炉高压主汽流量": {"KKS": "12LBA90CF101-SEL", "default": 295.641},
                "高压缸进汽压力": {"KKS": "13LBA40CP101", "default": 11.835},
                "高压缸进汽温度": {"KKS": "13LBA40CT601", "default": 537.059},
                "高排压力": {"KKS": "13LBC01CP101", "default": 3.352},
                "高排温度": {"KKS": "13LBC01CT601", "default": 359.035},
                "高排至1号炉流量": {"KKS": "11LBC80CF101-SEL", "default": 308.141},
                "高排至2号炉流量": {"KKS": "12LBC80CF101-SEL", "default": 333.668},
                "1号炉高旁减温水流量": {"KKS": "11LAE93CF101-CAL", "default": 0.000},
                "2号炉高旁减温水流量": {"KKS": "12LAE93CF101-CAL", "default": 2.906},
                "再热蒸汽母管压力": {"KKS": "13LBB40CP101", "default": 3.084},
                "再热蒸汽母管温度": {"KKS": "13LBB40CT601", "default": 566.345},
                "1号炉再热蒸汽流量": {"KKS": "11LBB80CF101-CAL", "default": 384.809},
                "2号炉再热蒸汽流量": {"KKS": "12LBX80CF101-SEL", "default": 409.371},
                "低压主蒸汽压力(中排混合前)": {"KKS": "13LBA60CP101", "default": 0.555},
                "低压主蒸汽温度(中排混合前)": {"KKS": "13LBA60CT601", "default": 247.790},
                "低压主汽至热网抽汽压力(中排混合后)": {"KKS": "10LBD92CP101", "default": 0.464},
                "低压主汽至热网抽汽温度(中排混合后)": {"KKS": "10LBD92CT601", "default": 314.482},
                "低压主汽至热网调门反馈(分支管)": {"KKS": "10LBA61AA102XB12", "default": 5.000},
                "低压主汽母管至热网调门": {"KKS": "13RCAOSE2502", "default": 4.719},
                "1号炉再热蒸汽减温水调门反馈": {"KKS": "11LAF80AA101XB12", "default": 0.093},
                "1号炉过热蒸汽减温水调门": {"KKS": "11LAE90AA101XB12", "default": 38.202},
                "1号机高压汽包连排调门": {"KKS": "11HAD95AA101XB12", "default": 3.050},
                "1号机中压汽包连排调门": {"KKS": "11HAD85AA101XB12", "default": 3.000},
                "1号机低压汽包连排调门": {"KKS": "11HAD75AA101XB12", "default": 3.080},
                "1号炉过热蒸汽减温水流量": {"KKS": "11LAE90CF101-COMP", "default": 5.594},
                "1号炉再热蒸汽减温水流量": {"KKS": "11LAF80CF101-COMP", "default": 0.000},
                "2号炉再热蒸汽减温水调门反馈": {"KKS": "12LAF80AA101XB12", "default": -0.090},
                "2号炉过热蒸汽减温水调门": {"KKS": "12LAE90AA101XB12", "default": 51.942},
                "2号机高压汽包连排调门": {"KKS": "12HAD95AA101XB12", "default": 3.040},
                "2号机中压汽包连排调门": {"KKS": "12HAD85AA101XB12", "default": 3.200},
                "2号机低压汽包连排调门": {"KKS": "12HAD75AA101XB12", "default": 3.100},
                "2号炉过热蒸汽减温水流量": {"KKS": "12LAE90CF101-COMP", "default": 4.254},
                "2号炉再热蒸汽减温水流量": {"KKS": "12LAF80CF101-COMP", "default": 0.000},
                "瞬时热网供水热量": {"KKS": "RWJL1-07", "default": 1207.560},
                "瞬时热网回水热量": {"KKS": "RWJL1-08", "default": 912.556},
                "热网瞬时供水流量": {"KKS": "RWJL1-09", "default": 3604.886},
                "热网瞬时回水流量": {"KKS": "RWJL1-10", "default": 3600.650},
                "热网供水压力": {"KKS": "10NDA60CP101", "default": 0.642},
                "热网回水压力": {"KKS": "10NDB10CP101", "default": 0.259},
                "热网供水温度": {"KKS": "RWJL1-11", "default": 79.755},
                "热网回水温度": {"KKS": "RWJL1-12", "default": 60.647},
                "1号炉烟气供热器供热量": {"KKS": "11LCC71CQ101", "default": 68.988},
                "2号炉烟气供热器供热量": {"KKS": "12LCC71CQ101", "default": 60.147},
                "凝汽器真空选择值": {"KKS": "10MAG10CP101-SEL", "default": 7.436},
                "凝汽器真空1": {"KKS": "10MAG10CP101", "default": 7.501},
                "凝汽器真空2": {"KKS": "10MAG10CP102", "default": 7.364},
                "汽轮发电机功率": {"KKS": "10CJA07FU101", "default": 279.990},
                "燃机功率1": {"KKS": "11GT-MW", "default": 305.151},
                "燃机功率2": {"KKS": "12GT-MW", "default": 303.749},
                "汽机循环水供水管道流量1": {"KKS": "10PAC11CF101", "default": 17362.748},
                "汽机循环水供水管道流量2": {"KKS": "10PAC11CF102", "default": 14625.702},
                "1号大循泵电流（额定298A）": {"KKS": "10PAC11AP001XQ01", "default": 235.388},
                "2号大循泵电流（额定298A）": {"KKS": "10PAC12AP001XQ01", "default": -0.787},
                "3号小循泵电流（高速）": {"KKS": "10PAC13AP001XQ01", "default": -0.499},
                "3号小循泵电流（低速）": {"KKS": "10PAC13AP001XQ02", "default": -0.181},
                "4号小循泵电流（高速）": {"KKS": "10PAC14AP001XQ01", "default": -0.224},
                "4号小循泵电流（低速）": {"KKS": "10PAC14AP001XQ02", "default": 0.002},
                "凝汽器进水压力1": {"KKS": "13MAG01CP103", "default": 0.158},
                "凝汽器进水压力2": {"KKS": "13MAG01CP105", "default": 0.158},
                "凝汽器进水温度1": {"KKS": "13MAG01CT303", "default": 26.286},
                "凝汽器进水温度2": {"KKS": "13MAG01CT305", "default": 26.097},
                "凝汽器回水温度1": {"KKS": "13MAG01CT304", "default": 40.115},
                "凝汽器回水温度2": {"KKS": "13MAG01CT306", "default": 40.052},
                "机力塔风机1电流": {"KKS": "10PAG11AN001XQ01", "default": 22.942},
                "机力塔风机2电流": {"KKS": "10PAG11AN002XQ01", "default": -0.072},
                "机力塔风机3电流": {"KKS": "10PAG11AN003XQ01", "default": 23.232},
                "机力塔风机4电流": {"KKS": "10PAG11AN004XQ01", "default": 23.816},
                "机力塔风机5电流": {"KKS": "10PAG11AN005XQ01", "default": 23.377},
                "机力塔风机6电流": {"KKS": "10PAG11AN006XQ01", "default": 23.103},
                "机力塔风机7电流": {"KKS": "10PAG11AN007XQ01", "default": 23.679},
                "机力塔风机8电流": {"KKS": "10PAG11AN008XQ01", "default": 22.019},
                "空气湿球温度": {"KKS": "10PAC11CT111", "default": 15.352},
                "风速": {"KKS": "10AC10FS101-X", "default": 3.166},
                "大气压力1": {"KKS": "11RCAOGC00702", "default": 1019.309},
                "大气压力2": {"KKS": "12RCAOGC00702", "default": 1019.717},
                "#1燃机环境温度": {"KKS": "11MBL01CT101", "default": 17.790},
                "#2燃机环境温度": {"KKS": "12MBL01CT101", "default": 17.377},
                "#1燃机进气湿度": {"KKS": "11MBL01CM101A", "default": 78.851},
                "#2燃机进气湿度": {"KKS": "12MBL01CM101A", "default": 71.840},
                "1号炉TCA入口温度": {"KKS": "11RCAOMD60502", "default": 164.378},
                "1号炉TCA入口流量": {"KKS": "11RCAOMD60501", "default": 99.127},
                "1号炉TCA出口温度": {"KKS": "11RCAOMD60503", "default": 270.086},
                "1号炉TCA入口压力": {"KKS": "11LAB90CP101-SEL", "default": 14.470},
                "1号炉高压给水流量(TCA分流后)": {"KKS": "11LAB90CF101-SEL", "default": 185.922},
                "1号炉高压给水温度": {"KKS": "11LAB90CT301-SEL", "default": 165.975},
                "1号炉中压给水流量": {"KKS": "11LAB80CF101-SEL", "default": 119.225},
                "1号炉中压给水压力": {"KKS": "11LAB81CP103-SEL", "default": 5.496},
                "1号炉中压给水温度": {"KKS": "11LAB80CT301-SEL", "default": 160.461},
                "1号FGH入口流量": {"KKS": "11RCAOME60410", "default": 37.362},
                "1号FGH入口温度": {"KKS": "11RCAOME60408", "default": 239.760},
                "1号FGH入口压力": {"KKS": "11LAB83CP101", "default": 5.140},
                "1号FGH出口温度": {"KKS": "11RCAOME60409", "default": 85.915},
                "1号炉凝结水流量(含FGH回水)": {"KKS": "11LAB70CF101-SEL", "default": 447.744},
                "1号炉凝结水压力(FGH混合后)": {"KKS": "11LCA70CP102", "default": 2.539},
                "1号炉凝结水温度(FGH混合后)": {"KKS": "11LCA70CT302", "default": 57.819},
                "2号炉TCA入口温度": {"KKS": "12RCAOMD60502", "default": 162.634},
                "2号炉TCA入口流量": {"KKS": "12RCAOMD60501", "default": 80.298},
                "2号炉TCA出口温度": {"KKS": "12RCAOMD60503", "default": 287.206},
                "2号炉TCA入口压力": {"KKS": "12LAB90CP101-SEL", "default": 14.593},
                "2号炉高压给水流量(TCA分流后)": {"KKS": "12LAB90CF101-SEL", "default": 202.594},
                "2号炉高压给水温度": {"KKS": "12LAB90CT301-SEL", "default": 165.581},
                "2号炉中压给水流量": {"KKS": "12LAB80CF101-SEL", "default": 116.211},
                "2号炉中压给水压力": {"KKS": "12LAB81CP103-SEL", "default": 5.595},
                "2号炉中压给水温度": {"KKS": "12LAB80CT301-SEL", "default": 164.071},
                "2号炉FGH入口流量": {"KKS": "12RCAOME60410", "default": 37.726},
                "2号炉FGH入口温度": {"KKS": "12RCAOME60408", "default": 240.540},
                "2号炉FGH入口压力": {"KKS": "12LAB83CP101", "default": 5.350},
                "2号炉FGH出口温度": {"KKS": "12RCAOME60409", "default": 87.798},
                "2号炉凝结水流量(含FGH回水)": {"KKS": "12LAB70CF101-SEL", "default": 450.884},
                "2号炉凝结水压力(FGH混合后)": {"KKS": "12LCA70CP102", "default": 2.534},
                "2号炉凝结水温度(FGH混合后)": {"KKS": "12LCA70CT302", "default": 58.391},
                "凝泵出口凝结水流量": {"KKS": "13LCA20CF101-SEL", "default": 728.051},
                "凝结水补水流量": {"KKS": "10GHA10CF101-SEL", "default": 0.000},
                "凝结水补水压力": {"KKS": "10GHA10CP101-CAL", "default": 0.399},
                "凝结水补水温度": {"KKS": "10GHA10CT301", "default": 23.389},
                "1号炉低过出口蒸汽流量": {"KKS": "11LBA70CF101-SEL", "default": 40.144},
                "2号炉低过出口蒸汽流量": {"KKS": "12LBA70CF101-SEL", "default": 37.835},
                "1号炉中压过热器出口流量": {"KKS": "11LBA80CF01-SEL", "default": -1},
                "1号炉凝结水再循环流量": {"KKS": "11LCC71CF101-COMP", "default": 202.291},
                "1号炉烟气加热器入口凝结水温度": {"KKS": "11LCC71CT301", "default": 170.774},
                "1号炉烟气加热器出口凝结水温度": {"KKS": "11LCC71CT302", "default": 81.122},
                "2号炉凝结水再循环流量": {"KKS": "12LCC71CF101-COMP", "default": 266.144},
                "2号炉烟气加热器入口凝结水温度": {"KKS": "12LCC71CT301", "default": 135.305},
                "2号炉烟气加热器出口凝结水温度": {"KKS": "12LCC71CT302", "default": 75.361},
                "2号炉中压过热器出口流量": {"KKS": "12LBA80CF01-SEL", "default": -1.000},
                "凝泵出口凝结水压力": {"KKS": "13LDB10CP101", "default": 0.300},
                "凝泵出口凝结水温度": {"KKS": "13LDB10CT301", "default": 43.408},
                "低压缸入口调门开度": {"KKS": "13RCAOSE2501", "default": 99.033},
                "背压模式": {"KKS": "10CJA03FU009", "default": 0}
            }
            self.snapshot: Optional[pd.Series, None] = None

        def Execute(self):
            # Input rule code here
            # Py_Action_1.Action.Execute(Py_Action_1)
            self.snapshot = self.get_snapshot()
            if self.snapshot['背压模式'] == 1:
                self.write_result(0, '001')
            self.cal_heat_load_condense()
            # 先计算当前运行方式对应的背压
            _input = [self.snapshot[col] for col in self.input_columns]
            self.snapshot['当前理论背压'] = self.model.prediction(_input)

            净功率增加 = 0
            n1 = self.snapshot['风机运行台数']
            n2 = self.snapshot['循泵运行方式数字编码']
            for i in range(9):  # 0-8 # 风机运行台数
                for j in range(1, 9):  # 1-8 # 循泵运行台数
                    _ = self.cal_net_power_inc(i, j)
                    if _ > 净功率增加 and _ > 0:
                        净功率增加 = _
                        n1 = i
                        n2 = j

            # 优化结果就是n1, n2
            self.write_result(n1, n2)
            return True

        def get_snapshot(self):
            kv = {}
            for des, obj in self.tags.items():
                kks = obj.get('KKS')
                default_val = obj.get('default')
                tag_value = PointLib.GetFloatPoint(kks)
                kv[des] = tag_value or default_val
            return pd.Series(kv)

        def write_result(self, fan_num, pump_type):
            if isinstance(pump_type, int):
                # 根据数字在self.循泵运行方式映射中查询对应的字符串
                # 根据数字在self.循泵运行方式映射中查询对应的字符串
                for key, value in self.循泵运行方式映射.items():
                    if value == pump_type:
                        pump_str = key
                        break
                else:
                    print('未找到循泵运行方式数字编码对应的运行方式，退出')
                    return
            else:
                pump_str = pump_type

        def cal_net_power_inc(self, i, j):
            """
            计算冷端运行方式对应的背压
            """
            input1 = [self.snapshot[col] for col in self.input_columns[:-2]]  # 去除最后两个，分别是风机和循泵运行台数
            input1.extend([i, j])
            bp = self.model.prediction(input1) # 预测的背压
            delta_power = self.cal_delta_power(bp) #
            delta_power_cold = self.cal_delta_power_cold(i, j)
            return delta_power - delta_power_cold  # 净功率增加

        def cal_delta_power(self, bp):
            """
            计算背压对应的功率变化
            """
            bp_lilun = self.snapshot['当前理论背压'] # 当前运行方式的理论背压
            bp_delta = bp - bp_lilun # 新运行方式下背压相对当前理论背压的变化值

            return delta_power

        def cal_delta_power_cold(self, i, j):
            """
            计算冷端运行方式对应的功率变化
            """
            delta_power = self.snapshot['背压'] * self.snapshot['Power_Cold_' + str(i) + '_' + str(j)]
            return delta_power

        def cal_heat_load_condense(self):
            """
            基于单个时刻的snapshot数据计算凝汽器热负荷
            """
            # 计算平均大气压力（单位转换：kPa -> MPa）
            self.snapshot['平均大气压力'] = (self.snapshot['大气压力1'] + self.snapshot['大气压力2']) / 2000

            # 计算环境温度和大气湿度平均值
            self.snapshot['环境温度'] = (self.snapshot['#1燃机环境温度'] + self.snapshot['#2燃机环境温度']) / 2
            self.snapshot['大气湿度'] = (self.snapshot['#1燃机进气湿度'] + self.snapshot['#2燃机进气湿度']) / 2

            # 计算高压缸进汽压力绝压
            self.snapshot['高压缸进汽压力绝压'] = self.snapshot['高压缸进汽压力'] + self.snapshot['平均大气压力']

            # 计算高压缸进汽总流量
            self.snapshot['高压缸进汽总流量'] = self.snapshot['1号锅炉高压主汽流量'] + self.snapshot[
                '2号锅炉高压主汽流量']

            # 计算高压缸进汽焓值
            self.snapshot['高压缸进汽焓'] = H_PT(self.snapshot['高压缸进汽压力绝压'], self.snapshot['高压缸进汽温度'])

            # 计算高压缸进汽能量
            self.snapshot['高压缸进汽能量'] = self.snapshot['高压缸进汽总流量'] * self.snapshot['高压缸进汽焓']

            # 计算凝结水绝对压力
            self.snapshot['凝结水绝对压力'] = self.snapshot['凝泵出口凝结水压力'] + self.snapshot['平均大气压力']

            # 计算平均凝结水焓
            self.snapshot['平均凝结水焓'] = H_PT(self.snapshot['凝结水绝对压力'], self.snapshot['凝泵出口凝结水温度'])

            # 计算再热蒸汽相关参数
            self.snapshot['再热蒸汽母管绝对压力'] = self.snapshot['再热蒸汽母管压力'] + self.snapshot['平均大气压力']
            self.snapshot['高排绝对压力'] = self.snapshot['高排压力'] + self.snapshot['平均大气压力']

            # 计算焓值
            self.snapshot['热再热蒸汽焓'] = H_PT(self.snapshot['再热蒸汽母管绝对压力'],
                                                 self.snapshot['再热蒸汽母管温度'])
            self.snapshot['高排焓'] = H_PT(self.snapshot['高排绝对压力'], self.snapshot['高排温度'])

            # 计算冷再热蒸汽流量
            self.snapshot['F_crh'] = self.snapshot['高排至1号炉流量'] + self.snapshot['高排至2号炉流量']

            # 计算冷再热蒸汽吸热量
            self.snapshot['冷再热蒸汽吸热量'] = self.snapshot['F_crh'] * (
                    self.snapshot['热再热蒸汽焓'] - self.snapshot['高排焓'])

            # 计算中压给水部分吸热量相关参数
            self.snapshot['F_is'] = (self.snapshot['1号炉再热蒸汽流量'] + self.snapshot['1号炉再热蒸汽流量'] -
                                     self.snapshot['F_crh'])
            self.snapshot['F_is*H_hrh'] = self.snapshot['F_is'] * self.snapshot['热再热蒸汽焓']

            # 计算低压部分吸热量
            self.snapshot['低压主蒸汽绝对压力混合前'] = (self.snapshot['低压主蒸汽压力(中排混合前)'] +
                                                         self.snapshot['平均大气压力'])
            self.snapshot['H_ls'] = H_PT(self.snapshot['低压主蒸汽绝对压力混合前'],
                                         self.snapshot['低压主蒸汽温度(中排混合前)'])
            self.snapshot['F_ls'] = self.snapshot['1号炉低过出口蒸汽流量'] + self.snapshot['2号炉低过出口蒸汽流量']
            self.snapshot['低压蒸汽能量'] = self.snapshot['F_ls'] * self.snapshot['H_ls']

            # 计算汽机轴功率当量热量
            self.snapshot['汽机轴功率当量热量'] = 3600 * (self.snapshot['汽轮发电机功率'] / 0.99 + 0.607)  # 单位为MJ/h

            # 计算低压缸排汽能量
            self.snapshot['低压缸排汽能量'] = (self.snapshot['高压缸进汽能量'] + self.snapshot['冷再热蒸汽吸热量'] +
                                               self.snapshot['F_is*H_hrh'] + self.snapshot['低压蒸汽能量'] -
                                               self.snapshot['汽机抽汽供热量'] * 1000 - self.snapshot[
                                                   '汽机轴功率当量热量'])

            # 计算补水焓
            self.snapshot['补水焓'] = H_PT(self.snapshot['凝结水补水压力'], self.snapshot['凝结水补水温度'])

            # 计算总凝结水流量
            self.snapshot['F_con'] = (self.snapshot['高压缸进汽总流量'] + self.snapshot['F_is'] +
                                      self.snapshot['F_ls'] + self.snapshot['凝结水补水流量'])

            # 最终计算凝汽器热负荷
            self.snapshot['凝汽器热负荷'] = (self.snapshot['低压缸排汽能量'] +
                                             self.snapshot['补水焓'] * self.snapshot['凝结水补水流量'] -
                                             self.snapshot['F_con'] * self.snapshot['平均凝结水焓'])

            # 计算风机运行台数（根据机力塔风机电流判断，假设电流>1A为运行）
            风机电流列 = [
                '机力塔风机1电流', '机力塔风机2电流', '机力塔风机3电流', '机力塔风机4电流',
                '机力塔风机5电流', '机力塔风机6电流', '机力塔风机7电流', '机力塔风机8电流'
            ]
            self.snapshot['风机运行台数'] = sum(1 for col in 风机电流列 if self.snapshot[col] > 1)

            # 计算循泵运行方式数字编码
            循泵状态 = {
                '1号大循泵': self.snapshot['1号大循泵电流（额定298A）'] > 1,
                '2号大循泵': self.snapshot['2号大循泵电流（额定298A）'] > 1,
                '3号小循泵高速': self.snapshot['3号小循泵电流（高速）'] > 1,
                '3号小循泵低速': self.snapshot['3号小循泵电流（低速）'] > 1,
                '4号小循泵高速': self.snapshot['4号小循泵电流（高速）'] > 1,
                '4号小循泵低速': self.snapshot['4号小循泵电流（低速）'] > 1
            }

            # 根据循泵运行状态确定运行方式编码
            # 构造运行方式字符串（格式如"001"表示1台低速小泵）
            运行方式 = ""
            # 大泵状态（最多2台）
            运行方式 += str(sum([循泵状态['1号大循泵'], 循泵状态['2号大循泵']]))
            # 高速小泵状态（最多2台）
            运行方式 += str(sum([循泵状态['3号小循泵高速'], 循泵状态['4号小循泵高速']]))
            # 低速小泵状态（最多2台）
            运行方式 += str(sum([循泵状态['3号小循泵低速'], 循泵状态['4号小循泵低速']]))

            # 根据运行方式字符串查找对应的数字编码
            self.snapshot['循泵运行方式数字编码'] = self.循泵运行方式映射.get(运行方式, 0)

            # 返回计算结果
            return [self.snapshot[col] if col in self.snapshot else 0 for col in self.input_columns]
