/*
 * Library context type test program
 *
 * Copyright (C) 2011-2020, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <file_stream.h>
#include <memory.h>
#include <types.h>

#if defined( HAVE_STDLIB_H ) || defined( WINAPI )
#include <stdlib.h>
#endif

#if defined( HAVE_LIBCRYPTO ) && defined( HAVE_OPENSSL_AES_H )
/* No additional includes necessary */

#elif defined( HAVE_LIBCRYPTO ) && defined( HAVE_OPENSSL_EVP_H )
#include <openssl/evp.h>
#endif

#if defined( HAVE_GNU_DL_DLSYM ) && defined( __GNUC__ ) && !defined( __clang__ ) && !defined( __CYGWIN__ )
#define __USE_GNU
#include <dlfcn.h>
#undef __USE_GNU
#endif

#include "caes_test_libcaes.h"
#include "caes_test_libcerror.h"
#include "caes_test_macros.h"
#include "caes_test_memory.h"
#include "caes_test_unused.h"

/* Make sure libcaes_definitions.h is included first to define LIBCAES_HAVE_AES_SUPPORT
 */
#include "../libcaes/libcaes_definitions.h"
#include "../libcaes/libcaes_context.h"

#if defined( HAVE_GNU_DL_DLSYM ) && defined( __GNUC__ ) && !defined( __clang__ ) && !defined( __CYGWIN__ )

#if defined( HAVE_LIBCRYPTO ) && defined( HAVE_OPENSSL_AES_H ) && ( defined( HAVE_AES_CBC_ENCRYPT ) || defined( HAVE_AES_EBC_ENCRYPT ) )

static int (*caes_test_real_AES_set_encrypt_key)(const unsigned char *, const int, AES_KEY *)                                                     = NULL;
static int (*caes_test_real_AES_set_decrypt_key)(const unsigned char *, const int, AES_KEY *)                                                     = NULL;

int caes_test_AES_set_encrypt_key_attempts_before_fail                                                                                            = -1;
int caes_test_AES_set_decrypt_key_attempts_before_fail                                                                                            = -1;

#elif defined( HAVE_LIBCRYPTO ) && defined( HAVE_OPENSSL_EVP_H ) && ( defined( HAVE_EVP_CRYPTO_AES_CBC ) || defined( HAVE_EVP_CRYPTO_AES_ECB ) )

static int (*caes_test_real_EVP_CIPHER_CTX_set_padding)(EVP_CIPHER_CTX *, int)                                                                    = NULL;
static int (*caes_test_real_EVP_CipherInit_ex)(EVP_CIPHER_CTX *, const EVP_CIPHER *, ENGINE *, const unsigned char *, const unsigned char *, int) = NULL;
static int (*caes_test_real_EVP_CipherUpdate)(EVP_CIPHER_CTX *, unsigned char *, int *, const unsigned char *, int)                               = NULL;

int caes_test_EVP_CIPHER_CTX_set_padding_attempts_before_fail                                                                                     = -1;
int caes_test_EVP_CipherInit_ex_attempts_before_fail                                                                                              = -1;
int caes_test_EVP_CipherUpdate_attempts_before_fail                                                                                               = -1;

#endif /* defined( HAVE_LIBCRYPTO ) && defined( HAVE_OPENSSL_AES_H ) && ( defined( HAVE_AES_CBC_ENCRYPT ) || defined( HAVE_AES_EBC_ENCRYPT ) ) */

#endif /* defined( HAVE_GNU_DL_DLSYM ) && defined( __GNUC__ ) && !defined( __clang__ ) && !defined( __CYGWIN__ ) */

typedef struct caes_test_context_test_vector caes_test_context_test_vector_t;

struct caes_test_context_test_vector
{
	/* The description
	 */
	const char *description;

	/* The mode
	 */
	int mode;

	/* The key
	 */
	uint8_t key[ 64 ];

	/* The key bit size
	 */
	size_t key_bit_size;

	/* The initialization vector
	 */
	uint8_t initialization_vector[ 16 ];

	/* The initialization vector size
	 */
	size_t initialization_vector_size;

	/* The input data
	 */
	uint8_t input_data[ 16 ];

	/* The input data size
	 */
	size_t input_data_size;

	/* The expected output data
	 */
	uint8_t output_data[ 16 ];

	/* The expected output data size
	 */
	size_t output_data_size;
};

caes_test_context_test_vector_t caes_context_aes_cbc_test_vectors[ 24 ] = {
	{ "NIST SP800-38A test vector AES-CBC 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d }, 16,
	  { 0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2 }, 16,
	  { 0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09, 0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16 }, 16,
	  { 0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09, 0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },

	{ "NIST SP800-38A test vector AES-CBC 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4, 0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8 }, 16,
	  { 0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4, 0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4, 0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0, 0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4, 0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a }, 16,
	  { 0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0, 0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0, 0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81, 0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62,
	    0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0, 0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0 }, 16,
	  { 0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81, 0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },

	{ "NIST SP800-38A test vector AES-CBC 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 }, 16,
	  { 0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61 }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d }, 16,
	  { 0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc, 0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b }, 16 },
	{ "NIST SP800-38A test vector AES-CBC 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61 }, 16,
	  { 0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc, 0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },
};

/* TODO: Values from NIST KAT-AES CFBVarKey128.rsp, CFBVarKey192.rsp and CFBVarKey256.rsp */

caes_test_context_test_vector_t caes_context_aes_cfb_test_vectors[ 8 ] = {
	{ "NIST SP800-38A test vector AES-CFB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20, 0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20, 0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20, 0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f, 0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20, 0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f, 0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f, 0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40, 0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f, 0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40, 0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40, 0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0xc0, 0x4b, 0x05, 0x35, 0x7c, 0x5d, 0x1c, 0x0e, 0xea, 0xc4, 0xc6, 0x6f, 0x9f, 0xf7, 0xf2, 0xe6 }, 16 },
	{ "NIST SP800-38A test vector AES-CFB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40, 0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0xc0, 0x4b, 0x05, 0x35, 0x7c, 0x5d, 0x1c, 0x0e, 0xea, 0xc4, 0xc6, 0x6f, 0x9f, 0xf7, 0xf2, 0xe6 }, 16 },
};

/* TODO
# For all CFB128 encrypts and decrypts, the transformed sequence is
#   AES-bits-CFB:key:IV/ciphertext':plaintext:ciphertext:encdec
# CFB128-AES192.Encrypt
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:000102030405060708090A0B0C0D0E0F:6BC1BEE22E409F96E93D7E117393172A:CDC80D6FDDF18CAB34C25909C99A4174:1
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:CDC80D6FDDF18CAB34C25909C99A4174:AE2D8A571E03AC9C9EB76FAC45AF8E51:67CE7F7F81173621961A2B70171D3D7A:1
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:67CE7F7F81173621961A2B70171D3D7A:30C81C46A35CE411E5FBC1191A0A52EF:2E1E8A1DD59B88B1C8E60FED1EFAC4C9:1
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:2E1E8A1DD59B88B1C8E60FED1EFAC4C9:F69F2445DF4F9B17AD2B417BE66C3710:C05F9F9CA9834FA042AE8FBA584B09FF:1
# CFB128-AES192.Decrypt
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:000102030405060708090A0B0C0D0E0F:6BC1BEE22E409F96E93D7E117393172A:CDC80D6FDDF18CAB34C25909C99A4174:0
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:CDC80D6FDDF18CAB34C25909C99A4174:AE2D8A571E03AC9C9EB76FAC45AF8E51:67CE7F7F81173621961A2B70171D3D7A:0
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:67CE7F7F81173621961A2B70171D3D7A:30C81C46A35CE411E5FBC1191A0A52EF:2E1E8A1DD59B88B1C8E60FED1EFAC4C9:0
AES-192-CFB:8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B:2E1E8A1DD59B88B1C8E60FED1EFAC4C9:F69F2445DF4F9B17AD2B417BE66C3710:C05F9F9CA9834FA042AE8FBA584B09FF:0
# CFB128-AES256.Encrypt 
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:000102030405060708090A0B0C0D0E0F:6BC1BEE22E409F96E93D7E117393172A:DC7E84BFDA79164B7ECD8486985D3860:1
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:DC7E84BFDA79164B7ECD8486985D3860:AE2D8A571E03AC9C9EB76FAC45AF8E51:39FFED143B28B1C832113C6331E5407B:1
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:39FFED143B28B1C832113C6331E5407B:30C81C46A35CE411E5FBC1191A0A52EF:DF10132415E54B92A13ED0A8267AE2F9:1
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:DF10132415E54B92A13ED0A8267AE2F9:F69F2445DF4F9B17AD2B417BE66C3710:75A385741AB9CEF82031623D55B1E471:1
# CFB128-AES256.Decrypt 
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:000102030405060708090A0B0C0D0E0F:6BC1BEE22E409F96E93D7E117393172A:DC7E84BFDA79164B7ECD8486985D3860:0
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:DC7E84BFDA79164B7ECD8486985D3860:AE2D8A571E03AC9C9EB76FAC45AF8E51:39FFED143B28B1C832113C6331E5407B:0
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:39FFED143B28B1C832113C6331E5407B:30C81C46A35CE411E5FBC1191A0A52EF:DF10132415E54B92A13ED0A8267AE2F9:0
AES-256-CFB:603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4:DF10132415E54B92A13ED0A8267AE2F9:F69F2445DF4F9B17AD2B417BE66C3710:75A385741AB9CEF82031623D55B1E471:0
 */

caes_test_context_test_vector_t caes_context_aes_ecb_test_vectors[ 1950 ] = {
	{ "FIPS-197 test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16,
	  { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30, 0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a }, 16 },
	{ "FIPS-197 test vector 1 AES-ECB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30, 0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16 },
	{ "FIPS-197 test vector 2 AES-ECB 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16,
	  { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0, 0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 }, 16 },
	{ "FIPS-197 test vector 2 AES-ECB 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0, 0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16 },
	{ "FIPS-197 test vector 2 AES-ECB 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16,
	  { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf, 0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 }, 16 },
	{ "FIPS-197 test vector 2 AES-ECB 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf, 0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 }, 16,
	  { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff }, 16 },

	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0x3a, 0xd7, 0x7b, 0xb4, 0x0d, 0x7a, 0x36, 0x60, 0xa8, 0x9e, 0xca, 0xf3, 0x24, 0x66, 0xef, 0x97 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xd7, 0x7b, 0xb4, 0x0d, 0x7a, 0x36, 0x60, 0xa8, 0x9e, 0xca, 0xf3, 0x24, 0x66, 0xef, 0x97 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0xf5, 0xd3, 0xd5, 0x85, 0x03, 0xb9, 0x69, 0x9d, 0xe7, 0x85, 0x89, 0x5a, 0x96, 0xfd, 0xba, 0xaf }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf5, 0xd3, 0xd5, 0x85, 0x03, 0xb9, 0x69, 0x9d, 0xe7, 0x85, 0x89, 0x5a, 0x96, 0xfd, 0xba, 0xaf }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0x43, 0xb1, 0xcd, 0x7f, 0x59, 0x8e, 0xce, 0x23, 0x88, 0x1b, 0x00, 0xe3, 0xed, 0x03, 0x06, 0x88 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xb1, 0xcd, 0x7f, 0x59, 0x8e, 0xce, 0x23, 0x88, 0x1b, 0x00, 0xe3, 0xed, 0x03, 0x06, 0x88 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0x7b, 0x0c, 0x78, 0x5e, 0x27, 0xe8, 0xad, 0x3f, 0x82, 0x23, 0x20, 0x71, 0x04, 0x72, 0x5d, 0xd4 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 128-bit encrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x0c, 0x78, 0x5e, 0x27, 0xe8, 0xad, 0x3f, 0x82, 0x23, 0x20, 0x71, 0x04, 0x72, 0x5d, 0xd4 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },

	{ "NIST SP800-38A test vector AES-ECB 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0xbd, 0x33, 0x4f, 0x1d, 0x6e, 0x45, 0xf2, 0x5f, 0xf7, 0x12, 0xa2, 0x14, 0x57, 0x1f, 0xa5, 0xcc }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbd, 0x33, 0x4f, 0x1d, 0x6e, 0x45, 0xf2, 0x5f, 0xf7, 0x12, 0xa2, 0x14, 0x57, 0x1f, 0xa5, 0xcc }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0x97, 0x41, 0x04, 0x84, 0x6d, 0x0a, 0xd3, 0xad, 0x77, 0x34, 0xec, 0xb3, 0xec, 0xee, 0x4e, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x41, 0x04, 0x84, 0x6d, 0x0a, 0xd3, 0xad, 0x77, 0x34, 0xec, 0xb3, 0xec, 0xee, 0x4e, 0xef }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0xef, 0x7a, 0xfd, 0x22, 0x70, 0xe2, 0xe6, 0x0a, 0xdc, 0xe0, 0xba, 0x2f, 0xac, 0xe6, 0x44, 0x4e }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x7a, 0xfd, 0x22, 0x70, 0xe2, 0xe6, 0x0a, 0xdc, 0xe0, 0xba, 0x2f, 0xac, 0xe6, 0x44, 0x4e }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0x9a, 0x4b, 0x41, 0xba, 0x73, 0x8d, 0x6c, 0x72, 0xfb, 0x16, 0x69, 0x16, 0x03, 0xc1, 0x8e, 0x0e }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 192-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9a, 0x4b, 0x41, 0xba, 0x73, 0x8d, 0x6c, 0x72, 0xfb, 0x16, 0x69, 0x16, 0x03, 0xc1, 0x8e, 0x0e }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },

	{ "NIST SP800-38A test vector AES-ECB 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16,
	  { 0xf3, 0xee, 0xd1, 0xbd, 0xb5, 0xd2, 0xa0, 0x3c, 0x06, 0x4b, 0x5a, 0x7e, 0x3d, 0xb1, 0x81, 0xf8 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0xee, 0xd1, 0xbd, 0xb5, 0xd2, 0xa0, 0x3c, 0x06, 0x4b, 0x5a, 0x7e, 0x3d, 0xb1, 0x81, 0xf8 }, 16,
	  { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16,
	  { 0x59, 0x1c, 0xcb, 0x10, 0xd4, 0x10, 0xed, 0x26, 0xdc, 0x5b, 0xa7, 0x4a, 0x31, 0x36, 0x28, 0x70 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0x1c, 0xcb, 0x10, 0xd4, 0x10, 0xed, 0x26, 0xdc, 0x5b, 0xa7, 0x4a, 0x31, 0x36, 0x28, 0x70 }, 16,
	  { 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16,
	  { 0xb6, 0xed, 0x21, 0xb9, 0x9c, 0xa6, 0xf4, 0xf9, 0xf1, 0x53, 0xe7, 0xb1, 0xbe, 0xaf, 0xed, 0x1d }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0xed, 0x21, 0xb9, 0x9c, 0xa6, 0xf4, 0xf9, 0xf1, 0x53, 0xe7, 0xb1, 0xbe, 0xaf, 0xed, 0x1d }, 16,
	  { 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16,
	  { 0x23, 0x30, 0x4b, 0x7a, 0x39, 0xf9, 0xf3, 0xff, 0x06, 0x7d, 0x8d, 0x8f, 0x9e, 0x24, 0xec, 0xc7 }, 16 },
	{ "NIST SP800-38A test vector AES-ECB 256-bit decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0x30, 0x4b, 0x7a, 0x39, 0xf9, 0xf3, 0xff, 0x06, 0x7d, 0x8d, 0x8f, 0x9e, 0x24, 0xec, 0xc7 }, 16,
	  { 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 }, 16 },

	{ "NIST KAT-AES ECBVarKey128.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0e, 0xdd, 0x33, 0xd3, 0xc6, 0x21, 0xe5, 0x46, 0x45, 0x5b, 0xd8, 0xba, 0x14, 0x18, 0xbe, 0xc8 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0xc3, 0xf8, 0x83, 0x45, 0x0c, 0x11, 0x3c, 0x64, 0xca, 0x42, 0xe1, 0x11, 0x2a, 0x9e, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xa1, 0xda, 0x77, 0x0f, 0x5d, 0x7a, 0xc4, 0xc9, 0xef, 0x94, 0xd8, 0x22, 0xaf, 0xfd, 0x97 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x00, 0x14, 0xd6, 0x34, 0xe2, 0xb7, 0x65, 0x07, 0x77, 0xe8, 0xe8, 0x4d, 0x03, 0xcc, 0xd8 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0x7e, 0x79, 0xae, 0xd0, 0xdb, 0x7e, 0x27, 0x9e, 0x95, 0x5b, 0x5f, 0x49, 0x38, 0x75, 0xa7 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9e, 0xd5, 0xa7, 0x51, 0x36, 0xa9, 0x40, 0xd0, 0x96, 0x3d, 0xa3, 0x79, 0xdb, 0x4a, 0xf2, 0x6a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x29, 0x5f, 0x83, 0x46, 0x5c, 0x77, 0x55, 0xe8, 0xfa, 0x36, 0x4b, 0xac, 0x6a, 0x7e, 0xa5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb1, 0xd7, 0x58, 0x25, 0x6b, 0x28, 0xfd, 0x85, 0x0a, 0xd4, 0x94, 0x42, 0x08, 0xcf, 0x11, 0x55 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x42, 0xff, 0xb3, 0x4c, 0x74, 0x3d, 0xe4, 0xd8, 0x8c, 0xa3, 0x80, 0x11, 0xc9, 0x90, 0x89, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x58, 0xf0, 0xec, 0xea, 0x8b, 0x21, 0x72, 0xc0, 0xc1, 0x99, 0x5f, 0x91, 0x82, 0xc0, 0xf3 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0x6d, 0x77, 0x98, 0xfa, 0xc2, 0x0f, 0x82, 0xa8, 0x82, 0x3f, 0x98, 0x4d, 0x06, 0xf7, 0xf5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x1b, 0xf4, 0x4f, 0x2d, 0x16, 0xbe, 0x92, 0x8c, 0xa4, 0x4a, 0xaf, 0x7b, 0x9b, 0x10, 0x6b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xf1, 0xa3, 0x3e, 0x50, 0xd4, 0x0d, 0x10, 0x37, 0x64, 0xc7, 0x6b, 0xd4, 0xc6, 0xb6, 0xf8 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 13 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x37, 0x05, 0x0c, 0x9f, 0xc0, 0xd4, 0x81, 0x7e, 0x2d, 0x69, 0xde, 0x87, 0x8a, 0xee, 0x8d }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 14 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x3e, 0xcb, 0xe4, 0xa4, 0x53, 0x26, 0x9a, 0x0d, 0xd2, 0x60, 0x69, 0x46, 0x7f, 0xb5, 0xb5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 15 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xd0, 0x75, 0x4f, 0xe6, 0x8f, 0x11, 0xb9, 0xe3, 0x75, 0xd0, 0x70, 0xa6, 0x08, 0xc8, 0x84 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 16 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0xa0, 0xb3, 0xe9, 0x98, 0xd0, 0x50, 0x68, 0xa5, 0x39, 0x97, 0x78, 0x40, 0x52, 0x00, 0xb4 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 17 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0x55, 0x6a, 0x33, 0x43, 0x8d, 0xb8, 0x7b, 0xc4, 0x1b, 0x17, 0x52, 0xc5, 0x5e, 0x5e, 0x49 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 18 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xfb, 0x12, 0x8d, 0x3a, 0x1a, 0xf6, 0xe5, 0x48, 0x52, 0x1b, 0xb9, 0x62, 0xbf, 0x1f, 0x05 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 19 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x29, 0x8e, 0x9c, 0x1d, 0xb5, 0x17, 0xc2, 0x15, 0xfa, 0xdf, 0xb7, 0xd2, 0xa8, 0xd6, 0x91 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 20 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa6, 0xcb, 0x76, 0x1d, 0x61, 0xf8, 0x29, 0x2d, 0x0d, 0xf3, 0x93, 0xa2, 0x79, 0xad, 0x03, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 21 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xac, 0xd8, 0x9b, 0x13, 0xcd, 0x5f, 0x87, 0x26, 0xe3, 0x4d, 0x44, 0xfd, 0x48, 0x61, 0x08 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 22 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0xb1, 0x70, 0x3f, 0xc5, 0x7b, 0xa0, 0x9f, 0xe0, 0xc3, 0x58, 0x0f, 0xeb, 0xdd, 0x7e, 0xd4 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 23 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xde, 0x11, 0x72, 0x2d, 0x89, 0x3e, 0x9f, 0x91, 0x21, 0xc3, 0x81, 0xbe, 0xcc, 0x1d, 0xa5, 0x9a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 24 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6d, 0x11, 0x4c, 0xcb, 0x27, 0xbf, 0x39, 0x10, 0x12, 0xe8, 0x97, 0x4c, 0x54, 0x6d, 0x9b, 0xf2 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 25 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5c, 0xe3, 0x7e, 0x17, 0xeb, 0x46, 0x46, 0xec, 0xfa, 0xc2, 0x9b, 0x9c, 0xc3, 0x8d, 0x93, 0x40 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 26 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0xc1, 0xb6, 0xe2, 0x15, 0x71, 0x22, 0x05, 0x6d, 0x02, 0x43, 0xd8, 0xa1, 0x65, 0xcd, 0xdb }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 27 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x69, 0x3e, 0x6a, 0x59, 0xd1, 0x36, 0x6c, 0x74, 0xd8, 0x23, 0x56, 0x2d, 0x7e, 0x14, 0x31 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 28 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x7c, 0x64, 0xdc, 0x84, 0xa8, 0xbb, 0xa7, 0x58, 0xed, 0x17, 0xeb, 0x02, 0x5a, 0x57, 0xe3 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 29 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe1, 0x7b, 0xc7, 0x9f, 0x30, 0xea, 0xab, 0x2f, 0xac, 0x2c, 0xbb, 0xe3, 0x45, 0x8d, 0x68, 0x7a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 30 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x14, 0xbc, 0x20, 0x28, 0x00, 0x9b, 0x92, 0x3f, 0x0b, 0x01, 0x91, 0x5c, 0xe5, 0xe7, 0xc4 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 31 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0x28, 0x52, 0x4a, 0x16, 0xa1, 0xe1, 0xc1, 0x45, 0x29, 0x71, 0xca, 0xa8, 0xd1, 0x34, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 32 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0x62, 0xe1, 0x63, 0x63, 0x63, 0x83, 0x60, 0xfd, 0xd6, 0xad, 0x62, 0x11, 0x27, 0x94, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 33 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5a, 0x86, 0x88, 0xf0, 0xb2, 0xa2, 0xc1, 0x62, 0x24, 0xc1, 0x61, 0x65, 0x8f, 0xfd, 0x40, 0x44 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 34 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0xf7, 0x10, 0x84, 0x2b, 0x9b, 0xb9, 0xc3, 0x2f, 0x26, 0x64, 0x8c, 0x78, 0x68, 0x07, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 35 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xa9, 0x8b, 0xf1, 0x1e, 0x16, 0x3f, 0x63, 0x2c, 0x47, 0xec, 0x6a, 0x49, 0x68, 0x3a, 0x89 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 36 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0f, 0x18, 0xaf, 0xf9, 0x42, 0x74, 0x69, 0x6d, 0x9b, 0x61, 0x84, 0x8b, 0xd5, 0x0a, 0xc5, 0xe5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 37 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x40, 0x85, 0x71, 0xc3, 0xe2, 0x42, 0x45, 0x40, 0x20, 0x7f, 0x83, 0x3b, 0x6d, 0xda, 0x69 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 38 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0x3f, 0xf9, 0x96, 0x94, 0x7f, 0x0c, 0x7d, 0x1f, 0x43, 0xc8, 0xf3, 0x02, 0x7b, 0x9b, 0x75 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 39 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0xf4, 0xda, 0xf4, 0xad, 0x29, 0xa3, 0x61, 0x5a, 0x9b, 0x6e, 0xce, 0x5c, 0x99, 0x51, 0x8a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 40 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x29, 0x54, 0xa4, 0x8d, 0x07, 0x74, 0xdb, 0x0b, 0x49, 0x71, 0xc5, 0x26, 0x26, 0x04, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 41 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xf9, 0xb7, 0x61, 0x12, 0xdc, 0x65, 0x31, 0xe0, 0x7d, 0x2c, 0xfd, 0xa0, 0x44, 0x11, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 42 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0x4d, 0x8e, 0x69, 0x91, 0x19, 0xe1, 0xfc, 0x87, 0x54, 0x5a, 0x64, 0x7f, 0xb1, 0xd3, 0x4f }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 43 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0xc4, 0x80, 0x7a, 0xe1, 0x1f, 0x36, 0xf0, 0x91, 0xc5, 0x7d, 0x9f, 0xb6, 0x85, 0x48, 0xd1 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 44 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xbf, 0x73, 0xaa, 0xd4, 0x9c, 0x82, 0x00, 0x7f, 0x77, 0xa5, 0xc1, 0xcc, 0xec, 0x6a, 0xb4 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 45 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xb2, 0x88, 0xcc, 0x20, 0x40, 0x04, 0x90, 0x01, 0xd2, 0xc7, 0x58, 0x5a, 0xd1, 0x23, 0xfc }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 46 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x04, 0x49, 0x71, 0x10, 0xef, 0xb9, 0xdc, 0xeb, 0x13, 0xe2, 0xb1, 0x3f, 0xb4, 0x46, 0x55, 0x64 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 47 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x55, 0x0e, 0x6c, 0xb5, 0xa8, 0x8e, 0x49, 0x63, 0x4c, 0x9a, 0xb6, 0x9e, 0xda, 0x04, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 48 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x76, 0x84, 0x73, 0xce, 0x98, 0x43, 0xea, 0x66, 0xa8, 0x14, 0x05, 0xdd, 0x50, 0xb3, 0x45 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 49 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x2f, 0x43, 0x03, 0x83, 0xf9, 0x08, 0x4e, 0x03, 0xa6, 0x53, 0x57, 0x1e, 0x06, 0x5d, 0xe6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 50 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x4e, 0x66, 0xc0, 0x7b, 0xae, 0x3e, 0x79, 0xfb, 0x7d, 0x21, 0x08, 0x47, 0xa3, 0xb0, 0xba }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 51 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x90, 0x78, 0x51, 0x25, 0x50, 0x5f, 0xad, 0x59, 0xb1, 0x3c, 0x18, 0x6d, 0xd6, 0x6c, 0xe3 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 52 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x52, 0x7a, 0x6a, 0xeb, 0xda, 0xec, 0x9e, 0xae, 0xf8, 0xed, 0xa2, 0xcb, 0x77, 0x83, 0xe5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 53 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xfd, 0xaf, 0x53, 0xeb, 0xbc, 0x98, 0x80, 0xc2, 0x28, 0x61, 0x7d, 0x6a, 0x9b, 0x54, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 54 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x78, 0x61, 0x04, 0xb9, 0x74, 0x4b, 0x98, 0xf0, 0x52, 0xc4, 0x6f, 0x1c, 0x85, 0x0d, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 55 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xab, 0x30, 0x13, 0xdd, 0x1e, 0x61, 0xdf, 0x06, 0xcb, 0xaf, 0x34, 0xca, 0x2a, 0xee, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 56 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x70, 0x46, 0x9b, 0xe9, 0x72, 0x30, 0x30, 0xfd, 0xcc, 0x73, 0xa8, 0xcd, 0x4f, 0xbb, 0x10 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 57 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa3, 0x5a, 0x63, 0xf5, 0x34, 0x3e, 0xbe, 0x9e, 0xf8, 0x16, 0x7b, 0xcb, 0x48, 0xad, 0x12, 0x2e }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 58 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0x86, 0x87, 0xf0, 0x75, 0x7a, 0x21, 0x0e, 0x9f, 0xdf, 0x18, 0x12, 0x04, 0xc3, 0x08, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 59 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0x18, 0x1e, 0x84, 0xbd, 0x54, 0x57, 0xd2, 0x6a, 0x88, 0xfb, 0xae, 0x96, 0x01, 0x8f, 0xb0 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 60 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x65, 0x33, 0x17, 0xb9, 0x36, 0x2b, 0x6f, 0x9b, 0x9e, 0x1a, 0x58, 0x0e, 0x68, 0xd4, 0x94, 0xb5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 61 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x5c, 0x9d, 0xc0, 0xb6, 0x89, 0xf0, 0x3c, 0x45, 0x86, 0x7b, 0x5f, 0xaa, 0x5c, 0x18, 0xd1 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 62 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xa4, 0xd9, 0x6d, 0x56, 0xdd, 0xa3, 0x98, 0xb9, 0xaa, 0xbe, 0xcf, 0xc7, 0x57, 0x29, 0xfd }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 63 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xbe, 0x19, 0xe0, 0x53, 0x63, 0x5f, 0x09, 0xf2, 0x66, 0x5e, 0x7b, 0xae, 0x85, 0xb4, 0x2d }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 64 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0xcd, 0x65, 0x28, 0x42, 0x92, 0x6a, 0xea, 0x4a, 0xa6, 0x13, 0x7b, 0xb2, 0xbe, 0x2b, 0x5e }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 65 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x3d, 0x4a, 0x4f, 0x38, 0xeb, 0xb3, 0x37, 0xd1, 0x0a, 0xa8, 0x4e, 0x91, 0x71, 0xa5, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 66 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0xbf, 0xf7, 0xff, 0x45, 0x4b, 0x0e, 0xc5, 0xa4, 0xa2, 0xa6, 0x95, 0x66, 0xe2, 0xcb, 0x84 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 67 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0x35, 0xd5, 0x65, 0xac, 0xe3, 0xf3, 0x1e, 0xb2, 0x49, 0xba, 0x2c, 0xc6, 0x76, 0x5d, 0x7a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 68 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x0e, 0x91, 0xfc, 0x32, 0x69, 0xee, 0xcf, 0x32, 0x31, 0xc6, 0xe9, 0x94, 0x56, 0x97, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 69 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x69, 0xcf, 0xad, 0xf5, 0x1f, 0x8e, 0x60, 0x4d, 0x9c, 0xc3, 0x71, 0x82, 0xf6, 0x63, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 70 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x78, 0x66, 0x37, 0x3f, 0x24, 0xa0, 0xb6, 0xed, 0x56, 0xe0, 0xd9, 0x6f, 0xcd, 0xaf, 0xb8, 0x77 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 71 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xa4, 0x48, 0xc2, 0xaa, 0xc9, 0x54, 0xf5, 0xd8, 0x12, 0xe9, 0xd7, 0x84, 0x94, 0x44, 0x6a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 72 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xac, 0xc5, 0x59, 0x9d, 0xd8, 0xac, 0x02, 0x23, 0x9a, 0x0f, 0xef, 0x4a, 0x36, 0xdd, 0x16, 0x68 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 73 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x76, 0x44, 0x68, 0xbb, 0x10, 0x38, 0x28, 0xcf, 0x7e, 0x14, 0x73, 0xce, 0x89, 0x50, 0x73 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 74 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x0d, 0x02, 0x89, 0x36, 0x83, 0xb9, 0xf1, 0x80, 0x45, 0x8e, 0x4a, 0xa6, 0xb7, 0x39, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 75 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0xd9, 0xb0, 0x17, 0xd3, 0x02, 0xdf, 0x41, 0x0a, 0x93, 0x7d, 0xcd, 0xb8, 0xbb, 0x6e, 0x43 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 76 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x16, 0x23, 0xcc, 0x44, 0x31, 0x3c, 0xff, 0x44, 0x0b, 0x15, 0x94, 0xa7, 0xe2, 0x1c, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 77 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x28, 0x4c, 0xa2, 0xfa, 0x35, 0x80, 0x7b, 0x8b, 0x0a, 0xe4, 0xd1, 0x9e, 0x11, 0xd7, 0xdb, 0xd7 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 78 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xe9, 0x76, 0x87, 0x57, 0x55, 0xf9, 0x40, 0x1d, 0x54, 0xf3, 0x6e, 0x2a, 0x23, 0xa5, 0x94 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 79 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xec, 0x19, 0x8a, 0x18, 0xe1, 0x0e, 0x53, 0x24, 0x03, 0xb7, 0xe2, 0x08, 0x87, 0xc8, 0xdd, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 80 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x54, 0x5d, 0x50, 0xeb, 0xd9, 0x19, 0xe4, 0xa6, 0x94, 0x9d, 0x96, 0xad, 0x47, 0xe4, 0x6a, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 81 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0xdf, 0xb5, 0x27, 0x06, 0x0e, 0x0a, 0x71, 0x00, 0x9c, 0x7b, 0xb0, 0xc6, 0x8f, 0x1d, 0x44 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 82 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xfa, 0x13, 0x22, 0xea, 0x33, 0xda, 0x21, 0x73, 0xa0, 0x24, 0xf2, 0xff, 0x0d, 0x89, 0x6d }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 83 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0x85, 0xb1, 0xa7, 0x5b, 0x0f, 0x3b, 0xd9, 0x58, 0xdc, 0xd0, 0xe2, 0x93, 0x18, 0xc5, 0x21 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 84 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x38, 0xf6, 0x7b, 0x9e, 0x98, 0xe4, 0xa9, 0x7b, 0x6d, 0xf0, 0x30, 0xa9, 0xfc, 0xdd, 0x01, 0x04 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 85 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x2a, 0xff, 0xfb, 0x2c, 0x88, 0x0e, 0x82, 0xb0, 0x59, 0x26, 0xd0, 0xfc, 0x6c, 0x44, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 86 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0x79, 0x80, 0xce, 0x7b, 0x10, 0x5c, 0xf5, 0x30, 0x95, 0x2d, 0x74, 0xda, 0xaf, 0x79, 0x8c }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 87 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x36, 0x95, 0xe1, 0x35, 0x1b, 0x9d, 0x68, 0x58, 0xbd, 0x95, 0x8c, 0xf5, 0x13, 0xef, 0x6c }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 88 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6d, 0xa0, 0x49, 0x0b, 0xa0, 0xba, 0x03, 0x43, 0xb9, 0x35, 0x68, 0x1d, 0x2c, 0xce, 0x5b, 0xa1 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 89 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0xea, 0x23, 0xaf, 0x08, 0x53, 0x40, 0x11, 0xc6, 0x00, 0x09, 0xab, 0x29, 0xad, 0xa2, 0xf1 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 90 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x13, 0x80, 0x6c, 0xf1, 0x9c, 0xc3, 0x87, 0x21, 0x55, 0x4d, 0x7c, 0x0f, 0xcd, 0xcd, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 91 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x38, 0xaf, 0x1f, 0x4f, 0x69, 0xba, 0xe9, 0xd8, 0x5d, 0xd1, 0x88, 0xdc, 0xdf, 0x06, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 92 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xcf, 0x44, 0xc9, 0x2d, 0x55, 0x0b, 0xfb, 0x1e, 0xd2, 0x8e, 0xf5, 0x83, 0xdd, 0xf5, 0xd7 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 93 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd0, 0x6e, 0x31, 0x95, 0xb5, 0x37, 0x6f, 0x10, 0x9d, 0x5c, 0x4e, 0xc6, 0xc5, 0xd6, 0x2c, 0xed }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 94 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x40, 0xde, 0x01, 0x4d, 0x3d, 0x61, 0x07, 0x07, 0x27, 0x9b, 0x13, 0x24, 0x2a, 0x5c, 0x36 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 95 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0xc5, 0xc6, 0xff, 0xa5, 0xe0, 0xbd, 0x3a, 0x94, 0xc8, 0x8f, 0x6b, 0x6f, 0x7c, 0x16, 0xb9 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 96 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3e, 0x40, 0xc3, 0x90, 0x1c, 0xd7, 0xef, 0xfc, 0x22, 0xbf, 0xfc, 0x35, 0xde, 0xe0, 0xb4, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 97 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x33, 0x05, 0xc7, 0x2b, 0xed, 0xfa, 0xb9, 0x73, 0x82, 0xc4, 0x06, 0xd0, 0xc4, 0x9b, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 98 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xbb, 0xaa, 0xb2, 0x2a, 0x6b, 0xd4, 0x92, 0x5a, 0x99, 0xa2, 0xb4, 0x08, 0xd2, 0xdb, 0xae }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 99 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0x7c, 0x5b, 0x8f, 0xcd, 0x05, 0x33, 0xab, 0x98, 0xbc, 0x51, 0xe2, 0x7a, 0x6c, 0xe4, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 100 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x9c, 0x04, 0xff, 0x4c, 0x07, 0x51, 0x3c, 0x0b, 0x3e, 0xf0, 0x5c, 0x03, 0xe3, 0x37, 0xb5 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 101 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0x7a, 0xf0, 0xe8, 0x95, 0xdd, 0xa5, 0xeb, 0x98, 0xef, 0xc6, 0x80, 0x66, 0xe8, 0x4c, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 102 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x71, 0x67, 0xf3, 0x81, 0x2a, 0xff, 0xf1, 0xff, 0xac, 0xb4, 0xa9, 0x34, 0x37, 0x9f, 0xc3 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 103 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xb1, 0xdc, 0x3a, 0x9c, 0x72, 0x97, 0x2e, 0x42, 0x5a, 0xe2, 0xef, 0x3e, 0xb5, 0x97, 0xcd }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 104 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xae, 0xaa, 0x3a, 0x21, 0x3e, 0x96, 0x8d, 0x4b, 0x5b, 0x67, 0x9d, 0x3a, 0x2c, 0x97, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 105 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x92, 0x41, 0xda, 0xca, 0x4f, 0xdd, 0x03, 0x4a, 0x82, 0x37, 0x2d, 0xb5, 0x0e, 0x1a, 0x0f, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 106 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0x45, 0x74, 0xd9, 0xcd, 0x00, 0xcf, 0x2b, 0x5a, 0x7f, 0x77, 0xe5, 0x3c, 0xd5, 0x78, 0x85 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 107 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0x3d, 0xe3, 0x92, 0x36, 0x57, 0x0a, 0xba, 0x83, 0xab, 0x9b, 0x73, 0x7c, 0xb5, 0x21, 0xc9 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 108 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x16, 0x59, 0x1c, 0x0f, 0x27, 0xd6, 0x0e, 0x29, 0xb8, 0x5a, 0x96, 0xc3, 0x38, 0x61, 0xa7, 0xef }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 109 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xfb, 0x5c, 0x4d, 0x4f, 0x5c, 0xb7, 0x9b, 0xe5, 0xc1, 0x74, 0xa3, 0xb1, 0xc9, 0x73, 0x48 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 110 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x4d, 0x2b, 0x61, 0x63, 0x3d, 0x16, 0x2b, 0xe5, 0x9d, 0xde, 0x04, 0x22, 0x2f, 0x47, 0x40 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 111 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0x75, 0x0f, 0xf2, 0x63, 0xa6, 0x5e, 0x1f, 0x9e, 0x92, 0x4c, 0xcf, 0xd9, 0x8f, 0x3e, 0x37 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 112 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0xd0, 0x66, 0x2d, 0x6e, 0xae, 0xdd, 0xed, 0xeb, 0xae, 0x7f, 0x7e, 0xa3, 0xa4, 0xf6, 0xb6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 113 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0xc4, 0x6b, 0xb3, 0x06, 0x92, 0xbe, 0x65, 0x7f, 0x7e, 0xaa, 0x93, 0xeb, 0xad, 0x98, 0x97 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 114 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0x39, 0x94, 0xcf, 0xb9, 0xda, 0x28, 0x5a, 0x5d, 0x96, 0x42, 0xe1, 0x75, 0x9b, 0x22, 0x4a }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 115 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xbf, 0x57, 0x87, 0x7b, 0x7b, 0x17, 0x38, 0x5c, 0x85, 0xd0, 0xb5, 0x48, 0x51, 0xe3, 0x71 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 116 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0xa5, 0xc0, 0x97, 0xcd, 0xc1, 0x53, 0x2a, 0xc0, 0x71, 0xd5, 0x7b, 0x1d, 0x28, 0xd1, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 117 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x0c, 0x53, 0xfa, 0x37, 0x31, 0x1f, 0xc1, 0x0b, 0xd2, 0xa9, 0x98, 0x1f, 0x51, 0x31, 0x74 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 118 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x4f, 0x97, 0x0c, 0x0a, 0x25, 0xc4, 0x18, 0x14, 0xbd, 0xae, 0x2e, 0x50, 0x6b, 0xe3, 0xb4 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 119 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2d, 0xce, 0x3a, 0xcb, 0x72, 0x7c, 0xd1, 0x3c, 0xcd, 0x76, 0xd4, 0x25, 0xea, 0x56, 0xe4, 0xf6 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 120 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0x60, 0x47, 0x4d, 0x50, 0x4b, 0x9b, 0x3e, 0xef, 0xb6, 0x8d, 0x35, 0xf2, 0x45, 0xf4, 0xb3 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 121 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xa8, 0xa9, 0x47, 0x76, 0x66, 0x35, 0xde, 0xc3, 0x75, 0x53, 0xd9, 0xa6, 0xc0, 0xcb, 0xb7 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 122 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0xd6, 0xcf, 0xe6, 0x88, 0x1f, 0x2b, 0xf4, 0x97, 0xdd, 0x14, 0xcd, 0x4d, 0xdf, 0x44, 0x5b }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 123 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xc7, 0x8c, 0x13, 0x5e, 0xd9, 0xe9, 0x8c, 0x09, 0x66, 0x40, 0x64, 0x72, 0x65, 0xda, 0x1e }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 124 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5a, 0x4d, 0x40, 0x4d, 0x89, 0x17, 0xe3, 0x53, 0xe9, 0x2a, 0x21, 0x07, 0x2c, 0x3b, 0x23, 0x05 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 125 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xbc, 0x96, 0x84, 0x6b, 0x3f, 0xdc, 0x71, 0x64, 0x3f, 0x38, 0x4c, 0xd3, 0xcc, 0x3e, 0xaf }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 126 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0xa4, 0xa9, 0x14, 0x3f, 0x4e, 0x5d, 0x40, 0x48, 0x52, 0x1c, 0x4f, 0x88, 0x77, 0xd8, 0x8e }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 127 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0xf6, 0x25, 0x8c, 0x87, 0x7d, 0x5f, 0xcd, 0x89, 0x64, 0x48, 0x45, 0x38, 0xbf, 0xc9, 0x2c }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 0 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0e, 0xdd, 0x33, 0xd3, 0xc6, 0x21, 0xe5, 0x46, 0x45, 0x5b, 0xd8, 0xba, 0x14, 0x18, 0xbe, 0xc8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 1 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0xc3, 0xf8, 0x83, 0x45, 0x0c, 0x11, 0x3c, 0x64, 0xca, 0x42, 0xe1, 0x11, 0x2a, 0x9e, 0x87 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 2 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xa1, 0xda, 0x77, 0x0f, 0x5d, 0x7a, 0xc4, 0xc9, 0xef, 0x94, 0xd8, 0x22, 0xaf, 0xfd, 0x97 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 3 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x00, 0x14, 0xd6, 0x34, 0xe2, 0xb7, 0x65, 0x07, 0x77, 0xe8, 0xe8, 0x4d, 0x03, 0xcc, 0xd8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 4 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0x7e, 0x79, 0xae, 0xd0, 0xdb, 0x7e, 0x27, 0x9e, 0x95, 0x5b, 0x5f, 0x49, 0x38, 0x75, 0xa7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 5 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9e, 0xd5, 0xa7, 0x51, 0x36, 0xa9, 0x40, 0xd0, 0x96, 0x3d, 0xa3, 0x79, 0xdb, 0x4a, 0xf2, 0x6a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 6 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x29, 0x5f, 0x83, 0x46, 0x5c, 0x77, 0x55, 0xe8, 0xfa, 0x36, 0x4b, 0xac, 0x6a, 0x7e, 0xa5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 7 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb1, 0xd7, 0x58, 0x25, 0x6b, 0x28, 0xfd, 0x85, 0x0a, 0xd4, 0x94, 0x42, 0x08, 0xcf, 0x11, 0x55 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 8 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x42, 0xff, 0xb3, 0x4c, 0x74, 0x3d, 0xe4, 0xd8, 0x8c, 0xa3, 0x80, 0x11, 0xc9, 0x90, 0x89, 0x0b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 9 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x58, 0xf0, 0xec, 0xea, 0x8b, 0x21, 0x72, 0xc0, 0xc1, 0x99, 0x5f, 0x91, 0x82, 0xc0, 0xf3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 10 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0x6d, 0x77, 0x98, 0xfa, 0xc2, 0x0f, 0x82, 0xa8, 0x82, 0x3f, 0x98, 0x4d, 0x06, 0xf7, 0xf5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 11 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x1b, 0xf4, 0x4f, 0x2d, 0x16, 0xbe, 0x92, 0x8c, 0xa4, 0x4a, 0xaf, 0x7b, 0x9b, 0x10, 0x6b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 12 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xf1, 0xa3, 0x3e, 0x50, 0xd4, 0x0d, 0x10, 0x37, 0x64, 0xc7, 0x6b, 0xd4, 0xc6, 0xb6, 0xf8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 13 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x37, 0x05, 0x0c, 0x9f, 0xc0, 0xd4, 0x81, 0x7e, 0x2d, 0x69, 0xde, 0x87, 0x8a, 0xee, 0x8d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 14 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x3e, 0xcb, 0xe4, 0xa4, 0x53, 0x26, 0x9a, 0x0d, 0xd2, 0x60, 0x69, 0x46, 0x7f, 0xb5, 0xb5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 15 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xd0, 0x75, 0x4f, 0xe6, 0x8f, 0x11, 0xb9, 0xe3, 0x75, 0xd0, 0x70, 0xa6, 0x08, 0xc8, 0x84 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 16 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0xa0, 0xb3, 0xe9, 0x98, 0xd0, 0x50, 0x68, 0xa5, 0x39, 0x97, 0x78, 0x40, 0x52, 0x00, 0xb4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 17 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0x55, 0x6a, 0x33, 0x43, 0x8d, 0xb8, 0x7b, 0xc4, 0x1b, 0x17, 0x52, 0xc5, 0x5e, 0x5e, 0x49 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 18 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xfb, 0x12, 0x8d, 0x3a, 0x1a, 0xf6, 0xe5, 0x48, 0x52, 0x1b, 0xb9, 0x62, 0xbf, 0x1f, 0x05 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 19 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x29, 0x8e, 0x9c, 0x1d, 0xb5, 0x17, 0xc2, 0x15, 0xfa, 0xdf, 0xb7, 0xd2, 0xa8, 0xd6, 0x91 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 20 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa6, 0xcb, 0x76, 0x1d, 0x61, 0xf8, 0x29, 0x2d, 0x0d, 0xf3, 0x93, 0xa2, 0x79, 0xad, 0x03, 0x80 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 21 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xac, 0xd8, 0x9b, 0x13, 0xcd, 0x5f, 0x87, 0x26, 0xe3, 0x4d, 0x44, 0xfd, 0x48, 0x61, 0x08 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 22 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0xb1, 0x70, 0x3f, 0xc5, 0x7b, 0xa0, 0x9f, 0xe0, 0xc3, 0x58, 0x0f, 0xeb, 0xdd, 0x7e, 0xd4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 23 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xde, 0x11, 0x72, 0x2d, 0x89, 0x3e, 0x9f, 0x91, 0x21, 0xc3, 0x81, 0xbe, 0xcc, 0x1d, 0xa5, 0x9a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 24 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6d, 0x11, 0x4c, 0xcb, 0x27, 0xbf, 0x39, 0x10, 0x12, 0xe8, 0x97, 0x4c, 0x54, 0x6d, 0x9b, 0xf2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 25 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5c, 0xe3, 0x7e, 0x17, 0xeb, 0x46, 0x46, 0xec, 0xfa, 0xc2, 0x9b, 0x9c, 0xc3, 0x8d, 0x93, 0x40 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 26 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0xc1, 0xb6, 0xe2, 0x15, 0x71, 0x22, 0x05, 0x6d, 0x02, 0x43, 0xd8, 0xa1, 0x65, 0xcd, 0xdb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 27 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x69, 0x3e, 0x6a, 0x59, 0xd1, 0x36, 0x6c, 0x74, 0xd8, 0x23, 0x56, 0x2d, 0x7e, 0x14, 0x31 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 28 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x7c, 0x64, 0xdc, 0x84, 0xa8, 0xbb, 0xa7, 0x58, 0xed, 0x17, 0xeb, 0x02, 0x5a, 0x57, 0xe3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 29 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe1, 0x7b, 0xc7, 0x9f, 0x30, 0xea, 0xab, 0x2f, 0xac, 0x2c, 0xbb, 0xe3, 0x45, 0x8d, 0x68, 0x7a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 30 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x14, 0xbc, 0x20, 0x28, 0x00, 0x9b, 0x92, 0x3f, 0x0b, 0x01, 0x91, 0x5c, 0xe5, 0xe7, 0xc4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 31 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0x28, 0x52, 0x4a, 0x16, 0xa1, 0xe1, 0xc1, 0x45, 0x29, 0x71, 0xca, 0xa8, 0xd1, 0x34, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 32 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0x62, 0xe1, 0x63, 0x63, 0x63, 0x83, 0x60, 0xfd, 0xd6, 0xad, 0x62, 0x11, 0x27, 0x94, 0xf0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 33 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5a, 0x86, 0x88, 0xf0, 0xb2, 0xa2, 0xc1, 0x62, 0x24, 0xc1, 0x61, 0x65, 0x8f, 0xfd, 0x40, 0x44 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 34 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0xf7, 0x10, 0x84, 0x2b, 0x9b, 0xb9, 0xc3, 0x2f, 0x26, 0x64, 0x8c, 0x78, 0x68, 0x07, 0xca }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 35 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xa9, 0x8b, 0xf1, 0x1e, 0x16, 0x3f, 0x63, 0x2c, 0x47, 0xec, 0x6a, 0x49, 0x68, 0x3a, 0x89 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 36 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0f, 0x18, 0xaf, 0xf9, 0x42, 0x74, 0x69, 0x6d, 0x9b, 0x61, 0x84, 0x8b, 0xd5, 0x0a, 0xc5, 0xe5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 37 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x40, 0x85, 0x71, 0xc3, 0xe2, 0x42, 0x45, 0x40, 0x20, 0x7f, 0x83, 0x3b, 0x6d, 0xda, 0x69 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 38 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0x3f, 0xf9, 0x96, 0x94, 0x7f, 0x0c, 0x7d, 0x1f, 0x43, 0xc8, 0xf3, 0x02, 0x7b, 0x9b, 0x75 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 39 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0xf4, 0xda, 0xf4, 0xad, 0x29, 0xa3, 0x61, 0x5a, 0x9b, 0x6e, 0xce, 0x5c, 0x99, 0x51, 0x8a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 40 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x29, 0x54, 0xa4, 0x8d, 0x07, 0x74, 0xdb, 0x0b, 0x49, 0x71, 0xc5, 0x26, 0x26, 0x04, 0x15 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 41 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xf9, 0xb7, 0x61, 0x12, 0xdc, 0x65, 0x31, 0xe0, 0x7d, 0x2c, 0xfd, 0xa0, 0x44, 0x11, 0xf0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 42 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0x4d, 0x8e, 0x69, 0x91, 0x19, 0xe1, 0xfc, 0x87, 0x54, 0x5a, 0x64, 0x7f, 0xb1, 0xd3, 0x4f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 43 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0xc4, 0x80, 0x7a, 0xe1, 0x1f, 0x36, 0xf0, 0x91, 0xc5, 0x7d, 0x9f, 0xb6, 0x85, 0x48, 0xd1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 44 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xbf, 0x73, 0xaa, 0xd4, 0x9c, 0x82, 0x00, 0x7f, 0x77, 0xa5, 0xc1, 0xcc, 0xec, 0x6a, 0xb4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 45 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xb2, 0x88, 0xcc, 0x20, 0x40, 0x04, 0x90, 0x01, 0xd2, 0xc7, 0x58, 0x5a, 0xd1, 0x23, 0xfc }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 46 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x04, 0x49, 0x71, 0x10, 0xef, 0xb9, 0xdc, 0xeb, 0x13, 0xe2, 0xb1, 0x3f, 0xb4, 0x46, 0x55, 0x64 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 47 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x55, 0x0e, 0x6c, 0xb5, 0xa8, 0x8e, 0x49, 0x63, 0x4c, 0x9a, 0xb6, 0x9e, 0xda, 0x04, 0x30 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 48 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x76, 0x84, 0x73, 0xce, 0x98, 0x43, 0xea, 0x66, 0xa8, 0x14, 0x05, 0xdd, 0x50, 0xb3, 0x45 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 49 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x2f, 0x43, 0x03, 0x83, 0xf9, 0x08, 0x4e, 0x03, 0xa6, 0x53, 0x57, 0x1e, 0x06, 0x5d, 0xe6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 50 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x4e, 0x66, 0xc0, 0x7b, 0xae, 0x3e, 0x79, 0xfb, 0x7d, 0x21, 0x08, 0x47, 0xa3, 0xb0, 0xba }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 51 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x90, 0x78, 0x51, 0x25, 0x50, 0x5f, 0xad, 0x59, 0xb1, 0x3c, 0x18, 0x6d, 0xd6, 0x6c, 0xe3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 52 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x52, 0x7a, 0x6a, 0xeb, 0xda, 0xec, 0x9e, 0xae, 0xf8, 0xed, 0xa2, 0xcb, 0x77, 0x83, 0xe5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 53 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xfd, 0xaf, 0x53, 0xeb, 0xbc, 0x98, 0x80, 0xc2, 0x28, 0x61, 0x7d, 0x6a, 0x9b, 0x54, 0x8b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 54 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x78, 0x61, 0x04, 0xb9, 0x74, 0x4b, 0x98, 0xf0, 0x52, 0xc4, 0x6f, 0x1c, 0x85, 0x0d, 0x0b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 55 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xab, 0x30, 0x13, 0xdd, 0x1e, 0x61, 0xdf, 0x06, 0xcb, 0xaf, 0x34, 0xca, 0x2a, 0xee, 0x78 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 56 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x70, 0x46, 0x9b, 0xe9, 0x72, 0x30, 0x30, 0xfd, 0xcc, 0x73, 0xa8, 0xcd, 0x4f, 0xbb, 0x10 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 57 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa3, 0x5a, 0x63, 0xf5, 0x34, 0x3e, 0xbe, 0x9e, 0xf8, 0x16, 0x7b, 0xcb, 0x48, 0xad, 0x12, 0x2e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 58 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0x86, 0x87, 0xf0, 0x75, 0x7a, 0x21, 0x0e, 0x9f, 0xdf, 0x18, 0x12, 0x04, 0xc3, 0x08, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 59 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0x18, 0x1e, 0x84, 0xbd, 0x54, 0x57, 0xd2, 0x6a, 0x88, 0xfb, 0xae, 0x96, 0x01, 0x8f, 0xb0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 60 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x65, 0x33, 0x17, 0xb9, 0x36, 0x2b, 0x6f, 0x9b, 0x9e, 0x1a, 0x58, 0x0e, 0x68, 0xd4, 0x94, 0xb5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 61 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x5c, 0x9d, 0xc0, 0xb6, 0x89, 0xf0, 0x3c, 0x45, 0x86, 0x7b, 0x5f, 0xaa, 0x5c, 0x18, 0xd1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 62 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xa4, 0xd9, 0x6d, 0x56, 0xdd, 0xa3, 0x98, 0xb9, 0xaa, 0xbe, 0xcf, 0xc7, 0x57, 0x29, 0xfd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 63 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xbe, 0x19, 0xe0, 0x53, 0x63, 0x5f, 0x09, 0xf2, 0x66, 0x5e, 0x7b, 0xae, 0x85, 0xb4, 0x2d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 64 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0xcd, 0x65, 0x28, 0x42, 0x92, 0x6a, 0xea, 0x4a, 0xa6, 0x13, 0x7b, 0xb2, 0xbe, 0x2b, 0x5e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 65 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x3d, 0x4a, 0x4f, 0x38, 0xeb, 0xb3, 0x37, 0xd1, 0x0a, 0xa8, 0x4e, 0x91, 0x71, 0xa5, 0x54 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 66 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0xbf, 0xf7, 0xff, 0x45, 0x4b, 0x0e, 0xc5, 0xa4, 0xa2, 0xa6, 0x95, 0x66, 0xe2, 0xcb, 0x84 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 67 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0x35, 0xd5, 0x65, 0xac, 0xe3, 0xf3, 0x1e, 0xb2, 0x49, 0xba, 0x2c, 0xc6, 0x76, 0x5d, 0x7a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 68 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x0e, 0x91, 0xfc, 0x32, 0x69, 0xee, 0xcf, 0x32, 0x31, 0xc6, 0xe9, 0x94, 0x56, 0x97, 0xc6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 69 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x69, 0xcf, 0xad, 0xf5, 0x1f, 0x8e, 0x60, 0x4d, 0x9c, 0xc3, 0x71, 0x82, 0xf6, 0x63, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 70 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x78, 0x66, 0x37, 0x3f, 0x24, 0xa0, 0xb6, 0xed, 0x56, 0xe0, 0xd9, 0x6f, 0xcd, 0xaf, 0xb8, 0x77 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 71 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xa4, 0x48, 0xc2, 0xaa, 0xc9, 0x54, 0xf5, 0xd8, 0x12, 0xe9, 0xd7, 0x84, 0x94, 0x44, 0x6a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 72 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xac, 0xc5, 0x59, 0x9d, 0xd8, 0xac, 0x02, 0x23, 0x9a, 0x0f, 0xef, 0x4a, 0x36, 0xdd, 0x16, 0x68 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 73 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x76, 0x44, 0x68, 0xbb, 0x10, 0x38, 0x28, 0xcf, 0x7e, 0x14, 0x73, 0xce, 0x89, 0x50, 0x73 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 74 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x0d, 0x02, 0x89, 0x36, 0x83, 0xb9, 0xf1, 0x80, 0x45, 0x8e, 0x4a, 0xa6, 0xb7, 0x39, 0x82 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 75 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0xd9, 0xb0, 0x17, 0xd3, 0x02, 0xdf, 0x41, 0x0a, 0x93, 0x7d, 0xcd, 0xb8, 0xbb, 0x6e, 0x43 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 76 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x16, 0x23, 0xcc, 0x44, 0x31, 0x3c, 0xff, 0x44, 0x0b, 0x15, 0x94, 0xa7, 0xe2, 0x1c, 0xc6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 77 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x28, 0x4c, 0xa2, 0xfa, 0x35, 0x80, 0x7b, 0x8b, 0x0a, 0xe4, 0xd1, 0x9e, 0x11, 0xd7, 0xdb, 0xd7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 78 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xe9, 0x76, 0x87, 0x57, 0x55, 0xf9, 0x40, 0x1d, 0x54, 0xf3, 0x6e, 0x2a, 0x23, 0xa5, 0x94 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 79 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xec, 0x19, 0x8a, 0x18, 0xe1, 0x0e, 0x53, 0x24, 0x03, 0xb7, 0xe2, 0x08, 0x87, 0xc8, 0xdd, 0x80 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 80 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x54, 0x5d, 0x50, 0xeb, 0xd9, 0x19, 0xe4, 0xa6, 0x94, 0x9d, 0x96, 0xad, 0x47, 0xe4, 0x6a, 0x80 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 81 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0xdf, 0xb5, 0x27, 0x06, 0x0e, 0x0a, 0x71, 0x00, 0x9c, 0x7b, 0xb0, 0xc6, 0x8f, 0x1d, 0x44 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 82 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xfa, 0x13, 0x22, 0xea, 0x33, 0xda, 0x21, 0x73, 0xa0, 0x24, 0xf2, 0xff, 0x0d, 0x89, 0x6d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 83 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0x85, 0xb1, 0xa7, 0x5b, 0x0f, 0x3b, 0xd9, 0x58, 0xdc, 0xd0, 0xe2, 0x93, 0x18, 0xc5, 0x21 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 84 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x38, 0xf6, 0x7b, 0x9e, 0x98, 0xe4, 0xa9, 0x7b, 0x6d, 0xf0, 0x30, 0xa9, 0xfc, 0xdd, 0x01, 0x04 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 85 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x2a, 0xff, 0xfb, 0x2c, 0x88, 0x0e, 0x82, 0xb0, 0x59, 0x26, 0xd0, 0xfc, 0x6c, 0x44, 0x8b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 86 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0x79, 0x80, 0xce, 0x7b, 0x10, 0x5c, 0xf5, 0x30, 0x95, 0x2d, 0x74, 0xda, 0xaf, 0x79, 0x8c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 87 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x36, 0x95, 0xe1, 0x35, 0x1b, 0x9d, 0x68, 0x58, 0xbd, 0x95, 0x8c, 0xf5, 0x13, 0xef, 0x6c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 88 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6d, 0xa0, 0x49, 0x0b, 0xa0, 0xba, 0x03, 0x43, 0xb9, 0x35, 0x68, 0x1d, 0x2c, 0xce, 0x5b, 0xa1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 89 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0xea, 0x23, 0xaf, 0x08, 0x53, 0x40, 0x11, 0xc6, 0x00, 0x09, 0xab, 0x29, 0xad, 0xa2, 0xf1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 90 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x13, 0x80, 0x6c, 0xf1, 0x9c, 0xc3, 0x87, 0x21, 0x55, 0x4d, 0x7c, 0x0f, 0xcd, 0xcd, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 91 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x38, 0xaf, 0x1f, 0x4f, 0x69, 0xba, 0xe9, 0xd8, 0x5d, 0xd1, 0x88, 0xdc, 0xdf, 0x06, 0x88 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 92 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xcf, 0x44, 0xc9, 0x2d, 0x55, 0x0b, 0xfb, 0x1e, 0xd2, 0x8e, 0xf5, 0x83, 0xdd, 0xf5, 0xd7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 93 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd0, 0x6e, 0x31, 0x95, 0xb5, 0x37, 0x6f, 0x10, 0x9d, 0x5c, 0x4e, 0xc6, 0xc5, 0xd6, 0x2c, 0xed }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 94 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x40, 0xde, 0x01, 0x4d, 0x3d, 0x61, 0x07, 0x07, 0x27, 0x9b, 0x13, 0x24, 0x2a, 0x5c, 0x36 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 95 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0xc5, 0xc6, 0xff, 0xa5, 0xe0, 0xbd, 0x3a, 0x94, 0xc8, 0x8f, 0x6b, 0x6f, 0x7c, 0x16, 0xb9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 96 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3e, 0x40, 0xc3, 0x90, 0x1c, 0xd7, 0xef, 0xfc, 0x22, 0xbf, 0xfc, 0x35, 0xde, 0xe0, 0xb4, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 97 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x33, 0x05, 0xc7, 0x2b, 0xed, 0xfa, 0xb9, 0x73, 0x82, 0xc4, 0x06, 0xd0, 0xc4, 0x9b, 0xc6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 98 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xbb, 0xaa, 0xb2, 0x2a, 0x6b, 0xd4, 0x92, 0x5a, 0x99, 0xa2, 0xb4, 0x08, 0xd2, 0xdb, 0xae }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 99 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0x7c, 0x5b, 0x8f, 0xcd, 0x05, 0x33, 0xab, 0x98, 0xbc, 0x51, 0xe2, 0x7a, 0x6c, 0xe4, 0x61 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 100 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x9c, 0x04, 0xff, 0x4c, 0x07, 0x51, 0x3c, 0x0b, 0x3e, 0xf0, 0x5c, 0x03, 0xe3, 0x37, 0xb5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 101 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0x7a, 0xf0, 0xe8, 0x95, 0xdd, 0xa5, 0xeb, 0x98, 0xef, 0xc6, 0x80, 0x66, 0xe8, 0x4c, 0x54 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 102 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x71, 0x67, 0xf3, 0x81, 0x2a, 0xff, 0xf1, 0xff, 0xac, 0xb4, 0xa9, 0x34, 0x37, 0x9f, 0xc3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 103 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xb1, 0xdc, 0x3a, 0x9c, 0x72, 0x97, 0x2e, 0x42, 0x5a, 0xe2, 0xef, 0x3e, 0xb5, 0x97, 0xcd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 104 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xae, 0xaa, 0x3a, 0x21, 0x3e, 0x96, 0x8d, 0x4b, 0x5b, 0x67, 0x9d, 0x3a, 0x2c, 0x97, 0xfe }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 105 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x92, 0x41, 0xda, 0xca, 0x4f, 0xdd, 0x03, 0x4a, 0x82, 0x37, 0x2d, 0xb5, 0x0e, 0x1a, 0x0f, 0x3f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 106 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0x45, 0x74, 0xd9, 0xcd, 0x00, 0xcf, 0x2b, 0x5a, 0x7f, 0x77, 0xe5, 0x3c, 0xd5, 0x78, 0x85 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 107 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0x3d, 0xe3, 0x92, 0x36, 0x57, 0x0a, 0xba, 0x83, 0xab, 0x9b, 0x73, 0x7c, 0xb5, 0x21, 0xc9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 108 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x16, 0x59, 0x1c, 0x0f, 0x27, 0xd6, 0x0e, 0x29, 0xb8, 0x5a, 0x96, 0xc3, 0x38, 0x61, 0xa7, 0xef }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 109 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xfb, 0x5c, 0x4d, 0x4f, 0x5c, 0xb7, 0x9b, 0xe5, 0xc1, 0x74, 0xa3, 0xb1, 0xc9, 0x73, 0x48 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 110 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x4d, 0x2b, 0x61, 0x63, 0x3d, 0x16, 0x2b, 0xe5, 0x9d, 0xde, 0x04, 0x22, 0x2f, 0x47, 0x40 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 111 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0x75, 0x0f, 0xf2, 0x63, 0xa6, 0x5e, 0x1f, 0x9e, 0x92, 0x4c, 0xcf, 0xd9, 0x8f, 0x3e, 0x37 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 112 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0xd0, 0x66, 0x2d, 0x6e, 0xae, 0xdd, 0xed, 0xeb, 0xae, 0x7f, 0x7e, 0xa3, 0xa4, 0xf6, 0xb6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 113 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0xc4, 0x6b, 0xb3, 0x06, 0x92, 0xbe, 0x65, 0x7f, 0x7e, 0xaa, 0x93, 0xeb, 0xad, 0x98, 0x97 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 114 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0x39, 0x94, 0xcf, 0xb9, 0xda, 0x28, 0x5a, 0x5d, 0x96, 0x42, 0xe1, 0x75, 0x9b, 0x22, 0x4a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 115 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xbf, 0x57, 0x87, 0x7b, 0x7b, 0x17, 0x38, 0x5c, 0x85, 0xd0, 0xb5, 0x48, 0x51, 0xe3, 0x71 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 116 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0xa5, 0xc0, 0x97, 0xcd, 0xc1, 0x53, 0x2a, 0xc0, 0x71, 0xd5, 0x7b, 0x1d, 0x28, 0xd1, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 117 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x0c, 0x53, 0xfa, 0x37, 0x31, 0x1f, 0xc1, 0x0b, 0xd2, 0xa9, 0x98, 0x1f, 0x51, 0x31, 0x74 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 118 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x4f, 0x97, 0x0c, 0x0a, 0x25, 0xc4, 0x18, 0x14, 0xbd, 0xae, 0x2e, 0x50, 0x6b, 0xe3, 0xb4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 119 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2d, 0xce, 0x3a, 0xcb, 0x72, 0x7c, 0xd1, 0x3c, 0xcd, 0x76, 0xd4, 0x25, 0xea, 0x56, 0xe4, 0xf6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 120 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0x60, 0x47, 0x4d, 0x50, 0x4b, 0x9b, 0x3e, 0xef, 0xb6, 0x8d, 0x35, 0xf2, 0x45, 0xf4, 0xb3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 121 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xa8, 0xa9, 0x47, 0x76, 0x66, 0x35, 0xde, 0xc3, 0x75, 0x53, 0xd9, 0xa6, 0xc0, 0xcb, 0xb7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 122 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0xd6, 0xcf, 0xe6, 0x88, 0x1f, 0x2b, 0xf4, 0x97, 0xdd, 0x14, 0xcd, 0x4d, 0xdf, 0x44, 0x5b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 123 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xc7, 0x8c, 0x13, 0x5e, 0xd9, 0xe9, 0x8c, 0x09, 0x66, 0x40, 0x64, 0x72, 0x65, 0xda, 0x1e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 124 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5a, 0x4d, 0x40, 0x4d, 0x89, 0x17, 0xe3, 0x53, 0xe9, 0x2a, 0x21, 0x07, 0x2c, 0x3b, 0x23, 0x05 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 125 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xbc, 0x96, 0x84, 0x6b, 0x3f, 0xdc, 0x71, 0x64, 0x3f, 0x38, 0x4c, 0xd3, 0xcc, 0x3e, 0xaf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 126 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0xa4, 0xa9, 0x14, 0x3f, 0x4e, 0x5d, 0x40, 0x48, 0x52, 0x1c, 0x4f, 0x88, 0x77, 0xd8, 0x8e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey128.rsp test vector 127 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0xf6, 0x25, 0x8c, 0x87, 0x7d, 0x5f, 0xcd, 0x89, 0x64, 0x48, 0x45, 0x38, 0xbf, 0xc9, 0x2c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },

	{ "NIST KAT-AES ECBVarKey192.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xde, 0x88, 0x5d, 0xc8, 0x7f, 0x5a, 0x92, 0x59, 0x40, 0x82, 0xd0, 0x2c, 0xc1, 0xe1, 0xb4, 0x2c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x2b, 0x07, 0x4e, 0x80, 0xf2, 0xa5, 0x97, 0xbf, 0x5f, 0xeb, 0xd8, 0xea, 0x5d, 0xa5, 0x5e }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0xcc, 0xed, 0xf8, 0xde, 0x59, 0x2c, 0x22, 0xfb, 0x81, 0x34, 0x7b, 0x79, 0xf2, 0xdb, 0x1f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0x0b, 0x09, 0xf2, 0x67, 0xc4, 0x51, 0x45, 0xdb, 0x2f, 0x82, 0x6c, 0x25, 0x82, 0xd3, 0x5c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xd8, 0x07, 0xef, 0x76, 0x52, 0xd7, 0xeb, 0x0e, 0x13, 0xc8, 0xb5, 0xe1, 0x5b, 0x3b, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x78, 0xbc, 0xf8, 0xdd, 0x8f, 0xd7, 0x22, 0x41, 0x22, 0x3a, 0xd2, 0x4b, 0x31, 0xb8, 0xa4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x10, 0xf6, 0x54, 0x34, 0x3e, 0x8f, 0x27, 0xe1, 0x2c, 0x83, 0xa4, 0x8d, 0x24, 0xff, 0x81 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x83, 0x3f, 0x71, 0x25, 0x8d, 0x53, 0x03, 0x6b, 0x02, 0x95, 0x2c, 0x76, 0xc7, 0x44, 0xf5, 0xa1 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0xa8, 0x3f, 0xf2, 0x00, 0xcf, 0xf9, 0x31, 0x8a, 0x92, 0xf8, 0x69, 0x1a, 0x06, 0xb0, 0x9f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x62, 0x0c, 0xcb, 0xe9, 0xf3, 0x29, 0x2a, 0xbd, 0xf2, 0x17, 0x6b, 0x09, 0xf0, 0x4e, 0xba }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0xba, 0xbc, 0x4b, 0x3f, 0x51, 0x6c, 0x9a, 0xaf, 0xb3, 0x5f, 0x41, 0x40, 0xb5, 0x48, 0xf9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0x18, 0x78, 0x24, 0xd9, 0xc4, 0x58, 0x2b, 0x09, 0x16, 0x49, 0x3e, 0xcb, 0xde, 0x8c, 0x57 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x0a, 0xd5, 0x53, 0x17, 0x7f, 0xd5, 0xea, 0x10, 0x92, 0xc9, 0xd6, 0x26, 0xa2, 0x9d, 0xc4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 13 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xdc, 0x46, 0xc3, 0x72, 0x61, 0x19, 0x41, 0x24, 0xec, 0xae, 0xbd, 0x68, 0x04, 0x08, 0xec }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 14 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe4, 0xf2, 0xf2, 0xae, 0x23, 0xe9, 0xb1, 0x0b, 0xac, 0xfa, 0x58, 0x60, 0x15, 0x31, 0xba, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 15 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0xd6, 0x7c, 0xf1, 0xa1, 0xe9, 0x1e, 0x8f, 0xf3, 0xa5, 0x7a, 0x17, 0x2c, 0x7b, 0xf4, 0x12 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 16 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x70, 0x6b, 0xe0, 0x69, 0x67, 0x88, 0x4e, 0x84, 0x7d, 0x13, 0x71, 0x28, 0xce, 0x47, 0xb3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 17 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0xf8, 0xb4, 0x09, 0xb0, 0x58, 0x59, 0x09, 0xaa, 0xd3, 0xa7, 0xb5, 0xa2, 0x19, 0x07, 0x2a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 18 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0x4b, 0x7b, 0xff, 0x02, 0x90, 0xc7, 0x83, 0x44, 0xc5, 0x4a, 0x23, 0xb7, 0x22, 0xcd, 0x20 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 19 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x09, 0x36, 0x57, 0x55, 0x2d, 0x44, 0x14, 0x22, 0x7c, 0xe1, 0x61, 0xe9, 0xeb, 0xf7, 0xdd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 20 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe1, 0xaf, 0x1e, 0x7d, 0x8b, 0xc2, 0x25, 0xed, 0x4d, 0xff, 0xb7, 0x71, 0xec, 0xbb, 0x9e, 0x67 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 21 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x65, 0x55, 0x25, 0x36, 0x35, 0xd8, 0x43, 0x21, 0x56, 0xcf, 0xd9, 0xc1, 0x1b, 0x14, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 22 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x40, 0x35, 0x07, 0x4a, 0x5d, 0x42, 0x60, 0xc9, 0x0c, 0xbd, 0x6d, 0xa6, 0xc3, 0xfc, 0xeb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 23 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0x6e, 0xe4, 0x16, 0xf9, 0xad, 0x1c, 0x10, 0x3e, 0xb0, 0xcc, 0x96, 0x75, 0x1c, 0x88, 0xe1 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 24 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x8a, 0xe2, 0xa4, 0x63, 0x7a, 0xc0, 0xa7, 0x89, 0x0a, 0x8f, 0xd1, 0x48, 0x54, 0x45, 0xc9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 25 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x20, 0x12, 0xec, 0x8f, 0xad, 0xed, 0x08, 0x25, 0xfb, 0x2f, 0xa7, 0x0a, 0xb3, 0x0c, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 26 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x8a, 0x64, 0xb4, 0x6b, 0x5d, 0x88, 0xbf, 0x7f, 0x24, 0x7d, 0x4d, 0xba, 0xf3, 0x8f, 0x05 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 27 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x68, 0x25, 0x37, 0x62, 0xe2, 0xcc, 0x81, 0xb4, 0x2d, 0x1e, 0x50, 0x01, 0x76, 0x26, 0x99 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 28 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x41, 0xf8, 0x3b, 0x38, 0xce, 0x50, 0x32, 0xc6, 0xcd, 0x7a, 0xf9, 0x8c, 0xf6, 0x20, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 29 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0xa8, 0x99, 0x90, 0xcd, 0x14, 0x11, 0x75, 0x0d, 0x5f, 0xb0, 0xdc, 0x98, 0x84, 0x47, 0xd4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 30 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xac, 0xcc, 0x8e, 0xd6, 0x29, 0xed, 0xf8, 0xc6, 0x8a, 0x53, 0x91, 0x83, 0xb1, 0xea, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 31 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb1, 0x6f, 0xa7, 0x1f, 0x84, 0x6b, 0x81, 0xa1, 0x3f, 0x36, 0x1c, 0x43, 0xa8, 0x51, 0xf2, 0x90 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 32 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xad, 0x6e, 0xfd, 0xff, 0x59, 0x75, 0xae, 0xe7, 0x69, 0x22, 0x34, 0xbc, 0xd5, 0x44, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 33 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0xfd, 0xb0, 0x5a, 0x78, 0x3d, 0x03, 0x08, 0x2d, 0xfe, 0x5f, 0xdd, 0x80, 0xa0, 0x0b, 0x17 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 34 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0x81, 0xb5, 0x84, 0x76, 0x69, 0x97, 0xaf, 0x6b, 0xa5, 0x52, 0x9d, 0x3b, 0xdd, 0x86, 0x09 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 35 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0c, 0xf4, 0xff, 0x4f, 0x49, 0xc8, 0xa0, 0xca, 0x06, 0x0c, 0x44, 0x34, 0x99, 0xe2, 0x93, 0x13 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 36 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x4b, 0xa8, 0xa8, 0xe0, 0x29, 0xf8, 0xb2, 0x6d, 0x8a, 0xff, 0xf9, 0xdf, 0x13, 0x3b, 0xb6 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 37 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0xfe, 0xbf, 0x64, 0x36, 0x0f, 0x38, 0xe4, 0xe6, 0x35, 0x58, 0xf0, 0xff, 0xc5, 0x50, 0xc3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 38 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xad, 0x98, 0xcb, 0xf7, 0x25, 0x13, 0x7d, 0x6a, 0x81, 0x08, 0xc2, 0xbe, 0xd9, 0x93, 0x22 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 39 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0xfa, 0xa9, 0x96, 0x22, 0x61, 0x98, 0xb3, 0xe2, 0x61, 0x04, 0x13, 0xce, 0x1b, 0x3f, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 40 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2a, 0x8c, 0xe6, 0x74, 0x7a, 0x7e, 0x39, 0x36, 0x78, 0x28, 0xe2, 0x90, 0x84, 0x85, 0x02, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 41 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x37, 0x36, 0xe8, 0xb8, 0xf8, 0x9c, 0xa1, 0xe3, 0x7b, 0x6d, 0xea, 0xb4, 0x0f, 0xac, 0xf1 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 42 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0xf7, 0x97, 0xe5, 0x04, 0x18, 0xb9, 0x5f, 0xa6, 0x01, 0x33, 0x33, 0x91, 0x7a, 0x94, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 43 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x58, 0xde, 0x37, 0xc2, 0xec, 0xe2, 0xa0, 0x2c, 0x73, 0xc0, 0x1f, 0xed, 0xc9, 0xa1, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 44 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x9b, 0x87, 0xae, 0x77, 0xba, 0xe7, 0x06, 0x80, 0x39, 0x66, 0xc6, 0x6c, 0x73, 0xad, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 45 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x65, 0xab, 0x8d, 0xf8, 0xff, 0xd7, 0x82, 0xe3, 0x58, 0x12, 0x1a, 0x4a, 0x4f, 0xc5, 0x41 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 46 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xdc, 0xd9, 0xe6, 0xf7, 0x5e, 0x6c, 0x36, 0xc8, 0xda, 0xee, 0x04, 0x66, 0xf0, 0xed, 0x74 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 47 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x9a, 0x63, 0x7b, 0xeb, 0x1c, 0x03, 0x04, 0xf1, 0x40, 0x14, 0xc0, 0x37, 0xe7, 0x36, 0xdd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 48 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0x5f, 0x0a, 0x25, 0xe8, 0x4a, 0xc9, 0x30, 0xd9, 0x96, 0x28, 0x1a, 0x5f, 0x95, 0x4d, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 49 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x42, 0xe4, 0x07, 0x4b, 0x29, 0x27, 0x97, 0x3e, 0x8d, 0x17, 0xff, 0xa9, 0x2f, 0x7f, 0xe6, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 50 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xe2, 0xa9, 0xd2, 0xc1, 0x82, 0x44, 0x49, 0xc6, 0x9e, 0x3e, 0x03, 0x98, 0xf1, 0x29, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 51 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0xf2, 0x9c, 0x1e, 0x1f, 0x62, 0x84, 0x7a, 0x15, 0x25, 0x3b, 0x28, 0xa1, 0xe9, 0xd7, 0x12 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 52 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xed, 0x5d, 0x29, 0xb9, 0x03, 0xf3, 0x1e, 0x89, 0x83, 0xef, 0x8b, 0x0a, 0x2b, 0xf9, 0x90 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 53 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xb8, 0x07, 0x02, 0x70, 0x81, 0x0f, 0x9d, 0x02, 0x3f, 0x9d, 0xd7, 0xff, 0x3b, 0x4a, 0xa2 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 54 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0xd4, 0x6e, 0x15, 0x5c, 0x12, 0x28, 0xf6, 0x1d, 0x1a, 0x0d, 0xb4, 0x81, 0x5e, 0xcc, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 55 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x61, 0x08, 0xd1, 0xd9, 0x80, 0x71, 0x42, 0x8e, 0xec, 0xee, 0xf1, 0x71, 0x4b, 0x96, 0xdd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 56 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0x5b, 0x25, 0xb7, 0x1b, 0x62, 0x96, 0xcf, 0x73, 0xdd, 0x2c, 0xdc, 0xac, 0x2f, 0x70, 0xb1 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 57 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xab, 0xa9, 0x5e, 0x8a, 0x06, 0xa2, 0xd9, 0xd3, 0x53, 0x0d, 0x26, 0x77, 0x87, 0x8c, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 58 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0x70, 0xd2, 0x0e, 0x89, 0xb4, 0x67, 0xe8, 0xf5, 0x17, 0x60, 0x61, 0xb8, 0x1d, 0xd3, 0x96 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 59 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x8f, 0x44, 0x67, 0xa5, 0xd8, 0xf1, 0xe7, 0x30, 0x7d, 0xc3, 0x0b, 0x34, 0xe4, 0x04, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 60 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbc, 0xea, 0x28, 0xe9, 0x07, 0x1b, 0x5a, 0x23, 0x02, 0x97, 0x0f, 0xf3, 0x52, 0x45, 0x1b, 0xc5 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 61 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x23, 0xc0, 0x0b, 0xc1, 0x77, 0xd3, 0x31, 0xad, 0x31, 0x2e, 0x09, 0xc9, 0x01, 0x5c, 0x1c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 62 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0xac, 0x61, 0xe3, 0x18, 0x37, 0x47, 0xb3, 0xf5, 0x83, 0x6d, 0xa2, 0x1a, 0x1b, 0xc4, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 63 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x7b, 0x07, 0x57, 0x91, 0x87, 0x88, 0x80, 0xb4, 0x41, 0x89, 0xd3, 0x52, 0x2b, 0x8c, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 64 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x71, 0x32, 0xd0, 0xc0, 0xe4, 0xa0, 0x75, 0x93, 0xcf, 0x12, 0xeb, 0xb1, 0x2b, 0xe7, 0x68, 0x8c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 65 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0xfb, 0xac, 0x16, 0x44, 0xde, 0xb0, 0xc7, 0x84, 0x27, 0x5f, 0xe5, 0x6e, 0x19, 0xea, 0xd3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 66 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x05, 0x06, 0x3f, 0x30, 0xf4, 0x22, 0x8b, 0x37, 0x4e, 0x24, 0x59, 0x73, 0x8f, 0x26, 0xbb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 67 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x97, 0x5b, 0x5f, 0x48, 0xbb, 0x68, 0xfc, 0xbb, 0xc7, 0xce, 0xa9, 0x3b, 0x45, 0x2e, 0xd7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 68 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x3f, 0x25, 0x76, 0xe2, 0xaf, 0xed, 0xc7, 0x4b, 0xb1, 0xca, 0x7e, 0xee, 0xc1, 0xc0, 0xe7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 69 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0xc4, 0x03, 0xf5, 0xf9, 0x66, 0xe0, 0xe3, 0xd9, 0xf2, 0x96, 0xd6, 0x22, 0x6d, 0xca, 0x28 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 70 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xc2, 0x09, 0x08, 0x24, 0x9a, 0xb4, 0xa3, 0x4d, 0x6d, 0xd0, 0xa3, 0x13, 0x27, 0xff, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 71 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0x54, 0x13, 0x29, 0xec, 0xb6, 0x15, 0x9a, 0xb2, 0x3b, 0x7f, 0xc5, 0xe6, 0xa2, 0x1b, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 72 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0xa1, 0xac, 0xf1, 0xa2, 0xed, 0x9b, 0xa7, 0x2b, 0xc6, 0xde, 0xb3, 0x1d, 0x88, 0xb8, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 73 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x8b, 0xd8, 0xed, 0xda, 0xbb, 0x6f, 0x3b, 0xf0, 0xd5, 0xa8, 0xa2, 0x7b, 0xe1, 0xfe, 0x8a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 74 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x3c, 0x7d, 0x76, 0x85, 0xe1, 0x4e, 0xc6, 0x6b, 0xbb, 0x96, 0xb8, 0xf0, 0x5b, 0x6d, 0xdd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 75 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0x75, 0x2e, 0xef, 0xc8, 0xc2, 0xa9, 0x3f, 0x91, 0xb6, 0xe7, 0x3e, 0xb0, 0x7c, 0xca, 0x6e }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 76 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x93, 0xe7, 0xd6, 0x2f, 0x6c, 0xe5, 0x02, 0xc6, 0x4f, 0x75, 0xe2, 0x81, 0xf9, 0xc0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 77 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0xfd, 0x99, 0x9b, 0xe5, 0xd0, 0xcf, 0xa3, 0x57, 0x32, 0xc0, 0xdd, 0xc8, 0x8f, 0xf5, 0xa5 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 78 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x64, 0x7c, 0x76, 0xa3, 0x00, 0xc3, 0x17, 0x3b, 0x84, 0x14, 0x87, 0xeb, 0x2b, 0xae, 0x9f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 79 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x17, 0x2d, 0xf8, 0xb0, 0x2f, 0x04, 0xb5, 0x3a, 0xda, 0xb0, 0x28, 0xb4, 0xe0, 0x1a, 0xcd, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 80 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0x4b, 0x3b, 0xf4, 0x99, 0x8a, 0xeb, 0x05, 0xaf, 0xd8, 0x7e, 0xc5, 0x36, 0x53, 0x3a, 0x36 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 81 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x83, 0xf7, 0xbf, 0x44, 0xc9, 0x7f, 0x06, 0x52, 0x58, 0xa6, 0x66, 0xca, 0xe0, 0x30, 0x20 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 82 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xd4, 0xc8, 0xa6, 0x3f, 0x80, 0x95, 0x41, 0x04, 0xde, 0x7b, 0x92, 0xce, 0xde, 0x1b, 0xe1 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 83 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0xfe, 0x61, 0x81, 0x0f, 0xd5, 0x46, 0x7c, 0xcd, 0xac, 0xb7, 0x58, 0x00, 0xf3, 0xac, 0x07 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 84 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x83, 0x0d, 0x8a, 0x25, 0x90, 0xf7, 0xd8, 0xe1, 0xb5, 0x5a, 0x73, 0x7f, 0x4a, 0xf4, 0x5f, 0x34 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 85 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfc, 0xd4, 0x68, 0x3f, 0x85, 0x80, 0x58, 0xe7, 0x43, 0x14, 0x67, 0x1d, 0x43, 0xfa, 0x2c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 86 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x52, 0x3d, 0x0b, 0xab, 0xbb, 0x82, 0xf4, 0x6e, 0xbc, 0x9e, 0x70, 0xb1, 0xcd, 0x41, 0xdd, 0xd0 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 87 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x4a, 0xab, 0x37, 0x08, 0x0d, 0x74, 0x86, 0xf7, 0xd5, 0x42, 0xa3, 0x09, 0xe5, 0x3e, 0xed }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 88 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0xc5, 0x60, 0x9d, 0x09, 0x06, 0xb2, 0x3a, 0xb9, 0xca, 0xca, 0x81, 0x6f, 0x5d, 0xbe, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 89 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x26, 0x02, 0x6e, 0xed, 0xd9, 0x1a, 0xdc, 0x6d, 0x83, 0x1c, 0xdf, 0x98, 0x94, 0xbd, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 90 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x88, 0x33, 0x0b, 0xaa, 0x4f, 0x2b, 0x61, 0x8f, 0xc9, 0xd9, 0xb0, 0x21, 0xbf, 0x50, 0x3d, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 91 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x9e, 0x0e, 0xa2, 0x24, 0x80, 0xb0, 0xba, 0xc9, 0x35, 0xc8, 0xa8, 0xeb, 0xef, 0xcd, 0xcf }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 92 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0xca, 0x77, 0x9f, 0x39, 0x8f, 0xb0, 0x4f, 0x86, 0x7d, 0xa7, 0xe8, 0xa4, 0x47, 0x56, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 93 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0xf8, 0x9c, 0x42, 0x98, 0x57, 0x86, 0xbf, 0xc4, 0x3c, 0x6d, 0xf8, 0xad, 0xa3, 0x68, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 94 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0xc1, 0xde, 0x5f, 0xb8, 0xf2, 0x1d, 0x87, 0x4e, 0x91, 0xc5, 0x3b, 0x56, 0x0c, 0x50, 0xe3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 95 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0xaa, 0x90, 0x58, 0x49, 0x0e, 0xda, 0x30, 0x60, 0x01, 0xa8, 0xa9, 0xf4, 0x8d, 0x0c, 0xa7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 96 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x4e, 0xc7, 0x1d, 0x61, 0x28, 0xd4, 0x87, 0x18, 0x65, 0xd6, 0x17, 0xc3, 0x0b, 0x37, 0xe3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 97 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0xbe, 0x1c, 0x53, 0x5b, 0x17, 0xca, 0xbd, 0x0c, 0x4d, 0x93, 0x52, 0x9d, 0x69, 0xbf, 0x47 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 98 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xef, 0x67, 0x75, 0x65, 0x07, 0xbe, 0xec, 0x9d, 0xd3, 0x86, 0x28, 0x83, 0x47, 0x80, 0x44 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 99 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xe2, 0x31, 0xfa, 0x5a, 0x59, 0x48, 0xce, 0x21, 0x34, 0xe9, 0x2f, 0xc0, 0x66, 0x4d, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 100 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x19, 0x4b, 0x8e, 0x5d, 0xda, 0x55, 0x30, 0xd0, 0xc6, 0x78, 0xc0, 0xb4, 0x8f, 0x5d, 0x92 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 101 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xbd, 0x08, 0x6f, 0x23, 0x7c, 0xc4, 0xfd, 0x99, 0xf4, 0xd7, 0x6b, 0xde, 0x6b, 0x48, 0x26 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 102 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x25, 0x97, 0x61, 0xca, 0x17, 0x13, 0x0d, 0x6e, 0xd8, 0x6d, 0x57, 0xcd, 0x79, 0x51, 0xee }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 103 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd7, 0xcb, 0xb3, 0xf3, 0x4b, 0x9b, 0x45, 0x0f, 0x24, 0xb0, 0xe8, 0x51, 0x8e, 0x54, 0xda, 0x6d }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 104 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0x5b, 0x9c, 0xae, 0xbe, 0x9f, 0x7f, 0x41, 0x7f, 0x40, 0x68, 0xd0, 0xd2, 0xee, 0x20, 0xb3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 105 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9d, 0x92, 0x4b, 0x93, 0x4a, 0x90, 0xce, 0x1f, 0xd3, 0x9b, 0x8a, 0x97, 0x94, 0xf8, 0x26, 0x72 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 106 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc5, 0x05, 0x62, 0xbf, 0x09, 0x45, 0x26, 0xa9, 0x1c, 0x5b, 0xc6, 0x3c, 0x0c, 0x22, 0x49, 0x95 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 107 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xf1, 0x18, 0x05, 0x04, 0x67, 0x43, 0xbd, 0x74, 0xf5, 0x71, 0x88, 0xd9, 0x18, 0x8d, 0xf7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 108 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0xd2, 0x74, 0xbd, 0x0f, 0x1b, 0x58, 0xae, 0x34, 0x5d, 0x9e, 0x72, 0x33, 0xf9, 0xb8, 0xf3 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 109 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9d, 0x6b, 0xdc, 0x8f, 0x4c, 0xe5, 0xfe, 0xb0, 0xf3, 0xbe, 0xd2, 0xe4, 0xb9, 0xa9, 0xbb, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 110 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0x55, 0x48, 0xbc, 0xf3, 0xf4, 0x25, 0x65, 0xf7, 0xef, 0xa9, 0x45, 0x62, 0x52, 0x8d, 0x46 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 111 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xcc, 0xae, 0xbd, 0x3a, 0x4c, 0x3e, 0x80, 0xb0, 0x63, 0x74, 0x81, 0x31, 0xba, 0x4a, 0x71 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 112 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x3c, 0xb2, 0x3d, 0x9e, 0x11, 0xc9, 0xd9, 0x3f, 0x11, 0x7e, 0x9c, 0x0a, 0x91, 0xb5, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 113 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x78, 0xf9, 0x33, 0xa2, 0x08, 0x1a, 0xc1, 0xdb, 0x84, 0xf6, 0x9d, 0x10, 0xf4, 0x52, 0x3f, 0xe0 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 114 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x61, 0xf7, 0x41, 0x2e, 0xd3, 0x20, 0xde, 0x0e, 0xdc, 0x88, 0x51, 0xc2, 0xe2, 0x43, 0x6f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 115 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x64, 0xba, 0x1c, 0xd0, 0x4c, 0xe6, 0xba, 0xb9, 0x84, 0x74, 0x33, 0x08, 0x14, 0xb4, 0xd4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 116 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x48, 0x39, 0x1b, 0xff, 0xb9, 0xcf, 0xff, 0x80, 0xac, 0x23, 0x8c, 0x88, 0x6e, 0xf0, 0xa4, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 117 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0xd2, 0xa6, 0x7d, 0xf5, 0xa9, 0x99, 0xfd, 0xbf, 0x93, 0xed, 0xd0, 0x34, 0x32, 0x96, 0xc9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 118 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xca, 0x73, 0x67, 0x39, 0x6b, 0x69, 0xa2, 0x21, 0xbd, 0x63, 0x2b, 0xea, 0x38, 0x6e, 0xec }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 119 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x0f, 0xd5, 0x02, 0x0d, 0xfe, 0x65, 0xf5, 0xf1, 0x62, 0x93, 0xec, 0x92, 0xc6, 0xfd, 0x89 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 120 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0x62, 0x99, 0x5b, 0x82, 0x17, 0xa6, 0x7f, 0x1a, 0xbc, 0x34, 0x2e, 0x14, 0x64, 0x06, 0xf8 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 121 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0xa6, 0x16, 0x4b, 0x7a, 0x60, 0xba, 0xe4, 0xe9, 0x86, 0xff, 0xac, 0x28, 0xdf, 0xad, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 122 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x64, 0xe0, 0xd7, 0xf9, 0x00, 0xe3, 0xd9, 0xc8, 0x3e, 0x4b, 0x8f, 0x96, 0x71, 0x7b, 0x21, 0x46 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 123 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1a, 0xd2, 0x56, 0x1d, 0xe8, 0xc1, 0x23, 0x2f, 0x5d, 0x8d, 0xba, 0xb4, 0x73, 0x9b, 0x6c, 0xbb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 124 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x96, 0x89, 0xe9, 0xa5, 0x57, 0xf5, 0x8b, 0x1c, 0x3b, 0xf4, 0x0c, 0x97, 0xa9, 0x09, 0x64 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 125 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x63, 0x7e, 0x4a, 0x5e, 0x63, 0x77, 0xf9, 0xcc, 0x5a, 0x86, 0x38, 0x04, 0x5d, 0xe0, 0x29 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 126 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x2e, 0x60, 0x7e, 0x5a, 0xea, 0x46, 0x88, 0x59, 0x4b, 0x45, 0xf3, 0xae, 0xe3, 0xdf, 0x90 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 127 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0xc4, 0xe4, 0x38, 0x1f, 0xee, 0xc7, 0x40, 0x54, 0x95, 0x4c, 0x05, 0xb7, 0x77, 0xa0, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 128 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x91, 0x54, 0x95, 0x14, 0x60, 0x5f, 0x38, 0x24, 0x6c, 0x9b, 0x72, 0x4a, 0xd8, 0x39, 0xf0, 0x1d }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 129 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0xb2, 0x4e, 0x3b, 0x6f, 0xef, 0xe4, 0x0a, 0x4f, 0x9e, 0xf7, 0xac, 0x6e, 0x44, 0xd7, 0x6a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 130 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x37, 0xa6, 0x83, 0xdc, 0x5d, 0x4b, 0x52, 0xab, 0xb4, 0xa1, 0x23, 0xa8, 0xdf, 0x86, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 131 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbb, 0x28, 0x52, 0xc8, 0x91, 0xc5, 0x94, 0x7d, 0x2e, 0xd4, 0x40, 0x32, 0xc4, 0x21, 0xb8, 0x5f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 132 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x9f, 0x5f, 0xbd, 0x5e, 0x8a, 0x42, 0x64, 0xc0, 0xa8, 0x5b, 0x80, 0x40, 0x9a, 0xfa, 0x5e }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 133 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xda, 0xb8, 0x09, 0xf8, 0x5a, 0x91, 0x7f, 0xe9, 0x24, 0x73, 0x3f, 0x42, 0x4a, 0xc5, 0x89 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 134 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0xef, 0x5c, 0x1f, 0x8d, 0x60, 0x51, 0x92, 0x64, 0x66, 0x95, 0xce, 0xad, 0xc6, 0x5f, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 135 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0xaa, 0x90, 0x04, 0x0b, 0x4c, 0x15, 0xa1, 0x23, 0x16, 0xb7, 0x8e, 0x0f, 0x95, 0x86, 0xfc }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 136 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xfa, 0xc8, 0x29, 0x7c, 0xea, 0xab, 0xc8, 0x7d, 0x45, 0x43, 0x50, 0x60, 0x1e, 0x06, 0x73 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 137 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0x47, 0xef, 0x56, 0x7a, 0xc2, 0x8d, 0xfe, 0x48, 0x84, 0x92, 0xf1, 0x57, 0xe2, 0xb2, 0xe0 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 138 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x84, 0x26, 0x02, 0x7d, 0xdb, 0x96, 0x2b, 0x5c, 0x5b, 0xa7, 0xeb, 0x8b, 0xc9, 0xab, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 139 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0x17, 0xfc, 0x77, 0xe7, 0x19, 0x92, 0xa1, 0x2d, 0xbe, 0x4c, 0x18, 0x06, 0x8b, 0xec, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 140 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xee, 0xbb, 0xc9, 0x88, 0x40, 0xf8, 0xae, 0x6d, 0xaf, 0x76, 0x57, 0x3b, 0x7e, 0x56, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 141 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4e, 0x11, 0xa9, 0xf7, 0x42, 0x05, 0x12, 0x5b, 0x61, 0xe0, 0xae, 0xe0, 0x47, 0xec, 0xa2, 0x0d }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 142 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x04, 0x67, 0xf5, 0x5a, 0x1f, 0x17, 0xea, 0xb8, 0x8e, 0x80, 0x01, 0x20, 0xcb, 0xc2, 0x84 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 143 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x36, 0x64, 0x9f, 0x60, 0x0b, 0x44, 0x9e, 0xe2, 0x76, 0x53, 0x0f, 0x0c, 0xd8, 0x3c, 0x11 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 144 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0xc0, 0xe3, 0x65, 0x6a, 0x9e, 0x3a, 0xc7, 0xcd, 0x37, 0x8a, 0x73, 0x7f, 0x53, 0xb6, 0x37 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 145 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xac, 0xae, 0x63, 0xd3, 0x3b, 0x92, 0x8a, 0xa8, 0x38, 0x0f, 0x8d, 0x54, 0xd8, 0x8c, 0x17 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 146 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x89, 0x35, 0xff, 0xbc, 0x75, 0xae, 0x62, 0x51, 0xbf, 0x8e, 0x85, 0x9f, 0x08, 0x5a, 0xdc, 0xb9 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 147 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xdc, 0x49, 0x70, 0xfe, 0x35, 0xf6, 0x77, 0x47, 0xcb, 0x05, 0x62, 0xc0, 0x6d, 0x87, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 148 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0xf9, 0xdf, 0x85, 0x89, 0x75, 0x85, 0x17, 0x97, 0xba, 0x60, 0x4f, 0xb0, 0xd1, 0x6c, 0xc7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 149 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xea, 0x0c, 0x98, 0xdc, 0xa1, 0x0b, 0x38, 0xc2, 0x1b, 0x3b, 0x14, 0xe8, 0xd1, 0xb7, 0x1f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 150 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0x09, 0x1b, 0x1b, 0x5b, 0x07, 0x49, 0xb2, 0xad, 0xc8, 0x03, 0xe6, 0x3d, 0xda, 0x9b, 0x72 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 151 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0xb3, 0x89, 0xe3, 0x32, 0x2c, 0x6d, 0xa0, 0x83, 0x84, 0x34, 0x5a, 0x41, 0x37, 0xfd, 0x08 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 152 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x38, 0x13, 0x08, 0xc4, 0x38, 0xf3, 0x5b, 0x39, 0x9f, 0x10, 0xad, 0x71, 0xb0, 0x50, 0x27, 0xd8 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 153 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0xc2, 0x30, 0xfc, 0xfa, 0x92, 0x79, 0xc3, 0x40, 0x9f, 0xc4, 0x23, 0xe2, 0xac, 0xbe, 0x04 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 154 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x84, 0xa4, 0x75, 0xac, 0xb0, 0x11, 0xf3, 0xf5, 0x9f, 0x4f, 0x46, 0xb7, 0x62, 0x74, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 155 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0x11, 0x9b, 0x68, 0xcb, 0x3f, 0x83, 0x99, 0xee, 0x60, 0x06, 0x6b, 0x56, 0x11, 0xa4, 0xd7 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 156 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0x23, 0x76, 0x2f, 0x52, 0x7a, 0x40, 0x60, 0xff, 0xca, 0x31, 0x2d, 0xcc, 0xa2, 0x2a, 0x16 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 157 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x61, 0xa2, 0x74, 0x5a, 0x33, 0xf0, 0x56, 0xa5, 0xac, 0x6a, 0xce, 0x2f, 0x08, 0xe3, 0x44 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 158 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0xf1, 0x45, 0x76, 0x6e, 0xca, 0x84, 0x9f, 0x5d, 0x01, 0x15, 0x36, 0xa6, 0x55, 0x7f, 0xdb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 159 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xaf, 0x27, 0xb2, 0xc8, 0x9c, 0x9b, 0x4c, 0xf4, 0xa0, 0xc4, 0x10, 0x6a, 0xc8, 0x03, 0x18 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 160 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x9c, 0x4f, 0x16, 0xc6, 0x21, 0xf4, 0xea, 0xb7, 0xe9, 0xac, 0x1d, 0x75, 0x51, 0xdd, 0x57 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 161 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x8e, 0x06, 0xfb, 0xa4, 0x66, 0xfa, 0x70, 0x85, 0x4d, 0x8c, 0x2e, 0x52, 0x4c, 0xff, 0xb2 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 162 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x4b, 0xc7, 0x8b, 0x22, 0x50, 0x70, 0x77, 0x3f, 0x04, 0xc4, 0x04, 0x66, 0xd4, 0xe9, 0x0c }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 163 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x2c, 0xbf, 0xf1, 0xed, 0x01, 0x50, 0xfe, 0xda, 0x8a, 0x47, 0x99, 0xbe, 0x94, 0x55, 0x1f }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 164 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0xb3, 0x0d, 0x7b, 0x3f, 0x27, 0x96, 0x27, 0x09, 0xa3, 0x6b, 0xca, 0xdf, 0xb9, 0x74, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 165 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xf6, 0xd3, 0x2e, 0x04, 0x4d, 0x77, 0xad, 0xcf, 0x37, 0xfb, 0x97, 0xac, 0x21, 0x33, 0x26 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 166 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xcb, 0x28, 0x4e, 0xcd, 0xcf, 0xd7, 0x81, 0xa8, 0xaf, 0xe3, 0x20, 0x77, 0x94, 0x9e, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 167 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x01, 0x7b, 0xb0, 0x2e, 0xc8, 0x7b, 0x2b, 0x94, 0xc9, 0x6e, 0x40, 0xa2, 0x6f, 0xc7, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 168 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc5, 0xc0, 0x38, 0xb6, 0x99, 0x06, 0x64, 0xab, 0x08, 0xa3, 0xaa, 0xa5, 0xdf, 0x9f, 0x32, 0x66 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 169 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0x70, 0x20, 0xbe, 0x37, 0xfa, 0xb6, 0x25, 0x9b, 0x2a, 0x27, 0xf4, 0xec, 0x55, 0x15, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 170 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0x13, 0x67, 0x03, 0x37, 0x4f, 0x64, 0xe8, 0x60, 0xb4, 0x8c, 0xe3, 0x1f, 0x93, 0x07, 0x16 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 171 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0x63, 0xa2, 0x69, 0xb1, 0x4d, 0x50, 0x6c, 0xcc, 0x40, 0x1a, 0xb8, 0xa9, 0xf1, 0xb5, 0x91 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 172 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x17, 0xf8, 0x1d, 0xc6, 0xaa, 0x45, 0x4a, 0xee, 0x4b, 0xd4, 0xa5, 0xa5, 0xcf, 0xf4, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 173 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xde, 0xce, 0xcd, 0x53, 0x54, 0xf0, 0x4d, 0x53, 0x0d, 0x76, 0xed, 0x88, 0x42, 0x46, 0xeb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 174 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xc5, 0x20, 0x5c, 0xc8, 0xfd, 0x8e, 0xda, 0x9a, 0x3c, 0xff, 0xd2, 0x51, 0x8f, 0x36, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 175 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x42, 0xfb, 0x47, 0x42, 0x93, 0xd9, 0x6e, 0xca, 0x9d, 0xb1, 0xb3, 0x7b, 0x1b, 0xa6, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 176 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa2, 0x31, 0x69, 0x26, 0x07, 0x16, 0x9b, 0x4e, 0xcd, 0xea, 0xd5, 0xcd, 0x3b, 0x10, 0xdb, 0x3e }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 177 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xac, 0xe4, 0xb9, 0x1c, 0x9c, 0x66, 0x9e, 0x77, 0xe7, 0xac, 0xac, 0xd1, 0x98, 0x59, 0xed, 0x49 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 178 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0xdb, 0x7c, 0xfd, 0x4a, 0x7b, 0x2b, 0x62, 0xab, 0x78, 0xa4, 0x8f, 0x3d, 0xda, 0xf4, 0xaf }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 179 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0xfa, 0xba, 0x2d, 0x46, 0xe2, 0x59, 0xcf, 0x48, 0x0d, 0x7c, 0x38, 0xe4, 0x57, 0x2a, 0x58 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 180 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x1c, 0x45, 0xbc, 0x6a, 0xe1, 0x6d, 0xee, 0x6e, 0xb7, 0xbe, 0xa1, 0x28, 0x70, 0x15, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 181 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0xd0, 0x30, 0x57, 0xcf, 0x13, 0x64, 0x42, 0x0c, 0x2b, 0x78, 0x06, 0x9a, 0x3e, 0x25, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 182 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xb5, 0x05, 0xe6, 0xcc, 0x13, 0x84, 0xcb, 0xae, 0xc1, 0xdf, 0x90, 0xb8, 0x0b, 0xeb, 0x20 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 183 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x74, 0xa3, 0xbe, 0xd2, 0x7b, 0xf4, 0xbd, 0x36, 0x22, 0xf9, 0xf5, 0xfe, 0x20, 0x83, 0x06 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 184 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x87, 0xf2, 0x6a, 0x89, 0xcf, 0xbf, 0xbb, 0x8e, 0x5e, 0xea, 0xc5, 0x40, 0x55, 0x31, 0x5e }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 185 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0x47, 0xdd, 0x32, 0xd3, 0xb2, 0x9a, 0xb6, 0xa4, 0xca, 0xeb, 0x60, 0x6c, 0x5b, 0x6f, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 186 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0x68, 0x61, 0xf8, 0xbc, 0x53, 0x86, 0xd3, 0x1f, 0xb7, 0x7f, 0x72, 0x0c, 0x32, 0x26, 0xe6 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 187 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0xcf, 0x1e, 0x6c, 0x42, 0x24, 0xef, 0xb3, 0x89, 0x00, 0xb1, 0x85, 0xab, 0x1d, 0xfd, 0x42 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 188 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x41, 0xaa, 0xb0, 0x5a, 0x42, 0xd3, 0x19, 0xde, 0x81, 0xd8, 0x74, 0xf5, 0xc7, 0xb9, 0x0d }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 189 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0xb9, 0xbc, 0x75, 0x9e, 0x2a, 0xd8, 0xd2, 0x14, 0x0a, 0x6c, 0x76, 0x2a, 0xe9, 0xe1, 0xab }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 190 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0x85, 0x96, 0xe1, 0x5e, 0x78, 0xe2, 0xc0, 0x64, 0x15, 0x9d, 0xef, 0xce, 0x5f, 0x30, 0x85 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 191 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0x8a, 0x49, 0x35, 0x14, 0x23, 0x1c, 0xbf, 0x56, 0xec, 0xce, 0xe4, 0xc4, 0x08, 0x89, 0xfb }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 0 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xde, 0x88, 0x5d, 0xc8, 0x7f, 0x5a, 0x92, 0x59, 0x40, 0x82, 0xd0, 0x2c, 0xc1, 0xe1, 0xb4, 0x2c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 1 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x2b, 0x07, 0x4e, 0x80, 0xf2, 0xa5, 0x97, 0xbf, 0x5f, 0xeb, 0xd8, 0xea, 0x5d, 0xa5, 0x5e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 2 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0xcc, 0xed, 0xf8, 0xde, 0x59, 0x2c, 0x22, 0xfb, 0x81, 0x34, 0x7b, 0x79, 0xf2, 0xdb, 0x1f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 3 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0x0b, 0x09, 0xf2, 0x67, 0xc4, 0x51, 0x45, 0xdb, 0x2f, 0x82, 0x6c, 0x25, 0x82, 0xd3, 0x5c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 4 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xd8, 0x07, 0xef, 0x76, 0x52, 0xd7, 0xeb, 0x0e, 0x13, 0xc8, 0xb5, 0xe1, 0x5b, 0x3b, 0xc0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 5 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x78, 0xbc, 0xf8, 0xdd, 0x8f, 0xd7, 0x22, 0x41, 0x22, 0x3a, 0xd2, 0x4b, 0x31, 0xb8, 0xa4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 6 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x10, 0xf6, 0x54, 0x34, 0x3e, 0x8f, 0x27, 0xe1, 0x2c, 0x83, 0xa4, 0x8d, 0x24, 0xff, 0x81 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 7 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x83, 0x3f, 0x71, 0x25, 0x8d, 0x53, 0x03, 0x6b, 0x02, 0x95, 0x2c, 0x76, 0xc7, 0x44, 0xf5, 0xa1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 8 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0xa8, 0x3f, 0xf2, 0x00, 0xcf, 0xf9, 0x31, 0x8a, 0x92, 0xf8, 0x69, 0x1a, 0x06, 0xb0, 0x9f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 9 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x62, 0x0c, 0xcb, 0xe9, 0xf3, 0x29, 0x2a, 0xbd, 0xf2, 0x17, 0x6b, 0x09, 0xf0, 0x4e, 0xba }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 10 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0xba, 0xbc, 0x4b, 0x3f, 0x51, 0x6c, 0x9a, 0xaf, 0xb3, 0x5f, 0x41, 0x40, 0xb5, 0x48, 0xf9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 11 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0x18, 0x78, 0x24, 0xd9, 0xc4, 0x58, 0x2b, 0x09, 0x16, 0x49, 0x3e, 0xcb, 0xde, 0x8c, 0x57 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 12 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x0a, 0xd5, 0x53, 0x17, 0x7f, 0xd5, 0xea, 0x10, 0x92, 0xc9, 0xd6, 0x26, 0xa2, 0x9d, 0xc4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 13 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xdc, 0x46, 0xc3, 0x72, 0x61, 0x19, 0x41, 0x24, 0xec, 0xae, 0xbd, 0x68, 0x04, 0x08, 0xec }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 14 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe4, 0xf2, 0xf2, 0xae, 0x23, 0xe9, 0xb1, 0x0b, 0xac, 0xfa, 0x58, 0x60, 0x15, 0x31, 0xba, 0x54 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 15 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0xd6, 0x7c, 0xf1, 0xa1, 0xe9, 0x1e, 0x8f, 0xf3, 0xa5, 0x7a, 0x17, 0x2c, 0x7b, 0xf4, 0x12 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 16 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x70, 0x6b, 0xe0, 0x69, 0x67, 0x88, 0x4e, 0x84, 0x7d, 0x13, 0x71, 0x28, 0xce, 0x47, 0xb3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 17 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0xf8, 0xb4, 0x09, 0xb0, 0x58, 0x59, 0x09, 0xaa, 0xd3, 0xa7, 0xb5, 0xa2, 0x19, 0x07, 0x2a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 18 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0x4b, 0x7b, 0xff, 0x02, 0x90, 0xc7, 0x83, 0x44, 0xc5, 0x4a, 0x23, 0xb7, 0x22, 0xcd, 0x20 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 19 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x09, 0x36, 0x57, 0x55, 0x2d, 0x44, 0x14, 0x22, 0x7c, 0xe1, 0x61, 0xe9, 0xeb, 0xf7, 0xdd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 20 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe1, 0xaf, 0x1e, 0x7d, 0x8b, 0xc2, 0x25, 0xed, 0x4d, 0xff, 0xb7, 0x71, 0xec, 0xbb, 0x9e, 0x67 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 21 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x65, 0x55, 0x25, 0x36, 0x35, 0xd8, 0x43, 0x21, 0x56, 0xcf, 0xd9, 0xc1, 0x1b, 0x14, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 22 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x40, 0x35, 0x07, 0x4a, 0x5d, 0x42, 0x60, 0xc9, 0x0c, 0xbd, 0x6d, 0xa6, 0xc3, 0xfc, 0xeb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 23 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0x6e, 0xe4, 0x16, 0xf9, 0xad, 0x1c, 0x10, 0x3e, 0xb0, 0xcc, 0x96, 0x75, 0x1c, 0x88, 0xe1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 24 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x8a, 0xe2, 0xa4, 0x63, 0x7a, 0xc0, 0xa7, 0x89, 0x0a, 0x8f, 0xd1, 0x48, 0x54, 0x45, 0xc9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 25 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x20, 0x12, 0xec, 0x8f, 0xad, 0xed, 0x08, 0x25, 0xfb, 0x2f, 0xa7, 0x0a, 0xb3, 0x0c, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 26 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x8a, 0x64, 0xb4, 0x6b, 0x5d, 0x88, 0xbf, 0x7f, 0x24, 0x7d, 0x4d, 0xba, 0xf3, 0x8f, 0x05 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 27 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x68, 0x25, 0x37, 0x62, 0xe2, 0xcc, 0x81, 0xb4, 0x2d, 0x1e, 0x50, 0x01, 0x76, 0x26, 0x99 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 28 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x41, 0xf8, 0x3b, 0x38, 0xce, 0x50, 0x32, 0xc6, 0xcd, 0x7a, 0xf9, 0x8c, 0xf6, 0x20, 0x61 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 29 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0xa8, 0x99, 0x90, 0xcd, 0x14, 0x11, 0x75, 0x0d, 0x5f, 0xb0, 0xdc, 0x98, 0x84, 0x47, 0xd4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 30 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xac, 0xcc, 0x8e, 0xd6, 0x29, 0xed, 0xf8, 0xc6, 0x8a, 0x53, 0x91, 0x83, 0xb1, 0xea, 0x82 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 31 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb1, 0x6f, 0xa7, 0x1f, 0x84, 0x6b, 0x81, 0xa1, 0x3f, 0x36, 0x1c, 0x43, 0xa8, 0x51, 0xf2, 0x90 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 32 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xad, 0x6e, 0xfd, 0xff, 0x59, 0x75, 0xae, 0xe7, 0x69, 0x22, 0x34, 0xbc, 0xd5, 0x44, 0x88 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 33 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0xfd, 0xb0, 0x5a, 0x78, 0x3d, 0x03, 0x08, 0x2d, 0xfe, 0x5f, 0xdd, 0x80, 0xa0, 0x0b, 0x17 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 34 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0x81, 0xb5, 0x84, 0x76, 0x69, 0x97, 0xaf, 0x6b, 0xa5, 0x52, 0x9d, 0x3b, 0xdd, 0x86, 0x09 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 35 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0c, 0xf4, 0xff, 0x4f, 0x49, 0xc8, 0xa0, 0xca, 0x06, 0x0c, 0x44, 0x34, 0x99, 0xe2, 0x93, 0x13 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 36 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x4b, 0xa8, 0xa8, 0xe0, 0x29, 0xf8, 0xb2, 0x6d, 0x8a, 0xff, 0xf9, 0xdf, 0x13, 0x3b, 0xb6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 37 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0xfe, 0xbf, 0x64, 0x36, 0x0f, 0x38, 0xe4, 0xe6, 0x35, 0x58, 0xf0, 0xff, 0xc5, 0x50, 0xc3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 38 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xad, 0x98, 0xcb, 0xf7, 0x25, 0x13, 0x7d, 0x6a, 0x81, 0x08, 0xc2, 0xbe, 0xd9, 0x93, 0x22 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 39 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0xfa, 0xa9, 0x96, 0x22, 0x61, 0x98, 0xb3, 0xe2, 0x61, 0x04, 0x13, 0xce, 0x1b, 0x3f, 0x78 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 40 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2a, 0x8c, 0xe6, 0x74, 0x7a, 0x7e, 0x39, 0x36, 0x78, 0x28, 0xe2, 0x90, 0x84, 0x85, 0x02, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 41 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x37, 0x36, 0xe8, 0xb8, 0xf8, 0x9c, 0xa1, 0xe3, 0x7b, 0x6d, 0xea, 0xb4, 0x0f, 0xac, 0xf1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 42 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0xf7, 0x97, 0xe5, 0x04, 0x18, 0xb9, 0x5f, 0xa6, 0x01, 0x33, 0x33, 0x91, 0x7a, 0x94, 0x80 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 43 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x58, 0xde, 0x37, 0xc2, 0xec, 0xe2, 0xa0, 0x2c, 0x73, 0xc0, 0x1f, 0xed, 0xc9, 0xa1, 0x32 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 44 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x9b, 0x87, 0xae, 0x77, 0xba, 0xe7, 0x06, 0x80, 0x39, 0x66, 0xc6, 0x6c, 0x73, 0xad, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 45 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x65, 0xab, 0x8d, 0xf8, 0xff, 0xd7, 0x82, 0xe3, 0x58, 0x12, 0x1a, 0x4a, 0x4f, 0xc5, 0x41 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 46 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xdc, 0xd9, 0xe6, 0xf7, 0x5e, 0x6c, 0x36, 0xc8, 0xda, 0xee, 0x04, 0x66, 0xf0, 0xed, 0x74 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 47 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x9a, 0x63, 0x7b, 0xeb, 0x1c, 0x03, 0x04, 0xf1, 0x40, 0x14, 0xc0, 0x37, 0xe7, 0x36, 0xdd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 48 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0x5f, 0x0a, 0x25, 0xe8, 0x4a, 0xc9, 0x30, 0xd9, 0x96, 0x28, 0x1a, 0x5f, 0x95, 0x4d, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 49 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x42, 0xe4, 0x07, 0x4b, 0x29, 0x27, 0x97, 0x3e, 0x8d, 0x17, 0xff, 0xa9, 0x2f, 0x7f, 0xe6, 0x15 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 50 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xe2, 0xa9, 0xd2, 0xc1, 0x82, 0x44, 0x49, 0xc6, 0x9e, 0x3e, 0x03, 0x98, 0xf1, 0x29, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 51 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0xf2, 0x9c, 0x1e, 0x1f, 0x62, 0x84, 0x7a, 0x15, 0x25, 0x3b, 0x28, 0xa1, 0xe9, 0xd7, 0x12 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 52 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xed, 0x5d, 0x29, 0xb9, 0x03, 0xf3, 0x1e, 0x89, 0x83, 0xef, 0x8b, 0x0a, 0x2b, 0xf9, 0x90 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 53 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xb8, 0x07, 0x02, 0x70, 0x81, 0x0f, 0x9d, 0x02, 0x3f, 0x9d, 0xd7, 0xff, 0x3b, 0x4a, 0xa2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 54 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0xd4, 0x6e, 0x15, 0x5c, 0x12, 0x28, 0xf6, 0x1d, 0x1a, 0x0d, 0xb4, 0x81, 0x5e, 0xcc, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 55 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x61, 0x08, 0xd1, 0xd9, 0x80, 0x71, 0x42, 0x8e, 0xec, 0xee, 0xf1, 0x71, 0x4b, 0x96, 0xdd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 56 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0x5b, 0x25, 0xb7, 0x1b, 0x62, 0x96, 0xcf, 0x73, 0xdd, 0x2c, 0xdc, 0xac, 0x2f, 0x70, 0xb1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 57 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0xab, 0xa9, 0x5e, 0x8a, 0x06, 0xa2, 0xd9, 0xd3, 0x53, 0x0d, 0x26, 0x77, 0x87, 0x8c, 0x80 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 58 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0x70, 0xd2, 0x0e, 0x89, 0xb4, 0x67, 0xe8, 0xf5, 0x17, 0x60, 0x61, 0xb8, 0x1d, 0xd3, 0x96 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 59 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x8f, 0x44, 0x67, 0xa5, 0xd8, 0xf1, 0xe7, 0x30, 0x7d, 0xc3, 0x0b, 0x34, 0xe4, 0x04, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 60 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbc, 0xea, 0x28, 0xe9, 0x07, 0x1b, 0x5a, 0x23, 0x02, 0x97, 0x0f, 0xf3, 0x52, 0x45, 0x1b, 0xc5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 61 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0x23, 0xc0, 0x0b, 0xc1, 0x77, 0xd3, 0x31, 0xad, 0x31, 0x2e, 0x09, 0xc9, 0x01, 0x5c, 0x1c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 62 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0xac, 0x61, 0xe3, 0x18, 0x37, 0x47, 0xb3, 0xf5, 0x83, 0x6d, 0xa2, 0x1a, 0x1b, 0xc4, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 63 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x7b, 0x07, 0x57, 0x91, 0x87, 0x88, 0x80, 0xb4, 0x41, 0x89, 0xd3, 0x52, 0x2b, 0x8c, 0x30 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 64 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x71, 0x32, 0xd0, 0xc0, 0xe4, 0xa0, 0x75, 0x93, 0xcf, 0x12, 0xeb, 0xb1, 0x2b, 0xe7, 0x68, 0x8c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 65 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0xfb, 0xac, 0x16, 0x44, 0xde, 0xb0, 0xc7, 0x84, 0x27, 0x5f, 0xe5, 0x6e, 0x19, 0xea, 0xd3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 66 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x05, 0x06, 0x3f, 0x30, 0xf4, 0x22, 0x8b, 0x37, 0x4e, 0x24, 0x59, 0x73, 0x8f, 0x26, 0xbb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 67 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x97, 0x5b, 0x5f, 0x48, 0xbb, 0x68, 0xfc, 0xbb, 0xc7, 0xce, 0xa9, 0x3b, 0x45, 0x2e, 0xd7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 68 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x3f, 0x25, 0x76, 0xe2, 0xaf, 0xed, 0xc7, 0x4b, 0xb1, 0xca, 0x7e, 0xee, 0xc1, 0xc0, 0xe7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 69 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0xc4, 0x03, 0xf5, 0xf9, 0x66, 0xe0, 0xe3, 0xd9, 0xf2, 0x96, 0xd6, 0x22, 0x6d, 0xca, 0x28 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 70 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xc2, 0x09, 0x08, 0x24, 0x9a, 0xb4, 0xa3, 0x4d, 0x6d, 0xd0, 0xa3, 0x13, 0x27, 0xff, 0x1a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 71 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0x54, 0x13, 0x29, 0xec, 0xb6, 0x15, 0x9a, 0xb2, 0x3b, 0x7f, 0xc5, 0xe6, 0xa2, 0x1b, 0xca }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 72 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0xa1, 0xac, 0xf1, 0xa2, 0xed, 0x9b, 0xa7, 0x2b, 0xc6, 0xde, 0xb3, 0x1d, 0x88, 0xb8, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 73 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x8b, 0xd8, 0xed, 0xda, 0xbb, 0x6f, 0x3b, 0xf0, 0xd5, 0xa8, 0xa2, 0x7b, 0xe1, 0xfe, 0x8a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 74 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x3c, 0x7d, 0x76, 0x85, 0xe1, 0x4e, 0xc6, 0x6b, 0xbb, 0x96, 0xb8, 0xf0, 0x5b, 0x6d, 0xdd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 75 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0x75, 0x2e, 0xef, 0xc8, 0xc2, 0xa9, 0x3f, 0x91, 0xb6, 0xe7, 0x3e, 0xb0, 0x7c, 0xca, 0x6e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 76 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x93, 0xe7, 0xd6, 0x2f, 0x6c, 0xe5, 0x02, 0xc6, 0x4f, 0x75, 0xe2, 0x81, 0xf9, 0xc0, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 77 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0xfd, 0x99, 0x9b, 0xe5, 0xd0, 0xcf, 0xa3, 0x57, 0x32, 0xc0, 0xdd, 0xc8, 0x8f, 0xf5, 0xa5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 78 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x64, 0x7c, 0x76, 0xa3, 0x00, 0xc3, 0x17, 0x3b, 0x84, 0x14, 0x87, 0xeb, 0x2b, 0xae, 0x9f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 79 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x17, 0x2d, 0xf8, 0xb0, 0x2f, 0x04, 0xb5, 0x3a, 0xda, 0xb0, 0x28, 0xb4, 0xe0, 0x1a, 0xcd, 0x87 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 80 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0x4b, 0x3b, 0xf4, 0x99, 0x8a, 0xeb, 0x05, 0xaf, 0xd8, 0x7e, 0xc5, 0x36, 0x53, 0x3a, 0x36 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 81 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x83, 0xf7, 0xbf, 0x44, 0xc9, 0x7f, 0x06, 0x52, 0x58, 0xa6, 0x66, 0xca, 0xe0, 0x30, 0x20 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 82 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xd4, 0xc8, 0xa6, 0x3f, 0x80, 0x95, 0x41, 0x04, 0xde, 0x7b, 0x92, 0xce, 0xde, 0x1b, 0xe1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 83 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0xfe, 0x61, 0x81, 0x0f, 0xd5, 0x46, 0x7c, 0xcd, 0xac, 0xb7, 0x58, 0x00, 0xf3, 0xac, 0x07 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 84 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x83, 0x0d, 0x8a, 0x25, 0x90, 0xf7, 0xd8, 0xe1, 0xb5, 0x5a, 0x73, 0x7f, 0x4a, 0xf4, 0x5f, 0x34 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 85 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfc, 0xd4, 0x68, 0x3f, 0x85, 0x80, 0x58, 0xe7, 0x43, 0x14, 0x67, 0x1d, 0x43, 0xfa, 0x2c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 86 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x52, 0x3d, 0x0b, 0xab, 0xbb, 0x82, 0xf4, 0x6e, 0xbc, 0x9e, 0x70, 0xb1, 0xcd, 0x41, 0xdd, 0xd0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 87 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x4a, 0xab, 0x37, 0x08, 0x0d, 0x74, 0x86, 0xf7, 0xd5, 0x42, 0xa3, 0x09, 0xe5, 0x3e, 0xed }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 88 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0xc5, 0x60, 0x9d, 0x09, 0x06, 0xb2, 0x3a, 0xb9, 0xca, 0xca, 0x81, 0x6f, 0x5d, 0xbe, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 89 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x26, 0x02, 0x6e, 0xed, 0xd9, 0x1a, 0xdc, 0x6d, 0x83, 0x1c, 0xdf, 0x98, 0x94, 0xbd, 0xc6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 90 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x88, 0x33, 0x0b, 0xaa, 0x4f, 0x2b, 0x61, 0x8f, 0xc9, 0xd9, 0xb0, 0x21, 0xbf, 0x50, 0x3d, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 91 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x9e, 0x0e, 0xa2, 0x24, 0x80, 0xb0, 0xba, 0xc9, 0x35, 0xc8, 0xa8, 0xeb, 0xef, 0xcd, 0xcf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 92 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0xca, 0x77, 0x9f, 0x39, 0x8f, 0xb0, 0x4f, 0x86, 0x7d, 0xa7, 0xe8, 0xa4, 0x47, 0x56, 0xcb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 93 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0xf8, 0x9c, 0x42, 0x98, 0x57, 0x86, 0xbf, 0xc4, 0x3c, 0x6d, 0xf8, 0xad, 0xa3, 0x68, 0x32 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 94 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0xc1, 0xde, 0x5f, 0xb8, 0xf2, 0x1d, 0x87, 0x4e, 0x91, 0xc5, 0x3b, 0x56, 0x0c, 0x50, 0xe3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 95 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0xaa, 0x90, 0x58, 0x49, 0x0e, 0xda, 0x30, 0x60, 0x01, 0xa8, 0xa9, 0xf4, 0x8d, 0x0c, 0xa7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 96 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x4e, 0xc7, 0x1d, 0x61, 0x28, 0xd4, 0x87, 0x18, 0x65, 0xd6, 0x17, 0xc3, 0x0b, 0x37, 0xe3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 97 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0xbe, 0x1c, 0x53, 0x5b, 0x17, 0xca, 0xbd, 0x0c, 0x4d, 0x93, 0x52, 0x9d, 0x69, 0xbf, 0x47 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 98 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xef, 0x67, 0x75, 0x65, 0x07, 0xbe, 0xec, 0x9d, 0xd3, 0x86, 0x28, 0x83, 0x47, 0x80, 0x44 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 99 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xe2, 0x31, 0xfa, 0x5a, 0x59, 0x48, 0xce, 0x21, 0x34, 0xe9, 0x2f, 0xc0, 0x66, 0x4d, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 100 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x19, 0x4b, 0x8e, 0x5d, 0xda, 0x55, 0x30, 0xd0, 0xc6, 0x78, 0xc0, 0xb4, 0x8f, 0x5d, 0x92 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 101 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xbd, 0x08, 0x6f, 0x23, 0x7c, 0xc4, 0xfd, 0x99, 0xf4, 0xd7, 0x6b, 0xde, 0x6b, 0x48, 0x26 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 102 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x25, 0x97, 0x61, 0xca, 0x17, 0x13, 0x0d, 0x6e, 0xd8, 0x6d, 0x57, 0xcd, 0x79, 0x51, 0xee }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 103 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd7, 0xcb, 0xb3, 0xf3, 0x4b, 0x9b, 0x45, 0x0f, 0x24, 0xb0, 0xe8, 0x51, 0x8e, 0x54, 0xda, 0x6d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 104 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0x5b, 0x9c, 0xae, 0xbe, 0x9f, 0x7f, 0x41, 0x7f, 0x40, 0x68, 0xd0, 0xd2, 0xee, 0x20, 0xb3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 105 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9d, 0x92, 0x4b, 0x93, 0x4a, 0x90, 0xce, 0x1f, 0xd3, 0x9b, 0x8a, 0x97, 0x94, 0xf8, 0x26, 0x72 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 106 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc5, 0x05, 0x62, 0xbf, 0x09, 0x45, 0x26, 0xa9, 0x1c, 0x5b, 0xc6, 0x3c, 0x0c, 0x22, 0x49, 0x95 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 107 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xf1, 0x18, 0x05, 0x04, 0x67, 0x43, 0xbd, 0x74, 0xf5, 0x71, 0x88, 0xd9, 0x18, 0x8d, 0xf7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 108 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0xd2, 0x74, 0xbd, 0x0f, 0x1b, 0x58, 0xae, 0x34, 0x5d, 0x9e, 0x72, 0x33, 0xf9, 0xb8, 0xf3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 109 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9d, 0x6b, 0xdc, 0x8f, 0x4c, 0xe5, 0xfe, 0xb0, 0xf3, 0xbe, 0xd2, 0xe4, 0xb9, 0xa9, 0xbb, 0x0b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 110 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0x55, 0x48, 0xbc, 0xf3, 0xf4, 0x25, 0x65, 0xf7, 0xef, 0xa9, 0x45, 0x62, 0x52, 0x8d, 0x46 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 111 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xcc, 0xae, 0xbd, 0x3a, 0x4c, 0x3e, 0x80, 0xb0, 0x63, 0x74, 0x81, 0x31, 0xba, 0x4a, 0x71 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 112 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x3c, 0xb2, 0x3d, 0x9e, 0x11, 0xc9, 0xd9, 0x3f, 0x11, 0x7e, 0x9c, 0x0a, 0x91, 0xb5, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 113 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x78, 0xf9, 0x33, 0xa2, 0x08, 0x1a, 0xc1, 0xdb, 0x84, 0xf6, 0x9d, 0x10, 0xf4, 0x52, 0x3f, 0xe0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 114 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x61, 0xf7, 0x41, 0x2e, 0xd3, 0x20, 0xde, 0x0e, 0xdc, 0x88, 0x51, 0xc2, 0xe2, 0x43, 0x6f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 115 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x64, 0xba, 0x1c, 0xd0, 0x4c, 0xe6, 0xba, 0xb9, 0x84, 0x74, 0x33, 0x08, 0x14, 0xb4, 0xd4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 116 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x48, 0x39, 0x1b, 0xff, 0xb9, 0xcf, 0xff, 0x80, 0xac, 0x23, 0x8c, 0x88, 0x6e, 0xf0, 0xa4, 0x61 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 117 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0xd2, 0xa6, 0x7d, 0xf5, 0xa9, 0x99, 0xfd, 0xbf, 0x93, 0xed, 0xd0, 0x34, 0x32, 0x96, 0xc9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 118 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xca, 0x73, 0x67, 0x39, 0x6b, 0x69, 0xa2, 0x21, 0xbd, 0x63, 0x2b, 0xea, 0x38, 0x6e, 0xec }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 119 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x0f, 0xd5, 0x02, 0x0d, 0xfe, 0x65, 0xf5, 0xf1, 0x62, 0x93, 0xec, 0x92, 0xc6, 0xfd, 0x89 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 120 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0x62, 0x99, 0x5b, 0x82, 0x17, 0xa6, 0x7f, 0x1a, 0xbc, 0x34, 0x2e, 0x14, 0x64, 0x06, 0xf8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 121 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0xa6, 0x16, 0x4b, 0x7a, 0x60, 0xba, 0xe4, 0xe9, 0x86, 0xff, 0xac, 0x28, 0xdf, 0xad, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 122 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x64, 0xe0, 0xd7, 0xf9, 0x00, 0xe3, 0xd9, 0xc8, 0x3e, 0x4b, 0x8f, 0x96, 0x71, 0x7b, 0x21, 0x46 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 123 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1a, 0xd2, 0x56, 0x1d, 0xe8, 0xc1, 0x23, 0x2f, 0x5d, 0x8d, 0xba, 0xb4, 0x73, 0x9b, 0x6c, 0xbb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 124 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x96, 0x89, 0xe9, 0xa5, 0x57, 0xf5, 0x8b, 0x1c, 0x3b, 0xf4, 0x0c, 0x97, 0xa9, 0x09, 0x64 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 125 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc4, 0x63, 0x7e, 0x4a, 0x5e, 0x63, 0x77, 0xf9, 0xcc, 0x5a, 0x86, 0x38, 0x04, 0x5d, 0xe0, 0x29 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 126 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x2e, 0x60, 0x7e, 0x5a, 0xea, 0x46, 0x88, 0x59, 0x4b, 0x45, 0xf3, 0xae, 0xe3, 0xdf, 0x90 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 127 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0xc4, 0xe4, 0x38, 0x1f, 0xee, 0xc7, 0x40, 0x54, 0x95, 0x4c, 0x05, 0xb7, 0x77, 0xa0, 0x0a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 128 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x91, 0x54, 0x95, 0x14, 0x60, 0x5f, 0x38, 0x24, 0x6c, 0x9b, 0x72, 0x4a, 0xd8, 0x39, 0xf0, 0x1d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 129 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0xb2, 0x4e, 0x3b, 0x6f, 0xef, 0xe4, 0x0a, 0x4f, 0x9e, 0xf7, 0xac, 0x6e, 0x44, 0xd7, 0x6a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 130 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x37, 0xa6, 0x83, 0xdc, 0x5d, 0x4b, 0x52, 0xab, 0xb4, 0xa1, 0x23, 0xa8, 0xdf, 0x86, 0xc6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 131 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbb, 0x28, 0x52, 0xc8, 0x91, 0xc5, 0x94, 0x7d, 0x2e, 0xd4, 0x40, 0x32, 0xc4, 0x21, 0xb8, 0x5f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 132 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x9f, 0x5f, 0xbd, 0x5e, 0x8a, 0x42, 0x64, 0xc0, 0xa8, 0x5b, 0x80, 0x40, 0x9a, 0xfa, 0x5e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 133 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xda, 0xb8, 0x09, 0xf8, 0x5a, 0x91, 0x7f, 0xe9, 0x24, 0x73, 0x3f, 0x42, 0x4a, 0xc5, 0x89 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 134 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0xef, 0x5c, 0x1f, 0x8d, 0x60, 0x51, 0x92, 0x64, 0x66, 0x95, 0xce, 0xad, 0xc6, 0x5f, 0x32 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 135 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0xaa, 0x90, 0x04, 0x0b, 0x4c, 0x15, 0xa1, 0x23, 0x16, 0xb7, 0x8e, 0x0f, 0x95, 0x86, 0xfc }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 136 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xfa, 0xc8, 0x29, 0x7c, 0xea, 0xab, 0xc8, 0x7d, 0x45, 0x43, 0x50, 0x60, 0x1e, 0x06, 0x73 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 137 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0x47, 0xef, 0x56, 0x7a, 0xc2, 0x8d, 0xfe, 0x48, 0x84, 0x92, 0xf1, 0x57, 0xe2, 0xb2, 0xe0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 138 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x84, 0x26, 0x02, 0x7d, 0xdb, 0x96, 0x2b, 0x5c, 0x5b, 0xa7, 0xeb, 0x8b, 0xc9, 0xab, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 139 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0x17, 0xfc, 0x77, 0xe7, 0x19, 0x92, 0xa1, 0x2d, 0xbe, 0x4c, 0x18, 0x06, 0x8b, 0xec, 0x82 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 140 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xee, 0xbb, 0xc9, 0x88, 0x40, 0xf8, 0xae, 0x6d, 0xaf, 0x76, 0x57, 0x3b, 0x7e, 0x56, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 141 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4e, 0x11, 0xa9, 0xf7, 0x42, 0x05, 0x12, 0x5b, 0x61, 0xe0, 0xae, 0xe0, 0x47, 0xec, 0xa2, 0x0d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 142 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf6, 0x04, 0x67, 0xf5, 0x5a, 0x1f, 0x17, 0xea, 0xb8, 0x8e, 0x80, 0x01, 0x20, 0xcb, 0xc2, 0x84 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 143 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x36, 0x64, 0x9f, 0x60, 0x0b, 0x44, 0x9e, 0xe2, 0x76, 0x53, 0x0f, 0x0c, 0xd8, 0x3c, 0x11 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 144 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0xc0, 0xe3, 0x65, 0x6a, 0x9e, 0x3a, 0xc7, 0xcd, 0x37, 0x8a, 0x73, 0x7f, 0x53, 0xb6, 0x37 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 145 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xac, 0xae, 0x63, 0xd3, 0x3b, 0x92, 0x8a, 0xa8, 0x38, 0x0f, 0x8d, 0x54, 0xd8, 0x8c, 0x17 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 146 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x89, 0x35, 0xff, 0xbc, 0x75, 0xae, 0x62, 0x51, 0xbf, 0x8e, 0x85, 0x9f, 0x08, 0x5a, 0xdc, 0xb9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 147 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xdc, 0x49, 0x70, 0xfe, 0x35, 0xf6, 0x77, 0x47, 0xcb, 0x05, 0x62, 0xc0, 0x6d, 0x87, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 148 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0xf9, 0xdf, 0x85, 0x89, 0x75, 0x85, 0x17, 0x97, 0xba, 0x60, 0x4f, 0xb0, 0xd1, 0x6c, 0xc7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 149 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xea, 0x0c, 0x98, 0xdc, 0xa1, 0x0b, 0x38, 0xc2, 0x1b, 0x3b, 0x14, 0xe8, 0xd1, 0xb7, 0x1f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 150 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0x09, 0x1b, 0x1b, 0x5b, 0x07, 0x49, 0xb2, 0xad, 0xc8, 0x03, 0xe6, 0x3d, 0xda, 0x9b, 0x72 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 151 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0xb3, 0x89, 0xe3, 0x32, 0x2c, 0x6d, 0xa0, 0x83, 0x84, 0x34, 0x5a, 0x41, 0x37, 0xfd, 0x08 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 152 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x38, 0x13, 0x08, 0xc4, 0x38, 0xf3, 0x5b, 0x39, 0x9f, 0x10, 0xad, 0x71, 0xb0, 0x50, 0x27, 0xd8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 153 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0xc2, 0x30, 0xfc, 0xfa, 0x92, 0x79, 0xc3, 0x40, 0x9f, 0xc4, 0x23, 0xe2, 0xac, 0xbe, 0x04 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 154 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x84, 0xa4, 0x75, 0xac, 0xb0, 0x11, 0xf3, 0xf5, 0x9f, 0x4f, 0x46, 0xb7, 0x62, 0x74, 0xc0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 155 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0x11, 0x9b, 0x68, 0xcb, 0x3f, 0x83, 0x99, 0xee, 0x60, 0x06, 0x6b, 0x56, 0x11, 0xa4, 0xd7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 156 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0x23, 0x76, 0x2f, 0x52, 0x7a, 0x40, 0x60, 0xff, 0xca, 0x31, 0x2d, 0xcc, 0xa2, 0x2a, 0x16 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 157 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x61, 0xa2, 0x74, 0x5a, 0x33, 0xf0, 0x56, 0xa5, 0xac, 0x6a, 0xce, 0x2f, 0x08, 0xe3, 0x44 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 158 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0xf1, 0x45, 0x76, 0x6e, 0xca, 0x84, 0x9f, 0x5d, 0x01, 0x15, 0x36, 0xa6, 0x55, 0x7f, 0xdb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 159 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xaf, 0x27, 0xb2, 0xc8, 0x9c, 0x9b, 0x4c, 0xf4, 0xa0, 0xc4, 0x10, 0x6a, 0xc8, 0x03, 0x18 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 160 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x9c, 0x4f, 0x16, 0xc6, 0x21, 0xf4, 0xea, 0xb7, 0xe9, 0xac, 0x1d, 0x75, 0x51, 0xdd, 0x57 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 161 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x8e, 0x06, 0xfb, 0xa4, 0x66, 0xfa, 0x70, 0x85, 0x4d, 0x8c, 0x2e, 0x52, 0x4c, 0xff, 0xb2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 162 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfb, 0x4b, 0xc7, 0x8b, 0x22, 0x50, 0x70, 0x77, 0x3f, 0x04, 0xc4, 0x04, 0x66, 0xd4, 0xe9, 0x0c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 163 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x2c, 0xbf, 0xf1, 0xed, 0x01, 0x50, 0xfe, 0xda, 0x8a, 0x47, 0x99, 0xbe, 0x94, 0x55, 0x1f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 164 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0xb3, 0x0d, 0x7b, 0x3f, 0x27, 0x96, 0x27, 0x09, 0xa3, 0x6b, 0xca, 0xdf, 0xb9, 0x74, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 165 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xf6, 0xd3, 0x2e, 0x04, 0x4d, 0x77, 0xad, 0xcf, 0x37, 0xfb, 0x97, 0xac, 0x21, 0x33, 0x26 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 166 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xcb, 0x28, 0x4e, 0xcd, 0xcf, 0xd7, 0x81, 0xa8, 0xaf, 0xe3, 0x20, 0x77, 0x94, 0x9e, 0x88 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 167 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x01, 0x7b, 0xb0, 0x2e, 0xc8, 0x7b, 0x2b, 0x94, 0xc9, 0x6e, 0x40, 0xa2, 0x6f, 0xc7, 0x1a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 168 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc5, 0xc0, 0x38, 0xb6, 0x99, 0x06, 0x64, 0xab, 0x08, 0xa3, 0xaa, 0xa5, 0xdf, 0x9f, 0x32, 0x66 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 169 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0x70, 0x20, 0xbe, 0x37, 0xfa, 0xb6, 0x25, 0x9b, 0x2a, 0x27, 0xf4, 0xec, 0x55, 0x15, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 170 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0x13, 0x67, 0x03, 0x37, 0x4f, 0x64, 0xe8, 0x60, 0xb4, 0x8c, 0xe3, 0x1f, 0x93, 0x07, 0x16 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 171 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0x63, 0xa2, 0x69, 0xb1, 0x4d, 0x50, 0x6c, 0xcc, 0x40, 0x1a, 0xb8, 0xa9, 0xf1, 0xb5, 0x91 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 172 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x17, 0xf8, 0x1d, 0xc6, 0xaa, 0x45, 0x4a, 0xee, 0x4b, 0xd4, 0xa5, 0xa5, 0xcf, 0xf4, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 173 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xde, 0xce, 0xcd, 0x53, 0x54, 0xf0, 0x4d, 0x53, 0x0d, 0x76, 0xed, 0x88, 0x42, 0x46, 0xeb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 174 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xc5, 0x20, 0x5c, 0xc8, 0xfd, 0x8e, 0xda, 0x9a, 0x3c, 0xff, 0xd2, 0x51, 0x8f, 0x36, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 175 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x42, 0xfb, 0x47, 0x42, 0x93, 0xd9, 0x6e, 0xca, 0x9d, 0xb1, 0xb3, 0x7b, 0x1b, 0xa6, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 176 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa2, 0x31, 0x69, 0x26, 0x07, 0x16, 0x9b, 0x4e, 0xcd, 0xea, 0xd5, 0xcd, 0x3b, 0x10, 0xdb, 0x3e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 177 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xac, 0xe4, 0xb9, 0x1c, 0x9c, 0x66, 0x9e, 0x77, 0xe7, 0xac, 0xac, 0xd1, 0x98, 0x59, 0xed, 0x49 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 178 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x75, 0xdb, 0x7c, 0xfd, 0x4a, 0x7b, 0x2b, 0x62, 0xab, 0x78, 0xa4, 0x8f, 0x3d, 0xda, 0xf4, 0xaf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 179 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0xfa, 0xba, 0x2d, 0x46, 0xe2, 0x59, 0xcf, 0x48, 0x0d, 0x7c, 0x38, 0xe4, 0x57, 0x2a, 0x58 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 180 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x1c, 0x45, 0xbc, 0x6a, 0xe1, 0x6d, 0xee, 0x6e, 0xb7, 0xbe, 0xa1, 0x28, 0x70, 0x15, 0x82 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 181 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0xd0, 0x30, 0x57, 0xcf, 0x13, 0x64, 0x42, 0x0c, 0x2b, 0x78, 0x06, 0x9a, 0x3e, 0x25, 0x02 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 182 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xb5, 0x05, 0xe6, 0xcc, 0x13, 0x84, 0xcb, 0xae, 0xc1, 0xdf, 0x90, 0xb8, 0x0b, 0xeb, 0x20 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 183 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x74, 0xa3, 0xbe, 0xd2, 0x7b, 0xf4, 0xbd, 0x36, 0x22, 0xf9, 0xf5, 0xfe, 0x20, 0x83, 0x06 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 184 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x87, 0xf2, 0x6a, 0x89, 0xcf, 0xbf, 0xbb, 0x8e, 0x5e, 0xea, 0xc5, 0x40, 0x55, 0x31, 0x5e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 185 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x05, 0x47, 0xdd, 0x32, 0xd3, 0xb2, 0x9a, 0xb6, 0xa4, 0xca, 0xeb, 0x60, 0x6c, 0x5b, 0x6f, 0x78 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 186 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x18, 0x68, 0x61, 0xf8, 0xbc, 0x53, 0x86, 0xd3, 0x1f, 0xb7, 0x7f, 0x72, 0x0c, 0x32, 0x26, 0xe6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 187 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0xcf, 0x1e, 0x6c, 0x42, 0x24, 0xef, 0xb3, 0x89, 0x00, 0xb1, 0x85, 0xab, 0x1d, 0xfd, 0x42 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 188 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x41, 0xaa, 0xb0, 0x5a, 0x42, 0xd3, 0x19, 0xde, 0x81, 0xd8, 0x74, 0xf5, 0xc7, 0xb9, 0x0d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 189 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0xb9, 0xbc, 0x75, 0x9e, 0x2a, 0xd8, 0xd2, 0x14, 0x0a, 0x6c, 0x76, 0x2a, 0xe9, 0xe1, 0xab }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 190 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0x85, 0x96, 0xe1, 0x5e, 0x78, 0xe2, 0xc0, 0x64, 0x15, 0x9d, 0xef, 0xce, 0x5f, 0x30, 0x85 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey192.rsp test vector 191 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0x8a, 0x49, 0x35, 0x14, 0x23, 0x1c, 0xbf, 0x56, 0xec, 0xce, 0xe4, 0xc4, 0x08, 0x89, 0xfb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },

	{ "NIST KAT-AES ECBVarKey256.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x5a, 0x6d, 0xcb, 0x19, 0xb2, 0x01, 0xa0, 0x1e, 0xbc, 0xfa, 0x8a, 0xa2, 0x2b, 0x57, 0x59 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0x91, 0x69, 0xcd, 0xcf, 0x2d, 0x83, 0xe8, 0x38, 0x12, 0x5a, 0x12, 0xee, 0x6a, 0xa4, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0xf3, 0xa7, 0x2f, 0xc3, 0xcd, 0xf7, 0x4d, 0xfa, 0xf6, 0xc3, 0xe6, 0xb9, 0x7b, 0x2f, 0xa6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x77, 0x76, 0x79, 0xd5, 0x00, 0x37, 0xc7, 0x94, 0x91, 0xa9, 0x4d, 0xa7, 0x6a, 0x9a, 0x35 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xf4, 0x89, 0x3e, 0xca, 0xfa, 0x0a, 0x02, 0x47, 0xa8, 0x98, 0xe0, 0x40, 0x69, 0x15, 0x59 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0xbb, 0x41, 0x37, 0x03, 0x73, 0x53, 0x26, 0x31, 0x0a, 0x26, 0x9b, 0xd3, 0xaa, 0x94, 0xb2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0xe3, 0x22, 0x46, 0xbe, 0xd2, 0xb0, 0xe8, 0x59, 0xe5, 0x5c, 0x1c, 0xc6, 0xb2, 0x65, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xec, 0x52, 0xa2, 0x12, 0xf8, 0x0a, 0x09, 0xdf, 0x63, 0x17, 0x02, 0x1b, 0xc2, 0xa9, 0x81, 0x9e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0x3e, 0x5b, 0x60, 0x0e, 0xb7, 0x0d, 0xbc, 0xcf, 0x6c, 0x0b, 0x1d, 0x9a, 0x68, 0x18, 0x2c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa3, 0xf5, 0x99, 0xd6, 0x3a, 0x82, 0xa9, 0x68, 0xc3, 0x3f, 0xe2, 0x65, 0x90, 0x74, 0x59, 0x70 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0xcc, 0xb9, 0xb1, 0x33, 0x70, 0x02, 0xcb, 0xac, 0x42, 0xc5, 0x20, 0xb5, 0xd6, 0x77, 0x22 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x11, 0x1f, 0x6c, 0x37, 0xcf, 0x40, 0xa1, 0x15, 0x9d, 0x00, 0xfb, 0x59, 0xfb, 0x04, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0x43, 0xb5, 0x1a, 0xb6, 0x09, 0x05, 0x23, 0x72, 0x98, 0x9a, 0x26, 0xe9, 0xcd, 0xd7, 0x14 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 13 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4d, 0xce, 0xde, 0x8d, 0xa9, 0xe2, 0x57, 0x8f, 0x39, 0x70, 0x3d, 0x44, 0x33, 0xdc, 0x64, 0x59 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 14 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1a, 0x4c, 0x1c, 0x26, 0x3b, 0xbc, 0xcf, 0xaf, 0xc1, 0x17, 0x82, 0x89, 0x46, 0x85, 0xe3, 0xa8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 15 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x7a, 0xd8, 0x48, 0x80, 0xdb, 0x50, 0x61, 0x34, 0x23, 0xd6, 0xd5, 0x27, 0xa2, 0x82, 0x3d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 16 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0x0b, 0x71, 0xdf, 0xc6, 0x88, 0xe1, 0x50, 0xd8, 0x15, 0x2c, 0x5b, 0x35, 0xeb, 0xc1, 0x4d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 17 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xef, 0x24, 0x95, 0xda, 0xbf, 0x32, 0x38, 0x85, 0xaa, 0xb3, 0x9c, 0x80, 0xf1, 0x8d, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 18 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x63, 0x3c, 0xaf, 0xea, 0x39, 0x5b, 0xc0, 0x3a, 0xda, 0xe3, 0xa1, 0xe2, 0x06, 0x8e, 0x4b, 0x4e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 19 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0x1b, 0x48, 0x2b, 0x53, 0x76, 0x1c, 0xf6, 0x31, 0x81, 0x9b, 0x74, 0x9a, 0x6f, 0x37, 0x24 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 20 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x6e, 0x6f, 0x85, 0x1a, 0xb5, 0x2c, 0x77, 0x19, 0x98, 0xdb, 0xb2, 0xd7, 0x1c, 0x75, 0xa9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 21 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x85, 0xf2, 0xba, 0x84, 0xf8, 0xc3, 0x07, 0xcf, 0x52, 0x5e, 0x12, 0x4c, 0x3e, 0x22, 0xe6, 0xcc }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 22 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xcc, 0xa9, 0x8b, 0xf6, 0xa8, 0x35, 0xfa, 0x64, 0x95, 0x5f, 0x72, 0xde, 0x41, 0x15, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 23 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0x75, 0xe2, 0xd3, 0x6e, 0xeb, 0xd6, 0x54, 0x11, 0xf1, 0x4f, 0xd0, 0xeb, 0x1d, 0x2a, 0x06 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 24 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbd, 0x49, 0x29, 0x50, 0x06, 0x25, 0x0f, 0xfc, 0xa5, 0x10, 0x0b, 0x60, 0x07, 0xa0, 0xea, 0xde }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 25 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x90, 0x52, 0x7d, 0x0e, 0xf7, 0xc7, 0x0f, 0x45, 0x9c, 0xd3, 0x94, 0x0d, 0xf3, 0x16, 0xec }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 26 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbb, 0xd1, 0x09, 0x7a, 0x62, 0x43, 0x3f, 0x79, 0x44, 0x9f, 0xa9, 0x7d, 0x4e, 0xe8, 0x0d, 0xbf }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 27 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x05, 0x8e, 0x40, 0x8f, 0x5b, 0x99, 0xb0, 0xe0, 0xf0, 0x61, 0xa1, 0x76, 0x1b, 0x5b, 0x3b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 28 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5f, 0xd1, 0xf1, 0x3f, 0xa0, 0xf3, 0x1e, 0x37, 0xfa, 0xbd, 0xe3, 0x28, 0xf8, 0x94, 0xea, 0xc2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 29 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x4a, 0xf7, 0xc9, 0x48, 0xdf, 0x26, 0xe2, 0xef, 0x3e, 0x01, 0xc1, 0xee, 0x5b, 0x8f, 0x6f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 30 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x9f, 0xd7, 0x20, 0x8f, 0xb9, 0x2d, 0x44, 0xa0, 0x74, 0xa6, 0x77, 0xee, 0x98, 0x61, 0xac }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 31 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xad, 0x9f, 0xc6, 0x13, 0xa7, 0x03, 0x25, 0x1b, 0x54, 0xc6, 0x4a, 0x0e, 0x76, 0x43, 0x17, 0x11 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 32 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xac, 0x9e, 0xcc, 0xc4, 0xcc, 0x75, 0xe2, 0x71, 0x16, 0x18, 0xf8, 0x0b, 0x15, 0x48, 0xe8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 33 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x20, 0x25, 0xc7, 0x4b, 0x8a, 0xd8, 0xf4, 0xcd, 0xa1, 0x7e, 0xe2, 0x04, 0x9c, 0x4c, 0x90, 0x2d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 34 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x5c, 0xa0, 0x5f, 0xe5, 0x28, 0xf1, 0xce, 0x9b, 0x79, 0x01, 0x66, 0xe8, 0xd5, 0x51, 0xe7 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 35 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6f, 0x62, 0x38, 0xd8, 0x96, 0x60, 0x48, 0xd4, 0x96, 0x71, 0x54, 0xe0, 0xda, 0xd5, 0xa6, 0xc9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 36 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xb2, 0x1b, 0x4e, 0x76, 0x40, 0xa9, 0xb3, 0x34, 0x6d, 0xe8, 0xb8, 0x2f, 0xb4, 0x1e, 0x49 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 37 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x36, 0xf2, 0x51, 0xad, 0x1d, 0x11, 0xd4, 0x9d, 0xc3, 0x44, 0x62, 0x8b, 0x18, 0x84, 0xe1 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 38 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x7e, 0x94, 0x70, 0xae, 0x7a, 0xbe, 0xa5, 0xa9, 0x76, 0x9d, 0x49, 0x18, 0x26, 0x28, 0xc3 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 39 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xdc, 0xc2, 0xd2, 0x7f, 0xc9, 0x86, 0x56, 0x33, 0xf8, 0x52, 0x23, 0xcf, 0x0d, 0x61, 0x1f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 40 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbe, 0x66, 0xcf, 0xea, 0x2f, 0xec, 0xd6, 0xbf, 0x0e, 0xc7, 0xb4, 0x35, 0x2c, 0x99, 0xbc, 0xaa }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 41 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0x31, 0x14, 0x4f, 0x87, 0xa2, 0xef, 0x52, 0x3f, 0xac, 0xdc, 0xf2, 0x1a, 0x42, 0x78, 0x04 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 42 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xbb, 0x0f, 0x56, 0x29, 0xfb, 0x6a, 0xae, 0x5e, 0x18, 0x39, 0xa3, 0xc3, 0x62, 0x5d, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 43 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0x9d, 0xb3, 0x33, 0x53, 0x06, 0xfe, 0x1e, 0xc6, 0x12, 0xbd, 0xbf, 0xae, 0x6b, 0x60, 0x28 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 44 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3d, 0xd5, 0xc3, 0x46, 0x34, 0xa7, 0x9d, 0x3c, 0xfc, 0xc8, 0x33, 0x97, 0x60, 0xe6, 0xf5, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 45 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0xbd, 0xa1, 0x18, 0xa3, 0xed, 0x7a, 0xf3, 0x14, 0xfa, 0x2c, 0xcc, 0x5c, 0x07, 0xb7, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 46 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x37, 0xa6, 0x4f, 0x7d, 0x4f, 0x46, 0xfe, 0x6f, 0xea, 0x3b, 0x34, 0x9e, 0xc7, 0x8e, 0x38 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 47 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x5f, 0x06, 0x8c, 0x28, 0x47, 0x66, 0x05, 0x73, 0x5a, 0xd6, 0x71, 0xbb, 0x8f, 0x39, 0xf3 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 48 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x68, 0x2c, 0x5e, 0xcd, 0x71, 0x89, 0x8e, 0x08, 0x94, 0x2a, 0xc9, 0xaa, 0x89, 0x87, 0x5c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 49 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0x03, 0x1c, 0xb9, 0xd6, 0x76, 0xc3, 0x02, 0x2d, 0x7f, 0x26, 0x22, 0x7e, 0x85, 0xc3, 0x8f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 50 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x84, 0x63, 0xfb, 0x06, 0x4d, 0xb5, 0xd5, 0x2b, 0xb6, 0x4b, 0xfe, 0xf6, 0x4f, 0x2d, 0xda }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 51 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8a, 0xa9, 0xb7, 0x5e, 0x78, 0x45, 0x93, 0x87, 0x6c, 0x53, 0xa0, 0x0e, 0xae, 0x5a, 0xf5, 0x2b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 52 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3f, 0x84, 0x56, 0x6d, 0xf2, 0x3d, 0xa4, 0x8a, 0xf6, 0x92, 0x72, 0x2f, 0xe9, 0x80, 0x57, 0x3a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 53 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x31, 0x69, 0x0b, 0x5e, 0xd4, 0x1c, 0x7e, 0xb4, 0x2a, 0x1e, 0x83, 0x27, 0x0a, 0x7f, 0xf0, 0xe6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 54 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xdd, 0x77, 0x02, 0x64, 0x6d, 0x55, 0xf0, 0x83, 0x65, 0xe4, 0x77, 0xd3, 0x59, 0x0e, 0xda }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 55 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4c, 0x02, 0x2a, 0xc6, 0x2b, 0x3c, 0xb7, 0x8d, 0x73, 0x9c, 0xc6, 0x7b, 0x3e, 0x20, 0xbb, 0x7e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 56 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x2f, 0xa1, 0x37, 0xce, 0x18, 0xb5, 0xdf, 0xe7, 0x90, 0x6f, 0x55, 0x0b, 0xb1, 0x33, 0x70 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 57 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3e, 0x0c, 0xda, 0xdf, 0x2e, 0x68, 0x35, 0x3c, 0x00, 0x27, 0x67, 0x2c, 0x97, 0x14, 0x4d, 0xd3 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 58 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0xc4, 0xb2, 0x00, 0xb3, 0x83, 0xfc, 0x1f, 0x2b, 0x2e, 0xa6, 0x77, 0x61, 0x8a, 0x1d, 0x27 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 59 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x82, 0x5f, 0x99, 0xb0, 0xe9, 0xbb, 0x34, 0x77, 0xc1, 0xc0, 0x71, 0x3b, 0x01, 0x5a, 0xac }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 60 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0xb9, 0xff, 0xfb, 0x5c, 0x18, 0x7f, 0x7d, 0xdc, 0x7a, 0xb1, 0x0f, 0x4f, 0xb7, 0x75, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 61 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xb4, 0xe8, 0x7a, 0x32, 0xb3, 0x7d, 0x6f, 0x2c, 0x83, 0x28, 0xd3, 0xb5, 0x37, 0x78, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 62 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x76, 0xc1, 0x3a, 0x5d, 0x22, 0x0f, 0x4d, 0xa9, 0x22, 0x4e, 0x74, 0x89, 0x63, 0x91, 0xce }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 63 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0xef, 0xe7, 0xa0, 0xe2, 0xe0, 0x31, 0xe2, 0x53, 0x6d, 0xa0, 0x1d, 0xf7, 0x99, 0xc9, 0x27 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 64 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0x8f, 0xd8, 0x22, 0x68, 0x0a, 0x85, 0x97, 0x4e, 0x53, 0xa5, 0xa8, 0xeb, 0x9d, 0x38, 0xde }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 65 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xf0, 0xa9, 0x1b, 0x2e, 0x45, 0xf8, 0xcc, 0x37, 0xb7, 0x80, 0x5a, 0x30, 0x42, 0x58, 0x8d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 66 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0x7a, 0x62, 0x52, 0x25, 0x5e, 0x46, 0xd6, 0x36, 0x4d, 0xbe, 0xed, 0xa3, 0x1e, 0x27, 0x9c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 67 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf5, 0x1a, 0x0f, 0x69, 0x44, 0x42, 0xb8, 0xf0, 0x55, 0x71, 0x79, 0x7f, 0xec, 0x7e, 0xe8, 0xbf }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 68 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9f, 0xf0, 0x71, 0xb1, 0x65, 0xb5, 0x19, 0x8a, 0x93, 0xdd, 0xde, 0xeb, 0xc5, 0x4d, 0x09, 0xb5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 69 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x0a, 0x19, 0xfd, 0x57, 0x58, 0xb0, 0xc4, 0xbc, 0x1a, 0x5d, 0xf8, 0x9c, 0xf7, 0x38, 0x77 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 70 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x12, 0x01, 0x66, 0x30, 0x71, 0x19, 0xca, 0x22, 0x80, 0xe9, 0x31, 0x56, 0x68, 0xe9, 0x6f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 71 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0x3b, 0x9f, 0x1e, 0x09, 0x9c, 0x2a, 0x09, 0xdc, 0x09, 0x1e, 0x90, 0xe4, 0xf1, 0x8f, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 72 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0x04, 0x0b, 0x89, 0x1d, 0x4b, 0x37, 0xf6, 0x85, 0x1f, 0x7e, 0xc2, 0x19, 0xcd, 0x3f, 0x6d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 73 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9f, 0x0f, 0xde, 0xc0, 0x8b, 0x7f, 0xd7, 0x9a, 0xa3, 0x95, 0x35, 0xbe, 0xa4, 0x2d, 0xb9, 0x2a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 74 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x70, 0xf1, 0x68, 0xfc, 0x74, 0xbf, 0x91, 0x1d, 0xf2, 0x40, 0xbc, 0xd2, 0xce, 0xf2, 0x36 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 75 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x46, 0x2c, 0xcd, 0x7f, 0x5f, 0xd1, 0x10, 0x8d, 0xbc, 0x15, 0x2f, 0x3c, 0xac, 0xad, 0x32, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 76 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0xaf, 0x53, 0x4a, 0x7d, 0x0b, 0x64, 0x3a, 0x01, 0x86, 0x87, 0x85, 0xd8, 0x6d, 0xfb, 0x95 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 77 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x98, 0x02, 0x96, 0x19, 0x7e, 0x1a, 0x50, 0x22, 0x32, 0x6c, 0x31, 0xda, 0x4b, 0xf6, 0xf3 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 78 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf9, 0x7d, 0x57, 0xb3, 0x33, 0x3b, 0x62, 0x81, 0xb0, 0x7d, 0x48, 0x6d, 0xb2, 0xd4, 0xe2, 0x0c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 79 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x3f, 0xa3, 0x67, 0x20, 0x23, 0x1a, 0xfe, 0x4c, 0x75, 0x9a, 0xde, 0x6b, 0xd6, 0x2e, 0xb6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 80 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xcf, 0xac, 0x0c, 0x02, 0xca, 0x53, 0x83, 0x43, 0xc6, 0x81, 0x17, 0xe0, 0xa1, 0x59, 0x38 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 81 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xad, 0x49, 0x16, 0xf5, 0xee, 0x57, 0x72, 0xbe, 0x76, 0x4f, 0xc0, 0x27, 0xb8, 0xa6, 0xe5, 0x39 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 82 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x16, 0x87, 0x3e, 0x16, 0x78, 0x61, 0x0d, 0x7e, 0x14, 0xc0, 0x2d, 0x00, 0x2e, 0xa8, 0x45 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 83 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4e, 0x6e, 0x62, 0x7c, 0x1a, 0xcc, 0x51, 0x34, 0x00, 0x53, 0xa8, 0x23, 0x6d, 0x57, 0x95, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 84 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x0c, 0x84, 0x10, 0xae, 0xee, 0xad, 0x92, 0xfe, 0xec, 0x1e, 0xb4, 0x30, 0xd6, 0x52, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 85 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0x6f, 0x7e, 0x23, 0xe8, 0x35, 0xe1, 0x14, 0x97, 0x7f, 0x60, 0xe1, 0xa5, 0x92, 0x20, 0x2e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 86 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0x8a, 0xd5, 0x05, 0x5a, 0x36, 0x70, 0x41, 0xfa, 0xde, 0x09, 0xd9, 0xa7, 0x0a, 0x79, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 87 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x91, 0x82, 0x3a, 0x3c, 0x66, 0x6b, 0xb6, 0x16, 0x28, 0x25, 0xe7, 0x86, 0x06, 0xa7, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 88 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xca, 0x36, 0x6a, 0x9b, 0xf4, 0x7b, 0x7b, 0x86, 0x8b, 0x77, 0xe2, 0x5c, 0x18, 0xa3, 0x64 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 89 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x4c, 0x9e, 0xfc, 0x23, 0x7e, 0x4a, 0x44, 0x29, 0x65, 0xf8, 0x4b, 0xce, 0x20, 0x24, 0x7a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 90 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x58, 0x41, 0x1f, 0xfb, 0xe6, 0x3f, 0xdb, 0x9c, 0x8a, 0xa1, 0xbf, 0xae, 0xd6, 0x7b, 0x52 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 91 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x04, 0xbc, 0x3d, 0xa2, 0x17, 0x9c, 0x30, 0x15, 0x49, 0x8b, 0x0e, 0x03, 0x91, 0x0d, 0xb5, 0xb8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 92 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x07, 0x1e, 0xea, 0xb3, 0xf9, 0x35, 0xdb, 0xc2, 0x5d, 0x00, 0x84, 0x14, 0x60, 0x26, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 93 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0e, 0xbd, 0x7c, 0x30, 0xed, 0x20, 0x16, 0xe0, 0x8b, 0xa8, 0x06, 0xdd, 0xb0, 0x08, 0xbc, 0xc8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 94 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0xc6, 0xbe, 0xcf, 0x0f, 0x4c, 0xec, 0x71, 0x29, 0xcb, 0xd2, 0x2d, 0x1a, 0x79, 0xb1, 0xb8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 95 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0a, 0xee, 0xde, 0x5b, 0x91, 0xf7, 0x21, 0x70, 0x0e, 0x9e, 0x62, 0xed, 0xbf, 0x60, 0xb7, 0x81 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 96 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x65, 0x81, 0xaf, 0x0d, 0xcf, 0xbe, 0xd1, 0x58, 0x5e, 0x0a, 0x24, 0x2c, 0x64, 0xb8, 0xdf }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 97 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x93, 0xdc, 0x91, 0x16, 0x62, 0xae, 0x47, 0x32, 0x16, 0xba, 0x22, 0x18, 0x9a, 0x51, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 98 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x76, 0x06, 0xfa, 0x36, 0xd8, 0x64, 0x73, 0xe6, 0xfb, 0x3a, 0x1b, 0xb0, 0xe2, 0xc0, 0xad, 0xf5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 99 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x20, 0x78, 0xe9, 0xe1, 0x1f, 0xbb, 0x78, 0xe2, 0x6f, 0xfb, 0x88, 0x99, 0xe9, 0x6b, 0x9a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 100 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xb2, 0x64, 0xe9, 0x21, 0xe9, 0xe4, 0xa8, 0x26, 0x94, 0x58, 0x9e, 0xf3, 0x79, 0x82, 0x62 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 101 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0x45, 0x95, 0xcb, 0x4f, 0xa7, 0x02, 0x67, 0x15, 0xf5, 0x5b, 0xd6, 0x8e, 0x28, 0x82, 0xf9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 102 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0x88, 0xa3, 0x02, 0xbd, 0xbc, 0x09, 0x19, 0x7d, 0xf1, 0xed, 0xae, 0x68, 0x92, 0x6e, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 103 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xf7, 0x50, 0x23, 0x90, 0xb8, 0xa4, 0xa2, 0x21, 0xcf, 0xec, 0xd0, 0x66, 0x66, 0x24, 0xba }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 104 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3d, 0x20, 0x25, 0x3a, 0xdb, 0xce, 0x3b, 0xe2, 0x37, 0x37, 0x67, 0xc4, 0xd8, 0x22, 0xc5, 0x66 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 105 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0x27, 0x34, 0xa3, 0x92, 0x9b, 0xf8, 0x4c, 0xf0, 0x11, 0x6c, 0x98, 0x56, 0xa3, 0xc1, 0x8c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 106 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0xab, 0xc4, 0x93, 0x94, 0x57, 0x42, 0x2b, 0xb9, 0x57, 0xda, 0x3c, 0x56, 0x93, 0x8c, 0x6d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 107 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x2b, 0xdd, 0x2e, 0x7c, 0x52, 0x51, 0x30, 0xfa, 0xdc, 0x8f, 0x76, 0xfc, 0x6f, 0x4b, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 108 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xa8, 0x3d, 0x7b, 0x94, 0xc6, 0x99, 0xcb, 0xcb, 0x8a, 0x7d, 0x9b, 0x61, 0xf6, 0x40, 0x93 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 109 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xce, 0x61, 0xd6, 0x35, 0x14, 0xad, 0xed, 0x03, 0xd4, 0x3e, 0x6e, 0xbf, 0xc3, 0xa9, 0x00, 0x1f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 110 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x83, 0x9d, 0xd5, 0x8e, 0xea, 0xe6, 0xb8, 0xa3, 0x6a, 0xf4, 0x8e, 0xd6, 0x3d, 0x2d, 0xc9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 111 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcd, 0x5e, 0xce, 0x55, 0xb8, 0xda, 0x3b, 0xf6, 0x22, 0xc4, 0x10, 0x0d, 0xf5, 0xde, 0x46, 0xf9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 112 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x6f, 0x46, 0xf4, 0x0e, 0x0a, 0xc5, 0xfc, 0x0a, 0x9c, 0x11, 0x05, 0xf8, 0x00, 0xf4, 0x8d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 113 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x26, 0xd4, 0x7d, 0xa3, 0xae, 0xb0, 0x28, 0xde, 0x4f, 0xb5, 0xb3, 0xa8, 0x54, 0xa2, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 114 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0xf5, 0x3b, 0xf6, 0x20, 0xd3, 0x67, 0x72, 0x68, 0x44, 0x52, 0x12, 0x90, 0x43, 0x89, 0xd5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 115 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0x61, 0x7d, 0x28, 0xb5, 0xe0, 0xf4, 0x60, 0x54, 0x92, 0xb1, 0x82, 0xa5, 0xd7, 0xf9, 0xf6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 116 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9a, 0xae, 0xc4, 0xfa, 0xbb, 0xf6, 0xfa, 0xe2, 0xa7, 0x1f, 0xef, 0xf0, 0x2e, 0x37, 0x2b, 0x39 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 117 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x90, 0xc6, 0x2d, 0x88, 0xb5, 0xc4, 0x28, 0x09, 0xab, 0xf7, 0x82, 0x48, 0x8e, 0xd1, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 118 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0xf1, 0xc5, 0xa4, 0x08, 0x99, 0xe1, 0x57, 0x72, 0x85, 0x7c, 0xcb, 0x65, 0xc7, 0xa0, 0x9a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 119 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x08, 0x43, 0xd2, 0x9b, 0x25, 0xa3, 0x89, 0x7c, 0x69, 0x2c, 0xe1, 0xdd, 0x81, 0xee, 0x52 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 120 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x66, 0xbc, 0x65, 0xb6, 0x94, 0x1d, 0x86, 0xe8, 0x42, 0x0a, 0x7f, 0xfb, 0x09, 0x64, 0xdb }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 121 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x81, 0x93, 0xc6, 0xff, 0x85, 0x22, 0x5c, 0xed, 0x42, 0x55, 0xe9, 0x2f, 0x6e, 0x07, 0x8a, 0x14 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 122 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x61, 0xcb, 0x24, 0x24, 0xd7, 0xd4, 0xa3, 0x80, 0xd5, 0x47, 0xf9, 0xe7, 0xec, 0x1c, 0xb9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 123 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0xf9, 0x3d, 0x9e, 0xc0, 0x84, 0x53, 0xa0, 0x71, 0xe2, 0xe2, 0x87, 0x78, 0x77, 0xa9, 0xc8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 124 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xee, 0xfa, 0x80, 0xce, 0x6a, 0x4a, 0x9d, 0x59, 0x8e, 0x3f, 0xec, 0x36, 0x54, 0x34, 0xd2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 125 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd6, 0x20, 0x68, 0x44, 0x45, 0x78, 0xe3, 0xab, 0x39, 0xce, 0x7e, 0xc9, 0x5d, 0xd0, 0x45, 0xdc }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 126 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xf7, 0x1d, 0x4d, 0xd9, 0xa7, 0x1f, 0xe5, 0xd8, 0xbc, 0x8b, 0xa7, 0xe6, 0xea, 0x30, 0x48 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 127 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x25, 0xa3, 0x47, 0xac, 0x47, 0x9d, 0x4f, 0x9d, 0x95, 0xc5, 0xcb, 0x8d, 0x3f, 0xd7, 0xe9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 128 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x71, 0x4e, 0x94, 0xa5, 0x77, 0x89, 0x55, 0xcc, 0x03, 0x46, 0x35, 0x8e, 0x94, 0x78, 0x3a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 129 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x36, 0xb4, 0x4b, 0xb2, 0x9e, 0x0c, 0x7d, 0x89, 0xfa, 0x4b, 0x2d, 0x4b, 0x67, 0x7d, 0x2a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 130 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5d, 0x45, 0x4b, 0x75, 0x02, 0x1d, 0x76, 0xd4, 0xb8, 0x4f, 0x87, 0x3a, 0x8f, 0x87, 0x7b, 0x92 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 131 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc3, 0x49, 0x8f, 0x7e, 0xce, 0xd2, 0x09, 0x53, 0x14, 0xfc, 0x28, 0x11, 0x58, 0x85, 0xb3, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 132 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0x66, 0x88, 0x56, 0x53, 0x9a, 0xd8, 0xe4, 0x05, 0xbd, 0x12, 0x3f, 0xe6, 0xc8, 0x85, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 133 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0x80, 0xdb, 0x7f, 0x3a, 0x87, 0xb8, 0x60, 0x55, 0x43, 0xcf, 0xdb, 0xe6, 0x75, 0x40, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 134 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x5d, 0x03, 0xb1, 0x30, 0x69, 0xc3, 0x65, 0x8b, 0x31, 0x79, 0xbe, 0x91, 0xb0, 0x80, 0x0c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 135 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x1b, 0x38, 0x4a, 0xc4, 0xd9, 0x3e, 0xda, 0x00, 0xc9, 0x2a, 0xdd, 0x09, 0x95, 0xea, 0x5f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 136 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbf, 0x81, 0x15, 0x80, 0x54, 0x71, 0x74, 0x1b, 0xd5, 0xad, 0x20, 0xa0, 0x39, 0x44, 0x79, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 137 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0x4c, 0x24, 0xb6, 0x89, 0x4b, 0x03, 0x8b, 0x3c, 0x0d, 0x09, 0xb1, 0xdf, 0x06, 0x8b, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 138 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x39, 0x67, 0xa1, 0x0c, 0xff, 0xe2, 0x7d, 0x01, 0x78, 0x54, 0x5f, 0xbf, 0x6a, 0x40, 0x54, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 139 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0x85, 0xe9, 0xc9, 0x5d, 0xe1, 0xa9, 0xec, 0x5a, 0x53, 0x63, 0xa8, 0xa0, 0x53, 0x47, 0x2d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 140 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa9, 0xee, 0xc0, 0x3c, 0x8a, 0xbe, 0xc7, 0xba, 0x68, 0x31, 0x5c, 0x2c, 0x8c, 0x23, 0x16, 0xe0 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 141 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0xc8, 0xe4, 0x14, 0xc2, 0xf3, 0x88, 0x22, 0x7a, 0xe1, 0x49, 0x86, 0xfc, 0x98, 0x35, 0x24 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 142 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5d, 0x94, 0x2b, 0x7f, 0x46, 0x22, 0xce, 0x05, 0x6c, 0x3c, 0xe3, 0xce, 0x5f, 0x1d, 0xd9, 0xd6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 143 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x40, 0xd6, 0x48, 0xce, 0x21, 0xa3, 0x02, 0x02, 0x82, 0xc3, 0xf1, 0xb5, 0x28, 0xa0, 0xb6 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 144 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0xd0, 0x89, 0xc3, 0x6d, 0x5c, 0x5a, 0x4e, 0xfc, 0x68, 0x9e, 0x3b, 0x0d, 0xe1, 0x0d, 0xd5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 145 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0xda, 0x5d, 0xf4, 0xbe, 0xcb, 0x54, 0x62, 0xe0, 0x3a, 0x0e, 0xd0, 0x0d, 0x29, 0x56, 0x29 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 146 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xf4, 0xe1, 0x29, 0x13, 0x6c, 0x1a, 0x4b, 0x7a, 0x0f, 0x38, 0x93, 0x5c, 0xc3, 0x4b, 0x2b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 147 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0xa4, 0xc7, 0x61, 0x8b, 0x0c, 0xe4, 0x8a, 0x3d, 0x5a, 0xee, 0x1a, 0x1c, 0x01, 0x14, 0xc4 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 148 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x35, 0x2d, 0xf0, 0x25, 0xc6, 0x5c, 0x7b, 0x0b, 0xf3, 0x06, 0xfb, 0xee, 0x0f, 0x36, 0xba }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 149 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0x8a, 0xca, 0x23, 0xfd, 0x34, 0x09, 0xf3, 0x8a, 0xf6, 0x33, 0x78, 0xed, 0x2f, 0x54, 0x73 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 150 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0x83, 0x6a, 0x0e, 0x06, 0xa7, 0x96, 0x91, 0xb3, 0x66, 0x67, 0xd5, 0x38, 0x0d, 0x81, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 151 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x90, 0x50, 0x80, 0xf7, 0xac, 0xf1, 0xcd, 0xae, 0x0a, 0x91, 0xfc, 0x3e, 0x85, 0xae, 0xe4 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 152 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xc9, 0xe4, 0x64, 0x6d, 0xbc, 0x3d, 0x63, 0x20, 0xfc, 0x66, 0x89, 0xd9, 0x3e, 0x88, 0x33 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 153 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x77, 0x41, 0x3d, 0xea, 0x59, 0x25, 0xb7, 0xf5, 0x41, 0x7e, 0xa4, 0x7f, 0xf1, 0x9f, 0x59 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 154 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0xae, 0x81, 0x29, 0xf8, 0x43, 0xd8, 0x6d, 0xc7, 0x86, 0xa0, 0xfb, 0x1a, 0x18, 0x49, 0x70 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 155 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0xfe, 0xfb, 0x53, 0x41, 0x00, 0x79, 0x6e, 0xeb, 0xbd, 0x99, 0x02, 0x06, 0x75, 0x4e, 0x19 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 156 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8c, 0x79, 0x1d, 0x5f, 0xdd, 0xdf, 0x47, 0x0d, 0xa0, 0x4f, 0x3e, 0x6d, 0xc4, 0xa5, 0xb5, 0xb5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 157 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0x3b, 0xbd, 0xc0, 0x7a, 0x46, 0x11, 0xae, 0x4b, 0xb2, 0x66, 0xea, 0x50, 0x34, 0xa3, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 158 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0x02, 0xe3, 0x8e, 0x48, 0x9a, 0xa7, 0x47, 0x62, 0xf3, 0xef, 0xc5, 0xbb, 0x23, 0x20, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 159 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x20, 0x14, 0x81, 0x66, 0x5c, 0xba, 0xfc, 0x1f, 0xcc, 0x22, 0x0b, 0xc5, 0x45, 0xfb, 0x3d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 160 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x60, 0x75, 0x7e, 0xc6, 0xce, 0x68, 0xcf, 0x19, 0x5e, 0x45, 0x4c, 0xfd, 0x0f, 0x32, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 161 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0xec, 0x7c, 0xe6, 0xa6, 0xcb, 0xd0, 0x7c, 0x04, 0x34, 0x16, 0x73, 0x7f, 0x1b, 0xbb, 0x33 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 162 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0xc5, 0x41, 0x39, 0x04, 0x48, 0x7a, 0x80, 0x5d, 0x70, 0xa8, 0xed, 0xd9, 0xc3, 0x55, 0x27 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 163 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x78, 0x46, 0xb2, 0xb2, 0xe3, 0x6f, 0x1f, 0x03, 0x24, 0xc8, 0x6f, 0x7f, 0x1b, 0x98, 0xe2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 164 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x2e, 0xee, 0x1a, 0x0c, 0xbd, 0x19, 0xca, 0x2d, 0x69, 0xb4, 0x26, 0x89, 0x40, 0x44, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 165 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0x6b, 0x5b, 0x39, 0x77, 0xba, 0x6e, 0xfa, 0x51, 0x28, 0xef, 0xbd, 0xa9, 0xff, 0x03, 0xcd }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 166 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x14, 0x45, 0xee, 0x94, 0xc0, 0xf0, 0x8c, 0xde, 0xe5, 0xc3, 0x44, 0xec, 0xd1, 0xe2, 0x33 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 167 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbe, 0x28, 0x83, 0x19, 0x02, 0x93, 0x63, 0xc2, 0x62, 0x2f, 0xeb, 0xa4, 0xb0, 0x5d, 0xfd, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 168 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0xd1, 0x87, 0x55, 0x23, 0xf3, 0xcd, 0x21, 0xc3, 0x95, 0x65, 0x1e, 0x6e, 0xe1, 0x5e, 0x56 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 169 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x5a, 0x40, 0x86, 0x57, 0x83, 0x7c, 0x53, 0xbf, 0x16, 0xf9, 0xd8, 0x46, 0x5d, 0xce, 0x19 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 170 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x0b, 0xf4, 0x2c, 0xb1, 0x07, 0xf5, 0x5c, 0xcf, 0xf2, 0xfc, 0x09, 0xee, 0x08, 0xca, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 171 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xd9, 0xbb, 0xb4, 0xa7, 0xdc, 0x2e, 0x4a, 0x23, 0x53, 0x6a, 0x58, 0x80, 0xa2, 0xdb, 0x67 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 172 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xe4, 0x47, 0xb3, 0x62, 0xc4, 0x84, 0x99, 0x3d, 0xec, 0x94, 0x42, 0xa3, 0xb4, 0x6a, 0xef }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 173 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0xdf, 0xfb, 0x05, 0x90, 0x4b, 0xff, 0x7c, 0x47, 0x81, 0xdf, 0x78, 0x0a, 0xd2, 0x68, 0x37 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 174 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc3, 0x3b, 0xc1, 0x3e, 0x8d, 0xe8, 0x8a, 0xc2, 0x52, 0x32, 0xaa, 0x74, 0x96, 0x39, 0x87, 0x83 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 175 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x35, 0x9c, 0x70, 0x80, 0x3a, 0x3b, 0x2a, 0x3d, 0x54, 0x2e, 0x87, 0x81, 0xde, 0xa9, 0x75 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 176 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbc, 0xc6, 0x5b, 0x52, 0x6f, 0x88, 0xd0, 0x5b, 0x89, 0xce, 0x8a, 0x52, 0x02, 0x1f, 0xdb, 0x06 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 177 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x91, 0xa3, 0x88, 0x55, 0xc8, 0xc4, 0x64, 0x38, 0x51, 0xfb, 0xfb, 0x35, 0x8b, 0x01, 0x09 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 178 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x6e, 0x88, 0x93, 0xa1, 0x14, 0xae, 0x8e, 0x27, 0xd5, 0xab, 0x03, 0xa5, 0x49, 0x96, 0x10 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 179 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x29, 0xd2, 0xb8, 0xdf, 0x97, 0xda, 0x72, 0x8c, 0xdd, 0x8b, 0x1e, 0x7f, 0x94, 0x50, 0x77 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 180 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0x70, 0xa5, 0xa1, 0x8c, 0xfc, 0x0d, 0xd5, 0x82, 0xf1, 0xd8, 0x8d, 0x5c, 0x9a, 0x17, 0x20 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 181 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xbc, 0x65, 0xaa, 0x8e, 0x89, 0x56, 0x2e, 0x3f, 0x27, 0x4d, 0x45, 0xaf, 0x1c, 0xd1, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 182 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x98, 0x55, 0x1d, 0xa1, 0xa6, 0x50, 0x32, 0x76, 0xae, 0x1c, 0x77, 0x62, 0x5f, 0x9e, 0xa6, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 183 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0d, 0xdf, 0xe5, 0x1c, 0xed, 0x7e, 0x3f, 0x4a, 0xe9, 0x27, 0xda, 0xa3, 0xfe, 0x45, 0x2c, 0xee }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 184 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x82, 0x62, 0x51, 0xe4, 0xce, 0x38, 0x4b, 0x80, 0x21, 0x8b, 0x0e, 0x1d, 0xa1, 0xdd, 0x4c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 185 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xac, 0xf7, 0x28, 0xb8, 0x8a, 0xbb, 0xad, 0x70, 0x11, 0xed, 0x0e, 0x64, 0xa1, 0x68, 0x0c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 186 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x0d, 0x8e, 0xe7, 0xc5, 0x67, 0x7e, 0x09, 0x9a, 0xc7, 0x4c, 0x99, 0x94, 0xee, 0x4c, 0xfb }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 187 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xf6, 0x1a, 0xe3, 0x62, 0xe8, 0x82, 0xdd, 0xc0, 0x16, 0x74, 0x74, 0xa7, 0xa7, 0x7f, 0x3a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 188 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0x68, 0xb0, 0x0b, 0xa7, 0x85, 0x9e, 0x09, 0x70, 0xec, 0xfd, 0x75, 0x7e, 0xfe, 0xcf, 0x7c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 189 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0x41, 0x54, 0x47, 0x86, 0x62, 0x30, 0xd2, 0x8b, 0xb1, 0xea, 0x18, 0xa4, 0xcd, 0xfd, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 190 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0x61, 0x83, 0x39, 0x2f, 0x7a, 0x87, 0x63, 0xaf, 0xec, 0x68, 0xa0, 0x60, 0x26, 0x41, 0x41 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 191 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0x56, 0x5c, 0x8d, 0x73, 0xcf, 0xd4, 0x13, 0x0b, 0x4a, 0xa1, 0x4d, 0x89, 0x11, 0x71, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 192 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x23, 0x2a, 0x4e, 0xd2, 0x1c, 0xcc, 0x27, 0xc1, 0x9c, 0x96, 0x10, 0x07, 0x8c, 0xab, 0xac }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 193 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x4f, 0x32, 0xea, 0x71, 0x82, 0x8c, 0x7d, 0x32, 0x90, 0x77, 0xe7, 0x12, 0x23, 0x16, 0x66 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 194 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd6, 0x44, 0x24, 0xf2, 0x3c, 0xb9, 0x72, 0x15, 0xe9, 0xc2, 0xc6, 0xf2, 0x8d, 0x29, 0xea, 0xb7 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 195 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x3e, 0x82, 0xb5, 0x33, 0xf6, 0x8c, 0x75, 0xc2, 0x38, 0xce, 0xbd, 0xb2, 0xee, 0x89, 0xa2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 196 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x3a, 0x3d, 0x24, 0x15, 0x7a, 0x51, 0xf1, 0xee, 0x08, 0x93, 0xf6, 0x77, 0x74, 0x17, 0xe7 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 197 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xec, 0xac, 0xfc, 0xd4, 0x00, 0x08, 0x4d, 0x07, 0x86, 0x12, 0xb1, 0x94, 0x5f, 0x2e, 0xf5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 198 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xcd, 0x8b, 0xb1, 0x73, 0x25, 0x9e, 0xb3, 0x3a, 0x52, 0x42, 0xb0, 0xde, 0x31, 0xa4, 0x55 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 199 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xe9, 0xed, 0xdb, 0xc3, 0x75, 0xe7, 0x92, 0xc1, 0x99, 0x92, 0xc1, 0x91, 0x65, 0x01, 0x2b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 200 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8a, 0x77, 0x22, 0x31, 0xc0, 0x1d, 0xfd, 0xd7, 0xc9, 0x8e, 0x4c, 0xfd, 0xdc, 0xc0, 0x80, 0x7a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 201 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0xda, 0x7f, 0xf6, 0xb8, 0x31, 0x91, 0x80, 0xff, 0x0d, 0x6e, 0x65, 0x62, 0x9d, 0x01, 0xc3 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 202 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x67, 0xef, 0x0e, 0x2d, 0x01, 0xa9, 0x93, 0x94, 0x4d, 0xd3, 0x97, 0x10, 0x14, 0x13, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 203 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0xf8, 0x0e, 0x9d, 0x84, 0x5b, 0xcc, 0x0f, 0x62, 0x92, 0x6a, 0xf7, 0x2e, 0xab, 0xca, 0x39 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 204 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x02, 0x99, 0x07, 0x27, 0xaa, 0x08, 0x78, 0x63, 0x7b, 0x45, 0xdc, 0xd3, 0xa3, 0xb0, 0x74 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 205 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x2e, 0x64, 0x7d, 0x53, 0x60, 0xe0, 0x92, 0x30, 0xa5, 0xd7, 0x38, 0xca, 0x33, 0x47, 0x1e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 206 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0x56, 0x41, 0x3c, 0x7a, 0xdd, 0x6f, 0x43, 0xd1, 0xd5, 0x6e, 0x4f, 0x02, 0x19, 0x03, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 207 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x69, 0xcd, 0x06, 0x06, 0xe1, 0x5a, 0xf7, 0x29, 0xd6, 0xbc, 0xa1, 0x43, 0x01, 0x6d, 0x98, 0x42 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 208 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x85, 0xd7, 0xc1, 0xa5, 0x00, 0x87, 0x3a, 0x20, 0x09, 0x9c, 0x4c, 0xaa, 0x3c, 0x3f, 0x5b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 209 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xc0, 0xd2, 0x30, 0xf8, 0x89, 0x14, 0x15, 0xb8, 0x7b, 0x83, 0xf9, 0x5f, 0x2e, 0x09, 0xd1 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 210 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0x27, 0xd0, 0x8c, 0x52, 0x3d, 0x8e, 0xba, 0x69, 0x7a, 0x43, 0x36, 0x50, 0x7d, 0x1f, 0x42 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 211 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0x15, 0xaa, 0xb8, 0x27, 0x01, 0xef, 0xa5, 0xae, 0x36, 0xab, 0x1d, 0x6b, 0x76, 0x29, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 212 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5b, 0xf0, 0x05, 0x18, 0x93, 0xa1, 0x8b, 0xb3, 0x0e, 0x13, 0x9a, 0x58, 0xfe, 0xd0, 0xfa, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 213 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xe8, 0xad, 0xf6, 0x56, 0x38, 0xfd, 0x9c, 0xdf, 0x3b, 0xc2, 0x2c, 0x17, 0xfe, 0x4d, 0xbd }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 214 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xe6, 0xee, 0x32, 0x65, 0x83, 0xa0, 0x58, 0x64, 0x91, 0xc9, 0x64, 0x18, 0xd1, 0xa3, 0x5d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 215 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0xb5, 0x49, 0xc2, 0xec, 0x75, 0x6f, 0x82, 0xec, 0xc4, 0x80, 0x08, 0xe5, 0x29, 0x95, 0x6b }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 216 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x37, 0x7b, 0x6d, 0xa6, 0x69, 0xb0, 0x72, 0x12, 0x9e, 0x05, 0x7c, 0xc2, 0x8e, 0x9c, 0xa5 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 217 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0x94, 0xb8, 0xb0, 0xcb, 0x8b, 0xcc, 0x91, 0x90, 0x72, 0x26, 0x2b, 0x3f, 0xa0, 0x5a, 0xd9 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 218 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0xbb, 0x83, 0xdf, 0xd0, 0xd7, 0xab, 0xcb, 0x05, 0xcd, 0x28, 0xca, 0xd2, 0xdf, 0xb5, 0x23 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 219 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x87, 0x78, 0x03, 0xde, 0x77, 0x74, 0x4b, 0xb9, 0x70, 0xd0, 0xa9, 0x1f, 0x4d, 0xeb, 0xae }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 220 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0x79, 0xf3, 0x37, 0x0c, 0xf6, 0xe5, 0xce, 0x12, 0xae, 0x59, 0x69, 0xc8, 0xee, 0xa3, 0x12 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 221 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xdc, 0x99, 0xfa, 0x3d, 0x4f, 0x98, 0xce, 0x80, 0x98, 0x5e, 0x72, 0x33, 0x88, 0x93, 0x13 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 222 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0x38, 0xe7, 0x59, 0x07, 0x5b, 0xa5, 0xca, 0xb6, 0x45, 0x7d, 0xa5, 0x18, 0x44, 0x29, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 223 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0xbe, 0xd8, 0xdb, 0xf6, 0x15, 0x86, 0x8a, 0x1f, 0x9d, 0x9b, 0x05, 0xd3, 0xe7, 0xa2, 0x67 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 224 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0x4b, 0x14, 0x8b, 0x8c, 0xb1, 0xd8, 0xc3, 0x2b, 0x28, 0x7e, 0x89, 0x69, 0x03, 0xd1, 0x50 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 225 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x4b, 0x03, 0x3d, 0xf4, 0xda, 0x85, 0x3f, 0x4b, 0xe3, 0xe2, 0x43, 0xf7, 0xe5, 0x13, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 226 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3f, 0x58, 0xc9, 0x50, 0xf0, 0x36, 0x71, 0x60, 0xad, 0xec, 0x45, 0xf2, 0x44, 0x1e, 0x74, 0x11 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 227 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0xf6, 0x55, 0x53, 0x6a, 0x70, 0x4e, 0x5a, 0xce, 0x18, 0x2d, 0x74, 0x2a, 0x82, 0x0c, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 228 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x7b, 0xd6, 0xbb, 0x63, 0x41, 0x87, 0x31, 0xae, 0xac, 0x79, 0x0f, 0xe4, 0x2d, 0x61, 0xe8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 229 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe7, 0x4a, 0x4c, 0x99, 0x9b, 0x4c, 0x06, 0x4e, 0x48, 0xbb, 0x1e, 0x41, 0x3f, 0x51, 0xe5, 0xea }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 230 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x9e, 0xbe, 0xfd, 0xb4, 0xcc, 0xf3, 0x0f, 0x29, 0x6c, 0xec, 0xb3, 0xbc, 0x19, 0x43, 0xe8 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 231 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x31, 0x94, 0x36, 0x7a, 0x48, 0x98, 0xc5, 0x02, 0xc1, 0x3b, 0xb7, 0x47, 0x86, 0x40, 0xa7, 0x2d }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 232 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xda, 0x79, 0x77, 0x13, 0x26, 0x3d, 0x6f, 0x33, 0xa5, 0x47, 0x8a, 0x65, 0xef, 0x60, 0xd4, 0x12 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 233 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0xac, 0x39, 0xbb, 0x1e, 0xf8, 0x6b, 0x9c, 0x13, 0x44, 0xf2, 0x14, 0x67, 0x9a, 0xa3, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 234 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0xde, 0xa9, 0xe6, 0x50, 0x53, 0x2b, 0xe5, 0xbc, 0x0e, 0x73, 0x25, 0x33, 0x7f, 0xd3, 0x63 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 235 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0xa2, 0x04, 0xdb, 0xd9, 0xc2, 0xaf, 0x15, 0x8b, 0x6c, 0xa6, 0x7a, 0x51, 0x56, 0xce, 0x4a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 236 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x0a, 0x0e, 0x75, 0xa8, 0xda, 0x36, 0x73, 0x5a, 0xee, 0x66, 0x84, 0xd9, 0x65, 0xa7, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 237 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x52, 0xfc, 0x3e, 0x62, 0x04, 0x92, 0xea, 0x99, 0x64, 0x1e, 0xa1, 0x68, 0xda, 0x5b, 0x6d, 0x52 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 238 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xe0, 0xc7, 0xf1, 0x5b, 0x47, 0x72, 0x46, 0x7d, 0x2c, 0xfc, 0x87, 0x30, 0x00, 0xb2, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 239 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x35, 0x31, 0x13, 0x5e, 0x0c, 0x4d, 0x70, 0xa3, 0x8f, 0x8b, 0xdb, 0x19, 0x0b, 0xa0, 0x4e }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 240 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0xa3, 0x9a, 0x0f, 0x56, 0x63, 0xf4, 0xc0, 0xfe, 0x5f, 0x2d, 0x3c, 0xaf, 0xff, 0x42, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 241 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0x4b, 0x5e, 0x90, 0xdb, 0x35, 0x4c, 0x1e, 0x42, 0xf6, 0x1f, 0xab, 0xe1, 0x67, 0xb2, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 242 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x50, 0xe6, 0xd3, 0xc9, 0xb6, 0x69, 0x8a, 0x7c, 0xd2, 0x76, 0xf9, 0x6b, 0x14, 0x73, 0xf3, 0x5a }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 243 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x38, 0xf0, 0x8e, 0x0e, 0xbe, 0xe9, 0x69, 0x05, 0xd8, 0xf2, 0xe8, 0x25, 0x20, 0x8f, 0x43 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 244 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x37, 0x8c, 0x86, 0x67, 0x2a, 0xa5, 0x4a, 0x3a, 0x26, 0x6b, 0xa1, 0x9d, 0x25, 0x80, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 245 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0xa7, 0xc3, 0x08, 0x6f, 0x5f, 0x95, 0x11, 0xb3, 0x12, 0x33, 0xda, 0x7c, 0xab, 0x91, 0x60 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 246 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5b, 0x40, 0xff, 0x4e, 0xc9, 0xbe, 0x53, 0x6b, 0xa2, 0x30, 0x35, 0xfa, 0x4f, 0x06, 0x06, 0x4c }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 247 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0xeb, 0x5a, 0xf8, 0x41, 0x6b, 0x25, 0x71, 0x49, 0x37, 0x21, 0x94, 0xe8, 0xb8, 0x87, 0x49 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 248 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0x00, 0x5a, 0x8a, 0xed, 0x8a, 0x36, 0x1c, 0x92, 0xe4, 0x40, 0xc1, 0x55, 0x20, 0xcb, 0xd1 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 249 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x03, 0x62, 0x76, 0x11, 0x67, 0x8a, 0x99, 0x77, 0x17, 0x57, 0x88, 0x07, 0xa8, 0x00, 0xe2 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 250 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x78, 0x61, 0x8f, 0x74, 0xf6, 0xf3, 0x69, 0x6e, 0x0a, 0x47, 0x79, 0xb9, 0x0b, 0x5a, 0x77 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 251 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x72, 0x03, 0x71, 0xa0, 0x49, 0x62, 0xea, 0xea, 0x0a, 0x85, 0x2e, 0x69, 0x97, 0x28, 0x58 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 252 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0x8a, 0x81, 0x33, 0xaa, 0x8c, 0xcf, 0x70, 0xe2, 0xbd, 0x32, 0x85, 0x83, 0x1c, 0xa6, 0xb7 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 253 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x93, 0x6b, 0xd2, 0x7f, 0xb1, 0x46, 0x8f, 0xc8, 0xb4, 0x8b, 0xc4, 0x83, 0x32, 0x17, 0x25 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 254 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb0, 0x7d, 0x4f, 0x3e, 0x2c, 0xd2, 0xef, 0x2e, 0xb5, 0x45, 0x98, 0x07, 0x54, 0xdf, 0xea, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 255 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0xf8, 0x5f, 0x1b, 0x5d, 0x54, 0xad, 0xbc, 0x30, 0x7b, 0x0a, 0x04, 0x83, 0x89, 0xad, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 0 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x5a, 0x6d, 0xcb, 0x19, 0xb2, 0x01, 0xa0, 0x1e, 0xbc, 0xfa, 0x8a, 0xa2, 0x2b, 0x57, 0x59 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 1 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0x91, 0x69, 0xcd, 0xcf, 0x2d, 0x83, 0xe8, 0x38, 0x12, 0x5a, 0x12, 0xee, 0x6a, 0xa4, 0x00 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 2 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0xf3, 0xa7, 0x2f, 0xc3, 0xcd, 0xf7, 0x4d, 0xfa, 0xf6, 0xc3, 0xe6, 0xb9, 0x7b, 0x2f, 0xa6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 3 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x77, 0x76, 0x79, 0xd5, 0x00, 0x37, 0xc7, 0x94, 0x91, 0xa9, 0x4d, 0xa7, 0x6a, 0x9a, 0x35 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 4 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xf4, 0x89, 0x3e, 0xca, 0xfa, 0x0a, 0x02, 0x47, 0xa8, 0x98, 0xe0, 0x40, 0x69, 0x15, 0x59 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 5 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0xbb, 0x41, 0x37, 0x03, 0x73, 0x53, 0x26, 0x31, 0x0a, 0x26, 0x9b, 0xd3, 0xaa, 0x94, 0xb2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 6 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0xe3, 0x22, 0x46, 0xbe, 0xd2, 0xb0, 0xe8, 0x59, 0xe5, 0x5c, 0x1c, 0xc6, 0xb2, 0x65, 0x02 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 7 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xec, 0x52, 0xa2, 0x12, 0xf8, 0x0a, 0x09, 0xdf, 0x63, 0x17, 0x02, 0x1b, 0xc2, 0xa9, 0x81, 0x9e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 8 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0x3e, 0x5b, 0x60, 0x0e, 0xb7, 0x0d, 0xbc, 0xcf, 0x6c, 0x0b, 0x1d, 0x9a, 0x68, 0x18, 0x2c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 9 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa3, 0xf5, 0x99, 0xd6, 0x3a, 0x82, 0xa9, 0x68, 0xc3, 0x3f, 0xe2, 0x65, 0x90, 0x74, 0x59, 0x70 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 10 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0xcc, 0xb9, 0xb1, 0x33, 0x70, 0x02, 0xcb, 0xac, 0x42, 0xc5, 0x20, 0xb5, 0xd6, 0x77, 0x22 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 11 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x11, 0x1f, 0x6c, 0x37, 0xcf, 0x40, 0xa1, 0x15, 0x9d, 0x00, 0xfb, 0x59, 0xfb, 0x04, 0x88 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 12 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0x43, 0xb5, 0x1a, 0xb6, 0x09, 0x05, 0x23, 0x72, 0x98, 0x9a, 0x26, 0xe9, 0xcd, 0xd7, 0x14 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 13 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4d, 0xce, 0xde, 0x8d, 0xa9, 0xe2, 0x57, 0x8f, 0x39, 0x70, 0x3d, 0x44, 0x33, 0xdc, 0x64, 0x59 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 14 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1a, 0x4c, 0x1c, 0x26, 0x3b, 0xbc, 0xcf, 0xaf, 0xc1, 0x17, 0x82, 0x89, 0x46, 0x85, 0xe3, 0xa8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 15 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x7a, 0xd8, 0x48, 0x80, 0xdb, 0x50, 0x61, 0x34, 0x23, 0xd6, 0xd5, 0x27, 0xa2, 0x82, 0x3d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 16 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0x0b, 0x71, 0xdf, 0xc6, 0x88, 0xe1, 0x50, 0xd8, 0x15, 0x2c, 0x5b, 0x35, 0xeb, 0xc1, 0x4d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 17 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xef, 0x24, 0x95, 0xda, 0xbf, 0x32, 0x38, 0x85, 0xaa, 0xb3, 0x9c, 0x80, 0xf1, 0x8d, 0x8b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 18 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x63, 0x3c, 0xaf, 0xea, 0x39, 0x5b, 0xc0, 0x3a, 0xda, 0xe3, 0xa1, 0xe2, 0x06, 0x8e, 0x4b, 0x4e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 19 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0x1b, 0x48, 0x2b, 0x53, 0x76, 0x1c, 0xf6, 0x31, 0x81, 0x9b, 0x74, 0x9a, 0x6f, 0x37, 0x24 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 20 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x6e, 0x6f, 0x85, 0x1a, 0xb5, 0x2c, 0x77, 0x19, 0x98, 0xdb, 0xb2, 0xd7, 0x1c, 0x75, 0xa9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 21 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x85, 0xf2, 0xba, 0x84, 0xf8, 0xc3, 0x07, 0xcf, 0x52, 0x5e, 0x12, 0x4c, 0x3e, 0x22, 0xe6, 0xcc }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 22 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0xcc, 0xa9, 0x8b, 0xf6, 0xa8, 0x35, 0xfa, 0x64, 0x95, 0x5f, 0x72, 0xde, 0x41, 0x15, 0xfe }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 23 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0x75, 0xe2, 0xd3, 0x6e, 0xeb, 0xd6, 0x54, 0x11, 0xf1, 0x4f, 0xd0, 0xeb, 0x1d, 0x2a, 0x06 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 24 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbd, 0x49, 0x29, 0x50, 0x06, 0x25, 0x0f, 0xfc, 0xa5, 0x10, 0x0b, 0x60, 0x07, 0xa0, 0xea, 0xde }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 25 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x90, 0x52, 0x7d, 0x0e, 0xf7, 0xc7, 0x0f, 0x45, 0x9c, 0xd3, 0x94, 0x0d, 0xf3, 0x16, 0xec }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 26 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbb, 0xd1, 0x09, 0x7a, 0x62, 0x43, 0x3f, 0x79, 0x44, 0x9f, 0xa9, 0x7d, 0x4e, 0xe8, 0x0d, 0xbf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 27 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x05, 0x8e, 0x40, 0x8f, 0x5b, 0x99, 0xb0, 0xe0, 0xf0, 0x61, 0xa1, 0x76, 0x1b, 0x5b, 0x3b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 28 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5f, 0xd1, 0xf1, 0x3f, 0xa0, 0xf3, 0x1e, 0x37, 0xfa, 0xbd, 0xe3, 0x28, 0xf8, 0x94, 0xea, 0xc2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 29 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x4a, 0xf7, 0xc9, 0x48, 0xdf, 0x26, 0xe2, 0xef, 0x3e, 0x01, 0xc1, 0xee, 0x5b, 0x8f, 0x6f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 30 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x9f, 0xd7, 0x20, 0x8f, 0xb9, 0x2d, 0x44, 0xa0, 0x74, 0xa6, 0x77, 0xee, 0x98, 0x61, 0xac }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 31 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xad, 0x9f, 0xc6, 0x13, 0xa7, 0x03, 0x25, 0x1b, 0x54, 0xc6, 0x4a, 0x0e, 0x76, 0x43, 0x17, 0x11 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 32 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xac, 0x9e, 0xcc, 0xc4, 0xcc, 0x75, 0xe2, 0x71, 0x16, 0x18, 0xf8, 0x0b, 0x15, 0x48, 0xe8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 33 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x20, 0x25, 0xc7, 0x4b, 0x8a, 0xd8, 0xf4, 0xcd, 0xa1, 0x7e, 0xe2, 0x04, 0x9c, 0x4c, 0x90, 0x2d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 34 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x5c, 0xa0, 0x5f, 0xe5, 0x28, 0xf1, 0xce, 0x9b, 0x79, 0x01, 0x66, 0xe8, 0xd5, 0x51, 0xe7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 35 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6f, 0x62, 0x38, 0xd8, 0x96, 0x60, 0x48, 0xd4, 0x96, 0x71, 0x54, 0xe0, 0xda, 0xd5, 0xa6, 0xc9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 36 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xb2, 0x1b, 0x4e, 0x76, 0x40, 0xa9, 0xb3, 0x34, 0x6d, 0xe8, 0xb8, 0x2f, 0xb4, 0x1e, 0x49 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 37 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x36, 0xf2, 0x51, 0xad, 0x1d, 0x11, 0xd4, 0x9d, 0xc3, 0x44, 0x62, 0x8b, 0x18, 0x84, 0xe1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 38 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x7e, 0x94, 0x70, 0xae, 0x7a, 0xbe, 0xa5, 0xa9, 0x76, 0x9d, 0x49, 0x18, 0x26, 0x28, 0xc3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 39 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xdc, 0xc2, 0xd2, 0x7f, 0xc9, 0x86, 0x56, 0x33, 0xf8, 0x52, 0x23, 0xcf, 0x0d, 0x61, 0x1f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 40 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbe, 0x66, 0xcf, 0xea, 0x2f, 0xec, 0xd6, 0xbf, 0x0e, 0xc7, 0xb4, 0x35, 0x2c, 0x99, 0xbc, 0xaa }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 41 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdf, 0x31, 0x14, 0x4f, 0x87, 0xa2, 0xef, 0x52, 0x3f, 0xac, 0xdc, 0xf2, 0x1a, 0x42, 0x78, 0x04 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 42 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xbb, 0x0f, 0x56, 0x29, 0xfb, 0x6a, 0xae, 0x5e, 0x18, 0x39, 0xa3, 0xc3, 0x62, 0x5d, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 43 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0x9d, 0xb3, 0x33, 0x53, 0x06, 0xfe, 0x1e, 0xc6, 0x12, 0xbd, 0xbf, 0xae, 0x6b, 0x60, 0x28 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 44 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3d, 0xd5, 0xc3, 0x46, 0x34, 0xa7, 0x9d, 0x3c, 0xfc, 0xc8, 0x33, 0x97, 0x60, 0xe6, 0xf5, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 45 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0xbd, 0xa1, 0x18, 0xa3, 0xed, 0x7a, 0xf3, 0x14, 0xfa, 0x2c, 0xcc, 0x5c, 0x07, 0xb7, 0x61 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 46 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x37, 0xa6, 0x4f, 0x7d, 0x4f, 0x46, 0xfe, 0x6f, 0xea, 0x3b, 0x34, 0x9e, 0xc7, 0x8e, 0x38 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 47 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x5f, 0x06, 0x8c, 0x28, 0x47, 0x66, 0x05, 0x73, 0x5a, 0xd6, 0x71, 0xbb, 0x8f, 0x39, 0xf3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 48 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x68, 0x2c, 0x5e, 0xcd, 0x71, 0x89, 0x8e, 0x08, 0x94, 0x2a, 0xc9, 0xaa, 0x89, 0x87, 0x5c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 49 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5e, 0x03, 0x1c, 0xb9, 0xd6, 0x76, 0xc3, 0x02, 0x2d, 0x7f, 0x26, 0x22, 0x7e, 0x85, 0xc3, 0x8f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 50 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x84, 0x63, 0xfb, 0x06, 0x4d, 0xb5, 0xd5, 0x2b, 0xb6, 0x4b, 0xfe, 0xf6, 0x4f, 0x2d, 0xda }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 51 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8a, 0xa9, 0xb7, 0x5e, 0x78, 0x45, 0x93, 0x87, 0x6c, 0x53, 0xa0, 0x0e, 0xae, 0x5a, 0xf5, 0x2b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 52 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3f, 0x84, 0x56, 0x6d, 0xf2, 0x3d, 0xa4, 0x8a, 0xf6, 0x92, 0x72, 0x2f, 0xe9, 0x80, 0x57, 0x3a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 53 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x31, 0x69, 0x0b, 0x5e, 0xd4, 0x1c, 0x7e, 0xb4, 0x2a, 0x1e, 0x83, 0x27, 0x0a, 0x7f, 0xf0, 0xe6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 54 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xdd, 0x77, 0x02, 0x64, 0x6d, 0x55, 0xf0, 0x83, 0x65, 0xe4, 0x77, 0xd3, 0x59, 0x0e, 0xda }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 55 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4c, 0x02, 0x2a, 0xc6, 0x2b, 0x3c, 0xb7, 0x8d, 0x73, 0x9c, 0xc6, 0x7b, 0x3e, 0x20, 0xbb, 0x7e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 56 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x2f, 0xa1, 0x37, 0xce, 0x18, 0xb5, 0xdf, 0xe7, 0x90, 0x6f, 0x55, 0x0b, 0xb1, 0x33, 0x70 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 57 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3e, 0x0c, 0xda, 0xdf, 0x2e, 0x68, 0x35, 0x3c, 0x00, 0x27, 0x67, 0x2c, 0x97, 0x14, 0x4d, 0xd3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 58 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0xc4, 0xb2, 0x00, 0xb3, 0x83, 0xfc, 0x1f, 0x2b, 0x2e, 0xa6, 0x77, 0x61, 0x8a, 0x1d, 0x27 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 59 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x82, 0x5f, 0x99, 0xb0, 0xe9, 0xbb, 0x34, 0x77, 0xc1, 0xc0, 0x71, 0x3b, 0x01, 0x5a, 0xac }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 60 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0xb9, 0xff, 0xfb, 0x5c, 0x18, 0x7f, 0x7d, 0xdc, 0x7a, 0xb1, 0x0f, 0x4f, 0xb7, 0x75, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 61 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xb4, 0xe8, 0x7a, 0x32, 0xb3, 0x7d, 0x6f, 0x2c, 0x83, 0x28, 0xd3, 0xb5, 0x37, 0x78, 0x02 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 62 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x76, 0xc1, 0x3a, 0x5d, 0x22, 0x0f, 0x4d, 0xa9, 0x22, 0x4e, 0x74, 0x89, 0x63, 0x91, 0xce }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 63 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0xef, 0xe7, 0xa0, 0xe2, 0xe0, 0x31, 0xe2, 0x53, 0x6d, 0xa0, 0x1d, 0xf7, 0x99, 0xc9, 0x27 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 64 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8f, 0x8f, 0xd8, 0x22, 0x68, 0x0a, 0x85, 0x97, 0x4e, 0x53, 0xa5, 0xa8, 0xeb, 0x9d, 0x38, 0xde }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 65 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xf0, 0xa9, 0x1b, 0x2e, 0x45, 0xf8, 0xcc, 0x37, 0xb7, 0x80, 0x5a, 0x30, 0x42, 0x58, 0x8d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 66 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0x7a, 0x62, 0x52, 0x25, 0x5e, 0x46, 0xd6, 0x36, 0x4d, 0xbe, 0xed, 0xa3, 0x1e, 0x27, 0x9c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 67 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf5, 0x1a, 0x0f, 0x69, 0x44, 0x42, 0xb8, 0xf0, 0x55, 0x71, 0x79, 0x7f, 0xec, 0x7e, 0xe8, 0xbf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 68 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9f, 0xf0, 0x71, 0xb1, 0x65, 0xb5, 0x19, 0x8a, 0x93, 0xdd, 0xde, 0xeb, 0xc5, 0x4d, 0x09, 0xb5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 69 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x0a, 0x19, 0xfd, 0x57, 0x58, 0xb0, 0xc4, 0xbc, 0x1a, 0x5d, 0xf8, 0x9c, 0xf7, 0x38, 0x77 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 70 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x12, 0x01, 0x66, 0x30, 0x71, 0x19, 0xca, 0x22, 0x80, 0xe9, 0x31, 0x56, 0x68, 0xe9, 0x6f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 71 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0x3b, 0x9f, 0x1e, 0x09, 0x9c, 0x2a, 0x09, 0xdc, 0x09, 0x1e, 0x90, 0xe4, 0xf1, 0x8f, 0x0a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 72 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xeb, 0x04, 0x0b, 0x89, 0x1d, 0x4b, 0x37, 0xf6, 0x85, 0x1f, 0x7e, 0xc2, 0x19, 0xcd, 0x3f, 0x6d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 73 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9f, 0x0f, 0xde, 0xc0, 0x8b, 0x7f, 0xd7, 0x9a, 0xa3, 0x95, 0x35, 0xbe, 0xa4, 0x2d, 0xb9, 0x2a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 74 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x70, 0xf1, 0x68, 0xfc, 0x74, 0xbf, 0x91, 0x1d, 0xf2, 0x40, 0xbc, 0xd2, 0xce, 0xf2, 0x36 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 75 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x46, 0x2c, 0xcd, 0x7f, 0x5f, 0xd1, 0x10, 0x8d, 0xbc, 0x15, 0x2f, 0x3c, 0xac, 0xad, 0x32, 0x8b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 76 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0xaf, 0x53, 0x4a, 0x7d, 0x0b, 0x64, 0x3a, 0x01, 0x86, 0x87, 0x85, 0xd8, 0x6d, 0xfb, 0x95 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 77 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x98, 0x02, 0x96, 0x19, 0x7e, 0x1a, 0x50, 0x22, 0x32, 0x6c, 0x31, 0xda, 0x4b, 0xf6, 0xf3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 78 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf9, 0x7d, 0x57, 0xb3, 0x33, 0x3b, 0x62, 0x81, 0xb0, 0x7d, 0x48, 0x6d, 0xb2, 0xd4, 0xe2, 0x0c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 79 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x3f, 0xa3, 0x67, 0x20, 0x23, 0x1a, 0xfe, 0x4c, 0x75, 0x9a, 0xde, 0x6b, 0xd6, 0x2e, 0xb6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 80 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xcf, 0xac, 0x0c, 0x02, 0xca, 0x53, 0x83, 0x43, 0xc6, 0x81, 0x17, 0xe0, 0xa1, 0x59, 0x38 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 81 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xad, 0x49, 0x16, 0xf5, 0xee, 0x57, 0x72, 0xbe, 0x76, 0x4f, 0xc0, 0x27, 0xb8, 0xa6, 0xe5, 0x39 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 82 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x16, 0x87, 0x3e, 0x16, 0x78, 0x61, 0x0d, 0x7e, 0x14, 0xc0, 0x2d, 0x00, 0x2e, 0xa8, 0x45 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 83 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4e, 0x6e, 0x62, 0x7c, 0x1a, 0xcc, 0x51, 0x34, 0x00, 0x53, 0xa8, 0x23, 0x6d, 0x57, 0x95, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 84 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x0c, 0x84, 0x10, 0xae, 0xee, 0xad, 0x92, 0xfe, 0xec, 0x1e, 0xb4, 0x30, 0xd6, 0x52, 0xcb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 85 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0x6f, 0x7e, 0x23, 0xe8, 0x35, 0xe1, 0x14, 0x97, 0x7f, 0x60, 0xe1, 0xa5, 0x92, 0x20, 0x2e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 86 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0x8a, 0xd5, 0x05, 0x5a, 0x36, 0x70, 0x41, 0xfa, 0xde, 0x09, 0xd9, 0xa7, 0x0a, 0x79, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 87 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x07, 0x91, 0x82, 0x3a, 0x3c, 0x66, 0x6b, 0xb6, 0x16, 0x28, 0x25, 0xe7, 0x86, 0x06, 0xa7, 0xfe }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 88 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xca, 0x36, 0x6a, 0x9b, 0xf4, 0x7b, 0x7b, 0x86, 0x8b, 0x77, 0xe2, 0x5c, 0x18, 0xa3, 0x64 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 89 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x4c, 0x9e, 0xfc, 0x23, 0x7e, 0x4a, 0x44, 0x29, 0x65, 0xf8, 0x4b, 0xce, 0x20, 0x24, 0x7a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 90 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x58, 0x41, 0x1f, 0xfb, 0xe6, 0x3f, 0xdb, 0x9c, 0x8a, 0xa1, 0xbf, 0xae, 0xd6, 0x7b, 0x52 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 91 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x04, 0xbc, 0x3d, 0xa2, 0x17, 0x9c, 0x30, 0x15, 0x49, 0x8b, 0x0e, 0x03, 0x91, 0x0d, 0xb5, 0xb8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 92 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x07, 0x1e, 0xea, 0xb3, 0xf9, 0x35, 0xdb, 0xc2, 0x5d, 0x00, 0x84, 0x14, 0x60, 0x26, 0x0f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 93 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0e, 0xbd, 0x7c, 0x30, 0xed, 0x20, 0x16, 0xe0, 0x8b, 0xa8, 0x06, 0xdd, 0xb0, 0x08, 0xbc, 0xc8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 94 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0xc6, 0xbe, 0xcf, 0x0f, 0x4c, 0xec, 0x71, 0x29, 0xcb, 0xd2, 0x2d, 0x1a, 0x79, 0xb1, 0xb8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 95 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0a, 0xee, 0xde, 0x5b, 0x91, 0xf7, 0x21, 0x70, 0x0e, 0x9e, 0x62, 0xed, 0xbf, 0x60, 0xb7, 0x81 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 96 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0x65, 0x81, 0xaf, 0x0d, 0xcf, 0xbe, 0xd1, 0x58, 0x5e, 0x0a, 0x24, 0x2c, 0x64, 0xb8, 0xdf }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 97 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x93, 0xdc, 0x91, 0x16, 0x62, 0xae, 0x47, 0x32, 0x16, 0xba, 0x22, 0x18, 0x9a, 0x51, 0x1a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 98 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x76, 0x06, 0xfa, 0x36, 0xd8, 0x64, 0x73, 0xe6, 0xfb, 0x3a, 0x1b, 0xb0, 0xe2, 0xc0, 0xad, 0xf5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 99 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x20, 0x78, 0xe9, 0xe1, 0x1f, 0xbb, 0x78, 0xe2, 0x6f, 0xfb, 0x88, 0x99, 0xe9, 0x6b, 0x9a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 100 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xb2, 0x64, 0xe9, 0x21, 0xe9, 0xe4, 0xa8, 0x26, 0x94, 0x58, 0x9e, 0xf3, 0x79, 0x82, 0x62 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 101 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8d, 0x45, 0x95, 0xcb, 0x4f, 0xa7, 0x02, 0x67, 0x15, 0xf5, 0x5b, 0xd6, 0x8e, 0x28, 0x82, 0xf9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 102 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0x88, 0xa3, 0x02, 0xbd, 0xbc, 0x09, 0x19, 0x7d, 0xf1, 0xed, 0xae, 0x68, 0x92, 0x6e, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 103 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xf7, 0x50, 0x23, 0x90, 0xb8, 0xa4, 0xa2, 0x21, 0xcf, 0xec, 0xd0, 0x66, 0x66, 0x24, 0xba }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 104 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3d, 0x20, 0x25, 0x3a, 0xdb, 0xce, 0x3b, 0xe2, 0x37, 0x37, 0x67, 0xc4, 0xd8, 0x22, 0xc5, 0x66 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 105 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0x27, 0x34, 0xa3, 0x92, 0x9b, 0xf8, 0x4c, 0xf0, 0x11, 0x6c, 0x98, 0x56, 0xa3, 0xc1, 0x8c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 106 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0xab, 0xc4, 0x93, 0x94, 0x57, 0x42, 0x2b, 0xb9, 0x57, 0xda, 0x3c, 0x56, 0x93, 0x8c, 0x6d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 107 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0x2b, 0xdd, 0x2e, 0x7c, 0x52, 0x51, 0x30, 0xfa, 0xdc, 0x8f, 0x76, 0xfc, 0x6f, 0x4b, 0x3f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 108 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xa8, 0x3d, 0x7b, 0x94, 0xc6, 0x99, 0xcb, 0xcb, 0x8a, 0x7d, 0x9b, 0x61, 0xf6, 0x40, 0x93 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 109 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xce, 0x61, 0xd6, 0x35, 0x14, 0xad, 0xed, 0x03, 0xd4, 0x3e, 0x6e, 0xbf, 0xc3, 0xa9, 0x00, 0x1f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 110 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x83, 0x9d, 0xd5, 0x8e, 0xea, 0xe6, 0xb8, 0xa3, 0x6a, 0xf4, 0x8e, 0xd6, 0x3d, 0x2d, 0xc9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 111 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcd, 0x5e, 0xce, 0x55, 0xb8, 0xda, 0x3b, 0xf6, 0x22, 0xc4, 0x10, 0x0d, 0xf5, 0xde, 0x46, 0xf9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 112 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x6f, 0x46, 0xf4, 0x0e, 0x0a, 0xc5, 0xfc, 0x0a, 0x9c, 0x11, 0x05, 0xf8, 0x00, 0xf4, 0x8d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 113 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x26, 0xd4, 0x7d, 0xa3, 0xae, 0xb0, 0x28, 0xde, 0x4f, 0xb5, 0xb3, 0xa8, 0x54, 0xa2, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 114 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0xf5, 0x3b, 0xf6, 0x20, 0xd3, 0x67, 0x72, 0x68, 0x44, 0x52, 0x12, 0x90, 0x43, 0x89, 0xd5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 115 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0x61, 0x7d, 0x28, 0xb5, 0xe0, 0xf4, 0x60, 0x54, 0x92, 0xb1, 0x82, 0xa5, 0xd7, 0xf9, 0xf6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 116 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9a, 0xae, 0xc4, 0xfa, 0xbb, 0xf6, 0xfa, 0xe2, 0xa7, 0x1f, 0xef, 0xf0, 0x2e, 0x37, 0x2b, 0x39 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 117 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x90, 0xc6, 0x2d, 0x88, 0xb5, 0xc4, 0x28, 0x09, 0xab, 0xf7, 0x82, 0x48, 0x8e, 0xd1, 0x30 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 118 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf1, 0xf1, 0xc5, 0xa4, 0x08, 0x99, 0xe1, 0x57, 0x72, 0x85, 0x7c, 0xcb, 0x65, 0xc7, 0xa0, 0x9a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 119 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x08, 0x43, 0xd2, 0x9b, 0x25, 0xa3, 0x89, 0x7c, 0x69, 0x2c, 0xe1, 0xdd, 0x81, 0xee, 0x52 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 120 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x66, 0xbc, 0x65, 0xb6, 0x94, 0x1d, 0x86, 0xe8, 0x42, 0x0a, 0x7f, 0xfb, 0x09, 0x64, 0xdb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 121 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x81, 0x93, 0xc6, 0xff, 0x85, 0x22, 0x5c, 0xed, 0x42, 0x55, 0xe9, 0x2f, 0x6e, 0x07, 0x8a, 0x14 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 122 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x61, 0xcb, 0x24, 0x24, 0xd7, 0xd4, 0xa3, 0x80, 0xd5, 0x47, 0xf9, 0xe7, 0xec, 0x1c, 0xb9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 123 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0xf9, 0x3d, 0x9e, 0xc0, 0x84, 0x53, 0xa0, 0x71, 0xe2, 0xe2, 0x87, 0x78, 0x77, 0xa9, 0xc8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 124 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0xee, 0xfa, 0x80, 0xce, 0x6a, 0x4a, 0x9d, 0x59, 0x8e, 0x3f, 0xec, 0x36, 0x54, 0x34, 0xd2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 125 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd6, 0x20, 0x68, 0x44, 0x45, 0x78, 0xe3, 0xab, 0x39, 0xce, 0x7e, 0xc9, 0x5d, 0xd0, 0x45, 0xdc }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 126 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb5, 0xf7, 0x1d, 0x4d, 0xd9, 0xa7, 0x1f, 0xe5, 0xd8, 0xbc, 0x8b, 0xa7, 0xe6, 0xea, 0x30, 0x48 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 127 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x25, 0xa3, 0x47, 0xac, 0x47, 0x9d, 0x4f, 0x9d, 0x95, 0xc5, 0xcb, 0x8d, 0x3f, 0xd7, 0xe9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 128 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe3, 0x71, 0x4e, 0x94, 0xa5, 0x77, 0x89, 0x55, 0xcc, 0x03, 0x46, 0x35, 0x8e, 0x94, 0x78, 0x3a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 129 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd8, 0x36, 0xb4, 0x4b, 0xb2, 0x9e, 0x0c, 0x7d, 0x89, 0xfa, 0x4b, 0x2d, 0x4b, 0x67, 0x7d, 0x2a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 130 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5d, 0x45, 0x4b, 0x75, 0x02, 0x1d, 0x76, 0xd4, 0xb8, 0x4f, 0x87, 0x3a, 0x8f, 0x87, 0x7b, 0x92 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 131 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc3, 0x49, 0x8f, 0x7e, 0xce, 0xd2, 0x09, 0x53, 0x14, 0xfc, 0x28, 0x11, 0x58, 0x85, 0xb3, 0x3f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 132 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0x66, 0x88, 0x56, 0x53, 0x9a, 0xd8, 0xe4, 0x05, 0xbd, 0x12, 0x3f, 0xe6, 0xc8, 0x85, 0x30 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 133 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0x80, 0xdb, 0x7f, 0x3a, 0x87, 0xb8, 0x60, 0x55, 0x43, 0xcf, 0xdb, 0xe6, 0x75, 0x40, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 134 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x5d, 0x03, 0xb1, 0x30, 0x69, 0xc3, 0x65, 0x8b, 0x31, 0x79, 0xbe, 0x91, 0xb0, 0x80, 0x0c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 135 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xef, 0x1b, 0x38, 0x4a, 0xc4, 0xd9, 0x3e, 0xda, 0x00, 0xc9, 0x2a, 0xdd, 0x09, 0x95, 0xea, 0x5f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 136 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbf, 0x81, 0x15, 0x80, 0x54, 0x71, 0x74, 0x1b, 0xd5, 0xad, 0x20, 0xa0, 0x39, 0x44, 0x79, 0x0f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 137 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc6, 0x4c, 0x24, 0xb6, 0x89, 0x4b, 0x03, 0x8b, 0x3c, 0x0d, 0x09, 0xb1, 0xdf, 0x06, 0x8b, 0x0b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 138 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x39, 0x67, 0xa1, 0x0c, 0xff, 0xe2, 0x7d, 0x01, 0x78, 0x54, 0x5f, 0xbf, 0x6a, 0x40, 0x54, 0x4b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 139 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0x85, 0xe9, 0xc9, 0x5d, 0xe1, 0xa9, 0xec, 0x5a, 0x53, 0x63, 0xa8, 0xa0, 0x53, 0x47, 0x2d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 140 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa9, 0xee, 0xc0, 0x3c, 0x8a, 0xbe, 0xc7, 0xba, 0x68, 0x31, 0x5c, 0x2c, 0x8c, 0x23, 0x16, 0xe0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 141 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0xc8, 0xe4, 0x14, 0xc2, 0xf3, 0x88, 0x22, 0x7a, 0xe1, 0x49, 0x86, 0xfc, 0x98, 0x35, 0x24 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 142 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5d, 0x94, 0x2b, 0x7f, 0x46, 0x22, 0xce, 0x05, 0x6c, 0x3c, 0xe3, 0xce, 0x5f, 0x1d, 0xd9, 0xd6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 143 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x40, 0xd6, 0x48, 0xce, 0x21, 0xa3, 0x02, 0x02, 0x82, 0xc3, 0xf1, 0xb5, 0x28, 0xa0, 0xb6 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 144 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0xd0, 0x89, 0xc3, 0x6d, 0x5c, 0x5a, 0x4e, 0xfc, 0x68, 0x9e, 0x3b, 0x0d, 0xe1, 0x0d, 0xd5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 145 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0xda, 0x5d, 0xf4, 0xbe, 0xcb, 0x54, 0x62, 0xe0, 0x3a, 0x0e, 0xd0, 0x0d, 0x29, 0x56, 0x29 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 146 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdc, 0xf4, 0xe1, 0x29, 0x13, 0x6c, 0x1a, 0x4b, 0x7a, 0x0f, 0x38, 0x93, 0x5c, 0xc3, 0x4b, 0x2b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 147 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0xa4, 0xc7, 0x61, 0x8b, 0x0c, 0xe4, 0x8a, 0x3d, 0x5a, 0xee, 0x1a, 0x1c, 0x01, 0x14, 0xc4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 148 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x35, 0x2d, 0xf0, 0x25, 0xc6, 0x5c, 0x7b, 0x0b, 0xf3, 0x06, 0xfb, 0xee, 0x0f, 0x36, 0xba }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 149 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0x8a, 0xca, 0x23, 0xfd, 0x34, 0x09, 0xf3, 0x8a, 0xf6, 0x33, 0x78, 0xed, 0x2f, 0x54, 0x73 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 150 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0x83, 0x6a, 0x0e, 0x06, 0xa7, 0x96, 0x91, 0xb3, 0x66, 0x67, 0xd5, 0x38, 0x0d, 0x81, 0x88 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 151 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x90, 0x50, 0x80, 0xf7, 0xac, 0xf1, 0xcd, 0xae, 0x0a, 0x91, 0xfc, 0x3e, 0x85, 0xae, 0xe4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 152 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xc9, 0xe4, 0x64, 0x6d, 0xbc, 0x3d, 0x63, 0x20, 0xfc, 0x66, 0x89, 0xd9, 0x3e, 0x88, 0x33 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 153 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x77, 0x41, 0x3d, 0xea, 0x59, 0x25, 0xb7, 0xf5, 0x41, 0x7e, 0xa4, 0x7f, 0xf1, 0x9f, 0x59 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 154 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0xae, 0x81, 0x29, 0xf8, 0x43, 0xd8, 0x6d, 0xc7, 0x86, 0xa0, 0xfb, 0x1a, 0x18, 0x49, 0x70 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 155 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0xfe, 0xfb, 0x53, 0x41, 0x00, 0x79, 0x6e, 0xeb, 0xbd, 0x99, 0x02, 0x06, 0x75, 0x4e, 0x19 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 156 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8c, 0x79, 0x1d, 0x5f, 0xdd, 0xdf, 0x47, 0x0d, 0xa0, 0x4f, 0x3e, 0x6d, 0xc4, 0xa5, 0xb5, 0xb5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 157 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0x3b, 0xbd, 0xc0, 0x7a, 0x46, 0x11, 0xae, 0x4b, 0xb2, 0x66, 0xea, 0x50, 0x34, 0xa3, 0x87 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 158 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc1, 0x02, 0xe3, 0x8e, 0x48, 0x9a, 0xa7, 0x47, 0x62, 0xf3, 0xef, 0xc5, 0xbb, 0x23, 0x20, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 159 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x20, 0x14, 0x81, 0x66, 0x5c, 0xba, 0xfc, 0x1f, 0xcc, 0x22, 0x0b, 0xc5, 0x45, 0xfb, 0x3d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 160 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0x60, 0x75, 0x7e, 0xc6, 0xce, 0x68, 0xcf, 0x19, 0x5e, 0x45, 0x4c, 0xfd, 0x0f, 0x32, 0xca }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 161 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0xec, 0x7c, 0xe6, 0xa6, 0xcb, 0xd0, 0x7c, 0x04, 0x34, 0x16, 0x73, 0x7f, 0x1b, 0xbb, 0x33 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 162 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0xc5, 0x41, 0x39, 0x04, 0x48, 0x7a, 0x80, 0x5d, 0x70, 0xa8, 0xed, 0xd9, 0xc3, 0x55, 0x27 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 163 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x78, 0x46, 0xb2, 0xb2, 0xe3, 0x6f, 0x1f, 0x03, 0x24, 0xc8, 0x6f, 0x7f, 0x1b, 0x98, 0xe2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 164 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x2e, 0xee, 0x1a, 0x0c, 0xbd, 0x19, 0xca, 0x2d, 0x69, 0xb4, 0x26, 0x89, 0x40, 0x44, 0xf0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 165 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x86, 0x6b, 0x5b, 0x39, 0x77, 0xba, 0x6e, 0xfa, 0x51, 0x28, 0xef, 0xbd, 0xa9, 0xff, 0x03, 0xcd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 166 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0x14, 0x45, 0xee, 0x94, 0xc0, 0xf0, 0x8c, 0xde, 0xe5, 0xc3, 0x44, 0xec, 0xd1, 0xe2, 0x33 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 167 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbe, 0x28, 0x83, 0x19, 0x02, 0x93, 0x63, 0xc2, 0x62, 0x2f, 0xeb, 0xa4, 0xb0, 0x5d, 0xfd, 0xfe }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 168 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0xd1, 0x87, 0x55, 0x23, 0xf3, 0xcd, 0x21, 0xc3, 0x95, 0x65, 0x1e, 0x6e, 0xe1, 0x5e, 0x56 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 169 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x5a, 0x40, 0x86, 0x57, 0x83, 0x7c, 0x53, 0xbf, 0x16, 0xf9, 0xd8, 0x46, 0x5d, 0xce, 0x19 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 170 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x0b, 0xf4, 0x2c, 0xb1, 0x07, 0xf5, 0x5c, 0xcf, 0xf2, 0xfc, 0x09, 0xee, 0x08, 0xca, 0x15 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 171 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfd, 0xd9, 0xbb, 0xb4, 0xa7, 0xdc, 0x2e, 0x4a, 0x23, 0x53, 0x6a, 0x58, 0x80, 0xa2, 0xdb, 0x67 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 172 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xe4, 0x47, 0xb3, 0x62, 0xc4, 0x84, 0x99, 0x3d, 0xec, 0x94, 0x42, 0xa3, 0xb4, 0x6a, 0xef }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 173 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0xdf, 0xfb, 0x05, 0x90, 0x4b, 0xff, 0x7c, 0x47, 0x81, 0xdf, 0x78, 0x0a, 0xd2, 0x68, 0x37 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 174 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc3, 0x3b, 0xc1, 0x3e, 0x8d, 0xe8, 0x8a, 0xc2, 0x52, 0x32, 0xaa, 0x74, 0x96, 0x39, 0x87, 0x83 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 175 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x35, 0x9c, 0x70, 0x80, 0x3a, 0x3b, 0x2a, 0x3d, 0x54, 0x2e, 0x87, 0x81, 0xde, 0xa9, 0x75 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 176 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xbc, 0xc6, 0x5b, 0x52, 0x6f, 0x88, 0xd0, 0x5b, 0x89, 0xce, 0x8a, 0x52, 0x02, 0x1f, 0xdb, 0x06 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 177 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x91, 0xa3, 0x88, 0x55, 0xc8, 0xc4, 0x64, 0x38, 0x51, 0xfb, 0xfb, 0x35, 0x8b, 0x01, 0x09 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 178 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xca, 0x6e, 0x88, 0x93, 0xa1, 0x14, 0xae, 0x8e, 0x27, 0xd5, 0xab, 0x03, 0xa5, 0x49, 0x96, 0x10 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 179 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x29, 0xd2, 0xb8, 0xdf, 0x97, 0xda, 0x72, 0x8c, 0xdd, 0x8b, 0x1e, 0x7f, 0x94, 0x50, 0x77 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 180 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x45, 0x70, 0xa5, 0xa1, 0x8c, 0xfc, 0x0d, 0xd5, 0x82, 0xf1, 0xd8, 0x8d, 0x5c, 0x9a, 0x17, 0x20 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 181 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x72, 0xbc, 0x65, 0xaa, 0x8e, 0x89, 0x56, 0x2e, 0x3f, 0x27, 0x4d, 0x45, 0xaf, 0x1c, 0xd1, 0x0b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 182 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x98, 0x55, 0x1d, 0xa1, 0xa6, 0x50, 0x32, 0x76, 0xae, 0x1c, 0x77, 0x62, 0x5f, 0x9e, 0xa6, 0x15 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 183 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0d, 0xdf, 0xe5, 0x1c, 0xed, 0x7e, 0x3f, 0x4a, 0xe9, 0x27, 0xda, 0xa3, 0xfe, 0x45, 0x2c, 0xee }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 184 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x82, 0x62, 0x51, 0xe4, 0xce, 0x38, 0x4b, 0x80, 0x21, 0x8b, 0x0e, 0x1d, 0xa1, 0xdd, 0x4c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 185 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xac, 0xf7, 0x28, 0xb8, 0x8a, 0xbb, 0xad, 0x70, 0x11, 0xed, 0x0e, 0x64, 0xa1, 0x68, 0x0c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 186 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0x0d, 0x8e, 0xe7, 0xc5, 0x67, 0x7e, 0x09, 0x9a, 0xc7, 0x4c, 0x99, 0x94, 0xee, 0x4c, 0xfb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 187 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0xf6, 0x1a, 0xe3, 0x62, 0xe8, 0x82, 0xdd, 0xc0, 0x16, 0x74, 0x74, 0xa7, 0xa7, 0x7f, 0x3a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 188 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0x68, 0xb0, 0x0b, 0xa7, 0x85, 0x9e, 0x09, 0x70, 0xec, 0xfd, 0x75, 0x7e, 0xfe, 0xcf, 0x7c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 189 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0x41, 0x54, 0x47, 0x86, 0x62, 0x30, 0xd2, 0x8b, 0xb1, 0xea, 0x18, 0xa4, 0xcd, 0xfd, 0x02 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 190 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x51, 0x61, 0x83, 0x39, 0x2f, 0x7a, 0x87, 0x63, 0xaf, 0xec, 0x68, 0xa0, 0x60, 0x26, 0x41, 0x41 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 191 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0x56, 0x5c, 0x8d, 0x73, 0xcf, 0xd4, 0x13, 0x0b, 0x4a, 0xa1, 0x4d, 0x89, 0x11, 0x71, 0x0f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 192 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x23, 0x2a, 0x4e, 0xd2, 0x1c, 0xcc, 0x27, 0xc1, 0x9c, 0x96, 0x10, 0x07, 0x8c, 0xab, 0xac }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 193 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x4f, 0x32, 0xea, 0x71, 0x82, 0x8c, 0x7d, 0x32, 0x90, 0x77, 0xe7, 0x12, 0x23, 0x16, 0x66 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 194 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd6, 0x44, 0x24, 0xf2, 0x3c, 0xb9, 0x72, 0x15, 0xe9, 0xc2, 0xc6, 0xf2, 0x8d, 0x29, 0xea, 0xb7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 195 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x3e, 0x82, 0xb5, 0x33, 0xf6, 0x8c, 0x75, 0xc2, 0x38, 0xce, 0xbd, 0xb2, 0xee, 0x89, 0xa2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 196 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x3a, 0x3d, 0x24, 0x15, 0x7a, 0x51, 0xf1, 0xee, 0x08, 0x93, 0xf6, 0x77, 0x74, 0x17, 0xe7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 197 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x84, 0xec, 0xac, 0xfc, 0xd4, 0x00, 0x08, 0x4d, 0x07, 0x86, 0x12, 0xb1, 0x94, 0x5f, 0x2e, 0xf5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 198 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1d, 0xcd, 0x8b, 0xb1, 0x73, 0x25, 0x9e, 0xb3, 0x3a, 0x52, 0x42, 0xb0, 0xde, 0x31, 0xa4, 0x55 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 199 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xe9, 0xed, 0xdb, 0xc3, 0x75, 0xe7, 0x92, 0xc1, 0x99, 0x92, 0xc1, 0x91, 0x65, 0x01, 0x2b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 200 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8a, 0x77, 0x22, 0x31, 0xc0, 0x1d, 0xfd, 0xd7, 0xc9, 0x8e, 0x4c, 0xfd, 0xdc, 0xc0, 0x80, 0x7a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 201 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6e, 0xda, 0x7f, 0xf6, 0xb8, 0x31, 0x91, 0x80, 0xff, 0x0d, 0x6e, 0x65, 0x62, 0x9d, 0x01, 0xc3 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 202 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x67, 0xef, 0x0e, 0x2d, 0x01, 0xa9, 0x93, 0x94, 0x4d, 0xd3, 0x97, 0x10, 0x14, 0x13, 0xcb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 203 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0xf8, 0x0e, 0x9d, 0x84, 0x5b, 0xcc, 0x0f, 0x62, 0x92, 0x6a, 0xf7, 0x2e, 0xab, 0xca, 0x39 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 204 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x02, 0x99, 0x07, 0x27, 0xaa, 0x08, 0x78, 0x63, 0x7b, 0x45, 0xdc, 0xd3, 0xa3, 0xb0, 0x74 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 205 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x2e, 0x64, 0x7d, 0x53, 0x60, 0xe0, 0x92, 0x30, 0xa5, 0xd7, 0x38, 0xca, 0x33, 0x47, 0x1e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 206 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0x56, 0x41, 0x3c, 0x7a, 0xdd, 0x6f, 0x43, 0xd1, 0xd5, 0x6e, 0x4f, 0x02, 0x19, 0x03, 0x30 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 207 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x69, 0xcd, 0x06, 0x06, 0xe1, 0x5a, 0xf7, 0x29, 0xd6, 0xbc, 0xa1, 0x43, 0x01, 0x6d, 0x98, 0x42 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 208 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0x85, 0xd7, 0xc1, 0xa5, 0x00, 0x87, 0x3a, 0x20, 0x09, 0x9c, 0x4c, 0xaa, 0x3c, 0x3f, 0x5b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 209 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4f, 0xc0, 0xd2, 0x30, 0xf8, 0x89, 0x14, 0x15, 0xb8, 0x7b, 0x83, 0xf9, 0x5f, 0x2e, 0x09, 0xd1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 210 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0x27, 0xd0, 0x8c, 0x52, 0x3d, 0x8e, 0xba, 0x69, 0x7a, 0x43, 0x36, 0x50, 0x7d, 0x1f, 0x42 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 211 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0x15, 0xaa, 0xb8, 0x27, 0x01, 0xef, 0xa5, 0xae, 0x36, 0xab, 0x1d, 0x6b, 0x76, 0x29, 0x0f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 212 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5b, 0xf0, 0x05, 0x18, 0x93, 0xa1, 0x8b, 0xb3, 0x0e, 0x13, 0x9a, 0x58, 0xfe, 0xd0, 0xfa, 0x54 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 213 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x97, 0xe8, 0xad, 0xf6, 0x56, 0x38, 0xfd, 0x9c, 0xdf, 0x3b, 0xc2, 0x2c, 0x17, 0xfe, 0x4d, 0xbd }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 214 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xe6, 0xee, 0x32, 0x65, 0x83, 0xa0, 0x58, 0x64, 0x91, 0xc9, 0x64, 0x18, 0xd1, 0xa3, 0x5d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 215 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x26, 0xb5, 0x49, 0xc2, 0xec, 0x75, 0x6f, 0x82, 0xec, 0xc4, 0x80, 0x08, 0xe5, 0x29, 0x95, 0x6b }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 216 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x37, 0x7b, 0x6d, 0xa6, 0x69, 0xb0, 0x72, 0x12, 0x9e, 0x05, 0x7c, 0xc2, 0x8e, 0x9c, 0xa5 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 217 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0x94, 0xb8, 0xb0, 0xcb, 0x8b, 0xcc, 0x91, 0x90, 0x72, 0x26, 0x2b, 0x3f, 0xa0, 0x5a, 0xd9 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 218 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0xbb, 0x83, 0xdf, 0xd0, 0xd7, 0xab, 0xcb, 0x05, 0xcd, 0x28, 0xca, 0xd2, 0xdf, 0xb5, 0x23 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 219 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x87, 0x78, 0x03, 0xde, 0x77, 0x74, 0x4b, 0xb9, 0x70, 0xd0, 0xa9, 0x1f, 0x4d, 0xeb, 0xae }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 220 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0x79, 0xf3, 0x37, 0x0c, 0xf6, 0xe5, 0xce, 0x12, 0xae, 0x59, 0x69, 0xc8, 0xee, 0xa3, 0x12 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 221 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0xdc, 0x99, 0xfa, 0x3d, 0x4f, 0x98, 0xce, 0x80, 0x98, 0x5e, 0x72, 0x33, 0x88, 0x93, 0x13 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 222 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0x38, 0xe7, 0x59, 0x07, 0x5b, 0xa5, 0xca, 0xb6, 0x45, 0x7d, 0xa5, 0x18, 0x44, 0x29, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 223 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0xbe, 0xd8, 0xdb, 0xf6, 0x15, 0x86, 0x8a, 0x1f, 0x9d, 0x9b, 0x05, 0xd3, 0xe7, 0xa2, 0x67 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 224 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x23, 0x4b, 0x14, 0x8b, 0x8c, 0xb1, 0xd8, 0xc3, 0x2b, 0x28, 0x7e, 0x89, 0x69, 0x03, 0xd1, 0x50 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 225 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x29, 0x4b, 0x03, 0x3d, 0xf4, 0xda, 0x85, 0x3f, 0x4b, 0xe3, 0xe2, 0x43, 0xf7, 0xe5, 0x13, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 226 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3f, 0x58, 0xc9, 0x50, 0xf0, 0x36, 0x71, 0x60, 0xad, 0xec, 0x45, 0xf2, 0x44, 0x1e, 0x74, 0x11 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 227 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0xf6, 0x55, 0x53, 0x6a, 0x70, 0x4e, 0x5a, 0xce, 0x18, 0x2d, 0x74, 0x2a, 0x82, 0x0c, 0xf4 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 228 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x7b, 0xd6, 0xbb, 0x63, 0x41, 0x87, 0x31, 0xae, 0xac, 0x79, 0x0f, 0xe4, 0x2d, 0x61, 0xe8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 229 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe7, 0x4a, 0x4c, 0x99, 0x9b, 0x4c, 0x06, 0x4e, 0x48, 0xbb, 0x1e, 0x41, 0x3f, 0x51, 0xe5, 0xea }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 230 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xba, 0x9e, 0xbe, 0xfd, 0xb4, 0xcc, 0xf3, 0x0f, 0x29, 0x6c, 0xec, 0xb3, 0xbc, 0x19, 0x43, 0xe8 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 231 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x31, 0x94, 0x36, 0x7a, 0x48, 0x98, 0xc5, 0x02, 0xc1, 0x3b, 0xb7, 0x47, 0x86, 0x40, 0xa7, 0x2d }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 232 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xda, 0x79, 0x77, 0x13, 0x26, 0x3d, 0x6f, 0x33, 0xa5, 0x47, 0x8a, 0x65, 0xef, 0x60, 0xd4, 0x12 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 233 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0xac, 0x39, 0xbb, 0x1e, 0xf8, 0x6b, 0x9c, 0x13, 0x44, 0xf2, 0x14, 0x67, 0x9a, 0xa3, 0x76 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 234 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0xde, 0xa9, 0xe6, 0x50, 0x53, 0x2b, 0xe5, 0xbc, 0x0e, 0x73, 0x25, 0x33, 0x7f, 0xd3, 0x63 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 235 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0xa2, 0x04, 0xdb, 0xd9, 0xc2, 0xaf, 0x15, 0x8b, 0x6c, 0xa6, 0x7a, 0x51, 0x56, 0xce, 0x4a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 236 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x0a, 0x0e, 0x75, 0xa8, 0xda, 0x36, 0x73, 0x5a, 0xee, 0x66, 0x84, 0xd9, 0x65, 0xa7, 0x78 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 237 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x52, 0xfc, 0x3e, 0x62, 0x04, 0x92, 0xea, 0x99, 0x64, 0x1e, 0xa1, 0x68, 0xda, 0x5b, 0x6d, 0x52 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 238 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xe0, 0xc7, 0xf1, 0x5b, 0x47, 0x72, 0x46, 0x7d, 0x2c, 0xfc, 0x87, 0x30, 0x00, 0xb2, 0xca }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 239 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x35, 0x31, 0x13, 0x5e, 0x0c, 0x4d, 0x70, 0xa3, 0x8f, 0x8b, 0xdb, 0x19, 0x0b, 0xa0, 0x4e }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 240 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0xa3, 0x9a, 0x0f, 0x56, 0x63, 0xf4, 0xc0, 0xfe, 0x5f, 0x2d, 0x3c, 0xaf, 0xff, 0x42, 0x1a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 241 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0x4b, 0x5e, 0x90, 0xdb, 0x35, 0x4c, 0x1e, 0x42, 0xf6, 0x1f, 0xab, 0xe1, 0x67, 0xb2, 0xc0 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 242 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x50, 0xe6, 0xd3, 0xc9, 0xb6, 0x69, 0x8a, 0x7c, 0xd2, 0x76, 0xf9, 0x6b, 0x14, 0x73, 0xf3, 0x5a }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 243 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0x38, 0xf0, 0x8e, 0x0e, 0xbe, 0xe9, 0x69, 0x05, 0xd8, 0xf2, 0xe8, 0x25, 0x20, 0x8f, 0x43 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 244 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x37, 0x8c, 0x86, 0x67, 0x2a, 0xa5, 0x4a, 0x3a, 0x26, 0x6b, 0xa1, 0x9d, 0x25, 0x80, 0xca }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 245 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcc, 0xa7, 0xc3, 0x08, 0x6f, 0x5f, 0x95, 0x11, 0xb3, 0x12, 0x33, 0xda, 0x7c, 0xab, 0x91, 0x60 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 246 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5b, 0x40, 0xff, 0x4e, 0xc9, 0xbe, 0x53, 0x6b, 0xa2, 0x30, 0x35, 0xfa, 0x4f, 0x06, 0x06, 0x4c }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 247 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x60, 0xeb, 0x5a, 0xf8, 0x41, 0x6b, 0x25, 0x71, 0x49, 0x37, 0x21, 0x94, 0xe8, 0xb8, 0x87, 0x49 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 248 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2f, 0x00, 0x5a, 0x8a, 0xed, 0x8a, 0x36, 0x1c, 0x92, 0xe4, 0x40, 0xc1, 0x55, 0x20, 0xcb, 0xd1 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 249 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0x03, 0x62, 0x76, 0x11, 0x67, 0x8a, 0x99, 0x77, 0x17, 0x57, 0x88, 0x07, 0xa8, 0x00, 0xe2 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 250 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x78, 0x61, 0x8f, 0x74, 0xf6, 0xf3, 0x69, 0x6e, 0x0a, 0x47, 0x79, 0xb9, 0x0b, 0x5a, 0x77 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 251 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x72, 0x03, 0x71, 0xa0, 0x49, 0x62, 0xea, 0xea, 0x0a, 0x85, 0x2e, 0x69, 0x97, 0x28, 0x58 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 252 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0x8a, 0x81, 0x33, 0xaa, 0x8c, 0xcf, 0x70, 0xe2, 0xbd, 0x32, 0x85, 0x83, 0x1c, 0xa6, 0xb7 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 253 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x93, 0x6b, 0xd2, 0x7f, 0xb1, 0x46, 0x8f, 0xc8, 0xb4, 0x8b, 0xc4, 0x83, 0x32, 0x17, 0x25 }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 254 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb0, 0x7d, 0x4f, 0x3e, 0x2c, 0xd2, 0xef, 0x2e, 0xb5, 0x45, 0x98, 0x07, 0x54, 0xdf, 0xea, 0x0f }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarKey256.rsp test vector 255 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
	    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0xf8, 0x5f, 0x1b, 0x5d, 0x54, 0xad, 0xbc, 0x30, 0x7b, 0x0a, 0x04, 0x83, 0x89, 0xad, 0xcb }, 16,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },

	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xd7, 0x8e, 0x72, 0x6c, 0x1e, 0xc0, 0x2b, 0x7e, 0xbf, 0xe9, 0x2b, 0x23, 0xd9, 0xec, 0x34 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xe5, 0x93, 0x9c, 0x8e, 0xfd, 0xf2, 0xf0, 0x4e, 0x60, 0xb9, 0xfe, 0x71, 0x17, 0xb2, 0xc2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0x31, 0xd4, 0xd7, 0x4f, 0x5d, 0xcb, 0xf3, 0x9d, 0xaa, 0xf8, 0xca, 0x3a, 0xf6, 0xe5, 0x27 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0xd9, 0xfd, 0x5c, 0xc4, 0xf0, 0x74, 0x41, 0x72, 0x7d, 0xf0, 0xf3, 0x3e, 0x40, 0x1a, 0x36 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xcc, 0xdb, 0x04, 0x46, 0x46, 0xd7, 0xe1, 0xf3, 0xcc, 0xea, 0x3d, 0xca, 0x08, 0xb8, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x16, 0xae, 0x4c, 0xe5, 0x04, 0x2a, 0x67, 0xee, 0x8e, 0x17, 0x7b, 0x7c, 0x58, 0x7e, 0xcc, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0xda, 0x0b, 0xb1, 0x1a, 0x23, 0x85, 0x5d, 0x9c, 0x5c, 0xb1, 0xb4, 0xc6, 0x41, 0x2e, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x4f, 0x1a, 0xa5, 0x30, 0x96, 0x7d, 0x67, 0x32, 0xce, 0x47, 0x15, 0xeb, 0x0e, 0xe2, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x17, 0x38, 0x25, 0x26, 0x21, 0xdd, 0x18, 0x0a, 0x34, 0xf3, 0x45, 0x5b, 0x4b, 0xaa, 0x2f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xe2, 0xb5, 0x08, 0xdb, 0x7f, 0xd8, 0x92, 0x34, 0xca, 0xf7, 0x93, 0x9e, 0xe5, 0x62, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0x49, 0x9c, 0x25, 0x1f, 0x84, 0x42, 0xee, 0x13, 0xf0, 0x93, 0x3b, 0x68, 0x8f, 0xcd, 0x19 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x51, 0x35, 0xf8, 0xa8, 0x1f, 0x25, 0xc9, 0xd6, 0x30, 0xb1, 0x75, 0x02, 0xf6, 0x8e, 0x53 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x87, 0x14, 0x5a, 0x01, 0xad, 0x1c, 0x6c, 0xed, 0xe9, 0x95, 0xea, 0x36, 0x70, 0x45, 0x4f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 13 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xae, 0x3b, 0x10, 0xa0, 0xc8, 0xca, 0x6d, 0x1d, 0x3b, 0x0f, 0xa6, 0x1e, 0x56, 0xb0, 0xb2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 14 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x64, 0xb4, 0xd6, 0x29, 0x81, 0x0f, 0xda, 0x6b, 0xaf, 0xdf, 0x08, 0xf3, 0xb0, 0xd8, 0xd2, 0xc5 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 15 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd7, 0xe5, 0xdb, 0xd3, 0x32, 0x45, 0x95, 0xf8, 0xfd, 0xc7, 0xd7, 0xc5, 0x71, 0xda, 0x6c, 0x2a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 16 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0xf7, 0x23, 0x75, 0x26, 0x4e, 0x16, 0x7f, 0xca, 0x9d, 0xe2, 0xc1, 0x52, 0x7d, 0x96, 0x06 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 17 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xe7, 0x9d, 0xd4, 0xf4, 0x01, 0xff, 0x9b, 0x7e, 0xa9, 0x45, 0xd8, 0x66, 0x66, 0xc1, 0x3b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 18 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0x35, 0xce, 0xa2, 0x79, 0x99, 0x40, 0xb4, 0x0d, 0xb3, 0xf8, 0x19, 0xcb, 0x94, 0xc0, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 19 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x69, 0x41, 0xcb, 0x6b, 0x3e, 0x08, 0xc2, 0xb7, 0xaf, 0xa5, 0x81, 0xeb, 0xdd, 0x60, 0x7b, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 20 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0x20, 0xf4, 0x39, 0xf6, 0xbb, 0x09, 0x7b, 0x29, 0xb8, 0xbd, 0x6d, 0x99, 0xaa, 0xd7, 0x99 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 21 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0x5d, 0x01, 0xf0, 0x58, 0xe5, 0x65, 0xf7, 0x7a, 0xe8, 0x63, 0x78, 0xbd, 0x2c, 0x49, 0xb3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 22 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0xb5, 0xfd, 0x98, 0x19, 0x0e, 0xf4, 0x5f, 0xbb, 0x43, 0x01, 0x43, 0x8d, 0x09, 0x59, 0x50 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 23 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x00, 0x1f, 0xf5, 0xd9, 0x98, 0x06, 0xef, 0xd2, 0x5d, 0xa3, 0x4f, 0x56, 0xbe, 0x85, 0x4b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 24 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x59, 0x4c, 0x60, 0xf5, 0xc8, 0x27, 0x7a, 0x51, 0x13, 0x67, 0x7f, 0x94, 0x20, 0x8d, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 25 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0xc0, 0xfc, 0x18, 0x18, 0xe4, 0xaa, 0x46, 0xbd, 0x2e, 0x39, 0xd6, 0x38, 0xf8, 0x9e, 0x05 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 26 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x02, 0x3e, 0xe9, 0xc3, 0xfd, 0xc4, 0x5a, 0x01, 0x9b, 0x4e, 0x98, 0x5c, 0x7e, 0x1a, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 27 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xf4, 0x01, 0x82, 0xab, 0x46, 0x62, 0xf3, 0x02, 0x3b, 0xae, 0xc1, 0xee, 0x79, 0x6b, 0x57 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 28 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xeb, 0xba, 0xd7, 0x30, 0x36, 0x49, 0xb4, 0x19, 0x4a, 0x69, 0x45, 0xc6, 0xcc, 0x36, 0x94 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 29 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa2, 0x12, 0x4b, 0xea, 0x53, 0xec, 0x28, 0x34, 0x27, 0x9b, 0xed, 0x7f, 0x7e, 0xb0, 0xf9, 0x38 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 30 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb9, 0xfb, 0x43, 0x99, 0xfa, 0x4f, 0xac, 0xc7, 0x30, 0x9e, 0x14, 0xec, 0x98, 0x36, 0x0b, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 31 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x62, 0x77, 0x43, 0x74, 0x20, 0xc5, 0xd6, 0x34, 0xf7, 0x15, 0xae, 0xa8, 0x1a, 0x91, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 32 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x17, 0x1a, 0x0e, 0x1b, 0x2d, 0xd4, 0x24, 0xf0, 0xe0, 0x89, 0xaf, 0x2c, 0x4c, 0x10, 0xf3, 0x2f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 33 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xad, 0xbe, 0x40, 0x2d, 0x1b, 0x20, 0x8f, 0xe7, 0x35, 0xed, 0xce, 0x00, 0xae, 0xe7, 0xce }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 34 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xb0, 0x2f, 0xf9, 0x29, 0xa1, 0x48, 0x5a, 0xf6, 0xf5, 0xc6, 0xd6, 0x55, 0x8b, 0xaa, 0x0f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 35 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x2f, 0xaa, 0xcc, 0x9b, 0xf4, 0x35, 0x08, 0xbf, 0x8f, 0xa8, 0x61, 0x3c, 0xa7, 0x5d, 0xea }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 36 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x2b, 0xf8, 0x28, 0x0f, 0x3f, 0x97, 0x42, 0xc7, 0xed, 0x51, 0x3f, 0xe8, 0x02, 0x62, 0x9c }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 37 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0x5a, 0x41, 0xee, 0x44, 0x2f, 0xa9, 0x92, 0xa6, 0xe3, 0x23, 0x98, 0x6d, 0xed, 0x3f, 0x68 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 38 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0x1e, 0x99, 0xcf, 0x4f, 0x0f, 0x77, 0xce, 0xa8, 0x36, 0xe1, 0x1a, 0x2f, 0xe7, 0x5f, 0xb1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 39 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0xe3, 0xa0, 0xca, 0x90, 0x79, 0xe6, 0x46, 0x33, 0x1d, 0xf8, 0xb4, 0xe7, 0x0d, 0x2c, 0xd6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 40 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4a, 0xfe, 0x7f, 0x12, 0x0c, 0xe7, 0x61, 0x3f, 0x74, 0xfc, 0x12, 0xa0, 0x1a, 0x82, 0x80, 0x73 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 41 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x7f, 0x00, 0x0e, 0x75, 0xe2, 0xc8, 0xb9, 0xd4, 0x79, 0xbe, 0xed, 0x91, 0x3f, 0xe6, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 42 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0x83, 0x0c, 0x8e, 0x7a, 0xae, 0xfe, 0x2d, 0x30, 0x31, 0x0e, 0xf3, 0x81, 0xcb, 0xf6, 0x91 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 43 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x1a, 0xa0, 0xf2, 0xc8, 0x57, 0x01, 0x44, 0xf3, 0x86, 0x57, 0xea, 0x40, 0x85, 0xeb, 0xe5 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 44 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x85, 0x06, 0x2c, 0x2c, 0x90, 0x9f, 0x15, 0xd9, 0x26, 0x9b, 0x6c, 0x18, 0xce, 0x99, 0xc4, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 45 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x80, 0x34, 0xdc, 0x9e, 0x41, 0xb5, 0xa5, 0x60, 0xed, 0x23, 0x9e, 0xea, 0xb1, 0xbc, 0x78 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 46 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0xf9, 0x3a, 0x4c, 0xe5, 0xab, 0x6d, 0x5d, 0x56, 0xf1, 0xb9, 0x3c, 0xf1, 0x99, 0x11, 0xc1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 47 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x31, 0x12, 0xbc, 0xb0, 0xc1, 0xdc, 0xc7, 0x49, 0xd7, 0x99, 0x74, 0x36, 0x91, 0xbf, 0x82 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 48 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0xc5, 0x5b, 0xd7, 0x5c, 0x7f, 0x9c, 0x88, 0x19, 0x89, 0xd3, 0xec, 0x19, 0x11, 0xc0, 0xd4 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 49 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x2e, 0x6b, 0x5e, 0xf1, 0x82, 0xb7, 0xdf, 0xf3, 0x62, 0x9a, 0xbd, 0x6a, 0x12, 0x04, 0x5f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 50 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x32, 0x23, 0x27, 0xe0, 0x17, 0x80, 0xb1, 0x73, 0x97, 0xf2, 0x40, 0x87, 0xf8, 0xcc, 0x6f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 51 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xca, 0xcb, 0x5c, 0xd1, 0x16, 0x92, 0xc3, 0x73, 0xb2, 0x41, 0x17, 0x68, 0x14, 0x9e, 0xe7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 52 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x8e, 0x3d, 0xbb, 0xca, 0x57, 0x78, 0x60, 0xda, 0xb6, 0xb8, 0x0d, 0xa3, 0x13, 0x92, 0x56 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 53 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0xb6, 0x1c, 0x37, 0xbf, 0x32, 0x8e, 0xcc, 0xa8, 0xd7, 0x43, 0x26, 0x5a, 0x3d, 0x42, 0x5c }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 54 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xd9, 0x9c, 0x6b, 0xcc, 0x1f, 0x06, 0xfd, 0xa8, 0xe2, 0x7e, 0x8a, 0xe3, 0xf1, 0xcc, 0xc7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 55 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0xfd, 0x4b, 0x91, 0xc7, 0x01, 0xfd, 0x6b, 0x61, 0xb7, 0xf9, 0x97, 0x82, 0x9d, 0x66, 0x3b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 56 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x00, 0x5d, 0x52, 0xf2, 0x5f, 0x16, 0xbd, 0xc9, 0x54, 0x5a, 0x87, 0x6a, 0x63, 0x49, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 57 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x4d, 0x35, 0x4f, 0x02, 0xbb, 0x5a, 0x5e, 0x47, 0xd3, 0x96, 0x66, 0x86, 0x7f, 0x24, 0x6a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 58 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x51, 0xb8, 0xd6, 0xe1, 0xe1, 0xa0, 0xeb, 0xb1, 0x55, 0xfb, 0xbf, 0x6e, 0x7b, 0x7d, 0xc3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 59 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x98, 0xd4, 0xf4, 0x2f, 0xa7, 0xba, 0x6a, 0x10, 0xac, 0x05, 0xe8, 0x7b, 0x9f, 0x20, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 60 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x11, 0x29, 0x5e, 0x73, 0x9c, 0xa7, 0xd9, 0xb5, 0x0f, 0x8e, 0x4c, 0x0e, 0x75, 0x4a, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 61 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0x33, 0xfc, 0x7d, 0x8a, 0xbe, 0x3c, 0xa1, 0x93, 0x67, 0x59, 0xf8, 0xf5, 0xde, 0xaf, 0x20 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 62 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x5e, 0x0f, 0x56, 0x6d, 0xc9, 0x6c, 0x29, 0x8f, 0x0c, 0x12, 0x63, 0x75, 0x39, 0xb2, 0x5c }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 63 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x07, 0xc3, 0xe7, 0x98, 0x5f, 0xe0, 0xf5, 0xa5, 0x0e, 0x2c, 0xdb, 0x25, 0xc5, 0x10, 0x9e }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 64 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xf9, 0x92, 0xa8, 0x56, 0xfb, 0x27, 0x8b, 0x38, 0x9a, 0x62, 0xf5, 0xd2, 0x74, 0xd7, 0xe9 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 65 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0xd3, 0xed, 0x7a, 0x6f, 0xe1, 0x5a, 0xb4, 0xd9, 0x1a, 0xcb, 0xc7, 0xd0, 0x76, 0x7a, 0xb1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 66 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0xfe, 0xec, 0xd4, 0x5b, 0x2e, 0x67, 0x59, 0x73, 0xac, 0x33, 0xbf, 0x0c, 0x54, 0x24, 0xfc }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 67 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0x80, 0xcb, 0x39, 0x55, 0xba, 0x62, 0xd0, 0x9e, 0xea, 0x66, 0x8f, 0x7c, 0x70, 0x88, 0x17 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 68 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x40, 0x40, 0x33, 0xd6, 0xb7, 0x2b, 0x60, 0x93, 0x54, 0xd5, 0x49, 0x6e, 0x7e, 0xb5, 0x11 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 69 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x31, 0x7a, 0x22, 0x0a, 0x7d, 0x70, 0x0d, 0xa2, 0xb1, 0xe0, 0x75, 0xb0, 0x02, 0x66, 0xe1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 70 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x3b, 0x89, 0x54, 0x22, 0x33, 0xf1, 0x27, 0x1b, 0xf8, 0xfd, 0x0c, 0x0f, 0x40, 0x35, 0x45 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 71 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0x3e, 0xae, 0x96, 0x6f, 0xac, 0x46, 0xdc, 0xa9, 0x27, 0xd6, 0xb1, 0x14, 0xfa, 0x3f, 0x9e }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 72 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0xde, 0xc5, 0x21, 0x31, 0x65, 0x03, 0xd9, 0xd5, 0xee, 0x65, 0xdf, 0x3e, 0xa9, 0x4d, 0xdf }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 73 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xee, 0xf4, 0x56, 0x43, 0x1d, 0xea, 0x8b, 0x4a, 0xcf, 0x83, 0xbd, 0xae, 0x37, 0x17, 0xf7, 0x5f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 74 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x06, 0xf2, 0x51, 0x9a, 0x2f, 0xaf, 0xaa, 0x59, 0x6b, 0xfe, 0xf5, 0xcf, 0xa1, 0x5c, 0x21, 0xb9 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 75 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0x1a, 0x7e, 0xac, 0x7e, 0x2f, 0xe8, 0x09, 0xe4, 0xaa, 0x8d, 0x0d, 0x70, 0x12, 0x53, 0x1a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 76 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0xff, 0xc1, 0x6e, 0x4c, 0x49, 0xb2, 0x68, 0xa2, 0x0f, 0x8d, 0x96, 0xa6, 0x0b, 0x40, 0x58 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 77 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0x86, 0xf9, 0x28, 0x19, 0x99, 0xc5, 0xbb, 0x3b, 0x3e, 0x88, 0x62, 0xe2, 0xf7, 0xc9, 0x88 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 78 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x3b, 0xf9, 0x0d, 0x61, 0xbe, 0xef, 0x39, 0xf4, 0x8d, 0xd6, 0x25, 0xfc, 0xef, 0x13, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 79 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4d, 0x37, 0xc8, 0x50, 0x64, 0x45, 0x63, 0xc6, 0x9f, 0xd0, 0xac, 0xd9, 0xa0, 0x49, 0x32, 0x5b }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 80 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0x7c, 0x92, 0x1b, 0x91, 0x82, 0x9e, 0xf3, 0xb1, 0x3c, 0xa5, 0x41, 0xee, 0x11, 0x30, 0xa6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 81 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x65, 0xeb, 0x6b, 0x6e, 0xa3, 0x83, 0xe1, 0x09, 0xac, 0xcc, 0xe8, 0x32, 0x6b, 0x03, 0x93 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 82 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xa5, 0x47, 0xf7, 0x43, 0x9e, 0xdc, 0x3e, 0x25, 0x5c, 0x0f, 0x4d, 0x49, 0xaa, 0x89, 0x90 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 83 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xe6, 0x52, 0x61, 0x4c, 0x93, 0x00, 0xf3, 0x78, 0x16, 0xb1, 0xf9, 0xfd, 0x0c, 0x87, 0xf9 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 84 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0x95, 0x4f, 0x0b, 0x46, 0x97, 0x77, 0x6f, 0x44, 0x49, 0x4f, 0xe4, 0x58, 0xd8, 0x14, 0xed }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 85 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0x8d, 0x9a, 0xb6, 0xc2, 0x76, 0x17, 0x23, 0xfe, 0x42, 0xf8, 0xbb, 0x50, 0x6c, 0xbc, 0xf7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 86 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x7e, 0x19, 0x32, 0x67, 0x9f, 0xdd, 0x99, 0x74, 0x2a, 0xab, 0x04, 0xaa, 0x0d, 0x5a, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 87 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4c, 0x6a, 0x1c, 0x83, 0xe5, 0x68, 0xcd, 0x10, 0xf2, 0x7c, 0x2d, 0x73, 0xde, 0xd1, 0x9c, 0x28 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 88 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xec, 0xbe, 0x61, 0x77, 0xe6, 0x74, 0xc9, 0x8d, 0xe4, 0x12, 0x41, 0x3f, 0x7a, 0xc9, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 89 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x68, 0x4a, 0x2a, 0xc5, 0x5f, 0xe1, 0xec, 0x2b, 0x8e, 0xbd, 0x56, 0x22, 0x52, 0x0b, 0x73 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 90 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x72, 0xf9, 0xa7, 0x98, 0x86, 0x07, 0xca, 0x79, 0x70, 0x77, 0x95, 0x99, 0x10, 0x35, 0xe6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 91 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0xaf, 0xf0, 0x89, 0x87, 0x8b, 0xf3, 0x35, 0x2f, 0x8d, 0xf1, 0x72, 0xa3, 0xae, 0x47, 0xd8 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 92 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x65, 0xc0, 0x52, 0x6c, 0xbe, 0x40, 0x16, 0x1b, 0x80, 0x19, 0xa2, 0xa3, 0x17, 0x1a, 0xbd, 0x23 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 93 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x7b, 0xe0, 0xbe, 0x33, 0xb4, 0xe3, 0xe3, 0x10, 0xb4, 0xaa, 0xbd, 0xa1, 0x73, 0xf8, 0x4f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 94 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0x02, 0xe9, 0xaa, 0x6f, 0x69, 0xde, 0x65, 0x04, 0xda, 0x8d, 0x20, 0xc4, 0xfc, 0xaa, 0x2f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 95 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0x3c, 0x1f, 0x4a, 0xf3, 0x13, 0xad, 0x8c, 0x2c, 0xe6, 0x48, 0xb2, 0xe7, 0x1f, 0xb6, 0xe1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 96 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0xfc, 0x62, 0x6d, 0x30, 0x20, 0x3d, 0xcd, 0xb0, 0x01, 0x9f, 0xb8, 0x0f, 0x72, 0x6c, 0xf4 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 97 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 16,
	  { 0x76, 0xda, 0x1f, 0xbe, 0x3a, 0x50, 0x72, 0x8c, 0x50, 0xfd, 0x2e, 0x62, 0x1b, 0x5a, 0xd8, 0x85 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 98 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x2e, 0xb8, 0xbe, 0x35, 0xf4, 0x42, 0xfb, 0x52, 0x66, 0x8e, 0x16, 0xa5, 0x91, 0xd1, 0xd6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 99 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0x56, 0xf9, 0xec, 0xf5, 0xfe, 0x27, 0xec, 0x3e, 0x4a, 0x73, 0xd0, 0x0c, 0x28, 0x2f, 0xb3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 100 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xa8, 0x20, 0x9d, 0x63, 0x27, 0x4c, 0xd9, 0xa2, 0x9b, 0xb7, 0x4b, 0xcd, 0x77, 0x68, 0x3a }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 101 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0xbf, 0x5d, 0xce, 0x14, 0xbb, 0x7d, 0xd7, 0x3a, 0x8e, 0x36, 0x11, 0xde, 0x7c, 0xe0, 0x26 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 102 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0x84, 0x99, 0x39, 0xa5, 0xd2, 0x93, 0x99, 0xf3, 0x44, 0xc4, 0xa0, 0xec, 0xa8, 0xa5, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 103 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0x3c, 0x0a, 0x94, 0xd5, 0x9b, 0xec, 0xe9, 0x88, 0x35, 0xda, 0x7a, 0xa4, 0xf0, 0x7c, 0xa2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 104 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 16,
	  { 0x63, 0x91, 0x9e, 0xd4, 0xce, 0x10, 0x19, 0x64, 0x38, 0xb6, 0xad, 0x09, 0xd9, 0x9c, 0xd7, 0x95 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 105 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 16,
	  { 0x76, 0x78, 0xf3, 0xa8, 0x33, 0xf1, 0x9f, 0xea, 0x95, 0xf3, 0xc6, 0x02, 0x9e, 0x2b, 0xc6, 0x10 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 106 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 16,
	  { 0x3a, 0xa4, 0x26, 0x83, 0x10, 0x67, 0xd3, 0x6b, 0x92, 0xbe, 0x7c, 0x5f, 0x81, 0xc1, 0x3c, 0x56 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 107 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 16,
	  { 0x92, 0x72, 0xe2, 0xd2, 0xcd, 0xd1, 0x10, 0x50, 0x99, 0x8c, 0x84, 0x50, 0x77, 0xa3, 0x0e, 0xa0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 108 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 16,
	  { 0x08, 0x8c, 0x4b, 0x53, 0xf5, 0xec, 0x0f, 0xf8, 0x14, 0xc1, 0x9a, 0xda, 0xe7, 0xf6, 0x24, 0x6c }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 109 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 16,
	  { 0x40, 0x10, 0xa5, 0xe4, 0x01, 0xfd, 0xf0, 0xa0, 0x35, 0x4d, 0xdb, 0xcc, 0x0d, 0x01, 0x2b, 0x17 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 110 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 16,
	  { 0xa8, 0x7a, 0x38, 0x57, 0x36, 0xc0, 0xa6, 0x18, 0x9b, 0xd6, 0x58, 0x9b, 0xd8, 0x44, 0x5a, 0x93 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 111 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 16,
	  { 0x54, 0x5f, 0x2b, 0x83, 0xd9, 0x61, 0x6d, 0xcc, 0xf6, 0x0f, 0xa9, 0x83, 0x0e, 0x9c, 0xd2, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 112 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 16,
	  { 0x4b, 0x70, 0x6f, 0x7f, 0x92, 0x40, 0x63, 0x52, 0x39, 0x40, 0x37, 0xa6, 0xd4, 0xf4, 0x68, 0x8d }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 113 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 16,
	  { 0xb7, 0x97, 0x2b, 0x39, 0x41, 0xc4, 0x4b, 0x90, 0xaf, 0xa7, 0xb2, 0x64, 0xbf, 0xba, 0x73, 0x87 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 114 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 16,
	  { 0x6f, 0x45, 0x73, 0x2c, 0xf1, 0x08, 0x81, 0x54, 0x6f, 0x0f, 0xd2, 0x38, 0x96, 0xd2, 0xbb, 0x60 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 115 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 16,
	  { 0x2e, 0x35, 0x79, 0xca, 0x15, 0xaf, 0x27, 0xf6, 0x4b, 0x3c, 0x95, 0x5a, 0x5b, 0xfc, 0x30, 0xba }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 116 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 16,
	  { 0x34, 0xa2, 0xc5, 0xa9, 0x1a, 0xe2, 0xae, 0xc9, 0x9b, 0x7d, 0x1b, 0x5f, 0xa6, 0x78, 0x04, 0x47 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 117 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 16,
	  { 0xa4, 0xd6, 0x61, 0x6b, 0xd0, 0x4f, 0x87, 0x33, 0x5b, 0x0e, 0x53, 0x35, 0x12, 0x27, 0xa9, 0xee }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 118 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 16,
	  { 0x7f, 0x69, 0x2b, 0x03, 0x94, 0x58, 0x67, 0xd1, 0x61, 0x79, 0xa8, 0xce, 0xfc, 0x83, 0xea, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 119 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 16,
	  { 0x3b, 0xd1, 0x41, 0xee, 0x84, 0xa0, 0xe6, 0x41, 0x4a, 0x26, 0xe7, 0xa4, 0xf2, 0x81, 0xf8, 0xa2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 120 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 16,
	  { 0xd1, 0x78, 0x8f, 0x57, 0x2d, 0x98, 0xb2, 0xb1, 0x6e, 0xc5, 0xd5, 0xf3, 0x92, 0x2b, 0x99, 0xbc }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 121 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 16,
	  { 0x08, 0x33, 0xff, 0x6f, 0x61, 0xd9, 0x8a, 0x57, 0xb2, 0x88, 0xe8, 0xc3, 0x58, 0x6b, 0x85, 0xa6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 122 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 16,
	  { 0x85, 0x68, 0x26, 0x17, 0x97, 0xde, 0x17, 0x6b, 0xf0, 0xb4, 0x3b, 0xec, 0xc6, 0x28, 0x5a, 0xfb }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 123 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 16,
	  { 0xf9, 0xb0, 0xfd, 0xa0, 0xc4, 0xa8, 0x98, 0xf5, 0xb9, 0xe6, 0xf6, 0x61, 0xc4, 0xce, 0x4d, 0x07 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 124 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 16,
	  { 0x8a, 0xde, 0x89, 0x59, 0x13, 0x68, 0x5c, 0x67, 0xc5, 0x26, 0x9f, 0x8a, 0xae, 0x42, 0x98, 0x3e }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 125 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 16,
	  { 0x39, 0xbd, 0xe6, 0x7d, 0x5c, 0x8e, 0xd8, 0xa8, 0xb1, 0xc3, 0x7e, 0xb8, 0xfa, 0x9f, 0x5a, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 126 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 16,
	  { 0x5c, 0x00, 0x5e, 0x72, 0xc1, 0x41, 0x8c, 0x44, 0xf5, 0x69, 0xf2, 0xea, 0x33, 0xba, 0x54, 0xf3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 127 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 16,
	  { 0x3f, 0x5b, 0x8c, 0xc9, 0xea, 0x85, 0x5a, 0x0a, 0xfa, 0x73, 0x47, 0xd2, 0x3e, 0x8d, 0x66, 0x4e }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 0 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xd7, 0x8e, 0x72, 0x6c, 0x1e, 0xc0, 0x2b, 0x7e, 0xbf, 0xe9, 0x2b, 0x23, 0xd9, 0xec, 0x34 }, 16,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 1 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0xe5, 0x93, 0x9c, 0x8e, 0xfd, 0xf2, 0xf0, 0x4e, 0x60, 0xb9, 0xfe, 0x71, 0x17, 0xb2, 0xc2 }, 16,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 2 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0x31, 0xd4, 0xd7, 0x4f, 0x5d, 0xcb, 0xf3, 0x9d, 0xaa, 0xf8, 0xca, 0x3a, 0xf6, 0xe5, 0x27 }, 16,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 3 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0xd9, 0xfd, 0x5c, 0xc4, 0xf0, 0x74, 0x41, 0x72, 0x7d, 0xf0, 0xf3, 0x3e, 0x40, 0x1a, 0x36 }, 16,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 4 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x30, 0xcc, 0xdb, 0x04, 0x46, 0x46, 0xd7, 0xe1, 0xf3, 0xcc, 0xea, 0x3d, 0xca, 0x08, 0xb8, 0xc0 }, 16,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 5 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x16, 0xae, 0x4c, 0xe5, 0x04, 0x2a, 0x67, 0xee, 0x8e, 0x17, 0x7b, 0x7c, 0x58, 0x7e, 0xcc, 0x82 }, 16,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 6 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0xda, 0x0b, 0xb1, 0x1a, 0x23, 0x85, 0x5d, 0x9c, 0x5c, 0xb1, 0xb4, 0xc6, 0x41, 0x2e, 0x0a }, 16,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 7 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x4f, 0x1a, 0xa5, 0x30, 0x96, 0x7d, 0x67, 0x32, 0xce, 0x47, 0x15, 0xeb, 0x0e, 0xe2, 0x4b }, 16,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 8 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x17, 0x38, 0x25, 0x26, 0x21, 0xdd, 0x18, 0x0a, 0x34, 0xf3, 0x45, 0x5b, 0x4b, 0xaa, 0x2f }, 16,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 9 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xe2, 0xb5, 0x08, 0xdb, 0x7f, 0xd8, 0x92, 0x34, 0xca, 0xf7, 0x93, 0x9e, 0xe5, 0x62, 0x1a }, 16,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 10 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0x49, 0x9c, 0x25, 0x1f, 0x84, 0x42, 0xee, 0x13, 0xf0, 0x93, 0x3b, 0x68, 0x8f, 0xcd, 0x19 }, 16,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 11 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x96, 0x51, 0x35, 0xf8, 0xa8, 0x1f, 0x25, 0xc9, 0xd6, 0x30, 0xb1, 0x75, 0x02, 0xf6, 0x8e, 0x53 }, 16,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 12 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0x87, 0x14, 0x5a, 0x01, 0xad, 0x1c, 0x6c, 0xed, 0xe9, 0x95, 0xea, 0x36, 0x70, 0x45, 0x4f }, 16,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 13 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xae, 0x3b, 0x10, 0xa0, 0xc8, 0xca, 0x6d, 0x1d, 0x3b, 0x0f, 0xa6, 0x1e, 0x56, 0xb0, 0xb2 }, 16,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 14 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x64, 0xb4, 0xd6, 0x29, 0x81, 0x0f, 0xda, 0x6b, 0xaf, 0xdf, 0x08, 0xf3, 0xb0, 0xd8, 0xd2, 0xc5 }, 16,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 15 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd7, 0xe5, 0xdb, 0xd3, 0x32, 0x45, 0x95, 0xf8, 0xfd, 0xc7, 0xd7, 0xc5, 0x71, 0xda, 0x6c, 0x2a }, 16,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 16 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0xf7, 0x23, 0x75, 0x26, 0x4e, 0x16, 0x7f, 0xca, 0x9d, 0xe2, 0xc1, 0x52, 0x7d, 0x96, 0x06 }, 16,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 17 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0xe7, 0x9d, 0xd4, 0xf4, 0x01, 0xff, 0x9b, 0x7e, 0xa9, 0x45, 0xd8, 0x66, 0x66, 0xc1, 0x3b }, 16,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 18 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0x35, 0xce, 0xa2, 0x79, 0x99, 0x40, 0xb4, 0x0d, 0xb3, 0xf8, 0x19, 0xcb, 0x94, 0xc0, 0x8b }, 16,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 19 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x69, 0x41, 0xcb, 0x6b, 0x3e, 0x08, 0xc2, 0xb7, 0xaf, 0xa5, 0x81, 0xeb, 0xdd, 0x60, 0x7b, 0x87 }, 16,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 20 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0x20, 0xf4, 0x39, 0xf6, 0xbb, 0x09, 0x7b, 0x29, 0xb8, 0xbd, 0x6d, 0x99, 0xaa, 0xd7, 0x99 }, 16,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 21 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0x5d, 0x01, 0xf0, 0x58, 0xe5, 0x65, 0xf7, 0x7a, 0xe8, 0x63, 0x78, 0xbd, 0x2c, 0x49, 0xb3 }, 16,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 22 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0xb5, 0xfd, 0x98, 0x19, 0x0e, 0xf4, 0x5f, 0xbb, 0x43, 0x01, 0x43, 0x8d, 0x09, 0x59, 0x50 }, 16,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 23 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x13, 0x00, 0x1f, 0xf5, 0xd9, 0x98, 0x06, 0xef, 0xd2, 0x5d, 0xa3, 0x4f, 0x56, 0xbe, 0x85, 0x4b }, 16,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 24 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x59, 0x4c, 0x60, 0xf5, 0xc8, 0x27, 0x7a, 0x51, 0x13, 0x67, 0x7f, 0x94, 0x20, 0x8d, 0x82 }, 16,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 25 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0xc0, 0xfc, 0x18, 0x18, 0xe4, 0xaa, 0x46, 0xbd, 0x2e, 0x39, 0xd6, 0x38, 0xf8, 0x9e, 0x05 }, 16,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 26 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x02, 0x3e, 0xe9, 0xc3, 0xfd, 0xc4, 0x5a, 0x01, 0x9b, 0x4e, 0x98, 0x5c, 0x7e, 0x1a, 0x54 }, 16,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 27 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xf4, 0x01, 0x82, 0xab, 0x46, 0x62, 0xf3, 0x02, 0x3b, 0xae, 0xc1, 0xee, 0x79, 0x6b, 0x57 }, 16,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 28 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xeb, 0xba, 0xd7, 0x30, 0x36, 0x49, 0xb4, 0x19, 0x4a, 0x69, 0x45, 0xc6, 0xcc, 0x36, 0x94 }, 16,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 29 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa2, 0x12, 0x4b, 0xea, 0x53, 0xec, 0x28, 0x34, 0x27, 0x9b, 0xed, 0x7f, 0x7e, 0xb0, 0xf9, 0x38 }, 16,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 30 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb9, 0xfb, 0x43, 0x99, 0xfa, 0x4f, 0xac, 0xc7, 0x30, 0x9e, 0x14, 0xec, 0x98, 0x36, 0x0b, 0x0a }, 16,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 31 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x62, 0x77, 0x43, 0x74, 0x20, 0xc5, 0xd6, 0x34, 0xf7, 0x15, 0xae, 0xa8, 0x1a, 0x91, 0x32 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 32 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x17, 0x1a, 0x0e, 0x1b, 0x2d, 0xd4, 0x24, 0xf0, 0xe0, 0x89, 0xaf, 0x2c, 0x4c, 0x10, 0xf3, 0x2f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 33 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xad, 0xbe, 0x40, 0x2d, 0x1b, 0x20, 0x8f, 0xe7, 0x35, 0xed, 0xce, 0x00, 0xae, 0xe7, 0xce }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 34 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xb0, 0x2f, 0xf9, 0x29, 0xa1, 0x48, 0x5a, 0xf6, 0xf5, 0xc6, 0xd6, 0x55, 0x8b, 0xaa, 0x0f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 35 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x2f, 0xaa, 0xcc, 0x9b, 0xf4, 0x35, 0x08, 0xbf, 0x8f, 0xa8, 0x61, 0x3c, 0xa7, 0x5d, 0xea }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 36 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcb, 0x2b, 0xf8, 0x28, 0x0f, 0x3f, 0x97, 0x42, 0xc7, 0xed, 0x51, 0x3f, 0xe8, 0x02, 0x62, 0x9c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 37 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0x5a, 0x41, 0xee, 0x44, 0x2f, 0xa9, 0x92, 0xa6, 0xe3, 0x23, 0x98, 0x6d, 0xed, 0x3f, 0x68 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 38 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0x1e, 0x99, 0xcf, 0x4f, 0x0f, 0x77, 0xce, 0xa8, 0x36, 0xe1, 0x1a, 0x2f, 0xe7, 0x5f, 0xb1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 39 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x95, 0xe3, 0xa0, 0xca, 0x90, 0x79, 0xe6, 0x46, 0x33, 0x1d, 0xf8, 0xb4, 0xe7, 0x0d, 0x2c, 0xd6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 40 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4a, 0xfe, 0x7f, 0x12, 0x0c, 0xe7, 0x61, 0x3f, 0x74, 0xfc, 0x12, 0xa0, 0x1a, 0x82, 0x80, 0x73 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 41 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x7f, 0x00, 0x0e, 0x75, 0xe2, 0xc8, 0xb9, 0xd4, 0x79, 0xbe, 0xed, 0x91, 0x3f, 0xe6, 0x78 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 42 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0x83, 0x0c, 0x8e, 0x7a, 0xae, 0xfe, 0x2d, 0x30, 0x31, 0x0e, 0xf3, 0x81, 0xcb, 0xf6, 0x91 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 43 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x19, 0x1a, 0xa0, 0xf2, 0xc8, 0x57, 0x01, 0x44, 0xf3, 0x86, 0x57, 0xea, 0x40, 0x85, 0xeb, 0xe5 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 44 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x85, 0x06, 0x2c, 0x2c, 0x90, 0x9f, 0x15, 0xd9, 0x26, 0x9b, 0x6c, 0x18, 0xce, 0x99, 0xc4, 0xf0 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 45 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x80, 0x34, 0xdc, 0x9e, 0x41, 0xb5, 0xa5, 0x60, 0xed, 0x23, 0x9e, 0xea, 0xb1, 0xbc, 0x78 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 46 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0xf9, 0x3a, 0x4c, 0xe5, 0xab, 0x6d, 0x5d, 0x56, 0xf1, 0xb9, 0x3c, 0xf1, 0x99, 0x11, 0xc1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 47 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x31, 0x12, 0xbc, 0xb0, 0xc1, 0xdc, 0xc7, 0x49, 0xd7, 0x99, 0x74, 0x36, 0x91, 0xbf, 0x82 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 48 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0xc5, 0x5b, 0xd7, 0x5c, 0x7f, 0x9c, 0x88, 0x19, 0x89, 0xd3, 0xec, 0x19, 0x11, 0xc0, 0xd4 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 49 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xea, 0x2e, 0x6b, 0x5e, 0xf1, 0x82, 0xb7, 0xdf, 0xf3, 0x62, 0x9a, 0xbd, 0x6a, 0x12, 0x04, 0x5f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 50 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0x32, 0x23, 0x27, 0xe0, 0x17, 0x80, 0xb1, 0x73, 0x97, 0xf2, 0x40, 0x87, 0xf8, 0xcc, 0x6f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 51 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0xca, 0xcb, 0x5c, 0xd1, 0x16, 0x92, 0xc3, 0x73, 0xb2, 0x41, 0x17, 0x68, 0x14, 0x9e, 0xe7 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 52 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x8e, 0x3d, 0xbb, 0xca, 0x57, 0x78, 0x60, 0xda, 0xb6, 0xb8, 0x0d, 0xa3, 0x13, 0x92, 0x56 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 53 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0xb6, 0x1c, 0x37, 0xbf, 0x32, 0x8e, 0xcc, 0xa8, 0xd7, 0x43, 0x26, 0x5a, 0x3d, 0x42, 0x5c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 54 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0xd9, 0x9c, 0x6b, 0xcc, 0x1f, 0x06, 0xfd, 0xa8, 0xe2, 0x7e, 0x8a, 0xe3, 0xf1, 0xcc, 0xc7 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 55 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0xfd, 0x4b, 0x91, 0xc7, 0x01, 0xfd, 0x6b, 0x61, 0xb7, 0xf9, 0x97, 0x82, 0x9d, 0x66, 0x3b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 56 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x00, 0x5d, 0x52, 0xf2, 0x5f, 0x16, 0xbd, 0xc9, 0x54, 0x5a, 0x87, 0x6a, 0x63, 0x49, 0x0a }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 57 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x4d, 0x35, 0x4f, 0x02, 0xbb, 0x5a, 0x5e, 0x47, 0xd3, 0x96, 0x66, 0x86, 0x7f, 0x24, 0x6a }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 58 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x51, 0xb8, 0xd6, 0xe1, 0xe1, 0xa0, 0xeb, 0xb1, 0x55, 0xfb, 0xbf, 0x6e, 0x7b, 0x7d, 0xc3 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 59 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x68, 0x98, 0xd4, 0xf4, 0x2f, 0xa7, 0xba, 0x6a, 0x10, 0xac, 0x05, 0xe8, 0x7b, 0x9f, 0x20, 0x80 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 60 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb6, 0x11, 0x29, 0x5e, 0x73, 0x9c, 0xa7, 0xd9, 0xb5, 0x0f, 0x8e, 0x4c, 0x0e, 0x75, 0x4a, 0x3f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 61 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0x33, 0xfc, 0x7d, 0x8a, 0xbe, 0x3c, 0xa1, 0x93, 0x67, 0x59, 0xf8, 0xf5, 0xde, 0xaf, 0x20 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 62 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0x5e, 0x0f, 0x56, 0x6d, 0xc9, 0x6c, 0x29, 0x8f, 0x0c, 0x12, 0x63, 0x75, 0x39, 0xb2, 0x5c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 63 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x07, 0xc3, 0xe7, 0x98, 0x5f, 0xe0, 0xf5, 0xa5, 0x0e, 0x2c, 0xdb, 0x25, 0xc5, 0x10, 0x9e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 64 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x41, 0xf9, 0x92, 0xa8, 0x56, 0xfb, 0x27, 0x8b, 0x38, 0x9a, 0x62, 0xf5, 0xd2, 0x74, 0xd7, 0xe9 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 65 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x10, 0xd3, 0xed, 0x7a, 0x6f, 0xe1, 0x5a, 0xb4, 0xd9, 0x1a, 0xcb, 0xc7, 0xd0, 0x76, 0x7a, 0xb1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 66 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0xfe, 0xec, 0xd4, 0x5b, 0x2e, 0x67, 0x59, 0x73, 0xac, 0x33, 0xbf, 0x0c, 0x54, 0x24, 0xfc }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 67 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0x80, 0xcb, 0x39, 0x55, 0xba, 0x62, 0xd0, 0x9e, 0xea, 0x66, 0x8f, 0x7c, 0x70, 0x88, 0x17 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 68 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x66, 0x40, 0x40, 0x33, 0xd6, 0xb7, 0x2b, 0x60, 0x93, 0x54, 0xd5, 0x49, 0x6e, 0x7e, 0xb5, 0x11 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 69 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x31, 0x7a, 0x22, 0x0a, 0x7d, 0x70, 0x0d, 0xa2, 0xb1, 0xe0, 0x75, 0xb0, 0x02, 0x66, 0xe1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 70 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xab, 0x3b, 0x89, 0x54, 0x22, 0x33, 0xf1, 0x27, 0x1b, 0xf8, 0xfd, 0x0c, 0x0f, 0x40, 0x35, 0x45 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 71 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd9, 0x3e, 0xae, 0x96, 0x6f, 0xac, 0x46, 0xdc, 0xa9, 0x27, 0xd6, 0xb1, 0x14, 0xfa, 0x3f, 0x9e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 72 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0xde, 0xc5, 0x21, 0x31, 0x65, 0x03, 0xd9, 0xd5, 0xee, 0x65, 0xdf, 0x3e, 0xa9, 0x4d, 0xdf }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 73 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xee, 0xf4, 0x56, 0x43, 0x1d, 0xea, 0x8b, 0x4a, 0xcf, 0x83, 0xbd, 0xae, 0x37, 0x17, 0xf7, 0x5f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 74 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x06, 0xf2, 0x51, 0x9a, 0x2f, 0xaf, 0xaa, 0x59, 0x6b, 0xfe, 0xf5, 0xcf, 0xa1, 0x5c, 0x21, 0xb9 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 75 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0x1a, 0x7e, 0xac, 0x7e, 0x2f, 0xe8, 0x09, 0xe4, 0xaa, 0x8d, 0x0d, 0x70, 0x12, 0x53, 0x1a }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 76 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0xff, 0xc1, 0x6e, 0x4c, 0x49, 0xb2, 0x68, 0xa2, 0x0f, 0x8d, 0x96, 0xa6, 0x0b, 0x40, 0x58 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 77 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0x86, 0xf9, 0x28, 0x19, 0x99, 0xc5, 0xbb, 0x3b, 0x3e, 0x88, 0x62, 0xe2, 0xf7, 0xc9, 0x88 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 78 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0x3b, 0xf9, 0x0d, 0x61, 0xbe, 0xef, 0x39, 0xf4, 0x8d, 0xd6, 0x25, 0xfc, 0xef, 0x13, 0x61 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 79 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4d, 0x37, 0xc8, 0x50, 0x64, 0x45, 0x63, 0xc6, 0x9f, 0xd0, 0xac, 0xd9, 0xa0, 0x49, 0x32, 0x5b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 80 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb8, 0x7c, 0x92, 0x1b, 0x91, 0x82, 0x9e, 0xf3, 0xb1, 0x3c, 0xa5, 0x41, 0xee, 0x11, 0x30, 0xa6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 81 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x65, 0xeb, 0x6b, 0x6e, 0xa3, 0x83, 0xe1, 0x09, 0xac, 0xcc, 0xe8, 0x32, 0x6b, 0x03, 0x93 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 82 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xa5, 0x47, 0xf7, 0x43, 0x9e, 0xdc, 0x3e, 0x25, 0x5c, 0x0f, 0x4d, 0x49, 0xaa, 0x89, 0x90 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 83 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xe6, 0x52, 0x61, 0x4c, 0x93, 0x00, 0xf3, 0x78, 0x16, 0xb1, 0xf9, 0xfd, 0x0c, 0x87, 0xf9 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 84 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x14, 0x95, 0x4f, 0x0b, 0x46, 0x97, 0x77, 0x6f, 0x44, 0x49, 0x4f, 0xe4, 0x58, 0xd8, 0x14, 0xed }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 85 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0x8d, 0x9a, 0xb6, 0xc2, 0x76, 0x17, 0x23, 0xfe, 0x42, 0xf8, 0xbb, 0x50, 0x6c, 0xbc, 0xf7 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 86 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x7e, 0x19, 0x32, 0x67, 0x9f, 0xdd, 0x99, 0x74, 0x2a, 0xab, 0x04, 0xaa, 0x0d, 0x5a, 0x80 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 87 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4c, 0x6a, 0x1c, 0x83, 0xe5, 0x68, 0xcd, 0x10, 0xf2, 0x7c, 0x2d, 0x73, 0xde, 0xd1, 0x9c, 0x28 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 88 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0xec, 0xbe, 0x61, 0x77, 0xe6, 0x74, 0xc9, 0x8d, 0xe4, 0x12, 0x41, 0x3f, 0x7a, 0xc9, 0x15 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 89 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x68, 0x4a, 0x2a, 0xc5, 0x5f, 0xe1, 0xec, 0x2b, 0x8e, 0xbd, 0x56, 0x22, 0x52, 0x0b, 0x73 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 90 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x72, 0xf9, 0xa7, 0x98, 0x86, 0x07, 0xca, 0x79, 0x70, 0x77, 0x95, 0x99, 0x10, 0x35, 0xe6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 91 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x56, 0xaf, 0xf0, 0x89, 0x87, 0x8b, 0xf3, 0x35, 0x2f, 0x8d, 0xf1, 0x72, 0xa3, 0xae, 0x47, 0xd8 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 92 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x65, 0xc0, 0x52, 0x6c, 0xbe, 0x40, 0x16, 0x1b, 0x80, 0x19, 0xa2, 0xa3, 0x17, 0x1a, 0xbd, 0x23 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 93 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x7b, 0xe0, 0xbe, 0x33, 0xb4, 0xe3, 0xe3, 0x10, 0xb4, 0xaa, 0xbd, 0xa1, 0x73, 0xf8, 0x4f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 94 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x94, 0x02, 0xe9, 0xaa, 0x6f, 0x69, 0xde, 0x65, 0x04, 0xda, 0x8d, 0x20, 0xc4, 0xfc, 0xaa, 0x2f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 95 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0x3c, 0x1f, 0x4a, 0xf3, 0x13, 0xad, 0x8c, 0x2c, 0xe6, 0x48, 0xb2, 0xe7, 0x1f, 0xb6, 0xe1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 96 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1f, 0xfc, 0x62, 0x6d, 0x30, 0x20, 0x3d, 0xcd, 0xb0, 0x01, 0x9f, 0xb8, 0x0f, 0x72, 0x6c, 0xf4 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 97 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x76, 0xda, 0x1f, 0xbe, 0x3a, 0x50, 0x72, 0x8c, 0x50, 0xfd, 0x2e, 0x62, 0x1b, 0x5a, 0xd8, 0x85 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 98 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x2e, 0xb8, 0xbe, 0x35, 0xf4, 0x42, 0xfb, 0x52, 0x66, 0x8e, 0x16, 0xa5, 0x91, 0xd1, 0xd6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 99 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe6, 0x56, 0xf9, 0xec, 0xf5, 0xfe, 0x27, 0xec, 0x3e, 0x4a, 0x73, 0xd0, 0x0c, 0x28, 0x2f, 0xb3 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 100 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xa8, 0x20, 0x9d, 0x63, 0x27, 0x4c, 0xd9, 0xa2, 0x9b, 0xb7, 0x4b, 0xcd, 0x77, 0x68, 0x3a }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 101 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x79, 0xbf, 0x5d, 0xce, 0x14, 0xbb, 0x7d, 0xd7, 0x3a, 0x8e, 0x36, 0x11, 0xde, 0x7c, 0xe0, 0x26 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 102 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0x84, 0x99, 0x39, 0xa5, 0xd2, 0x93, 0x99, 0xf3, 0x44, 0xc4, 0xa0, 0xec, 0xa8, 0xa5, 0x76 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 103 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xed, 0x3c, 0x0a, 0x94, 0xd5, 0x9b, 0xec, 0xe9, 0x88, 0x35, 0xda, 0x7a, 0xa4, 0xf0, 0x7c, 0xa2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 104 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x63, 0x91, 0x9e, 0xd4, 0xce, 0x10, 0x19, 0x64, 0x38, 0xb6, 0xad, 0x09, 0xd9, 0x9c, 0xd7, 0x95 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 105 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x76, 0x78, 0xf3, 0xa8, 0x33, 0xf1, 0x9f, 0xea, 0x95, 0xf3, 0xc6, 0x02, 0x9e, 0x2b, 0xc6, 0x10 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 106 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0xa4, 0x26, 0x83, 0x10, 0x67, 0xd3, 0x6b, 0x92, 0xbe, 0x7c, 0x5f, 0x81, 0xc1, 0x3c, 0x56 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 107 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x92, 0x72, 0xe2, 0xd2, 0xcd, 0xd1, 0x10, 0x50, 0x99, 0x8c, 0x84, 0x50, 0x77, 0xa3, 0x0e, 0xa0 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 108 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x8c, 0x4b, 0x53, 0xf5, 0xec, 0x0f, 0xf8, 0x14, 0xc1, 0x9a, 0xda, 0xe7, 0xf6, 0x24, 0x6c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 109 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x10, 0xa5, 0xe4, 0x01, 0xfd, 0xf0, 0xa0, 0x35, 0x4d, 0xdb, 0xcc, 0x0d, 0x01, 0x2b, 0x17 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 110 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0x7a, 0x38, 0x57, 0x36, 0xc0, 0xa6, 0x18, 0x9b, 0xd6, 0x58, 0x9b, 0xd8, 0x44, 0x5a, 0x93 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 111 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x54, 0x5f, 0x2b, 0x83, 0xd9, 0x61, 0x6d, 0xcc, 0xf6, 0x0f, 0xa9, 0x83, 0x0e, 0x9c, 0xd2, 0x87 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 112 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4b, 0x70, 0x6f, 0x7f, 0x92, 0x40, 0x63, 0x52, 0x39, 0x40, 0x37, 0xa6, 0xd4, 0xf4, 0x68, 0x8d }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 113 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0x97, 0x2b, 0x39, 0x41, 0xc4, 0x4b, 0x90, 0xaf, 0xa7, 0xb2, 0x64, 0xbf, 0xba, 0x73, 0x87 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 114 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6f, 0x45, 0x73, 0x2c, 0xf1, 0x08, 0x81, 0x54, 0x6f, 0x0f, 0xd2, 0x38, 0x96, 0xd2, 0xbb, 0x60 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 115 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x35, 0x79, 0xca, 0x15, 0xaf, 0x27, 0xf6, 0x4b, 0x3c, 0x95, 0x5a, 0x5b, 0xfc, 0x30, 0xba }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 116 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0xa2, 0xc5, 0xa9, 0x1a, 0xe2, 0xae, 0xc9, 0x9b, 0x7d, 0x1b, 0x5f, 0xa6, 0x78, 0x04, 0x47 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 117 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0xd6, 0x61, 0x6b, 0xd0, 0x4f, 0x87, 0x33, 0x5b, 0x0e, 0x53, 0x35, 0x12, 0x27, 0xa9, 0xee }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 118 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7f, 0x69, 0x2b, 0x03, 0x94, 0x58, 0x67, 0xd1, 0x61, 0x79, 0xa8, 0xce, 0xfc, 0x83, 0xea, 0x3f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 119 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3b, 0xd1, 0x41, 0xee, 0x84, 0xa0, 0xe6, 0x41, 0x4a, 0x26, 0xe7, 0xa4, 0xf2, 0x81, 0xf8, 0xa2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 120 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd1, 0x78, 0x8f, 0x57, 0x2d, 0x98, 0xb2, 0xb1, 0x6e, 0xc5, 0xd5, 0xf3, 0x92, 0x2b, 0x99, 0xbc }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 121 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x33, 0xff, 0x6f, 0x61, 0xd9, 0x8a, 0x57, 0xb2, 0x88, 0xe8, 0xc3, 0x58, 0x6b, 0x85, 0xa6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 122 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x85, 0x68, 0x26, 0x17, 0x97, 0xde, 0x17, 0x6b, 0xf0, 0xb4, 0x3b, 0xec, 0xc6, 0x28, 0x5a, 0xfb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 123 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf9, 0xb0, 0xfd, 0xa0, 0xc4, 0xa8, 0x98, 0xf5, 0xb9, 0xe6, 0xf6, 0x61, 0xc4, 0xce, 0x4d, 0x07 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 124 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8a, 0xde, 0x89, 0x59, 0x13, 0x68, 0x5c, 0x67, 0xc5, 0x26, 0x9f, 0x8a, 0xae, 0x42, 0x98, 0x3e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 125 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x39, 0xbd, 0xe6, 0x7d, 0x5c, 0x8e, 0xd8, 0xa8, 0xb1, 0xc3, 0x7e, 0xb8, 0xfa, 0x9f, 0x5a, 0xc0 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 126 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x5c, 0x00, 0x5e, 0x72, 0xc1, 0x41, 0x8c, 0x44, 0xf5, 0x69, 0xf2, 0xea, 0x33, 0xba, 0x54, 0xf3 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarTxt128.rsp test vector 127 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 128,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3f, 0x5b, 0x8c, 0xc9, 0xea, 0x85, 0x5a, 0x0a, 0xfa, 0x73, 0x47, 0xd2, 0x3e, 0x8d, 0x66, 0x4e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 16 },

	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0xd0, 0x25, 0x13, 0xe8, 0xd4, 0xdc, 0x98, 0x6b, 0x4a, 0xfe, 0x08, 0x7a, 0x60, 0xbd, 0x0c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xe1, 0xf8, 0xb7, 0xe3, 0x06, 0x27, 0xc1, 0xc4, 0x51, 0x9e, 0xad, 0xa4, 0x4b, 0xc4, 0x36 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x46, 0xb5, 0xf8, 0x7a, 0xf4, 0x46, 0xf5, 0x79, 0x6c, 0x1f, 0xee, 0x63, 0xa2, 0xda, 0x24 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2a, 0x56, 0x03, 0x64, 0xce, 0x52, 0x9e, 0xfc, 0x21, 0x78, 0x87, 0x79, 0x56, 0x8d, 0x55, 0x55 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xc1, 0x47, 0x18, 0x37, 0xaf, 0x44, 0x61, 0x53, 0xbc, 0xe5, 0x5d, 0x5b, 0xa7, 0x2a, 0x0a }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xce, 0x60, 0xbc, 0x52, 0x38, 0x62, 0x34, 0xf1, 0x58, 0xf8, 0x43, 0x41, 0xe5, 0x34, 0xcd, 0x9e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8c, 0x7c, 0x27, 0xff, 0x32, 0xbc, 0xf8, 0xdc, 0x2d, 0xc5, 0x7c, 0x90, 0xc2, 0x90, 0x39, 0x61 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0xbb, 0x6a, 0x7e, 0xc8, 0x44, 0x99, 0xe1, 0x66, 0xf9, 0x36, 0x00, 0x3d, 0x55, 0xa5, 0xbb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xc7, 0x72, 0xe5, 0xc6, 0x26, 0x31, 0xef, 0x66, 0x0e, 0xe1, 0xd5, 0x87, 0x7f, 0x6d, 0x1b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x0d, 0x7e, 0x5b, 0x64, 0xf3, 0x80, 0xa7, 0xe4, 0xea, 0x53, 0x87, 0xb5, 0xcd, 0x7f, 0x49 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0d, 0xc9, 0xa2, 0x61, 0x00, 0x37, 0x00, 0x9b, 0x69, 0x8f, 0x11, 0xbb, 0x7e, 0x86, 0xc8, 0x3e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x46, 0x61, 0x2c, 0x76, 0x6d, 0x18, 0x40, 0xc2, 0x26, 0x36, 0x4f, 0x1f, 0xa7, 0xed, 0x72 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x48, 0x80, 0xc7, 0xe0, 0x8f, 0x27, 0xbe, 0xfe, 0x78, 0x59, 0x07, 0x43, 0xc0, 0x5e, 0x69, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 13 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0x20, 0xce, 0x82, 0x9a, 0x26, 0x57, 0x7f, 0x0f, 0x48, 0x22, 0xc4, 0xec, 0xc8, 0x74, 0x01 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 14 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0x65, 0xe8, 0xac, 0xc1, 0x69, 0x75, 0x83, 0x19, 0xcb, 0x46, 0xdc, 0x7b, 0xcf, 0x3d, 0xca }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 15 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0x8f, 0x4b, 0xa4, 0xf0, 0x73, 0xdf, 0x4b, 0xaa, 0x11, 0x6d, 0x01, 0x1d, 0xc2, 0x4a, 0x28 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 16 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x78, 0xf6, 0x8c, 0x5d, 0xbf, 0x59, 0xe2, 0x11, 0xb3, 0xa6, 0x59, 0xa7, 0x31, 0x7d, 0x94 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 17 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x28, 0x3d, 0x3b, 0x06, 0x9d, 0x8e, 0xb9, 0xfb, 0x43, 0x2d, 0x74, 0xb9, 0x6c, 0xa7, 0x62, 0xb4 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 18 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0xe1, 0x84, 0x2e, 0x8a, 0x87, 0x86, 0x1c, 0x22, 0x1a, 0x50, 0x08, 0x83, 0x24, 0x5c, 0x51 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 19 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xaa, 0x27, 0x04, 0x71, 0x88, 0x1b, 0xe0, 0x70, 0xfb, 0x52, 0xc7, 0x06, 0x7c, 0xe7, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 20 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0xb0, 0xf4, 0x76, 0xd4, 0x84, 0xf4, 0x3f, 0x1a, 0xeb, 0x6e, 0xfa, 0x93, 0x61, 0xa8, 0xac }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 21 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x3a, 0x94, 0xf1, 0xc0, 0x52, 0xc5, 0x5c, 0x2d, 0x83, 0x59, 0xaf, 0xf2, 0x16, 0x3b, 0x4f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 22 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0xa0, 0x67, 0xb6, 0x04, 0xd5, 0x37, 0x3d, 0x8b, 0x0f, 0x2e, 0x05, 0xa0, 0x3b, 0x34, 0x1b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 23 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x87, 0x6e, 0xc8, 0x7f, 0x5a, 0x09, 0xbf, 0xea, 0x42, 0xc7, 0x7d, 0xa3, 0x0f, 0xd5, 0x0e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 24 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0c, 0xf3, 0xe9, 0xd3, 0xa4, 0x2b, 0xe5, 0xb8, 0x54, 0xca, 0x65, 0xb1, 0x3f, 0x35, 0xf4, 0x8d }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 25 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x62, 0xf6, 0xbb, 0xca, 0xb7, 0xc3, 0xe8, 0x21, 0xc9, 0x29, 0x0f, 0x08, 0x89, 0x2d, 0xda }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 26 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7f, 0x5e, 0x05, 0xbd, 0x20, 0x68, 0x73, 0x81, 0x96, 0xfe, 0xe7, 0x9a, 0xce, 0x7e, 0x3a, 0xec }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 27 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0x0e, 0x0d, 0x73, 0x32, 0x55, 0xcd, 0xa9, 0x2f, 0xb4, 0x6e, 0x84, 0x2f, 0xe5, 0x80, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 28 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0x5d, 0x5b, 0x1c, 0x4e, 0xa1, 0xb7, 0xa2, 0x2e, 0x55, 0x83, 0xac, 0x2e, 0x9e, 0xd8, 0xa7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 29 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xe5, 0x37, 0xe8, 0x9e, 0x84, 0x91, 0xe8, 0x66, 0x2a, 0xae, 0x3b, 0xc8, 0x09, 0x42, 0x1d }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 30 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x7d, 0xd3, 0xe9, 0xf1, 0x59, 0x8b, 0xfa, 0x73, 0xf7, 0x59, 0x73, 0xf7, 0xe9, 0x3b, 0x76 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 31 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x38, 0xd4, 0xf7, 0x45, 0x2a, 0xfe, 0xfc, 0xb7, 0xfc, 0x72, 0x12, 0x44, 0xe4, 0xb7, 0x2e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 32 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0b, 0xe2, 0xb1, 0x82, 0x52, 0xe7, 0x74, 0xdd, 0xa3, 0x0c, 0xdd, 0xa0, 0x2c, 0x69, 0x06, 0xe3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 33 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x69, 0x5e, 0x59, 0xc2, 0x03, 0x61, 0xd8, 0x26, 0x52, 0xd7, 0xd5, 0x8b, 0x6f, 0x11, 0xb2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 34 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x2d, 0x88, 0xd1, 0x3e, 0xae, 0x52, 0x08, 0x9a, 0xbd, 0x61, 0x43, 0xcf, 0xe3, 0x94, 0xe9 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 35 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x9b, 0xce, 0xb3, 0xb8, 0x23, 0xfe, 0xdd, 0x60, 0x2c, 0x30, 0x53, 0x45, 0x73, 0x4b, 0xd2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 36 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x7b, 0x1d, 0xbb, 0x0f, 0xfa, 0x40, 0xef, 0x7d, 0x95, 0xde, 0xf4, 0x21, 0x23, 0x3f, 0xae }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 37 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xa0, 0xc1, 0xd9, 0x33, 0x56, 0xd9, 0xeb, 0x8a, 0xa9, 0x52, 0x08, 0x4d, 0x75, 0xf9, 0x13 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 38 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xcb, 0xf9, 0xcb, 0x18, 0x6e, 0x27, 0x0d, 0xd7, 0xbd, 0xb0, 0xc2, 0x8f, 0xeb, 0xc5, 0x7d }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 39 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0x43, 0x37, 0xc3, 0x7c, 0x4e, 0x79, 0x0a, 0xb4, 0x57, 0x80, 0xbd, 0x9c, 0x36, 0x74, 0xa0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 40 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0x35, 0x58, 0xc1, 0x35, 0x25, 0x2f, 0xb9, 0xc9, 0xf3, 0x67, 0xed, 0x60, 0x94, 0x67, 0xa1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 41 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x72, 0xee, 0xae, 0xe4, 0x89, 0x9b, 0x44, 0x39, 0x14, 0xe5, 0xb3, 0xa5, 0x7f, 0xba, 0x92 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 42 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0x18, 0x65, 0xf9, 0x1b, 0xc5, 0x68, 0x68, 0xd0, 0x51, 0xe5, 0x2c, 0x9e, 0xfd, 0x59, 0xb7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 43 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe4, 0x77, 0x13, 0x18, 0xad, 0x7a, 0x63, 0xdd, 0x68, 0x0f, 0x6e, 0x58, 0x3b, 0x77, 0x47, 0xea }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 44 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0xe3, 0xd1, 0x94, 0x08, 0x8d, 0xc8, 0xd9, 0x7e, 0x9e, 0x6d, 0xb3, 0x74, 0x57, 0xea, 0xc5 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 45 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xff, 0x1e, 0xc9, 0xcc, 0xfb, 0xc3, 0x49, 0xe5, 0xd3, 0x56, 0xd0, 0x63, 0x69, 0x3a, 0xd6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 46 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0xc9, 0xe9, 0xa9, 0xbe, 0x8c, 0xc3, 0xf6, 0xfb, 0x2e, 0xa2, 0x40, 0x88, 0xe9, 0xbb, 0x19 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 47 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xe5, 0xab, 0x00, 0x3d, 0xc8, 0x72, 0x2e, 0x74, 0x90, 0x5d, 0x9a, 0x8f, 0xe3, 0xd3, 0x50 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 48 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x53, 0x39, 0x31, 0x95, 0x84, 0xb0, 0xa4, 0x12, 0x41, 0x28, 0x69, 0xd6, 0xc2, 0xea, 0xda }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 49 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0xd4, 0x96, 0x91, 0x81, 0x15, 0xd1, 0x4e, 0xd5, 0x38, 0x08, 0x52, 0x71, 0x6c, 0x88, 0x14 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 50 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x3a, 0xb2, 0xf2, 0xb4, 0xa3, 0x66, 0xa5, 0x7d, 0x58, 0x2a, 0x33, 0x93, 0x13, 0xc8, 0xb1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 51 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x33, 0x65, 0xa9, 0xff, 0xbe, 0x3b, 0x0c, 0xa6, 0x1e, 0x98, 0x50, 0x75, 0x54, 0x16, 0x8b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 52 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaf, 0xa9, 0x9c, 0x99, 0x7a, 0xc4, 0x78, 0xa0, 0xde, 0xa4, 0x11, 0x9c, 0x9e, 0x45, 0xf8, 0xb1 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 53 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x92, 0x16, 0x30, 0x9a, 0x78, 0x42, 0x43, 0x0b, 0x83, 0xff, 0xb9, 0x86, 0x38, 0x01, 0x15, 0x12 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 54 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0xab, 0xc7, 0x92, 0x28, 0x82, 0x58, 0x49, 0x2a, 0x7c, 0xb4, 0x51, 0x45, 0xf4, 0xb7, 0x59 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 55 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x49, 0x23, 0xc1, 0x69, 0xd5, 0x04, 0xd7, 0x51, 0x9c, 0x15, 0xd3, 0x0e, 0x75, 0x6c, 0x50 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 56 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfa, 0x75, 0xe0, 0x5b, 0xcd, 0xc7, 0xe0, 0x0c, 0x27, 0x3f, 0xa3, 0x3f, 0x6e, 0xe4, 0x41, 0xd2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 57 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0x35, 0x0f, 0xa6, 0x05, 0x70, 0x80, 0xf1, 0x08, 0x6a, 0x56, 0xb1, 0x7e, 0xc2, 0x40, 0xdb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 58 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x4e, 0x4a, 0x63, 0x24, 0xea, 0x4a, 0x5c, 0x39, 0xa6, 0x61, 0xc8, 0xfe, 0x5a, 0xda, 0x8f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 59 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x82, 0xa1, 0x6f, 0x44, 0x08, 0x8d, 0x42, 0x44, 0x7a, 0x29, 0xac, 0x09, 0x0e, 0xc1, 0x7e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 60 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x3c, 0x15, 0xbf, 0xc1, 0x1a, 0x95, 0x37, 0xc1, 0x30, 0x68, 0x70, 0x04, 0xe1, 0x36, 0xee }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 61 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0xc0, 0xa7, 0x67, 0x8d, 0xc6, 0xd8, 0xcf, 0x5c, 0x8a, 0x6d, 0x5a, 0x99, 0x60, 0x76, 0x7c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 62 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0x6b, 0x09, 0x80, 0x9d, 0x68, 0xb9, 0xa4, 0x56, 0x43, 0x2a, 0x79, 0xbd, 0xc2, 0xe3, 0x8c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 63 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xba, 0xaf, 0xfb, 0x35, 0xfb, 0xe7, 0x39, 0xc1, 0x7c, 0x6a, 0xc2, 0x2e, 0xec, 0xf1, 0x8f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 64 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xaa, 0x80, 0xa7, 0x85, 0x06, 0x75, 0xbc, 0x00, 0x7c, 0x46, 0xdf, 0x06, 0xb4, 0x98, 0x68 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 65 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xc6, 0xf3, 0x87, 0x7a, 0xf4, 0x21, 0xa9, 0x18, 0xa8, 0x4b, 0x77, 0x58, 0x58, 0x02, 0x1d }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 66 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xf1, 0x23, 0x28, 0x2c, 0x5d, 0x63, 0x39, 0x24, 0xf7, 0xd5, 0xba, 0x3f, 0x3c, 0xab, 0x11 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 67 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0xf1, 0x61, 0x00, 0x27, 0x33, 0xe9, 0x3c, 0xa4, 0x52, 0x7d, 0x22, 0xc1, 0xa0, 0xc5, 0xbb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 68 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0x2f, 0x70, 0xeb, 0xf3, 0xe3, 0xfd, 0xa2, 0x3f, 0x50, 0x8e, 0xec, 0x76, 0xb4, 0x2c, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 69 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0x9d, 0x96, 0x5e, 0x62, 0x74, 0x14, 0x3f, 0x25, 0xaf, 0xdc, 0xfc, 0x88, 0xff, 0xd7, 0x7c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 70 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0xc7, 0x4f, 0xd0, 0xb9, 0x36, 0x17, 0x64, 0xce, 0x91, 0xc5, 0x20, 0x0b, 0x09, 0x53, 0x57 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 71 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x1d, 0x1f, 0xdc, 0x2b, 0xd2, 0xc3, 0x46, 0xcd, 0x50, 0x46, 0xa8, 0xc6, 0x20, 0x91, 0x46 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 72 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe2, 0xa3, 0x75, 0x80, 0x11, 0x6c, 0xfb, 0x71, 0x85, 0x62, 0x54, 0x49, 0x6a, 0xb0, 0xac, 0xa8 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 73 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xb3, 0xa0, 0x07, 0x85, 0x91, 0x7c, 0x7e, 0xfc, 0x9a, 0xdb, 0xa3, 0x22, 0x81, 0x35, 0x71 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 74 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0x3d, 0x41, 0xf4, 0x72, 0x7b, 0x5e, 0xf0, 0xdf, 0x4a, 0xf4, 0xcf, 0x3c, 0xff, 0xa0, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 75 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa9, 0x9e, 0xbb, 0x03, 0x02, 0x60, 0x82, 0x6f, 0x98, 0x1a, 0xd3, 0xe6, 0x44, 0x90, 0xaa, 0x4f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 76 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0xf3, 0x4c, 0x7d, 0x3e, 0xae, 0x5e, 0x80, 0x08, 0x2c, 0x16, 0x47, 0x52, 0x43, 0x08, 0xee }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 77 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xeb, 0xd5, 0xad, 0x08, 0x23, 0x45, 0xb7, 0xa2, 0x09, 0x7c, 0xcd, 0x34, 0x64, 0xda, 0x02 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 78 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xc4, 0xae, 0x9a, 0x42, 0x4b, 0x2c, 0xec, 0x90, 0xc9, 0x71, 0x53, 0xc2, 0x45, 0x7e, 0xc5 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 79 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x54, 0xd6, 0x32, 0xd0, 0x3a, 0xba, 0x0b, 0xd0, 0xf9, 0x18, 0x77, 0xeb, 0xdd, 0x4d, 0x09, 0xcb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 80 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x42, 0x7b, 0xe7, 0xe4, 0xd2, 0x7c, 0xd5, 0x4f, 0x5f, 0xe3, 0x7b, 0x03, 0xcf, 0x08, 0x97 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 81 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0x09, 0x97, 0x95, 0xe8, 0x8c, 0xc1, 0x58, 0xfd, 0x75, 0xea, 0x13, 0x3d, 0x7e, 0x7f, 0xbe }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 82 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa6, 0xca, 0xe4, 0x6f, 0xb6, 0xfa, 0xdf, 0xe7, 0xa2, 0xc3, 0x02, 0xa3, 0x42, 0x42, 0x81, 0x7b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 83 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x6a, 0x70, 0x24, 0xd6, 0xa9, 0x02, 0xe0, 0xb3, 0xff, 0xcc, 0xba, 0xa9, 0x10, 0xcc, 0x3f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 84 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0x6f, 0x07, 0x76, 0x7a, 0x85, 0xa4, 0x31, 0x23, 0x21, 0xf6, 0x39, 0x68, 0x33, 0x8a, 0x01 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 85 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0xee, 0xc9, 0xeb, 0xf4, 0x2b, 0x9c, 0xa7, 0x68, 0x97, 0xd2, 0xcd, 0x6c, 0x5a, 0x12, 0xe2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 86 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x0d, 0x3a, 0x6f, 0xdc, 0xc1, 0x3f, 0x91, 0x5e, 0x2b, 0x30, 0x2c, 0xee, 0xb7, 0x0f, 0xd8 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 87 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x71, 0xdb, 0xf3, 0x7e, 0x87, 0xa2, 0xe3, 0x4d, 0x15, 0xb2, 0x0e, 0x8f, 0x10, 0xe4, 0x89, 0x24 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 88 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x45, 0xc4, 0x51, 0xe9, 0x6f, 0xf3, 0xc0, 0x45, 0xe4, 0x36, 0x7c, 0x83, 0x3e, 0x3b, 0x54 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 89 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x0d, 0xa0, 0x9c, 0x2d, 0xd1, 0x1c, 0x3b, 0x67, 0x9d, 0x08, 0xcc, 0xd2, 0x7d, 0xd5, 0x95 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 90 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x79, 0xf7, 0xc0, 0xc2, 0xa0, 0x3e, 0xe6, 0x60, 0xc6, 0xd3, 0x92, 0xdb, 0x02, 0x5d, 0x18 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 91 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0xb2, 0xc7, 0xd8, 0xeb, 0xa5, 0x31, 0xff, 0x47, 0xc5, 0x04, 0x1a, 0x55, 0xfb, 0xd1, 0xec }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 92 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x56, 0x9a, 0x2c, 0xa5, 0xa7, 0xbd, 0x51, 0x31, 0xce, 0x8d, 0xc7, 0xcb, 0xfb, 0xf7, 0x2f }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 93 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x13, 0xda, 0x0c, 0x02, 0x19, 0xb6, 0x34, 0x54, 0x03, 0x56, 0x13, 0xb5, 0xa4, 0x03, 0xdd }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 94 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x88, 0x27, 0x55, 0x1d, 0xdc, 0xc9, 0xdf, 0x23, 0xfa, 0x72, 0xa3, 0xde, 0x4e, 0x9f, 0x0b, 0x07 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 95 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x3f, 0xeb, 0xfd, 0x62, 0x5b, 0xfc, 0xd0, 0xa2, 0xc0, 0x6e, 0xb4, 0x60, 0xda, 0x17, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 96 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 16,
	  { 0xee, 0x82, 0xe6, 0xba, 0x48, 0x81, 0x56, 0xf7, 0x64, 0x96, 0x31, 0x1d, 0xa6, 0x94, 0x1d, 0xeb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 97 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 16,
	  { 0x47, 0x70, 0x44, 0x6f, 0x01, 0xd1, 0xf3, 0x91, 0x25, 0x6e, 0x85, 0xa1, 0xb3, 0x0d, 0x89, 0xd3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 98 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 16,
	  { 0xaf, 0x04, 0xb6, 0x8f, 0x10, 0x4f, 0x21, 0xef, 0x2a, 0xfb, 0x47, 0x67, 0xcf, 0x74, 0x14, 0x3c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 99 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x35, 0x79, 0xa9, 0xba, 0x38, 0xc8, 0xe4, 0x36, 0x53, 0x17, 0x3e, 0x14, 0xf3, 0xa4, 0xc6 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 100 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 16,
	  { 0xb3, 0xbb, 0xa9, 0x04, 0xf4, 0x95, 0x3e, 0x09, 0xb5, 0x48, 0x00, 0xaf, 0x2f, 0x62, 0xe7, 0xd4 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 101 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x42, 0x49, 0x65, 0x6e, 0x14, 0xb2, 0x9e, 0xb9, 0xc4, 0x48, 0x29, 0xb4, 0xc5, 0x9a, 0x46 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 102 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0x31, 0x56, 0x8f, 0xeb, 0xe8, 0x1c, 0xfc, 0x2e, 0x65, 0xaf, 0x1c, 0x86, 0xd1, 0xa3, 0x08 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 103 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xa0, 0x9c, 0x25, 0xf2, 0x73, 0xa7, 0x66, 0xdb, 0x98, 0xa4, 0x80, 0xce, 0x8d, 0xfe, 0xdc }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 104 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 16,
	  { 0xb9, 0x09, 0x92, 0x57, 0x86, 0xf3, 0x4c, 0x3c, 0x92, 0xd9, 0x71, 0x88, 0x3c, 0x9f, 0xbe, 0xdf }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 105 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 16,
	  { 0x82, 0x64, 0x7f, 0x13, 0x32, 0xfe, 0x57, 0x0a, 0x9d, 0x4d, 0x92, 0xb2, 0xee, 0x77, 0x1d, 0x3b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 106 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 16,
	  { 0x36, 0x04, 0xa7, 0xe8, 0x08, 0x32, 0xb3, 0xa9, 0x99, 0x54, 0xbc, 0xa6, 0xf5, 0xb9, 0xf5, 0x01 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 107 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 16,
	  { 0x88, 0x46, 0x07, 0xb1, 0x28, 0xc5, 0xde, 0x3a, 0xb3, 0x9a, 0x52, 0x9a, 0x1e, 0xf5, 0x1b, 0xef }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 108 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 16,
	  { 0x67, 0x0c, 0xfa, 0x09, 0x3d, 0x1d, 0xbd, 0xb2, 0x31, 0x70, 0x41, 0x40, 0x41, 0x02, 0x43, 0x5e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 109 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 16,
	  { 0x7a, 0x86, 0x71, 0x95, 0xf3, 0xce, 0x87, 0x69, 0xcb, 0xd3, 0x36, 0x50, 0x2f, 0xbb, 0x51, 0x30 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 110 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 16,
	  { 0x52, 0xef, 0xcf, 0x64, 0xc7, 0x2b, 0x2f, 0x7c, 0xa5, 0xb3, 0xc8, 0x36, 0xb1, 0x07, 0x8c, 0x15 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 111 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 16,
	  { 0x40, 0x19, 0x25, 0x0f, 0x6e, 0xef, 0xb2, 0xac, 0x5c, 0xcb, 0xca, 0xe0, 0x44, 0xe7, 0x5c, 0x7e }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 112 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 16,
	  { 0x02, 0x2c, 0x4f, 0x6f, 0x5a, 0x01, 0x7d, 0x29, 0x27, 0x85, 0x62, 0x76, 0x67, 0xdd, 0xef, 0x24 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 113 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 16,
	  { 0xe9, 0xc2, 0x10, 0x78, 0xa2, 0xeb, 0x7e, 0x03, 0x25, 0x0f, 0x71, 0x00, 0x0f, 0xa9, 0xe3, 0xed }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 114 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 16,
	  { 0xa1, 0x3e, 0xae, 0xeb, 0x9c, 0xd3, 0x91, 0xda, 0x4e, 0x2b, 0x09, 0x49, 0x0b, 0x3e, 0x7f, 0xad }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 115 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 16,
	  { 0xc9, 0x58, 0xa1, 0x71, 0xdc, 0xa1, 0xd4, 0xed, 0x53, 0xe1, 0xaf, 0x1d, 0x38, 0x08, 0x03, 0xa9 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 116 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 16,
	  { 0x21, 0x44, 0x2e, 0x07, 0xa1, 0x10, 0x66, 0x7f, 0x25, 0x83, 0xea, 0xee, 0xee, 0x44, 0xdc, 0x8c }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 117 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 16,
	  { 0x59, 0xbb, 0xb3, 0x53, 0xcf, 0x1d, 0xd8, 0x67, 0xa6, 0xe3, 0x37, 0x37, 0xaf, 0x65, 0x5e, 0x99 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 118 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 16,
	  { 0x43, 0xcd, 0x3b, 0x25, 0x37, 0x5d, 0x0c, 0xe4, 0x10, 0x87, 0xff, 0x9f, 0xe2, 0x82, 0x96, 0x39 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 119 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 16,
	  { 0x6b, 0x98, 0xb1, 0x7e, 0x80, 0xd1, 0x11, 0x8e, 0x35, 0x16, 0xbd, 0x76, 0x8b, 0x28, 0x5a, 0x84 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 120 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 16,
	  { 0xae, 0x47, 0xed, 0x36, 0x76, 0xca, 0x0c, 0x08, 0xde, 0xea, 0x02, 0xd9, 0x5b, 0x81, 0xdb, 0x58 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 121 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 16,
	  { 0x34, 0xec, 0x40, 0xdc, 0x20, 0x41, 0x37, 0x95, 0xed, 0x53, 0x62, 0x8e, 0xa7, 0x48, 0x72, 0x0b }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 122 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 16,
	  { 0x4d, 0xc6, 0x81, 0x63, 0xf8, 0xe9, 0x83, 0x54, 0x73, 0x25, 0x35, 0x42, 0xc8, 0xa6, 0x5d, 0x46 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 123 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 16,
	  { 0x2a, 0xab, 0xb9, 0x99, 0xf4, 0x36, 0x93, 0x17, 0x5a, 0xf6, 0x5c, 0x6c, 0x61, 0x2c, 0x46, 0xfb }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 124 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 16,
	  { 0xe0, 0x1f, 0x94, 0x49, 0x9d, 0xac, 0x35, 0x47, 0x51, 0x5c, 0x5b, 0x1d, 0x75, 0x6f, 0x0f, 0x58 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 125 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 16,
	  { 0x9d, 0x12, 0x43, 0x5a, 0x46, 0x48, 0x0c, 0xe0, 0x0e, 0xa3, 0x49, 0xf7, 0x17, 0x99, 0xdf, 0x9a }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 126 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 16,
	  { 0xce, 0xf4, 0x1d, 0x16, 0xd2, 0x66, 0xbd, 0xfe, 0x46, 0x93, 0x8a, 0xd7, 0x88, 0x4c, 0xc0, 0xcf }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 127 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 16,
	  { 0xb1, 0x3d, 0xb4, 0xda, 0x1f, 0x71, 0x8b, 0xc6, 0x90, 0x47, 0x97, 0xc8, 0x2b, 0xcf, 0x2d, 0x32 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 0 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0xd0, 0x25, 0x13, 0xe8, 0xd4, 0xdc, 0x98, 0x6b, 0x4a, 0xfe, 0x08, 0x7a, 0x60, 0xbd, 0x0c }, 16,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 1 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2c, 0xe1, 0xf8, 0xb7, 0xe3, 0x06, 0x27, 0xc1, 0xc4, 0x51, 0x9e, 0xad, 0xa4, 0x4b, 0xc4, 0x36 }, 16,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 2 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x46, 0xb5, 0xf8, 0x7a, 0xf4, 0x46, 0xf5, 0x79, 0x6c, 0x1f, 0xee, 0x63, 0xa2, 0xda, 0x24 }, 16,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 3 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2a, 0x56, 0x03, 0x64, 0xce, 0x52, 0x9e, 0xfc, 0x21, 0x78, 0x87, 0x79, 0x56, 0x8d, 0x55, 0x55 }, 16,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 4 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x35, 0xc1, 0x47, 0x18, 0x37, 0xaf, 0x44, 0x61, 0x53, 0xbc, 0xe5, 0x5d, 0x5b, 0xa7, 0x2a, 0x0a }, 16,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 5 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xce, 0x60, 0xbc, 0x52, 0x38, 0x62, 0x34, 0xf1, 0x58, 0xf8, 0x43, 0x41, 0xe5, 0x34, 0xcd, 0x9e }, 16,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 6 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8c, 0x7c, 0x27, 0xff, 0x32, 0xbc, 0xf8, 0xdc, 0x2d, 0xc5, 0x7c, 0x90, 0xc2, 0x90, 0x39, 0x61 }, 16,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 7 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x32, 0xbb, 0x6a, 0x7e, 0xc8, 0x44, 0x99, 0xe1, 0x66, 0xf9, 0x36, 0x00, 0x3d, 0x55, 0xa5, 0xbb }, 16,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 8 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa5, 0xc7, 0x72, 0xe5, 0xc6, 0x26, 0x31, 0xef, 0x66, 0x0e, 0xe1, 0xd5, 0x87, 0x7f, 0x6d, 0x1b }, 16,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 9 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x03, 0x0d, 0x7e, 0x5b, 0x64, 0xf3, 0x80, 0xa7, 0xe4, 0xea, 0x53, 0x87, 0xb5, 0xcd, 0x7f, 0x49 }, 16,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 10 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0d, 0xc9, 0xa2, 0x61, 0x00, 0x37, 0x00, 0x9b, 0x69, 0x8f, 0x11, 0xbb, 0x7e, 0x86, 0xc8, 0x3e }, 16,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 11 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x00, 0x46, 0x61, 0x2c, 0x76, 0x6d, 0x18, 0x40, 0xc2, 0x26, 0x36, 0x4f, 0x1f, 0xa7, 0xed, 0x72 }, 16,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 12 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x48, 0x80, 0xc7, 0xe0, 0x8f, 0x27, 0xbe, 0xfe, 0x78, 0x59, 0x07, 0x43, 0xc0, 0x5e, 0x69, 0x8b }, 16,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 13 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x25, 0x20, 0xce, 0x82, 0x9a, 0x26, 0x57, 0x7f, 0x0f, 0x48, 0x22, 0xc4, 0xec, 0xc8, 0x74, 0x01 }, 16,
	  { 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 14 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x87, 0x65, 0xe8, 0xac, 0xc1, 0x69, 0x75, 0x83, 0x19, 0xcb, 0x46, 0xdc, 0x7b, 0xcf, 0x3d, 0xca }, 16,
	  { 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 15 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0x8f, 0x4b, 0xa4, 0xf0, 0x73, 0xdf, 0x4b, 0xaa, 0x11, 0x6d, 0x01, 0x1d, 0xc2, 0x4a, 0x28 }, 16,
	  { 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 16 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x78, 0xf6, 0x8c, 0x5d, 0xbf, 0x59, 0xe2, 0x11, 0xb3, 0xa6, 0x59, 0xa7, 0x31, 0x7d, 0x94 }, 16,
	  { 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 17 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x28, 0x3d, 0x3b, 0x06, 0x9d, 0x8e, 0xb9, 0xfb, 0x43, 0x2d, 0x74, 0xb9, 0x6c, 0xa7, 0x62, 0xb4 }, 16,
	  { 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 18 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0xe1, 0x84, 0x2e, 0x8a, 0x87, 0x86, 0x1c, 0x22, 0x1a, 0x50, 0x08, 0x83, 0x24, 0x5c, 0x51 }, 16,
	  { 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 19 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xaa, 0x27, 0x04, 0x71, 0x88, 0x1b, 0xe0, 0x70, 0xfb, 0x52, 0xc7, 0x06, 0x7c, 0xe7, 0x32 }, 16,
	  { 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 20 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0xb0, 0xf4, 0x76, 0xd4, 0x84, 0xf4, 0x3f, 0x1a, 0xeb, 0x6e, 0xfa, 0x93, 0x61, 0xa8, 0xac }, 16,
	  { 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 21 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1c, 0x3a, 0x94, 0xf1, 0xc0, 0x52, 0xc5, 0x5c, 0x2d, 0x83, 0x59, 0xaf, 0xf2, 0x16, 0x3b, 0x4f }, 16,
	  { 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 22 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe8, 0xa0, 0x67, 0xb6, 0x04, 0xd5, 0x37, 0x3d, 0x8b, 0x0f, 0x2e, 0x05, 0xa0, 0x3b, 0x34, 0x1b }, 16,
	  { 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 23 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa7, 0x87, 0x6e, 0xc8, 0x7f, 0x5a, 0x09, 0xbf, 0xea, 0x42, 0xc7, 0x7d, 0xa3, 0x0f, 0xd5, 0x0e }, 16,
	  { 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 24 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0c, 0xf3, 0xe9, 0xd3, 0xa4, 0x2b, 0xe5, 0xb8, 0x54, 0xca, 0x65, 0xb1, 0x3f, 0x35, 0xf4, 0x8d }, 16,
	  { 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 25 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6c, 0x62, 0xf6, 0xbb, 0xca, 0xb7, 0xc3, 0xe8, 0x21, 0xc9, 0x29, 0x0f, 0x08, 0x89, 0x2d, 0xda }, 16,
	  { 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 26 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7f, 0x5e, 0x05, 0xbd, 0x20, 0x68, 0x73, 0x81, 0x96, 0xfe, 0xe7, 0x9a, 0xce, 0x7e, 0x3a, 0xec }, 16,
	  { 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 27 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x44, 0x0e, 0x0d, 0x73, 0x32, 0x55, 0xcd, 0xa9, 0x2f, 0xb4, 0x6e, 0x84, 0x2f, 0xe5, 0x80, 0x54 }, 16,
	  { 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 28 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaa, 0x5d, 0x5b, 0x1c, 0x4e, 0xa1, 0xb7, 0xa2, 0x2e, 0x55, 0x83, 0xac, 0x2e, 0x9e, 0xd8, 0xa7 }, 16,
	  { 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 29 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x77, 0xe5, 0x37, 0xe8, 0x9e, 0x84, 0x91, 0xe8, 0x66, 0x2a, 0xae, 0x3b, 0xc8, 0x09, 0x42, 0x1d }, 16,
	  { 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 30 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x99, 0x7d, 0xd3, 0xe9, 0xf1, 0x59, 0x8b, 0xfa, 0x73, 0xf7, 0x59, 0x73, 0xf7, 0xe9, 0x3b, 0x76 }, 16,
	  { 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 31 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x38, 0xd4, 0xf7, 0x45, 0x2a, 0xfe, 0xfc, 0xb7, 0xfc, 0x72, 0x12, 0x44, 0xe4, 0xb7, 0x2e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 32 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0b, 0xe2, 0xb1, 0x82, 0x52, 0xe7, 0x74, 0xdd, 0xa3, 0x0c, 0xdd, 0xa0, 0x2c, 0x69, 0x06, 0xe3 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 33 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd2, 0x69, 0x5e, 0x59, 0xc2, 0x03, 0x61, 0xd8, 0x26, 0x52, 0xd7, 0xd5, 0x8b, 0x6f, 0x11, 0xb2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 34 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x90, 0x2d, 0x88, 0xd1, 0x3e, 0xae, 0x52, 0x08, 0x9a, 0xbd, 0x61, 0x43, 0xcf, 0xe3, 0x94, 0xe9 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 35 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd4, 0x9b, 0xce, 0xb3, 0xb8, 0x23, 0xfe, 0xdd, 0x60, 0x2c, 0x30, 0x53, 0x45, 0x73, 0x4b, 0xd2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 36 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x70, 0x7b, 0x1d, 0xbb, 0x0f, 0xfa, 0x40, 0xef, 0x7d, 0x95, 0xde, 0xf4, 0x21, 0x23, 0x3f, 0xae }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 37 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xa0, 0xc1, 0xd9, 0x33, 0x56, 0xd9, 0xeb, 0x8a, 0xa9, 0x52, 0x08, 0x4d, 0x75, 0xf9, 0x13 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 38 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf2, 0xcb, 0xf9, 0xcb, 0x18, 0x6e, 0x27, 0x0d, 0xd7, 0xbd, 0xb0, 0xc2, 0x8f, 0xeb, 0xc5, 0x7d }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 39 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0x43, 0x37, 0xc3, 0x7c, 0x4e, 0x79, 0x0a, 0xb4, 0x57, 0x80, 0xbd, 0x9c, 0x36, 0x74, 0xa0 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 40 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8e, 0x35, 0x58, 0xc1, 0x35, 0x25, 0x2f, 0xb9, 0xc9, 0xf3, 0x67, 0xed, 0x60, 0x94, 0x67, 0xa1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 41 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1b, 0x72, 0xee, 0xae, 0xe4, 0x89, 0x9b, 0x44, 0x39, 0x14, 0xe5, 0xb3, 0xa5, 0x7f, 0xba, 0x92 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 42 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x01, 0x18, 0x65, 0xf9, 0x1b, 0xc5, 0x68, 0x68, 0xd0, 0x51, 0xe5, 0x2c, 0x9e, 0xfd, 0x59, 0xb7 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 43 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe4, 0x77, 0x13, 0x18, 0xad, 0x7a, 0x63, 0xdd, 0x68, 0x0f, 0x6e, 0x58, 0x3b, 0x77, 0x47, 0xea }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 44 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x61, 0xe3, 0xd1, 0x94, 0x08, 0x8d, 0xc8, 0xd9, 0x7e, 0x9e, 0x6d, 0xb3, 0x74, 0x57, 0xea, 0xc5 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 45 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0xff, 0x1e, 0xc9, 0xcc, 0xfb, 0xc3, 0x49, 0xe5, 0xd3, 0x56, 0xd0, 0x63, 0x69, 0x3a, 0xd6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 46 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3c, 0xc9, 0xe9, 0xa9, 0xbe, 0x8c, 0xc3, 0xf6, 0xfb, 0x2e, 0xa2, 0x40, 0x88, 0xe9, 0xbb, 0x19 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 47 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x1e, 0xe5, 0xab, 0x00, 0x3d, 0xc8, 0x72, 0x2e, 0x74, 0x90, 0x5d, 0x9a, 0x8f, 0xe3, 0xd3, 0x50 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 48 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x24, 0x53, 0x39, 0x31, 0x95, 0x84, 0xb0, 0xa4, 0x12, 0x41, 0x28, 0x69, 0xd6, 0xc2, 0xea, 0xda }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 49 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7b, 0xd4, 0x96, 0x91, 0x81, 0x15, 0xd1, 0x4e, 0xd5, 0x38, 0x08, 0x52, 0x71, 0x6c, 0x88, 0x14 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 50 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x27, 0x3a, 0xb2, 0xf2, 0xb4, 0xa3, 0x66, 0xa5, 0x7d, 0x58, 0x2a, 0x33, 0x93, 0x13, 0xc8, 0xb1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 51 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x11, 0x33, 0x65, 0xa9, 0xff, 0xbe, 0x3b, 0x0c, 0xa6, 0x1e, 0x98, 0x50, 0x75, 0x54, 0x16, 0x8b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 52 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaf, 0xa9, 0x9c, 0x99, 0x7a, 0xc4, 0x78, 0xa0, 0xde, 0xa4, 0x11, 0x9c, 0x9e, 0x45, 0xf8, 0xb1 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 53 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x92, 0x16, 0x30, 0x9a, 0x78, 0x42, 0x43, 0x0b, 0x83, 0xff, 0xb9, 0x86, 0x38, 0x01, 0x15, 0x12 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 54 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x62, 0xab, 0xc7, 0x92, 0x28, 0x82, 0x58, 0x49, 0x2a, 0x7c, 0xb4, 0x51, 0x45, 0xf4, 0xb7, 0x59 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 55 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x53, 0x49, 0x23, 0xc1, 0x69, 0xd5, 0x04, 0xd7, 0x51, 0x9c, 0x15, 0xd3, 0x0e, 0x75, 0x6c, 0x50 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 56 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfa, 0x75, 0xe0, 0x5b, 0xcd, 0xc7, 0xe0, 0x0c, 0x27, 0x3f, 0xa3, 0x3f, 0x6e, 0xe4, 0x41, 0xd2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 57 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7d, 0x35, 0x0f, 0xa6, 0x05, 0x70, 0x80, 0xf1, 0x08, 0x6a, 0x56, 0xb1, 0x7e, 0xc2, 0x40, 0xdb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 58 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf3, 0x4e, 0x4a, 0x63, 0x24, 0xea, 0x4a, 0x5c, 0x39, 0xa6, 0x61, 0xc8, 0xfe, 0x5a, 0xda, 0x8f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 59 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x08, 0x82, 0xa1, 0x6f, 0x44, 0x08, 0x8d, 0x42, 0x44, 0x7a, 0x29, 0xac, 0x09, 0x0e, 0xc1, 0x7e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 60 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x3a, 0x3c, 0x15, 0xbf, 0xc1, 0x1a, 0x95, 0x37, 0xc1, 0x30, 0x68, 0x70, 0x04, 0xe1, 0x36, 0xee }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 61 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x22, 0xc0, 0xa7, 0x67, 0x8d, 0xc6, 0xd8, 0xcf, 0x5c, 0x8a, 0x6d, 0x5a, 0x99, 0x60, 0x76, 0x7c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 62 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb4, 0x6b, 0x09, 0x80, 0x9d, 0x68, 0xb9, 0xa4, 0x56, 0x43, 0x2a, 0x79, 0xbd, 0xc2, 0xe3, 0x8c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 63 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x93, 0xba, 0xaf, 0xfb, 0x35, 0xfb, 0xe7, 0x39, 0xc1, 0x7c, 0x6a, 0xc2, 0x2e, 0xec, 0xf1, 0x8f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 64 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc8, 0xaa, 0x80, 0xa7, 0x85, 0x06, 0x75, 0xbc, 0x00, 0x7c, 0x46, 0xdf, 0x06, 0xb4, 0x98, 0x68 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 65 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x12, 0xc6, 0xf3, 0x87, 0x7a, 0xf4, 0x21, 0xa9, 0x18, 0xa8, 0x4b, 0x77, 0x58, 0x58, 0x02, 0x1d }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 66 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x33, 0xf1, 0x23, 0x28, 0x2c, 0x5d, 0x63, 0x39, 0x24, 0xf7, 0xd5, 0xba, 0x3f, 0x3c, 0xab, 0x11 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 67 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa8, 0xf1, 0x61, 0x00, 0x27, 0x33, 0xe9, 0x3c, 0xa4, 0x52, 0x7d, 0x22, 0xc1, 0xa0, 0xc5, 0xbb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 68 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb7, 0x2f, 0x70, 0xeb, 0xf3, 0xe3, 0xfd, 0xa2, 0x3f, 0x50, 0x8e, 0xec, 0x76, 0xb4, 0x2c, 0x02 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 69 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6a, 0x9d, 0x96, 0x5e, 0x62, 0x74, 0x14, 0x3f, 0x25, 0xaf, 0xdc, 0xfc, 0x88, 0xff, 0xd7, 0x7c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 70 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa0, 0xc7, 0x4f, 0xd0, 0xb9, 0x36, 0x17, 0x64, 0xce, 0x91, 0xc5, 0x20, 0x0b, 0x09, 0x53, 0x57 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 71 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x09, 0x1d, 0x1f, 0xdc, 0x2b, 0xd2, 0xc3, 0x46, 0xcd, 0x50, 0x46, 0xa8, 0xc6, 0x20, 0x91, 0x46 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 72 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe2, 0xa3, 0x75, 0x80, 0x11, 0x6c, 0xfb, 0x71, 0x85, 0x62, 0x54, 0x49, 0x6a, 0xb0, 0xac, 0xa8 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 73 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0xb3, 0xa0, 0x07, 0x85, 0x91, 0x7c, 0x7e, 0xfc, 0x9a, 0xdb, 0xa3, 0x22, 0x81, 0x35, 0x71 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 74 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0x3d, 0x41, 0xf4, 0x72, 0x7b, 0x5e, 0xf0, 0xdf, 0x4a, 0xf4, 0xcf, 0x3c, 0xff, 0xa0, 0xcb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 75 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa9, 0x9e, 0xbb, 0x03, 0x02, 0x60, 0x82, 0x6f, 0x98, 0x1a, 0xd3, 0xe6, 0x44, 0x90, 0xaa, 0x4f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 76 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x73, 0xf3, 0x4c, 0x7d, 0x3e, 0xae, 0x5e, 0x80, 0x08, 0x2c, 0x16, 0x47, 0x52, 0x43, 0x08, 0xee }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 77 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0xeb, 0xd5, 0xad, 0x08, 0x23, 0x45, 0xb7, 0xa2, 0x09, 0x7c, 0xcd, 0x34, 0x64, 0xda, 0x02 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 78 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7c, 0xc4, 0xae, 0x9a, 0x42, 0x4b, 0x2c, 0xec, 0x90, 0xc9, 0x71, 0x53, 0xc2, 0x45, 0x7e, 0xc5 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 79 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x54, 0xd6, 0x32, 0xd0, 0x3a, 0xba, 0x0b, 0xd0, 0xf9, 0x18, 0x77, 0xeb, 0xdd, 0x4d, 0x09, 0xcb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 80 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xd3, 0x42, 0x7b, 0xe7, 0xe4, 0xd2, 0x7c, 0xd5, 0x4f, 0x5f, 0xe3, 0x7b, 0x03, 0xcf, 0x08, 0x97 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 81 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb2, 0x09, 0x97, 0x95, 0xe8, 0x8c, 0xc1, 0x58, 0xfd, 0x75, 0xea, 0x13, 0x3d, 0x7e, 0x7f, 0xbe }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 82 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa6, 0xca, 0xe4, 0x6f, 0xb6, 0xfa, 0xdf, 0xe7, 0xa2, 0xc3, 0x02, 0xa3, 0x42, 0x42, 0x81, 0x7b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 83 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x6a, 0x70, 0x24, 0xd6, 0xa9, 0x02, 0xe0, 0xb3, 0xff, 0xcc, 0xba, 0xa9, 0x10, 0xcc, 0x3f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 84 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0x6f, 0x07, 0x76, 0x7a, 0x85, 0xa4, 0x31, 0x23, 0x21, 0xf6, 0x39, 0x68, 0x33, 0x8a, 0x01 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 85 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x15, 0xee, 0xc9, 0xeb, 0xf4, 0x2b, 0x9c, 0xa7, 0x68, 0x97, 0xd2, 0xcd, 0x6c, 0x5a, 0x12, 0xe2 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 86 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdb, 0x0d, 0x3a, 0x6f, 0xdc, 0xc1, 0x3f, 0x91, 0x5e, 0x2b, 0x30, 0x2c, 0xee, 0xb7, 0x0f, 0xd8 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 87 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x71, 0xdb, 0xf3, 0x7e, 0x87, 0xa2, 0xe3, 0x4d, 0x15, 0xb2, 0x0e, 0x8f, 0x10, 0xe4, 0x89, 0x24 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 88 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc7, 0x45, 0xc4, 0x51, 0xe9, 0x6f, 0xf3, 0xc0, 0x45, 0xe4, 0x36, 0x7c, 0x83, 0x3e, 0x3b, 0x54 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 89 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0x0d, 0xa0, 0x9c, 0x2d, 0xd1, 0x1c, 0x3b, 0x67, 0x9d, 0x08, 0xcc, 0xd2, 0x7d, 0xd5, 0x95 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 90 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x79, 0xf7, 0xc0, 0xc2, 0xa0, 0x3e, 0xe6, 0x60, 0xc6, 0xd3, 0x92, 0xdb, 0x02, 0x5d, 0x18 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 91 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa4, 0xb2, 0xc7, 0xd8, 0xeb, 0xa5, 0x31, 0xff, 0x47, 0xc5, 0x04, 0x1a, 0x55, 0xfb, 0xd1, 0xec }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 92 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x74, 0x56, 0x9a, 0x2c, 0xa5, 0xa7, 0xbd, 0x51, 0x31, 0xce, 0x8d, 0xc7, 0xcb, 0xfb, 0xf7, 0x2f }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 93 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x37, 0x13, 0xda, 0x0c, 0x02, 0x19, 0xb6, 0x34, 0x54, 0x03, 0x56, 0x13, 0xb5, 0xa4, 0x03, 0xdd }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 94 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x88, 0x27, 0x55, 0x1d, 0xdc, 0xc9, 0xdf, 0x23, 0xfa, 0x72, 0xa3, 0xde, 0x4e, 0x9f, 0x0b, 0x07 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 95 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2e, 0x3f, 0xeb, 0xfd, 0x62, 0x5b, 0xfc, 0xd0, 0xa2, 0xc0, 0x6e, 0xb4, 0x60, 0xda, 0x17, 0x32 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 96 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xee, 0x82, 0xe6, 0xba, 0x48, 0x81, 0x56, 0xf7, 0x64, 0x96, 0x31, 0x1d, 0xa6, 0x94, 0x1d, 0xeb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 97 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x47, 0x70, 0x44, 0x6f, 0x01, 0xd1, 0xf3, 0x91, 0x25, 0x6e, 0x85, 0xa1, 0xb3, 0x0d, 0x89, 0xd3 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 98 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xaf, 0x04, 0xb6, 0x8f, 0x10, 0x4f, 0x21, 0xef, 0x2a, 0xfb, 0x47, 0x67, 0xcf, 0x74, 0x14, 0x3c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 99 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xcf, 0x35, 0x79, 0xa9, 0xba, 0x38, 0xc8, 0xe4, 0x36, 0x53, 0x17, 0x3e, 0x14, 0xf3, 0xa4, 0xc6 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 100 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb3, 0xbb, 0xa9, 0x04, 0xf4, 0x95, 0x3e, 0x09, 0xb5, 0x48, 0x00, 0xaf, 0x2f, 0x62, 0xe7, 0xd4 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 101 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x42, 0x49, 0x65, 0x6e, 0x14, 0xb2, 0x9e, 0xb9, 0xc4, 0x48, 0x29, 0xb4, 0xc5, 0x9a, 0x46 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 102 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0x31, 0x56, 0x8f, 0xeb, 0xe8, 0x1c, 0xfc, 0x2e, 0x65, 0xaf, 0x1c, 0x86, 0xd1, 0xa3, 0x08 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 103 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9c, 0xa0, 0x9c, 0x25, 0xf2, 0x73, 0xa7, 0x66, 0xdb, 0x98, 0xa4, 0x80, 0xce, 0x8d, 0xfe, 0xdc }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 104 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb9, 0x09, 0x92, 0x57, 0x86, 0xf3, 0x4c, 0x3c, 0x92, 0xd9, 0x71, 0x88, 0x3c, 0x9f, 0xbe, 0xdf }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 105 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x82, 0x64, 0x7f, 0x13, 0x32, 0xfe, 0x57, 0x0a, 0x9d, 0x4d, 0x92, 0xb2, 0xee, 0x77, 0x1d, 0x3b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 106 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x36, 0x04, 0xa7, 0xe8, 0x08, 0x32, 0xb3, 0xa9, 0x99, 0x54, 0xbc, 0xa6, 0xf5, 0xb9, 0xf5, 0x01 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 107 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x88, 0x46, 0x07, 0xb1, 0x28, 0xc5, 0xde, 0x3a, 0xb3, 0x9a, 0x52, 0x9a, 0x1e, 0xf5, 0x1b, 0xef }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 108 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x67, 0x0c, 0xfa, 0x09, 0x3d, 0x1d, 0xbd, 0xb2, 0x31, 0x70, 0x41, 0x40, 0x41, 0x02, 0x43, 0x5e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 109 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7a, 0x86, 0x71, 0x95, 0xf3, 0xce, 0x87, 0x69, 0xcb, 0xd3, 0x36, 0x50, 0x2f, 0xbb, 0x51, 0x30 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 110 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x52, 0xef, 0xcf, 0x64, 0xc7, 0x2b, 0x2f, 0x7c, 0xa5, 0xb3, 0xc8, 0x36, 0xb1, 0x07, 0x8c, 0x15 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 111 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x40, 0x19, 0x25, 0x0f, 0x6e, 0xef, 0xb2, 0xac, 0x5c, 0xcb, 0xca, 0xe0, 0x44, 0xe7, 0x5c, 0x7e }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 112 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x02, 0x2c, 0x4f, 0x6f, 0x5a, 0x01, 0x7d, 0x29, 0x27, 0x85, 0x62, 0x76, 0x67, 0xdd, 0xef, 0x24 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 113 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe9, 0xc2, 0x10, 0x78, 0xa2, 0xeb, 0x7e, 0x03, 0x25, 0x0f, 0x71, 0x00, 0x0f, 0xa9, 0xe3, 0xed }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 114 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xa1, 0x3e, 0xae, 0xeb, 0x9c, 0xd3, 0x91, 0xda, 0x4e, 0x2b, 0x09, 0x49, 0x0b, 0x3e, 0x7f, 0xad }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 115 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc9, 0x58, 0xa1, 0x71, 0xdc, 0xa1, 0xd4, 0xed, 0x53, 0xe1, 0xaf, 0x1d, 0x38, 0x08, 0x03, 0xa9 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 116 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x21, 0x44, 0x2e, 0x07, 0xa1, 0x10, 0x66, 0x7f, 0x25, 0x83, 0xea, 0xee, 0xee, 0x44, 0xdc, 0x8c }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 117 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x59, 0xbb, 0xb3, 0x53, 0xcf, 0x1d, 0xd8, 0x67, 0xa6, 0xe3, 0x37, 0x37, 0xaf, 0x65, 0x5e, 0x99 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 118 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x43, 0xcd, 0x3b, 0x25, 0x37, 0x5d, 0x0c, 0xe4, 0x10, 0x87, 0xff, 0x9f, 0xe2, 0x82, 0x96, 0x39 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 119 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x6b, 0x98, 0xb1, 0x7e, 0x80, 0xd1, 0x11, 0x8e, 0x35, 0x16, 0xbd, 0x76, 0x8b, 0x28, 0x5a, 0x84 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 120 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xae, 0x47, 0xed, 0x36, 0x76, 0xca, 0x0c, 0x08, 0xde, 0xea, 0x02, 0xd9, 0x5b, 0x81, 0xdb, 0x58 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x80 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 121 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x34, 0xec, 0x40, 0xdc, 0x20, 0x41, 0x37, 0x95, 0xed, 0x53, 0x62, 0x8e, 0xa7, 0x48, 0x72, 0x0b }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 122 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x4d, 0xc6, 0x81, 0x63, 0xf8, 0xe9, 0x83, 0x54, 0x73, 0x25, 0x35, 0x42, 0xc8, 0xa6, 0x5d, 0x46 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 123 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x2a, 0xab, 0xb9, 0x99, 0xf4, 0x36, 0x93, 0x17, 0x5a, 0xf6, 0x5c, 0x6c, 0x61, 0x2c, 0x46, 0xfb }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 124 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x1f, 0x94, 0x49, 0x9d, 0xac, 0x35, 0x47, 0x51, 0x5c, 0x5b, 0x1d, 0x75, 0x6f, 0x0f, 0x58 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8 }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 125 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9d, 0x12, 0x43, 0x5a, 0x46, 0x48, 0x0c, 0xe0, 0x0e, 0xa3, 0x49, 0xf7, 0x17, 0x99, 0xdf, 0x9a }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 126 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xce, 0xf4, 0x1d, 0x16, 0xd2, 0x66, 0xbd, 0xfe, 0x46, 0x93, 0x8a, 0xd7, 0x88, 0x4c, 0xc0, 0xcf }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe }, 16 },
	{ "NIST KAT-AES ECBVarTxt192.rsp test vector 127 decrypt", LIBCAES_CRYPT_MODE_DECRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 192,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xb1, 0x3d, 0xb4, 0xda, 0x1f, 0x71, 0x8b, 0xc6, 0x90, 0x47, 0x97, 0xc8, 0x2b, 0xcf, 0x2d, 0x32 }, 16,
	  { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, 16 },

	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 0 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xdd, 0xc6, 0xbf, 0x79, 0x0c, 0x15, 0x76, 0x0d, 0x8d, 0x9a, 0xeb, 0x6f, 0x9a, 0x75, 0xfd, 0x4e }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 1 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0a, 0x6b, 0xdc, 0x6d, 0x4c, 0x1e, 0x62, 0x80, 0x30, 0x1f, 0xd8, 0xe9, 0x7d, 0xdb, 0xe6, 0x01 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 2 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x9b, 0x80, 0xee, 0xfb, 0x7e, 0xbe, 0x2d, 0x2b, 0x16, 0x24, 0x7a, 0xa0, 0xef, 0xc7, 0x2f, 0x5d }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 3 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x7f, 0x2c, 0x5e, 0xce, 0x07, 0xa9, 0x8d, 0x8b, 0xee, 0x13, 0xc5, 0x11, 0x77, 0x39, 0x5f, 0xf7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 4 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x78, 0x18, 0xd8, 0x00, 0xdc, 0xf6, 0xf4, 0xbe, 0x1e, 0x0e, 0x94, 0xf4, 0x03, 0xd1, 0xe4, 0xc2 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 5 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xe7, 0x4c, 0xd1, 0xc9, 0x2f, 0x09, 0x19, 0xc3, 0x5a, 0x03, 0x24, 0x12, 0x3d, 0x61, 0x77, 0xd3 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 6 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x80, 0x92, 0xa4, 0xdc, 0xf2, 0xda, 0x7e, 0x77, 0xe9, 0x3b, 0xdd, 0x37, 0x1d, 0xfe, 0xd8, 0x2e }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 7 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x49, 0xaf, 0x6b, 0x37, 0x21, 0x35, 0xac, 0xef, 0x10, 0x13, 0x2e, 0x54, 0x8f, 0x21, 0x7b, 0x17 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 8 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x8b, 0xcd, 0x40, 0xf9, 0x4e, 0xbb, 0x63, 0xb9, 0xf7, 0x90, 0x96, 0x76, 0xe6, 0x67, 0xf1, 0xe7 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 9 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xfe, 0x1c, 0xff, 0xb8, 0x3f, 0x45, 0xdc, 0xfb, 0x38, 0xb2, 0x9b, 0xe4, 0x38, 0xdb, 0xd3, 0xab }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 10 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x0d, 0xc5, 0x8a, 0x8d, 0x88, 0x66, 0x23, 0x70, 0x5a, 0xec, 0x15, 0xcb, 0x1e, 0x70, 0xdc, 0x0e }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 11 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xc2, 0x18, 0xfa, 0xa1, 0x60, 0x56, 0xbd, 0x07, 0x74, 0xc3, 0xe8, 0xd7, 0x9c, 0x35, 0xa5, 0xe4 }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 12 encrypt", LIBCAES_CRYPT_MODE_ENCRYPT,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 256,
	  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, 16,
	  { 0x04, 0x7b, 0xba, 0x83, 0xf7, 0xaa, 0x84, 0x17, 0x31, 0x50, 0x4e, 0x01, 0x22, 0x08, 0xfc, 0x9e }, 16 },
	{ "NIST KAT-AES ECBVarTxt256.rsp test vector 13 enc