/*
 * Export handle
 *
 * Copyright (C) 2011-2019, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#if !defined( _EXPORT_HANDLE_H )
#define _EXPORT_HANDLE_H

#include <common.h>
#include <file_stream.h>
#include <types.h>

#include "evttools_libcerror.h"
#include "evttools_libevt.h"
#include "log_handle.h"
#include "message_handle.h"
#include "message_string.h"

#if defined( __cplusplus )
extern "C" {
#endif

enum EXPORT_MODES
{
	EXPORT_MODE_ALL				= (int) 'a',
	EXPORT_MODE_ITEMS			= (int) 'i',
	EXPORT_MODE_RECOVERED			= (int) 'r'
};

typedef struct export_handle export_handle_t;

struct export_handle
{
	/* The export mode
	 */
	uint8_t export_mode;

	/* The libevt input file
	 */
	libevt_file_t *input_file;

	/* The message handle
	 */
	message_handle_t *message_handle;

	/* The event log type
	 */
	int event_log_type;

	/* Value to indicate the input is open
	 */
	int input_is_open;

	/* The ascii codepage
	 */
	int ascii_codepage;

	/* The notification output stream
	 */
	FILE *notify_stream;

	/* Value to indicate if abort was signalled
	 */
	int abort;
};

const char *export_handle_get_event_log_key_name(
             int event_log_type );

const char *export_handle_get_event_type(
             uint16_t event_type );

int export_handle_initialize(
     export_handle_t **export_handle,
     libcerror_error_t **error );

int export_handle_free(
     export_handle_t **export_handle,
     libcerror_error_t **error );

int export_handle_signal_abort(
     export_handle_t *export_handle,
     libcerror_error_t **error );

int export_handle_set_export_mode(
     export_handle_t *export_handle,
     const system_character_t *string,
     libcerror_error_t **error );

int export_handle_set_ascii_codepage(
     export_handle_t *export_handle,
     const system_character_t *string,
     libcerror_error_t **error );

int export_handle_set_preferred_language_identifier(
     export_handle_t *export_handle,
     uint32_t preferred_language_identifier,
     libcerror_error_t **error );

int export_handle_set_event_log_type(
     export_handle_t *export_handle,
     const system_character_t *string,
     libcerror_error_t **error );

int export_handle_set_event_log_type_from_filename(
     export_handle_t *export_handle,
     const system_character_t *filename,
     libcerror_error_t **error );

int export_handle_set_software_registry_filename(
     export_handle_t *export_handle,
     const system_character_t *filename,
     libcerror_error_t **error );

int export_handle_set_system_registry_filename(
     export_handle_t *export_handle,
     const system_character_t *filename,
     libcerror_error_t **error );

int export_handle_set_registry_directory_name(
     export_handle_t *export_handle,
     const system_character_t *name,
     libcerror_error_t **error );

int export_handle_set_resource_files_path(
     export_handle_t *export_handle,
     const system_character_t *path,
     libcerror_error_t **error );

int export_handle_open_input(
     export_handle_t *export_handle,
     const system_character_t *filename,
     libcerror_error_t **error );

int export_handle_close_input(
     export_handle_t *export_handle,
     libcerror_error_t **error );

int export_handle_posix_time_value_fprint(
     export_handle_t *export_handle,
     const char *value_name,
     uint32_t value_32bit,
     libcerror_error_t **error );

/* Record specific export functions
 */
int export_handle_export_record_event_category(
     export_handle_t *export_handle,
     libevt_record_t *record,
     const system_character_t *event_source,
     size_t event_source_length,
     log_handle_t *log_handle,
     libcerror_error_t **error );

int export_handle_export_record_event_message(
     export_handle_t *export_handle,
     libevt_record_t *record,
     const system_character_t *event_source,
     size_t event_source_length,
     uint32_t event_identifier,
     log_handle_t *log_handle,
     libcerror_error_t **error );

int export_handle_export_record(
     export_handle_t *export_handle,
     libevt_record_t *record,
     log_handle_t *log_handle,
     libcerror_error_t **error );

/* File export functions
 */
int export_handle_export_records(
     export_handle_t *export_handle,
     libevt_file_t *file,
     log_handle_t *log_handle,
     libcerror_error_t **error );

int export_handle_export_recovered_records(
     export_handle_t *export_handle,
     libevt_file_t *file,
     log_handle_t *log_handle,
     libcerror_error_t **error );

int export_handle_export_file(
     export_handle_t *export_handle,
     log_handle_t *log_handle,
     libcerror_error_t **error );

#if defined( __cplusplus )
}
#endif

#endif /* !defined( _EXPORT_HANDLE_H ) */

