/*
 * Debug functions
 *
 * Copyright (C) 2011-2023, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <memory.h>
#include <narrow_string.h>
#include <system_string.h>
#include <types.h>
#include <wide_string.h>

#include "libwrc_debug.h"
#include "libwrc_definitions.h"
#include "libwrc_libbfio.h"
#include "libwrc_libcerror.h"
#include "libwrc_libcnotify.h"
#include "libwrc_libuna.h"

#if defined( HAVE_DEBUG_OUTPUT )

/* Returns a string with the resource identifier description
 */
const char *libwrc_debug_get_resource_identifier(
             uint32_t resource_identifier )
{
	switch( resource_identifier )
	{
		case LIBWRC_RESOURCE_IDENTIFIER_CURSOR:
			return( "Cursor" );

		case LIBWRC_RESOURCE_IDENTIFIER_BITMAP:
			return( "Bitmap" );

		case LIBWRC_RESOURCE_IDENTIFIER_ICON:
			return( "Icon" );

		case LIBWRC_RESOURCE_IDENTIFIER_MENU:
			return( "Menu" );

		case LIBWRC_RESOURCE_IDENTIFIER_DIALOG:
			return( "Dialog" );

		case LIBWRC_RESOURCE_IDENTIFIER_STRING_TABLE:
			return( "String table" );

		case LIBWRC_RESOURCE_IDENTIFIER_FONT_DIRECTORY:
			return( "Font Directory" );

		case LIBWRC_RESOURCE_IDENTIFIER_FONT:
			return( "Font" );

		case LIBWRC_RESOURCE_IDENTIFIER_ACCELERATOR:
			return( "Accelerator" );

		case LIBWRC_RESOURCE_IDENTIFIER_RAW_DATA:
			return( "Raw data" );

		case LIBWRC_RESOURCE_IDENTIFIER_MESSAGE_TABLE:
			return( "Message Table" );

		case LIBWRC_RESOURCE_IDENTIFIER_GROUP_CURSOR:
			return( "Group Cursor" );

		case LIBWRC_RESOURCE_IDENTIFIER_GROUP_ICON:
			return( "Group Icon" );

		case LIBWRC_RESOURCE_IDENTIFIER_VERSION_INFORMATION:
			return( "Version information" );

		case LIBWRC_RESOURCE_IDENTIFIER_DIALOG_INCLUDE:
			return( "Dialog Include" );

		case LIBWRC_RESOURCE_IDENTIFIER_PLUG_AND_PLAY:
			return( "Plug and Play" );

		case LIBWRC_RESOURCE_IDENTIFIER_VXD:
			return( "VXD" );

		case LIBWRC_RESOURCE_IDENTIFIER_ANIMATED_CURSOR:
			return( "Animated Cursor" );

		case LIBWRC_RESOURCE_IDENTIFIER_ANIMATED_ICON:
			return( "Animated Icon" );

		case LIBWRC_RESOURCE_IDENTIFIER_HTML:
			return( "HTML" );

		case LIBWRC_RESOURCE_IDENTIFIER_MANIFEST:
			return( "Manifest" );

		default:
			break;
	}
	return( "UNKNOWN" );
}

/* Prints an UTF-16 string value
 * Returns 1 if successful or -1 on error
 */
int libwrc_debug_print_utf16_string_value(
     const char *function_name,
     const char *value_name,
     const uint8_t *byte_stream,
     size_t byte_stream_size,
     int byte_order,
     libcerror_error_t **error )
{
	system_character_t *string = NULL;
	static char *function      = "libwrc_debug_print_utf16_string_value";
	size_t string_size         = 0;
	int result                 = 0;

	if( ( byte_stream == NULL )
	 || ( byte_stream_size == 0 ) )
	{
		libcnotify_printf(
		 "%s: %s:\n",
		 function_name,
		 value_name );

		return( 1 );
	}
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
	result = libuna_utf16_string_size_from_utf16_stream(
	          byte_stream,
	          byte_stream_size,
	          byte_order,
	          &string_size,
	          error );
#else
	result = libuna_utf8_string_size_from_utf16_stream(
	          byte_stream,
	          byte_stream_size,
	          byte_order,
	          &string_size,
	          error );
#endif
	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
		 "%s: unable to determine size of string.",
		 function );

		goto on_error;
	}
	if( string_size > ( (size_t) SSIZE_MAX / sizeof( system_character_t ) ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid string size value exceeds maximum.",
		 function );

		goto on_error;
	}
	libcnotify_printf(
	 "%s: %s:",
	 function_name,
	 value_name );

	if( string_size > 0 )
	{
		string = system_string_allocate(
		          string_size );

		if( string == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
			 "%s: unable to create string.",
			 function );

			goto on_error;
		}
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
		result = libuna_utf16_string_copy_from_utf16_stream(
		          (libuna_utf16_character_t *) string,
		          string_size,
		          byte_stream,
		          byte_stream_size,
		          byte_order,
		          error );
#else
		result = libuna_utf8_string_copy_from_utf16_stream(
		          (libuna_utf8_character_t *) string,
		          string_size,
		          byte_stream,
		          byte_stream_size,
		          byte_order,
		          error );
#endif
		if( result != 1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to set string.",
			 function );

			goto on_error;
		}
		libcnotify_printf(
		 " %s",
		 string );

		memory_free(
		 string );
	}
	libcnotify_printf(
	 "\n" );

	return( 1 );

on_error:
	if( string != NULL )
	{
		memory_free(
		 string );
	}
	return( -1 );
}

/* Prints the read offsets
 * Returns 1 if successful or -1 on error
 */
int libwrc_debug_print_read_offsets(
     libbfio_handle_t *file_io_handle,
     libcerror_error_t **error )
{
	static char *function = "libwrc_debug_print_read_offsets";
	off64_t offset        = 0;
	size64_t size         = 0;
	int number_of_offsets = 0;
	int offset_iterator   = 0;

	if( file_io_handle == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid file io handle.",
		 function );

		return( -1 );
	}
	if( libbfio_handle_get_number_of_offsets_read(
	     file_io_handle,
	     &number_of_offsets,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
		 "%s: unable to retrieve number of offsets read.",
		 function );

		return( -1 );
	}
	libcnotify_printf(
	 "Offsets read:\n" );

	for( offset_iterator = 0;
	     offset_iterator < number_of_offsets;
	     offset_iterator++ )
	{
		if( libbfio_handle_get_offset_read(
		     file_io_handle,
		     offset_iterator,
		     &offset,
		     &size,
		     error ) != 1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to retrieve offset: %d.",
			 function,
			 ( offset_iterator + 1 ) );

			return( -1 );
		}
		libcnotify_printf(
		 "%08" PRIi64 " ( 0x%08" PRIx64 " ) - %08" PRIi64 " ( 0x%08" PRIx64 " ) size: %" PRIu64 "\n",
		 offset,
		 offset,
		 offset + size,
		 offset + size,
		 size );
	}
	libcnotify_printf(
	 "\n" );

	return( 1 );
}

#endif /* defined( HAVE_DEBUG_OUTPUT ) */

