//! A LRM (linear reference model) is an abstract representation
//! where the geometry and real distances are not considered.

use geo::Point;
use thiserror::Error;

use crate::lrs::Properties;

/// Measurement along the `Curve`. Typically in meters.
pub type CurvePosition = f64;

/// Measurement along the `LrmScale`. Often in meters, but it could be anything.
pub type ScalePosition = f64;

/// Errors when manipulating a `LrmScale`.
#[derive(Error, Debug, PartialEq)]
pub enum LrmScaleError {
    /// Returned when building a `LrmScale` from a builder and less than 2 [NamedAnchor] objects were provided.
    #[error("a scale needs at least two named anchor")]
    NoEnoughNamedAnchor,
    /// All the [NamedAnchor] objects must be unique within a same `LrmScale`.
    #[error("duplicated anchor: {0}")]
    DuplicatedAnchorName(String),
    /// Could not find the position on the `Curve` as the [Anchor] is not known.
    #[error("anchor is unknown in the LrmScale")]
    UnknownAnchorName,
    /// Could not find an [Anchor] that matches a given offset.
    #[error("no anchor found")]
    NoAnchorFound,
}

/// An unnamed anchor is an anchor that is not a landmark, and no point will be referenced from that anchor.
///
/// It is used to match a scale position with a `Curve` position.
/// This can happen when an object between two `NamedAnchor` has a bad offset:
/// if it is said to be 300m away from the previous named anchor, but in reality it is 322m.
/// This makes sure that an object label at +310m is located after the one at +300m.
#[derive(PartialEq, Debug, Clone)]
pub struct UnnamedAnchor {
    /// Distance from the start of the scale in the scale space, can be negative.
    pub scale_position: ScalePosition,

    /// Real distance from the start of the `Curve`.
    /// The `Curve` might not start at the same 0 (e.g. the `Curve` is longer than the scale),
    /// or the `Curve` might not progress at the same rate (e.g. the `Curve` is a schematic representation that distorts distances).
    pub curve_position: CurvePosition,

    /// Position of the anchor on the `Curve`.
    pub point: Option<Point>,

    /// Metadata to describe the node
    pub properties: Properties,
}

/// A named anchor is an anchor that is a known reference point.
///
/// It often is a milestone (such a km 42), but it can be any landmark (a bridge, a notable building…)
#[derive(PartialEq, Debug, Clone)]
pub struct NamedAnchor {
    /// The name that identifies the anchor. It must be unique within the `LrmScale`.
    pub name: String,
    /// Distance from the start of the scale in the scale space, can be negative.
    pub scale_position: ScalePosition,

    /// Real distance from the start of the `Curve`.
    /// The `Curve` might not start at the same 0 (e.g. the `Curve` is longer than the scale),
    /// or the `Curve` might not progress at the same rate (e.g. the `Curve` is a schematic representation that distorts distances).
    pub curve_position: CurvePosition,

    /// Position of the anchor on the `Curve`.
    pub point: Option<Point>,

    /// Metadata to describe the node
    pub properties: Properties,
}

/// An anchor is a reference point on a `LrmScale`.
///
/// It can either be a `NamedAnchor` or an `UnnamedAnchor`.
#[derive(Debug, PartialEq, Clone)]
pub enum Anchor {
    /// A `NamedAnchor`
    Named(NamedAnchor),
    /// An `UnnamedAnchor`
    Unnamed(UnnamedAnchor),
}

impl Anchor {
    /// Builds a named `Anchor`.
    pub fn new_named(
        name: &str,
        scale_position: ScalePosition,
        curve_position: CurvePosition,
        point: Option<Point>,
        properties: Properties,
    ) -> Self {
        Self::Named(NamedAnchor {
            name: name.to_owned(),
            scale_position,
            curve_position,
            point,
            properties,
        })
    }

    /// Create an unnamed anchor.
    pub fn new_unnamed(
        scale_position: ScalePosition,
        curve_position: CurvePosition,
        point: Option<Point>,
        properties: Properties,
    ) -> Self {
        Self::Unnamed(UnnamedAnchor {
            scale_position,
            curve_position,
            point,
            properties,
        })
    }

    /// Position of the anchor on the scale.
    ///
    /// This value is arbitrary. It typically is expressed in meters, but can be any unit.
    /// It is common, but not necessary that the scale starts at 0 at the start of the curve.
    pub fn scale_position(&self) -> ScalePosition {
        match self {
            Anchor::Named(anchor) => anchor.scale_position,
            Anchor::Unnamed(anchor) => anchor.scale_position,
        }
    }

    /// Position of the anchor on the curve.
    ///
    /// The value is between 0 and 1 and represents a fraction of the curve.
    /// The can sometimes be negative to represent an anchor located before the curve starts.
    pub fn curve_position(&self) -> ScalePosition {
        match self {
            Anchor::Named(anchor) => anchor.curve_position,
            Anchor::Unnamed(anchor) => anchor.curve_position,
        }
    }

    /// Properties of the anchor
    pub fn properties(&self) -> &Properties {
        match self {
            Anchor::Named(anchor) => &anchor.properties,
            Anchor::Unnamed(anchor) => &anchor.properties,
        }
    }

    /// Geographical position of the anchor
    ///
    /// The location can be outside of the curve (a landmark visible from the curve)
    /// It is optional as it might be defined only with the curve position
    pub fn point(&self) -> Option<Point> {
        match self {
            Anchor::Named(anchor) => anchor.point,
            Anchor::Unnamed(anchor) => anchor.point,
        }
    }
}

/// A measure defines a location on the [LrmScale].
/// It is given as an [Anchor] name and an `offset` on that scale.
/// It is often represented as `12+100` to say `“100 scale units after the Anchor 12`”.
#[derive(Clone, Debug)]
pub struct LrmScaleMeasure {
    /// `Name` of the [Anchor]. While it is often named after a kilometer position,
    /// it can be anything (a letter, a landmark).
    pub anchor_name: String,
    /// The `offset` from the anchor in the scale units.
    /// there is no guarantee that its value matches actual distance on the `Curve` and is defined in scale units.
    pub scale_offset: ScalePosition,
}

impl LrmScaleMeasure {
    /// Builds a new `LrmMeasure` from an [Anchor] `name` and the `offset` on the [LrmScale].
    pub fn new(anchor_name: &str, scale_offset: ScalePosition) -> Self {
        Self {
            anchor_name: anchor_name.to_owned(),
            scale_offset,
        }
    }
}

/// Represents an `LrmScale` and allows to map [Measure] to a position along a `Curve`.
#[derive(PartialEq, Debug, Clone)]
pub struct LrmScale {
    /// Unique identifier.
    pub id: String,
    /// The [Anchor] objects are reference points on the scale from which relative distances are used.
    pub anchors: Vec<Anchor>,
}

impl LrmScale {
    /// Locates a point along a `Curve` given an [Anchor] and an `offset`,
    /// which might be negative.
    /// The `[CurvePosition]` is between 0.0 and 1.0, both included
    pub fn locate_point(&self, measure: &LrmScaleMeasure) -> Result<CurvePosition, LrmScaleError> {
        let named_anchor = self
            .iter_named()
            .find(|anchor| anchor.name == measure.anchor_name)
            .ok_or(LrmScaleError::UnknownAnchorName)?;

        let scale_position = named_anchor.scale_position + measure.scale_offset;
        let anchors = self
            .anchors
            .windows(2)
            .find(|window| window[1].scale_position() >= scale_position)
            .or_else(|| self.anchors.windows(2).last())
            .ok_or(LrmScaleError::NoAnchorFound)?;

        let scale_interval = anchors[0].scale_position() - anchors[1].scale_position();
        let curve_interval = anchors[0].curve_position() - anchors[1].curve_position();
        Ok(anchors[0].curve_position()
            + curve_interval * (scale_position - anchors[0].scale_position()) / scale_interval)
    }

    /// Returns a measure given a distance along the `Curve`.
    /// The corresponding [Anchor] is the named `Anchor` that gives the smallest positive `offset`.
    /// If such an `Anchor` does not exists, the first named `Anchor` is used.
    pub fn locate_anchor(
        &self,
        curve_position: CurvePosition,
    ) -> Result<LrmScaleMeasure, LrmScaleError> {
        // First, we find the nearest named Anchor to the Curve.
        let named_anchor = self
            .nearest_named(curve_position)
            .ok_or(LrmScaleError::NoAnchorFound)?;

        // Then we search the nearest Anchor that will be the reference
        // to convert from Curve units to scale units.
        let nearest_anchor = if named_anchor.curve_position < curve_position {
            self.next_anchor(&named_anchor.name)
                .or(self.previous_anchor(&named_anchor.name))
        } else {
            self.previous_anchor(&named_anchor.name)
                .or(self.next_anchor(&named_anchor.name))
        }
        .ok_or(LrmScaleError::NoAnchorFound)?;

        let ratio = (nearest_anchor.scale_position() - named_anchor.scale_position)
            / (nearest_anchor.curve_position() - named_anchor.curve_position);

        Ok(LrmScaleMeasure {
            anchor_name: named_anchor.name.clone(),
            scale_offset: (curve_position - named_anchor.curve_position) * ratio,
        })
    }

    /// Returns a measure given a distance along the `LrmScale`.
    /// The corresponding [Anchor] is the named `Anchor` that gives the smallest positive `offset`.
    /// If such an `Anchor` does not exists, the first named `Anchor` is used.
    pub fn get_measure(
        &self,
        scale_position: ScalePosition,
    ) -> Result<LrmScaleMeasure, LrmScaleError> {
        let named_anchor = self
            .scale_nearest_named(scale_position)
            .ok_or(LrmScaleError::NoAnchorFound)?;

        Ok(LrmScaleMeasure {
            anchor_name: named_anchor.name.clone(),
            scale_offset: scale_position - named_anchor.scale_position,
        })
    }

    /// Locates a point along the scale given an [Anchor] and an `offset`,
    /// which might be negative.
    pub fn get_position(&self, measure: LrmScaleMeasure) -> Result<ScalePosition, LrmScaleError> {
        let named_anchor = self
            .iter_named()
            .find(|anchor| anchor.name == measure.anchor_name)
            .ok_or(LrmScaleError::UnknownAnchorName)?;

        Ok(named_anchor.scale_position + measure.scale_offset)
    }

    fn nearest_named(&self, curve_position: CurvePosition) -> Option<&NamedAnchor> {
        // Tries to find the Anchor whose curve_position is the biggest possible, yet smaller than Curve position
        // Otherwise take the first named
        // Anchor names   ----A----B----
        // Curve positions    2    3
        // With Curve position = 2.1, we want A
        // With Curve position = 2.9, we want A
        //                       3.5, we want B
        //                       1.5, we want A
        self.iter_named()
            .rev()
            .find(|anchor| anchor.curve_position <= curve_position)
            .or_else(|| self.iter_named().next())
    }

    fn scale_nearest_named(&self, scale_position: ScalePosition) -> Option<&NamedAnchor> {
        // Like nearest_named, but our position is along the scale
        self.iter_named()
            .rev()
            .find(|anchor| anchor.scale_position <= scale_position)
            .or_else(|| self.iter_named().next())
    }

    // Finds the closest Anchor before the Anchor having the name `name`
    fn previous_anchor(&self, name: &str) -> Option<&Anchor> {
        self.anchors
            .iter()
            .rev()
            .skip_while(|anchor| match anchor {
                Anchor::Named(anchor) => anchor.name != name,
                Anchor::Unnamed(_) => true,
            })
            .nth(1)
    }

    // Finds the closest Anchor after the Anchor having the name `name`
    fn next_anchor(&self, name: &str) -> Option<&Anchor> {
        self.anchors
            .iter()
            .skip_while(|anchor| match anchor {
                Anchor::Named(anchor) => anchor.name != name,
                Anchor::Unnamed(_) => true,
            })
            .nth(1)
    }

    // Iterates only on named Anchor objects
    fn iter_named(&self) -> impl DoubleEndedIterator<Item = &NamedAnchor> + '_ {
        self.anchors.iter().filter_map(|anchor| match anchor {
            Anchor::Named(anchor) => Some(anchor),
            Anchor::Unnamed(_) => None,
        })
    }
}

#[cfg(test)]
pub(crate) mod tests {
    use crate::properties;

    use super::*;

    pub(crate) fn scale() -> LrmScale {
        LrmScale {
            id: "id".to_owned(),
            anchors: vec![
                Anchor::new_named("a", 0., 0., None, properties!()),
                Anchor::new_named("b", 10., 0.5, None, properties!()),
            ],
        }
    }

    #[test]
    fn locate_point() {
        // Everything a usual
        assert_eq!(
            scale().locate_point(&LrmScaleMeasure::new("a", 5.)),
            Ok(0.25)
        );
        assert_eq!(
            scale().locate_point(&LrmScaleMeasure::new("b", 5.)),
            Ok(0.75)
        );

        // Negative offsets
        assert_eq!(
            scale().locate_point(&LrmScaleMeasure::new("a", -5.)),
            Ok(-0.25)
        );

        // Unknown Anchor
        assert_eq!(
            scale().locate_point(&LrmScaleMeasure::new("c", 5.)),
            Err(LrmScaleError::UnknownAnchorName)
        );
    }

    #[test]
    fn nearest_named() {
        let scale = LrmScale {
            id: "id".to_owned(),
            anchors: vec![
                Anchor::new_named("a", 0., 2., None, properties!()),
                Anchor::new_named("b", 10., 3., None, properties!()),
            ],
        };

        assert_eq!(scale.nearest_named(2.1).unwrap().name, "a");
        assert_eq!(scale.nearest_named(2.9).unwrap().name, "a");
        assert_eq!(scale.nearest_named(1.5).unwrap().name, "a");
        assert_eq!(scale.nearest_named(3.5).unwrap().name, "b");
    }

    #[test]
    fn locate_anchor() {
        let measure = scale().locate_anchor(0.2).unwrap();
        assert_eq!(measure.anchor_name, "a");
        assert_eq!(measure.scale_offset, 4.);

        let measure = scale().locate_anchor(0.75).unwrap();
        assert_eq!(measure.anchor_name, "b");
        assert_eq!(measure.scale_offset, 5.);

        let measure = scale().locate_anchor(-0.05).unwrap();
        assert_eq!(measure.anchor_name, "a");
        assert_eq!(measure.scale_offset, -1.);
    }

    #[test]
    fn locate_anchor_with_unnamed() {
        // ----Unnamed(100)----A(200)----B(300)----Unnamed(400)---
        let scale = LrmScale {
            id: "id".to_owned(),
            anchors: vec![
                Anchor::new_unnamed(0., 100., None, properties!()),
                Anchor::new_named("a", 1., 200., None, properties!()),
                Anchor::new_named("b", 3., 300., None, properties!()),
                Anchor::new_unnamed(4., 400., None, properties!()),
            ],
        };

        // Unnamed----position----Named
        let measure = scale.locate_anchor(150.).unwrap();
        assert_eq!(measure.anchor_name, "a");
        assert_eq!(measure.scale_offset, -0.5);

        // position----Unnamed----Named
        let measure = scale.locate_anchor(50.).unwrap();
        assert_eq!(measure.anchor_name, "a");
        assert_eq!(measure.scale_offset, -1.5);

        // Unnamed----Named----position----Unnamed
        let measure = scale.locate_anchor(350.).unwrap();
        assert_eq!(measure.anchor_name, "b");
        assert_eq!(measure.scale_offset, 0.5);

        // Unnamed----Named----Unnamed----position
        let measure = scale.locate_anchor(500.).unwrap();
        assert_eq!(measure.anchor_name, "b");
        assert_eq!(measure.scale_offset, 2.);
    }

    #[test]
    fn get_measure() {
        // a(scale 0)----measure(scale 5)----b(scale 10)
        let measure = scale().get_measure(5.).unwrap();
        assert_eq!(measure.anchor_name, "a");
        assert_eq!(measure.scale_offset, 5.);

        // a(scale 0)----b(scale 10)----measure(scale 25)
        let measure = scale().get_measure(25.).unwrap();
        assert_eq!(measure.anchor_name, "b");
        assert_eq!(measure.scale_offset, 15.);
    }

    #[test]
    fn get_position() {
        // a(scale 0)----position(scale a+5)----b(scale 10)
        let position = scale()
            .get_position(LrmScaleMeasure {
                anchor_name: "a".to_string(),
                scale_offset: 5.,
            })
            .unwrap();
        assert_eq!(position, 5.);

        // a(scale 0)----b(scale 10)----position(scale b+15)
        let position = scale()
            .get_position(LrmScaleMeasure {
                anchor_name: "b".to_string(),
                scale_offset: 15.,
            })
            .unwrap();
        assert_eq!(position, 25.);
    }

    #[test]
    fn single_anchor() {
        // Scenario where the curve is to short to have an anchor
        // The scale of the curve goes from 1200 to 1300
        // The anchor corresponding to the 0 of the scale is at -2. of the curve
        // Anchor(curve: -2, scale: 1000)    [curve begin]------Unamed(curve: 0, scale: 1300)
        let scale = LrmScale {
            id: "id".to_owned(),
            anchors: vec![
                Anchor::new_named("a", 1000. + 0., -2., None, properties!()),
                Anchor::new_unnamed(1000. + 300., 1., None, properties!()),
            ],
        };

        let position = scale
            .locate_point(&LrmScaleMeasure {
                anchor_name: "a".to_string(),
                scale_offset: 200.,
            })
            .unwrap();

        assert_eq!(position, 0.);
    }

    #[test]
    fn irregular_scale() {
        // Let’s imagine a scale that growth weirdly
        // 0----1--9----10
        // If we want the position +5, it should be 0.5 on the scale
        let scale = LrmScale {
            id: "id".to_owned(),
            anchors: vec![
                Anchor::new_named("a", 0., 0., None, properties!()),
                Anchor::new_unnamed(1., 0.4, None, properties!()),
                Anchor::new_unnamed(9., 0.6, None, properties!()),
            ],
        };

        let position = scale
            .locate_point(&LrmScaleMeasure {
                anchor_name: "a".to_string(),
                scale_offset: 5.,
            })
            .unwrap();

        assert_eq!(position, 0.5);
    }
}
