/*
 * Library file type test program
 *
 * Copyright (C) 2008-2019, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <file_stream.h>
#include <narrow_string.h>
#include <system_string.h>
#include <types.h>
#include <wide_string.h>

#if defined( HAVE_STDLIB_H ) || defined( WINAPI )
#include <stdlib.h>
#endif

#include "olecf_test_functions.h"
#include "olecf_test_getopt.h"
#include "olecf_test_libbfio.h"
#include "olecf_test_libcerror.h"
#include "olecf_test_libolecf.h"
#include "olecf_test_macros.h"
#include "olecf_test_memory.h"
#include "olecf_test_unused.h"

#include "../libolecf/libolecf_file.h"

#if !defined( LIBOLECF_HAVE_BFIO )

LIBOLECF_EXTERN \
int libolecf_check_file_signature_file_io_handle(
     libbfio_handle_t *file_io_handle,
     libcerror_error_t **error );

LIBOLECF_EXTERN \
int libolecf_file_open_file_io_handle(
     libolecf_file_t *file,
     libbfio_handle_t *file_io_handle,
     int access_flags,
     libolecf_error_t **error );

#endif /* !defined( LIBOLECF_HAVE_BFIO ) */

#if defined( HAVE_WIDE_SYSTEM_CHARACTER ) && SIZEOF_WCHAR_T != 2 && SIZEOF_WCHAR_T != 4
#error Unsupported size of wchar_t
#endif

/* Define to make olecf_test_file generate verbose output
#define OLECF_TEST_FILE_VERBOSE
 */

/* Creates and opens a source file
 * Returns 1 if successful or -1 on error
 */
int olecf_test_file_open_source(
     libolecf_file_t **file,
     libbfio_handle_t *file_io_handle,
     libcerror_error_t **error )
{
	static char *function = "olecf_test_file_open_source";
	int result            = 0;

	if( file == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid file.",
		 function );

		return( -1 );
	}
	if( file_io_handle == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid file IO handle.",
		 function );

		return( -1 );
	}
	if( libolecf_file_initialize(
	     file,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_INITIALIZE_FAILED,
		 "%s: unable to initialize file.",
		 function );

		goto on_error;
	}
	result = libolecf_file_open_file_io_handle(
	          *file,
	          file_io_handle,
	          LIBOLECF_OPEN_READ,
	          error );

	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_IO,
		 LIBCERROR_IO_ERROR_OPEN_FAILED,
		 "%s: unable to open file.",
		 function );

		goto on_error;
	}
	return( 1 );

on_error:
	if( *file != NULL )
	{
		libolecf_file_free(
		 file,
		 NULL );
	}
	return( -1 );
}

/* Closes and frees a source file
 * Returns 1 if successful or -1 on error
 */
int olecf_test_file_close_source(
     libolecf_file_t **file,
     libcerror_error_t **error )
{
	static char *function = "olecf_test_file_close_source";
	int result            = 0;

	if( file == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid file.",
		 function );

		return( -1 );
	}
	if( libolecf_file_close(
	     *file,
	     error ) != 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_IO,
		 LIBCERROR_IO_ERROR_CLOSE_FAILED,
		 "%s: unable to close file.",
		 function );

		result = -1;
	}
	if( libolecf_file_free(
	     file,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_FINALIZE_FAILED,
		 "%s: unable to free file.",
		 function );

		result = -1;
	}
	return( result );
}

/* Tests the libolecf_file_initialize function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_initialize(
     void )
{
	libcerror_error_t *error        = NULL;
	libolecf_file_t *file           = NULL;
	int result                      = 0;

#if defined( HAVE_OLECF_TEST_MEMORY )
	int number_of_malloc_fail_tests = 1;
	int number_of_memset_fail_tests = 1;
	int test_number                 = 0;
#endif

	/* Test regular cases
	 */
	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libolecf_file_free(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test error cases
	 */
	result = libolecf_file_initialize(
	          NULL,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	file = (libolecf_file_t *) 0x12345678UL;

	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	file = NULL;

#if defined( HAVE_OLECF_TEST_MEMORY )

	for( test_number = 0;
	     test_number < number_of_malloc_fail_tests;
	     test_number++ )
	{
		/* Test libolecf_file_initialize with malloc failing
		 */
		olecf_test_malloc_attempts_before_fail = test_number;

		result = libolecf_file_initialize(
		          &file,
		          &error );

		if( olecf_test_malloc_attempts_before_fail != -1 )
		{
			olecf_test_malloc_attempts_before_fail = -1;

			if( file != NULL )
			{
				libolecf_file_free(
				 &file,
				 NULL );
			}
		}
		else
		{
			OLECF_TEST_ASSERT_EQUAL_INT(
			 "result",
			 result,
			 -1 );

			OLECF_TEST_ASSERT_IS_NULL(
			 "file",
			 file );

			OLECF_TEST_ASSERT_IS_NOT_NULL(
			 "error",
			 error );

			libcerror_error_free(
			 &error );
		}
	}
	for( test_number = 0;
	     test_number < number_of_memset_fail_tests;
	     test_number++ )
	{
		/* Test libolecf_file_initialize with memset failing
		 */
		olecf_test_memset_attempts_before_fail = test_number;

		result = libolecf_file_initialize(
		          &file,
		          &error );

		if( olecf_test_memset_attempts_before_fail != -1 )
		{
			olecf_test_memset_attempts_before_fail = -1;

			if( file != NULL )
			{
				libolecf_file_free(
				 &file,
				 NULL );
			}
		}
		else
		{
			OLECF_TEST_ASSERT_EQUAL_INT(
			 "result",
			 result,
			 -1 );

			OLECF_TEST_ASSERT_IS_NULL(
			 "file",
			 file );

			OLECF_TEST_ASSERT_IS_NOT_NULL(
			 "error",
			 error );

			libcerror_error_free(
			 &error );
		}
	}
#endif /* defined( HAVE_OLECF_TEST_MEMORY ) */

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	return( 0 );
}

/* Tests the libolecf_file_free function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_free(
     void )
{
	libcerror_error_t *error = NULL;
	int result               = 0;

	/* Test error cases
	 */
	result = libolecf_file_free(
	          NULL,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_open function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_open(
     const system_character_t *source )
{
	char narrow_source[ 256 ];

	libcerror_error_t *error = NULL;
	libolecf_file_t *file    = NULL;
	int result               = 0;

	/* Initialize test
	 */
	result = olecf_test_get_narrow_source(
	          source,
	          narrow_source,
	          256,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test open
	 */
	result = libolecf_file_open(
	          file,
	          narrow_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test error cases
	 */
	result = libolecf_file_open(
	          NULL,
	          narrow_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open(
	          file,
	          NULL,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open(
	          file,
	          narrow_source,
	          -1,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Test open when already opened
	 */
	result = libolecf_file_open(
	          file,
	          narrow_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Clean up
	 */
	result = libolecf_file_free(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	return( 0 );
}

#if defined( HAVE_WIDE_CHARACTER_TYPE )

/* Tests the libolecf_file_open_wide function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_open_wide(
     const system_character_t *source )
{
	wchar_t wide_source[ 256 ];

	libcerror_error_t *error = NULL;
	libolecf_file_t *file    = NULL;
	int result               = 0;

	/* Initialize test
	 */
	result = olecf_test_get_wide_source(
	          source,
	          wide_source,
	          256,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test open
	 */
	result = libolecf_file_open_wide(
	          file,
	          wide_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test error cases
	 */
	result = libolecf_file_open_wide(
	          NULL,
	          wide_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open_wide(
	          file,
	          NULL,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open_wide(
	          file,
	          wide_source,
	          -1,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Test open when already opened
	 */
	result = libolecf_file_open_wide(
	          file,
	          wide_source,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Clean up
	 */
	result = libolecf_file_free(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	return( 0 );
}

#endif /* defined( HAVE_WIDE_CHARACTER_TYPE ) */

/* Tests the libolecf_file_open_file_io_handle function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_open_file_io_handle(
     const system_character_t *source )
{
	libbfio_handle_t *file_io_handle = NULL;
	libcerror_error_t *error         = NULL;
	libolecf_file_t *file            = NULL;
	size_t string_length             = 0;
	int result                       = 0;

	/* Initialize test
	 */
	result = libbfio_file_initialize(
	          &file_io_handle,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        OLECF_TEST_ASSERT_IS_NOT_NULL(
         "file_io_handle",
         file_io_handle );

        OLECF_TEST_ASSERT_IS_NULL(
         "error",
         error );

	string_length = system_string_length(
	                 source );

#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
	result = libbfio_file_set_name_wide(
	          file_io_handle,
	          source,
	          string_length,
	          &error );
#else
	result = libbfio_file_set_name(
	          file_io_handle,
	          source,
	          string_length,
	          &error );
#endif
	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

        OLECF_TEST_ASSERT_IS_NULL(
         "error",
         error );

	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test open
	 */
	result = libolecf_file_open_file_io_handle(
	          file,
	          file_io_handle,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test error cases
	 */
	result = libolecf_file_open_file_io_handle(
	          NULL,
	          file_io_handle,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open_file_io_handle(
	          file,
	          NULL,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	result = libolecf_file_open_file_io_handle(
	          file,
	          file_io_handle,
	          -1,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Test open when already opened
	 */
	result = libolecf_file_open_file_io_handle(
	          file,
	          file_io_handle,
	          LIBOLECF_OPEN_READ,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	/* Clean up
	 */
	result = libolecf_file_free(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libbfio_handle_free(
	          &file_io_handle,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
         "file_io_handle",
         file_io_handle );

        OLECF_TEST_ASSERT_IS_NULL(
         "error",
         error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	if( file_io_handle != NULL )
	{
		libbfio_handle_free(
		 &file_io_handle,
		 NULL );
	}
	return( 0 );
}

/* Tests the libolecf_file_close function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_close(
     void )
{
	libcerror_error_t *error = NULL;
	int result               = 0;

	/* Test error cases
	 */
	result = libolecf_file_close(
	          NULL,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_open and libolecf_file_close functions
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_open_close(
     const system_character_t *source )
{
	libcerror_error_t *error = NULL;
	libolecf_file_t *file    = NULL;
	int result               = 0;

	/* Initialize test
	 */
	result = libolecf_file_initialize(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test open and close
	 */
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
	result = libolecf_file_open_wide(
	          file,
	          source,
	          LIBOLECF_OPEN_READ,
	          &error );
#else
	result = libolecf_file_open(
	          file,
	          source,
	          LIBOLECF_OPEN_READ,
	          &error );
#endif

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libolecf_file_close(
	          file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 0 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test open and close a second time to validate clean up on close
	 */
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
	result = libolecf_file_open_wide(
	          file,
	          source,
	          LIBOLECF_OPEN_READ,
	          &error );
#else
	result = libolecf_file_open(
	          file,
	          source,
	          LIBOLECF_OPEN_READ,
	          &error );
#endif

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	result = libolecf_file_close(
	          file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 0 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Clean up
	 */
	result = libolecf_file_free(
	          &file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "file",
	 file );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	return( 0 );
}

/* Tests the libolecf_file_signal_abort function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_signal_abort(
     libolecf_file_t *file )
{
	libcerror_error_t *error = NULL;
	int result               = 0;

	/* Test regular cases
	 */
	result = libolecf_file_signal_abort(
	          file,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	/* Test error cases
	 */
	result = libolecf_file_signal_abort(
	          NULL,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_get_sector_size function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_get_sector_size(
     libolecf_file_t *file )
{
	libcerror_error_t *error = NULL;
	size32_t sector_size     = 0;
	int result               = 0;
	int sector_size_is_set   = 0;

	/* Test regular cases
	 */
	result = libolecf_file_get_sector_size(
	          file,
	          &sector_size,
	          &error );

	OLECF_TEST_ASSERT_NOT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	sector_size_is_set = result;

	/* Test error cases
	 */
	result = libolecf_file_get_sector_size(
	          NULL,
	          &sector_size,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	if( sector_size_is_set != 0 )
	{
		result = libolecf_file_get_sector_size(
		          file,
		          NULL,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_get_short_sector_size function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_get_short_sector_size(
     libolecf_file_t *file )
{
	libcerror_error_t *error     = NULL;
	size32_t short_sector_size   = 0;
	int result                   = 0;
	int short_sector_size_is_set = 0;

	/* Test regular cases
	 */
	result = libolecf_file_get_short_sector_size(
	          file,
	          &short_sector_size,
	          &error );

	OLECF_TEST_ASSERT_NOT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	short_sector_size_is_set = result;

	/* Test error cases
	 */
	result = libolecf_file_get_short_sector_size(
	          NULL,
	          &short_sector_size,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	if( short_sector_size_is_set != 0 )
	{
		result = libolecf_file_get_short_sector_size(
		          file,
		          NULL,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_get_ascii_codepage function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_get_ascii_codepage(
     libolecf_file_t *file )
{
	libcerror_error_t *error  = NULL;
	int ascii_codepage        = 0;
	int ascii_codepage_is_set = 0;
	int result                = 0;

	/* Test regular cases
	 */
	result = libolecf_file_get_ascii_codepage(
	          file,
	          &ascii_codepage,
	          &error );

	OLECF_TEST_ASSERT_NOT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	ascii_codepage_is_set = result;

	/* Test error cases
	 */
	result = libolecf_file_get_ascii_codepage(
	          NULL,
	          &ascii_codepage,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	if( ascii_codepage_is_set != 0 )
	{
		result = libolecf_file_get_ascii_codepage(
		          file,
		          NULL,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_set_ascii_codepage function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_set_ascii_codepage(
     libolecf_file_t *file )
{
	int supported_codepages[ 15 ] = {
		LIBOLECF_CODEPAGE_ASCII,
		LIBOLECF_CODEPAGE_WINDOWS_874,
		LIBOLECF_CODEPAGE_WINDOWS_932,
		LIBOLECF_CODEPAGE_WINDOWS_936,
		LIBOLECF_CODEPAGE_WINDOWS_949,
		LIBOLECF_CODEPAGE_WINDOWS_950,
		LIBOLECF_CODEPAGE_WINDOWS_1250,
		LIBOLECF_CODEPAGE_WINDOWS_1251,
		LIBOLECF_CODEPAGE_WINDOWS_1252,
		LIBOLECF_CODEPAGE_WINDOWS_1253,
		LIBOLECF_CODEPAGE_WINDOWS_1254,
		LIBOLECF_CODEPAGE_WINDOWS_1255,
		LIBOLECF_CODEPAGE_WINDOWS_1256,
		LIBOLECF_CODEPAGE_WINDOWS_1257,
		LIBOLECF_CODEPAGE_WINDOWS_1258 };

	int unsupported_codepages[ 17 ] = {
		LIBOLECF_CODEPAGE_ISO_8859_1,
		LIBOLECF_CODEPAGE_ISO_8859_2,
		LIBOLECF_CODEPAGE_ISO_8859_3,
		LIBOLECF_CODEPAGE_ISO_8859_4,
		LIBOLECF_CODEPAGE_ISO_8859_5,
		LIBOLECF_CODEPAGE_ISO_8859_6,
		LIBOLECF_CODEPAGE_ISO_8859_7,
		LIBOLECF_CODEPAGE_ISO_8859_8,
		LIBOLECF_CODEPAGE_ISO_8859_9,
		LIBOLECF_CODEPAGE_ISO_8859_10,
		LIBOLECF_CODEPAGE_ISO_8859_11,
		LIBOLECF_CODEPAGE_ISO_8859_13,
		LIBOLECF_CODEPAGE_ISO_8859_14,
		LIBOLECF_CODEPAGE_ISO_8859_15,
		LIBOLECF_CODEPAGE_ISO_8859_16,
		LIBOLECF_CODEPAGE_KOI8_R,
		LIBOLECF_CODEPAGE_KOI8_U };

	libcerror_error_t *error = NULL;
	int codepage             = 0;
	int index                = 0;
	int result               = 0;

	/* Test set ASCII codepage
	 */
	for( index = 0;
	     index < 15;
	     index++ )
	{
		codepage = supported_codepages[ index ];

		result = libolecf_file_set_ascii_codepage(
		          file,
		          codepage,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

		OLECF_TEST_ASSERT_IS_NULL(
		 "error",
		 error );
	}
	/* Test error cases
	 */
	result = libolecf_file_set_ascii_codepage(
	          NULL,
	          LIBOLECF_CODEPAGE_ASCII,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	for( index = 0;
	     index < 17;
	     index++ )
	{
		codepage = unsupported_codepages[ index ];

		result = libolecf_file_set_ascii_codepage(
		          file,
		          codepage,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	/* Clean up
	 */
	result = libolecf_file_set_ascii_codepage(
	          file,
	          LIBOLECF_CODEPAGE_WINDOWS_1252,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_get_number_of_unallocated_blocks function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_get_number_of_unallocated_blocks(
     libolecf_file_t *file )
{
	libcerror_error_t *error                = NULL;
	int number_of_unallocated_blocks        = 0;
	int number_of_unallocated_blocks_is_set = 0;
	int result                              = 0;

	/* Test regular cases
	 */
	result = libolecf_file_get_number_of_unallocated_blocks(
	          file,
	          &number_of_unallocated_blocks,
	          &error );

	OLECF_TEST_ASSERT_NOT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	number_of_unallocated_blocks_is_set = result;

	/* Test error cases
	 */
	result = libolecf_file_get_number_of_unallocated_blocks(
	          NULL,
	          &number_of_unallocated_blocks,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	if( number_of_unallocated_blocks_is_set != 0 )
	{
		result = libolecf_file_get_number_of_unallocated_blocks(
		          file,
		          NULL,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* Tests the libolecf_file_get_root_item function
 * Returns 1 if successful or 0 if not
 */
int olecf_test_file_get_root_item(
     libolecf_file_t *file )
{
	libcerror_error_t *error   = NULL;
	libolecf_item_t *root_item = 0;
	int result                 = 0;
	int root_item_is_set       = 0;

	/* Test regular cases
	 */
	result = libolecf_file_get_root_item(
	          file,
	          &root_item,
	          &error );

	OLECF_TEST_ASSERT_NOT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "error",
	 error );

	root_item_is_set = result;

	if( root_item_is_set != 0 )
	{
		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "root_item",
		 root_item );

		result = libolecf_item_free(
		          &root_item,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

		OLECF_TEST_ASSERT_IS_NULL(
		 "error",
		 error );
	}
	/* Test error cases
	 */
	result = libolecf_file_get_root_item(
	          NULL,
	          &root_item,
	          &error );

	OLECF_TEST_ASSERT_EQUAL_INT(
	 "result",
	 result,
	 -1 );

	OLECF_TEST_ASSERT_IS_NULL(
	 "root_item",
	 root_item );

	OLECF_TEST_ASSERT_IS_NOT_NULL(
	 "error",
	 error );

	libcerror_error_free(
	 &error );

	if( root_item_is_set != 0 )
	{
		result = libolecf_file_get_root_item(
		          file,
		          NULL,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NULL(
		 "root_item",
		 root_item );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "error",
		 error );

		libcerror_error_free(
		 &error );
	}
	return( 1 );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	return( 0 );
}

/* The main program
 */
#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
int wmain(
     int argc,
     wchar_t * const argv[] )
#else
int main(
     int argc,
     char * const argv[] )
#endif
{
	libbfio_handle_t *file_io_handle = NULL;
	libcerror_error_t *error         = NULL;
	libolecf_file_t *file            = NULL;
	system_character_t *source       = NULL;
	system_integer_t option          = 0;
	size_t string_length             = 0;
	int result                       = 0;

	while( ( option = olecf_test_getopt(
	                   argc,
	                   argv,
	                   _SYSTEM_STRING( "" ) ) ) != (system_integer_t) -1 )
	{
		switch( option )
		{
			case (system_integer_t) '?':
			default:
				fprintf(
				 stderr,
				 "Invalid argument: %" PRIs_SYSTEM ".\n",
				 argv[ optind - 1 ] );

				return( EXIT_FAILURE );
		}
	}
	if( optind < argc )
	{
		source = argv[ optind ];
	}
#if defined( HAVE_DEBUG_OUTPUT ) && defined( OLECF_TEST_FILE_VERBOSE )
	libolecf_notify_set_verbose(
	 1 );
	libolecf_notify_set_stream(
	 stderr,
	 NULL );
#endif

	OLECF_TEST_RUN(
	 "libolecf_file_initialize",
	 olecf_test_file_initialize );

	OLECF_TEST_RUN(
	 "libolecf_file_free",
	 olecf_test_file_free );

#if !defined( __BORLANDC__ ) || ( __BORLANDC__ >= 0x0560 )
	if( source != NULL )
	{
		result = libbfio_file_initialize(
		          &file_io_handle,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

	        OLECF_TEST_ASSERT_IS_NOT_NULL(
	         "file_io_handle",
	         file_io_handle );

	        OLECF_TEST_ASSERT_IS_NULL(
	         "error",
	         error );

		string_length = system_string_length(
		                 source );

#if defined( HAVE_WIDE_SYSTEM_CHARACTER )
		result = libbfio_file_set_name_wide(
		          file_io_handle,
		          source,
		          string_length,
		          &error );
#else
		result = libbfio_file_set_name(
		          file_io_handle,
		          source,
		          string_length,
		          &error );
#endif
		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

	        OLECF_TEST_ASSERT_IS_NULL(
	         "error",
	         error );

		result = libolecf_check_file_signature_file_io_handle(
		          file_io_handle,
		          &error );

		OLECF_TEST_ASSERT_NOT_EQUAL_INT(
		 "result",
		 result,
		 -1 );

		OLECF_TEST_ASSERT_IS_NULL(
		 "error",
		 error );
	}
	if( result != 0 )
	{
		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_open",
		 olecf_test_file_open,
		 source );

#if defined( HAVE_WIDE_CHARACTER_TYPE )

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_open_wide",
		 olecf_test_file_open_wide,
		 source );

#endif /* defined( HAVE_WIDE_CHARACTER_TYPE ) */

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_open_file_io_handle",
		 olecf_test_file_open_file_io_handle,
		 source );

		OLECF_TEST_RUN(
		 "libolecf_file_close",
		 olecf_test_file_close );

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_open_close",
		 olecf_test_file_open_close,
		 source );

		/* Initialize file for tests
		 */
		result = olecf_test_file_open_source(
		          &file,
		          file_io_handle,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

		OLECF_TEST_ASSERT_IS_NOT_NULL(
		 "file",
		 file );

		OLECF_TEST_ASSERT_IS_NULL(
		 "error",
		 error );

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_signal_abort",
		 olecf_test_file_signal_abort,
		 file );

#if defined( __GNUC__ ) && !defined( LIBOLECF_DLL_IMPORT )

		/* TODO: add tests for libolecf_file_open_read */

#endif /* defined( __GNUC__ ) && !defined( LIBOLECF_DLL_IMPORT ) */

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_get_sector_size",
		 olecf_test_file_get_sector_size,
		 file );

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_get_short_sector_size",
		 olecf_test_file_get_short_sector_size,
		 file );

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_get_ascii_codepage",
		 olecf_test_file_get_ascii_codepage,
		 file );

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_set_ascii_codepage",
		 olecf_test_file_set_ascii_codepage,
		 file );

		/* TODO: add tests for libolecf_file_get_format_version */

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_get_number_of_unallocated_blocks",
		 olecf_test_file_get_number_of_unallocated_blocks,
		 file );

		/* TODO: add tests for libolecf_file_get_unallocated_block */

		OLECF_TEST_RUN_WITH_ARGS(
		 "libolecf_file_get_root_item",
		 olecf_test_file_get_root_item,
		 file );

		/* TODO: add tests for libolecf_file_get_item_by_utf8_path */

		/* TODO: add tests for libolecf_file_get_item_by_utf16_path */

		/* Clean up
		 */
		result = olecf_test_file_close_source(
		          &file,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 0 );

		OLECF_TEST_ASSERT_IS_NULL(
		 "file",
		 file );

		OLECF_TEST_ASSERT_IS_NULL(
		 "error",
		 error );
	}
	if( file_io_handle != NULL )
	{
		result = libbfio_handle_free(
		          &file_io_handle,
		          &error );

		OLECF_TEST_ASSERT_EQUAL_INT(
		 "result",
		 result,
		 1 );

		OLECF_TEST_ASSERT_IS_NULL(
	         "file_io_handle",
	         file_io_handle );

	        OLECF_TEST_ASSERT_IS_NULL(
	         "error",
	         error );
	}
#endif /* !defined( __BORLANDC__ ) || ( __BORLANDC__ >= 0x0560 ) */

	return( EXIT_SUCCESS );

on_error:
	if( error != NULL )
	{
		libcerror_error_free(
		 &error );
	}
	if( file != NULL )
	{
		libolecf_file_free(
		 &file,
		 NULL );
	}
	if( file_io_handle != NULL )
	{
		libbfio_handle_free(
		 &file_io_handle,
		 NULL );
	}
	return( EXIT_FAILURE );
}

