import importlib
import json
import os.path
import pkgutil

import localstack.utils.container_utils.docker_cmd_client
import pytest
import requests
from click.testing import CliRunner
from localstack import config, constants
from localstack.cli.localstack import localstack as cli
from localstack.config import in_docker
from localstack.constants import MODULE_MAIN_PATH
from localstack.utils.bootstrap import in_ci
from localstack.utils.common import poll_condition
from localstack.utils.files import mkdir
from localstack.utils.run import run, to_str


@pytest.fixture
def runner():
    return CliRunner()


def container_exists(client, container_name):
    try:
        container_id = client.get_container_id(container_name)
        return True if container_id else False
    except Exception:
        return False


@pytest.fixture(autouse=True)
def container_client():
    client = localstack.utils.container_utils.docker_cmd_client.CmdDockerClient()

    yield client

    try:
        client.stop_container(config.MAIN_CONTAINER_NAME, timeout=5)
    except Exception:
        pass

    # wait until container has been removed
    assert poll_condition(
        lambda: not container_exists(client, config.MAIN_CONTAINER_NAME), timeout=20
    )


@pytest.mark.skipif(condition=in_docker(), reason="cannot run CLI tests in docker")
class TestCliContainerLifecycle:
    def test_start_wait_stop(self, runner, container_client, monkeypatch, caplog):
        monkeypatch.setenv("DEBUG", "1")
        result = runner.invoke(cli, ["start", "-d"])
        assert result.exit_code == 0
        assert "starting LocalStack" in result.output

        # ensure that all entrypoints could be imported
        assert "error importing entrypoint" not in caplog.text

        result = runner.invoke(cli, ["wait", "-t", "180"])
        assert result.exit_code == 0

        assert container_client.is_container_running(
            config.MAIN_CONTAINER_NAME
        ), "container name was not running after wait"

        health = requests.get(config.internal_service_url() + "/_localstack/health")
        assert health.ok, "health request did not return OK: %s" % health.text

        result = runner.invoke(cli, ["stop"])
        assert result.exit_code == 0

        with pytest.raises(requests.ConnectionError):
            requests.get(config.internal_service_url() + "/_localstack/health")

    def test_wait_timeout_raises_exception(self, runner, container_client):
        # assume a wait without start fails
        result = runner.invoke(cli, ["wait", "-t", "0.5"])
        assert result.exit_code != 0

    def test_logs(self, runner, container_client):
        result = runner.invoke(cli, ["logs"])
        assert result.exit_code != 0

        runner.invoke(cli, ["start", "-d"])
        runner.invoke(cli, ["wait", "-t", "180"])

        result = runner.invoke(cli, ["logs"])
        assert constants.READY_MARKER_OUTPUT in result.output

    def test_status_services(self, runner):
        result = runner.invoke(cli, ["status", "services"])
        assert result.exit_code != 0
        assert "could not connect to LocalStack health endpoint" in result.output

        runner.invoke(cli, ["start", "-d"])
        runner.invoke(cli, ["wait", "-t", "180"])

        result = runner.invoke(cli, ["status", "services"])

        # just a smoke test
        assert "dynamodb" in result.output
        for line in result.output.splitlines():
            if "dynamodb" in line:
                assert "available" in line

    def test_custom_docker_flags(self, runner, tmp_path, monkeypatch, container_client):
        volume = tmp_path / "volume"
        volume.mkdir()

        monkeypatch.setattr(config, "DOCKER_FLAGS", f"-p 42069 -v {volume}:{volume}")

        runner.invoke(cli, ["start", "-d"])
        runner.invoke(cli, ["wait", "-t", "180"])

        inspect = container_client.inspect_container(config.MAIN_CONTAINER_NAME)
        assert "42069/tcp" in inspect["HostConfig"]["PortBindings"]
        assert f"{volume}:{volume}" in inspect["HostConfig"]["Binds"]

    def test_volume_dir_mounted_correctly(self, runner, tmp_path, monkeypatch, container_client):
        volume_dir = tmp_path / "volume"

        # set different directories and make sure they are mounted correctly
        monkeypatch.setenv("LOCALSTACK_VOLUME_DIR", str(volume_dir))
        monkeypatch.setattr(config, "VOLUME_DIR", str(volume_dir))

        runner.invoke(cli, ["start", "-d"])
        runner.invoke(cli, ["wait", "-t", "60"])

        # check that mounts were created correctly
        inspect = container_client.inspect_container(config.MAIN_CONTAINER_NAME)
        binds = inspect["HostConfig"]["Binds"]
        assert f"{volume_dir}:{constants.DEFAULT_VOLUME_DIR}" in binds

    def test_container_starts_non_root(self, runner, monkeypatch, container_client):
        user = "localstack"
        monkeypatch.setattr(config, "DOCKER_FLAGS", f"--user={user}")

        if in_ci() and os.path.exists("/home/runner"):
            volume_dir = "/home/runner/.cache/localstack/volume/"
            mkdir(volume_dir)
            run(["sudo", "chmod", "-R", "777", volume_dir])

        runner.invoke(cli, ["start", "-d"])
        runner.invoke(cli, ["wait", "-t", "180"])

        cmd = ["awslocal", "stepfunctions", "list-state-machines"]
        output = container_client.exec_in_container(config.MAIN_CONTAINER_NAME, cmd)
        result = json.loads(output[0])
        assert "stateMachines" in result

        output = container_client.exec_in_container(config.MAIN_CONTAINER_NAME, ["ps", "-fu", user])
        assert "localstack-supervisor" in to_str(output[0])

    def test_start_cli_within_container(self, runner, container_client):
        output = container_client.run_container(
            # CAVEAT: Updates to the Docker image are not immediately reflected when using the latest image from
            # DockerHub in the CI.
            # Re-build the Docker image locally with `bin/docker-helper.sh build` in community for local testing.
            "localstack/localstack",
            remove=True,
            entrypoint="",
            command=["bin/localstack", "start", "-d"],
            volumes=[
                ("/var/run/docker.sock", "/var/run/docker.sock"),
                (MODULE_MAIN_PATH, "/opt/code/localstack/localstack"),
            ],
            env_vars={"LOCALSTACK_VOLUME_DIR": "/tmp/ls-volume"},
        )
        stdout = to_str(output[0])
        assert "starting LocalStack" in stdout
        assert "detaching" in stdout

        # assert that container is running
        runner.invoke(cli, ["wait", "-t", "180"])


class TestImports:
    def test_cli_imports_from_bootstrap(self):
        """
        Simple test to assert we can import different code paths within the `bootstrap` module with
        different Python versions (covered by CLI tests which are parametrized with different versions).
        """
        from localstack.pro.core import bootstrap

        # TODO: fix in upstream and remove these checks!
        try:
            from localstack.aws import connect  # noqa
        except ImportError as e:
            if "functools" in str(e):
                pytest.skip("Skipping, as `functools.cache` not supported in Python <=3.8")
            raise
        except TypeError as e:
            if "unsupported operand type" in str(e):
                pytest.skip("Skipping, as union type hints with `|` not supported in Python <=3.9")
            raise

        def import_all_recursively(module):
            module_path = getattr(module, "__path__", None)
            if not module_path:
                return
            for loader, name, is_pkg in pkgutil.walk_packages(module_path):
                sub_module = importlib.import_module(f"{module.__name__}.{name}")
                import_all_recursively(sub_module)

        import_all_recursively(bootstrap)
