import re
from typing import Optional

from jsonpath_ng.ext import parse
from localstack.testing.snapshots.transformer_utility import TransformerUtility
from localstack_snapshot.snapshots.transformer import (
    SNAPSHOT_LOGGER,
    JsonpathTransformer,
    KeyValueBasedTransformer,
    TransformContext,
)


class TransformerUtilityExt(TransformerUtility):
    """
    Transformer Utility for ext: can be used to group common transformers for APIs.
    This class should group all APIs that are only available in LocalStack Pro.
    """

    @staticmethod
    def dms_api():
        return [
            TransformerUtility.key_value(
                "AvailabilityZone", "availability-zone", reference_replacement=False
            ),
            JsonpathTransformer(
                "$..SubnetAvailabilityZone.Name", "subnet-name", replace_reference=False
            ),
            TransformerUtility.key_value("ServerName", reference_replacement=False),
            TransformerUtility.key_value("EndpointIdentifier"),
            TransformerUtility.key_value("ReplicationInstanceIdentifier"),
            TransformerUtility.key_value("PreferredMaintenanceWindow"),
            TransformerUtility.key_value("ReplicationTaskIdentifier"),
            TransformerUtility.key_value("ReplicationInstancePrivateIpAddress"),
            TransformerUtility.key_value("ReplicationInstancePublicIpAddress"),
            TransformerUtility.key_value("VpcSecurityGroupId"),
            TransformerUtility.key_value("VpcId"),
            TransformerUtility.key_value("ElapsedTimeMillis", reference_replacement=False),
            TransformerUtility.key_value("CloudWatchLogGroup"),
            KeyValueBasedTransformer(_dms_autogenerated_arn_transformer, "identifier"),
            JsonpathTransformer("$..Endpoints..Port", "<port>", replace_reference=False),
            KeyValueBasedTransformer(
                _dms_recovery_checkpoint_transformer,
                replacement="recovery-checkpoint",
                replace_reference=False,
            ),
            KeyValueBasedTransformer(
                _dms_cdc_start_position_transformer,
                replacement="start-position-date",
                replace_reference=False,
            ),
            KeyValueBasedTransformer(
                _dms_transaction_id, replacement="12345678901", replace_reference=False
            ),
            TransformerUtility.resource_name(),
            KeyValueBasedTransformer(
                _dms_timestamp_transformer,
                "<timestamp:2022-07-13T13:48:01.000Z>",
                replace_reference=False,
            ),
        ]

    @staticmethod
    def ecs_api():
        return [TransformerUtility.resource_name(), TransformerUtility.key_value("clusterName")]

    @staticmethod
    def rds_api():
        return [
            TransformerUtility.key_value(
                "DBInstanceIdentifier", value_replacement="db-instance-id"
            ),
            TransformerUtility.key_value("DBClusterIdentifier", value_replacement="db-cluster-id"),
            TransformerUtility.key_value("SubnetIdentifier", value_replacement="subnet-id"),
            TransformerUtility.key_value("GlobalClusterIdentifier"),
            TransformerUtility.key_value("GlobalClusterResourceId"),
            KeyValueBasedTransformer(
                _preferred_backup_window_transformer, "<backup-window>", replace_reference=False
            ),
            KeyValueBasedTransformer(
                _preferred_maintenance_window_transformer,
                "<maintenance-window>",
                replace_reference=False,
            ),
            TransformerUtility.key_value("VpcSecurityGroupId"),
            TransformerUtility.key_value("VpcId"),
            TransformerUtility.key_value("SubnetIdentifier"),
            JsonpathTransformer("$..SubnetAvailabilityZone.Name", "subnet-name"),
            TransformerUtility.key_value("DbiResourceId"),
            TransformerUtility.key_value("CACertificateIdentifier"),
            TransformerUtility.key_value("HostedZoneId"),
            TransformerUtility.key_value("DbClusterResourceId"),
            TransformerUtility.key_value("KmsKeyId"),
            TransformerUtility.key_value("Port", reference_replacement=False),
            TransformerUtility.key_value("ReaderEndpoint", reference_replacement=False),
            TransformerUtility.key_value("Address", reference_replacement=False),
        ]

    def cloudtrail_api(self):
        return [
            TransformerUtility.key_value("EventId"),
            TransformerUtility.key_value("EventTime", "datetime", reference_replacement=False),
            TransformerUtility.key_value("AccessKeyId"),
            TransformerUtility.key_value("principalId"),
            TransformerUtility.key_value("Username"),
            # from the "CloudTrailEvent" details:
            TransformerUtility.key_value("userAgent", reference_replacement=False),
            TransformerUtility.key_value("recipientAccountId"),
            TransformerUtility.key_value("sourceIPAddress", reference_replacement=False),
            TransformerUtility.key_value("x-amz-id-2", reference_replacement=False),
            TransformerUtility.key_value("NextToken", reference_replacement=False),
        ]

    @staticmethod
    def appsync_api():
        return [
            TransformerUtility.key_value("functionId"),
            TransformerUtility.key_value("apiId"),
            TransformerUtility.key_value("name"),
            JsonpathTransformer("$..uris.GRAPHQL", "graphql-endpoint"),
            JsonpathTransformer("$..uris.REALTIME", "websocket-endpoint"),
        ]

    @staticmethod
    def cognito_idp_api():
        return [
            TransformerUtility.key_value("userPoolId"),
            TransformerUtility.key_value("sub"),
            TransformerUtility.key_value("userName"),
            TransformerUtility.key_value("clientId"),
        ]

    @staticmethod
    def redshift_api():
        return [
            TransformerUtility.key_value("AvailabilityZone"),
            TransformerUtility.key_value("AvailabilityZoneRelocationStatus"),
            TransformerUtility.key_value("ClusterIdentifier"),
            TransformerUtility.key_value("ClusterNamespaceArn"),
            TransformerUtility.key_value("ClusterNodes", reference_replacement=False),
            TransformerUtility.key_value("ClusterPublicKey", reference_replacement=False),
            TransformerUtility.key_value("ClusterSubnetGroupName"),
            TransformerUtility.key_value("Address", reference_replacement=False),
            TransformerUtility.key_value("Port", reference_replacement=False),
            TransformerUtility.key_value("PreferredMaintenanceWindow", reference_replacement=False),
            TransformerUtility.key_value("VpcId"),
            TransformerUtility.key_value("VpcSecurityGroupId"),
            TransformerUtility.key_value("KmsKeyId", reference_replacement=False),
        ]

    @staticmethod
    def logs_api():
        # FIXME: reference_replacement seems to be broken here
        return [
            TransformerUtility.key_value("eventId", reference_replacement=False),
            TransformerUtility.key_value("ingestionTime", reference_replacement=False),
            TransformerUtility.key_value("logStreamName", reference_replacement=False),
            TransformerUtility.key_value("timestamp", reference_replacement=False),
            TransformerUtility.key_value("nextToken", reference_replacement=False),
        ]

    @staticmethod
    def glue_api():
        return [
            # Generic transformer rules.
            TransformerUtility.key_value(
                "JobName", value_replacement="job-name", reference_replacement=True
            ),
            TransformerUtility.key_value(
                "JobRunId", value_replacement="job-run-id", reference_replacement=True
            ),
            # Transformer rules for JobRun fields.
            TransformerUtility.key_value(
                "AllocatedCapacity", "allocated_capacity", reference_replacement=False
            ),
            TransformerUtility.key_value(
                "CompletedOn", "completed_on", reference_replacement=False
            ),
            TransformerUtility.key_value(
                "ExecutionTime", "execution_time", reference_replacement=False
            ),
            TransformerUtility.key_value("Id", "id", reference_replacement=True),
            TransformerUtility.key_value(
                "LastModifiedOn", "last_modified_on", reference_replacement=False
            ),
            TransformerUtility.key_value(
                "MaxCapacity", "max_capacity", reference_replacement=False
            ),
            TransformerUtility.key_value("StartedOn", "started_on", reference_replacement=False),
        ]

    @staticmethod
    def structure(jsonpath: str):
        return StructureTransformer(jsonpath)


def _preferred_backup_window_transformer(key: str, val: str) -> str:
    if key == "PreferredBackupWindow":
        match = re.match(r"\d{2}:\d{2}-\d{2}:\d{2}", val)
        if match:
            return val
    return None


def _preferred_maintenance_window_transformer(key: str, val: str) -> Optional[str]:
    if key == "PreferredMaintenanceWindow":
        match = re.match(r"[a-z]{3}:\d{2}:\d{2}-[a-z]{3}:\d{2}:\d{2}", val)
        if match:
            return val
    return None


def _dms_autogenerated_arn_transformer(key: str, val: str) -> Optional[str]:
    # the EndpointArn + ReplicationInstanceArn includes the resource-id which will be auto-generated if not set,
    # it used to be 39 chars long, but for some reason the returned values from AWS changed to 26 long
    if isinstance(val, str) and key in ("ReplicationInstanceArn", "EndpointArn"):
        match = re.match(".*:([A-Z0-9]{26,39})", val)
        if match:
            return match.groups()[-1]
        return None


def _dms_timestamp_transformer(key: str, val: str) -> Optional[str]:
    """dms follows a pattern for timestamps that has not been covered yet be the new timestamp transformers in the external lib
    adding it now may causes issues with existing snapshots. At some point we may need to consolidate this
    """
    if "timestamp" in key.lower() and isinstance(val, str):
        match = re.match(r"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{1,6}Z", val)
        if match:
            return val
    return None


def _dms_recovery_checkpoint_transformer(key: str, val: str) -> Optional[str]:
    if isinstance(val, str) and key == "RecoveryCheckpoint":
        # TODO maybe add more verification
        # example for value: 'checkpoint:V1#38#mysql-bin-changelog.000017:487:-1:514:73014444260:mysql-bin-changelog.000016:9523#0#0#*#0#133'
        if val.startswith("checkpoint:"):
            return val
    return None


def _dms_cdc_start_position_transformer(key: str, val: str) -> Optional[str]:
    if isinstance(val, str) and key == "CdcStartPosition":
        # pattern: 2024-02-12T11:13:38
        match = re.match("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}", val)
        if match:
            return val
    return None


def _dms_transaction_id(key: str, val: int) -> Optional[str]:
    if isinstance(val, int) and key == "transaction-id":
        # samples: 47244647739 or 77309413425 (not enough evidence not be sure it's always 11 chars
        value = str(val)
        match = re.match("[0-9]+", value)
        if match:
            return value
    return None


class StructureTransformer:
    def __init__(
        self,
        jsonpath: str,
    ) -> None:
        self.jsonpath = jsonpath

    def transform(self, input_data: dict, *, ctx: TransformContext) -> dict:
        pattern = parse(self.jsonpath)

        res = pattern.find(input_data)
        if not res:
            SNAPSHOT_LOGGER.debug(f"No match found for JsonPath '{self.jsonpath}'")
            return input_data
        for r in res:
            r.full_path.update(input_data, self._structure_replacement(r.value))

        return input_data

    def _structure_replacement(self, value):
        if isinstance(value, dict):
            result = {}
            for k, v in value.items():
                result[k] = self._structure_replacement(v)
            return result
        elif isinstance(value, list):
            return [self._structure_replacement(val) for val in value]
        else:
            return type(value).__name__
