import io
import json
import os
import textwrap

import aws_cdk as cdk
import aws_cdk.aws_lambda as awslambda
import aws_cdk.aws_rds as rds
import pytest
from localstack.testing.aws.util import is_aws_cloud
from localstack.testing.pytest import markers
from localstack.testing.scenario.cdk_lambda_helper import load_python_lambda_to_s3
from localstack.testing.scenario.provisioning import InfraProvisioner
from localstack.utils.strings import short_uid, to_bytes, to_str
from localstack.utils.sync import retry

from tests.aws.services.rds.test_rds import (
    TestRDSBase,
    _add_role_to_db_instance,
)


@markers.only_in_docker
@pytest.mark.parametrize("engine_version", ["11", "12", "13", "14", "15", "16"])
class TestRdsPostgresExtensions(TestRDSBase):
    STACK_OUTPUT = "PostgresExtensions"

    @pytest.fixture(scope="class")
    def infrastructure(self, aws_client, infrastructure_setup):
        """
        Stack sets up database instances for version 11,12,13,14,15,16 with autogenerated password
        One lambda is configured for each instance, that can be used to run queries
        outputs for each instance: lambda-function name, database identifier, database arn
        """
        infra = infrastructure_setup(namespace="PostgresExtension", force_synth=False)
        infra.add_custom_setup(
            lambda: load_python_lambda_to_s3(
                aws_client.s3,
                infra.get_asset_bucket(),
                "query-lambda-helper.zip",
                os.path.join(os.path.dirname(__file__), "functions/helper_postgres_query.py"),
                additional_python_packages=["pg8000"],
            )
        )
        stack = cdk.Stack(infra.cdk_app, self.STACK_OUTPUT)
        vpc = cdk.aws_ec2.Vpc(stack, "vpc", restrict_default_security_group=False)
        cdk.aws_ec2.SecurityGroup(stack, "securityGroup", vpc=vpc)
        database_versions = {
            "11": rds.PostgresEngineVersion.VER_11,
            "12": rds.PostgresEngineVersion.VER_12,
            "13": rds.PostgresEngineVersion.VER_13,
            "14": rds.PostgresEngineVersion.VER_14,
            "15": rds.PostgresEngineVersion.VER_15,
            "16": rds.PostgresEngineVersion.VER_16,
        }
        bucket = cdk.aws_s3.Bucket.from_bucket_name(
            stack,
            "bucket_name",
            bucket_name=InfraProvisioner.get_asset_bucket_cdk(stack),
        )

        for name, engine_version in database_versions.items():
            rds_database = rds.DatabaseInstance(
                stack,
                id=f"PostgresDBInstance-{name}",
                vpc=vpc,
                credentials=rds.Credentials.from_username(username="myuser"),
                removal_policy=cdk.RemovalPolicy.DESTROY,
                engine=rds.DatabaseInstanceEngine.postgres(version=engine_version),
                backup_retention=cdk.Duration.days(0),
                database_name="mydb",
            )

            helper_func = awslambda.Function(
                stack,
                f"HelperLambda-{name}",
                runtime=awslambda.Runtime.PYTHON_3_10,
                handler="index.handler",
                vpc=vpc,
                code=awslambda.S3Code(bucket=bucket, key="query-lambda-helper.zip"),
                environment={"RDS_SECRET": rds_database.secret.secret_arn},
            )
            rds_database.secret.grant_read(helper_func)
            rds_database.connections.allow_default_port_from_any_ipv4("Open to the world")
            cdk.CfnOutput(stack, f"HelperFunc{name}", value=helper_func.function_name)
            cdk.CfnOutput(stack, f"DatabaseArn{name}", value=rds_database.instance_arn)
            cdk.CfnOutput(
                stack, f"DatabaseIdentifier{name}", value=rds_database.instance_identifier
            )

        with infra.provisioner(skip_teardown=False) as prov:
            yield prov

    @markers.aws.validated
    def test_postgres_s3_extension_helpers(self, infrastructure, aws_client, engine_version):
        if engine_version == "11":
            return pytest.skip("postgres version 11 does not support aws_lambda on AWS")

        outputs = infrastructure.get_stack_outputs(self.STACK_OUTPUT)

        fun_name = outputs.get(f"HelperFunc{engine_version}")

        # test create_s3_uri function
        query = "SELECT aws_commons.create_s3_uri('bucket123', 'filepath123', 'us-west-2')"
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": query})),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not create_s3_uri with engine version {engine_version}: {result.get('error')}"

        result = result.get("results")[0]
        assert "bucket123" in result[0]
        assert "filepath123" in result[0]
        assert "us-west-2" in result[0]

        # test create_aws_credentials function
        query = "SELECT aws_commons.create_aws_credentials('access123', 'secret123', 'token123')"
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": query})),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not create_s3_uri with engine version {engine_version}: {result.get('error')}"

        result = result.get("results")[0]
        assert "access123" in result[0]
        assert "secret123" in result[0]
        assert "token123" in result[0]

    @markers.aws.validated
    def test_postgres_extension_table_import_from_s3(
        self,
        infrastructure,
        aws_client,
        engine_version,
        s3_bucket,
        snapshot,
        create_iam_role_s3_access_lambda_invoke_for_db,
        cleanups,
    ):
        # setup role
        outputs = infrastructure.get_stack_outputs(self.STACK_OUTPUT)
        instance_id = outputs.get(f"DatabaseIdentifier{engine_version}")
        instance_arn = outputs.get(f"DatabaseArn{engine_version}")
        role = create_iam_role_s3_access_lambda_invoke_for_db(source_arn=instance_arn)
        _add_role_to_db_instance(aws_client.rds, instance_id, "s3Import", role)
        cleanups.append(
            lambda: aws_client.rds.remove_role_from_db_instance(
                DBInstanceIdentifier=instance_id, FeatureName="s3Import", RoleArn=role
            )
        )
        # create test CSV file
        csv_file = textwrap.dedent(
            """
        t1.1, t1.2, t1.3
        t2.1, t2.2, t2.3
        t3.1, t3.2, t3.3
        """
        ).strip()
        s3_key = "test/data.csv"
        aws_client.s3.put_object(Bucket=s3_bucket, Key=s3_key, Body=io.BytesIO(to_bytes(csv_file)))

        fun_name = outputs.get(f"HelperFunc{engine_version}")

        # create target test table
        table_name = f"t1{short_uid()}"
        query = f"CREATE TABLE {table_name} (c1 text, c2 text, c3 text)"

        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": query})),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not table with engine version {engine_version}: {result.get('error')}"

        # activate the extension - required for AWS parity
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(
                json.dumps({"sqlQuery": "CREATE EXTENSION IF NOT EXISTS aws_s3 CASCADE"})
            ),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not activate extension with engine version {engine_version}: {result.get('error')}"

        # run import query from S3
        query = f"""
        SELECT aws_s3.table_import_from_s3(
            '{table_name}',
            'c1, c2, c3',
            '(format csv)',
            aws_commons.create_s3_uri('{s3_bucket}', '{s3_key}', '{aws_client.s3.meta.region_name}')
        )
        """

        def _run_import():
            result = aws_client.lambda_.invoke(
                FunctionName=fun_name,
                Payload=to_bytes(json.dumps({"sqlQuery": query})),
            )
            result = json.load(result["Payload"])
            assert result.get("status") == "SUCCESS"
            return result.get("results")

        # IAM is eventually consistent - retry to fix "credentials stored with the DB cluster can’t be accessed"
        result = retry(_run_import, retries=30, sleep=5 if is_aws_cloud() else 1)

        # returns the number of inserted rows
        snapshot.match("import-result", result)

        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": f"SELECT * FROM {table_name}"})),
        )
        result = json.load(result["Payload"])
        assert result.get("status") == "SUCCESS"
        snapshot.match("select-result", result.get("results"))

    @markers.aws.validated
    def test_postgres_extension_query_export_to_s3(
        self,
        aws_client,
        infrastructure,
        engine_version,
        s3_bucket,
        create_iam_role_s3_access_lambda_invoke_for_db,
        cleanups,
    ):
        # setup role
        outputs = infrastructure.get_stack_outputs(self.STACK_OUTPUT)
        instance_id = outputs.get(f"DatabaseIdentifier{engine_version}")
        instance_arn = outputs.get(f"DatabaseArn{engine_version}")
        role = create_iam_role_s3_access_lambda_invoke_for_db(source_arn=instance_arn)
        _add_role_to_db_instance(aws_client.rds, instance_id, "s3Export", role)
        cleanups.append(
            lambda: aws_client.rds.remove_role_from_db_instance(
                DBInstanceIdentifier=instance_id, FeatureName="s3Export", RoleArn=role
            )
        )

        fun_name = outputs.get(f"HelperFunc{engine_version}")

        # create test table
        table_name = f"t1{short_uid()}"
        query = f"CREATE TABLE {table_name} (c1 text, c2 text, c3 text)"
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": query})),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not create table on database engine version {engine_version}: {result.get('error')}"

        # insert test data
        num_rows = 3
        for i in range(1, num_rows + 1):
            query = f"INSERT INTO {table_name} (c1, c2, c3) VALUES ('t{i}.1', 't{i}.2', 't{i}.3')"
            result = aws_client.lambda_.invoke(
                FunctionName=fun_name,
                Payload=to_bytes(json.dumps({"sqlQuery": query})),
            )
            result = json.load(result["Payload"])
            assert (
                result.get("status") == "SUCCESS"
            ), f"could not insert data {i} into database engine version {engine_version}: {result.get('error')}"

        # activate the extension - required for AWS parity
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(
                json.dumps({"sqlQuery": "CREATE EXTENSION IF NOT EXISTS aws_s3 CASCADE;"})
            ),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not create extension aws_s3 database engine version {engine_version}: {result.get('error')}"

        # run query to export table data to S3
        s3_key = "result/data.csv"
        query = f"""
        SELECT * FROM aws_s3.query_export_to_s3(
            'SELECT * FROM {table_name}',
            aws_commons.create_s3_uri('{s3_bucket}', '{s3_key}', '{aws_client.s3.meta.region_name}'),
            options :='FORMAT csv'
        )"""

        def _run_export():
            # FIXME: for some reason this query times out on AWS for postgres 16 (works with all others)
            #  I assume this is due to a specific setting, but I couldn't find it yet
            result = aws_client.lambda_.invoke(
                FunctionName=fun_name,
                Payload=to_bytes(json.dumps({"sqlQuery": query})),
            )
            result = json.load(result["Payload"])
            assert (
                result.get("status") == "SUCCESS"
            ), f"could not export engine version {engine_version}: {result.get('error')}"
            return result

        # IAM is eventually consistent - retry to fix "credentials stored with the DB cluster can’t be accessed"
        retry(_run_export, retries=60 if is_aws_cloud() else 10, sleep=5 if is_aws_cloud() else 1)

        # get file from S3, assert correct content
        result = aws_client.s3.get_object(Bucket=s3_bucket, Key=s3_key)
        content = to_str(result["Body"].read())
        for i in range(1, num_rows + 1):
            assert f"t{i}.1,t{i}.2,t{i}.3" in content

    @markers.aws.validated
    def test_installed_postgres_version(self, aws_client, infrastructure, engine_version):
        outputs = infrastructure.get_stack_outputs(self.STACK_OUTPUT)

        fun_name = outputs.get(f"HelperFunc{engine_version}")
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps({"sqlQuery": "SELECT version();"})),
        )
        result = json.load(result["Payload"])
        assert (
            result.get("status") == "SUCCESS"
        ), f"could not get version from engine {engine_version}: {result.get('error')}"

        # assert major version is the same
        assert f"PostgreSQL {engine_version.split('.')[0]}" in result.get("results")[0][0]

    @markers.aws.validated
    def test_postgis_extension(self, aws_client, infrastructure, engine_version, snapshot):
        # i run the tests manually against LS using the versions:
        # VER_11_22, VER_12_17, VER_13_13, VER_14_10, VER_15_5, VER_16_1
        outputs = infrastructure.get_stack_outputs(self.STACK_OUTPUT)

        fun_name = outputs.get(f"HelperFunc{engine_version}")
        # extensions should all be available for postgis:
        # https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Appendix.PostgreSQL.CommonDBATasks.PostGIS.html

        extensions = [
            "postgis",
            "postgis_raster",
            "fuzzystrmatch",
            "postgis_tiger_geocoder",
            "postgis_topology",
            "address_standardizer_data_us",
        ]

        for extension in extensions:
            result = aws_client.lambda_.invoke(
                FunctionName=fun_name,
                Payload=to_bytes(
                    json.dumps({"sqlQuery": f"CREATE EXTENSION IF NOT EXISTS {extension};"})
                ),
            )
            result = json.load(result["Payload"])
            assert (
                result.get("status") == "SUCCESS"
            ), f"could not create extension {extension}: {result.get('error')}"

        # sample query for normalize
        verify_postgis_normalize = {
            "sqlQuery": """SELECT address, streetname, streettypeabbrev, zip
                            FROM normalize_address('1 Devonshire Place, Boston, MA 02109') AS na;"""
        }
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps(verify_postgis_normalize)),
        )
        result = json.load(result["Payload"])
        snapshot.match("postgis_normalize", result)

        # use a random name here to ensure it can be re-run
        topo_name = "my_new_topo_{short_uid()"
        snapshot.add_transformer(snapshot.transform.regex(topo_name, "my_new_topo"))
        verify_postgis_topology = {
            "sqlQuery": f"SELECT topology.createtopology('my_new_topo_{short_uid()}',26986,0.5);"
        }
        result = aws_client.lambda_.invoke(
            FunctionName=fun_name,
            Payload=to_bytes(json.dumps(verify_postgis_topology)),
        )
        result = json.load(result["Payload"])
        # the query returns 1 or 2 depending on the state of the database, which seems to be expected
        # only verify that the query runs successfully
        assert (
            result.get("status") == "SUCCESS"
        ), f"postgis query failed on engine {engine_version}: {result.get('error')}"
