from datetime import datetime
from enum import StrEnum
from typing import TypedDict

from localstack.aws.api import RequestContext, ServiceException, ServiceRequest, handler

AWSAccountId = str
AmazonResourceName = str
Arn = str
AttrKey = str
AttrValue = str
Code = str
DiscoverMaxResults = int
ErrorMessage = str
FailureThreshold = int
FilterValue = str
InstanceId = str
MaxResults = int
Message = str
NamespaceName = str
NamespaceNameHttp = str
NamespaceNamePrivate = str
NamespaceNamePublic = str
NextToken = str
OperationId = str
ResourceCount = int
ResourceDescription = str
ResourceId = str
ResourcePath = str
ServiceAttributeKey = str
ServiceAttributeValue = str
ServiceName = str
TagKey = str
TagValue = str


class CustomHealthStatus(StrEnum):
    HEALTHY = "HEALTHY"
    UNHEALTHY = "UNHEALTHY"


class FilterCondition(StrEnum):
    EQ = "EQ"
    IN = "IN"
    BETWEEN = "BETWEEN"
    BEGINS_WITH = "BEGINS_WITH"


class HealthCheckType(StrEnum):
    HTTP = "HTTP"
    HTTPS = "HTTPS"
    TCP = "TCP"


class HealthStatus(StrEnum):
    HEALTHY = "HEALTHY"
    UNHEALTHY = "UNHEALTHY"
    UNKNOWN = "UNKNOWN"


class HealthStatusFilter(StrEnum):
    HEALTHY = "HEALTHY"
    UNHEALTHY = "UNHEALTHY"
    ALL = "ALL"
    HEALTHY_OR_ELSE_ALL = "HEALTHY_OR_ELSE_ALL"


class NamespaceFilterName(StrEnum):
    TYPE = "TYPE"
    NAME = "NAME"
    HTTP_NAME = "HTTP_NAME"
    RESOURCE_OWNER = "RESOURCE_OWNER"


class NamespaceType(StrEnum):
    DNS_PUBLIC = "DNS_PUBLIC"
    DNS_PRIVATE = "DNS_PRIVATE"
    HTTP = "HTTP"


class OperationFilterName(StrEnum):
    NAMESPACE_ID = "NAMESPACE_ID"
    SERVICE_ID = "SERVICE_ID"
    STATUS = "STATUS"
    TYPE = "TYPE"
    UPDATE_DATE = "UPDATE_DATE"


class OperationStatus(StrEnum):
    SUBMITTED = "SUBMITTED"
    PENDING = "PENDING"
    SUCCESS = "SUCCESS"
    FAIL = "FAIL"


class OperationTargetType(StrEnum):
    NAMESPACE = "NAMESPACE"
    SERVICE = "SERVICE"
    INSTANCE = "INSTANCE"


class OperationType(StrEnum):
    CREATE_NAMESPACE = "CREATE_NAMESPACE"
    DELETE_NAMESPACE = "DELETE_NAMESPACE"
    UPDATE_NAMESPACE = "UPDATE_NAMESPACE"
    UPDATE_SERVICE = "UPDATE_SERVICE"
    REGISTER_INSTANCE = "REGISTER_INSTANCE"
    DEREGISTER_INSTANCE = "DEREGISTER_INSTANCE"


class RecordType(StrEnum):
    SRV = "SRV"
    A = "A"
    AAAA = "AAAA"
    CNAME = "CNAME"


class RoutingPolicy(StrEnum):
    MULTIVALUE = "MULTIVALUE"
    WEIGHTED = "WEIGHTED"


class ServiceFilterName(StrEnum):
    NAMESPACE_ID = "NAMESPACE_ID"
    RESOURCE_OWNER = "RESOURCE_OWNER"


class ServiceType(StrEnum):
    HTTP = "HTTP"
    DNS_HTTP = "DNS_HTTP"
    DNS = "DNS"


class ServiceTypeOption(StrEnum):
    HTTP = "HTTP"


class CustomHealthNotFound(ServiceException):
    """The health check for the instance that's specified by ``ServiceId`` and
    ``InstanceId`` isn't a custom health check.
    """

    code: str = "CustomHealthNotFound"
    sender_fault: bool = False
    status_code: int = 400


class DuplicateRequest(ServiceException):
    """The operation is already in progress."""

    code: str = "DuplicateRequest"
    sender_fault: bool = False
    status_code: int = 400
    DuplicateOperationId: OperationId | None


class InstanceNotFound(ServiceException):
    """No instance exists with the specified ID, or the instance was recently
    registered, and information about the instance hasn't propagated yet.
    """

    code: str = "InstanceNotFound"
    sender_fault: bool = False
    status_code: int = 400


class InvalidInput(ServiceException):
    """One or more specified values aren't valid. For example, a required value
    might be missing, a numeric value might be outside the allowed range, or
    a string value might exceed length constraints.
    """

    code: str = "InvalidInput"
    sender_fault: bool = False
    status_code: int = 400


class NamespaceAlreadyExists(ServiceException):
    """The namespace that you're trying to create already exists."""

    code: str = "NamespaceAlreadyExists"
    sender_fault: bool = False
    status_code: int = 400
    CreatorRequestId: ResourceId | None
    NamespaceId: ResourceId | None


class NamespaceNotFound(ServiceException):
    """No namespace exists with the specified ID."""

    code: str = "NamespaceNotFound"
    sender_fault: bool = False
    status_code: int = 400


class OperationNotFound(ServiceException):
    """No operation exists with the specified ID."""

    code: str = "OperationNotFound"
    sender_fault: bool = False
    status_code: int = 400


class RequestLimitExceeded(ServiceException):
    """The operation can't be completed because you've reached the quota for
    the number of requests. For more information, see `Cloud Map API request
    throttling
    quota <https://docs.aws.amazon.com/cloud-map/latest/dg/throttling.html>`__
    in the *Cloud Map Developer Guide*.
    """

    code: str = "RequestLimitExceeded"
    sender_fault: bool = False
    status_code: int = 400


class ResourceInUse(ServiceException):
    """The specified resource can't be deleted because it contains other
    resources. For example, you can't delete a service that contains any
    instances.
    """

    code: str = "ResourceInUse"
    sender_fault: bool = False
    status_code: int = 400


class ResourceLimitExceeded(ServiceException):
    """The resource can't be created because you've reached the quota on the
    number of resources.
    """

    code: str = "ResourceLimitExceeded"
    sender_fault: bool = False
    status_code: int = 400


class ResourceNotFoundException(ServiceException):
    """The operation can't be completed because the resource was not found."""

    code: str = "ResourceNotFoundException"
    sender_fault: bool = False
    status_code: int = 400


class ServiceAlreadyExists(ServiceException):
    """The service can't be created because a service with the same name
    already exists.
    """

    code: str = "ServiceAlreadyExists"
    sender_fault: bool = False
    status_code: int = 400
    CreatorRequestId: ResourceId | None
    ServiceId: ResourceId | None
    ServiceArn: Arn | None


class ServiceAttributesLimitExceededException(ServiceException):
    """The attribute can't be added to the service because you've exceeded the
    quota for the number of attributes you can add to a service.
    """

    code: str = "ServiceAttributesLimitExceededException"
    sender_fault: bool = False
    status_code: int = 400


class ServiceNotFound(ServiceException):
    """No service exists with the specified ID."""

    code: str = "ServiceNotFound"
    sender_fault: bool = False
    status_code: int = 400


class TooManyTagsException(ServiceException):
    """The list of tags on the resource is over the quota. The maximum number
    of tags that can be applied to a resource is 50.
    """

    code: str = "TooManyTagsException"
    sender_fault: bool = False
    status_code: int = 400
    ResourceName: AmazonResourceName | None


Attributes = dict[AttrKey, AttrValue]


class Tag(TypedDict, total=False):
    """A custom key-value pair that's associated with a resource."""

    Key: TagKey
    Value: TagValue


TagList = list[Tag]


class CreateHttpNamespaceRequest(ServiceRequest):
    Name: NamespaceNameHttp
    CreatorRequestId: ResourceId | None
    Description: ResourceDescription | None
    Tags: TagList | None


class CreateHttpNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


RecordTTL = int


class SOA(TypedDict, total=False):
    """Start of Authority (SOA) properties for a public or private DNS
    namespace.
    """

    TTL: RecordTTL


class PrivateDnsPropertiesMutable(TypedDict, total=False):
    """DNS properties for the private DNS namespace."""

    SOA: SOA


class PrivateDnsNamespaceProperties(TypedDict, total=False):
    """DNS properties for the private DNS namespace."""

    DnsProperties: PrivateDnsPropertiesMutable


class CreatePrivateDnsNamespaceRequest(ServiceRequest):
    Name: NamespaceNamePrivate
    CreatorRequestId: ResourceId | None
    Description: ResourceDescription | None
    Vpc: ResourceId
    Tags: TagList | None
    Properties: PrivateDnsNamespaceProperties | None


class CreatePrivateDnsNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class PublicDnsPropertiesMutable(TypedDict, total=False):
    """DNS properties for the public DNS namespace."""

    SOA: SOA


class PublicDnsNamespaceProperties(TypedDict, total=False):
    """DNS properties for the public DNS namespace."""

    DnsProperties: PublicDnsPropertiesMutable


class CreatePublicDnsNamespaceRequest(ServiceRequest):
    Name: NamespaceNamePublic
    CreatorRequestId: ResourceId | None
    Description: ResourceDescription | None
    Tags: TagList | None
    Properties: PublicDnsNamespaceProperties | None


class CreatePublicDnsNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class HealthCheckCustomConfig(TypedDict, total=False):
    """A complex type that contains information about an optional custom health
    check. A custom health check, which requires that you use a third-party
    health checker to evaluate the health of your resources, is useful in
    the following circumstances:

    -  You can't use a health check that's defined by ``HealthCheckConfig``
       because the resource isn't available over the internet. For example,
       you can use a custom health check when the instance is in an Amazon
       VPC. (To check the health of resources in a VPC, the health checker
       must also be in the VPC.)

    -  You want to use a third-party health checker regardless of where your
       resources are located.

    If you specify a health check configuration, you can specify either
    ``HealthCheckCustomConfig`` or ``HealthCheckConfig`` but not both.

    To change the status of a custom health check, submit an
    ``UpdateInstanceCustomHealthStatus`` request. Cloud Map doesn't monitor
    the status of the resource, it just keeps a record of the status
    specified in the most recent ``UpdateInstanceCustomHealthStatus``
    request.

    Here's how custom health checks work:

    #. You create a service.

    #. You register an instance.

    #. You configure a third-party health checker to monitor the resource
       that's associated with the new instance.

       Cloud Map doesn't check the health of the resource directly.

    #. The third-party health-checker determines that the resource is
       unhealthy and notifies your application.

    #. Your application submits an ``UpdateInstanceCustomHealthStatus``
       request.

    #. Cloud Map waits for 30 seconds.

    #. If another ``UpdateInstanceCustomHealthStatus`` request doesn't
       arrive during that time to change the status back to healthy, Cloud
       Map stops routing traffic to the resource.
    """

    FailureThreshold: FailureThreshold | None


class HealthCheckConfig(TypedDict, total=False):
    """*Public DNS and HTTP namespaces only.* A complex type that contains
    settings for an optional health check. If you specify settings for a
    health check, Cloud Map associates the health check with the records
    that you specify in ``DnsConfig``.

    If you specify a health check configuration, you can specify either
    ``HealthCheckCustomConfig`` or ``HealthCheckConfig`` but not both.

    Health checks are basic Route 53 health checks that monitor an Amazon
    Web Services endpoint. For information about pricing for health checks,
    see `Amazon Route 53
    Pricing <http://aws.amazon.com/route53/pricing/>`__.

    Note the following about configuring health checks.

    A and AAAA records
       If ``DnsConfig`` includes configurations for both ``A`` and ``AAAA``
       records, Cloud Map creates a health check that uses the IPv4 address
       to check the health of the resource. If the endpoint tthat's
       specified by the IPv4 address is unhealthy, Route 53 considers both
       the ``A`` and ``AAAA`` records to be unhealthy.

    CNAME records
       You can't specify settings for ``HealthCheckConfig`` when the
       ``DNSConfig`` includes ``CNAME`` for the value of ``Type``. If you
       do, the ``CreateService`` request will fail with an ``InvalidInput``
       error.

    Request interval
       A Route 53 health checker in each health-checking Amazon Web Services
       Region sends a health check request to an endpoint every 30 seconds.
       On average, your endpoint receives a health check request about every
       two seconds. However, health checkers don't coordinate with one
       another. Therefore, you might sometimes see several requests in one
       second that's followed by a few seconds with no health checks at all.

    Health checking regions
       Health checkers perform checks from all Route 53 health-checking
       Regions. For a list of the current Regions, see
       `Regions <https://docs.aws.amazon.com/Route53/latest/APIReference/API_HealthCheckConfig.html#Route53-Type-HealthCheckConfig-Regions>`__.

    Alias records
       When you register an instance, if you include the
       ``AWS_ALIAS_DNS_NAME`` attribute, Cloud Map creates a Route 53 alias
       record. Note the following:

       -  Route 53 automatically sets ``EvaluateTargetHealth`` to true for
          alias records. When ``EvaluateTargetHealth`` is true, the alias
          record inherits the health of the referenced Amazon Web Services
          resource. such as an ELB load balancer. For more information, see
          `EvaluateTargetHealth <https://docs.aws.amazon.com/Route53/latest/APIReference/API_AliasTarget.html#Route53-Type-AliasTarget-EvaluateTargetHealth>`__.

       -  If you include ``HealthCheckConfig`` and then use the service to
          register an instance that creates an alias record, Route 53
          doesn't create the health check.

    Charges for health checks
       Health checks are basic Route 53 health checks that monitor an Amazon
       Web Services endpoint. For information about pricing for health
       checks, see `Amazon Route 53
       Pricing <http://aws.amazon.com/route53/pricing/>`__.
    """

    Type: HealthCheckType
    ResourcePath: ResourcePath | None
    FailureThreshold: FailureThreshold | None


class DnsRecord(TypedDict, total=False):
    """A complex type that contains information about the Route 53 DNS records
    that you want Cloud Map to create when you register an instance.
    """

    Type: RecordType
    TTL: RecordTTL


DnsRecordList = list[DnsRecord]


class DnsConfig(TypedDict, total=False):
    """A complex type that contains information about the Amazon Route 53 DNS
    records that you want Cloud Map to create when you register an instance.
    """

    NamespaceId: ResourceId | None
    RoutingPolicy: RoutingPolicy | None
    DnsRecords: DnsRecordList


class CreateServiceRequest(ServiceRequest):
    Name: ServiceName
    NamespaceId: Arn | None
    CreatorRequestId: ResourceId | None
    Description: ResourceDescription | None
    DnsConfig: DnsConfig | None
    HealthCheckConfig: HealthCheckConfig | None
    HealthCheckCustomConfig: HealthCheckCustomConfig | None
    Tags: TagList | None
    Type: ServiceTypeOption | None


Timestamp = datetime


class Service(TypedDict, total=False):
    """A complex type that contains information about the specified service."""

    Id: ResourceId | None
    Arn: Arn | None
    ResourceOwner: AWSAccountId | None
    Name: ServiceName | None
    NamespaceId: ResourceId | None
    Description: ResourceDescription | None
    InstanceCount: ResourceCount | None
    DnsConfig: DnsConfig | None
    Type: ServiceType | None
    HealthCheckConfig: HealthCheckConfig | None
    HealthCheckCustomConfig: HealthCheckCustomConfig | None
    CreateDate: Timestamp | None
    CreatorRequestId: ResourceId | None
    CreatedByAccount: AWSAccountId | None


class CreateServiceResponse(TypedDict, total=False):
    Service: Service | None


class DeleteNamespaceRequest(ServiceRequest):
    Id: Arn


class DeleteNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


ServiceAttributeKeyList = list[ServiceAttributeKey]


class DeleteServiceAttributesRequest(ServiceRequest):
    ServiceId: Arn
    Attributes: ServiceAttributeKeyList


class DeleteServiceAttributesResponse(TypedDict, total=False):
    pass


class DeleteServiceRequest(ServiceRequest):
    Id: Arn


class DeleteServiceResponse(TypedDict, total=False):
    pass


class DeregisterInstanceRequest(ServiceRequest):
    ServiceId: Arn
    InstanceId: ResourceId


class DeregisterInstanceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class DiscoverInstancesRequest(ServiceRequest):
    NamespaceName: NamespaceName
    ServiceName: ServiceName
    MaxResults: DiscoverMaxResults | None
    QueryParameters: Attributes | None
    OptionalParameters: Attributes | None
    HealthStatus: HealthStatusFilter | None
    OwnerAccount: AWSAccountId | None


Revision = int


class HttpInstanceSummary(TypedDict, total=False):
    """In a response to a
    `DiscoverInstances <https://docs.aws.amazon.com/cloud-map/latest/api/API_DiscoverInstances.html>`__
    request, ``HttpInstanceSummary`` contains information about one instance
    that matches the values that you specified in the request.
    """

    InstanceId: ResourceId | None
    NamespaceName: NamespaceNameHttp | None
    ServiceName: ServiceName | None
    HealthStatus: HealthStatus | None
    Attributes: Attributes | None


HttpInstanceSummaryList = list[HttpInstanceSummary]


class DiscoverInstancesResponse(TypedDict, total=False):
    Instances: HttpInstanceSummaryList | None
    InstancesRevision: Revision | None


class DiscoverInstancesRevisionRequest(ServiceRequest):
    NamespaceName: NamespaceName
    ServiceName: ServiceName
    OwnerAccount: AWSAccountId | None


class DiscoverInstancesRevisionResponse(TypedDict, total=False):
    InstancesRevision: Revision | None


class DnsConfigChange(TypedDict, total=False):
    """A complex type that contains information about changes to the Route 53
    DNS records that Cloud Map creates when you register an instance.
    """

    DnsRecords: DnsRecordList


class DnsProperties(TypedDict, total=False):
    """A complex type that contains the ID for the Route 53 hosted zone that
    Cloud Map creates when you create a namespace.
    """

    HostedZoneId: ResourceId | None
    SOA: SOA | None


FilterValues = list[FilterValue]


class GetInstanceRequest(ServiceRequest):
    ServiceId: Arn
    InstanceId: ResourceId


class Instance(TypedDict, total=False):
    """A complex type that contains information about an instance that Cloud
    Map creates when you submit a ``RegisterInstance`` request.
    """

    Id: ResourceId
    CreatorRequestId: ResourceId | None
    Attributes: Attributes | None
    CreatedByAccount: AWSAccountId | None


class GetInstanceResponse(TypedDict, total=False):
    ResourceOwner: AWSAccountId | None
    Instance: Instance | None


InstanceIdList = list[ResourceId]


class GetInstancesHealthStatusRequest(ServiceRequest):
    ServiceId: Arn
    Instances: InstanceIdList | None
    MaxResults: MaxResults | None
    NextToken: NextToken | None


InstanceHealthStatusMap = dict[ResourceId, HealthStatus]


class GetInstancesHealthStatusResponse(TypedDict, total=False):
    Status: InstanceHealthStatusMap | None
    NextToken: NextToken | None


class GetNamespaceRequest(ServiceRequest):
    Id: Arn


class HttpProperties(TypedDict, total=False):
    """A complex type that contains the name of an HTTP namespace."""

    HttpName: NamespaceName | None


class NamespaceProperties(TypedDict, total=False):
    """A complex type that contains information that's specific to the
    namespace type.
    """

    DnsProperties: DnsProperties | None
    HttpProperties: HttpProperties | None


class Namespace(TypedDict, total=False):
    """A complex type that contains information about a specified namespace."""

    Id: ResourceId | None
    Arn: Arn | None
    ResourceOwner: AWSAccountId | None
    Name: NamespaceName | None
    Type: NamespaceType | None
    Description: ResourceDescription | None
    ServiceCount: ResourceCount | None
    Properties: NamespaceProperties | None
    CreateDate: Timestamp | None
    CreatorRequestId: ResourceId | None


class GetNamespaceResponse(TypedDict, total=False):
    Namespace: Namespace | None


class GetOperationRequest(ServiceRequest):
    OperationId: OperationId
    OwnerAccount: AWSAccountId | None


OperationTargetsMap = dict[OperationTargetType, ResourceId]


class Operation(TypedDict, total=False):
    """A complex type that contains information about a specified operation."""

    Id: OperationId | None
    OwnerAccount: AWSAccountId | None
    Type: OperationType | None
    Status: OperationStatus | None
    ErrorMessage: Message | None
    ErrorCode: Code | None
    CreateDate: Timestamp | None
    UpdateDate: Timestamp | None
    Targets: OperationTargetsMap | None


class GetOperationResponse(TypedDict, total=False):
    Operation: Operation | None


class GetServiceAttributesRequest(ServiceRequest):
    ServiceId: Arn


ServiceAttributesMap = dict[ServiceAttributeKey, ServiceAttributeValue]


class ServiceAttributes(TypedDict, total=False):
    """A complex type that contains information about attributes associated
    with a specific service.
    """

    ServiceArn: Arn | None
    ResourceOwner: AWSAccountId | None
    Attributes: ServiceAttributesMap | None


class GetServiceAttributesResponse(TypedDict, total=False):
    ServiceAttributes: ServiceAttributes | None


class GetServiceRequest(ServiceRequest):
    Id: Arn


class GetServiceResponse(TypedDict, total=False):
    Service: Service | None


class HttpNamespaceChange(TypedDict, total=False):
    """Updated properties for the HTTP namespace."""

    Description: ResourceDescription


class InstanceSummary(TypedDict, total=False):
    """A complex type that contains information about the instances that you
    registered by using a specified service.
    """

    Id: ResourceId | None
    Attributes: Attributes | None
    CreatedByAccount: AWSAccountId | None


InstanceSummaryList = list[InstanceSummary]


class ListInstancesRequest(ServiceRequest):
    ServiceId: Arn
    NextToken: NextToken | None
    MaxResults: MaxResults | None


class ListInstancesResponse(TypedDict, total=False):
    ResourceOwner: AWSAccountId | None
    Instances: InstanceSummaryList | None
    NextToken: NextToken | None


class NamespaceFilter(TypedDict, total=False):
    """A complex type that identifies the namespaces that you want to list. You
    can choose to list public or private namespaces.
    """

    Name: NamespaceFilterName
    Values: FilterValues
    Condition: FilterCondition | None


NamespaceFilters = list[NamespaceFilter]


class ListNamespacesRequest(ServiceRequest):
    NextToken: NextToken | None
    MaxResults: MaxResults | None
    Filters: NamespaceFilters | None


class NamespaceSummary(TypedDict, total=False):
    """A complex type that contains information about a namespace."""

    Id: ResourceId | None
    Arn: Arn | None
    ResourceOwner: AWSAccountId | None
    Name: NamespaceName | None
    Type: NamespaceType | None
    Description: ResourceDescription | None
    ServiceCount: ResourceCount | None
    Properties: NamespaceProperties | None
    CreateDate: Timestamp | None


NamespaceSummariesList = list[NamespaceSummary]


class ListNamespacesResponse(TypedDict, total=False):
    Namespaces: NamespaceSummariesList | None
    NextToken: NextToken | None


class OperationFilter(TypedDict, total=False):
    """A complex type that lets you select the operations that you want to
    list.
    """

    Name: OperationFilterName
    Values: FilterValues
    Condition: FilterCondition | None


OperationFilters = list[OperationFilter]


class ListOperationsRequest(ServiceRequest):
    NextToken: NextToken | None
    MaxResults: MaxResults | None
    Filters: OperationFilters | None


class OperationSummary(TypedDict, total=False):
    """A complex type that contains information about an operation that matches
    the criteria that you specified in a
    `ListOperations <https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html>`__
    request.
    """

    Id: OperationId | None
    Status: OperationStatus | None


OperationSummaryList = list[OperationSummary]


class ListOperationsResponse(TypedDict, total=False):
    Operations: OperationSummaryList | None
    NextToken: NextToken | None


class ServiceFilter(TypedDict, total=False):
    """A complex type that lets you specify the namespaces that you want to
    list services for.
    """

    Name: ServiceFilterName
    Values: FilterValues
    Condition: FilterCondition | None


ServiceFilters = list[ServiceFilter]


class ListServicesRequest(ServiceRequest):
    NextToken: NextToken | None
    MaxResults: MaxResults | None
    Filters: ServiceFilters | None


class ServiceSummary(TypedDict, total=False):
    """A complex type that contains information about a specified service."""

    Id: ResourceId | None
    Arn: Arn | None
    ResourceOwner: AWSAccountId | None
    Name: ServiceName | None
    Type: ServiceType | None
    Description: ResourceDescription | None
    InstanceCount: ResourceCount | None
    DnsConfig: DnsConfig | None
    HealthCheckConfig: HealthCheckConfig | None
    HealthCheckCustomConfig: HealthCheckCustomConfig | None
    CreateDate: Timestamp | None
    CreatedByAccount: AWSAccountId | None


ServiceSummariesList = list[ServiceSummary]


class ListServicesResponse(TypedDict, total=False):
    Services: ServiceSummariesList | None
    NextToken: NextToken | None


class ListTagsForResourceRequest(ServiceRequest):
    ResourceARN: AmazonResourceName


class ListTagsForResourceResponse(TypedDict, total=False):
    Tags: TagList | None


class SOAChange(TypedDict, total=False):
    """Updated Start of Authority (SOA) properties for a public or private DNS
    namespace.
    """

    TTL: RecordTTL


class PrivateDnsPropertiesMutableChange(TypedDict, total=False):
    """Updated DNS properties for the private DNS namespace."""

    SOA: SOAChange


class PrivateDnsNamespacePropertiesChange(TypedDict, total=False):
    """Updated properties for the private DNS namespace."""

    DnsProperties: PrivateDnsPropertiesMutableChange


class PrivateDnsNamespaceChange(TypedDict, total=False):
    """Updated properties for the private DNS namespace."""

    Description: ResourceDescription | None
    Properties: PrivateDnsNamespacePropertiesChange | None


class PublicDnsPropertiesMutableChange(TypedDict, total=False):
    """Updated DNS properties for the public DNS namespace."""

    SOA: SOAChange


class PublicDnsNamespacePropertiesChange(TypedDict, total=False):
    """Updated properties for the public DNS namespace."""

    DnsProperties: PublicDnsPropertiesMutableChange


class PublicDnsNamespaceChange(TypedDict, total=False):
    """Updated properties for the public DNS namespace."""

    Description: ResourceDescription | None
    Properties: PublicDnsNamespacePropertiesChange | None


class RegisterInstanceRequest(ServiceRequest):
    ServiceId: Arn
    InstanceId: InstanceId
    CreatorRequestId: ResourceId | None
    Attributes: Attributes


class RegisterInstanceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class ServiceChange(TypedDict, total=False):
    """A complex type that contains changes to an existing service."""

    Description: ResourceDescription | None
    DnsConfig: DnsConfigChange | None
    HealthCheckConfig: HealthCheckConfig | None


TagKeyList = list[TagKey]


class TagResourceRequest(ServiceRequest):
    ResourceARN: AmazonResourceName
    Tags: TagList


class TagResourceResponse(TypedDict, total=False):
    pass


class UntagResourceRequest(ServiceRequest):
    ResourceARN: AmazonResourceName
    TagKeys: TagKeyList


class UntagResourceResponse(TypedDict, total=False):
    pass


class UpdateHttpNamespaceRequest(ServiceRequest):
    Id: Arn
    UpdaterRequestId: ResourceId | None
    Namespace: HttpNamespaceChange


class UpdateHttpNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class UpdateInstanceCustomHealthStatusRequest(ServiceRequest):
    ServiceId: Arn
    InstanceId: ResourceId
    Status: CustomHealthStatus


class UpdatePrivateDnsNamespaceRequest(ServiceRequest):
    Id: Arn
    UpdaterRequestId: ResourceId | None
    Namespace: PrivateDnsNamespaceChange


class UpdatePrivateDnsNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class UpdatePublicDnsNamespaceRequest(ServiceRequest):
    Id: Arn
    UpdaterRequestId: ResourceId | None
    Namespace: PublicDnsNamespaceChange


class UpdatePublicDnsNamespaceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class UpdateServiceAttributesRequest(ServiceRequest):
    ServiceId: Arn
    Attributes: ServiceAttributesMap


class UpdateServiceAttributesResponse(TypedDict, total=False):
    pass


class UpdateServiceRequest(ServiceRequest):
    Id: Arn
    Service: ServiceChange


class UpdateServiceResponse(TypedDict, total=False):
    OperationId: OperationId | None


class ServicediscoveryApi:
    service: str = "servicediscovery"
    version: str = "2017-03-14"

    @handler("CreateHttpNamespace")
    def create_http_namespace(
        self,
        context: RequestContext,
        name: NamespaceNameHttp,
        creator_request_id: ResourceId | None = None,
        description: ResourceDescription | None = None,
        tags: TagList | None = None,
        **kwargs,
    ) -> CreateHttpNamespaceResponse:
        """Creates an HTTP namespace. Service instances registered using an HTTP
        namespace can be discovered using a ``DiscoverInstances`` request but
        can't be discovered using DNS.

        For the current quota on the number of namespaces that you can create
        using the same Amazon Web Services account, see `Cloud Map
        quotas <https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html>`__
        in the *Cloud Map Developer Guide*.

        :param name: The name that you want to assign to this namespace.
        :param creator_request_id: A unique string that identifies the request and that allows failed
        ``CreateHttpNamespace`` requests to be retried without the risk of
        running the operation twice.
        :param description: A description for the namespace.
        :param tags: The tags to add to the namespace.
        :returns: CreateHttpNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceAlreadyExists:
        :raises ResourceLimitExceeded:
        :raises DuplicateRequest:
        :raises TooManyTagsException:
        """
        raise NotImplementedError

    @handler("CreatePrivateDnsNamespace")
    def create_private_dns_namespace(
        self,
        context: RequestContext,
        name: NamespaceNamePrivate,
        vpc: ResourceId,
        creator_request_id: ResourceId | None = None,
        description: ResourceDescription | None = None,
        tags: TagList | None = None,
        properties: PrivateDnsNamespaceProperties | None = None,
        **kwargs,
    ) -> CreatePrivateDnsNamespaceResponse:
        """Creates a private namespace based on DNS, which is visible only inside a
        specified Amazon VPC. The namespace defines your service naming scheme.
        For example, if you name your namespace ``example.com`` and name your
        service ``backend``, the resulting DNS name for the service is
        ``backend.example.com``. Service instances that are registered using a
        private DNS namespace can be discovered using either a
        ``DiscoverInstances`` request or using DNS. For the current quota on the
        number of namespaces that you can create using the same Amazon Web
        Services account, see `Cloud Map
        quotas <https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html>`__
        in the *Cloud Map Developer Guide*.

        :param name: The name that you want to assign to this namespace.
        :param vpc: The ID of the Amazon VPC that you want to associate the namespace with.
        :param creator_request_id: A unique string that identifies the request and that allows failed
        ``CreatePrivateDnsNamespace`` requests to be retried without the risk of
        running the operation twice.
        :param description: A description for the namespace.
        :param tags: The tags to add to the namespace.
        :param properties: Properties for the private DNS namespace.
        :returns: CreatePrivateDnsNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceAlreadyExists:
        :raises ResourceLimitExceeded:
        :raises DuplicateRequest:
        :raises TooManyTagsException:
        """
        raise NotImplementedError

    @handler("CreatePublicDnsNamespace")
    def create_public_dns_namespace(
        self,
        context: RequestContext,
        name: NamespaceNamePublic,
        creator_request_id: ResourceId | None = None,
        description: ResourceDescription | None = None,
        tags: TagList | None = None,
        properties: PublicDnsNamespaceProperties | None = None,
        **kwargs,
    ) -> CreatePublicDnsNamespaceResponse:
        """Creates a public namespace based on DNS, which is visible on the
        internet. The namespace defines your service naming scheme. For example,
        if you name your namespace ``example.com`` and name your service
        ``backend``, the resulting DNS name for the service is
        ``backend.example.com``. You can discover instances that were registered
        with a public DNS namespace by using either a ``DiscoverInstances``
        request or using DNS. For the current quota on the number of namespaces
        that you can create using the same Amazon Web Services account, see
        `Cloud Map
        quotas <https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html>`__
        in the *Cloud Map Developer Guide*.

        The ``CreatePublicDnsNamespace`` API operation is not supported in the
        Amazon Web Services GovCloud (US) Regions.

        :param name: The name that you want to assign to this namespace.
        :param creator_request_id: A unique string that identifies the request and that allows failed
        ``CreatePublicDnsNamespace`` requests to be retried without the risk of
        running the operation twice.
        :param description: A description for the namespace.
        :param tags: The tags to add to the namespace.
        :param properties: Properties for the public DNS namespace.
        :returns: CreatePublicDnsNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceAlreadyExists:
        :raises ResourceLimitExceeded:
        :raises DuplicateRequest:
        :raises TooManyTagsException:
        """
        raise NotImplementedError

    @handler("CreateService", expand=False)
    def create_service(
        self, context: RequestContext, request: CreateServiceRequest, **kwargs
    ) -> CreateServiceResponse:
        """Creates a service. This action defines the configuration for the
        following entities:

        -  For public and private DNS namespaces, one of the following
           combinations of DNS records in Amazon Route 53:

           -  ``A``

           -  ``AAAA``

           -  ``A`` and ``AAAA``

           -  ``SRV``

           -  ``CNAME``

        -  Optionally, a health check

        After you create the service, you can submit a
        `RegisterInstance <https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html>`__
        request, and Cloud Map uses the values in the configuration to create
        the specified entities.

        For the current quota on the number of instances that you can register
        using the same namespace and using the same service, see `Cloud Map
        quotas <https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html>`__
        in the *Cloud Map Developer Guide*.

        :param name: The name that you want to assign to the service.
        :param namespace_id: The ID or Amazon Resource Name (ARN) of the namespace that you want to
        use to create the service.
        :param creator_request_id: A unique string that identifies the request and that allows failed
        ``CreateService`` requests to be retried without the risk of running the
        operation twice.
        :param description: A description for the service.
        :param dns_config: A complex type that contains information about the Amazon Route 53
        records that you want Cloud Map to create when you register an instance.
        :param health_check_config: *Public DNS and HTTP namespaces only.
        :param health_check_custom_config: A complex type that contains information about an optional custom health
        check.
        :param tags: The tags to add to the service.
        :param type: If present, specifies that the service instances are only discoverable
        using the ``DiscoverInstances`` API operation.
        :returns: CreateServiceResponse
        :raises InvalidInput:
        :raises ResourceLimitExceeded:
        :raises NamespaceNotFound:
        :raises ServiceAlreadyExists:
        :raises TooManyTagsException:
        """
        raise NotImplementedError

    @handler("DeleteNamespace")
    def delete_namespace(
        self, context: RequestContext, id: Arn, **kwargs
    ) -> DeleteNamespaceResponse:
        """Deletes a namespace from the current account. If the namespace still
        contains one or more services, the request fails.

        :param id: The ID or Amazon Resource Name (ARN) of the namespace that you want to
        delete.
        :returns: DeleteNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceNotFound:
        :raises ResourceInUse:
        :raises DuplicateRequest:
        """
        raise NotImplementedError

    @handler("DeleteService")
    def delete_service(self, context: RequestContext, id: Arn, **kwargs) -> DeleteServiceResponse:
        """Deletes a specified service and all associated service attributes. If
        the service still contains one or more registered instances, the request
        fails.

        :param id: The ID or Amazon Resource Name (ARN) of the service that you want to
        delete.
        :returns: DeleteServiceResponse
        :raises InvalidInput:
        :raises ServiceNotFound:
        :raises ResourceInUse:
        """
        raise NotImplementedError

    @handler("DeleteServiceAttributes")
    def delete_service_attributes(
        self,
        context: RequestContext,
        service_id: Arn,
        attributes: ServiceAttributeKeyList,
        **kwargs,
    ) -> DeleteServiceAttributesResponse:
        """Deletes specific attributes associated with a service.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service from which the
        attributes will be deleted.
        :param attributes: A list of keys corresponding to each attribute that you want to delete.
        :returns: DeleteServiceAttributesResponse
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("DeregisterInstance")
    def deregister_instance(
        self, context: RequestContext, service_id: Arn, instance_id: ResourceId, **kwargs
    ) -> DeregisterInstanceResponse:
        """Deletes the Amazon Route 53 DNS records and health check, if any, that
        Cloud Map created for the specified instance.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that the instance is
        associated with.
        :param instance_id: The value that you specified for ``Id`` in the
        `RegisterInstance <https://docs.
        :returns: DeregisterInstanceResponse
        :raises DuplicateRequest:
        :raises InvalidInput:
        :raises InstanceNotFound:
        :raises ResourceInUse:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("DiscoverInstances")
    def discover_instances(
        self,
        context: RequestContext,
        namespace_name: NamespaceName,
        service_name: ServiceName,
        max_results: DiscoverMaxResults | None = None,
        query_parameters: Attributes | None = None,
        optional_parameters: Attributes | None = None,
        health_status: HealthStatusFilter | None = None,
        owner_account: AWSAccountId | None = None,
        **kwargs,
    ) -> DiscoverInstancesResponse:
        """Discovers registered instances for a specified namespace and service.
        You can use ``DiscoverInstances`` to discover instances for any type of
        namespace. ``DiscoverInstances`` returns a randomized list of instances
        allowing customers to distribute traffic evenly across instances. For
        public and private DNS namespaces, you can also use DNS queries to
        discover instances.

        :param namespace_name: The ``HttpName`` name of the namespace.
        :param service_name: The name of the service that you specified when you registered the
        instance.
        :param max_results: The maximum number of instances that you want Cloud Map to return in the
        response to a ``DiscoverInstances`` request.
        :param query_parameters: Filters to scope the results based on custom attributes for the instance
        (for example, ``{version=v1, az=1a}``).
        :param optional_parameters: Opportunistic filters to scope the results based on custom attributes.
        :param health_status: The health status of the instances that you want to discover.
        :param owner_account: The ID of the Amazon Web Services account that owns the namespace
        associated with the instance, as specified in the namespace
        ``ResourceOwner`` field.
        :returns: DiscoverInstancesResponse
        :raises ServiceNotFound:
        :raises NamespaceNotFound:
        :raises InvalidInput:
        :raises RequestLimitExceeded:
        """
        raise NotImplementedError

    @handler("DiscoverInstancesRevision")
    def discover_instances_revision(
        self,
        context: RequestContext,
        namespace_name: NamespaceName,
        service_name: ServiceName,
        owner_account: AWSAccountId | None = None,
        **kwargs,
    ) -> DiscoverInstancesRevisionResponse:
        """Discovers the increasing revision associated with an instance.

        :param namespace_name: The ``HttpName`` name of the namespace.
        :param service_name: The name of the service that you specified when you registered the
        instance.
        :param owner_account: The ID of the Amazon Web Services account that owns the namespace
        associated with the instance, as specified in the namespace
        ``ResourceOwner`` field.
        :returns: DiscoverInstancesRevisionResponse
        :raises ServiceNotFound:
        :raises NamespaceNotFound:
        :raises InvalidInput:
        :raises RequestLimitExceeded:
        """
        raise NotImplementedError

    @handler("GetInstance")
    def get_instance(
        self, context: RequestContext, service_id: Arn, instance_id: ResourceId, **kwargs
    ) -> GetInstanceResponse:
        """Gets information about a specified instance.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that the instance is
        associated with.
        :param instance_id: The ID of the instance that you want to get information about.
        :returns: GetInstanceResponse
        :raises InstanceNotFound:
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("GetInstancesHealthStatus")
    def get_instances_health_status(
        self,
        context: RequestContext,
        service_id: Arn,
        instances: InstanceIdList | None = None,
        max_results: MaxResults | None = None,
        next_token: NextToken | None = None,
        **kwargs,
    ) -> GetInstancesHealthStatusResponse:
        """Gets the current health status (``Healthy``, ``Unhealthy``, or
        ``Unknown``) of one or more instances that are associated with a
        specified service.

        There's a brief delay between when you register an instance and when the
        health status for the instance is available.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that the instance is
        associated with.
        :param instances: An array that contains the IDs of all the instances that you want to get
        the health status for.
        :param max_results: The maximum number of instances that you want Cloud Map to return in the
        response to a ``GetInstancesHealthStatus`` request.
        :param next_token: For the first ``GetInstancesHealthStatus`` request, omit this value.
        :returns: GetInstancesHealthStatusResponse
        :raises InstanceNotFound:
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("GetNamespace")
    def get_namespace(self, context: RequestContext, id: Arn, **kwargs) -> GetNamespaceResponse:
        """Gets information about a namespace.

        :param id: The ID or Amazon Resource Name (ARN) of the namespace that you want to
        get information about.
        :returns: GetNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceNotFound:
        """
        raise NotImplementedError

    @handler("GetOperation")
    def get_operation(
        self,
        context: RequestContext,
        operation_id: OperationId,
        owner_account: AWSAccountId | None = None,
        **kwargs,
    ) -> GetOperationResponse:
        """Gets information about any operation that returns an operation ID in the
        response, such as a ``CreateHttpNamespace`` request.

        To get a list of operations that match specified criteria, see
        `ListOperations <https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html>`__.

        :param operation_id: The ID of the operation that you want to get more information about.
        :param owner_account: The ID of the Amazon Web Services account that owns the namespace
        associated with the operation, as specified in the namespace
        ``ResourceOwner`` field.
        :returns: GetOperationResponse
        :raises InvalidInput:
        :raises OperationNotFound:
        """
        raise NotImplementedError

    @handler("GetService")
    def get_service(self, context: RequestContext, id: Arn, **kwargs) -> GetServiceResponse:
        """Gets the settings for a specified service.

        :param id: The ID or Amazon Resource Name (ARN) of the service that you want to get
        settings for.
        :returns: GetServiceResponse
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("GetServiceAttributes")
    def get_service_attributes(
        self, context: RequestContext, service_id: Arn, **kwargs
    ) -> GetServiceAttributesResponse:
        """Returns the attributes associated with a specified service.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that you want to get
        attributes for.
        :returns: GetServiceAttributesResponse
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("ListInstances")
    def list_instances(
        self,
        context: RequestContext,
        service_id: Arn,
        next_token: NextToken | None = None,
        max_results: MaxResults | None = None,
        **kwargs,
    ) -> ListInstancesResponse:
        """Lists summary information about the instances that you registered by
        using a specified service.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that you want to
        list instances for.
        :param next_token: For the first ``ListInstances`` request, omit this value.
        :param max_results: The maximum number of instances that you want Cloud Map to return in the
        response to a ``ListInstances`` request.
        :returns: ListInstancesResponse
        :raises ServiceNotFound:
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("ListNamespaces")
    def list_namespaces(
        self,
        context: RequestContext,
        next_token: NextToken | None = None,
        max_results: MaxResults | None = None,
        filters: NamespaceFilters | None = None,
        **kwargs,
    ) -> ListNamespacesResponse:
        """Lists summary information about the namespaces that were created by the
        current Amazon Web Services account and shared with the current Amazon
        Web Services account.

        :param next_token: For the first ``ListNamespaces`` request, omit this value.
        :param max_results: The maximum number of namespaces that you want Cloud Map to return in
        the response to a ``ListNamespaces`` request.
        :param filters: A complex type that contains specifications for the namespaces that you
        want to list.
        :returns: ListNamespacesResponse
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("ListOperations")
    def list_operations(
        self,
        context: RequestContext,
        next_token: NextToken | None = None,
        max_results: MaxResults | None = None,
        filters: OperationFilters | None = None,
        **kwargs,
    ) -> ListOperationsResponse:
        """Lists operations that match the criteria that you specify.

        :param next_token: For the first ``ListOperations`` request, omit this value.
        :param max_results: The maximum number of items that you want Cloud Map to return in the
        response to a ``ListOperations`` request.
        :param filters: A complex type that contains specifications for the operations that you
        want to list, for example, operations that you started between a
        specified start date and end date.
        :returns: ListOperationsResponse
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("ListServices")
    def list_services(
        self,
        context: RequestContext,
        next_token: NextToken | None = None,
        max_results: MaxResults | None = None,
        filters: ServiceFilters | None = None,
        **kwargs,
    ) -> ListServicesResponse:
        """Lists summary information for all the services that are associated with
        one or more namespaces.

        :param next_token: For the first ``ListServices`` request, omit this value.
        :param max_results: The maximum number of services that you want Cloud Map to return in the
        response to a ``ListServices`` request.
        :param filters: A complex type that contains specifications for the namespaces that you
        want to list services for.
        :returns: ListServicesResponse
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("ListTagsForResource")
    def list_tags_for_resource(
        self, context: RequestContext, resource_arn: AmazonResourceName, **kwargs
    ) -> ListTagsForResourceResponse:
        """Lists tags for the specified resource.

        :param resource_arn: The Amazon Resource Name (ARN) of the resource that you want to retrieve
        tags for.
        :returns: ListTagsForResourceResponse
        :raises ResourceNotFoundException:
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("RegisterInstance")
    def register_instance(
        self,
        context: RequestContext,
        service_id: Arn,
        instance_id: InstanceId,
        attributes: Attributes,
        creator_request_id: ResourceId | None = None,
        **kwargs,
    ) -> RegisterInstanceResponse:
        """Creates or updates one or more records and, optionally, creates a health
        check based on the settings in a specified service. When you submit a
        ``RegisterInstance`` request, the following occurs:

        -  For each DNS record that you define in the service that's specified
           by ``ServiceId``, a record is created or updated in the hosted zone
           that's associated with the corresponding namespace.

        -  If the service includes ``HealthCheckConfig``, a health check is
           created based on the settings in the health check configuration.

        -  The health check, if any, is associated with each of the new or
           updated records.

        One ``RegisterInstance`` request must complete before you can submit
        another request and specify the same service ID and instance ID.

        For more information, see
        `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`__.

        When Cloud Map receives a DNS query for the specified DNS name, it
        returns the applicable value:

        -  **If the health check is healthy**: returns all the records

        -  **If the health check is unhealthy**: returns the applicable value
           for the last healthy instance

        -  **If you didn't specify a health check configuration**: returns all
           the records

        For the current quota on the number of instances that you can register
        using the same namespace and using the same service, see `Cloud Map
        quotas <https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html>`__
        in the *Cloud Map Developer Guide*.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that you want to use
        for settings for the instance.
        :param instance_id: An identifier that you want to associate with the instance.
        :param attributes: A string map that contains the following information for the service
        that you specify in ``ServiceId``:

        -  The attributes that apply to the records that are defined in the
           service.
        :param creator_request_id: A unique string that identifies the request and that allows failed
        ``RegisterInstance`` requests to be retried without the risk of
        executing the operation twice.
        :returns: RegisterInstanceResponse
        :raises DuplicateRequest:
        :raises InvalidInput:
        :raises ResourceInUse:
        :raises ResourceLimitExceeded:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("TagResource")
    def tag_resource(
        self, context: RequestContext, resource_arn: AmazonResourceName, tags: TagList, **kwargs
    ) -> TagResourceResponse:
        """Adds one or more tags to the specified resource.

        :param resource_arn: The Amazon Resource Name (ARN) of the resource that you want to retrieve
        tags for.
        :param tags: The tags to add to the specified resource.
        :returns: TagResourceResponse
        :raises ResourceNotFoundException:
        :raises TooManyTagsException:
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("UntagResource")
    def untag_resource(
        self,
        context: RequestContext,
        resource_arn: AmazonResourceName,
        tag_keys: TagKeyList,
        **kwargs,
    ) -> UntagResourceResponse:
        """Removes one or more tags from the specified resource.

        :param resource_arn: The Amazon Resource Name (ARN) of the resource that you want to retrieve
        tags for.
        :param tag_keys: The tag keys to remove from the specified resource.
        :returns: UntagResourceResponse
        :raises ResourceNotFoundException:
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("UpdateHttpNamespace")
    def update_http_namespace(
        self,
        context: RequestContext,
        id: Arn,
        namespace: HttpNamespaceChange,
        updater_request_id: ResourceId | None = None,
        **kwargs,
    ) -> UpdateHttpNamespaceResponse:
        """Updates an HTTP namespace.

        :param id: The ID or Amazon Resource Name (ARN) of the namespace that you want to
        update.
        :param namespace: Updated properties for the the HTTP namespace.
        :param updater_request_id: A unique string that identifies the request and that allows failed
        ``UpdateHttpNamespace`` requests to be retried without the risk of
        running the operation twice.
        :returns: UpdateHttpNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceNotFound:
        :raises ResourceInUse:
        :raises DuplicateRequest:
        """
        raise NotImplementedError

    @handler("UpdateInstanceCustomHealthStatus")
    def update_instance_custom_health_status(
        self,
        context: RequestContext,
        service_id: Arn,
        instance_id: ResourceId,
        status: CustomHealthStatus,
        **kwargs,
    ) -> None:
        """Submits a request to change the health status of a custom health check
        to healthy or unhealthy.

        You can use ``UpdateInstanceCustomHealthStatus`` to change the status
        only for custom health checks, which you define using
        ``HealthCheckCustomConfig`` when you create a service. You can't use it
        to change the status for Route 53 health checks, which you define using
        ``HealthCheckConfig``.

        For more information, see
        `HealthCheckCustomConfig <https://docs.aws.amazon.com/cloud-map/latest/api/API_HealthCheckCustomConfig.html>`__.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that includes the
        configuration for the custom health check that you want to change the
        status for.
        :param instance_id: The ID of the instance that you want to change the health status for.
        :param status: The new status of the instance, ``HEALTHY`` or ``UNHEALTHY``.
        :raises InstanceNotFound:
        :raises ServiceNotFound:
        :raises CustomHealthNotFound:
        :raises InvalidInput:
        """
        raise NotImplementedError

    @handler("UpdatePrivateDnsNamespace")
    def update_private_dns_namespace(
        self,
        context: RequestContext,
        id: Arn,
        namespace: PrivateDnsNamespaceChange,
        updater_request_id: ResourceId | None = None,
        **kwargs,
    ) -> UpdatePrivateDnsNamespaceResponse:
        """Updates a private DNS namespace.

        :param id: The ID or Amazon Resource Name (ARN) of the namespace that you want to
        update.
        :param namespace: Updated properties for the private DNS namespace.
        :param updater_request_id: A unique string that identifies the request and that allows failed
        ``UpdatePrivateDnsNamespace`` requests to be retried without the risk of
        running the operation twice.
        :returns: UpdatePrivateDnsNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceNotFound:
        :raises ResourceInUse:
        :raises DuplicateRequest:
        """
        raise NotImplementedError

    @handler("UpdatePublicDnsNamespace")
    def update_public_dns_namespace(
        self,
        context: RequestContext,
        id: Arn,
        namespace: PublicDnsNamespaceChange,
        updater_request_id: ResourceId | None = None,
        **kwargs,
    ) -> UpdatePublicDnsNamespaceResponse:
        """Updates a public DNS namespace.

        :param id: The ID or Amazon Resource Name (ARN) of the namespace being updated.
        :param namespace: Updated properties for the public DNS namespace.
        :param updater_request_id: A unique string that identifies the request and that allows failed
        ``UpdatePublicDnsNamespace`` requests to be retried without the risk of
        running the operation twice.
        :returns: UpdatePublicDnsNamespaceResponse
        :raises InvalidInput:
        :raises NamespaceNotFound:
        :raises ResourceInUse:
        :raises DuplicateRequest:
        """
        raise NotImplementedError

    @handler("UpdateService")
    def update_service(
        self, context: RequestContext, id: Arn, service: ServiceChange, **kwargs
    ) -> UpdateServiceResponse:
        """Submits a request to perform the following operations:

        -  Update the TTL setting for existing ``DnsRecords`` configurations

        -  Add, update, or delete ``HealthCheckConfig`` for a specified service

           You can't add, update, or delete a ``HealthCheckCustomConfig``
           configuration.

        For public and private DNS namespaces, note the following:

        -  If you omit any existing ``DnsRecords`` or ``HealthCheckConfig``
           configurations from an ``UpdateService`` request, the configurations
           are deleted from the service.

        -  If you omit an existing ``HealthCheckCustomConfig`` configuration
           from an ``UpdateService`` request, the configuration isn't deleted
           from the service.

        You can't call ``UpdateService`` and update settings in the following
        scenarios:

        -  When the service is associated with an HTTP namespace

        -  When the service is associated with a shared namespace and contains
           instances that were registered by Amazon Web Services accounts other
           than the account making the ``UpdateService`` call

        When you update settings for a service, Cloud Map also updates the
        corresponding settings in all the records and health checks that were
        created by using the specified service.

        :param id: The ID or Amazon Resource Name (ARN) of the service that you want to
        update.
        :param service: A complex type that contains the new settings for the service.
        :returns: UpdateServiceResponse
        :raises DuplicateRequest:
        :raises InvalidInput:
        :raises ServiceNotFound:
        """
        raise NotImplementedError

    @handler("UpdateServiceAttributes")
    def update_service_attributes(
        self, context: RequestContext, service_id: Arn, attributes: ServiceAttributesMap, **kwargs
    ) -> UpdateServiceAttributesResponse:
        """Submits a request to update a specified service to add service-level
        attributes.

        :param service_id: The ID or Amazon Resource Name (ARN) of the service that you want to
        update.
        :param attributes: A string map that contains attribute key-value pairs.
        :returns: UpdateServiceAttributesResponse
        :raises InvalidInput:
        :raises ServiceNotFound:
        :raises ServiceAttributesLimitExceededException:
        """
        raise NotImplementedError
