//=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//
// LSDChiTools
// Land Surface Dynamics ChiTools object
//
// An object within the University
//  of Edinburgh Land Surface Dynamics group topographic toolbox
//  for performing various analyses in chi space
//
//
// Developed by:
//  Simon M. Mudd
//  Martin D. Hurst
//  David T. Milodowski
//  Stuart W.D. Grieve
//  Declan A. Valters
//  Fiona Clubb
//  Boris Gailleton
//
// Copyright (C) 2016 Simon M. Mudd 2016
//
// Developer can be contacted by simon.m.mudd _at_ ed.ac.uk
//
//    Simon Mudd
//    University of Edinburgh
//    School of GeoSciences
//    Drummond Street
//    Edinburgh, EH8 9XP
//    Scotland
//    United Kingdom
//
// This program is free software;
// you can redistribute it and/or modify it under the terms of the
// GNU General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY;
// without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU General Public License for more details.
//
// You should have received a copy of the
// GNU General Public License along with this program;
// if not, write to:
// Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301
// USA
//
//=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

#ifndef LSDTribBasModel_CPP
#define LSDTribBasModel_CPP
#include <iostream>
#include <string>
#include <vector>
#include <map>
#include <ctime>
#include <fstream>
#include <queue>
#include "LSDStatsTools.hpp"
#include "LSDChiNetwork.hpp"
#include "LSDRaster.hpp"
#include "LSDRasterInfo.hpp"
#include "LSDIndexRaster.hpp"
#include "LSDFlowInfo.hpp"
#include "LSDJunctionNetwork.hpp"
#include "LSDIndexChannelTree.hpp"
#include "LSDBasin.hpp"
#include "LSDChiTools.hpp"
#include "LSDParameterParser.hpp"
#include "LSDSpatialCSVReader.hpp"
#include "LSDShapeTools.hpp"
#include "LSDRasterMaker.hpp"

#include <omp.h>

#include "LSDTribBasModel.hpp"

#include "xtensor-python/pyarray.hpp"
#include "xtensor-python/pytensor.hpp"
#include "xtensor-python/pyvectorize.hpp"
#include "xtensor/xadapt.hpp"

#include "LSDStatsTools.hpp"

#include "xtensor/xmath.hpp"
#include "xtensor/xarray.hpp"
#include "xtensor/xtensor.hpp"
#include <iostream>
#include <numeric>
#include <cmath>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>


// Dealing with muddpile stuff
#ifdef __linux__ 
#include "LSDRasterModel.hpp"
#endif




// All the functions with the spacename TriBas are used in this model
namespace TriBas
{


  //#######################################################################################################
  // DEFINING STRUCT
  //#######################################################################################################
  // Probably a temporary structure. I kind of struggle to get the node ordered in the right way I need for
  // running bottom to top my model.
  // sortation structure attempts to solve that in a hacky way where I am using a priority queue 
  // to sort my rivers bottom to top.
  struct sortation
  {
    // Vector of river node
    std::vector<int> Node_of_river;
    // Elevation at eh base of the river
    float base_elevation;
  };

  // These are the operator used to sort the river per grater/lower elevation in the priority queue
  bool operator>( const sortation& lhs, const sortation& rhs )
  {
    return lhs.base_elevation > rhs.base_elevation;
  }
  bool operator<( const sortation& lhs, const sortation& rhs )
  {
    return lhs.base_elevation < rhs.base_elevation;
  }
  //#######################################################################################################
  //#######################################################################################################



	// Alright let's begin that. B.G. 21/11/2018 at 4 AM in Edinburgh's airport

  //#######################################################################################################
  //#######################################################################################################
  // MAIN FUNCTIONS


  ///@brief Functions that sets up the models from a range of parameters.
  ///@details The model needs a specific structure to be build once. This function etract the river network and all the information the models need to know to run (e.g., node IDs and their receivers, elevqtions, ...).
  ///@details It requires Information to understance the raster and on what you want to calculate mchi which basin and which channel heads you want.
  ///@details It then returns a map of all these data ready to be ran in the run function(s)
  ///@param nrows (int): Number of rows of your raster
  ///@param ncols (int): Number of cols of your raster
  ///@param xmin (float): the minimum x of your raster
  ///@param ymin (float): the minimum y of your raster
  ///@param ndv (float): No Data Value (Which value I will ignore)
  ///@param data (2Dtensor float): the raster array
  ///@param x_sources (1Dtensor double): x coordinates of our cannel heads (Same coordinate system than our raster!!)
  ///@param y_sources (1Dtensor double): y coordinates of our cannel heads (Same coordinate system than our raster!!)
  ///@param Northing: basin outlet y coordinate (Same coordinate system than our raster!!)
  ///@param Easting: basin outlet x coordinate (Same coordinate system than our raster!!)
  ///@param search_node_radius (int): How many nodes should I check around your basin outlet input for the nearest consequent Junction
  ///@param target_nodes (int)/n_iteration (int)/ skip (int)/ minimum_segment_length/ sigma (int)/ m (float)/ n (float)/ A0 (float): parameters linked to mchi extraction (see Mudd et al., 2014 and associated documentation for details).
  ///@return A map of vectors (Node, Elevation,X,Y...)
  ///@authors B.G.
  ///@date 25/11/2018
  std::map<std::string, xt::pytensor<float,1> > setting_up_TriBAs(int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, 
    xt::pytensor<float,2>& data, xt::pytensor<float,1>& x_sources, xt::pytensor<float,1>& y_sources, 
    float m, float n, float A0, double Northing, double Easting, int search_radius_nodes,
    int target_nodes, int n_iterations, int skip, int minimum_segment_length, int sigma)
  {

    std::cout << "I am loading the data into LSDTT" << std::endl;
    // This will host the data. At the moment I need to copy the data from pytensors to TNT array. Not ideal but No other solution unfortunately.
    // Memory leak can become a problem for realy big rasters, however, the time loss is not important (few milliseconds).
    TNT::Array2D<float> data_pointerast(nrows,ncols,ndv); // Trying here to create a raster of pointers
    for(size_t i=0;i<nrows;i++)
    for(size_t j=0;j<ncols;j++)
      data_pointerast[i][j] = data(i,j);

    // Data is copied! I am ready to feed a LSDRaster object
    // Here we go: 
    LSDRaster PP_raster(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);

    // Now I need all the flow informations
    std::vector<std::string> bound = {"n","n","n","n"};
    LSDFlowInfo FlowInfo(bound,PP_raster);
    // OK I got it
    
    // Calculating the flow accumulation and the ...
    LSDIndexRaster FlowAcc = FlowInfo.write_NContributingNodes_to_LSDIndexRaster();
    // Drainage area
    LSDRaster DrainageArea = FlowInfo.write_DrainageArea_to_LSDRaster();
    // DOne

    // Getting the flow distance
    LSDRaster DistanceFromOutlet = FlowInfo.distance_from_outlet();

    // Initialising the channel heads
    std::vector<int> sources; //  sources will contain the node index (unique identifer for each node) of me sources
    // # I first need to copy the xy cordinates into vector for lsdtt
    std::vector<float> x_sources_v, y_sources_v;
    for (size_t yuio = 0; yuio<x_sources.size(); yuio ++)
      {x_sources_v.push_back(x_sources(yuio));y_sources_v.push_back(y_sources(yuio));}
    // ~ DOne
    sources = FlowInfo.Ingest_Channel_Heads(x_sources_v,y_sources_v);
    // I got me channels head

    // Lets extract my channels
    // now get the junction network
    LSDJunctionNetwork JunctionNetwork(sources, FlowInfo);
    // Ok, great, I preprocessed all my raw rivers

    // Now let's get the basin outlet
    // # A bit of hacky way to get the data into the right format
    std::vector<float> fUTM_easting; fUTM_easting.push_back(Easting);
    std::vector<float> fUTM_northing; fUTM_northing.push_back(Northing);
    int threshold_stream_order = 2;
    std::vector<int> valid_cosmo_points;         // a vector to hold the valid nodes
    std::vector<int> snapped_node_indices;       // a vector to hold the valid node indices
    std::vector<int> snapped_junction_indices;   // a vector to hold the valid junction indices
    // Sending the coordinates to the outlet tracker
    JunctionNetwork.snap_point_locations_to_channels(fUTM_easting, fUTM_northing, 
                search_radius_nodes, threshold_stream_order, FlowInfo, 
                valid_cosmo_points, snapped_node_indices, snapped_junction_indices);
    std::vector<int> BaseLevelJunctions = snapped_junction_indices;
    // Done

    // Results check
    if(BaseLevelJunctions.size() == 0)
    {
      std::cout << "ERROR: I couldn't find any basin with your criteri. Increase search_radius_nodes maybe??" << std::endl;
      std::exit(EXIT_FAILURE);
    }
    // Let me tell you outside of the code just in case you are wondering what is happenning
    std::cout << "Successfully detected the outlet of a basin! Hopefully the one you had in mind" << std::endl;

    // Ignore but keep that in case
    // BaseLevelJunctions = JunctionNetwork.Prune_Junctions_If_Nested(BaseLevelJunctions,FlowInfo, FlowAcc);
    // Small but required cleaning
    std::vector<int> outlet_nodes, baselevel_node_of_each_basin;
    JunctionNetwork.get_overlapping_channels(FlowInfo, BaseLevelJunctions, DistanceFromOutlet,
                                    sources, outlet_nodes,baselevel_node_of_each_basin,10);

    // The steps below are getting (i) mchi and related values and (ii) the node structure of our river network
    // Now let's get the chitools N stuffs
    LSDChiTools ChiTool(FlowInfo);
    // Getting chi because we kind of need it
    LSDRaster chi_coordinate = FlowInfo.get_upslope_chi_from_all_baselevel_nodes(m/n,A0,1);
    // getting mchi
    ChiTool.chi_map_automator(FlowInfo, sources, outlet_nodes, baselevel_node_of_each_basin,
                          PP_raster, DistanceFromOutlet,
                          DrainageArea, chi_coordinate, target_nodes,
                          n_iterations, skip, minimum_segment_length, sigma);
    const int n_dfdg = 10000;
    ChiTool.segment_counter(FlowInfo, n_dfdg);

    // Now getting the ndoe sequence and the data maps
    // The node_sequence correspond to the ordered fastscape node index
    // The maps contains elevation, flow distance ... for each of these node indexes
    std::map<std::string, std::map<int,float> > mahmaps =  ChiTool.get_data_maps();
    std::vector<int> node_sequence =  ChiTool.get_vectors_of_node();

    // Generating the data structure for the output
    std::map<std::string,xt::pytensor<float,1> > output;
    // # THat's the way you create a xtensor array
    std::array<size_t, 1> shape = { node_sequence.size() };
    xt::xtensor<float,1> elev(shape);
    xt::xtensor<float,1> chi(shape);
    xt::xtensor<float,1> flowdis(shape);
    xt::xtensor<float,1> node_seq(shape);
    xt::xtensor<float,1> rec(shape);
    xt::xtensor<float,1> SK(shape);
    xt::xtensor<float,1> DA(shape);
    xt::xtensor<int,1> row(shape);
    xt::xtensor<int,1> col(shape);
    xt::xtensor<float,1> these_x(shape);
    xt::xtensor<float,1> these_y(shape);
    xt::xtensor<float,1> mchi(shape);

    // OK now I am dealing with node ordering. I did not manage to extrct it correctly within LSDTT for some reason
    // So here is a quick and dirty way to order my nodes in a modelling friendly way
    // # First I am creating a priority queue with a greater logic 
    std::priority_queue< sortation, std::vector<sortation>, std::greater<sortation> > myp;
    std::vector<int> temp_node, totnode;// temp variables
    // Sorting my rivers:
    float last_elevation_for_sorting = mahmaps["elevation"][node_sequence[0]];
    int this_SK = mahmaps["SK"][node_sequence[0]]; int last_SK = this_SK;
    for (size_t i=0;i<node_sequence.size(); i++)
    {
      // the idea is to separate each of my rivers and get the lower ones at the bottom of my queue
      // Getting the current cource key
      this_SK = mahmaps["SK"][node_sequence[i]];
      if(this_SK != last_SK)
      {
        // If I changed e source key (my river chunk) I ewant to save it into my queue
        // Nodes are sorted top to bottom per rivers -> here is the bottom to top one
        std::reverse(temp_node.begin(), temp_node.end());
        // Creating my node structure here, see description above
        sortation temp_sortation;
        temp_sortation.Node_of_river = temp_node; // vectore of nodes
        temp_sortation.base_elevation = last_elevation_for_sorting; // elevation of the lowest cell
        myp.push(temp_sortation); // feeding my queue (highly questionable sentence is translated in french)
        temp_node.clear(); // Getting it ready for my next loop run 
      }

      // I do that all the time anyway
      temp_node.push_back(node_sequence[i]); // adding my node to the vector
      // Saving the last elevation in case next one is a change
      last_elevation_for_sorting = mahmaps["elevation"][node_sequence[i]];
      // saving current sk to check if change
      last_SK = this_SK;
    }
    // I need to do it a last time to save the last river
    std::reverse(temp_node.begin(), temp_node.end());
    // Creating my node structure here, see description above
    sortation temp_sortation;
    temp_sortation.Node_of_river = temp_node; // vectore of nodes
    temp_sortation.base_elevation = last_elevation_for_sorting; // elevation of the lowest cell
    myp.push(temp_sortation); // feeding my queue (highly questionable sentence is translated in french)
    temp_node.clear(); // Getting it ready for my next loop run 
    // DOne
    // Now I kind of need to recompile my data into one single tensor
    while(myp.size()>0) // While my priority queue is not empty
    {
      // Getting my vector of lowest river nodes
      std::vector<int> this_veconde;
      sortation hugh = myp.top(); // this gives the elements at the top of the queue
      this_veconde = hugh.Node_of_river;
      myp.pop(); //  this removes the abovementioned river from my stack
      // Now simply pushing the node into me global node vector
      for (size_t u =0 ; u < this_veconde.size() ; u++) {totnode.push_back(this_veconde[u]);}
    }
    // Nodes should be ordered now

    // Last but not least I am feeding my output map. Not detailing here as it is simple copy.
    for(size_t i=0; i<totnode.size(); i++)
    {
      int this_node = totnode[i]; int tr; int tc;
      FlowInfo.retrieve_current_row_and_col(this_node,tr,tc);
      node_seq[i] = this_node;
      int recnode =0;
      FlowInfo.retrieve_receiver_information(this_node,recnode, tr,tc);
      rec[i] = recnode;
      chi[i] = mahmaps["chi"][this_node];
      flowdis[i] = mahmaps["flow_distance"][this_node];
      SK[i] = mahmaps["SK"][this_node];
      DA[i] = mahmaps["DA"][this_node];
      row[i] = tr;
      col[i] = tc;
      float tx,ty;
      FlowInfo.get_x_and_y_locations(tr,tc,tx,ty);
      these_x[i] = tx;
      these_y[i] = ty;
      elev[i] = mahmaps["elevation"][this_node];
      mchi[i] = mahmaps["m_chi"][this_node];

      // if(elev[i] == -9999 || mahmaps["elevation"].count(this_node) == 0)
      //   {std::cout << "GARGGGGGGG: " << elev[i] << std::endl;}
      // if(elev[i] == -9999 || mahmaps["elevation"].count(recnode) == 0)
      //   {std::cout << "GARGGGGGGG_recnode: " << elev[i] << std::endl;}
      if(mahmaps["elevation"].count(recnode) == 0)
      {
        bool keep_checking = true;
        int noronode = 0;
        while(keep_checking)
        {
          FlowInfo.retrieve_receiver_information(recnode,noronode,tr,tc);
          if(mahmaps["elevation"].count(noronode)==0 || noronode == -9999)
          {
            keep_checking = false;
          }
          else
          {
            recnode = noronode;
          }
        }
        rec[i] = noronode;
      }
    }

    output["nodeID"] = node_seq;
    output["chi"] = chi;
    output["elevation"] = elev;
    output["flow_distance"] = flowdis;
    output["source_key"] = SK;
    output["drainage_area"] = DA;
    output["receivers"] = rec;
    output["row"] = row;
    output["col"] = col;
    output["x"] = these_x;
    output["y"] = these_y;
    output["m_chi"] = mchi;

    // Ready to return!!
    return output;
  }


  ///@brief this is the basic run of the model: it runs on a prebuilt model for a given time, a spatial uplift field and a spatial erodibility and given m and n.
  ///@details requires a prebuilt model to run. It calculate and save for given time steps the resulting elevation at the end of the run.
  ///@param meq, neq (flaot): the SPL constants
  ///@param n_dt (int): the nuber of time step to run 
  ///@param timestep (int): The value in years of each time step
  ///@param save_step (int): sve the elevation numpy array every X time step. Beware of your memory for large run!! I'll implement a disk I/O option if it becomes a problem
  ///@param uplift (xtensor<float,1>): the spatially variable uplift field (Doesn't have to be variable though, it has to be an array with the same dimension then the input)
  ///@param erodability_K (xtensor<float,1>): the spatially variable erodibility (Doesn't have to be variable though, it has to be an array with the same dimension then the input)
  ///@param prebuilt (map<string,xtensor<float,1>>): the prebuilt model, ouput from the building functions. It contains useful informationand the initial condition of the model.
  ///@return A map/dictionnary(in python) of xtensor/array(in python) of the elevation at each saved time steps.
  ///@authors B.G.
  ///@date 27/11/2018
  std::map<std::string, xt::pytensor<float,1> > run_model(float meq, float neq, int n_timestep, int timestep, int save_step, xt::pytensor<float,1>& uplift, xt::pytensor<float,1>& erodility_K, std::map<std::string, xt::pytensor<float,1> >& prebuilt )
  {
    
    // Creating the final output to fill
    std::map<std::string, xt::pytensor<float,1> > output;

    //global size of my arrays
    size_t siz = prebuilt["elevation"].size();

    // Setting arrays to store the new and last elevation
    std::array<size_t, 1> shape = { siz };
    xt::xtensor<float,1> last_elev(shape), this_new_elev(shape);
    // Setting base variables I need
    float baselevel = prebuilt["elevation"][0]; // elevation of the very first element. So far has to be fixed
    float last_h = baselevel, this_h=0; // navigating through the different elevtion layers
    float tol = 1e-3;// Tolerance for the newton-rhapson itarative solver for the stream power law
    int this_node,last_node;// Navigating through the node and its receivers
    last_elev = prebuilt["elevation"]; // initializing our previous elevation to the first one

    // Alright let's run the model
    // for(int tTt = 0; tTt < 1; tTt++) // Ingore that it's when I need to only run one model to check some stuff
    // Real run here
    for(int tTt = 0; tTt < n_timestep; tTt++)
    {
      // These maps are storing the elevation and flow distance for each node IDs. 
      // this idea is to rn the model bottom up, store informations for current nodes and 
      // therefore get the receiver elevation/flow distance of previously processed node.
      std::map<int,float> rec_to_elev, rec_to_fd;
      // How far are we in the model. Let me stop my work here, there are pretty intense turbulences in my flights.
      // OK seems better now, f*ck these were impressive
      // So I was saying that we want to know which timestep we are at.
      std::cout << "Processing: " << tTt+1 << "||" << n_timestep << "\r"; // the last sign means "Return to the beginning of my line and flush"
      // Running through the river network:
      for(size_t i=0; i<siz;i++)
      {
        // We are in the first element, it is a bit different so we need a special case:
        // Saving elevation and stuff as baselevel, and giving the first receiver information
        if(i==0)
          {this_new_elev[i]=baselevel;last_h = baselevel; rec_to_elev[prebuilt["nodeID"][i]] = baselevel;rec_to_fd[prebuilt["nodeID"][i]] = prebuilt["flow_distance"][i];continue;}
        // Done, the continue statement at the end of last if makes sure it stop this loop run here if i == 0

        // I am getting my receiver and current nodes (available in the prebuilt model)
        last_node = prebuilt["receivers"][i];
        this_node = prebuilt["nodeID"][i];
        // Getting the length between these nodes (dx)
        float length = prebuilt["flow_distance"][i] - rec_to_fd[last_node];
        // getting the flow distance ready when next nodes will query this one as receiver
        rec_to_fd[this_node] = prebuilt["flow_distance"][i];
        // Getting the drainage area (from prebuilt model)
        float tDA = prebuilt["drainage_area"][i];
        // Getting the erodibility from input
        float keq = erodility_K[i];
        // Newton-rahpson method to solve non linear equation
        float epsilon;     // in newton's method, z_n+1 = z_n - f(z_n)/f'(z_n)
                         // and here epsilon =   f(z_n)/f'(z_n)
                         // f(z_n) = -z_n + z_old - dt*K*A^m*( (z_n-z_r)/dx )^n
                         // We differentiate the above equation to get f'(z_n)
                         // the resulting equation f(z_n)/f'(z_n) is seen below
        // A bit of factorisation to clarify the equation
        float streamPowerFactor = keq * pow(tDA, meq) * timestep;
        float slope; // Slope.
        float dx = length; // Embarrassing renaming of variables because I am lazy and got confused
        float new_zeta = last_elev[i]; float old_zeta = new_zeta; // zeta = z = elevation
        // iterate until you converge on a solution. Uses Newton's method.
        int iter_count = 0;
        do
        {
          // Get the slope
          slope = (new_zeta - rec_to_elev[last_node]) / dx;
          // Check backslope or no slope ie no erosion
          if(slope <= 0)
          {
            epsilon = 0;
          }
          else
          {
            // Applying the newton's method
            epsilon = (new_zeta - old_zeta + streamPowerFactor * std::pow(slope, neq)) /
                 (1 + streamPowerFactor * (neq/dx) * std::pow(slope, neq-1));
          }
          // iterate the result
          new_zeta -= epsilon;

          // This limits the number of iterations, it is a safety check to avoid infinite loop
          // Thsi will begin to split some  nan or inf if it diverges
          iter_count++;
          if(iter_count > 100)
          {
            // std::cout << "Too many iterations! epsilon is: " << std::abs(epsilon) << std::endl;
            epsilon = 0.5e-6;
          }
          // I want it to run while it can still have an effect on the elevation
        } while (abs(epsilon) > tol);

        // Avioding inversion there!!
        if(new_zeta < rec_to_elev[last_node])
          new_zeta = rec_to_elev[last_node];

        // embarassing renaming here as well
        float hnew = new_zeta;

        // Applying the new elevation field
        this_new_elev[i] = hnew;
        rec_to_elev[this_node] = hnew;
      }
      // Done with the elevation stuff

      // computing uplift
      for(size_t i=1; i<siz;i++)
      {
        this_new_elev[i] = this_new_elev[i] + uplift[i]*timestep;
      }

      // Saving this step if required
      // and adding to the global map if required
      if(tTt != 0)
      {
        if((tTt%save_step) == 0)
        {
          std::string gh = itoa(tTt * timestep); // Calculate the real time
          output[gh] = this_new_elev;
        }
      }
      // IMPoORTANT: giving the next elevation the new one
      last_elev = this_new_elev;
    }

    //We are basically done
    return output;


  }

  // Brief description before real doc: This burns to any map <string,float> ontaining x and y key the underlying raster info
  // No security check, make sure your rasters are good
  std::map<string,xt::pytensor<float,1> > burn_external_to_map(std::map<string,xt::pytensor<float,1> >& prebuilt, int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, 
    xt::pytensor<float,2>& data, std::string name_of_burned_column)
  {
    std::cout << "I am loading the data into LSDTT" << std::endl;
    // This will host the data. At the moment I need to copy the data from pytensors to TNT array. Not ideal but No other solution unfortunately.
    // Memory leak can become a problem for realy big rasters, however, the time loss is not important (few milliseconds).
    TNT::Array2D<float> data_pointerast(nrows,ncols,ndv); // Trying here to create a raster of pointers
    for(size_t i=0;i<nrows;i++)
    for(size_t j=0;j<ncols;j++)
      data_pointerast[i][j] = data(i,j);

    // Data is copied! I am ready to feed a LSDRaster object
    // Here we go: 
    LSDRaster PP_raster(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);
    std::cout << "Extracting the xy data, assigning nodata if point outside of the raster." << std::endl;

    // creating the array for output
    size_t siz = prebuilt["x"].size();
    std::array<size_t, 1> shape = {siz};
    xt::xtensor<float,1> new_col(shape);

    for(size_t t = 0; t < prebuilt["x"].size(); t++)
    {
      float tx = prebuilt["x"][t];
      float ty = prebuilt["y"][t];
      if(PP_raster.check_if_point_is_in_raster(tx,ty)){new_col[t] = PP_raster.get_value_of_point(tx,ty);}
      else {new_col[t] = ndv;}
    }

    // Done if in place stuff worked. I didn't, I am returning the shit
    prebuilt[name_of_burned_column] = new_col;
    std::cout << "Entry " << name_of_burned_column << " burned to the dictionnary/map" << std::endl; 

    return prebuilt;
  }




  std::map<std::string, xt::pytensor<float,1> > run_to_steady_state(float meq, float neq, int timestep, int save_step, float ratio_of_tolerance, float delta_tolerance, int min_iterations , xt::pytensor<float,1>& uplift, xt::pytensor<float,1>& erodility_K, std::map<std::string, xt::pytensor<float,1> >& prebuilt, int max_timestep )
  {

    // Creating the final output to fill
    std::map<std::string, xt::pytensor<float,1> > output;

    //global size of my arrays
    size_t siz = prebuilt["elevation"].size();

    // Setting arrays to store the new and last elevation
    std::array<size_t, 1> shape = { siz };
    xt::xtensor<float,1> last_elev(shape), this_new_elev(shape);
    // Setting base variables I need
    float baselevel = prebuilt["elevation"][0]; // elevation of the very first element. So far has to be fixed
    float last_h = baselevel, this_h=0; // navigating through the different elevtion layers
    float tol = 1e-3;// Tolerance for the newton-rhapson itarative solver for the stream power law
    int this_node,last_node;// Navigating through the node and its receivers
    last_elev = prebuilt["elevation"]; // initializing our previous elevation to the first one

    // Alright let's run the model
    // for(int tTt = 0; tTt < 1; tTt++) // Ingore that it's when I need to only run one model to check some stuff
    // Real run here
    int tTt = 0;
    bool not_steady_state = true;
    float ratio_tol = 0;
    while(not_steady_state && tTt < max_timestep)
    {

      tTt++;  // just a simple counter

      // These maps are storing the elevation and flow distance for each node IDs. 
      // this idea is to rn the model bottom up, store informations for current nodes and 
      // therefore get the receiver elevation/flow distance of previously processed node.
      std::map<int,float> rec_to_elev, rec_to_fd;
      // How far are we in the model. Let me stop my work here, there are pretty intense turbulences in my flights.
      // OK seems better now, f*ck these were impressive
      // So I was saying that we want to know which timestep we are at.
      // std::cout << "Processing time step: " << tTt+1 << " and tolerance ratio is " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"
      // Running through the river network:
      for(size_t i=0; i<siz;i++)
      {
        // We are in the first element, it is a bit different so we need a special case:
        // Saving elevation and stuff as baselevel, and giving the first receiver information
        if(i==0)
          {this_new_elev[i]=baselevel;last_h = baselevel; rec_to_elev[prebuilt["nodeID"][i]] = baselevel;rec_to_fd[prebuilt["nodeID"][i]] = prebuilt["flow_distance"][i];continue;}
        // Done, the continue statement at the end of last if makes sure it stop this loop run here if i == 0

        // I am getting my receiver and current nodes (available in the prebuilt model)
        last_node = prebuilt["receivers"][i];
        this_node = prebuilt["nodeID"][i];
        // Getting the length between these nodes (dx)
        float length = prebuilt["flow_distance"][i] - rec_to_fd[last_node];
        // getting the flow distance ready when next nodes will query this one as receiver
        rec_to_fd[this_node] = prebuilt["flow_distance"][i];
        // Getting the drainage area (from prebuilt model)
        float tDA = prebuilt["drainage_area"][i];
        // Getting the erodibility from input
        float keq = erodility_K[i];
        // Newton-rahpson method to solve non linear equation
        float epsilon;     // in newton's method, z_n+1 = z_n - f(z_n)/f'(z_n)
                         // and here epsilon =   f(z_n)/f'(z_n)
                         // f(z_n) = -z_n + z_old - dt*K*A^m*( (z_n-z_r)/dx )^n
                         // We differentiate the above equation to get f'(z_n)
                         // the resulting equation f(z_n)/f'(z_n) is seen below
        // A bit of factorisation to clarify the equation
        float streamPowerFactor = keq * pow(tDA, meq) * timestep;
        float slope; // Slope.
        float dx = length; // Embarrassing renaming of variables because I am lazy and got confused
        float new_zeta = last_elev[i]; float old_zeta = new_zeta; // zeta = z = elevation
        // iterate until you converge on a solution. Uses Newton's method.
        int iter_count = 0;
        do
        {
          // Get the slope
          slope = (new_zeta - rec_to_elev[last_node]) / dx;
          // Check backslope or no slope ie no erosion
          if(slope <= 0)
          {
            epsilon = 0;
          }
          else
          {
            // Applying the newton's method
            epsilon = (new_zeta - old_zeta + streamPowerFactor * std::pow(slope, neq)) /
                 (1 + streamPowerFactor * (neq/dx) * std::pow(slope, neq-1));
          }
          // iterate the result
          new_zeta -= epsilon;

          // This limits the number of iterations, it is a safety check to avoid infinite loop
          // Thsi will begin to split some  nan or inf if it diverges
          iter_count++;
          if(iter_count > 100)
          {
            // std::cout << "Too many iterations! epsilon is: " << std::abs(epsilon) << std::endl;
            epsilon = 0.5e-6;
          }
          // I want it to run while it can still have an effect on the elevation
        } while (abs(epsilon) > tol);

        // Avioding inversion there!!
        if(new_zeta < rec_to_elev[last_node])
          new_zeta = rec_to_elev[last_node];

        // embarassing renaming here as well
        float hnew = new_zeta;

        // Applying the new elevation field
        this_new_elev[i] = hnew;
        rec_to_elev[this_node] = hnew;
      }
      // Done with the elevation stuff

      // computing uplift
      for(size_t i=1; i<siz;i++)
      {
        this_new_elev[i] = this_new_elev[i] + uplift[i]*timestep;
      }

      // Saving this step if required
      // and adding to the global map if required
      if(tTt != 0)
      {
        if((tTt%save_step) == 0)
        {
          std::string gh = itoa(tTt * timestep); // Calculate the real time
          output[gh] = this_new_elev;
        }
      }


      // Now checking if we keep on eroding
      // Map has these entry:
      // delta_median
      // delta_mean
      // delta_stddev
      // delta_stderr
      // delta_FQ
      // delta_TQ
      // n_above
      // n_below
      // n_tolerated
      // n_untolerated
          // Setting arrays to store the new and last elevation
      xt::xtensor<float,1> flowdist(shape);
      flowdist = prebuilt["flow_distance"];
      std::map<std::string, float > compstat; compstat = calculate_comparative_metrics(last_elev, this_new_elev, flowdist, 50, delta_tolerance);
      ratio_tol = compstat["n_tolerated"]/siz;
      std::cout << "TS: " << tTt+1 << " tolrat: " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"

      // Let's try this simple approach:
      // Steady-state = when most of my points doesn't evolve?
      if(ratio_tol>ratio_of_tolerance && tTt > min_iterations)
      {
        not_steady_state = false;
      }

      // IMPoORTANT: giving the 
      // next elevation the new one
      last_elev = this_new_elev;
    }

    // Just saving the last time_step here
    std::string gh = itoa(tTt * timestep); // Calculate the real time
    output[gh] = this_new_elev;


    //We are basically done
    return output;

  }


  ///@brief Internal building function to generate Multi Threading Order (MTO) for my river system. 
  ///@details Multithreading is organised in the following way: first my main river is processed with a single thread (MTO=1)
  ///@details then all the channels draining into this first one can be processed parallely (MTO=2)
  ///@details This process can be repeated until the highest MTO is reached
  ///@param prebuilt: a classic prebuilt of TribBass
  ///@return I don't know yet actually...
  ///@author B.G.
  ///@Date 02/12/2018
  std::map<std::string, xt::pytensor<float,1> > generate_MTO(std::map<std::string, xt::pytensor<float,1> >& prebuilt)
  {

    size_t siz = prebuilt["elevation"].size();
    std::array<size_t,1> shape = {siz};
    xt::xtensor<float,1> MTO(shape);
    std::map<int, int> node_to_MTO;
    int this_MTO = 1, this_SK = prebuilt["elevation"][0]; int last_SK = this_SK;
    for(size_t i=0; i<prebuilt["elevation"].size(); i++)
    {
      int this_node = prebuilt["nodeID"][i];
      int receiver_node = prebuilt["receivers"][i];
      this_SK = prebuilt["source_key"][i];
      if(last_SK != this_SK)
      {
        this_MTO = node_to_MTO[receiver_node]+1;
      }

      node_to_MTO[i] = this_MTO;
      MTO[i] = this_MTO;
    }

    prebuilt["MTO"] = MTO;

    // OK Now I have my MTO, I need to gather the index info for each node
    std::pair<int,int> this_first_to_last;
    std::vector<std::pair<int,int> > these_pair_of_ID ; // This will store all pair of first-to-last element
    std::map<int, std::vector<std::pair<int,int> > > MTO_to_idx;
    for(size_t i=0; i< siz; i++)
    {

    }


  }



  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################
  //###################################################################################################




  //###################################################################################################
  //###################################################################################################
  //Internal_functions
  //###################################################################################################


  ///@brief Calculates basic m_chi (dz/dchi) inplace on given xtensors
  ///@param chi,elevation and m_chi xtensor<float,1>.
  ///@param node_index (xtensor<int,1>): contains the ID of each node
  ///@param node_to_rec (map<int,int>): contains the receiver of each node ID
  ///@author B.G.
  ///@date 27/11/2018
  void calculate_m_chi_basic_inplace(xt::xtensor<float,1>& elevation, xt::xtensor<float,1>& chi, xt::xtensor<float,1>& m_chi, xt::xtensor<int,1>& node_index, std::map<int,int>& node_to_rec)
  {

    // Few variables I need first
    std::map<int,float> rec_to_elev,rec_to_chi; // This will contain the lasts chi and elevation for receivers. Thanks to node ordering it can be filled in time.
    int this_node, last_node;
    //Let's loop
    for(size_t i =0; i< elevation.size();i++)
    {
      //First run = simple
      if(i==0){this_node=node_index[i];rec_to_elev[this_node]=elevation[i];rec_to_chi[this_node]=chi[i];m_chi[i]=0;continue;}

      //Others:
      this_node=node_index[i];
      float dz = elevation[i]-rec_to_elev[this_node]; rec_to_elev[this_node] = elevation[i];
      float dchi = chi[i]-rec_to_chi[this_node]; rec_to_chi[this_node]=chi[i];
      m_chi[i]=dz/dchi;
    }
  }

  ///@brief Evaluate a series of metrics to compare 2 1D y profiles having a common x.
  ///@brief This is to help detecting steady-state.
  ///@param profile 1 and profile 2 xtensor<float,1> and their x
  ///@param x_bin (float): the binning for the median profile NOT READY YET, NOT SURE IF I WILL NEED IR
  ///@param tolerance_detla (float): simply counts how many points are within a tolerance compare to their brother 
  ///@return comparative_metrics (map<string,xtensor<float,1> >)
  ///@return -> So far it has only the global comparison: 
  ///@return -> delta_median, delta_mean, delta_stddev, delta_stderr, delta_FQ, delta_TQ, n_above, n_below, n_tolerated, n_untolerated
  ///@autors B.G.
  ///@date 27/11/2018
  std::map<std::string, float > calculate_comparative_metrics(xt::xtensor<float,1>& elevation_1, xt::xtensor<float,1>& elevation_2, xt::xtensor<float, 1>& common_x, float x_bin, float tolerance_delta)
  {
    std::map<std::string, float > output;

    size_t sizla = elevation_1.size();
    // std::array<size_t,1> shape = {sizla};
    // xt::xtensor<float,1> temp_output(shape);
    std::vector<float> temp_delta; // I need a temp vector to use the different statstools in LSDTT 
    int n_below=0, n_above=0, n_within_tol = 0, n_outside_tol = 0; // stores how many values  are above or below
    // First I want to get the delta and associated stats
    for (size_t i=0; i<sizla; i++)
    {
      float this_delta = elevation_1[i]-elevation_2[i];
      temp_delta.push_back(this_delta);
      if(this_delta>=0){n_above++;}else{n_below++;}
      if(abs(this_delta)>tolerance_delta){n_outside_tol++;}else{n_within_tol++;}
    }
    // Alright let's get basic stats out of that now
    output["delta_median"] = get_median(temp_delta);
    float mean = get_mean(temp_delta);
    output["delta_mean"] = mean;
    output["delta_stddev"] = get_standard_deviation(temp_delta, mean);
    output["delta_stderr"] = get_standard_error(temp_delta, output["delta_stddev"]);
    output["delta_FQ"] = get_percentile(temp_delta, 25);
    output["delta_TQ"] = get_percentile(temp_delta, 75);
    output["n_above"] = float(n_above); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
    output["n_below"] = float(n_below); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
    output["n_tolerated"] = float(n_within_tol); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
    output["n_untolerated"] = float(n_outside_tol); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it


    return output;
  }


};

////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////// Builing a new Class here, cleaner approach now that first tests are promising ///////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////



void LSDTribBas::create()
{ 
  std::cout << "I need information to be created!" << std::endl; 
  std::exit(EXIT_FAILURE);
}

void LSDTribBas::create(int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, 
  xt::pytensor<float,2>& data, xt::pytensor<float,1>& x_sources, xt::pytensor<float,1>& y_sources, 
  float m, float n, float A0, double Northing, double Easting, int search_radius_nodes,
  int target_nodes, int n_iterations, int skip, int minimum_segment_length, int sigma)
{
  std::cout << "I am loading the data into LSDTT" << std::endl;
  // This will host the data. At the moment I need to copy the data from pytensors to TNT array. Not ideal but No other solution unfortunately.
  // Memory leak can become a problem for realy big rasters, however, the time loss is not important (few milliseconds, yes I benchmarked it because of my coding OCD).
  TNT::Array2D<float> data_pointerast(nrows,ncols,ndv); // Trying here to create a raster of pointers
  for(size_t i=0;i<nrows;i++)
  for(size_t j=0;j<ncols;j++)
    data_pointerast[i][j] = data(i,j);
  std::cout << "Loaded" << std::endl;
  // Global variable set: I want my m and n to be constant!! or eventually to really specify which one we want
  meq = m;
  neq = n;

  // Data is copied! I am ready to feed a LSDRaster object
  // Here we go: 
  LSDRaster PP_raster(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);

  // Now I need all the flow informations
  std::vector<std::string> bound = {"n","n","n","n"};
  LSDFlowInfo FlowInfo(bound,PP_raster);
  // OK I got it
  
  // Calculating the flow accumulation and the ...
  LSDIndexRaster FlowAcc = FlowInfo.write_NContributingNodes_to_LSDIndexRaster();
  // Drainage area
  LSDRaster DrainageArea = FlowInfo.write_DrainageArea_to_LSDRaster();
  // DOne

  // Getting the flow distance
  LSDRaster DistanceFromOutlet = FlowInfo.distance_from_outlet();

  // Initialising the channel heads
  std::vector<int> sources; //  sources will contain the node index (unique identifer for each node) of me sources
  // # I first need to copy the xy cordinates into vector for lsdtt
  std::vector<float> x_sources_v, y_sources_v;
  for (size_t yuio = 0; yuio<x_sources.size(); yuio ++)
    {x_sources_v.push_back(x_sources(yuio));y_sources_v.push_back(y_sources(yuio));}
  // ~ DOne
  sources = FlowInfo.Ingest_Channel_Heads(x_sources_v,y_sources_v);
  // I got me channels head

  // Lets extract my channels
  // now get the junction network
  LSDJunctionNetwork JunctionNetwork(sources, FlowInfo);
  // Ok, great, I preprocessed all my raw rivers

  // Now let's get the basin outlet
  // # A bit of hacky way to get the data into the right format
  std::vector<float> fUTM_easting; fUTM_easting.push_back(Easting);
  std::vector<float> fUTM_northing; fUTM_northing.push_back(Northing);
  int threshold_stream_order = 2;
  std::vector<int> valid_cosmo_points;         // a vector to hold the valid nodes
  std::vector<int> snapped_node_indices;       // a vector to hold the valid node indices
  std::vector<int> snapped_junction_indices;   // a vector to hold the valid junction indices
  // Sending the coordinates to the outlet tracker
  JunctionNetwork.snap_point_locations_to_channels(fUTM_easting, fUTM_northing, 
              search_radius_nodes, threshold_stream_order, FlowInfo, 
              valid_cosmo_points, snapped_node_indices, snapped_junction_indices);
  std::vector<int> BaseLevelJunctions = snapped_junction_indices;
  // Done

  // Results check
  if(BaseLevelJunctions.size() == 0)
  {
    std::cout << "ERROR: I couldn't find any basin with your criteri. Increase search_radius_nodes maybe??" << std::endl;
    std::exit(EXIT_FAILURE);
  }
  // Let me tell you outside of the code just in case you are wondering what is happenning
  std::cout << "Successfully detected the outlet of a basin! Hopefully the one you had in mind" << std::endl;

  // Ignore but keep that in case
  // BaseLevelJunctions = JunctionNetwork.Prune_Junctions_If_Nested(BaseLevelJunctions,FlowInfo, FlowAcc);
  // Small but required cleaning
  std::vector<int> outlet_nodes, baselevel_node_of_each_basin;
  JunctionNetwork.get_overlapping_channels(FlowInfo, BaseLevelJunctions, DistanceFromOutlet,
                                  sources, outlet_nodes,baselevel_node_of_each_basin,10);

  // The steps below are getting (i) mchi and related values and (ii) the node structure of our river network
  // Now let's get the chitools N stuffs
  LSDChiTools ChiTool(FlowInfo);
  // Getting chi because we kind of need it
  LSDRaster chi_coordinate = FlowInfo.get_upslope_chi_from_all_baselevel_nodes(m/n,A0,1);
  // getting mchi
  ChiTool.chi_map_automator(FlowInfo, sources, outlet_nodes, baselevel_node_of_each_basin,
                        PP_raster, DistanceFromOutlet,
                        DrainageArea, chi_coordinate, target_nodes,
                        n_iterations, skip, minimum_segment_length, sigma);
  ChiTool.segment_counter(FlowInfo, 1000000);

  // Now getting the ndoe sequence and the data maps
  // The node_sequence correspond to the ordered fastscape node index
  // The maps contains elevation, flow distance ... for each of these node indexes
  std::map<std::string, std::map<int,float> > mahmaps =  ChiTool.get_data_maps();
  std::vector<int> node_sequence =  ChiTool.get_vectors_of_node();


  // # THat's the way you create a xtensor array
  std::array<size_t, 1> shape = { node_sequence.size() };
  xt::xtensor<float,1> TT_elev(shape);
  xt::xtensor<float,1> TT_chi(shape);
  xt::xtensor<float,1> TT_flowdis(shape);
  xt::xtensor<float,1> TT_node_seq(shape);
  xt::xtensor<float,1> TT_rec(shape);
  xt::xtensor<float,1> TT_SK(shape);
  xt::xtensor<float,1> TT_DA(shape);
  xt::xtensor<int,1> TT_row(shape);
  xt::xtensor<int,1> TT_col(shape);
  xt::xtensor<float,1> TT_these_x(shape);
  xt::xtensor<float,1> TT_these_y(shape);
  xt::xtensor<float,1> TT_mchi(shape);

  // OK now I am dealing with node ordering. I did not manage to extrct it correctly within LSDTT for some reason
  // So here is a quick and dirty way to order my nodes in a modelling friendly way
  // # First I am creating a priority queue with a greater logic 
  std::priority_queue< TriBas::sortation, std::vector<TriBas::sortation>, std::greater<TriBas::sortation> > myp;
  std::vector<int> temp_node, totnode;// temp variables
  // Sorting my rivers:
  float last_elevation_for_sorting = mahmaps["elevation"][node_sequence[0]];
  int this_SK = mahmaps["SK"][node_sequence[0]]; int last_SK = this_SK;
  for (size_t i=0;i<node_sequence.size(); i++)
  {
    // the idea is to separate each of my rivers and get the lower ones at the bottom of my queue
    // Getting the current cource key
    this_SK = mahmaps["SK"][node_sequence[i]];
    if(this_SK != last_SK)
    {
      // If I changed e source key (my river chunk) I ewant to save it into my queue
      // Nodes are sorted top to bottom per rivers -> here is the bottom to top one
      std::reverse(temp_node.begin(), temp_node.end());
      // Creating my node structure here, see description above
      TriBas::sortation temp_sortation;
      temp_sortation.Node_of_river = temp_node; // vectore of nodes
      temp_sortation.base_elevation = last_elevation_for_sorting; // elevation of the lowest cell
      myp.push(temp_sortation); // feeding my queue (highly questionable sentence is translated in french)
      temp_node.clear(); // Getting it ready for my next loop run 
    }

    // I do that all the time anyway
    temp_node.push_back(node_sequence[i]); // adding my node to the vector
    // Saving the last elevation in case next one is a change
    last_elevation_for_sorting = mahmaps["elevation"][node_sequence[i]];
    // saving current sk to check if change
    last_SK = this_SK;
  }
  // I need to do it a last time to save the last river
  std::reverse(temp_node.begin(), temp_node.end());
  // Creating my node structure here, see description above
  TriBas::sortation temp_sortation;
  temp_sortation.Node_of_river = temp_node; // vectore of nodes
  temp_sortation.base_elevation = last_elevation_for_sorting; // elevation of the lowest cell
  myp.push(temp_sortation); // feeding my queue (highly questionable sentence is translated in french)
  temp_node.clear(); // Getting it ready for my next loop run 
  // DOne
  // Now I kind of need to recompile my data into one single tensor
  while(myp.size()>0) // While my priority queue is not empty
  {
    // Getting my vector of lowest river nodes
    std::vector<int> this_veconde;
    TriBas::sortation hugh = myp.top(); // this gives the elements at the top of the queue
    this_veconde = hugh.Node_of_river;
    myp.pop(); //  this removes the abovementioned river from my stack
    // Now simply pushing the node into me global node vector
    for (size_t u =0 ; u < this_veconde.size() ; u++) {totnode.push_back(this_veconde[u]);}
  }
  // Nodes should be ordered now

  // Last but not least I am feeding my output map. Not detailing here as it is simple copy.
  for(size_t i=0; i<totnode.size(); i++)
  {
    int this_node = totnode[i]; int tr; int tc;
    FlowInfo.retrieve_current_row_and_col(this_node,tr,tc);
    TT_node_seq[i] = this_node;
    int recnode =0;
    FlowInfo.retrieve_receiver_information(this_node,recnode, tr,tc);
    TT_rec[i] = recnode;
    // std::cout << recnode << "||" << std::endl;
    recnode_global[this_node] = recnode;
    TT_chi[i] = mahmaps["chi"][this_node];
    TT_flowdis[i] = mahmaps["flow_distance"][this_node];
    TT_SK[i] = mahmaps["SK"][this_node];
    TT_DA[i] = mahmaps["DA"][this_node];
    TT_row[i] = tr;
    TT_col[i] = tc;
    float tx,ty;
    FlowInfo.get_x_and_y_locations(tr,tc,tx,ty);
    TT_these_x[i] = tx;
    TT_these_y[i] = ty;
    TT_elev[i] = mahmaps["elevation"][this_node];
    TT_mchi[i] = mahmaps["m_chi"][this_node];

    // if(elev[i] == -9999 || mahmaps["elevation"].count(this_node) == 0)
    //   {std::cout << "GARGGGGGGG: " << elev[i] << std::endl;}
    // if(elev[i] == -9999 || mahmaps["elevation"].count(recnode) == 0)
    //   {std::cout << "GARGGGGGGG_recnode: " << elev[i] << std::endl;}
    if(mahmaps["elevation"].count(recnode) == 0)
    {
      bool keep_checking = true;
      int noronode = 0;
      while(keep_checking)
      {
        FlowInfo.retrieve_receiver_information(recnode,noronode,tr,tc);
        if(mahmaps["elevation"].count(noronode)==0 || noronode == -9999)
        {
          keep_checking = false;
        }
        else
        {
          recnode = noronode;
        }
      }
      TT_rec[i] = recnode;
    }
  }

  // transferring to the global variables
  node_ID = TT_node_seq;
  chi = TT_chi;
  elevation = TT_elev;
  flow_distance = TT_flowdis;
  source_key = TT_SK;
  drainage_area = TT_DA;
  receivers = TT_rec;
  raster_row = TT_row;
  raster_col = TT_col;
  x_coord = TT_these_x;
  y_coord = TT_these_y;
  m_chi = TT_mchi;

  // MTO generation
  generate_MTO();

}

struct rivIndexByLength
{
  float length;
  std::pair<int,int> indexes_bounds;    
};
// These are the operator used to sort the river per grater/lower length in the priority queue
bool operator>( const rivIndexByLength& lhs, const rivIndexByLength& rhs )
{
  return lhs.length > rhs.length;
};
bool operator<( const rivIndexByLength& lhs, const rivIndexByLength& rhs )
{
  return lhs.length < rhs.length;
};

///@brief Internal building function to generate Multi Threading Order (MTO) for my river system. 
///@details Multithreading is organised in the following way: first my main river is processed with a single thread (MTO=1)
///@details then all the channels draining into this first one can be processed parallely (MTO=2)
///@details This process can be repeated until the highest MTO is reached
///@param prebuilt: a classic prebuilt of TribBass
///@return I don't know yet actually...
///@author B.G.
///@Date 02/12/2018
void LSDTribBas::generate_MTO()
{

  // Creating a structure only valid for this function: 
  // I need to sort my indexes within each MTO to get the longest river processed firts
  // using a priority queue with struct to do that

  size_t siz = elevation.size();
  std::array<size_t,1> shape = {siz};
  xt::xtensor<float,1> MTO(shape);
  std::map<int, int> node_to_MTO;
  int this_MTO = 1, this_SK = source_key[0]; int last_SK = this_SK;

  max_MTO = 0;
  for(size_t i=0; i<siz; i++)
  {
    int this_node = node_ID[i];
    int receiver_node = receivers[i];
    this_SK = source_key[i];
    if(last_SK != this_SK)
    {
      this_MTO = node_to_MTO[receiver_node]+1;
      if(this_MTO > max_MTO){max_MTO=this_MTO;}
    }

    node_to_MTO[this_node] = this_MTO;
    MTO[i] = this_MTO;
    last_SK = this_SK;
  }

  MTO_at_node = MTO;
  // max_MTO=this_MTO+1;
  // OK Now I have my MTO, I need to gather the index info for each node
  std::pair<int,int> this_first_to_last;
  std::vector<std::vector<std::pair<int,int> > > TGH(max_MTO);
  std::vector<std::pair<int,int> > these_pair_of_ID ; // This will store all pair of first-to-last element
  for(size_t t=0; t<TGH.size();t++){TGH[t]=these_pair_of_ID;}
  MTO_global_index = TGH;

  int first_ID = 0; int last_ID = 0; this_MTO = MTO_at_node[0]; int last_MTO = MTO_at_node[0];

  for(size_t i=0; i< siz; i++)
  {

    this_MTO = MTO_at_node[i];

    if(this_MTO != last_MTO)
    {

      this_first_to_last = std::make_pair(first_ID, i-1);

      first_ID = int(i);
      if((this_MTO-1)>= MTO_global_index.size()){std::cout << "DEBUG:: You should definitely not see this message... Node ordering can be fucked up: MTO = " << this_MTO << std::endl;}
      else{MTO_global_index[this_MTO-1].push_back(this_first_to_last);}


    }
    if(i == siz-1)
    {
      this_first_to_last = std::make_pair(first_ID, i);
      MTO_global_index[this_MTO-1].push_back(this_first_to_last);
    }
    last_MTO = this_MTO;
  }

  // Now last step is to sort my nodes into the priority queue
  for(size_t i=0; i<MTO_global_index.size(); i++)
  {
    // std::Lower because I need the biggest river to be first 
    std::priority_queue< rivIndexByLength, std::vector<rivIndexByLength>, std::less<rivIndexByLength> > datQ;
    std::vector<std::pair<int,int> > datVec = MTO_global_index[i];
    // Feeding the Queue
    for (size_t j=0; j<datVec.size(); j++)
    {
      // Getting all the data I need
      std::pair<int,int> datPair = datVec[j];
      int datFirstIndex = datPair.first;
      int datLastIndex = datPair.second;
      rivIndexByLength datRiv; datRiv.length = flow_distance[datFirstIndex] - flow_distance[datFirstIndex]; datRiv.indexes_bounds = datVec[j];
      // implementing
      datQ.push(datRiv);
    }

    //OK now I have my rivers by size
    // I need to refeed my MTO_gobal_index
    size_t cpt = 0;
    while(datQ.size()>0) // While my priority queue is not empty
    {
      // Getting my vector of lowest river nodes
      std::pair<int,int> datPair;
      rivIndexByLength hugh = datQ.top(); // this gives the elements at the top of the queue
      datPair = hugh.indexes_bounds;
      datQ.pop(); //  this removes the abovementioned river from my stack
      // Now simply correcting the 
      MTO_global_index[i][cpt] = datPair;
    }

  }


  // SHould be Done now


}




float LSDTribBas::newton_rahpson_solver(float keq, float meq, float neq, float tDA, float dt, float dx, float this_elevation, float receiving_elevation ,float tol)
{
  // Newton-rahpson method to solve non linear equation, see Braun and Willett 2013
  float epsilon;     // in newton's method, z_n+1 = z_n - f(z_n)/f'(z_n)
                   // and here epsilon =   f(z_n)/f'(z_n)
                   // f(z_n) = -z_n + z_old - dt*K*A^m*( (z_n-z_r)/dx )^n
                   // We differentiate the above equation to get f'(z_n)
                   // the resulting equation f(z_n)/f'(z_n) is seen below
  // A bit of factorisation to clarify the equation
  float streamPowerFactor = keq * pow(tDA, meq) * dt;
  float slope; // Slope.
  float new_zeta = this_elevation; float old_zeta = new_zeta; // zeta = z = elevation
  // iterate until you converge on a solution. Uses Newton's method.
  int iter_count = 0;
  do
  {
    // Get the slope
    slope = (new_zeta - receiving_elevation) / dx;
    // Check backslope or no slope ie no erosion
    if(slope <= 0)
    {
      epsilon = 0;
    }
    else
    {
      // Applying the newton's method
      epsilon = (new_zeta - old_zeta + streamPowerFactor * std::pow(slope, neq)) /
           (1 + streamPowerFactor * (neq/dx) * std::pow(slope, neq-1));
    }
    // iterate the result
    new_zeta -= epsilon;

    // This limits the number of iterations, it is a safety check to avoid infinite loop
    // Thsi will begin to split some  nan or inf if it diverges
    iter_count++;
    if(iter_count > 100)
    {
      std::cout << "Too many iterations! epsilon is: " << std::abs(epsilon) << std::endl;
      epsilon = 0.5e-6;
    }
    // I want it to run while it can still have an effect on the elevation
  } while (abs(epsilon) > tol);

  // Avioding inversion there!!
  if(new_zeta < receiving_elevation)
    new_zeta = receiving_elevation;

  // std::cout << receiving_elevation<< "||" <<this_elevation <<"||"<< new_zeta << std::endl;

  return new_zeta;
}

// Brief description before real doc: This burns to any map <string,float> ontaining x and y key the underlying raster info
// No security check, make sure your rasters are good
void LSDTribBas::ingest_external_data_from_xy(int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, 
  xt::pytensor<float,2>& data, std::string name_of_burned_column)
{
  std::cout << "I am loading the data into LSDTT" << std::endl;
  // This will host the data. At the moment I need to copy the data from pytensors to TNT array. Not ideal but No other solution unfortunately.
  // Memory leak can become a problem for realy big rasters, however, the time loss is not important (few milliseconds).
  TNT::Array2D<float> data_pointerast(nrows,ncols,ndv); // Trying here to create a raster of pointers
  for(size_t i=0;i<nrows;i++)
  for(size_t j=0;j<ncols;j++)
    data_pointerast[i][j] = data(i,j);

  // Data is copied! I am ready to feed a LSDRaster object
  // Here we go: 
  LSDRaster PP_raster(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);
  std::cout << "Extracting the xy data, assigning nodata if point outside of the raster." << std::endl;

  // creating the array for output
  size_t siz = elevation.size();
  std::array<size_t, 1> shape = {siz};
  xt::xtensor<float,1> new_col(shape);

  for(size_t t = 0; t < elevation.size(); t++)
  {
    float tx = x_coord[t];
    float ty = y_coord[t];
    if(PP_raster.check_if_point_is_in_raster(tx,ty)){new_col[t] = PP_raster.get_value_of_point(tx,ty);}
    else {new_col[t] = ndv;}
  }

  // Done if in place stuff worked. I didn't, I am returning the shit
  external_data[name_of_burned_column] = new_col;
  std::cout << "Entry " << name_of_burned_column << " burned to the LSDObject" << std::endl; 

}


std::map<std::string,xt::pytensor<float,1> > LSDTribBas::run_model(int timestep, int save_step, float ratio_of_tolerance, float delta_tolerance, int min_iterations , xt::pytensor<float,1>& uplift, xt::pytensor<float,1>& erodility_K, int max_timestep )
{

  // Creating the final output to fill
  std::map<std::string, xt::pytensor<float,1> > output;

  //global size of my arrays (elevation is an attribute)
  size_t siz = elevation.size();

  // Setting arrays to store the new and last elevation
  std::array<size_t, 1> shape = { siz };
  xt::xtensor<float,1> last_elev(shape), this_new_elev(shape);
  // Setting base variables I need
  float baselevel = elevation[0]; // elevation of the very first element. So far has to be fixed
  float last_h = baselevel, this_h=0; // navigating through the different elevtion layers
  float tol = 1e-3;// Tolerance for the newton-rhapson itarative solver for the stream power law
  int this_node,last_node;// Navigating through the node and its receivers
  last_elev = elevation; // initializing our previous elevation to the first one

  // Alright let's run the model
  // for(int tTt = 0; tTt < 1; tTt++) // Ingore that it's when I need to only run one model to check some stuff
  // Real run here
  int tTt = 0;
  bool not_steady_state = true;
  float ratio_tol = 0;
  while(not_steady_state && tTt < max_timestep)
  {

    tTt++;  // just a simple counter

    // These maps are storing the elevation and flow distance for each node IDs. 
    // this idea is to rn the model bottom up, store informations for current nodes and 
    // therefore get the receiver elevation/flow distance of previously processed node.
    std::map<int,float> rec_to_elev, rec_to_fd;
    // How far are we in the model. Let me stop my work here, there are pretty intense turbulences in my flights.
    // OK seems better now, f*ck these were impressive
    // So I was saying that we want to know which timestep we are at.
    // std::cout << "Processing time step: " << tTt+1 << " and tolerance ratio is " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"
    // Running through the river network:
    for(size_t i=0; i<siz;i++)
    {
      // We are in the first element, it is a bit different so we need a special case:
      // Saving elevation and stuff as baselevel, and giving the first receiver information
      if(i==0)
        {this_new_elev[i]=baselevel;last_h = baselevel; rec_to_elev[node_ID[i]] = baselevel;rec_to_fd[node_ID[i]] = flow_distance[i];continue;}
      // Done, the continue statement at the end of last if makes sure it stop this loop run here if i == 0

      // I am getting my receiver and current nodes (available in the prebuilt model)
      last_node = receivers[i];
      // std::cout << "VAGUL" << last_node << std::endl;
      this_node = node_ID[i];
      // Getting the length between these nodes (dx)
      float length = flow_distance[i] - rec_to_fd[last_node];
      // getting the flow distance ready when next nodes will query this one as receiver
      rec_to_fd[this_node] = flow_distance[i];
      // Getting the drainage area (from prebuilt model)
      float tDA = drainage_area[i];
      // Getting the erodibility from input
      float keq = erodility_K[i];

      float hnew = newton_rahpson_solver(keq, meq, neq,  tDA,  timestep,  length, last_elev[i], rec_to_elev[last_node] ,tol);
       

      // Applying the new elevation field
      this_new_elev[i] = hnew;
      rec_to_elev[this_node] = hnew;
    }
    // Done with the elevation stuff

    // computing uplift
    for(size_t i=1; i<siz;i++)
    {
      this_new_elev[i] = this_new_elev[i] + uplift[i]*timestep;
    }

    // Saving this step if required
    // and adding to the global map if required
    if(tTt != 0)
    {
      if((tTt%save_step) == 0)
      {
        std::string gh = itoa(tTt * timestep); // Calculate the real time
        output[gh] = this_new_elev;
      }
    }


    // Now checking if we keep on eroding
    // Map has these entry:
    // delta_median
    // delta_mean
    // delta_stddev
    // delta_stderr
    // delta_FQ
    // delta_TQ
    // n_above
    // n_below
    // n_tolerated
    // n_untolerated
    // Setting arrays to store the new and last elevation
    xt::xtensor<float,1> flowdist(shape);
    flowdist = flow_distance;
    std::map<std::string, float > compstat; compstat = TriBas::calculate_comparative_metrics(last_elev, this_new_elev, flowdist, 50, delta_tolerance);
    ratio_tol = compstat["n_tolerated"]/siz;
    // std::cout << "TS: " << tTt+1 << " tolrat: " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"

    // Let's try this simple approach:
    // Steady-state = when most of my points doesn't evolve?
    if(ratio_tol>ratio_of_tolerance && tTt > min_iterations)
    {
      not_steady_state = false;
    }

    // IMPoORTANT: giving the 
    // next elevation the new one
    last_elev = this_new_elev;
  }

  // Just saving the last time_step here
  std::string gh = itoa(tTt * timestep); // Calculate the real time
  output[gh] = this_new_elev;

  // Saving my last stage because you might need it at some point
  last_elevation_modelled = this_new_elev;
  last_U_modelled = uplift;
  last_K_modelled = erodility_K;


  //We are basically done
  return output;

}

// Derive chi-elevation profile from custom chi and elevation array, but with the same receiver-node indexing
// this is suitable for modelled data
xt::pytensor<float,1> LSDTribBas::first_order_m_chi_from_custarray(xt::pytensor<float,1>& this_chi, xt::pytensor<float,1>& this_elevation)
{

  // Alright let's do it
  // output initialization
  size_t shape = chi.size();
  std::array<size_t,1> sizla = {shape};
  xt::xtensor<float,1> c_m_chi(sizla); c_m_chi[0] = 0;
  std::map<int,float> chi_receiver,elev_receiver;
  chi_receiver[node_ID[0]] = this_chi[0]; elev_receiver[node_ID[0]] = this_elevation[0];

  for(size_t i=1; i<shape; i++)
  {
    int this_node = node_ID[i]; int rec_node = receivers[i];
    float dz = this_elevation[i] - elev_receiver[rec_node];
    float dchi = this_chi[i] - chi_receiver[i];
    chi_receiver[this_node] = this_chi[i];
    elev_receiver[this_node] = this_elevation[i];
    // Let's get mchi
    c_m_chi[i] = dz/dchi;
    // Alright done
  }

  return c_m_chi;
}


///@brief Run the model, Multithreading version (I want to keep those separated, small catchment will be faster in serial).
///@brief Work in progress. Unstable.
std::map<std::string,xt::pytensor<float,1> > LSDTribBas::run_model_parallel(int timestep, int save_step, float ratio_of_tolerance, float delta_tolerance, int min_iterations , xt::pytensor<float,1>& uplift, xt::pytensor<float,1>& erodility_K, int max_timestep )
{
  
  // Creating the final output to fill
  std::map<std::string, xt::pytensor<float,1> > output;

  //global size of my arrays (elevation is an attribute)
  size_t siz = elevation.size();

  // Setting arrays to store the new and last elevation
  std::array<size_t, 1> shape = { siz };
  xt::xtensor<float,1> last_elev(shape), this_new_elev(shape);
  // Setting base variables I need
  float baselevel = elevation[0]; // elevation of the very first element. So far has to be fixed
  float last_h = baselevel, this_h=0; // navigating through the different elevtion layers
  float tol = 1e-3;// Tolerance for the newton-rhapson itarative solver for the stream power law
  int this_node,last_node;// Navigating through the node and its receivers
  last_elev = elevation; // initializing our previous elevation to the first one

  // Alright let's run the model
  // for(int tTt = 0; tTt < 1; tTt++) // Ingore that it's when I need to only run one model to check some stuff
  // Real run here
  int tTt = 0;
  bool not_steady_state = true;
  float ratio_tol = 0;
  while(not_steady_state && tTt < max_timestep)
  {

    tTt++;  // just a simple counter

    // These maps are storing the elevation and flow distance for each node IDs. 
    // this idea is to rn the model bottom up, store informations for current nodes and 
    // therefore get the receiver elevation/flow distance of previously processed node.
    std::map<int,float> rec_to_elev, rec_to_fd;
    // How far are we in the model. Let me stop my work here, there are pretty intense turbulences in my flights.
    // OK seems better now, f*ck these were impressive
    // So I was saying that we want to know which timestep we are at.
    // std::cout << "Processing time step: " << tTt+1 << " and tolerance ratio is " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"
    // Running through the river network:
    for(size_t i=0; i<siz;i++)
    {
      // We are in the first element, it is a bit different so we need a special case:
      // Saving elevation and stuff as baselevel, and giving the first receiver information
      if(i==0)
        {this_new_elev[i]=baselevel;last_h = baselevel; rec_to_elev[node_ID[i]] = baselevel;rec_to_fd[node_ID[i]] = flow_distance[i];continue;}
      // Done, the continue statement at the end of last if makes sure it stop this loop run here if i == 0

      // I am getting my receiver and current nodes (available in the prebuilt model)
      last_node = receivers[i];
      this_node = node_ID[i];
      // Getting the length between these nodes (dx)
      float length = flow_distance[i] - rec_to_fd[last_node];
      // getting the flow distance ready when next nodes will query this one as receiver
      rec_to_fd[this_node] = flow_distance[i];
      // Getting the drainage area (from prebuilt model)
      float tDA = drainage_area[i];
      // Getting the erodibility from input
      float keq = erodility_K[i];

      float hnew = newton_rahpson_solver(keq, meq, neq,  tDA,  timestep,  length, last_elev[i], rec_to_elev[node_ID[i]] ,tol);
       

      // Applying the new elevation field
      this_new_elev[i] = hnew;
      rec_to_elev[this_node] = hnew;
    }
    // Done with the elevation stuff

    // computing uplift
    for(size_t i=1; i<siz;i++)
    {
      this_new_elev[i] = this_new_elev[i] + uplift[i]*timestep;
    }

    // Saving this step if required
    // and adding to the global map if required
    if(tTt != 0)
    {
      if((tTt%save_step) == 0)
      {
        std::string gh = itoa(tTt * timestep); // Calculate the real time
        output[gh] = this_new_elev;
      }
    }


    // Now checking if we keep on eroding
    // Map has these entry:
    // delta_median
    // delta_mean
    // delta_stddev
    // delta_stderr
    // delta_FQ
    // delta_TQ
    // n_above
    // n_below
    // n_tolerated
    // n_untolerated
    // Setting arrays to store the new and last elevation
    xt::xtensor<float,1> flowdist(shape);
    flowdist = flow_distance;
    std::map<std::string, float > compstat; compstat = TriBas::calculate_comparative_metrics(last_elev, this_new_elev, flowdist, 50, delta_tolerance);
    ratio_tol = compstat["n_tolerated"]/siz;
    // std::cout << "TS: " << tTt+1 << " tolrat: " << ratio_tol << "\r"; // the last sign means "Return to the beginning of my line and flush"

    // Let's try this simple approach:
    // Steady-state = when most of my points doesn't evolve?
    if(ratio_tol>ratio_of_tolerance && tTt > min_iterations)
    {
      not_steady_state = false;
    }

    // IMPoORTANT: giving the 
    // next elevation the new one
    last_elev = this_new_elev;
  }

  // Just saving the last time_step here
  std::string gh = itoa(tTt * timestep); // Calculate the real time
  output[gh] = this_new_elev;

  // Saving my last stage because you might need it at some point
  last_elevation_modelled = this_new_elev;
  last_U_modelled = uplift;
  last_K_modelled = erodility_K;


  //We are basically done
  return output;

}



// ///@brief Evaluate a series of metrics to compare 2 1D y profiles having a common x.
// ///@brief This is to help detecting steady-state.
// ///@param profile 1 and profile 2 xtensor<float,1> and their x
// ///@param x_bin (float): the binning for the median profile NOT READY YET, NOT SURE IF I WILL NEED IR
// ///@param tolerance_detla (float): simply counts how many points are within a tolerance compare to their brother 
// ///@return comparative_metrics (map<string,xtensor<float,1> >)
// ///@return -> So far it has only the global comparison: 
// ///@return -> delta_median, delta_mean, delta_stddev, delta_stderr, delta_FQ, delta_TQ, n_above, n_below, n_tolerated, n_untolerated
// ///@autors B.G.
// ///@date 27/11/2018
// std::map<std::string, float > LSDTribBas::calculate_comparative_metrics(xt::xtensor<float,1>& elevation_1, xt::xtensor<float,1>& elevation_2, xt::xtensor<float, 1>& common_x, float x_bin, float tolerance_delta)
// {
//   std::map<std::string, float > output;

//   size_t sizla = elevation_1.size();
//   // std::array<size_t,1> shape = {sizla};
//   // xt::xtensor<float,1> temp_output(shape);
//   std::vector<float> temp_delta; // I need a temp vector to use the different statstools in LSDTT 
//   int n_below=0, n_above=0, n_within_tol = 0, n_outside_tol = 0; // stores how many values  are above or below
//   // First I want to get the delta and associated stats
//   for (size_t i=0; i<sizla; i++)
//   {
//     float this_delta = elevation_1[i]-elevation_2[i];
//     temp_delta.push_back(this_delta);
//     if(this_delta>=0){n_above++;}else{n_below++;}
//     if(abs(this_delta)>tolerance_delta){n_outside_tol++;}else{n_within_tol++;}
//   }
//   // Alright let's get basic stats out of that now
//   output["delta_median"] = get_median(temp_delta);
//   float mean = get_mean(temp_delta);
//   output["delta_mean"] = mean;
//   output["delta_stddev"] = get_standard_deviation(temp_delta, mean);
//   output["delta_stderr"] = get_standard_error(temp_delta, output["delta_stddev"]);
//   output["delta_FQ"] = get_percentile(temp_delta, 25);
//   output["delta_TQ"] = get_percentile(temp_delta, 75);
//   output["n_above"] = float(n_above); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
//   output["n_below"] = float(n_below); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
//   output["n_tolerated"] = float(n_within_tol); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it
//   output["n_untolerated"] = float(n_outside_tol); // Bad recasting here but I am lazy... class approach would solve that. I'll think about it


//   return output;
// }








void LSDTribBas::simplification_by_median(float flow_distance_bin)
{
  // This function should take the prebuit model and downgrade to smaller size. The idea is to optimise it by finding the right states for simpler profile before going to the full version of it
  // I Will use the MTO order to generate the simplification
  std::vector<int> node_simID_temp;
  for (int tmto = 0; tmto< max_MTO; tmto++)
  {
    // first step is to get each river bounds on 
    std::vector<std::pair<int,int> > these_river = MTO_global_index[tmto];
    for(size_t it1=0; it1<these_river.size();it1++)

    {
      size_t first_index = these_river[it1].first;
      size_t last_index = these_river[it1].second;
    
      // If my river is too small, I am not even bothering with it
      float size_of_this_river = flow_distance[last_index] - flow_distance[last_index];
      if(size_of_this_river <= flow_distance_bin)
      {std::cout << "getting there but on hold for now" << std::endl;}
    }

  }

}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////// Bindings for Muddpile thereafter //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


#ifdef __linux__

void muddpyle::create()
{
  // Empty creator
  std::cout << "I am an empty creator. You may need me for something but it might trigger a bug. Who knows? Anyway, have a good day." << std::endl;

}

void muddpyle::create(bool tquiet)
{
  if(tquiet == false)
  {
    std::cout << "Welcome to MuddPyle, python binding for Muddpile. You can find documentation here:" << std::endl << "https://lsdtopotools.github.io/LSDTT_documentation/LSDTT_MuddPILE.html" << std::endl;
  }
  // Initialiseing Muddpile
  mymod = LSDRasterModel();
  // set level of quietness
  mymod.set_quiet(tquiet);
  verbose = tquiet;

  // Parameters specific to the python binding
  mymod.set_print_hillshade(false);
  // Done
}

// initialise the model with an already existing topography
void muddpyle::create(int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, xt::pytensor<float,2>& data, float m, 
  float n, int dt, int Umod, float save_step, float default_D, float default_Sc, xt::pytensor<float,2>& K_field, xt::pytensor<float,2>& U_field, bool hillslope_diffusion,
  bool use_adaptive_timestep, float max_dt, std::string OUT_DIR, std::string OUT_ID, std::vector<std::string> bc )
{
  TNT::Array2D<float> data_pointerast(nrows,ncols,ndv), tKdat(nrows,ncols,ndv), tUdat(nrows,ncols,ndv); // Trying here to create a raster of pointers

  for(size_t i=0;i<nrows;i++)
  for(size_t j=0;j<ncols;j++)
    {data_pointerast[i][j] = data(i,j);tKdat[i][j] = K_field(i,j);tUdat[i][j] = U_field(i,j);}

  // Data is copied! I am ready to feed a LSDRaster object
  // Here we go: 
  LSDRaster temprast(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);
  LSDRaster this_KRaster(nrows, ncols, xmin, ymin, cellsize, ndv, tKdat);
  LSDRaster this_URaster(nrows, ncols, xmin, ymin, cellsize, ndv, tUdat);

  LSDRasterModel mod(temprast);
  mod.add_path_to_names(OUT_DIR);
  mod.set_name(OUT_DIR+OUT_ID);

  // m and n are the exponent of spls
  mod.set_m(m);
  mod.set_n(n);
  // The mode of uplift. Options are:
  // default == block uplift
  // 1 == tilt block
  // 2 == gaussian
  // 3 == quadratic
  // 4 == periodic
  mod.set_uplift_mode(Umod);
  // time step in years and max time
  mod.set_timeStep(dt);
  // mod.set_maxtimeStep(max_dt);
  mod.set_endTime(max_dt);
  mod.set_endTime_mode(1);
  // save step in number of timestep (not in year!)
  mod.set_float_print_interval(save_step);
  // Default diffusion and critical slope param for hillslope diffusion
  mod.set_D( default_D);
  mod.set_S_c(default_Sc);
  // Default Erodibility param for river system
  mod.set_K(this_KRaster.get_data_element(0,0));

  // Remove edge artefacts
  mod.initialise_taper_edges_and_raise_raster(1);

  // ACtivate (or not the hillslope diffusion)
  mod.set_hillslope(hillslope_diffusion);
  // need to tell the model the first time to print (I guess???????)
  mod.set_next_printing_time(0);
  mod.set_current_frame(0);
  mod.set_print_hillshade(false);
  mod.set_boundary_conditions(bc);

  mod.raise_and_fill_raster();



  mod.run_components_combined(this_URaster,this_KRaster,use_adaptive_timestep);
  // Done basically???

}

// this function initialises the model with real topography
void muddpyle::initialise_model_with_dem(int nrows, int ncols, float xmin, float ymin, float cellsize, float ndv, xt::pytensor<float,2>& data)
{
  // transferring the array
  TNT::Array2D<float> data_pointerast(nrows,ncols,ndv);
  for(size_t i=0;i<nrows;i++)
  for(size_t j=0;j<ncols;j++)
    {data_pointerast[i][j] = data(i,j);}

  LSDRaster topo(nrows, ncols, xmin, ymin, cellsize, ndv, data_pointerast);
  mymod = LSDRasterModel(topo);
  base_nrows = nrows;
  base_ncols = ncols;
  base_xmin = xmin;
  base_ymin = ymin;
  base_resolution = cellsize;
  base_ndv = ndv;

  if(verbose)
    std::cout<< "I have feeded the model with real topography, I just need few internal routines to make it ready for muddpilisations." << std::endl;

  mymod.raise_and_fill_raster();
  mymod.set_print_hillshade(false);


}

// This function initialise (or at least attempt to initialise) the model with a diamond square method crafted by simon.
// It generates nice initial topography, detail in the doc
// Diamonds -> mineral -> main deposit type -> kimberlite. Funny innit?. I don't care I find it funny anyway and noone will read that yo.
// B.G. 2019
void muddpyle::initialise_model_kimberlite(int nrows, int ncols, float resolution, int diamond_square_feature_order, float diamond_square_relief, float parabola_relief, float roughness_relief, int prediff_step)
{
  
  if(this->verbose)
    std::cout << "Looking for kimberlites ..." << std::endl;
  // reshaping the model.
  mymod.resize_and_reset(nrows,ncols,resolution);
  base_nrows = nrows;
  base_ncols = ncols;
  base_xmin = 0;
  base_ymin = 0;
  base_resolution = resolution;
  base_ndv = -9999;
  // Spotting the smallest/largest (possible) dimension mate
  int smallest_dim = 0; if(nrows>=ncols){smallest_dim = ncols;}else{smallest_dim=nrows;} 
  float lps = floor( log2( float(smallest_dim) ) ); int largest_possible_scale = int(lps);
  // You don't want to create larger figures than possible
  if(diamond_square_feature_order>largest_possible_scale)
    diamond_square_feature_order = largest_possible_scale;
  // let's goooooooooooooooooooooooooooo
  mymod.intialise_diamond_square_fractal_surface(diamond_square_feature_order, diamond_square_relief);
  // Wanna add parabola relief on that?
  if(parabola_relief>0)
    mymod.superimpose_parabolic_surface(parabola_relief);
  // A bit of noise never hurts right?
  if(roughness_relief > 0)
  {
    mymod.set_noise(roughness_relief);
    mymod.random_surface_noise();
  }

  if(prediff_step>0)
  {
    if(verbose)
      cout << "I am going to diffuse the initial surface for you." << endl;

    float tpts = mymod.get_timeStep();
    mymod.set_timeStep( 0.5 );
    for (int i = 0; i< prediff_step; i++)
    {
      mymod.MuddPILE_nl_soil_diffusion_nouplift();
    }
    mymod.set_timeStep(tpts);
  }

  mymod.initialise_taper_edges_and_raise_raster(1);

  // NOT SURE IF I NEED THAT
  mymod.raise_and_fill_raster();
  // printing the outputs
  if(verbose)
    std::cout << "I am printing the initial surface, if nothing happens, you forgot to tell me the output loc and prefix!!" << std::endl; 
  mymod.print_rasters_and_csv(-1);
  if(verbose)
    std::cout << "Awrite mate I found the kimberlite." << std::endl; 

}

void muddpyle::run_model(bool fluvial, bool hillslope, bool spatially_variable_UK, int save_frame, bool use_adaptative_timestep)
{
  mymod.set_next_printing_time(0);
  mymod.set_current_frame(save_frame);
  mymod.set_print_hillshade(false);
  mymod.set_hillslope(hillslope);
  mymod.set_fluvial(fluvial);

  // mymod.raise_and_fill_raster();
  if(spatially_variable_UK)
    mymod.run_components_combined(UpliftRaster,KRaster,use_adaptative_timestep);
  else
    mymod.run_components_combined();
}

// end of the only-linux part 
#endif






















#endif