import warnings

from lys.Qt import QtCore
from lys.errors import NotImplementedWarning

from .CanvasBase import saveCanvas
from .AnnotationData import AnnotationData
from .Font import FontInfo


class TextAnnotation(AnnotationData):
    """
    Interface to access text annotations in canvas.

    *TextAnnotation* is usually generated by addText method in canvas.

    Args:
        canvas(Canvas): canvas to which the text annotation is added.
        text(str): The text of the annotation.
        pos(length 2 sequence): The psition of the annotation in the form of (x,y).
        axis('BottomLeft', 'BottomRight', 'TopLeft', or 'TopRight'): The axis to which the text annotation is added.

    Example::

        from lys import display
        g = display()
        line = g.addText('test')
    """

    edited = QtCore.pyqtSignal(str)
    """Pyqtsignal that is emitted when the text is edited."""

    def __init__(self, canvas, text, pos, axis):
        super().__init__(canvas, "text", axis)
        self._initialize(text, pos, axis)
        self._text = ""
        self.setText(text)
        self.setPosition(pos)

    @saveCanvas
    def setText(self, text):
        """
        Set the text of the annotation.

        Args:
            text(str): The text of the annotation.
        """
        if text != self._text:
            self._setText(text)
            self._text = text
            self.edited.emit(text)

    def getText(self):
        """
        Get the text of the annotation.

        Return:
            str: The text of the annotation.
        """
        return self._text

    @saveCanvas
    def setPosition(self, pos):
        """
        Set the position of the annotation.

        Args:
            pos(length 2 sequence): The position of the annotation.
        """
        self._setPosition(pos)
        self._pos = tuple(pos)

    def getPosition(self):
        """
        Get the position of the annotation.

        Return:
            length 2 sequence: The position of the annotation.
        """
        return self._pos

    @saveCanvas
    def setTransform(self, transformation):
        """
        Set the transformation of the annotation.

        When the *transformation* is 'axes', the annotation is fixed in the canvas.

        When the *transformation* is 'data', the position of the annotation is changed when the view range of the canvas is changed.

        Args:
            transformation('axes' or 'data'): The transformation.
        """
        self._setTransform(transformation)
        self._appearance['transform'] = transformation

    def getTransform(self):
        """
        Get the transformation of the annotation.

        Return:
            str: The transformation string.
        """
        return self._appearance['transform']

    @saveCanvas
    def setFont(self, fname, size=10, color="black"):
        """
        Set the font of the annotation.

        Args:
            family(str): The name of the font.
            size(int): The size of the font.
            color(str): The color of the font such as '#111111'.
        """
        if isinstance(fname, FontInfo):
            font = fname
        else:
            font = FontInfo(fname, size, color)
        self._setFont(font)
        self._appearance['font'] = font.toDict()

    def getFont(self):
        """
        Get the font of the label.

        Args:
            axis('Left' or 'Right' or 'Top' or 'Bottom'): The axis.

        Return:
            dict: The information of font. See :meth:`setFont`
        """
        return self._appearance['font']

    @saveCanvas
    def setBoxStyle(self, style):
        """
        Set the style of the bounding box.

        Args:
            style(str): 'none', 'square', 'circle', 'round', 'round4', 'larrow', 'rarrow', 'darrow', 'roundtooth', or 'sawtooth'
        """
        self._setBoxStyle(style)
        self._appearance['boxStyle'] = style

    def getBoxStyle(self):
        """
        Get the style of the bounding box.

        Return:
            str: The style of the bounding box. See :meth:`setBoxStyle`
        """
        return self._appearance['boxStyle']

    @saveCanvas
    def setBoxColor(self, faceColor, edgeColor):
        """
        Set the color of the bounding box.

        Args:
            faceColor: The face color in the form of color string such as #111111
            edgeColor: The edge color in the form of color string such as #111111
        """
        self._setBoxColor(faceColor, edgeColor)
        self._appearance['boxFaceColor'] = faceColor
        self._appearance['boxEdgeColor'] = edgeColor

    def getBoxColor(self):
        return self._appearance.get('boxFaceColor', 'white'), self._appearance.get('boxEdgeColor', 'black')

    def _loadAppearance(self, appearance):
        pos = self.getPosition()
        self.setTransform(appearance.get('transform', 'axes'))
        self.setPosition(pos)
        font = appearance.get('font', {"fname": FontInfo.defaultFont(), "size": 10, "color": "black"})
        self.setFont(FontInfo.fromDict(font))
        self.setBoxStyle(appearance.get('boxStyle', 'none'))
        self.setBoxColor(appearance.get('boxFaceColor', 'white'), appearance.get('boxEdgeColor', 'black'))

    def _initialize(self, text, axis):
        warnings.warn(str(type(self)) + " does not implement _initialize(text, axis) method.", NotImplementedWarning)

    def _setText(self, text):
        warnings.warn(str(type(self)) + " does not implement _setText(text) method.", NotImplementedWarning)

    def _setPosition(self, pos):
        warnings.warn(str(type(self)) + " does not implement _setPosition(pos) method.", NotImplementedWarning)

    def _setTransform(self, transformation):
        warnings.warn(str(type(self)) + " does not implement _setTransformation(transformation) method.", NotImplementedWarning)

    def _setFont(self, font):
        warnings.warn(str(type(self)) + " does not implement _setFont(font) method.", NotImplementedWarning)

    def _setBoxStyle(self, style):
        warnings.warn(str(type(self)) + " does not implement _setBoxStyle(style) method.", NotImplementedWarning)
