import warnings
from lys import filters
from lys.Qt import QtCore
from lys.errors import NotImplementedWarning

from .CanvasBase import CanvasPart, saveCanvas


class WaveData(CanvasPart):
    """
    Interface to access wave data in the canvas.

    This class is inherited by :class:`.Line.LineData`, :class:`.Image.ImageData`, :class:`.RGB.RGBData`,  :class:`.Vector.VectorData`, and :class:`.Contour.ContourData`.

    Instance of WaveData is automatically generated by display or append methods.

    """
    modified = QtCore.pyqtSignal()
    """This pyqtSignal is emimtted when the data is changed."""

    def __init__(self, canvas, wave, axis):
        super().__init__(canvas)
        self._wave = wave
        self._wave.modified.connect(self._update)
        self._axis = axis
        self._appearance = {}
        self._offset = (0, 0, 0, 0)
        self._filter = None
        self._filteredWave = wave
        self._z = 0

    @saveCanvas
    def _update(self, *args, **kwargs):
        if self._offset == (0, 0, 0, 0) and self._filter is None:
            self._filteredWave = self._wave
        else:
            self._filteredWave = self._calcFilteredWave(self._wave, self._offset, self._filter)
        self._updateData()
        self.modified.emit()

    def _calcFilteredWave(self, w, offset, filter):
        if filter is None:
            filt = filters.Filters([filters.OffsetFilter(offset)])
        else:
            filt = filter + filters.OffsetFilter(offset)
        w = filt.execute(w)
        if w.data.ndim == 1 and w.data.dtype == complex:
            w = filters.ComplexFilter("absolute").execute(w)
        return w

    def getWave(self):
        """
        Get the wave.

        Return:
            Wave: The wave.
        """
        return self._wave

    def getFilteredWave(self):
        """
        Get the wave to which offset and filter have been applied.

        Return:
            Wave: The filtered wave.
        """
        return self._filteredWave

    def getName(self):
        """
        Get the name of the wave data.

        Return:
            str: The name.
        """
        return self._appearance.get('DataName', self.getWave().name)

    def setName(self, name):
        """
        Set the name of the data.

        Args:
            name(str): The name of the data.
        """
        self._appearance['DataName'] = name

    def getAxis(self):
        """
        Get the axis to which the data was added.

        Return:
            str: The axis ('BottomLeft', 'TopLeft', 'BottomRight', or 'TopRight').
        """
        return self._axis

    @saveCanvas
    def setVisible(self, visible):
        """
        Set the visibility of the data.

        Args:
            visible(bool): The visibility of the data.
        """
        self._setVisible(visible)
        self._appearance['Visible'] = visible

    def getVisible(self):
        """
        Get the visibility of the data.

        Return:
            bool: The visibility of the data.
        """
        return self._appearance.get('Visible', True)

    @saveCanvas
    def setOffset(self, offset):
        """
        Set the offset to the data.

        The data is offset as x'=x*x1+x0 and y'=y*y1+y0.

        Args:
            offset(tuple of length 4 float): The offset in the form of (x0, y0, x1, y1).
        """
        if self._offset != offset:
            self._offset = offset
            self._update()

    def getOffset(self):
        """
        Get the offset to the data.

        See :meth:`setOffset` for detail.

        Return:
            tuple of length 4 float: The offset in the form of (x0, y0, x1, y1).
        """
        return tuple(self._offset)

    @saveCanvas
    def setFilter(self, filter=None):
        """
        Apply filter to the data.

        Args:
            filter(filterr): The filter. See :class:`lys.filters.filter.FilterInterface.FilterInterface`
        """
        if self._filter != filter:
            self._filter = filter
            self._update()

    def getFilter(self):
        return self._filter

    @saveCanvas
    def setZOrder(self, z):
        """
        Set the z order of the data.

        Args:
            z(int): The z order.
        """
        self._setZ(z)
        self._z = z

    def getZOrder(self):
        """
        Get the z order of the data.

        Return:
            int: The z order.
        """
        return self._z

    def saveAppearance(self):
        """
        Save appearance from dictionary.

        Users can save/load appearance of data by save/loadAppearance methods.

        Return:
            dict: dictionary that include all appearance information.
        """
        return dict(self._appearance)

    @saveCanvas
    def loadAppearance(self, appearance):
        """
        Load appearance from dictionary.

        Users can save/load appearance of data by save/loadAppearance methods.

        Args:
            appearance(dict): dictionary that include all appearance information, which is usually generated by :meth:`saveAppearance` method.
        """
        self.setVisible(appearance.get('Visible', True))
        self.setName(appearance.get('DataName', self.getWave().name))
        self._loadAppearance(appearance)

    def _setVisible(self, visible):
        warnings.warn(str(type(self)) + " does not implement _setVisible(visible) method.", NotImplementedWarning)

    def _setZ(self, z):
        warnings.warn(str(type(self)) + " does not implement _setZ(z) method.", NotImplementedWarning)

    def _updateData(self):
        raise NotImplementedError(str(type(self)) + " does not implement _updateData() method.")

    def _loadAppearance(self, appearance):
        raise NotImplementedError(str(type(self)) + " does not implement _loadAppearance(appearance) method.")
