import warnings

from lys.Qt import QtCore
from lys.errors import NotImplementedWarning

from .CanvasBase import saveCanvas
from .AnnotationData import AnnotationWithLine


class LineAnnotation(AnnotationWithLine):
    """
    Interface to access line annotations in canvas.

    *LineAnnotation* is usually generated by addLineAnnotation method in canvas.

    Several methods related to the appearance of line is inherited from :class:`.AnnotationData.AnnotationWithLine`

    Args:
        canvas(Canvas): canvas to which the line annotation is added.
        pos(2 * 2 sequence): The position of the line annotation.
        axis('BottomLeft', 'BottomRight', 'TopLeft', or 'TopRight'): The axis to which the line annotation is added.

    Example::

        from lys import display
        g = display()
        line = g.addLineAnnotation()
        line.setLineColor("#ff0000")
    """
    positionChanged = QtCore.pyqtSignal(tuple)
    """PyqtSignal that is emitted when the position of the line is changed."""

    def __init__(self, canvas, pos, axis):
        super().__init__(canvas, "line", axis)
        self._initialize(pos, axis)
        self._pos = pos

    @saveCanvas
    def setPosition(self, pos):
        """
        Set the position of the line.

        Args:
            pos(2*2 sequence): The position of the line in the form of [(x0,y0),(x1,y1)].
        """
        pos = tuple((tuple(pos[0]), tuple(pos[1])))
        if pos != self._pos:
            self._pos = pos
            self._setPosition(pos)
            self.positionChanged.emit(tuple(self._pos))

    def getPosition(self):
        """
        Get the position of the line.

        Return:
            2*2 sequence: The position of the line in the form of [(x0,y0),(x1,y1)].
        """
        return self._pos

    def _initialize(self, pos, axis):
        warnings.warn(str(type(self)) + " does not implement _initialize(text, axis) method.", NotImplementedWarning)

    def _setPosition(self, pos):
        warnings.warn(str(type(self)) + " does not implement _setPosition(pos) method.", NotImplementedWarning)


class InfiniteLineAnnotation(AnnotationWithLine):
    """
    Interface to access infinite line annotations in canvas.

    *InfiniteLineAnnotation* is usually generated by addInfiniteLineAnnotation method in canvas.

    Several methods related to the appearance of line is inherited from :class:`.AnnotationData.AnnotationWithLine`

    Args:
        canvas(Canvas): canvas to which the line annotation is added.
        pos(float): The position of the line annotation.
        type('vertical' or 'horizontal'): The direction of the infinite line.
        axis('BottomLeft', 'BottomRight', 'TopLeft', or 'TopRight'): The axis to which the line annotation is added.

    Example::

        from lys import display
        g = display()
        line = g.canvas.addInfiniteLineAnnotation()
        line.setLineColor("#ff0000")
    """
    positionChanged = QtCore.pyqtSignal(float)
    """PyqtSignal that is emitted when the position of the line is changed."""

    def __init__(self, canvas, pos, orientation, axis):
        super().__init__(canvas, "inf line", axis)
        self._initialize(pos, orientation, axis)
        self._pos = pos
        self._orientation = orientation

    @saveCanvas
    def setPosition(self, pos):
        """
        Set the position of the line.

        Args:
            pos(float): The position of the line.
        """
        if pos != self._pos:
            self._pos = pos
            self._setPosition(pos)
            self.positionChanged.emit(self._pos)

    def getPosition(self):
        """
        Get the position of the line.

        Args:
            float: The position of the line.
        """
        return self._pos

    def getOrientation(self):
        return self._orientation

    def _initialize(self, pos, orientation, axis):
        warnings.warn(str(type(self)) + " does not implement _initialize(pos, orientation, axis) method.", NotImplementedWarning)

    def _setPosition(self, pos):
        warnings.warn(str(type(self)) + " does not implement _setPosition(pos) method.", NotImplementedWarning)
