import logging
import warnings
import weakref

from lys import DaskWave, Wave, filters
from lys.Qt import QtCore
from lys.decorators import avoidCircularReference


class MultiCutCUI(QtCore.QObject):
    """
    MultiCutCUI is a class that manage contents of :class:`lys.multicut.MultiCutGUI.MultiCut`.

    The instance of this class is automatically generated by :class:`lys.multicut.MultiCutGUI.MultiCut` and therefore users should not instantiate this class.

    The functionarities of this class is composed of :class:`MultiCutWave`, :class:`ChildWaves`, :class:`AxesRangeManger`, and :class:`FreeLineManager`.

    The methods of above classes can be accessed directly from this class.

    Args:
        wave(Wave or DaskWave): The wave to be analyzed by MultiCut.

    """

    def __init__(self, wave):
        super().__init__()
        self._wave = MultiCutWave(wave)
        self._axesRange = AxesRangeManager(self.getRawWave())
        self._freeLine = FreeLineManager()
        self._children = ChildWaves(self)
        self._wave.dimensionChanged.connect(self.__reset)

    def __reset(self, wave):
        self._children.clear()
        self._axesRange.reset(wave)
        self._freeLine.clear()

    def __getattr__(self, key):
        if "_axesRange" in self.__dict__:
            if hasattr(self._axesRange, key):
                return getattr(self._axesRange, key)
        if "_wave" in self.__dict__:
            if hasattr(self._wave, key):
                return getattr(self._wave, key)
        if "_freeLine" in self.__dict__:
            if hasattr(self._freeLine, key):
                return getattr(self._freeLine, key)
        if "_children" in self.__dict__:
            if hasattr(self._children, key):
                return getattr(self._children, key)
        return super().__getattr__(key)

    def saveAsDictionary(self, **kwargs):
        """
        Save present state as dictionary.
        """
        return {
            "range": self._axesRange.saveAsDictionary(**kwargs),
            "children": self._children.saveAsDictionary(**kwargs),
            "fline": self._freeLine.saveAsDictionary(**kwargs)
        }

    def loadFromDictionary(self, d, **kwargs):
        """
        Load the state stored in the dictionary.
        """
        self._axesRange.loadFromDictionary(d.get("range", {}), **kwargs)
        self._freeLine.loadFromDictionary(d.get("fline", {}), **kwargs)
        self._children.loadFromDictionary(d.get("children", {}), **kwargs)


class MultiCutWave(QtCore.QObject):
    """
    This class manage wave and filter for analysis.

    All methods in this class can be accessed from :class:`MultiCutCUI`.

    Args:
        wave(Wave or DaskWave): The wave to be analyzed.

    """
    rawDataChanged = QtCore.pyqtSignal(object)
    """
    Emitted when the raw data is changed.
    """
    dimensionChanged = QtCore.pyqtSignal(object)
    """
    Emitted when the dimension of the filtered wave is changed.
    """
    filterApplied = QtCore.pyqtSignal(object)
    """
    Emitted when filter is applied.
    """

    def __init__(self, wave):
        super().__init__()
        self._wave = self._filtered = self._load(wave)
        self._wave.persist()
        self._filter = None
        self._useDask = True

    def _load(self, data):
        if isinstance(data, Wave) or isinstance(data, DaskWave):
            return DaskWave(data)
        else:
            return DaskWave(Wave(data))

    def applyFilter(self, filt):
        """
        Apply filter to the wave.

        Args:
            filt(Filter): The filter to be applied.
        """
        if filt is None:
            filt = filters.EmptyFilter()
        self._filter = filt
        dim_old = self._filtered.ndim
        wave = filt.execute(self._wave)
        wave.persist()
        if self._useDask:
            self._filtered = wave
            logging.info("[MultiCut] DaskWave set. shape = " + str(wave.data.shape) + ", dtype = " + str(wave.data.dtype) + ", chunksize = " + str(wave.data.chunksize))
        else:
            self._filtered = wave.compute()
            logging.info("[MultiCut] Wave set. shape = " + str(wave.data.shape) + ", dtype = " + str(wave.data.dtype))
        if dim_old != self._filtered.ndim:
            self.dimensionChanged.emit(self._filtered)
        else:
            self.filterApplied.emit(self._filtered)

    def setRawWave(self, wave):
        """
        Set the raw wave. The existing raw wave will be replaced if it exists. Use :meth:`updateRawWave` method for fast update of the data.

        Args:
            wave(Wave or DaskWave): The raw wave.
        """
        shape_old = self._wave.shape
        if not isinstance(wave, Wave) or isinstance(wave, DaskWave):
            wave = Wave(wave)
        if self._wave.data.shape == wave.shape:
            self._wave.data[:] = wave.data
            self._wave.axes = wave.axes
            self._wave.note = wave.note
        else:
            self._wave = self._load(wave)
        self._wave.persist()
        self.rawDataChanged.emit(self._wave)
        if self._wave.shape != shape_old:
            self._filter = None
            self._filtered = self._wave
            self.dimensionChanged.emit(self._filtered)
        else:
            self.applyFilter(self._filter)

    def getRawWave(self):
        """
        Get raw wave instance.

        Returns:
            DaskWave: The instance of the raw wave.
        """
        return self._wave

    def updateRawWave(self, data=None, axes=None, update=True):
        """
        Update raw wavefor fast update of the data.

        Args:
            data(dict): The dictionary that contains change of the data. The key of the dictionary should be index of the array. RawWave[key] will be replaced by data[key]
            axes(list): The new axes. If None, the axes will not be changed.
            update(bool): If True, the MultiCut result will be replaced by new data. Set False only when you want to continuously update the data before updating GUIs.
        """
        if data is not None:
            for idx, frame in data.items():
                self._wave.data[idx] = frame
            self._wave.data.persist()
        if axes is not None:
            self._wave.axes = axes
        if update:
            self.applyFilter(self._filter)

    def getFilteredWave(self):
        """
        Get filtered wave instance.

        Returns:
            DaskWave or Wave: The filtered wave.
        """
        return self._filtered

    def useDask(self, b):
        """
        Specify whether the DaskWave is used for processing.

        Args:
            b(bool): If True, DaskWave is used for processing.
        """
        self._useDask = b


class ChildWaves(QtCore.QObject):
    """
    This class manages child waves in MultiCut.

    """
    childWavesChanged = QtCore.pyqtSignal()
    """
    Emitted when the child waves are added or removed.
    """

    def __init__(self, cui):
        super().__init__()
        self._cui = weakref.ref(cui)
        self.cui.axesRangeChanged.connect(self.__update)
        self.cui.freeLineMoved.connect(self.__update)
        self.cui.filterApplied.connect(lambda x: self.__update())
        self._sumType = "Mean"
        self._waves = []

    def clear(self):
        self._waves = []
        self.childWavesChanged.emit()

    @property
    def cui(self):
        """
        Return the cui of MultiCut.
        """
        return self._cui()

    def setSumType(self, sumType):
        """
        Set the sum type.

        Args:
            sumType("Sum", "Mean", "Median", "Max", or "Min"): The sum type.
        """
        self._sumType = sumType

    def addWave(self, axes, filter=None, name=None):
        """
        Add new wave to MultiCut.
        The child waves that is generated by this method is automatically updated when axes range of free lines are changed. 

        Args:
            axes(list of int): The axes of the child wave.
            filter(filter): The filter for postprocess.
            name(str): The name of the child wave.

        Returns:
            _ChildWave: The object that include information of the child wave.
        """
        w = self._makeWave(axes)
        if name is not None:
            w.name = name
        item = _ChildWave(w, axes, filter)
        self._waves.append(item)
        self.childWavesChanged.emit()
        return item

    def remove(self, obj):
        """
        Remove a child wave.

        Args:
            obj(_ChildWave): The child wave to be removed.
        """
        self._waves.remove(obj)
        self.childWavesChanged.emit()

    def _makeWave(self, axes):
        wave = self.cui.getFilteredWave()
        ignored = sorted([ax for ax in axes if not isinstance(ax, str)] + self._freeLineAxes(axes))
        ranges = [self.cui.getAxisRange(i) for i in range(wave.ndim)]
        for ax in ignored:
            ranges[ax] = None
        f = [filters.IntegralFilter(ranges, self._sumType)] + self.__getFreeLineTransposeFilter(axes, ignored)
        return filters.Filters(f).execute(wave)

    def _freeLineAxes(self, axes):
        res = []
        for ax in axes:
            if isinstance(ax, str):
                res.extend(self.cui.getFreeLine(ax).getAxes())
        return res

    def __getFreeLineTransposeFilter(self, axes_orig, ignored):
        """
        Get freeline and transpose filter for multicut.
        Since several axes are integrated, axes for free line should be modified.
        This method calculate appropriate free line filter and additional transpose filter if needed.

        Args:
            axes_orig(list): The axes of data in the original wave.
            ignored: The axes of data in the original wave that is not integrated.
        """
        axes_final = list(ignored)
        filts = []
        for ax in axes_orig:
            if isinstance(ax, str):
                line = self.cui.getFreeLine(ax)
                line_axes_orig = list(line.getAxes())                      # Axes for non-integrated data
                line_axes = [axes_final.index(a) for a in line_axes_orig]  # Axes for integrated data used to create free line filter
                filts.append(line.getFilter(line_axes))
                axes_final[axes_final.index(line_axes_orig[0])] = ax       # Replace original axes to the name of line.
                axes_final.remove(line_axes_orig[1])
        if axes_final != axes_orig and len(axes_orig) != 1:
            filts.append(filters.TransposeFilter([axes_final.index(a) for a in axes_orig]))
        return filts

    def getChildWaves(self):
        """
        Return the list of child waves:

        Returns:
            list of _ChildWave: The list of child waves.
        """
        return self._waves

    def __update(self, axes=None):
        for child in self._waves:
            # update all
            if axes is None:
                self.__updateSingleWave(child)
            # line is moved
            elif isinstance(axes, _FreeLine):
                if axes.getName() in child.getAxes():
                    self.__updateSingleWave(child)
            # range changed
            else:
                ax = []
                for a in child.getAxes():
                    if isinstance(a, str):
                        ax.extend(self._freeLineAxes([a]))
                    else:
                        ax.append(a)
                if not set(axes).issubset(ax):
                    self.__updateSingleWave(child)

    def __updateSingleWave(self, child):
        if not child.isEnabled():
            return
        try:
            with warnings.catch_warnings():
                warnings.simplefilter("ignore")
                wav = self._makeWave(child.getAxes())
                child.update(wav)
        except Exception:
            import traceback
            traceback.print_exc()

    def saveAsDictionary(self, **kwargs):
        """
        Save the present state as dictionary.
        """
        items = []
        for w in self._waves:
            d = {"axes": w.getAxes(), "name": w.name()}
            f = w.postProcess()
            if f is not None:
                d["filter"] = filters.toString(w.postProcess())
            items.append(d)
        return {"Items": items}

    def loadFromDictionary(self, d, axesMap=None, **kwargs):
        """
        Load the state stored in the dictionary.
        """
        self.clear()
        for item in d.get("Items", []):
            if "filter" in item:
                item["filter"] = filters.fromString(item["filter"])
            if axesMap is not None:
                item["axes"] = [axesMap[ax] if ax in axesMap else ax for ax in item["axes"]]
            self.addWave(**item)


class _ChildWave(QtCore.QObject):
    def __init__(self, wave, axes, filter=None):
        super().__init__()
        self._orig = wave
        self._post = filter
        self._axes = axes
        self._filt = self.__apply(wave)
        self._enabled = True

    def getAxes(self):
        return tuple(self._axes)

    def getRawWave(self):
        return self._orig

    def getFilteredWave(self):
        return self._filt

    def setEnabled(self, b):
        self._enabled = b

    def isEnabled(self):
        return self._enabled

    def setPostProcess(self, post):
        self._post = post
        self.update(self._orig)

    def postProcess(self):
        return self._post

    def update(self, wave):
        self._orig = wave
        name = str(self._filt.name)
        wave = self.__apply(wave)
        self._filt.data = wave.data
        self._filt.axes = wave.axes
        self._filt.note = wave.note
        self._filt.name = name

    def __apply(self, wave):
        post = self.postProcess()
        if post is not None:
            wave = post.execute(wave)
        if isinstance(wave, DaskWave):
            wave = wave.compute()
        return wave

    def name(self):
        return self._filt.name


class AxesRangeManager(QtCore.QObject):
    """
    AxesRangeManager manage axes range that determines the result of MultiCut.

    This class is automatically instanciated by MultiCutCUI.

    Do not directly instanciate this class.
    """
    axesRangeChanged = QtCore.pyqtSignal(tuple)
    """
    Emitted after :meth:`setAxisRange` is called.
    """

    def __init__(self, wave):
        super().__init__()
        self.reset(wave)

    def reset(self, wave):
        """
        Reset all axes ranges to default.
        """
        self._ranges = [ax[0] for ax in wave.axes]
        self.axesRangeChanged.emit(tuple(range(len(wave.axes))))

    @avoidCircularReference
    def setAxisRange(self, axis, range):
        """
        Set the integrated range for MultiCut.

        Args:
            axis(int): The axis of which integrated range is set.
            range(float or length 2 sequence): The integrated range. If *range* is a float, only a point is used for integration. 
        """
        if hasattr(axis, "__iter__"):
            for ax, r in zip(axis, range):
                self._ranges[ax] = r
            self.axesRangeChanged.emit(tuple(axis))
        else:
            self._ranges[axis] = range
            self.axesRangeChanged.emit((axis,))

    def getAxisRange(self, axis):
        """
        Get the integrated range for MultiCut.

        Args:
            axis(int): The axis.

        Returns:    
            float or length 2 sequence: See :meth:`setAxisRange`.
        """
        return self._ranges[axis]

    def getAxisRangeType(self, axis):
        """
        Get the axis range type for the specified axis.

        Returns:
            'point' or 'range': The axis range type.
        """
        r = self.getAxisRange(axis)
        if hasattr(r, "__iter__"):
            return 'range'
        else:
            return 'point'

    def saveAsDictionary(self, useRange=False, **kwargs):
        """
        Save the present state as dictionary.

        Args:
            useRange(bool): If True, save the ranges.
        """
        if useRange:
            return {"range": self._ranges}
        else:
            return {}

    def loadFromDictionary(self, d, useRange=False, axesMap=None, **kwargs):
        """
        Load the state from the dictionary.

        Args:
            d(dict): The dictionary.
            useRange(bool): If True, the range is loaded.
        """
        if useRange:
            if "range" in d:
                self._ranges = d["range"]
                if axesMap is not None:
                    self._ranges = [self._ranges[axesMap[i]] for i in range(len(self._ranges))]
                self.axesRangeChanged.emit(tuple(range(len(self._ranges))))


class FreeLineManager(QtCore.QObject):
    """
    FreeLineManager manages free lines in MultiCut.

    Methods in this class can be accessed from MultiCutCUI.

    Do not instantiate this class except in MultiCutCUI.
    """
    freeLineChanged = QtCore.pyqtSignal()
    """Emitted when free lines are added or removed."""
    freeLineMoved = QtCore.pyqtSignal(object)
    """Emitted when one of the free line is moved."""

    def __init__(self):
        super().__init__()
        self.clear()

    def clear(self):
        """
        Clear all free lines.
        """
        self._fregs = []
        self.freeLineChanged.emit()

    def addFreeLine(self, axes, position=[[0, 0], [1, 1]], width=1, name=None):
        """
        Add free line to MultiCut.

        Args:
            axes(length 2 sequence): The axes for which the free line is added.
            position(2*2 array): The position of the free line in the form of [(x1,y1), (x2,y2)]
            width(float): The width of the free line.
            name(str): The name of the free line. If omitted, the line name is automatically generated.

        Returns:
            _FreeLine: The free line object.
        """
        if name is None:
            name = self.__getName()
        obj = _FreeLine(name, axes, position, width)
        obj.lineChanged.connect(lambda: self.freeLineMoved.emit(obj))
        self._fregs.append(obj)
        self.freeLineChanged.emit()
        return obj

    def __getName(self):
        i = 0
        names = [f.name() for f in self._fregs]
        while "Line" + str(i) in names:
            i += 1
        return "Line" + str(i)

    def removeFreeLine(self, obj):
        """
        Remove free line.

        Args:
            obj(_FreeLine): The free line to be removed.
        """
        self._fregs.remove(obj)
        self.freeLineChanged.emit()

    def getFreeLines(self):
        """
        Return list of free lines.

        Returns:
            list: The list of free lines.
        """
        return self._fregs

    def getFreeLine(self, name):
        """
        Get free line from name.

        Args:
            name(str): The name of free line.

        Returns:
            _FreeLine: The free line object.
        """
        for line in self._fregs:
            if line.getName() == name:
                return line

    def saveAsDictionary(self, useLine=False, **kwargs):
        """
        Save information of free lines as dictionary.

        Args:
            useLine: If it is False, the position and width is not saved.

        Returns:
            dict: The dictonary that contains the information of free line.
        """
        res = []
        for f in self._fregs:
            data = {"axes": f.getAxes(), "name": f.getName()}
            if useLine:
                data["position"] = f.getPosition()
                data["width"] = f.getWidth()
            res.append(data)
        return {"freeLines": res}

    def loadFromDictionary(self, d, useLine=False, axesMap=None, **kwargs):
        """
        Load information of free lines from dictionary.

        Args:
            d(dict): The dictionary that is generated by saveAsDictionary method.
            useLine(bool): It determines to load position and width of the free lines.
        """
        self.clear()
        for f in d.get("freeLines", []):
            if not useLine:
                if "position" in f:
                    del f["position"]
                if "width" in f:
                    del f["width"]
            if axesMap is not None:
                f["axes"] = [axesMap[ax] for ax in f["axes"]]
            self.addFreeLine(**f)


class _FreeLine(QtCore.QObject):
    lineChanged = QtCore.pyqtSignal()
    """
    Emitted when the line position or width is changed.
    """

    def __init__(self, name, axes, position=[[0, 0], [1, 1]], width=1):
        super().__init__()
        self._name = name
        self._axes = axes
        self._pos = position
        self._width = width

    def getName(self):
        """
        Return the name of the free line.
        """
        return self._name

    def getAxes(self):
        """
        Return the axes of the free line.
        """
        return self._axes

    def setPosition(self, pos):
        """
        Set the position of the free line.
        pos(2*2 array): The position of the free line in the form of [(x1,y1), (x2,y2)]
        """
        self._pos = pos
        self.lineChanged.emit()

    def getPosition(self):
        """
        Get the position of the free line.
        """
        return self._pos

    def setWidth(self, width):
        """
        Set the width of the free line.

        Args:
            width(float): The width of the free line.
        """
        self._width = width
        self.lineChanged.emit()

    def getWidth(self):
        """
        Get the width of the free line.
        """
        return self._width

    def getFilter(self, axes):
        """
        Return FreeLineFilter.

        Args:
            axes(length 2 sequence): The axes that the FreeLineFilter is applied.

        Returns:
            filters.FreeLineFilter: The filter.
        """
        return filters.FreeLineFilter(axes, self._pos, self._width)
