import pandas
import geopandas
import numpy

from map2loop.mapdata import MapData
from map2loop.thickness_calculator import StructuralPoint
from map2loop._datasets.geodata_files.load_map2loop_data import load_hamersley_geology
from map2loop.m2l_enums import Datatype

####################################################################
### Define the test data for ThicknessCalculator StructuralPoint ###
####################################################################

# Sample stratigraphic units data
st_units = pandas.DataFrame(
    {
        'Unnamed: 0': [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
        'layerId': [7, 0, 10, 8, 1, 5, 9, 4, 3, 2, 6],
        'name': [
            'Turee_Creek_Group',
            'Boolgeeda_Iron_Formation',
            'Woongarra_Rhyolite',
            'Weeli_Wolli_Formation',
            'Brockman_Iron_Formation',
            'Mount_McRae_Shale_and_Mount_Sylvia_Formation',
            'Wittenoom_Formation',
            'Marra_Mamba_Iron_Formation',
            'Jeerinah_Formation',
            'Bunjinah_Formation',
            'Pyradie_Formation',
        ],
        'minAge': [0.0] * 11,
        'maxAge': [100000.0] * 11,
        'group': [None] * 11,
        'supergroup': [None] * 11,
        'ThicknessMean_ThicknessCalculatorAlpha': [0.0] * 11,
        'ThicknessMedian_ThicknessCalculatorAlpha': [0.0] * 11,
        'ThicknessStdDev_ThicknessCalculatorAlpha': [0.0] * 11,
        'stratigraphic_Order': list(range(11)),
        'colour': [
            '#5d7e60',
            '#387866',
            '#628304',
            '#a2f290',
            '#0c2562',
            '#5fb3c5',
            '#f48b70',
            '#1932e2',
            '#106e8a',
            '#d0d47c',
            '#e7f2f3',
        ],
    }
)

st_column = [
    'Turee_Creek_Group',
    'Boolgeeda_Iron_Formation',
    'Woongarra_Rhyolite',
    'Weeli_Wolli_Formation',
    'Brockman_Iron_Formation',
    'Mount_McRae_Shale_and_Mount_Sylvia_Formation',
    'Wittenoom_Formation',
    'Marra_Mamba_Iron_Formation',
    'Jeerinah_Formation',
    'Bunjinah_Formation',
    'Pyradie_Formation',
    'Fortescue_Group',
]

# 3. map_data.basal_contacts
basal_c = [
    {'ID': 0, 'basal_unit': 'Turee_Creek_Group', 'type': 'BASAL'},
    {'ID': 1, 'basal_unit': 'Boolgeeda_Iron_Formation', 'type': 'BASAL'},
    {'ID': 1, 'basal_unit': 'Brockman_Iron_Formation', 'type': 'BASAL'},
    {'ID': 1, 'basal_unit': 'Weeli_Wolli_Formation', 'type': 'BASAL'},
    {'ID': 4, 'basal_unit': 'Jeerinah_Formation', 'type': 'BASAL'},
    {'ID': 5, 'basal_unit': 'Bunjinah_Formation', 'type': 'BASAL'},
    {'ID': 6, 'basal_unit': 'Marra_Mamba_Iron_Formation', 'type': 'BASAL'},
    {'ID': 7, 'basal_unit': 'Mount_McRae_Shale_and_Mount_Sylvia_Formation', 'type': 'ABNORMAL'},
    {'ID': 8, 'basal_unit': 'Mount_McRae_Shale_and_Mount_Sylvia_Formation', 'type': 'ABNORMAL'},
    {'ID': 9, 'basal_unit': 'Wittenoom_Formation', 'type': 'BASAL'},
    {'ID': 10, 'basal_unit': 'Mount_McRae_Shale_and_Mount_Sylvia_Formation', 'type': 'BASAL'},
    {'ID': 11, 'basal_unit': 'Woongarra_Rhyolite', 'type': 'BASAL'},
]
bc_geoms = geopandas.GeoSeries.from_wkt(
    [
        'MULTILINESTRING ((520000.0000000000000000 7506463.0589317083358765, 520000.0000000000000000 7506464.0583261400461197, 521798.5784270099829882 7506667.5086746597662568, 524292.0583883899962530 7507105.0368000296875834, 525888.7772752699675038 7507345.5593877695500851, 526916.8410634599858895 7507564.5817003501579165, 528098.0088455299846828 7507783.0997126800939441, 529344.8316910399589688 7507958.1297348998486996, 530263.5311044099507853 7508220.6498684398829937, 531357.2706492099678144 7508395.6678343201056123, 532013.4010351699544117 7508548.6697535198181868, 532538.3520777500234544 7508548.6775182196870446, 532932.1086069300072268 7508483.1717491699382663, 533085.1519033299991861 7508308.1688938299193978, 533107.0194513100432232 7508045.6694244500249624, 533041.4099823100259528 7507717.6704597100615501, 532866.4200693099992350 7507455.1614942299202085, 531685.2195994600187987 7506755.1383216800168157, 530875.8998375800438225 7506317.6329081403091550, 530175.9800506900064647 7506011.6195486104115844, 529279.1401694599771872 7505442.5995908901095390, 528426.1115621499484405 7505027.0810785600915551, 526654.3696992900222540 7504655.0613197097554803, 525210.7310010100482032 7504567.5296983895823359, 523067.1991962700267322 7504217.5111659299582243, 521098.6489177999901585 7503867.9904129095375538, 520414.0195790100260638 7503782.4897812502458692, 520000.0000000000000000 7503710.2290291301906109, 520000.0000000000000000 7503711.2441460378468037))',
        'MULTILINESTRING ((520000.0000000000000000 7503245.2598590906709433, 520000.0000000000000000 7503246.2595371659845114, 520271.9002807399956509 7503274.9797609802335501, 520982.5192159600555897 7503437.4804947897791862, 521754.8289336899761111 7503561.4991028197109699, 522673.5081250499933958 7503780.5075413398444653, 523439.0304306600592099 7503868.0182901099324226, 525079.5116741600213572 7504086.5288977697491646, 526041.9114880500128493 7504086.5408970797434449, 527157.4277337800012901 7504239.5578404404222965, 528710.4300827099941671 7504589.5780827598646283, 529322.8810412200400606 7504699.1013501295819879, 529519.7383159700548276 7504721.1013953704386950, 529782.1897067499812692 7504852.1116183400154114, 529846.5792647199705243 7505018.1105302302166820, 530434.1504480800358579 7505322.1195039302110672, 531001.4219496899750084 7505585.1193884601816535, 531730.7807765699690208 7505767.6295145899057388, 531892.8382770799798891 7505828.6311367200687528, 532231.0984191700117663 7506036.1303443796932697, 532535.6622749799862504 7506280.1284634796902537, 533205.6884558700257912 7506665.6397261302918196, 533571.2115816300502047 7506929.6612275000661612, 534139.7121901500504464 7507214.1599170295521617, 534444.3105956399813294 7507356.1602175496518612, 534769.1609872799599543 7507478.1677699098363519, 534911.2916754200123250 7507518.6696750596165657, 535114.3212854999583215 7507640.6607389999553561, 535236.1208061299985275 7507823.1785498997196555, 535358.0217514700489119 7508351.1807611100375652, 535520.5093329700175673 7508594.6879638703539968, 535743.8194368999684229 7508818.1783391702920198, 536007.7499623399926350 7508980.6794774401932955, 536576.2579947899794206 7509386.6783681297674775, 536718.4177284899633378 7509630.1906582303345203, 536637.2394275299739093 7509975.6913913199678063, 536604.7901494553079829 7510000.0000000000000000, 536606.4580603817012161 7510000.0000000000000000), (533429.9102329264860600 7510000.0000000000000000, 533432.8839227686403319 7510000.0000000000000000, 533307.4205499000381678 7509955.1997412098571658, 532718.6210473099490628 7509792.6996827302500606, 531297.2519883899949491 7509264.6781625803560019, 530525.7001818099524826 7509021.1595332501456141, 529551.0197801099857315 7508858.6513834102079272, 528251.4977760600158945 7508655.6207956997677684, 527134.7110856899525970 7508351.0991824995726347, 525794.5900796599453315 7508066.5579961901530623, 524596.6507077199639753 7507782.5406945496797562, 523662.6590976800071076 7507681.0381808001548052, 522830.1799164600088261 7507498.0297148795798421, 522038.3083402099437080 7507295.0188862504437566, 520677.9482569899992086 7507010.9990820102393627, 520000.0000000000000000 7506956.5663501676172018, 520000.0000000000000000 7506957.5695682624354959), (550000.0000000000000000 7490095.5414067916572094, 550000.0000000000000000 7490094.5419130371883512, 549848.4078108200337738 7490011.2184614501893520, 549837.1990841374499723 7490000.0000000000000000, 549836.1991053668316454 7490000.0000000000000000))',
        'MULTILINESTRING ((520000.0000000000000000 7500803.4896762184798717, 520000.0000000000000000 7500804.4889609171077609, 520473.5791580000077374 7500907.4720744201913476, 521144.7272773912409320 7501014.1576358489692211, 521145.5101736339274794 7501013.5373818902298808), (523032.4352534925565124 7510000.0000000000000000, 523034.2386217917664908 7510000.0000000000000000, 522935.2510594900231808 7509934.0400712601840496, 521858.9592969599762000 7509706.5850364100188017, 521402.0897869099862874 7509610.0089996904134750, 520906.0067273600143380 7509526.0576650602743030, 520005.1707852099789307 7509373.5673304200172424, 520000.0000000000000000 7509372.6918550245463848, 520000.0000000000000000 7509373.7060850486159325), (521279.9150416324264370 7501034.6344962781295180, 521278.8505533785792068 7501035.4778430974110961, 521543.4397763700108044 7501077.5368893602862954, 522373.8810591999790631 7501209.4896944900974631, 523030.4162207188783213 7501391.7815449377521873, 523031.2068098201416433 7501391.1711508315056562), (527953.2198819570476189 7501611.1241540247574449, 527952.4394709372427315 7501611.9876126917079091, 527953.2198819570476189 7501611.1241540247574449, 528269.5491108499700204 7501675.5494663203135133, 528644.5797393700340763 7501769.5606017401441932, 529113.4114604999776930 7501800.5704611297696829, 529800.9698748099617660 7501832.0698146400973201, 530707.2799876299686730 7501832.0906658703461289, 531832.3692170899594203 7502050.6015144297853112, 532080.1110293077072129 7502146.1749884476885200, 532080.7953341902466491 7502145.4462257148697972), (532471.3224294331157580 7501854.8021797873079777, 532470.6117427451536059 7501856.0567162586376071, 532551.2005992300109938 7501832.0981646096333861, 533020.0292473699664697 7501925.6022116597741842, 533613.8304010899737477 7502019.6212948998436332, 534488.9388966499827802 7502207.1201175795868039, 534926.4299064800143242 7502457.1309860097244382, 535239.0000169699778780 7502707.1384293204173446, 535551.5432611800497398 7503144.6412358498200774, 535676.5302752000279725 7503519.6384271895512938, 535926.5516183800064027 7503894.6519359098747373, 536270.3683300199918449 7504082.1422335496172309, 536676.5897916300455108 7504113.6475562499836087, 536989.1799710299819708 7504051.1521394196897745, 537895.5108797400025651 7503957.1480598496273160, 538364.2599795899586752 7504113.6601144503802061, 539051.8112280699424446 7504394.6494075302034616, 539676.8889227600302547 7504613.6505787996575236, 540145.7008413899457082 7504988.6477605598047376, 540770.7923636999912560 7505332.1517110802233219, 541145.7601316999644041 7505676.1485500596463680, 541614.6014505899511278 7506144.6516708899289370, 541895.8687167000025511 7506645.1619760403409600, 542083.3585663399426267 7507145.1590369800105691, 542208.3892406800296158 7507707.6589486598968506, 542145.9084491999819875 7508332.6592682404443622, 542239.6581270300084725 7508645.1607529902830720, 542458.4708741100039333 7508957.6600440395995975, 542677.2416874299524352 7509082.6624572100117803, 542864.7195600100094453 7509239.1611510002985597, 542927.2699991799890995 7509989.1508510801941156, 542927.2698631049133837 7510000.0000000000000000, 542928.2698600001167506 7510000.0000000000000000), (542679.7431424340466037 7490560.7071135109290481, 542680.4317333664512262 7490559.9283441118896008, 542592.3105463000247255 7490517.6513492902740836, 542387.8711325100157410 7490321.1590552795678377, 542278.4307441100245342 7490048.1597586404532194, 542272.8549225005554035 7490000.0000000000000000, 542271.8556170752272010 7490000.0000000000000000), (544273.9324713232927024 7491662.8551605539396405, 544274.6237809687154368 7491662.1230727611109614, 544238.3782646199688315 7491635.1603938303887844, 544135.6185495100216940 7491432.1611171104013920, 543918.3223162599606439 7491197.6698569804430008, 543420.6783191299764439 7490869.6585929403081536, 543038.3205035700229928 7490699.6702497899532318, 542815.3584537300048396 7490624.6607478400692344, 542773.7009090904612094 7490604.6751364599913359, 542773.0674823645967990 7490605.4474027175456285), (550000.0000000000000000 7492637.9491975577548146, 550000.0000000000000000 7492636.9501683469861746, 549769.4085893699666485 7492521.7100056502968073, 549501.4603817999595776 7492462.1984654497355223, 549174.0184454700211063 7492372.7009732704609632, 548668.0100578600540757 7492343.2083122497424483, 547774.9591860000509769 7492343.1903728395700455, 547358.2082652900135145 7492402.6892563700675964, 546971.2795143900439143 7492432.1799347596243024, 546524.7789019900374115 7492402.6676745600998402, 546018.6897931500570849 7492283.6581326704472303, 545304.2811550099868327 7491926.1592656997963786, 545036.3909502900205553 7491777.1605294896289706, 544828.0104751100298017 7491628.6593068800866604, 544708.9605470900423825 7491509.6700969301164150, 544618.5316022647311911 7491404.1884233895689249, 544617.8842772342031822 7491404.9697802281007171))',
        'MULTILINESTRING ((520293.4320175029570237 7501708.4171284157782793, 520293.4981995084672235 7501707.4205201249569654, 520000.0000000000000000 7501674.5829317998141050, 520000.0000000000000000 7501675.5891712857410312), (520000.0000000000000000 7508845.7206690181046724, 520000.0000000000000000 7508846.7200987627729774, 520052.9280038099968806 7508862.0008838102221489, 521090.5512352299992926 7509073.0105239599943161, 522005.9512440299731679 7508998.0294947298243642, 522324.3404851399827749 7509085.0209561996161938, 522451.5710224300273694 7509097.0301381601020694, 522591.6214781600283459 7509122.0274216998368502, 524151.1692813800182194 7509438.0516055300831795, 524456.6897462300257757 7509487.5595593899488449, 525140.4391166700515896 7509496.5704689295962453, 526119.0513356799492612 7509602.5791763495653868, 527124.1199973500333726 7509920.1001460999250412, 527347.8288250340847299 7510000.0000000000000000, 527348.8276206540176645 7510000.0000000000000000), (522351.1343196252128109 7501916.1655494002625346, 522352.2528255800134502 7501915.3037830777466297, 521104.1391963799833320 7501751.4809457501396537, 520737.7721352900261991 7501723.9786810996010900, 520420.9591136300005019 7501715.1471717469394207, 520420.1756578796193935 7501715.7678689807653427), (527125.4548547224840149 7502377.7747816061601043, 527126.3209163650171831 7502376.8168430794030428, 526700.8509709000354633 7502355.5393781904131174, 525246.2184770300518721 7502408.5209231497719884, 524294.1117317799944431 7502249.5209683403372765, 523447.7796929900068790 7502091.0098048197105527, 522583.0080486031947657 7501863.5510688340291381, 522582.2173395422287285 7501864.1615555742755532), (530798.8710058355936781 7503475.2625857004895806, 530799.6446038441499695 7503474.4564733263105154, 530292.3180354699725285 7503316.0992012796923518, 529603.4816261499654502 7503154.0790386795997620, 529076.7311885600211099 7502911.0787954898551106, 528266.3094259300269186 7502728.5696691796183586, 527759.8099331599660218 7502688.0595766501501203, 527233.1392903799423948 7502607.0608489904552698, 527065.4771157877985388 7502593.2268756395205855, 527064.8072133261011913 7502593.9678452722728252), (539955.9743753559887409 7510000.0000000000000000, 539956.9738961729453877 7510000.0000000000000000, 540057.3710624600062147 7509681.6689012898132205, 540295.3694572099484503 7509391.1672181095927954, 540480.4705794400069863 7509232.1700646700337529, 540612.7384639199590310 7509073.6586520997807384, 540665.5814983600284904 7508571.1700814096257091, 540348.2114157699979842 7507671.6596398204565048, 539925.0498745300574228 7507090.1577369300648570, 539607.6706715000327677 7506746.1599598200991750, 539052.2007989300182089 7506296.6606875201687217, 538205.9094809900270775 7505952.6699313903227448, 537068.6509273999836296 7505741.1603932101279497, 536169.3901651899795979 7505661.6614198600873351, 535217.2919880599947646 7505503.1506273699924350, 534635.4301719899522141 7505317.6512130200862885, 534027.0999437200371176 7504974.1400412498041987, 533603.8817716699559242 7504709.6272615399211645, 533233.6514340500580147 7504312.6317273303866386, 532466.6187711399979889 7503969.1208717301487923, 531858.3409144999459386 7503651.6198519701138139, 531329.3389675599755719 7503360.6116438498720527, 531139.0552856920985505 7503265.5203097239136696, 531138.3627150403335690 7503266.2411932516843081), (546233.0059536607004702 7490000.0000000000000000, 546232.0059536602348089 7490000.0000000000000000, 546232.0095286100404337 7490128.6799554601311684, 546425.2915191700449213 7490406.1914659300819039, 546769.6909271699842066 7490574.1895378697663546, 547082.8998588599497452 7490707.6977680595591664, 547486.2694205499719828 7490854.1999811800196767, 548366.2786710499785841 7491037.7003361200913787, 548806.2997251900378615 7491074.2083575604483485, 549612.9294149799970910 7491257.7084028301760554, 550000.0000000000000000 7491301.9012328926473856, 550000.0000000000000000 7491300.8947363123297691))',
        'MULTILINESTRING ((520000.0000000000000000 7496937.7387266764417291, 520000.0000000000000000 7496940.1299340603873134, 520002.3885029399534687 7496934.9419469097629189, 520209.1083645300241187 7496271.9499225998297334, 520062.0993496900191531 7496119.9397722100839019, 520000.0000000000000000 7496068.6646597366780043, 520000.0000000000000000 7496069.9614912457764149), (523381.4890356060350314 7498414.2473349031060934, 523382.0906666574883275 7498413.4010553266853094, 523169.7519518999615684 7498336.9916863301768899, 523067.8999133500619791 7498311.9897804697975516, 522895.7909434399916790 7498236.4781237496063113, 522615.2015952100045979 7498072.4703307598829269, 521882.6812058400246315 7497821.9590960601344705, 521526.3299231799901463 7497779.4584824498742819, 521208.3405302499886602 7497768.4611136997118592, 520979.3200964700081386 7497743.9708666298538446, 520246.9187887199805118 7497531.9419434396550059, 520000.0000000000000000 7497468.9978941539302468, 520000.0000000000000000 7497470.0298743853345513), (525280.9678179419133812 7499008.5035365633666515, 525281.4949324887711555 7499007.5775622371584177, 525203.0376252799760550 7498994.5084805004298687, 524869.2387656100327149 7498918.5108462898060679, 524080.2303868400631472 7498843.0101468600332737, 523609.8704664499964565 7498706.4891386404633522, 523343.3935284681501798 7498640.4214922022074461, 523342.8145249948720448 7498641.2359428005293012), (534272.9718145289225504 7499007.8629990173503757, 534273.7464286693139002 7499006.7735539842396975, 533821.5094927400350571 7499055.6005880599841475, 533411.8113768999464810 7499131.0899759698659182, 532789.7513422400224954 7499313.0907301902770996, 532046.2501181999687105 7499359.0887924302369356, 531560.6700862100115046 7499404.5924122100695968, 531257.2500160000054166 7499404.5794073604047298, 530968.9126983700552955 7499419.5698141297325492, 530650.2800068800570443 7499495.5805069999769330, 530361.9716437599854544 7499601.5801136298105121, 530210.2997778200078756 7499647.0707991197705269, 530073.7308944800170138 7499647.0701257800683379, 529891.5828257299726829 7499586.5608606198802590, 529755.0991281700553373 7499495.5807948503643274, 529648.8292915900237858 7499434.5578709095716476, 528920.5188349500531331 7499359.0610773200169206, 527979.7492919899523258 7499328.5417763004079461, 527312.1506295599974692 7499283.0298913996666670, 526705.1804492699448019 7499192.0412488700821996, 525809.9994778400287032 7499116.0194425499066710, 525476.2009420599788427 7499040.0108542302623391, 525386.5961969492491335 7499025.0848800987005234, 525386.1020599714247510 7499025.9529232168570161), (547423.9512601103633642 7510000.0000000000000000, 547425.9598791532916948 7510000.0000000000000000, 547270.3128660599468276 7509910.6501949401572347, 547123.4215484600281343 7509784.1305715003982186, 546931.6794150100322440 7509581.6403483096510172, 546815.5487570599652827 7509245.1501061804592609, 546813.4198143399553373 7508786.6386062400415540, 546793.0495528799947351 7508519.6417614798992872, 546760.1718947299523279 7508297.1516221202909946, 546630.1392768600489944 7507705.6514096902683377, 546603.1807938199490309 7507387.6502600098028779, 546506.4978075700346380 7507121.1480851704254746, 546347.0615013099741191 7507032.6372693302109838, 546098.2091887400019914 7506843.1486029401421547, 545931.3088385299779475 7506525.6525994800031185, 545783.2304036399582401 7506157.6511867698282003, 545590.6004368900321424 7505770.1490056701004505, 545302.2208498599939048 7505307.1615147199481726, 544943.5218329799827188 7504761.6597001496702433, 544725.5899502099491656 7504406.6611849600449204, 544501.3301299399463460 7504057.6593816699460149, 544123.7492673799861223 7503569.6505599301308393, 543753.3717356600100175 7503247.1585790300741792, 543357.1403491899836808 7502835.6512618502601981, 543024.9804781300481409 7502538.1512261899188161, 541869.5108813600381836 7501596.1483753900974989, 541378.0882920400472358 7501255.1483567599207163, 540868.3004191899672151 7501022.1397865395992994, 540434.9686214999528602 7500840.1404092302545905, 539886.7884534699842334 7500575.6385581698268652, 539472.3796638699714094 7500361.6388751100748777, 539128.0079635200090706 7500159.6402211003005505, 538782.9978578999871388 7499830.6288305800408125, 538476.5108094000024721 7499571.6312278602272272, 538240.7492773899575695 7499470.6311592804268003, 537903.1197484800359234 7499358.1302875401452184, 537565.7982155899517238 7499296.1290474496781826, 536808.4890007500071079 7499179.6114020701497793, 536356.4701961500104517 7499080.1195044601336122, 535866.5394221700262278 7499031.6102768797427416, 535319.4902977800229564 7498996.6221683695912361, 534772.7408331099431962 7499031.6016195202246308, 534285.8976723682135344 7499162.2542209504172206, 534285.3182902499102056 7499163.0690846843644977), (537098.9795529744587839 7490000.0000000000000000, 537097.9807571568526328 7490000.0000000000000000, 537082.7614457299932837 7490032.1078298501670361, 537064.9390341599937528 7490293.1209420198574662, 537085.1899132600519806 7490547.6190917901694775, 537118.4598460316192359 7490667.2163930963724852, 537119.4298784348648041 7490666.9776619225740433), (537417.0468347219284624 7490706.0521089499816298, 537416.5515447265934199 7490706.2860060287639499, 537416.1256071323296055 7490706.4396555135026574, 537487.4418410599464551 7490882.6095111798495054, 537654.0587315800366923 7491149.1116127697750926, 537858.8020591400563717 7491415.1179784098640084, 537992.9993732899893075 7491548.1193249300122261, 538056.9795873910188675 7491588.6379907196387649, 538057.9059167269151658 7491588.2641664957627654), (538156.4071117863059044 7491653.3615979626774788, 538155.5143074670340866 7491653.8223220268264413, 538299.8710431599756703 7491877.6190374298021197, 538434.6211939599597827 7492137.6200953898951411, 538652.5613591700093821 7492492.6281045097857714, 538768.5901984019437805 7492667.4812555732205510, 538769.1307524497387931 7492666.4872721917927265), (539076.4825414990773425 7492310.6365013578906655, 539076.0048737846082076 7492311.5148478839546442, 539256.6677916899789125 7492496.1301840599626303, 539703.5787173799471930 7492843.6307888999581337, 540053.9791185399517417 7492994.6384690301492810, 540455.2317104099784046 7493113.1314762597903609, 540786.5202559200115502 7493232.1414313204586506, 541041.4298104699701071 7493351.6418332597240806, 541239.2018903200514615 7493478.1304302699863911, 541332.9279536200920120 7493553.2876135194674134, 541333.7303589903749526 7493552.6927433693781495), (541996.2659877514233813 7493323.2507506581023335, 541995.3182718952884898 7493323.6649971948936582, 542500.6798850599443540 7493961.1386070298030972, 542743.6307616099948063 7494233.6408173600211740, 543082.0077898399904370 7494518.1417143298313022, 543356.1986791399540380 7494663.1389560597017407, 543789.2602507199626416 7494794.6498441295698285, 544139.2511316499439999 7494849.6397826299071312, 544641.6300744400359690 7494840.6485728900879622, 545175.3190517800394446 7494736.1496383799239993, 545741.0922157400054857 7494688.1493368400260806, 546307.7507409299723804 7494838.1492867600172758, 546786.0567151700379327 7495109.1620891699567437, 547188.0597629300318658 7495399.6700470102950931, 547621.6999697199789807 7495652.1683375202119350, 547965.6989555200561881 7495764.6696619400754571, 548207.3907420800533146 7495776.1700877603143454, 548481.1595977500546724 7495844.6798296095803380, 548863.0792892000172287 7495925.1790779400616884, 549506.2123941599857062 7496087.1879144096747041, 550000.0000000000000000 7496197.8834195006638765, 550000.0000000000000000 7496196.8585999859496951))',
        'MULTILINESTRING ((524266.0954626141465269 7490000.0000000000000000, 524264.0581623602192849 7490000.0000000000000000, 524274.6716975499875844 7490005.9794874498620629, 524580.9296716400422156 7490207.9900786597281694, 525263.6310803899541497 7490681.4799169795587659, 525633.8797883000224829 7490965.9987491900101304, 526080.4097873299615458 7491262.5022844001650810, 526392.7913924700114876 7491426.5083817597478628, 526680.5080642091343179 7491700.2571335136890411, 526680.9617382686119527 7491699.3084705946967006), (527736.6980700913118199 7490014.9772448325529695, 527736.3587646851083264 7490015.9580855472013354, 528010.6905607599765062 7490033.0087006296962500, 528554.6395948999561369 7490014.5194057403132319, 528659.0582184605300426 7490000.0000000000000000, 528651.7973667406477034 7490000.0000000000000000), (528242.3239959123311564 7492064.7222724705934525, 528241.7088020627852529 7492065.7254664935171604, 528666.1973901799647138 7492063.5307877697050571, 529212.3307416000170633 7491907.5399811398237944, 529719.6322913799667731 7491605.5284209195524454, 530163.0910896200221032 7491234.0305473301559687, 530511.5112620899453759 7490958.5415406301617622, 531453.6296071013202891 7490000.0000000000000000, 531452.2274564296239987 7490000.0000000000000000))',
        'MULTILINESTRING ((522423.1507045699981973 7499760.7176261981949210, 522423.9203800179529935 7499759.6364277126267552, 522349.8299966399208643 7499766.9809384401887655, 522165.4282854500343092 7499767.9805885404348373, 521783.7590640200069174 7499750.9782179798930883, 521446.5883731800131500 7499727.4790324503555894, 521058.2386758999782614 7499640.4595605703070760, 520701.5588286300189793 7499521.4718082202598453, 520204.5092078600428067 7499313.9601706201210618, 520000.0000000000000000 7499193.2638115361332893, 520000.0000000000000000 7499194.4249778995290399), (522555.9904544320888817 7499746.5444441922008991, 522555.4111143556656316 7499747.3582698311656713, 522555.9904544320888817 7499746.5444441922008991, 522960.0394677100121044 7499706.4914416698738933, 523297.1696609099744819 7499723.9880460202693939, 523539.3288009500829503 7499824.4885594900697470, 523807.0801918199867941 7499963.0007417099550366, 524452.2040554300183430 7500171.9706728672608733, 524451.4118937810417265 7500172.7652285397052765), (529030.4932832464110106 7500200.7058669319376349, 529031.2178299439838156 7500199.9257171414792538, 528991.0805857500527054 7500183.0489843999966979, 528679.6486600999487564 7500223.0495249899104238, 527986.5626644400181249 7500239.5480527197942138, 527668.6504820300033316 7500241.0489183496683836, 526409.2975434400141239 7500216.0304062804207206, 525849.7599196099909022 7500225.5205484097823501, 525373.0094749700510874 7500266.0211616195738316, 525080.3788815899752080 7500248.5201182495802641, 524717.1588156500365585 7500210.0084451204165816, 524719.1996265298221260 7500211.2304345155134797), (532993.0805511008948088 7500834.6403764961287379, 532993.3489576445426792 7500834.0452068466693163, 532993.5077827317873016 7500833.7281568944454193, 532828.7419588699704036 7500779.5898232003673911, 532059.2178660000208765 7500783.5925765298306942, 531715.7397694600513205 7500766.5896713202819228, 531009.3391590700484812 7500675.0807855604216456, 530322.5721776599530131 7500685.0713867703452706, 529769.5689500400330871 7500739.0687459995970130, 529089.1306385099887848 7500749.0586953703314066, 528660.8910052364226431 7500745.6074274424463511, 528660.2112883693771437 7500746.3392704948782921), (545728.9288492670748383 7510000.0000000000000000, 545730.2700878457399085 7510000.0000000000000000, 545584.7022705799899995 7509837.1403594203293324, 545386.0617979000089690 7509520.1391145400702953, 545238.0399551700102165 7509170.6511606499552727, 545122.0989813000196591 7508859.6512553403154016, 544986.9678246000548825 7508529.6407046103850007, 544743.6496689900523052 7508180.6489076800644398, 544659.9523990500019863 7507964.6498591499403119, 544499.4011402300093323 7507634.6525363801047206, 544320.0205451600486413 7507368.1501189004629850, 544044.8593324699904770 7507007.1477869795635343, 543852.7414086499484256 7506721.6492448104545474, 543666.6799941799836233 7506391.6518390402197838, 543486.6189286799635738 7505966.1496158502995968, 543177.9487287199590355 7505267.6511040404438972, 542851.5999981700442731 7504830.1483773496001959, 542628.1512679100269452 7504653.1477385796606541, 541996.4995115000056103 7504230.1516731502488256, 541486.3089837899897248 7503927.6487837303429842, 541052.4398514899658039 7503624.6398146301507950, 540714.5785347600467503 7503454.6588033195585012, 540331.7501426199451089 7503195.6492001600563526, 539885.0605549899628386 7502867.1490661101415753, 539534.2795183400157839 7502671.6515465499833226, 539088.3392634700285271 7502508.6507563600316644, 538115.0518896100111306 7502437.6390984999015927, 537790.3291457899613306 7502369.1493166498839855, 537573.5679509800393134 7502255.6492295796051621, 537394.9012426600093022 7502116.6387358000501990, 537330.5981657799566165 7501970.6416400596499443, 537323.5123430900275707 7501824.6311678895726800, 537322.0002546999603510 7501506.1379719302058220, 537263.7595541899790987 7501296.6407880699262023, 537039.8101914300350472 7501017.6294886004179716, 536612.6102296400349587 7500784.6198029303923249, 536218.0508771500317380 7500716.6298277098685503, 535734.4196069199824706 7500662.1207142304629087, 534972.0878904700512066 7500831.6092656701803207, 534616.1697530100354925 7500865.1187592102214694, 534266.3894202300580218 7500867.1096779303625226, 534081.7079486900474876 7500817.1092230295762420, 533864.9377766799880192 7500710.1096216002479196, 533501.7106121799442917 7500546.6094194203615189, 533336.0891072107478976 7500498.7944972664117813, 533335.5105264986632392 7500499.6086738053709269), (540652.7093492220155895 7490387.5987470783293247, 540651.8620166190667078 7490388.1283743241801858, 540716.9413671500515193 7490648.6385009195655584, 540890.1206744499504566 7490959.6384145403280854, 541069.2507452500285581 7491181.1377072203904390, 541209.8013249200303108 7491327.1401432603597641, 541318.3109204999636859 7491402.6507735904306173, 541522.3790933899581432 7491516.1502364799380302, 541726.1788735899608582 7491597.6481381803750992, 541866.3201340900268406 7491635.1395976599305868, 542152.3343121195212007 7491644.4789796750992537, 542152.5743099044775590 7491643.5087957028299570), (542310.5385527068283409 7491467.5166142378002405, 542310.2017283077584580 7491468.4781577382236719, 542387.0478783199796453 7491479.6506026098504663, 542508.2986270999535918 7491568.1487769903615117, 542623.6207645100075752 7491745.6495260195806623, 542707.3887829299783334 7491974.6494024097919464, 542804.3198439499828964 7492292.1578110801056027, 542874.8705274199601263 7492419.1503577204421163, 543015.3294375799596310 7492539.1386309601366520, 543157.3559995990945026 7492607.0648720515891910, 543158.2323379423469305 7492606.5854770792648196), (543285.3188949242467061 7492641.8684887290000916, 543284.1699949501780793 7492642.7301141498610377, 543785.6400337499799207 7492706.6602805098518729, 544142.1796537400223315 7492794.1595931304618716, 544346.3083736399421468 7492926.6612573396414518, 544550.7481039999984205 7493129.1590701797977090, 544793.1985071500530466 7493287.1483336500823498, 545098.8382594300201163 7493361.6606107000261545, 545620.5201594299869612 7493403.1614144695922732, 546046.6321061899652705 7493420.1597612500190735, 546440.8310840800404549 7493399.1707573700696230, 546822.0285851999651641 7493326.6682714400812984, 547241.4683955300133675 7493267.1695830002427101, 548296.9793255199911073 7493223.1783396899700165, 548710.8022534899646416 7493316.6910186298191547, 548978.0409311200492084 7493359.6988639002665877, 549474.6489930299576372 7493478.1999030597507954, 549870.0192977499682456 7493704.7011540196835995, 550000.0000000000000000 7493802.1282507702708244, 550000.0000000000000000 7493800.8785204347223043))',
        'MULTILINESTRING ((524718.0163713778601959 7500210.5219292528927326, 524717.1588156500365585 7500210.0084451204165816, 524574.9860907683614641 7500190.5280320746824145, 524574.2798689020564780 7500191.2344054849818349))',
        'MULTILINESTRING ((524193.8328189906897023 7500431.1064537204802036, 524194.6307847223361023 7500430.3067480474710464, 523397.8001402500085533 7500183.4910121802240610, 522854.1015355099807493 7500070.4906302299350500, 522324.1118496610433795 7500072.2749801976606250, 522323.5325372974039055 7500073.0887669073417783), (527905.8476731716655195 7501050.4671189449727535, 527908.1407460733316839 7501050.5568969398736954, 527430.0695936999982223 7500795.5491229798644781, 526591.8898101799422875 7500682.0311559904366732, 525549.8513239700114354 7500614.0300792902708054, 525164.7414901100564748 7500478.0103883901610970, 524717.1588156500365585 7500210.0084451204165816, 524716.1685345589648932 7500209.8727574581280351))',
        'MULTILINESTRING ((522198.7010203180252574 7500076.0088302092626691, 522199.4341642370563932 7500074.9790627742186189, 521744.1016206499771215 7500092.9814525097608566, 521177.7584161399863660 7500024.9713861504569650, 520656.7282556199934334 7499911.9717762302607298, 520226.3289424299728125 7499730.4618171695619822, 520000.0000000000000000 7499617.2908613989129663, 520000.0000000000000000 7499618.4089082013815641), (532860.9091847165254876 7501127.3776061432436109, 532861.3487732587382197 7501126.4021477382630110, 532255.2395347800338641 7501090.0919910203665495, 531190.5300996799487621 7501090.0910327201709151, 530442.9822135099675506 7501135.0692772204056382, 529763.3494318500161171 7501135.0693844202905893, 529061.0896587100578472 7501044.5682494696229696, 528473.1436554730171338 7500979.1873466055840254, 528472.5273607608396560 7500979.9734598090872169), (544737.4093717507785186 7510000.0000000000000000, 544738.7021026653237641 7510000.0000000000000000, 544626.1803077399963513 7509862.6509016100317240, 544599.6914848999585956 7509307.1512340800836682, 544520.3602272400166839 7508619.1491003800183535, 544255.8195605799555779 7507878.6505708796903491, 543753.3111433800077066 7507006.1485301395878196, 543197.8599954500095919 7506291.6510444404557347, 542748.2417293100152165 7505604.1493647499009967, 542034.1492941799806431 7504704.6514335004612803, 541293.5891299600480124 7504202.1505518397316337, 540447.2613627200480551 7503647.1504111299291253, 539204.1899481900036335 7503038.6409026896581054, 538490.0402022900525481 7502853.6400044597685337, 537934.5981001099571586 7502959.1494819503277540, 537458.5619110600091517 7503118.1395992599427700, 536956.0390386499930173 7503170.6392884301021695, 536770.9210307099856436 7502932.6405451204627752, 536718.0086382699664682 7502615.6299286996945739, 536718.0077891800319776 7502060.1385293100029230, 536585.7404540400020778 7501557.6296280203387141, 536215.4412918200250715 7501319.6187131898477674, 535871.5983779799425974 7501240.1199650401249528, 535289.7498613100033253 7501266.6198128499090672, 534813.6585787199437618 7501213.6187100997194648, 533649.9409386699553579 7501055.1196561502292752, 533032.6647448980947956 7500990.1133196894079447, 533032.2540093590505421 7500991.0240919245406985), (540991.6145223230123520 7490051.9104150431230664, 540990.9120639010798186 7490052.6205057417973876, 540994.1994521199958399 7490125.1404810203239322, 541020.3115120199508965 7490265.1499587204307318, 541071.8600603099912405 7490404.6513284202665091, 541225.5412088100565597 7490626.6515073096379638, 541378.8100700799841434 7490759.6511909803375602, 541621.1913036600453779 7490904.6496356101706624, 541850.5094163799658418 7490992.6486446103081107, 542117.9989447799744084 7491067.6498956503346562, 542339.6836309707723558 7491081.8338681170716882, 542340.0139575036009774 7491080.8908742861822248), (542448.3669101102277637 7491085.2667160956189036, 542447.8281487500062212 7491086.2563206022605300, 542645.9278556300560012 7491090.1496953703463078, 542881.8001631699735299 7491216.1593797197565436, 543022.5110186899546534 7491374.6574428202584386, 543093.4196471800096333 7491577.6600298201665282, 543100.6207402900326997 7491755.6605573296546936, 543146.0011632499517873 7491940.1524548903107643, 543242.2113231299445033 7492092.6493198703974485, 543420.5594578799791634 7492167.6605370296165347, 543736.0608669177163392 7492178.8340415228158236, 543736.8605833266628906 7492178.2342887129634619), (543994.7063397207530215 7492105.4846383240073919, 543993.9952419346664101 7492106.2410740470513701, 544164.7493750499561429 7492195.6611232999712229, 544311.5085405800491571 7492303.1592243798077106, 544713.1909174999454990 7492510.6592104202136397, 545414.1404617800144479 7492806.1695769801735878, 545739.0797240600222722 7492931.6574951196089387, 546146.6309219299582765 7493050.6718051703646779, 546566.3691623499616981 7493067.1693900702521205, 547017.7696958100423217 7493033.1687579201534390, 547411.5708465500501916 7492935.1810991801321507, 547824.2799819100182503 7492786.6796223297715187, 548154.3804447900038213 7492676.6903927102684975, 548555.1085242800181732 7492681.1924451002851129, 548956.0910621000220999 7492755.1982696000486612, 549427.3525090899784118 7492886.2018355997279286, 549924.2300843800185248 7493068.2112118601799011, 550000.0000000000000000 7493102.4734313925728202, 550000.0000000000000000 7493101.3759462386369705))',
        'MULTILINESTRING ((520000.0000000000000000 7500333.5864726584404707, 520000.0000000000000000 7500334.5854991283267736, 520251.9812008599983528 7500460.4708616798743606, 520790.4299846600042656 7500582.9693757295608521, 521328.8990372599801049 7500656.4806792000308633, 521537.9934471686137840 7500702.5902975173667073, 521538.7771375657757744 7500701.9694143859669566, 521537.9934471686137840 7500702.5902975173667073), (521826.3962308935006149 7510000.0000000000000000, 521830.2234692216152325 7510000.0000000000000000, 521671.4390298799844459 7509957.0189364701509476, 520790.3393039599759504 7509810.0001919697970152, 520000.0000000000000000 7509619.7119117267429829, 520000.0000000000000000 7509618.6830340670421720, 520000.0000000000000000 7509619.7119117267429829), (532600.3302651896374300 7501627.1083485167473555, 532599.7396073570707813 7501628.1497170943766832, 532856.6521893100580201 7501611.1112058302387595, 533444.0185844299849123 7501684.6103292601183057, 534325.1210312099428847 7501880.1196714900434017, 534985.9214213599916548 7502076.1217858698219061, 535646.8110275299986824 7502516.6302939895540476, 535891.5287010800093412 7502908.1374861896038055, 536013.9019787500146776 7503250.6406890796497464, 536258.6794978299876675 7503520.1420491002500057, 536527.9085336000425741 7503593.6495772004127502, 536992.9320167599944398 7503520.1422608699649572, 537555.8628507399698719 7503544.6488754795864224, 538020.8807973100338131 7503667.1522581996396184, 538755.1426198399858549 7503960.6502358699217439, 539244.6298891199985519 7504132.1499195201322436, 540125.7701951599447057 7504572.6497226702049375, 540762.0786962399724871 7504939.6401027701795101, 541716.5790550899691880 7505698.1617022398859262, 541936.8897461800370365 7506065.6509176101535559, 542157.1883640999440104 7506530.6489702202379704, 542426.4309210999635980 7507142.6610005302354693, 542597.7511561999563128 7507729.6500914199277759, 542622.2203807899495587 7508170.6593472696840763, 542866.9608011499512941 7508635.6476191803812981, 543136.2108244899427518 7508978.1516894698143005, 543283.1098597400123253 7509198.1474713198840618, 543356.4892539000138640 7509687.6487833503633738, 543368.5279655156191438 7510000.0000000000000000, 543369.5268157882383093 7510000.0000000000000000), (542475.7818017150275409 7490825.7722711022943258, 542476.2623097165487707 7490824.8896671906113625, 542121.6113261700375006 7490675.1504720998927951, 541942.0503731799544767 7490540.6584189096465707, 541762.5393725200556219 7490428.6507309796288610, 541605.5117328099440783 7490293.6494011301547289, 541560.6191276300232857 7490159.1616177195683122, 541538.0986778000369668 7490024.6489791097119451, 541545.8831280654994771 7490000.0000000000000000, 541544.8344431390287355 7490000.0000000000000000), (544083.8251957478933036 7491864.6897576972842216, 544084.5100616407580674 7491863.9622678663581610, 543916.8111612600041553 7491662.6594603899866343, 543467.9909774400293827 7491303.6579534802585840, 543288.5212596700293943 7491169.1625096900388598, 543041.6283885700395331 7491034.6487735398113728, 542929.4202479800442234 7490989.6511897603049874, 542705.0600305399857461 7490877.1614052504301071, 542571.8228042328264564 7490858.5011789817363024, 542571.3451686579501256 7490859.3785067796707153), (550000.0000000000000000 7492916.1826057024300098, 550000.0000000000000000 7492915.1826563579961658, 549683.9776767699513584 7492784.7010641396045685, 549457.1530677999835461 7492715.7030614195391536, 549167.8620524300495163 7492627.6976040797308087, 549044.6585097600473091 7492599.7510081203654408, 548674.1506595800165087 7492515.7009411500766873, 548609.5248929499648511 7492518.2924996195361018, 548113.1504889399511740 7492538.1886065695434809, 547800.5016407900257036 7492572.8691065600141883, 547379.9360571899451315 7492619.5098762298002839, 547103.3177899100119248 7492650.1804305203258991, 546676.9415344200097024 7492650.1683770902454853, 546362.8189851900096983 7492582.6708111502230167, 546048.6516915999818593 7492493.1693898700177670, 545734.4392289100214839 7492380.6604282101616263, 545353.0321352999890223 7492201.1703274799510837, 544926.6382374600507319 7492066.6692933803424239, 544724.7222984499530867 7491977.1600893298164010, 544410.5210711299441755 7491797.6681312201544642, 544324.8354344329563901 7491754.6016815919429064, 544324.1489822026342154 7491755.3286254471167922))',
        'MULTILINESTRING ((530271.9887491008266807 7504023.8181659672409296, 530272.8096558250254020 7504022.9626687979325652, 529716.5596407300326973 7503911.6010149903595448, 529213.5023450599983335 7503780.5819101296365261, 528797.9000049700262025 7503627.0802137600257993, 527507.3776305499486625 7503343.0594466198235750, 526960.5505161000182852 7503255.5484263198450208, 526610.5993896899744868 7503255.5402162401005626, 526129.4297748099779710 7503146.0497182803228498, 525298.2094937399961054 7503146.0410827100276947, 524313.9806665199575946 7502993.0200903099030256, 522979.7296735499985516 7502796.0099626500159502, 521973.5802435199730098 7502730.4904361795634031, 520967.4508550300379284 7502533.4807597603648901, 520000.0000000000000000 7502409.3710406739264727, 520000.0000000000000000 7502410.3792356532067060), (520000.0000000000000000 7507850.7456019073724747, 520000.0000000000000000 7507851.7444353895261884, 520459.1205355499987490 7507962.4997558500617743, 521252.5692508600186557 7508095.0084923598915339, 522046.0097163000609726 7508306.5182301895692945, 522495.6195994799491018 7508412.0197906903922558, 523447.7511029100278392 7508491.5316420802846551, 524135.4115040699834935 7508571.0508861597627401, 524796.6501091100508347 7508809.0602384395897388, 525537.1616826100507751 7508888.5604379596188664, 526304.1784057499608025 7509047.0794246401637793, 527150.5723233999451622 7509258.6008259104564786, 527891.1703374399803579 7509549.6305759903043509, 528759.1812102999538183 7509752.1486169397830963, 529408.9095765600213781 7509934.6589543297886848, 529609.4298291478771716 7510000.0000000000000000, 529610.4290333546232432 7510000.0000000000000000), (538922.8089907000539824 7510000.0000000000000000, 538923.8080818294547498 7510000.0000000000000000, 538972.1599029799690470 7509914.6897562900558114, 539114.2378285899758339 7509528.6699069095775485, 539134.5507683800533414 7509041.6695242598652840, 539012.7302670100471005 7508493.1818856503814459, 538586.3296623999485746 7507965.1697401199489832, 538180.2115608500316739 7507599.6715208096429706, 537631.9808313100365922 7507214.1684171101078391, 537428.9305590899894014 7507011.1716584600508213, 537002.5204438799992204 7506686.1613326398655772, 536616.7715214400086552 7506544.1692011002451181, 535743.6882477200124413 7506361.1594366002827883, 534931.5578075499506667 7506158.1588880904018879, 534038.1795864100567997 7505813.1479306500405073, 533489.9179332499625161 7505488.1394624896347523, 533022.9010144800413400 7505082.1303953798487782, 532495.0114416599972174 7504737.1207175096496940, 531906.1711659600259736 7504513.6188210602849722, 530728.5093100100057200 7504046.6094597103074193, 530502.3308432935737073 7503928.6902586026117206, 530501.6387504261219874 7503929.4114401936531067), (549037.6348608708940446 7490000.0000000000000000, 549036.4332247237907723 7490000.0000000000000000, 549043.9079029599670321 7490011.2185191400349140, 549204.8611184799810871 7490100.2200202103704214, 549687.5501057700021192 7490279.2203355301171541, 550000.0000000000000000 7490413.2414639443159103, 550000.0000000000000000 7490412.1533525427803397))',
    ]
)
bc_gdf = geopandas.GeoDataFrame(basal_c, geometry=bc_geoms, crs='EPSG:28350')

structures = pandas.DataFrame(
    {
        'ID': [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
        'DIPDIR': [190.0, 190.0, 0.0, 330.0, 165.0, 150.0, 180.0, 210.0, 240.0, 270.0, 300.0],
        'DIP': [55, 55, 15, 15, 0, 45, 30, 20, 10, 5, 50],
        'OVERTURNED': [False] * 11,
        'BEDDING': [False] * 11,
        'X': [
            548279.320612,
            548249.155883,
            546137.857561,
            543754.180680,
            520512.912720,
            528512.912720,
            529512.912720,
            530512.912720,
            531512.912720,
            532512.912720,
            533512.912720,
        ],
        'Y': [
            7.493304e06,
            7.493512e06,
            7.494607e06,
            7.504599e06,
            7.497506e06,
            7.497806e06,
            7.498506e06,
            7.499506e06,
            7.500506e06,
            7.501506e06,
            7.502506e06,
        ],
        'Z': [543.0, 543.0, 532.0, 559.0, 503.0, 553.0, 563.0, 573.0, 583.0, 593.0, 603.0],
        'layerID': [3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2],
    }
)

# sampled contacts

IDS = [
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    1,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    4,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    3,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    8,
    9,
    9,
    9,
    9,
    9,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    7,
    5,
    5,
    5,
    5,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    6,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    5,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
    2,
]
X = [
    520000.0,
    520992.6699257682,
    521984.6869456721,
    522969.6388939231,
    523954.5908421741,
    524942.1013152138,
    525930.4850180566,
    526908.535657469,
    527891.805258285,
    528880.6323520339,
    529855.6352847969,
    530832.075832642,
    531813.1634102948,
    532804.1384816798,
    533019.2882480841,
    532243.2409444518,
    531376.1445685071,
    530480.68269627,
    529612.3980015446,
    528734.9447395488,
    527783.6457775467,
    526804.9871671252,
    525809.8226356399,
    524816.1487769039,
    523829.21986169985,
    522842.8215091018,
    521858.22077695775,
    520000.0,
    520979.22944531543,
    521963.40533887857,
    522941.7587850634,
    523934.147302897,
    524925.392627546,
    525924.0314485825,
    526915.8477737093,
    527895.0821625991,
    528872.0657047849,
    529800.4947308041,
    530625.584657682,
    531569.6370222451,
    532437.052847155,
    533287.2652070294,
    534152.4134851344,
    535068.2792519473,
    535460.6891555252,
    536224.4983751376,
    533429.9102329265,
    532481.4641957916,
    531544.0571021343,
    530594.6994174849,
    529610.6855646572,
    528622.769870241,
    527649.2613247755,
    526678.2236776681,
    525700.529526533,
    524727.502968803,
    523736.1966217523,
    522758.8160138651,
    521787.52440495713,
    520000.0,
    523032.43525349256,
    522075.0050621681,
    521094.2495766504,
    521279.9150416324,
    527953.219881957,
    528932.1661473936,
    529931.0489500397,
    530926.9444223469,
    532471.3224294331,
    533451.0808071761,
    534430.5181256596,
    535259.9903455864,
    535733.754706194,
    536530.414399986,
    537520.9367684029,
    538483.8842249501,
    539416.5488248907,
    540254.201848269,
    541073.3185078846,
    541731.634304068,
    542111.7328311033,
    542153.227246826,
    542648.3360907623,
    544273.9324713233,
    550000.0,
    549046.9473048343,
    548047.5904092644,
    547052.7023355255,
    546066.4155333972,
    520000.0,
    520977.906499162,
    521972.6466978357,
    522950.7862194081,
    523930.86647590954,
    524922.3632146807,
    525917.7966582401,
    522351.1343196252,
    527125.4548547225,
    526128.5235193562,
    525130.7081813341,
    524144.8882983563,
    530798.8710058356,
    529837.3189358161,
    528901.4390783398,
    539955.974375356,
    540509.7202600716,
    540554.4697958604,
    540124.5665895239,
    539457.7385073705,
    538609.0930945011,
    537650.6482494324,
    536662.2132646953,
    535670.9952466968,
    534702.7550180865,
    533831.4503084399,
    533061.3425394666,
    532157.7972663342,
    546233.0059536607,
    546906.7057886998,
    547857.5988940151,
    548844.1779325017,
    523381.48903560604,
    522468.0997263235,
    521503.9257619144,
    525280.9678179419,
    534272.9718145289,
    533289.726130804,
    532311.6002925185,
    531314.2350300908,
    530343.2573956609,
    529404.0267836585,
    528406.8877962828,
    527408.438540811,
    526416.4831948339,
    547423.9512601104,
    546818.8696391915,
    546751.8787156106,
    546564.3507434212,
    545942.2297277196,
    545512.8225309709,
    544971.65760404,
    544429.4936548981,
    543746.9451913793,
    543030.8690031096,
    542256.0397376382,
    541457.6565650662,
    540552.3950139998,
    539652.2355835018,
    538839.7378067289,
    537990.4340473542,
    537007.3752328141,
    536022.1700130996,
    537417.0468347219,
    538156.4071117863,
    539076.4825414991,
    539863.903717873,
    541996.2659877514,
    542624.1105710816,
    543421.4023585871,
    544400.4790130118,
    545389.5363446891,
    546360.828213827,
    547201.2564344314,
    548117.0745406685,
    524266.09546261415,
    525089.486889288,
    528242.3239959123,
    529217.7910440405,
    530038.4913963537,
    522423.15070457,
    521427.2884053698,
    522555.9904544321,
    529030.4932832464,
    528038.4263267842,
    527038.5688849417,
    526038.746341665,
    532993.0805511009,
    532003.260610268,
    531009.5137000929,
    530011.0692840518,
    545728.9288492671,
    545228.7527245631,
    544793.6165900896,
    544324.0658601674,
    543757.1054321213,
    543343.5919286992,
    542816.8120473484,
    541997.2329619491,
    541153.4587884084,
    540302.4609239949,
    539458.859568292,
    538484.4409203371,
    537531.6718874933,
    537267.7621267324,
    536473.9828749119,
    535486.8616121166,
    534501.006691062,
    540652.709349222,
    541131.4261559306,
    542310.5385527068,
    543285.3188949242,
    544247.7959926978,
    545089.9263310316,
    546087.6202142882,
    547077.7264906017,
    548075.363336421,
    549059.0245534881,
    524193.8328189907,
    527905.8476731717,
    526978.3198189315,
    525983.1425512254,
    522198.701020318,
    521204.9277157221,
    532860.9091847165,
    531863.5053478461,
    530864.0956853683,
    529864.8572444214,
    544737.4093717508,
    544569.3339346765,
    544327.771360188,
    543863.477011355,
    543275.0341084594,
    542715.4199777856,
    542093.6525093828,
    541285.7709317384,
    540449.5406616033,
    539551.5492211859,
    538610.5312377414,
    537640.4148034687,
    536770.6743346298,
    536686.2888205624,
    536054.8844726445,
    535061.7313647007,
    534070.1286353774,
    540991.614522323,
    542448.3669101102,
    543994.7063397208,
    544871.8490877205,
    545799.7056238599,
    546784.2744169813,
    547750.3419044648,
    548724.9651694401,
    520000.0,
    521826.3962308935,
    532600.3302651896,
    533589.8503320153,
    534561.7759873917,
    535449.8915779426,
    535993.0214361007,
    536819.0801195968,
    537807.793036829,
    538744.8306413345,
    539665.2325617559,
    540546.0055358304,
    541349.6936467969,
    541980.9483774537,
    542394.1133221515,
    542614.8288802537,
    543077.7762354391,
    542475.781801715,
    544083.8251957479,
    550000.0,
    549053.1439277239,
    548063.5281953025,
    547069.414890058,
    546087.5377901661,
    530271.9887491008,
    529298.9639373667,
    528340.209952825,
    527361.9910066663,
    526373.1685692647,
    525379.3992484873,
    524390.3065754601,
    523401.1210929096,
    522406.9032212814,
    521418.36474354746,
    520000.0,
    520978.6362345788,
    521950.47117176966,
    522933.3988561018,
    523928.4054918701,
    524884.9875455543,
    525874.1190162523,
    526848.2966478025,
    527791.3040361393,
    538922.8089907,
    539134.0778793262,
    538736.0836775531,
    538004.1174260699,
    537223.7768297916,
    536312.6322191659,
    535337.4942052161,
    534389.0342257542,
    533501.4998407771,
    532715.1008693202,
    531806.479694858,
    549037.6348608709,
]
Y = [
    7506463.058931708,
    7506576.346475119,
    7506700.1649271855,
    7506872.99333815,
    7507045.821749114,
    7507202.956357559,
    7507354.44495147,
    7507562.8122875495,
    7507744.951704663,
    7507892.96504778,
    7508104.092849379,
    7508311.6272335,
    7508501.976685256,
    7508504.460999884,
    7507684.484812225,
    7507085.842450439,
    7506588.057267194,
    7506144.839214603,
    7505654.042415291,
    7505177.516716159,
    7504892.17995596,
    7504686.687074701,
    7504603.854204072,
    7504503.098141689,
    7504341.941955539,
    7504177.672377438,
    7504002.8541911375,
    7503245.259859091,
    7503436.728206725,
    7503611.222655113,
    7503811.172638808,
    7503933.967406592,
    7504066.000381116,
    7504086.539427338,
    7504206.419975162,
    7504405.812567679,
    7504618.483028057,
    7504899.302718434,
    7505410.872728581,
    7505727.305923727,
    7506201.128560977,
    7506724.563539154,
    7507220.081115803,
    7507612.996219244,
    7508505.040226217,
    7509135.469929169,
    7510000.0,
    7509704.59876308,
    7509356.363295672,
    7509042.9372015195,
    7508868.599440474,
    7508713.626424883,
    7508491.404999385,
    7508254.175495736,
    7508044.257347997,
    7507813.564222933,
    7507689.029948036,
    7507479.73426719,
    7507242.659510909,
    7500803.4896762185,
    7510000.0,
    7509752.2424489865,
    7509557.913698296,
    7501034.634496278,
    7501611.124154025,
    7501788.582382207,
    7501832.072807334,
    7501874.753115009,
    7501854.802179787,
    7501993.852453728,
    7502194.603011602,
    7502736.52101189,
    7503605.470838049,
    7504102.310629672,
    7503995.998654295,
    7504162.548298732,
    7504522.438259361,
    7505048.271868659,
    7505609.690332235,
    7506352.909638542,
    7507272.811883032,
    7508259.448754307,
    7509066.146225987,
    7491662.855160554,
    7492637.949197558,
    7492365.294641344,
    7492343.195849396,
    7492425.974104291,
    7492294.881093971,
    7508845.720669018,
    7509050.10324461,
    7509000.757492466,
    7509194.807969997,
    7509393.409809908,
    7509493.696517873,
    7509580.778152611,
    7501916.1655494,
    7502377.774781606,
    7502376.385050711,
    7502389.230930838,
    7502221.572623042,
    7503475.2625857005,
    7503209.079556105,
    7502871.602548231,
    7510000.0,
    7509197.116898042,
    7508256.250238412,
    7507364.330453778,
    7506624.831329774,
    7506116.551366933,
    7505849.401377124,
    7505705.229350432,
    7505578.685761046,
    7505339.114590024,
    7504851.858361741,
    7504235.464249766,
    7503807.926211051,
    7490000.0,
    7490632.5934013035,
    7490931.629951538,
    7491082.825264233,
    7498414.247334903,
    7498022.163648057,
    7497778.683655275,
    7499008.503536563,
    7499007.862999017,
    7499166.809369003,
    7499342.672418874,
    7499404.581849788,
    7499607.193046319,
    7499409.181598686,
    7499342.398484021,
    7499289.594081205,
    7499167.5240981905,
    7510000.0,
    7509254.772409647,
    7508259.427108908,
    7507280.617024052,
    7506546.4275029935,
    7505645.278178703,
    7504804.447980701,
    7503964.813485547,
    7503240.48427183,
    7502543.425302152,
    7501911.2681202525,
    7501310.361086443,
    7500889.459500117,
    7500454.516031106,
    7499884.737649882,
    7499387.224113458,
    7499210.211525513,
    7499047.019637048,
    7490706.05210895,
    7491653.361597963,
    7492310.636501358,
    7492912.724049956,
    7493323.250750658,
    7494099.582786533,
    7494682.93982123,
    7494844.964517663,
    7494717.975402998,
    7494868.223497515,
    7495407.3541522715,
    7495771.872569879,
    7490000.0,
    7490560.701632605,
    7492064.722272471,
    7491904.289302209,
    7491338.4112052,
    7499760.717626198,
    7499723.154391495,
    7499746.544444192,
    7500200.705866932,
    7500238.3134670025,
    7500228.5316139795,
    7500222.315208985,
    7500834.640376496,
    7500780.822571908,
    7500675.103396037,
    7500715.487725784,
    7510000.0,
    7509145.739106326,
    7508252.316578914,
    7507374.160168251,
    7506552.029851974,
    7505642.489778755,
    7504802.59176734,
    7504230.642840398,
    7503695.190221784,
    7503174.1096064225,
    7502644.083926628,
    7502464.589954097,
    7502223.052264384,
    7501311.038411527,
    7500760.731710168,
    7500717.160069187,
    7500865.774257174,
    7490387.598747078,
    7491245.724857572,
    7491467.516614238,
    7492641.868488729,
    7492862.716044601,
    7493359.487961125,
    7493417.977361985,
    7493290.396821679,
    7493232.414779238,
    7493379.023244919,
    7500431.1064537205,
    7501050.467118945,
    7500734.366883766,
    7500642.3056855835,
    7500076.008830209,
    7500028.234047118,
    7501127.377606143,
    7501090.091638437,
    7501109.731843927,
    7501135.069368409,
    7510000.0,
    7509043.874690335,
    7508080.057040733,
    7507197.428797968,
    7506390.922978456,
    7505562.805840937,
    7504779.60394153,
    7504197.023577066,
    7503648.645117141,
    7503208.680546424,
    7502884.853275787,
    7503057.40301376,
    7502931.162530749,
    7501939.632101527,
    7501282.496918272,
    7501241.235538427,
    7501112.3494773,
    7490051.910415043,
    7491085.266716096,
    7492105.484638324,
    7492577.547240268,
    7492949.361650263,
    7493050.756214061,
    7492813.284106591,
    7492712.54139396,
    7500333.586472658,
    7510000.0,
    7501627.108348517,
    7501716.96918736,
    7501950.3146539405,
    7502385.375857322,
    7503192.199394774,
    7503547.6234244285,
    7503611.016851668,
    7503956.5283480855,
    7504342.417675098,
    7504815.020093556,
    7505406.605492606,
    7506158.648222525,
    7507069.200253583,
    7508037.442205912,
    7508903.818976895,
    7490825.772271102,
    7491864.689757697,
    7492916.182605702,
    7492601.675778644,
    7492543.692947827,
    7492650.179472104,
    7492504.2474401975,
    7504023.818165967,
    7503802.840013571,
    7503526.350863925,
    7503319.79261976,
    7503201.512659313,
    7503146.041926193,
    7503004.886707106,
    7502858.230922394,
    7502758.70803037,
    7502621.773975609,
    7507850.745601907,
    7508049.260711595,
    7508281.05024187,
    7508448.578437336,
    7508547.11325418,
    7508818.54401165,
    7508958.199252176,
    7509183.0594342025,
    7509510.386527048,
    7510000.0,
    7509039.540376887,
    7508150.610234313,
    7507475.846219083,
    7506854.803228106,
    7506480.417593711,
    7506259.626961826,
    7505948.643392906,
    7505495.005195306,
    7504880.9632680155,
    7504474.085528871,
    7490000.0,
]
Z = [
    533.0,
    533.0,
    540.0,
    540.0,
    549.0,
    550.0,
    556.0,
    558.0,
    568.0,
    571.0,
    582.0,
    586.0,
    593.0,
    598.0,
    591.0,
    582.0,
    569.0,
    565.0,
    563.0,
    556.0,
    554.0,
    549.0,
    543.0,
    537.0,
    535.0,
    531.0,
    525.0,
    518.0,
    528.0,
    536.0,
    538.0,
    531.0,
    539.0,
    544.0,
    546.0,
    551.0,
    556.0,
    562.0,
    571.0,
    574.0,
    575.0,
    581.0,
    586.0,
    601.0,
    609.0,
    619.0,
    609.0,
    606.0,
    597.0,
    588.0,
    581.0,
    574.0,
    569.0,
    563.0,
    561.0,
    556.0,
    553.0,
    542.0,
    539.0,
    640.0,
    577.0,
    601.0,
    584.0,
    629.0,
    561.0,
    562.0,
    600.0,
    658.0,
    660.0,
    767.0,
    778.0,
    724.0,
    772.0,
    739.0,
    791.0,
    812.0,
    675.0,
    649.0,
    690.0,
    670.0,
    682.0,
    761.0,
    804.0,
    629.0,
    582.0,
    596.0,
    585.0,
    611.0,
    634.0,
    650.0,
    594.0,
    558.0,
    607.0,
    578.0,
    599.0,
    594.0,
    551.0,
    555.0,
    558.0,
    553.0,
    561.0,
    590.0,
    578.0,
    560.0,
    667.0,
    656.0,
    649.0,
    638.0,
    631.0,
    623.0,
    623.0,
    615.0,
    608.0,
    607.0,
    602.0,
    602.0,
    606.0,
    643.0,
    617.0,
    594.0,
    572.0,
    504.0,
    484.0,
    497.0,
    503.0,
    507.0,
    521.0,
    521.0,
    520.0,
    512.0,
    506.0,
    502.0,
    502.0,
    504.0,
    598.0,
    575.0,
    580.0,
    564.0,
    567.0,
    555.0,
    540.0,
    539.0,
    552.0,
    537.0,
    527.0,
    532.0,
    547.0,
    534.0,
    521.0,
    496.0,
    516.0,
    503.0,
    510.0,
    503.0,
    532.0,
    529.0,
    566.0,
    528.0,
    555.0,
    558.0,
    533.0,
    535.0,
    547.0,
    538.0,
    600.0,
    549.0,
    465.0,
    533.0,
    521.0,
    536.0,
    561.0,
    559.0,
    583.0,
    560.0,
    533.0,
    556.0,
    544.0,
    561.0,
    560.0,
    543.0,
    691.0,
    719.0,
    667.0,
    642.0,
    621.0,
    640.0,
    625.0,
    598.0,
    614.0,
    585.0,
    596.0,
    636.0,
    634.0,
    571.0,
    561.0,
    534.0,
    565.0,
    552.0,
    573.0,
    585.0,
    546.0,
    547.0,
    555.0,
    540.0,
    543.0,
    544.0,
    551.0,
    545.0,
    546.0,
    544.0,
    555.0,
    546.0,
    529.0,
    541.0,
    569.0,
    553.0,
    561.0,
    681.0,
    646.0,
    639.0,
    619.0,
    612.0,
    602.0,
    600.0,
    591.0,
    579.0,
    580.0,
    602.0,
    593.0,
    585.0,
    570.0,
    552.0,
    554.0,
    560.0,
    636.0,
    602.0,
    600.0,
    608.0,
    591.0,
    585.0,
    573.0,
    571.0,
    581.0,
    564.0,
    625.0,
    650.0,
    664.0,
    608.0,
    635.0,
    644.0,
    659.0,
    812.0,
    616.0,
    611.0,
    622.0,
    629.0,
    677.0,
    641.0,
    647.0,
    602.0,
    600.0,
    603.0,
    582.0,
    573.0,
    591.0,
    634.0,
    573.0,
    566.0,
    554.0,
    549.0,
    545.0,
    549.0,
    550.0,
    550.0,
    543.0,
    532.0,
    545.0,
    546.0,
    544.0,
    549.0,
    551.0,
    568.0,
    577.0,
    580.0,
    584.0,
    629.0,
    628.0,
    617.0,
    609.0,
    600.0,
    594.0,
    591.0,
    594.0,
    592.0,
    596.0,
    592.0,
    579.0,
]
featureid = [
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '0',
    '1',
    '1',
    '1',
    '2',
    '3',
    '3',
    '3',
    '3',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '6',
    '7',
    '7',
    '7',
    '7',
    '7',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '2',
    '3',
    '3',
    '3',
    '3',
    '4',
    '4',
    '4',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '6',
    '6',
    '6',
    '6',
    '1',
    '1',
    '1',
    '2',
    '3',
    '3',
    '3',
    '3',
    '3',
    '3',
    '3',
    '3',
    '3',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '6',
    '7',
    '8',
    '8',
    '9',
    '9',
    '9',
    '9',
    '9',
    '9',
    '9',
    '9',
    '0',
    '0',
    '2',
    '2',
    '2',
    '0',
    '0',
    '1',
    '2',
    '2',
    '2',
    '2',
    '3',
    '3',
    '3',
    '3',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '4',
    '5',
    '5',
    '6',
    '7',
    '7',
    '7',
    '7',
    '7',
    '7',
    '7',
    '0',
    '1',
    '1',
    '1',
    '0',
    '0',
    '1',
    '1',
    '1',
    '1',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '3',
    '4',
    '5',
    '5',
    '5',
    '5',
    '5',
    '5',
    '0',
    '1',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '3',
    '4',
    '5',
    '5',
    '5',
    '5',
    '5',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '0',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '1',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '2',
    '3',
]

s_c = pandas.DataFrame({'X': X, 'Y': Y, 'Z': Z, 'featureId': featureid})


############################
### TEST StructuralPoint ###
############################

geology = load_hamersley_geology()
geology.rename(columns={'unitname': 'UNITNAME', 'code': 'CODE'}, inplace=True)


def check_thickness_values(result, column, description):
    for order, position in [
        (max(st_units['stratigraphic_Order']), 'bottom'),
        (min(st_units['stratigraphic_Order']), 'top'),
    ]:
        assert (
            result[result['stratigraphic_Order'] == order][column].values == -1
        ), f"StructuralPoint: {position} unit not assigned as -1 ({description})"


def test_calculate_thickness_structural_point():
    # Run the calculation
    thickness_calculator = StructuralPoint()

    md = MapData()
    md.sampled_contacts = s_c
    md.sampled_contacts = s_c
    md.raw_data[Datatype.GEOLOGY] = geology
    md.load_map_data(Datatype.GEOLOGY)
    md.check_map(Datatype.GEOLOGY)
    md.parse_geology_map()

    result = thickness_calculator.compute(
        units=st_units,
        stratigraphic_order=st_column,
        basal_contacts=bc_gdf,
        structure_data=structures,
        map_data=md,
    )

    # is thickness calc alpha the label?
    assert (
        thickness_calculator.thickness_calculator_label == 'StructuralPoint'
    ), 'StructuralPoint: thickness calculator name not set correctly'

    # is the result a pandas dataframe?
    assert isinstance(result, pandas.DataFrame), 'StructuralPoint result not a pandas DataFrame'

    # Check if there is mean, std, and median in results
    required_columns = ['ThicknessMean', 'ThicknessMedian', 'ThicknessStdDev']
    for column in required_columns:
        assert column in result.columns, f'{column} not in StructuralPoint result'

    # check if all units are in the results
    assert 'name' in result.columns, 'unit_name not in StructuralPoint result'
    assert all(
        name in result['name'].values for name in st_units['name'].values
    ), 'units missing from in StructuralPoint result'

    # are bottom and top units being assigned -1
    for column, description in [
        ('ThicknessMean', 'mean'),
        ('ThicknessMedian', 'median'),
        ('ThicknessStdDev', 'std dev'),
    ]:
        check_thickness_values(result, column, description)

    # are the dtypes numpy.float?
    for column in required_columns:
        assert (
            result[column].dtype == numpy.float64
        ), f'StructuralPoint: result column {column} not numpy.float64'

    # check for nans in the results
    for column in required_columns:
        assert (
            not result[column].isnull().values.any()
        ), f'StructuralPoint: result column {column} has NaN values'
