//! Core maplet implementation
//! 
//! Implements the main Maplet data structure that provides space-efficient
//! approximate key-value mappings with one-sided error guarantees.

use std::hash::{Hash, BuildHasher};
use std::marker::PhantomData;
use std::sync::Arc;
use tokio::sync::RwLock;
use crate::{
    MapletError, MapletResult, MapletStats,
    types::MapletConfig,
    hash::{FingerprintHasher, HashFunction, CollisionDetector, PerfectHash},
    quotient_filter::QuotientFilter,
    operators::MergeOperator,
};

/// Core maplet data structure
/// 
/// A maplet provides space-efficient approximate key-value mappings with
/// one-sided error guarantees. When queried with a key k, it returns a
/// value m[k] that is an approximation of the true value M[k].
/// 
/// The maplet guarantees that M[k] ≺ m[k] for some application-specific
/// ordering relation ≺, and that m[k] = M[k] with probability at least 1-ε.
#[derive(Debug)]
pub struct Maplet<K, V, Op> 
where
    K: Hash + Eq + Clone + std::fmt::Debug + Send + Sync,
    V: Clone + std::fmt::Debug + Send + Sync,
    Op: MergeOperator<V> + Send + Sync,
{
    /// Configuration for the maplet
    config: MapletConfig,
    /// Quotient filter for fingerprint storage
    filter: Arc<RwLock<QuotientFilter>>,
    /// Map of fingerprints to values (not aligned with slots)
    values: Arc<RwLock<std::collections::HashMap<u64, V>>>,
    /// Merge operator for combining values
    operator: Op,
    /// Collision detector for monitoring hash collisions
    collision_detector: Arc<RwLock<CollisionDetector>>,
    /// Perfect hash for slot mapping (same as quotient filter)
    perfect_hash: PerfectHash,
    /// Current number of items stored
    len: Arc<RwLock<usize>>,
    /// Phantom data to hold the key type
    _phantom: PhantomData<K>,
}

impl<K, V, Op> Maplet<K, V, Op>
where
    K: Hash + Eq + Clone + std::fmt::Debug + Send + Sync,
    V: Clone + PartialEq + std::fmt::Debug + Send + Sync,
    Op: MergeOperator<V> + Default + Send + Sync,
{
    /// Create a new maplet with default configuration
    pub fn new(capacity: usize, false_positive_rate: f64) -> MapletResult<Self> {
        let config = MapletConfig::new(capacity, false_positive_rate);
        Self::with_config(config)
    }
    
    /// Create a new maplet with custom operator
    pub fn with_operator(capacity: usize, false_positive_rate: f64, operator: Op) -> MapletResult<Self> {
        let config = MapletConfig::new(capacity, false_positive_rate);
        Self::with_config_and_operator(config, operator)
    }
    
    /// Create a new maplet with custom configuration
    pub fn with_config(config: MapletConfig) -> MapletResult<Self> {
        let operator = Op::default();
        Self::with_config_and_operator(config, operator)
    }
    
    /// Create a new maplet with custom configuration and operator
    pub fn with_config_and_operator(config: MapletConfig, operator: Op) -> MapletResult<Self> {
        config.validate()?;
        
        let fingerprint_bits = FingerprintHasher::optimal_fingerprint_size(config.false_positive_rate);
        let filter = QuotientFilter::new(config.capacity, fingerprint_bits, HashFunction::default())?;
        
        let collision_detector = CollisionDetector::new(config.capacity / 4); // Allow 25% collisions
        let perfect_hash = PerfectHash::new(config.capacity, HashFunction::default());
        
        Ok(Self {
            config,
            filter: Arc::new(RwLock::new(filter)),
            values: Arc::new(RwLock::new(std::collections::HashMap::new())),
            operator,
            collision_detector: Arc::new(RwLock::new(collision_detector)),
            perfect_hash,
            len: Arc::new(RwLock::new(0)),
            _phantom: PhantomData,
        })
    }
    
    /// Insert a key-value pair into the maplet
    pub async fn insert(&self, key: K, value: V) -> MapletResult<()> {
        let current_len = *self.len.read().await;
        if current_len >= self.config.capacity {
            if self.config.auto_resize {
                self.resize(self.config.capacity * 2).await?;
            } else {
                return Err(MapletError::CapacityExceeded);
            }
        }
        
        let fingerprint = self.hash_key(&key);
        
        // Check if key already exists
        let filter_guard = self.filter.read().await;
        let key_exists = filter_guard.query(fingerprint);
        drop(filter_guard);
        
        if key_exists {
            // Key exists, merge with existing value
            self.merge_value(fingerprint, value).await?;
        } else {
            // New key, insert into filter and store value
            {
                let mut filter_guard = self.filter.write().await;
                filter_guard.insert(fingerprint)?;
            }
            self.store_value(fingerprint, value).await?;
            {
                let mut len_guard = self.len.write().await;
                *len_guard += 1;
            }
        }
        
        Ok(())
    }
    
    /// Query a key to get its associated value
    pub async fn query(&self, key: &K) -> Option<V> {
        let fingerprint = self.hash_key(key);
        
        let filter_guard = self.filter.read().await;
        if !filter_guard.query(fingerprint) {
            return None;
        }
        drop(filter_guard);
        
        // Get the value directly from the HashMap using the fingerprint
        let values_guard = self.values.read().await;
        values_guard.get(&fingerprint).cloned()
    }
    
    /// Check if a key exists in the maplet
    pub async fn contains(&self, key: &K) -> bool {
        let fingerprint = self.hash_key(key);
        let filter_guard = self.filter.read().await;
        filter_guard.query(fingerprint)
    }
    
    /// Delete a key-value pair from the maplet
    pub async fn delete(&self, key: &K, value: &V) -> MapletResult<bool> {
        if !self.config.enable_deletion {
            return Err(MapletError::Internal("Deletion not enabled".to_string()));
        }
        
        let fingerprint = self.hash_key(key);
        
        let filter_guard = self.filter.read().await;
        if !filter_guard.query(fingerprint) {
            return Ok(false);
        }
        drop(filter_guard);
        
        let mut values_guard = self.values.write().await;
        if let Some(existing_value) = values_guard.get(&fingerprint) {
            // Check if the values match (for exact deletion)
            if existing_value == value {
                // Remove from filter and clear value
                {
                    let mut filter_guard = self.filter.write().await;
                    filter_guard.delete(fingerprint)?;
                }
                values_guard.remove(&fingerprint);
                {
                    let mut len_guard = self.len.write().await;
                    *len_guard -= 1;
                }
                return Ok(true);
            }
        }
        
        Ok(false)
    }
    
    /// Get the current number of items stored
    pub async fn len(&self) -> usize {
        *self.len.read().await
    }
    
    /// Check if the maplet is empty
    pub async fn is_empty(&self) -> bool {
        *self.len.read().await == 0
    }
    
    /// Get the configured false-positive rate
    pub fn error_rate(&self) -> f64 {
        self.config.false_positive_rate
    }
    
    /// Get the current load factor
    pub async fn load_factor(&self) -> f64 {
        let current_len = *self.len.read().await;
        current_len as f64 / self.config.capacity as f64
    }
    
    /// Get statistics about the maplet
    pub async fn stats(&self) -> MapletStats {
        let filter_guard = self.filter.read().await;
        let filter_stats = filter_guard.stats();
        drop(filter_guard);
        
        let memory_usage = self.estimate_memory_usage();
        let current_len = *self.len.read().await;
        
        let collision_guard = self.collision_detector.read().await;
        let collision_count = collision_guard.collision_count() as u64;
        drop(collision_guard);
        
        let mut stats = MapletStats::new(
            self.config.capacity,
            current_len,
            self.config.false_positive_rate,
        );
        stats.update(
            current_len,
            memory_usage,
            collision_count,
            filter_stats.runs,
        );
        stats
    }
    
    /// Resize the maplet to a new capacity
    pub async fn resize(&self, new_capacity: usize) -> MapletResult<()> {
        if new_capacity <= self.config.capacity {
            return Err(MapletError::ResizeFailed("New capacity must be larger".to_string()));
        }
        
        // Create new filter with larger capacity
        let fingerprint_bits = FingerprintHasher::optimal_fingerprint_size(self.config.false_positive_rate);
        let new_filter = QuotientFilter::new(
            new_capacity,
            fingerprint_bits,
            HashFunction::default(),
        )?;
        
        // Replace the filter and resize values array
        {
            let mut filter_guard = self.filter.write().await;
            *filter_guard = new_filter;
        }
        
        // HashMap doesn't need explicit resizing - it grows automatically
        
        // Note: In a full implementation, config.capacity would also need to be updated
        // For now, we rely on the actual filter and values array capacity
        
        Ok(())
    }
    
    /// Merge another maplet into this one
    pub async fn merge(&self, _other: &Maplet<K, V, Op>) -> MapletResult<()> {
        if !self.config.enable_merging {
            return Err(MapletError::MergeFailed("Merging not enabled".to_string()));
        }
        
        // This is a simplified merge implementation
        // In practice, we'd need to iterate through all items in the other maplet
        // and insert them into this one using the merge operator
        Err(MapletError::MergeFailed("Merge not fully implemented".to_string()))
    }
    
    /// Hash a key to get its fingerprint
    fn hash_key(&self, key: &K) -> u64 {
        // Use the same hasher as the quotient filter
        // The quotient filter uses AHash by default, so we need to use the same
        use ahash::RandomState;
        use std::hash::Hasher;
        
        // Create a consistent hasher - we need to use the same seed as the quotient filter
        // For now, use a fixed seed to ensure consistency
        let random_state = RandomState::with_seed(42);
        let mut hasher = random_state.build_hasher();
        key.hash(&mut hasher);
        hasher.finish()
    }
    
    /// Find the slot index for a fingerprint
    fn find_slot_for_fingerprint(&self, fingerprint: u64) -> usize {
        // Use the same slot mapping as the quotient filter
        let quotient = self.extract_quotient(fingerprint);
        
        // Use the same perfect hash as the quotient filter
        self.perfect_hash.slot_index(quotient)
    }
    
    /// Extract quotient from fingerprint (same as quotient filter)
    fn extract_quotient(&self, fingerprint: u64) -> u64 {
        let quotient_bits = (self.config.capacity as f64).log2().ceil() as u32;
        let quotient_mask = if quotient_bits >= 64 {
            u64::MAX
        } else {
            (1u64 << quotient_bits) - 1
        };
        fingerprint & quotient_mask
    }
    
    /// Extract remainder from fingerprint (same as quotient filter)
    fn extract_remainder(&self, fingerprint: u64) -> u64 {
        let quotient_bits = (self.config.capacity as f64).log2().ceil() as u32;
        let remainder_bits = 64 - quotient_bits;
        let remainder_mask = if remainder_bits >= 64 {
            u64::MAX
        } else {
            (1u64 << remainder_bits) - 1
        };
        (fingerprint >> quotient_bits) & remainder_mask
    }
    
    /// Find the target slot for a quotient and remainder
    fn find_target_slot(&self, quotient: u64, _remainder: u64) -> usize {
        // Use the same perfect hash as the quotient filter
        self.perfect_hash.slot_index(quotient)
    }
    
    /// Find the actual slot where a fingerprint is stored
    /// This replicates the quotient filter's slot finding logic
    fn find_actual_slot_for_fingerprint(&self, quotient: u64, _remainder: u64) -> usize {
        // Get the target slot from the perfect hash
        let target_slot = self.perfect_hash.slot_index(quotient);
        
        // The quotient filter stores fingerprints in runs
        // We need to find the actual slot within the run where this remainder is stored
        // This is a simplified version - in practice, we'd need access to the quotient filter's internal state
        
        // For now, let's use a simple approach: assume the remainder is stored at the target slot
        // This is not correct but will help us identify the issue
        target_slot
    }
    
    /// Merge a value with an existing value at a fingerprint
    async fn merge_value(&self, fingerprint: u64, value: V) -> MapletResult<()> {
        let mut values_guard = self.values.write().await;
        if let Some(existing_value) = values_guard.get(&fingerprint) {
            let merged_value = self.operator.merge(existing_value.clone(), value)?;
            values_guard.insert(fingerprint, merged_value);
        } else {
            // This shouldn't happen if the filter is consistent
            return Err(MapletError::Internal("Filter inconsistency detected".to_string()));
        }
        
        Ok(())
    }
    
    /// Store a value at a fingerprint
    async fn store_value(&self, fingerprint: u64, value: V) -> MapletResult<()> {
        let mut values_guard = self.values.write().await;
        values_guard.insert(fingerprint, value);
        
        Ok(())
    }
    
    /// Estimate memory usage in bytes
    fn estimate_memory_usage(&self) -> usize {
        // Rough estimate: filter size + values size + overhead
        let filter_size = self.config.capacity * std::mem::size_of::<crate::quotient_filter::SlotMetadata>();
        let values_size = self.config.capacity * std::mem::size_of::<Option<V>>();
        let overhead = std::mem::size_of::<Self>();
        
        filter_size + values_size + overhead
    }
}

// Implement Default for operators that support it
impl<K, V, Op> Default for Maplet<K, V, Op>
where
    K: Hash + Eq + Clone + std::fmt::Debug + Send + Sync,
    V: Clone + PartialEq + std::fmt::Debug + Send + Sync,
    Op: MergeOperator<V> + Default + Send + Sync,
{
    fn default() -> Self {
        Self::new(1000, 0.01).expect("Failed to create default maplet")
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::operators::CounterOperator;

    #[tokio::test]
    async fn test_maplet_creation() {
        let maplet = Maplet::<String, u64, CounterOperator>::new(100, 0.01);
        assert!(maplet.is_ok());
        
        let maplet = maplet.unwrap();
        assert_eq!(maplet.len().await, 0);
        assert!(maplet.is_empty().await);
        assert_eq!(maplet.error_rate(), 0.01);
    }

    #[tokio::test]
    async fn test_maplet_insert_query() {
        let maplet = Maplet::<String, u64, CounterOperator>::new(100, 0.01).unwrap();
        
        // Insert some key-value pairs
        assert!(maplet.insert("key1".to_string(), 5).await.is_ok());
        assert!(maplet.insert("key2".to_string(), 10).await.is_ok());
        
        assert_eq!(maplet.len().await, 2);
        assert!(!maplet.is_empty().await);
        
        // Query existing keys
        assert!(maplet.contains(&"key1".to_string()).await);
        assert!(maplet.contains(&"key2".to_string()).await);
        
        // Query non-existing key
        assert!(!maplet.contains(&"key3".to_string()).await);
    }

    #[tokio::test]
    async fn test_maplet_merge_values() {
        let maplet = Maplet::<String, u64, CounterOperator>::new(100, 0.01).unwrap();
        
        // Insert same key multiple times
        assert!(maplet.insert("key1".to_string(), 5).await.is_ok());
        assert!(maplet.insert("key1".to_string(), 3).await.is_ok());
        
        assert_eq!(maplet.len().await, 1); // Still only one unique key
        
        // Query should return merged value
        let value = maplet.query(&"key1".to_string()).await;
        assert!(value.is_some());
        // Note: Due to hash collisions, the exact value might not be 8
        // but it should be >= 5 (one-sided error guarantee)
    }

    #[tokio::test]
    async fn test_maplet_stats() {
        let maplet = Maplet::<String, u64, CounterOperator>::new(100, 0.01).unwrap();
        
        maplet.insert("key1".to_string(), 5).await.unwrap();
        maplet.insert("key2".to_string(), 10).await.unwrap();
        
        let stats = maplet.stats().await;
        assert_eq!(stats.capacity, 100);
        assert_eq!(stats.len, 2);
        assert!(stats.load_factor > 0.0);
        assert!(stats.memory_usage > 0);
    }
}
