"""Tests for group theory implementations"""

from mathcode.abstract_algebra import CyclicGroup, SymmetricGroup, Permutation


class TestCyclicGroup:
    """Tests for CyclicGroup"""

    def test_initialization(self):
        """Test basic initialization"""
        G = CyclicGroup(6)
        assert G.n == 6
        assert G.order() == 6

    def test_elements(self):
        """Test elements generation"""
        G = CyclicGroup(5)
        assert G.elements() == [0, 1, 2, 3, 4]

    def test_operation(self):
        """Test group operation (addition mod n)"""
        G = CyclicGroup(7)
        assert G.op(3, 5) == 1  # 3 + 5 = 8 = 1 mod 7
        assert G.op(6, 2) == 1  # 6 + 2 = 8 = 1 mod 7

    def test_identity(self):
        """Test identity element"""
        G = CyclicGroup(10)
        assert G.identity() == 0

        # Identity property
        for a in G.elements():
            assert G.op(a, G.identity()) == a
            assert G.op(G.identity(), a) == a

    def test_inverse(self):
        """Test inverse elements"""
        G = CyclicGroup(8)

        assert G.inverse(0) == 0
        assert G.inverse(3) == 5  # 3 + 5 = 8 = 0 mod 8
        assert G.inverse(7) == 1  # 7 + 1 = 8 = 0 mod 8

        # Inverse property
        for a in G.elements():
            assert G.op(a, G.inverse(a)) == G.identity()

    def test_is_abelian(self):
        """Test that cyclic groups are abelian"""
        G = CyclicGroup(6)
        assert G.is_abelian()

    def test_generators(self):
        """Test generator finding"""
        G = CyclicGroup(12)
        gens = G.all_generators()

        # Generators are elements coprime to n
        assert 1 in gens
        assert 5 in gens
        assert 7 in gens
        assert 11 in gens

        # Non-generators
        assert 2 not in gens
        assert 3 not in gens
        assert 4 not in gens

    def test_element_order(self):
        """Test order of elements"""
        G = CyclicGroup(6)

        assert G.element_order(0) == 1
        assert G.element_order(1) == 6
        assert G.element_order(2) == 3
        assert G.element_order(3) == 2

    def test_subgroup_generation(self):
        """Test subgroup generation"""
        G = CyclicGroup(12)

        # Subgroup generated by 3
        H = G.generate_subgroup(3)
        assert set(H) == {0, 3, 6, 9}

        # Subgroup generated by 4
        K = G.generate_subgroup(4)
        assert set(K) == {0, 4, 8}


class TestPermutation:
    """Tests for Permutation class"""

    def test_initialization(self):
        """Test permutation initialization"""
        p = Permutation([1, 2, 0])
        assert p.n == 3
        assert p.mapping == [1, 2, 0]

    def test_initialization_dict(self):
        """Test initialization with dictionary"""
        p = Permutation({0: 1, 1: 2, 2: 0})
        assert p.mapping == [1, 2, 0]

    def test_application(self):
        """Test applying permutation"""
        p = Permutation([2, 0, 1])
        assert p(0) == 2
        assert p(1) == 0
        assert p(2) == 1

    def test_composition(self):
        """Test permutation composition"""
        p = Permutation([1, 2, 0])  # (0 1 2)
        q = Permutation([1, 0, 2])  # (0 1)

        # (0 1 2) * (0 1) = (0 2)
        r = p * q
        assert r(0) == 2
        assert r(1) == 1
        assert r(2) == 0

    def test_inverse(self):
        """Test permutation inverse"""
        p = Permutation([2, 0, 1])
        p_inv = p.inverse()

        # Check p * p^-1 = identity
        identity = p * p_inv
        assert identity.mapping == [0, 1, 2]

    def test_cycle_decomposition(self):
        """Test cycle decomposition"""
        p = Permutation([1, 2, 0])  # (0 1 2)
        cycles = p.cycle_decomposition()
        assert len(cycles) == 1
        assert (0, 1, 2) in cycles or (1, 2, 0) in cycles or (2, 0, 1) in cycles

        # Permutation with two cycles
        p2 = Permutation([1, 0, 3, 2])  # (0 1)(2 3)
        cycles2 = p2.cycle_decomposition()
        assert len(cycles2) == 2

        # Identity has no cycles
        identity = Permutation([0, 1, 2])
        assert identity.cycle_decomposition() == []

    def test_order(self):
        """Test permutation order"""
        p1 = Permutation([1, 2, 0])  # (0 1 2) - order 3
        assert p1.order() == 3

        p2 = Permutation([1, 0, 2])  # (0 1) - order 2
        assert p2.order() == 2

        identity = Permutation([0, 1, 2, 3])
        assert identity.order() == 1

    def test_sign(self):
        """Test permutation sign"""
        # Even permutations
        p1 = Permutation([1, 0, 2])  # (0 1) - odd
        assert p1.sign() == -1
        assert p1.is_odd()

        p2 = Permutation([1, 2, 0])  # (0 1 2) - even
        assert p2.sign() == 1
        assert p2.is_even()

        # Identity is even
        identity = Permutation([0, 1, 2])
        assert identity.sign() == 1

    def test_equality(self):
        """Test permutation equality"""
        p1 = Permutation([1, 2, 0])
        p2 = Permutation([1, 2, 0])
        p3 = Permutation([2, 1, 0])

        assert p1 == p2
        assert p1 != p3


class TestSymmetricGroup:
    """Tests for SymmetricGroup"""

    def test_initialization(self):
        """Test basic initialization"""
        S3 = SymmetricGroup(3)
        assert S3.n == 3
        assert S3.order() == 6  # 3! = 6

    def test_order(self):
        """Test group order"""
        S2 = SymmetricGroup(2)
        assert S2.order() == 2

        S4 = SymmetricGroup(4)
        assert S4.order() == 24  # 4! = 24

    def test_elements(self):
        """Test element generation"""
        S2 = SymmetricGroup(2)
        elems = S2.elements()
        assert len(elems) == 2

        S3 = SymmetricGroup(3)
        elems3 = S3.elements()
        assert len(elems3) == 6

    def test_identity(self):
        """Test identity element"""
        S3 = SymmetricGroup(3)
        e = S3.identity()
        assert e.mapping == [0, 1, 2]

    def test_operation(self):
        """Test group operation"""
        S3 = SymmetricGroup(3)
        p = Permutation([1, 2, 0])
        q = Permutation([1, 0, 2])

        result = S3.op(p, q)
        assert isinstance(result, Permutation)

    def test_inverse(self):
        """Test inverse computation"""
        S3 = SymmetricGroup(3)
        p = Permutation([2, 0, 1])
        p_inv = S3.inverse(p)

        # Check inverse property
        product = S3.op(p, p_inv)
        assert product == S3.identity()

    def test_not_abelian(self):
        """Test that S_n is not abelian for n >= 3"""
        S3 = SymmetricGroup(3)
        assert not S3.is_abelian()

    def test_s2_abelian(self):
        """Test that S_2 is abelian"""
        S2 = SymmetricGroup(2)
        assert S2.is_abelian()

    def test_alternating_group(self):
        """Test alternating group (even permutations)"""
        S3 = SymmetricGroup(3)
        A3 = S3.alternating_group()

        # A_3 has 3 elements (half of 6)
        assert len(A3) == 3

        # All elements should be even
        for p in A3:
            assert p.is_even()

    def test_transpositions(self):
        """Test transposition generation"""
        S3 = SymmetricGroup(3)
        trans = S3.transpositions()

        # S_3 has 3 transpositions: (0 1), (0 2), (1 2)
        assert len(trans) == 3

        # All should be odd permutations
        for t in trans:
            assert t.is_odd()


class TestGroupProperties:
    """Test general group properties"""

    def test_associativity(self):
        """Test associativity of group operation"""
        G = CyclicGroup(4)

        # Check associativity for all triples
        for a in range(4):
            for b in range(4):
                for c in range(4):
                    left = G.op(G.op(a, b), c)
                    right = G.op(a, G.op(b, c))
                    assert left == right

    def test_cayley_table(self):
        """Test Cayley table generation"""
        G = CyclicGroup(3)
        table = G.cayley_table()

        # Should be 3x3
        assert len(table) == 3
        assert all(len(row) == 3 for row in table)

        # Check specific entries
        assert table[0][0] == 0  # 0 + 0 = 0
        assert table[1][2] == 0  # 1 + 2 = 3 = 0 mod 3
        assert table[2][1] == 0  # 2 + 1 = 3 = 0 mod 3
