# Python: 3.8.2 (tags/v3.8.2:7b3ab59, Feb 25 2020, 23:03:10) [MSC v.1916 64 bit (AMD64)]
# Library: scipy, version: 1.6.2
# Module: scipy.spatial.transform.rotation, version: unspecified
import typing
import builtins as _mod_builtins

class Rotation(_mod_builtins.object):
    "Rotation in 3 dimensions.\n\n    This class provides an interface to initialize from and represent rotations\n    with:\n\n    - Quaternions\n    - Rotation Matrices\n    - Rotation Vectors\n    - Modified Rodrigues Parameters\n    - Euler Angles\n\n    The following operations on rotations are supported:\n\n    - Application on vectors\n    - Rotation Composition\n    - Rotation Inversion\n    - Rotation Indexing\n\n    Indexing within a rotation is supported since multiple rotation transforms\n    can be stored within a single `Rotation` instance.\n\n    To create `Rotation` objects use ``from_...`` methods (see examples below).\n    ``Rotation(...)`` is not supposed to be instantiated directly.\n\n    Attributes\n    ----------\n    single\n\n    Methods\n    -------\n    __len__\n    from_quat\n    from_matrix\n    from_rotvec\n    from_mrp\n    from_euler\n    as_quat\n    as_matrix\n    as_rotvec\n    as_mrp\n    as_euler\n    apply\n    __mul__\n    inv\n    magnitude\n    mean\n    reduce\n    create_group\n    __getitem__\n    identity\n    random\n    align_vectors\n\n    See Also\n    --------\n    Slerp\n\n    Notes\n    -----\n    .. versionadded: 1.2.0\n\n    Examples\n    --------\n    >>> from scipy.spatial.transform import Rotation as R\n\n    A `Rotation` instance can be initialized in any of the above formats and\n    converted to any of the others. The underlying object is independent of the\n    representation used for initialization.\n\n    Consider a counter-clockwise rotation of 90 degrees about the z-axis. This\n    corresponds to the following quaternion (in scalar-last format):\n\n    >>> r = R.from_quat([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])\n\n    The rotation can be expressed in any of the other formats:\n\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n    [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n    [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The same rotation can be initialized using a rotation matrix:\n\n    >>> r = R.from_matrix([[0, -1, 0],\n    ...                    [1, 0, 0],\n    ...                    [0, 0, 1]])\n\n    Representation in other formats:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The rotation vector corresponding to this rotation is given by:\n\n    >>> r = R.from_rotvec(np.pi/2 * np.array([0, 0, 1]))\n\n    Representation in other formats:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The ``from_euler`` method is quite flexible in the range of input formats\n    it supports. Here we initialize a single rotation about a single axis:\n\n    >>> r = R.from_euler('z', 90, degrees=True)\n\n    Again, the object is representation independent and can be converted to any\n    other format:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n\n    It is also possible to initialize multiple rotations in a single instance\n    using any of the `from_...` functions. Here we initialize a stack of 3\n    rotations using the ``from_euler`` method:\n\n    >>> r = R.from_euler('zyx', [\n    ... [90, 0, 0],\n    ... [0, 45, 0],\n    ... [45, 60, 30]], degrees=True)\n\n    The other representations also now return a stack of 3 rotations. For\n    example:\n\n    >>> r.as_quat()\n    array([[0.        , 0.        , 0.70710678, 0.70710678],\n           [0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n\n    Applying the above rotations onto a vector:\n\n    >>> v = [1, 2, 3]\n    >>> r.apply(v)\n    array([[-2.        ,  1.        ,  3.        ],\n           [ 2.82842712,  2.        ,  1.41421356],\n           [ 2.24452282,  0.78093109,  2.89002836]])\n\n    A `Rotation` instance can be indexed and sliced as if it were a single\n    1D array or list:\n\n    >>> r.as_quat()\n    array([[0.        , 0.        , 0.70710678, 0.70710678],\n           [0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n    >>> p = r[0]\n    >>> p.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> q = r[1:3]\n    >>> q.as_quat()\n    array([[0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n\n    In fact it can be converted to numpy.array:\n\n    >>> r_array = np.asarray(r)\n    >>> r_array.shape\n    (3,)\n    >>> r_array[0].as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n\n    Multiple rotations can be composed using the ``*`` operator:\n\n    >>> r1 = R.from_euler('z', 90, degrees=True)\n    >>> r2 = R.from_rotvec([np.pi/4, 0, 0])\n    >>> v = [1, 2, 3]\n    >>> r2.apply(r1.apply(v))\n    array([-2.        , -1.41421356,  2.82842712])\n    >>> r3 = r2 * r1 # Note the order\n    >>> r3.apply(v)\n    array([-2.        , -1.41421356,  2.82842712])\n\n    Finally, it is also possible to invert rotations:\n\n    >>> r1 = R.from_euler('z', [90, 45], degrees=True)\n    >>> r2 = r1.inv()\n    >>> r2.as_euler('zyx', degrees=True)\n    array([[-90.,   0.,   0.],\n           [-45.,   0.,   0.]])\n\n    These examples serve as an overview into the `Rotation` class and highlight\n    major functionalities. For more thorough examples of the range of input and\n    output formats supported, consult the individual method's examples.\n\n    "
    def __getitem__(self, index: int) -> typing.Any:
        'Extract rotation(s) at given index(es) from object.\n\n        Create a new `Rotation` instance containing a subset of rotations\n        stored in this object.\n\n        Parameters\n        ----------\n        indexer : index, slice, or index array\n            Specifies which rotation(s) to extract. A single indexer must be\n            specified, i.e. as if indexing a 1 dimensional array or list.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Contains\n                - a single rotation, if `indexer` is a single index\n                - a stack of rotation(s), if `indexer` is a slice, or and index\n                  array.\n\n        Raises\n        ------\n        TypeError if the instance was created as a single rotation.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n        >>> r = R.from_quat([\n        ... [1, 1, 0, 0],\n        ... [0, 1, 0, 1],\n        ... [1, 1, -1, 0]])\n        >>> r.as_quat()\n        array([[ 0.70710678,  0.70710678,  0.        ,  0.        ],\n               [ 0.        ,  0.70710678,  0.        ,  0.70710678],\n               [ 0.57735027,  0.57735027, -0.57735027,  0.        ]])\n\n        Indexing using a single index:\n\n        >>> p = r[0]\n        >>> p.as_quat()\n        array([0.70710678, 0.70710678, 0.        , 0.        ])\n\n        Array slicing:\n\n        >>> q = r[1:3]\n        >>> q.as_quat()\n        array([[ 0.        ,  0.70710678,  0.        ,  0.70710678],\n               [ 0.57735027,  0.57735027, -0.57735027,  0.        ]])\n\n        '
        ...
    
    def __getstate__(self) -> typing.Any:
        ...
    
    def __init__(self, *args, **kwargs) -> None:
        "Rotation in 3 dimensions.\n\n    This class provides an interface to initialize from and represent rotations\n    with:\n\n    - Quaternions\n    - Rotation Matrices\n    - Rotation Vectors\n    - Modified Rodrigues Parameters\n    - Euler Angles\n\n    The following operations on rotations are supported:\n\n    - Application on vectors\n    - Rotation Composition\n    - Rotation Inversion\n    - Rotation Indexing\n\n    Indexing within a rotation is supported since multiple rotation transforms\n    can be stored within a single `Rotation` instance.\n\n    To create `Rotation` objects use ``from_...`` methods (see examples below).\n    ``Rotation(...)`` is not supposed to be instantiated directly.\n\n    Attributes\n    ----------\n    single\n\n    Methods\n    -------\n    __len__\n    from_quat\n    from_matrix\n    from_rotvec\n    from_mrp\n    from_euler\n    as_quat\n    as_matrix\n    as_rotvec\n    as_mrp\n    as_euler\n    apply\n    __mul__\n    inv\n    magnitude\n    mean\n    reduce\n    create_group\n    __getitem__\n    identity\n    random\n    align_vectors\n\n    See Also\n    --------\n    Slerp\n\n    Notes\n    -----\n    .. versionadded: 1.2.0\n\n    Examples\n    --------\n    >>> from scipy.spatial.transform import Rotation as R\n\n    A `Rotation` instance can be initialized in any of the above formats and\n    converted to any of the others. The underlying object is independent of the\n    representation used for initialization.\n\n    Consider a counter-clockwise rotation of 90 degrees about the z-axis. This\n    corresponds to the following quaternion (in scalar-last format):\n\n    >>> r = R.from_quat([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])\n\n    The rotation can be expressed in any of the other formats:\n\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n    [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n    [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The same rotation can be initialized using a rotation matrix:\n\n    >>> r = R.from_matrix([[0, -1, 0],\n    ...                    [1, 0, 0],\n    ...                    [0, 0, 1]])\n\n    Representation in other formats:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The rotation vector corresponding to this rotation is given by:\n\n    >>> r = R.from_rotvec(np.pi/2 * np.array([0, 0, 1]))\n\n    Representation in other formats:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_euler('zyx', degrees=True)\n    array([90.,  0.,  0.])\n\n    The ``from_euler`` method is quite flexible in the range of input formats\n    it supports. Here we initialize a single rotation about a single axis:\n\n    >>> r = R.from_euler('z', 90, degrees=True)\n\n    Again, the object is representation independent and can be converted to any\n    other format:\n\n    >>> r.as_quat()\n    array([0.        , 0.        , 0.70710678, 0.70710678])\n    >>> r.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> r.as_rotvec()\n    array([0.        , 0.        , 1.57079633])\n\n    It is also possible to initialize multiple rotations in a single instance\n    using any of the `from_...` functions. Here we initialize a stack of 3\n    rotations using the ``from_euler`` method:\n\n    >>> r = R.from_euler('zyx', [\n    ... [90, 0, 0],\n    ... [0, 45, 0],\n    ... [45, 60, 30]], degrees=True)\n\n    The other representations also now return a stack of 3 rotations. For\n    example:\n\n    >>> r.as_quat()\n    array([[0.        , 0.        , 0.70710678, 0.70710678],\n           [0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n\n    Applying the above rotations onto a vector:\n\n    >>> v = [1, 2, 3]\n    >>> r.apply(v)\n    array([[-2.        ,  1.        ,  3.        ],\n           [ 2.82842712,  2.        ,  1.41421356],\n           [ 2.24452282,  0.78093109,  2.89002836]])\n\n    A `Rotation` instance can be indexed and sliced as if it were a single\n    1D array or list:\n\n    >>> r.as_quat()\n    array([[0.        , 0.        , 0.70710678, 0.70710678],\n           [0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n    >>> p = r[0]\n    >>> p.as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n    >>> q = r[1:3]\n    >>> q.as_quat()\n    array([[0.        , 0.38268343, 0.        , 0.92387953],\n           [0.39190384, 0.36042341, 0.43967974, 0.72331741]])\n\n    In fact it can be converted to numpy.array:\n\n    >>> r_array = np.asarray(r)\n    >>> r_array.shape\n    (3,)\n    >>> r_array[0].as_matrix()\n    array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n           [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n           [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n\n    Multiple rotations can be composed using the ``*`` operator:\n\n    >>> r1 = R.from_euler('z', 90, degrees=True)\n    >>> r2 = R.from_rotvec([np.pi/4, 0, 0])\n    >>> v = [1, 2, 3]\n    >>> r2.apply(r1.apply(v))\n    array([-2.        , -1.41421356,  2.82842712])\n    >>> r3 = r2 * r1 # Note the order\n    >>> r3.apply(v)\n    array([-2.        , -1.41421356,  2.82842712])\n\n    Finally, it is also possible to invert rotations:\n\n    >>> r1 = R.from_euler('z', [90, 45], degrees=True)\n    >>> r2 = r1.inv()\n    >>> r2.as_euler('zyx', degrees=True)\n    array([[-90.,   0.,   0.],\n           [-45.,   0.,   0.]])\n\n    These examples serve as an overview into the `Rotation` class and highlight\n    major functionalities. For more thorough examples of the range of input and\n    output formats supported, consult the individual method's examples.\n\n    "
        ...
    
    @classmethod
    def __init_subclass__(cls) -> None:
        'This method is called when a class is subclassed.\n\nThe default implementation does nothing. It may be\noverridden to extend subclasses.\n'
        ...
    
    def __len__(self) -> int:
        'Number of rotations contained in this object.\n\n        Multiple rotations can be stored in a single instance.\n\n        Returns\n        -------\n        length : int\n            Number of rotations stored in object.\n\n        Raises\n        ------\n        TypeError if the instance was created as a single rotation.\n        '
        ...
    
    def __mul__(self) -> Rotation:
        "Compose this rotation with the other.\n\n        If `p` and `q` are two rotations, then the composition of 'q followed\n        by p' is equivalent to `p * q`. In terms of rotation matrices,\n        the composition can be expressed as\n        ``p.as_matrix().dot(q.as_matrix())``.\n\n        Parameters\n        ----------\n        other : `Rotation` instance\n            Object containing the rotations to be composed with this one. Note\n            that rotation compositions are not commutative, so ``p * q`` is\n            different from ``q * p``.\n\n        Returns\n        -------\n        composition : `Rotation` instance\n            This function supports composition of multiple rotations at a time.\n            The following cases are possible:\n\n            - Either ``p`` or ``q`` contains a single rotation. In this case\n              `composition` contains the result of composing each rotation in\n              the other object with the single rotation.\n            - Both ``p`` and ``q`` contain ``N`` rotations. In this case each\n              rotation ``p[i]`` is composed with the corresponding rotation\n              ``q[i]`` and `output` contains ``N`` rotations.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Composition of two single rotations:\n\n        >>> p = R.from_quat([0, 0, 1, 1])\n        >>> q = R.from_quat([1, 0, 0, 1])\n        >>> p.as_matrix()\n        array([[ 0., -1.,  0.],\n               [ 1.,  0.,  0.],\n               [ 0.,  0.,  1.]])\n        >>> q.as_matrix()\n        array([[ 1.,  0.,  0.],\n               [ 0.,  0., -1.],\n               [ 0.,  1.,  0.]])\n        >>> r = p * q\n        >>> r.as_matrix()\n        array([[0., 0., 1.],\n               [1., 0., 0.],\n               [0., 1., 0.]])\n\n        Composition of two objects containing equal number of rotations:\n\n        >>> p = R.from_quat([[0, 0, 1, 1], [1, 0, 0, 1]])\n        >>> q = R.from_rotvec([[np.pi/4, 0, 0], [-np.pi/4, 0, np.pi/4]])\n        >>> p.as_quat()\n        array([[0.        , 0.        , 0.70710678, 0.70710678],\n               [0.70710678, 0.        , 0.        , 0.70710678]])\n        >>> q.as_quat()\n        array([[ 0.38268343,  0.        ,  0.        ,  0.92387953],\n               [-0.37282173,  0.        ,  0.37282173,  0.84971049]])\n        >>> r = p * q\n        >>> r.as_quat()\n        array([[ 0.27059805,  0.27059805,  0.65328148,  0.65328148],\n               [ 0.33721128, -0.26362477,  0.26362477,  0.86446082]])\n\n        "
        ...
    
    def __reduce_cython__(self) -> typing.Any:
        ...
    
    def __rmul__(self, value) -> Rotation:
        'Return value*self.'
        ...
    
    def __setstate__(self, state: typing.Any) -> None:
        ...
    
    def __setstate_cython__(self) -> typing.Any:
        ...
    
    @classmethod
    def __subclasshook__(cls, subclass: typing.Any) -> bool:
        'Abstract classes can override this to customize issubclass().\n\nThis is invoked early on by abc.ABCMeta.__subclasscheck__().\nIt should return True, False or NotImplemented.  If it returns\nNotImplemented, the normal algorithm is used.  Otherwise, it\noverrides the normal algorithm (and the outcome is cached).\n'
        ...
    
    @classmethod
    def align_vectors(cls) -> typing.Any:
        'Estimate a rotation to optimally align two sets of vectors.\n\n        Find a rotation between frames A and B which best aligns a set of\n        vectors `a` and `b` observed in these frames. The following loss\n        function is minimized to solve for the rotation matrix\n        :math:`C`:\n\n        .. math::\n\n            L(C) = \\frac{1}{2} \\sum_{i = 1}^{n} w_i \\lVert \\mathbf{a}_i -\n            C \\mathbf{b}_i \\rVert^2 ,\n\n        where :math:`w_i`\'s are the `weights` corresponding to each vector.\n\n        The rotation is estimated with Kabsch algorithm [1]_.\n\n        Parameters\n        ----------\n        a : array_like, shape (N, 3)\n            Vector components observed in initial frame A. Each row of `a`\n            denotes a vector.\n        b : array_like, shape (N, 3)\n            Vector components observed in another frame B. Each row of `b`\n            denotes a vector.\n        weights : array_like shape (N,), optional\n            Weights describing the relative importance of the vector\n            observations. If None (default), then all values in `weights` are\n            assumed to be 1.\n        return_sensitivity : bool, optional\n            Whether to return the sensitivity matrix. See Notes for details.\n            Default is False.\n\n        Returns\n        -------\n        estimated_rotation : `Rotation` instance\n            Best estimate of the rotation that transforms `b` to `a`.\n        rmsd : float\n            Root mean square distance (weighted) between the given set of\n            vectors after alignment. It is equal to ``sqrt(2 * minimum_loss)``,\n            where ``minimum_loss`` is the loss function evaluated for the\n            found optimal rotation.\n        sensitivity_matrix : ndarray, shape (3, 3)\n            Sensitivity matrix of the estimated rotation estimate as explained\n            in Notes. Returned only when `return_sensitivity` is True.\n\n        Notes\n        -----\n        This method can also compute the sensitivity of the estimated rotation\n        to small perturbations of the vector measurements. Specifically we\n        consider the rotation estimate error as a small rotation vector of\n        frame A. The sensitivity matrix is proportional to the covariance of\n        this rotation vector assuming that the vectors in `a` was measured with\n        errors significantly less than their lengths. To get the true\n        covariance matrix, the returned sensitivity matrix must be multiplied\n        by harmonic mean [3]_ of variance in each observation. Note that\n        `weights` are supposed to be inversely proportional to the observation\n        variances to get consistent results. For example, if all vectors are\n        measured with the same accuracy of 0.01 (`weights` must be all equal),\n        then you should multiple the sensitivity matrix by 0.01**2 to get the\n        covariance.\n\n        Refer to [2]_ for more rigorous discussion of the covariance\n        estimation.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Kabsch_algorithm\n        .. [2] F. Landis Markley,\n                "Attitude determination using vector observations: a fast\n                optimal matrix algorithm", Journal of Astronautical Sciences,\n                Vol. 41, No.2, 1993, pp. 261-280.\n        .. [3] https://en.wikipedia.org/wiki/Harmonic_mean\n        '
        ...
    
    def apply(self) -> typing.Any:
        "Apply this rotation to a set of vectors.\n\n        If the original frame rotates to the final frame by this rotation, then\n        its application to a vector can be seen in two ways:\n\n            - As a projection of vector components expressed in the final frame\n              to the original frame.\n            - As the physical rotation of a vector being glued to the original\n              frame as it rotates. In this case the vector components are\n              expressed in the original frame before and after the rotation.\n\n        In terms of rotation matricies, this application is the same as\n        ``self.as_matrix().dot(vectors)``.\n\n        Parameters\n        ----------\n        vectors : array_like, shape (3,) or (N, 3)\n            Each `vectors[i]` represents a vector in 3D space. A single vector\n            can either be specified with shape `(3, )` or `(1, 3)`. The number\n            of rotations and number of vectors given must follow standard numpy\n            broadcasting rules: either one of them equals unity or they both\n            equal each other.\n        inverse : boolean, optional\n            If True then the inverse of the rotation(s) is applied to the input\n            vectors. Default is False.\n\n        Returns\n        -------\n        rotated_vectors : ndarray, shape (3,) or (N, 3)\n            Result of applying rotation on input vectors.\n            Shape depends on the following cases:\n\n                - If object contains a single rotation (as opposed to a stack\n                  with a single rotation) and a single vector is specified with\n                  shape ``(3,)``, then `rotated_vectors` has shape ``(3,)``.\n                - In all other cases, `rotated_vectors` has shape ``(N, 3)``,\n                  where ``N`` is either the number of rotations or vectors.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Single rotation applied on a single vector:\n\n        >>> vector = np.array([1, 0, 0])\n        >>> r = R.from_rotvec([0, 0, np.pi/2])\n        >>> r.as_matrix()\n        array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n               [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n               [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n        >>> r.apply(vector)\n        array([2.22044605e-16, 1.00000000e+00, 0.00000000e+00])\n        >>> r.apply(vector).shape\n        (3,)\n\n        Single rotation applied on multiple vectors:\n\n        >>> vectors = np.array([\n        ... [1, 0, 0],\n        ... [1, 2, 3]])\n        >>> r = R.from_rotvec([0, 0, np.pi/4])\n        >>> r.as_matrix()\n        array([[ 0.70710678, -0.70710678,  0.        ],\n               [ 0.70710678,  0.70710678,  0.        ],\n               [ 0.        ,  0.        ,  1.        ]])\n        >>> r.apply(vectors)\n        array([[ 0.70710678,  0.70710678,  0.        ],\n               [-0.70710678,  2.12132034,  3.        ]])\n        >>> r.apply(vectors).shape\n        (2, 3)\n\n        Multiple rotations on a single vector:\n\n        >>> r = R.from_rotvec([[0, 0, np.pi/4], [np.pi/2, 0, 0]])\n        >>> vector = np.array([1,2,3])\n        >>> r.as_matrix()\n        array([[[ 7.07106781e-01, -7.07106781e-01,  0.00000000e+00],\n                [ 7.07106781e-01,  7.07106781e-01,  0.00000000e+00],\n                [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]],\n               [[ 1.00000000e+00,  0.00000000e+00,  0.00000000e+00],\n                [ 0.00000000e+00,  2.22044605e-16, -1.00000000e+00],\n                [ 0.00000000e+00,  1.00000000e+00,  2.22044605e-16]]])\n        >>> r.apply(vector)\n        array([[-0.70710678,  2.12132034,  3.        ],\n               [ 1.        , -3.        ,  2.        ]])\n        >>> r.apply(vector).shape\n        (2, 3)\n\n        Multiple rotations on multiple vectors. Each rotation is applied on the\n        corresponding vector:\n\n        >>> r = R.from_euler('zxy', [\n        ... [0, 0, 90],\n        ... [45, 30, 60]], degrees=True)\n        >>> vectors = [\n        ... [1, 2, 3],\n        ... [1, 0, -1]]\n        >>> r.apply(vectors)\n        array([[ 3.        ,  2.        , -1.        ],\n               [-0.09026039,  1.11237244, -0.86860844]])\n        >>> r.apply(vectors).shape\n        (2, 3)\n\n        It is also possible to apply the inverse rotation:\n\n        >>> r = R.from_euler('zxy', [\n        ... [0, 0, 90],\n        ... [45, 30, 60]], degrees=True)\n        >>> vectors = [\n        ... [1, 2, 3],\n        ... [1, 0, -1]]\n        >>> r.apply(vectors, inverse=True)\n        array([[-3.        ,  2.        ,  1.        ],\n               [ 1.09533535, -0.8365163 ,  0.3169873 ]])\n\n        "
        ...
    
    def as_euler(self) -> typing.Any:
        'Represent as Euler angles.\n\n        Any orientation can be expressed as a composition of 3 elementary\n        rotations. Once the axis sequence has been chosen, Euler angles define\n        the angle of rotation around each respective axis [1]_.\n\n        The algorithm from [2]_ has been used to calculate Euler angles for the\n        rotation about a given sequence of axes.\n\n        Euler angles suffer from the problem of gimbal lock [3]_, where the\n        representation loses a degree of freedom and it is not possible to\n        determine the first and third angles uniquely. In this case,\n        a warning is raised, and the third angle is set to zero. Note however\n        that the returned angles still represent the correct rotation.\n\n        Parameters\n        ----------\n        seq : string, length 3\n            3 characters belonging to the set {\'X\', \'Y\', \'Z\'} for intrinsic\n            rotations, or {\'x\', \'y\', \'z\'} for extrinsic rotations [1]_.\n            Adjacent axes cannot be the same.\n            Extrinsic and intrinsic rotations cannot be mixed in one function\n            call.\n        degrees : boolean, optional\n            Returned angles are in degrees if this flag is True, else they are\n            in radians. Default is False.\n\n        Returns\n        -------\n        angles : ndarray, shape (3,) or (N, 3)\n            Shape depends on shape of inputs used to initialize object.\n            The returned angles are in the range:\n\n            - First angle belongs to [-180, 180] degrees (both inclusive)\n            - Third angle belongs to [-180, 180] degrees (both inclusive)\n            - Second angle belongs to:\n\n                - [-90, 90] degrees if all axes are different (like xyz)\n                - [0, 180] degrees if first and third axes are the same\n                  (like zxz)\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Euler_angles#Definition_by_intrinsic_rotations\n        .. [2] Malcolm D. Shuster, F. Landis Markley, "General formula for\n               extraction the Euler angles", Journal of guidance, control, and\n               dynamics, vol. 29.1, pp. 215-221. 2006\n        .. [3] https://en.wikipedia.org/wiki/Gimbal_lock#In_applied_mathematics\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Represent a single rotation:\n\n        >>> r = R.from_rotvec([0, 0, np.pi/2])\n        >>> r.as_euler(\'zxy\', degrees=True)\n        array([90.,  0.,  0.])\n        >>> r.as_euler(\'zxy\', degrees=True).shape\n        (3,)\n\n        Represent a stack of single rotation:\n\n        >>> r = R.from_rotvec([[0, 0, np.pi/2]])\n        >>> r.as_euler(\'zxy\', degrees=True)\n        array([[90.,  0.,  0.]])\n        >>> r.as_euler(\'zxy\', degrees=True).shape\n        (1, 3)\n\n        Represent multiple rotations in a single object:\n\n        >>> r = R.from_rotvec([\n        ... [0, 0, np.pi/2],\n        ... [0, -np.pi/3, 0],\n        ... [np.pi/4, 0, 0]])\n        >>> r.as_euler(\'zxy\', degrees=True)\n        array([[ 90.,   0.,   0.],\n               [  0.,   0., -60.],\n               [  0.,  45.,   0.]])\n        >>> r.as_euler(\'zxy\', degrees=True).shape\n        (3, 3)\n\n        '
        ...
    
    def as_matrix(self) -> typing.Any:
        'Represent as rotation matrix.\n\n        3D rotations can be represented using rotation matrices, which\n        are 3 x 3 real orthogonal matrices with determinant equal to +1 [1]_.\n\n        Returns\n        -------\n        matrix : ndarray, shape (3, 3) or (N, 3, 3)\n            Shape depends on shape of inputs used for initialization.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Rotation_matrix#In_three_dimensions\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Represent a single rotation:\n\n        >>> r = R.from_rotvec([0, 0, np.pi/2])\n        >>> r.as_matrix()\n        array([[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n               [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n               [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]])\n        >>> r.as_matrix().shape\n        (3, 3)\n\n        Represent a stack with a single rotation:\n\n        >>> r = R.from_quat([[1, 1, 0, 0]])\n        >>> r.as_matrix()\n        array([[[ 0.,  1.,  0.],\n                [ 1.,  0.,  0.],\n                [ 0.,  0., -1.]]])\n        >>> r.as_matrix().shape\n        (1, 3, 3)\n\n        Represent multiple rotations:\n\n        >>> r = R.from_rotvec([[np.pi/2, 0, 0], [0, 0, np.pi/2]])\n        >>> r.as_matrix()\n        array([[[ 1.00000000e+00,  0.00000000e+00,  0.00000000e+00],\n                [ 0.00000000e+00,  2.22044605e-16, -1.00000000e+00],\n                [ 0.00000000e+00,  1.00000000e+00,  2.22044605e-16]],\n               [[ 2.22044605e-16, -1.00000000e+00,  0.00000000e+00],\n                [ 1.00000000e+00,  2.22044605e-16,  0.00000000e+00],\n                [ 0.00000000e+00,  0.00000000e+00,  1.00000000e+00]]])\n        >>> r.as_matrix().shape\n        (2, 3, 3)\n\n        Notes\n        -----\n        This function was called as_dcm before.\n\n        .. versionadded:: 1.4.0\n        '
        ...
    
    def as_mrp(self) -> typing.Any:
        'Represent as Modified Rodrigues Parameters (MRPs).\n\n        MRPs are a 3 dimensional vector co-directional to the axis of rotation and whose\n        magnitude is equal to ``tan(theta / 4)``, where ``theta`` is the angle of rotation\n        (in radians) [1]_.\n\n        MRPs have a singuarity at 360 degrees which can be avoided by ensuring the angle of\n        rotation does not exceed 180 degrees, i.e. switching the direction of the rotation when\n        it is past 180 degrees. This function will always return MRPs corresponding to a rotation\n        of less than or equal to 180 degrees.\n\n        Returns\n        -------\n        mrps : ndarray, shape (3,) or (N, 3)\n            Shape depends on shape of inputs used for initialization.\n\n        References\n        ----------\n        .. [1] Shuster, M. D. "A Survery of Attitude Representations",\n               The Journal of Astronautical Sciences, Vol. 41, No.4, 1993,\n               pp. 475-476\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Represent a single rotation:\n\n        >>> r = R.from_rotvec([0, 0, np.pi])\n        >>> r.as_mrp()\n        array([0.        , 0.        , 1.         ])\n        >>> r.as_mrp().shape\n        (3,)\n\n        Represent a stack with a single rotation:\n\n        >>> r = R.from_euler(\'xyz\', [[180, 0, 0]], degrees=True)\n        >>> r.as_mrp()\n        array([[1.       , 0.        , 0.         ]])\n        >>> r.as_mrp().shape\n        (1, 3)\n\n        Represent multiple rotations:\n\n        >>> r = R.from_rotvec([[np.pi/2, 0, 0], [0, 0, np.pi/2]])\n        >>> r.as_mrp()\n        array([[0.41421356, 0.        , 0.        ],\n               [0.        , 0.        , 0.41421356]])\n        >>> r.as_mrp().shape\n        (2, 3)\n\n        Notes\n        -----\n\n        .. versionadded:: 1.6.0\n        '
        ...
    
    def as_quat(self) -> typing.Any:
        'Represent as quaternions.\n\n        Rotations in 3 dimensions can be represented using unit norm\n        quaternions [1]_. The mapping from quaternions to rotations is\n        two-to-one, i.e. quaternions ``q`` and ``-q``, where ``-q`` simply\n        reverses the sign of each component, represent the same spatial\n        rotation. The returned value is in scalar-last (x, y, z, w) format.\n\n        Returns\n        -------\n        quat : `numpy.ndarray`, shape (4,) or (N, 4)\n            Shape depends on shape of inputs used for initialization.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Quaternions_and_spatial_rotation\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Represent a single rotation:\n\n        >>> r = R.from_matrix([[0, -1, 0],\n        ...                    [1, 0, 0],\n        ...                    [0, 0, 1]])\n        >>> r.as_quat()\n        array([0.        , 0.        , 0.70710678, 0.70710678])\n        >>> r.as_quat().shape\n        (4,)\n\n        Represent a stack with a single rotation:\n\n        >>> r = R.from_quat([[0, 0, 0, 1]])\n        >>> r.as_quat().shape\n        (1, 4)\n\n        Represent multiple rotations in a single object:\n\n        >>> r = R.from_rotvec([[np.pi, 0, 0], [0, 0, np.pi/2]])\n        >>> r.as_quat().shape\n        (2, 4)\n\n        '
        ...
    
    def as_rotvec(self) -> typing.Any:
        "Represent as rotation vectors.\n\n        A rotation vector is a 3 dimensional vector which is co-directional to\n        the axis of rotation and whose norm gives the angle of rotation (in\n        radians) [1]_.\n\n        Returns\n        -------\n        rotvec : ndarray, shape (3,) or (N, 3)\n            Shape depends on shape of inputs used for initialization.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Axis%E2%80%93angle_representation#Rotation_vector\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Represent a single rotation:\n\n        >>> r = R.from_euler('z', 90, degrees=True)\n        >>> r.as_rotvec()\n        array([0.        , 0.        , 1.57079633])\n        >>> r.as_rotvec().shape\n        (3,)\n\n        Represent a stack with a single rotation:\n\n        >>> r = R.from_quat([[0, 0, 1, 1]])\n        >>> r.as_rotvec()\n        array([[0.        , 0.        , 1.57079633]])\n        >>> r.as_rotvec().shape\n        (1, 3)\n\n        Represent multiple rotations in a single object:\n\n        >>> r = R.from_quat([[0, 0, 1, 1], [1, 1, 0, 1]])\n        >>> r.as_rotvec()\n        array([[0.        , 0.        , 1.57079633],\n               [1.35102172, 1.35102172, 0.        ]])\n        >>> r.as_rotvec().shape\n        (2, 3)\n\n        "
        ...
    
    @classmethod
    def create_group(cls) -> typing.Any:
        "Create a 3D rotation group.\n\n        Parameters\n        ----------\n        group : string\n            The name of the group. Must be one of 'I', 'O', 'T', 'Dn', 'Cn',\n            where `n` is a positive integer. The groups are:\n\n                * I: Icosahedral group\n                * O: Octahedral group\n                * T: Tetrahedral group\n                * D: Dicyclic group\n                * C: Cyclic group\n\n        axis : integer\n            The cyclic rotation axis. Must be one of ['X', 'Y', 'Z'] (or\n            lowercase). Default is 'Z'. Ignored for groups 'I', 'O', and 'T'.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the elements of the rotation group.\n\n        Notes\n        -----\n        This method generates rotation groups only. The full 3-dimensional\n        point groups [PointGroups]_ also contain reflections.\n\n        References\n        ----------\n        .. [PointGroups] `Point groups\n           <https://en.wikipedia.org/wiki/Point_groups_in_three_dimensions>`_\n           on Wikipedia.\n        "
        ...
    
    @classmethod
    def from_euler(cls) -> typing.Any:
        "Initialize from Euler angles.\n\n        Rotations in 3-D can be represented by a sequence of 3\n        rotations around a sequence of axes. In theory, any three axes spanning\n        the 3-D Euclidean space are enough. In practice, the axes of rotation are\n        chosen to be the basis vectors.\n\n        The three rotations can either be in a global frame of reference\n        (extrinsic) or in a body centred frame of reference (intrinsic), which\n        is attached to, and moves with, the object under rotation [1]_.\n\n        Parameters\n        ----------\n        seq : string\n            Specifies sequence of axes for rotations. Up to 3 characters\n            belonging to the set {'X', 'Y', 'Z'} for intrinsic rotations, or\n            {'x', 'y', 'z'} for extrinsic rotations. Extrinsic and intrinsic\n            rotations cannot be mixed in one function call.\n        angles : float or array_like, shape (N,) or (N, [1 or 2 or 3])\n            Euler angles specified in radians (`degrees` is False) or degrees\n            (`degrees` is True).\n            For a single character `seq`, `angles` can be:\n\n            - a single value\n            - array_like with shape (N,), where each `angle[i]`\n              corresponds to a single rotation\n            - array_like with shape (N, 1), where each `angle[i, 0]`\n              corresponds to a single rotation\n\n            For 2- and 3-character wide `seq`, `angles` can be:\n\n            - array_like with shape (W,) where `W` is the width of\n              `seq`, which corresponds to a single rotation with `W` axes\n            - array_like with shape (N, W) where each `angle[i]`\n              corresponds to a sequence of Euler angles describing a single\n              rotation\n\n        degrees : bool, optional\n            If True, then the given angles are assumed to be in degrees.\n            Default is False.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the rotation represented by the sequence of\n            rotations around given axes with given angles.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Euler_angles#Definition_by_intrinsic_rotations\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Initialize a single rotation along a single axis:\n\n        >>> r = R.from_euler('x', 90, degrees=True)\n        >>> r.as_quat().shape\n        (4,)\n\n        Initialize a single rotation with a given axis sequence:\n\n        >>> r = R.from_euler('zyx', [90, 45, 30], degrees=True)\n        >>> r.as_quat().shape\n        (4,)\n\n        Initialize a stack with a single rotation around a single axis:\n\n        >>> r = R.from_euler('x', [90], degrees=True)\n        >>> r.as_quat().shape\n        (1, 4)\n\n        Initialize a stack with a single rotation with an axis sequence:\n\n        >>> r = R.from_euler('zyx', [[90, 45, 30]], degrees=True)\n        >>> r.as_quat().shape\n        (1, 4)\n\n        Initialize multiple elementary rotations in one object:\n\n        >>> r = R.from_euler('x', [90, 45, 30], degrees=True)\n        >>> r.as_quat().shape\n        (3, 4)\n\n        Initialize multiple rotations in one object:\n\n        >>> r = R.from_euler('zyx', [[90, 45, 30], [35, 45, 90]], degrees=True)\n        >>> r.as_quat().shape\n        (2, 4)\n\n        "
        ...
    
    @classmethod
    def from_matrix(cls) -> typing.Any:
        'Initialize from rotation matrix.\n\n        Rotations in 3 dimensions can be represented with 3 x 3 proper\n        orthogonal matrices [1]_. If the input is not proper orthogonal,\n        an approximation is created using the method described in [2]_.\n\n        Parameters\n        ----------\n        matrix : array_like, shape (N, 3, 3) or (3, 3)\n            A single matrix or a stack of matrices, where ``matrix[i]`` is\n            the i-th matrix.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the rotations represented by the rotation\n            matrices.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Rotation_matrix#In_three_dimensions\n        .. [2] F. Landis Markley, "Unit Quaternion from Rotation Matrix",\n               Journal of guidance, control, and dynamics vol. 31.2, pp.\n               440-442, 2008.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Initialize a single rotation:\n\n        >>> r = R.from_matrix([\n        ... [0, -1, 0],\n        ... [1, 0, 0],\n        ... [0, 0, 1]])\n        >>> r.as_matrix().shape\n        (3, 3)\n\n        Initialize multiple rotations in a single object:\n\n        >>> r = R.from_matrix([\n        ... [\n        ...     [0, -1, 0],\n        ...     [1, 0, 0],\n        ...     [0, 0, 1],\n        ... ],\n        ... [\n        ...     [1, 0, 0],\n        ...     [0, 0, -1],\n        ...     [0, 1, 0],\n        ... ]])\n        >>> r.as_matrix().shape\n        (2, 3, 3)\n\n        If input matrices are not special orthogonal (orthogonal with\n        determinant equal to +1), then a special orthogonal estimate is stored:\n\n        >>> a = np.array([\n        ... [0, -0.5, 0],\n        ... [0.5, 0, 0],\n        ... [0, 0, 0.5]])\n        >>> np.linalg.det(a)\n        0.12500000000000003\n        >>> r = R.from_matrix(a)\n        >>> matrix = r.as_matrix()\n        >>> matrix\n        array([[-0.38461538, -0.92307692,  0.        ],\n               [ 0.92307692, -0.38461538,  0.        ],\n               [ 0.        ,  0.        ,  1.        ]])\n        >>> np.linalg.det(matrix)\n        1.0000000000000002\n\n        It is also possible to have a stack containing a single rotation:\n\n        >>> r = R.from_matrix([[\n        ... [0, -1, 0],\n        ... [1, 0, 0],\n        ... [0, 0, 1]]])\n        >>> r.as_matrix()\n        array([[[ 0., -1.,  0.],\n                [ 1.,  0.,  0.],\n                [ 0.,  0.,  1.]]])\n        >>> r.as_matrix().shape\n        (1, 3, 3)\n\n        Notes\n        -----\n        This function was called from_dcm before.\n\n        .. versionadded:: 1.4.0\n        '
        ...
    
    @classmethod
    def from_mrp(cls) -> typing.Any:
        'Initialize from Modified Rodrigues Parameters (MRPs).\n\n        MRPs are a 3 dimensional vector co-directional to the axis of rotation and whose\n        magnitude is equal to ``tan(theta / 4)``, where ``theta`` is the angle of rotation\n        (in radians) [1]_.\n\n        MRPs have a singuarity at 360 degrees which can be avoided by ensuring the angle of\n        rotation does not exceed 180 degrees, i.e. switching the direction of the rotation when\n        it is past 180 degrees.\n\n        Parameters\n        ----------\n        mrp : array_like, shape (N, 3) or (3,)\n            A single vector or a stack of vectors, where `mrp[i]` gives\n            the ith set of MRPs.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the rotations represented by input MRPs.\n\n        References\n        ----------\n        .. [1] Shuster, M. D. "A Survery of Attitude Representations",\n               The Journal of Astronautical Sciences, Vol. 41, No.4, 1993,\n               pp. 475-476\n\n        Notes\n        -----\n\n        .. versionadded:: 1.6.0\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Initialize a single rotation:\n\n        >>> r = R.from_mrp([0, 0, 1])\n        >>> r.as_euler(\'xyz\', degrees=True)\n        array([0.        , 0.        , 180.      ])\n        >>> r.as_euler(\'xyz\').shape\n        (3,)\n\n        Initialize multiple rotations in one object:\n\n        >>> r = R.from_mrp([\n        ... [0, 0, 1],\n        ... [1, 0, 0]])\n        >>> r.as_euler(\'xyz\', degrees=True)\n        array([[0.        , 0.        , 180.      ],\n               [180.0     , 0.        , 0.        ]])\n        >>> r.as_euler(\'xyz\').shape\n        (2, 3)\n\n        It is also possible to have a stack of a single rotation:\n\n        >>> r = R.from_mrp([[0, 0, np.pi/2]])\n        >>> r.as_euler(\'xyz\').shape\n        (1, 3)\n\n        '
        ...
    
    @classmethod
    def from_quat(cls) -> typing.Any:
        'Initialize from quaternions.\n\n        3D rotations can be represented using unit-norm quaternions [1]_.\n\n        Parameters\n        ----------\n        quat : array_like, shape (N, 4) or (4,)\n            Each row is a (possibly non-unit norm) quaternion in scalar-last\n            (x, y, z, w) format. Each quaternion will be normalized to unit\n            norm.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the rotations represented by input quaternions.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Quaternions_and_spatial_rotation\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Initialize a single rotation:\n\n        >>> r = R.from_quat([1, 0, 0, 0])\n        >>> r.as_quat()\n        array([1., 0., 0., 0.])\n        >>> r.as_quat().shape\n        (4,)\n\n        Initialize multiple rotations in a single object:\n\n        >>> r = R.from_quat([\n        ... [1, 0, 0, 0],\n        ... [0, 0, 0, 1]\n        ... ])\n        >>> r.as_quat()\n        array([[1., 0., 0., 0.],\n               [0., 0., 0., 1.]])\n        >>> r.as_quat().shape\n        (2, 4)\n\n        It is also possible to have a stack of a single rotation:\n\n        >>> r = R.from_quat([[0, 0, 0, 1]])\n        >>> r.as_quat()\n        array([[0., 0., 0., 1.]])\n        >>> r.as_quat().shape\n        (1, 4)\n\n        Quaternions are normalized before initialization.\n\n        >>> r = R.from_quat([0, 0, 1, 1])\n        >>> r.as_quat()\n        array([0.        , 0.        , 0.70710678, 0.70710678])\n        '
        ...
    
    @classmethod
    def from_rotvec(cls) -> typing.Any:
        'Initialize from rotation vectors.\n\n        A rotation vector is a 3 dimensional vector which is co-directional to\n        the axis of rotation and whose norm gives the angle of rotation (in\n        radians) [1]_.\n\n        Parameters\n        ----------\n        rotvec : array_like, shape (N, 3) or (3,)\n            A single vector or a stack of vectors, where `rot_vec[i]` gives\n            the ith rotation vector.\n\n        Returns\n        -------\n        rotation : `Rotation` instance\n            Object containing the rotations represented by input rotation\n            vectors.\n\n        References\n        ----------\n        .. [1] https://en.wikipedia.org/wiki/Axis%E2%80%93angle_representation#Rotation_vector\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Initialize a single rotation:\n\n        >>> r = R.from_rotvec(np.pi/2 * np.array([0, 0, 1]))\n        >>> r.as_rotvec()\n        array([0.        , 0.        , 1.57079633])\n        >>> r.as_rotvec().shape\n        (3,)\n\n        Initialize multiple rotations in one object:\n\n        >>> r = R.from_rotvec([\n        ... [0, 0, np.pi/2],\n        ... [np.pi/2, 0, 0]])\n        >>> r.as_rotvec()\n        array([[0.        , 0.        , 1.57079633],\n               [1.57079633, 0.        , 0.        ]])\n        >>> r.as_rotvec().shape\n        (2, 3)\n\n        It is also possible to have a stack of a single rotaton:\n\n        >>> r = R.from_rotvec([[0, 0, np.pi/2]])\n        >>> r.as_rotvec().shape\n        (1, 3)\n\n        '
        ...
    
    @classmethod
    def identity(cls) -> typing.Any:
        'Get identity rotation(s).\n\n        Composition with the identity rotation has no effect.\n\n        Parameters\n        ----------\n        num : int or None, optional\n            Number of identity rotations to generate. If None (default), then a\n            single rotation is generated.\n\n        Returns\n        -------\n        identity : Rotation object\n            The identity rotation.\n        '
        ...
    
    def inv(self) -> typing.Any:
        "Invert this rotation.\n\n        Composition of a rotation with its inverse results in an identity\n        transformation.\n\n        Returns\n        -------\n        inverse : `Rotation` instance\n            Object containing inverse of the rotations in the current instance.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Inverting a single rotation:\n\n        >>> p = R.from_euler('z', 45, degrees=True)\n        >>> q = p.inv()\n        >>> q.as_euler('zyx', degrees=True)\n        array([-45.,   0.,   0.])\n\n        Inverting multiple rotations:\n\n        >>> p = R.from_rotvec([[0, 0, np.pi/3], [-np.pi/4, 0, 0]])\n        >>> q = p.inv()\n        >>> q.as_rotvec()\n        array([[-0.        , -0.        , -1.04719755],\n               [ 0.78539816, -0.        , -0.        ]])\n\n        "
        ...
    
    def magnitude(self) -> typing.Any:
        'Get the magnitude(s) of the rotation(s).\n\n        Returns\n        -------\n        magnitude : ndarray or float\n            Angle(s) in radians, float if object contains a single rotation\n            and ndarray if object contains multiple rotations.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n        >>> r = R.from_quat(np.eye(4))\n        >>> r.magnitude()\n        array([3.14159265, 3.14159265, 3.14159265, 0.        ])\n\n        Magnitude of a single rotation:\n\n        >>> r[0].magnitude()\n        3.141592653589793\n        '
        ...
    
    def mean(self) -> typing.Any:
        "Get the mean of the rotations.\n\n        Parameters\n        ----------\n        weights : array_like shape (N,), optional\n            Weights describing the relative importance of the rotations. If\n            None (default), then all values in `weights` are assumed to be\n            equal.\n\n        Returns\n        -------\n        mean : `Rotation` instance\n            Object containing the mean of the rotations in the current\n            instance.\n\n        Notes\n        -----\n        The mean used is the chordal L2 mean (also called the projected or\n        induced arithmetic mean). If ``p`` is a set of rotations with mean\n        ``m``, then ``m`` is the rotation which minimizes\n        ``(weights[:, None, None] * (p.as_matrix() - m.as_matrix())**2).sum()``.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n        >>> r = R.from_euler('zyx', [[0, 0, 0],\n        ...                          [1, 0, 0],\n        ...                          [0, 1, 0],\n        ...                          [0, 0, 1]], degrees=True)\n        >>> r.mean().as_euler('zyx', degrees=True)\n        array([0.24945696, 0.25054542, 0.24945696])\n        "
        ...
    
    @classmethod
    def random(cls) -> typing.Any:
        "Generate uniformly distributed rotations.\n\n        Parameters\n        ----------\n        num : int or None, optional\n            Number of random rotations to generate. If None (default), then a\n            single rotation is generated.\n        random_state : int, RandomState instance or None, optional\n            Accepts an integer as a seed for the random generator or a\n            RandomState object. If None (default), uses global `numpy.random`\n            random state.\n\n        Returns\n        -------\n        random_rotation : `Rotation` instance\n            Contains a single rotation if `num` is None. Otherwise contains a\n            stack of `num` rotations.\n\n        Notes\n        -----\n        This function is optimized for efficiently sampling random rotation\n        matrices in three dimensions. For generating random rotation matrices\n        in higher dimensions, see `scipy.stats.special_ortho_group`.\n\n        Examples\n        --------\n        >>> from scipy.spatial.transform import Rotation as R\n\n        Sample a single rotation:\n\n        >>> R.random(random_state=1234).as_euler('zxy', degrees=True)\n        array([-110.5976185 ,   55.32758512,   76.3289269 ])\n\n        Sample a stack of rotations:\n\n        >>> R.random(5, random_state=1234).as_euler('zxy', degrees=True)\n        array([[-110.5976185 ,   55.32758512,   76.3289269 ],\n               [ -91.59132005,  -14.3629884 ,  -93.91933182],\n               [  25.23835501,   45.02035145, -121.67867086],\n               [ -51.51414184,  -15.29022692, -172.46870023],\n               [ -81.63376847,  -27.39521579,    2.60408416]])\n\n        See Also\n        --------\n        scipy.stats.special_ortho_group\n\n       "
        ...
    
    def reduce(self) -> typing.Any:
        'Reduce this rotation with the provided rotation groups.\n\n        Reduction of a rotation ``p`` is a transformation of the form\n        ``q = l * p * r``, where ``l`` and ``r`` are chosen from `left` and\n        `right` respectively, such that rotation ``q`` has the smallest\n        magnitude.\n\n        If `left` and `right` are rotation groups representing symmetries of\n        two objects rotated by ``p``, then ``q`` is the rotation of the\n        smallest magnitude to align these objects considering their symmetries.\n\n        Parameters\n        ----------\n        left : `Rotation` instance, optional\n            Object containing the left rotation(s). Default value (None)\n            corresponds to the identity rotation.\n        right : `Rotation` instance, optional\n            Object containing the right rotation(s). Default value (None)\n            corresponds to the identity rotation.\n        return_indices : bool, optional\n            Whether to return the indices of the rotations from `left` and\n            `right` used for reduction.\n\n        Returns\n        -------\n        reduced : `Rotation` instance\n            Object containing reduced rotations.\n        left_best, right_best: integer ndarray\n            Indices of elements from `left` and `right` used for reduction.\n        '
        ...
    
    @property
    def single(self) -> typing.Any:
        'Whether this instance represents a single rotation.'
        ...
    
    def __getattr__(self, name) -> typing.Any:
        ...
    

class Slerp(_mod_builtins.object):
    "Spherical Linear Interpolation of Rotations.\n\n    The interpolation between consecutive rotations is performed as a rotation\n    around a fixed axis with a constant angular velocity [1]_. This ensures\n    that the interpolated rotations follow the shortest path between initial\n    and final orientations.\n\n    Parameters\n    ----------\n    times : array_like, shape (N,)\n        Times of the known rotations. At least 2 times must be specified.\n    rotations : `Rotation` instance\n        Rotations to perform the interpolation between. Must contain N\n        rotations.\n\n    Methods\n    -------\n    __call__\n\n    See Also\n    --------\n    Rotation\n\n    Notes\n    -----\n    .. versionadded:: 1.2.0\n\n    References\n    ----------\n    .. [1] https://en.wikipedia.org/wiki/Slerp#Quaternion_Slerp\n\n    Examples\n    --------\n    >>> from scipy.spatial.transform import Rotation as R\n    >>> from scipy.spatial.transform import Slerp\n\n    Setup the fixed keyframe rotations and times:\n\n    >>> key_rots = R.random(5, random_state=2342345)\n    >>> key_times = [0, 1, 2, 3, 4]\n\n    Create the interpolator object:\n\n    >>> slerp = Slerp(key_times, key_rots)\n\n    Interpolate the rotations at the given times:\n\n    >>> times = [0, 0.5, 0.25, 1, 1.5, 2, 2.75, 3, 3.25, 3.60, 4]\n    >>> interp_rots = slerp(times)\n\n    The keyframe rotations expressed as Euler angles:\n\n    >>> key_rots.as_euler('xyz', degrees=True)\n    array([[ 14.31443779, -27.50095894,  -3.7275787 ],\n           [ -1.79924227, -24.69421529, 164.57701743],\n           [146.15020772,  43.22849451, -31.34891088],\n           [ 46.39959442,  11.62126073, -45.99719267],\n           [-88.94647804, -49.64400082, -65.80546984]])\n\n    The interpolated rotations expressed as Euler angles. These agree with the\n    keyframe rotations at both endpoints of the range of keyframe times.\n\n    >>> interp_rots.as_euler('xyz', degrees=True)\n    array([[  14.31443779,  -27.50095894,   -3.7275787 ],\n           [   4.74588574,  -32.44683966,   81.25139984],\n           [  10.71094749,  -31.56690154,   38.06896408],\n           [  -1.79924227,  -24.69421529,  164.57701743],\n           [  11.72796022,   51.64207311, -171.7374683 ],\n           [ 146.15020772,   43.22849451,  -31.34891088],\n           [  68.10921869,   20.67625074,  -48.74886034],\n           [  46.39959442,   11.62126073,  -45.99719267],\n           [  12.35552615,    4.21525086,  -64.89288124],\n           [ -30.08117143,  -19.90769513,  -78.98121326],\n           [ -88.94647804,  -49.64400082,  -65.80546984]])\n\n    "
    def __call__(self, times) -> typing.Any:
        'Interpolate rotations.\n\n        Compute the interpolated rotations at the given `times`.\n\n        Parameters\n        ----------\n        times : array_like\n            Times to compute the interpolations at. Can be a scalar or\n            1-dimensional.\n\n        Returns\n        -------\n        interpolated_rotation : `Rotation` instance\n            Object containing the rotations computed at given `times`.\n\n        '
        ...
    
    __dict__: typing.Dict[str, typing.Any]
    def __init__(self, times, rotations) -> None:
        "Spherical Linear Interpolation of Rotations.\n\n    The interpolation between consecutive rotations is performed as a rotation\n    around a fixed axis with a constant angular velocity [1]_. This ensures\n    that the interpolated rotations follow the shortest path between initial\n    and final orientations.\n\n    Parameters\n    ----------\n    times : array_like, shape (N,)\n        Times of the known rotations. At least 2 times must be specified.\n    rotations : `Rotation` instance\n        Rotations to perform the interpolation between. Must contain N\n        rotations.\n\n    Methods\n    -------\n    __call__\n\n    See Also\n    --------\n    Rotation\n\n    Notes\n    -----\n    .. versionadded:: 1.2.0\n\n    References\n    ----------\n    .. [1] https://en.wikipedia.org/wiki/Slerp#Quaternion_Slerp\n\n    Examples\n    --------\n    >>> from scipy.spatial.transform import Rotation as R\n    >>> from scipy.spatial.transform import Slerp\n\n    Setup the fixed keyframe rotations and times:\n\n    >>> key_rots = R.random(5, random_state=2342345)\n    >>> key_times = [0, 1, 2, 3, 4]\n\n    Create the interpolator object:\n\n    >>> slerp = Slerp(key_times, key_rots)\n\n    Interpolate the rotations at the given times:\n\n    >>> times = [0, 0.5, 0.25, 1, 1.5, 2, 2.75, 3, 3.25, 3.60, 4]\n    >>> interp_rots = slerp(times)\n\n    The keyframe rotations expressed as Euler angles:\n\n    >>> key_rots.as_euler('xyz', degrees=True)\n    array([[ 14.31443779, -27.50095894,  -3.7275787 ],\n           [ -1.79924227, -24.69421529, 164.57701743],\n           [146.15020772,  43.22849451, -31.34891088],\n           [ 46.39959442,  11.62126073, -45.99719267],\n           [-88.94647804, -49.64400082, -65.80546984]])\n\n    The interpolated rotations expressed as Euler angles. These agree with the\n    keyframe rotations at both endpoints of the range of keyframe times.\n\n    >>> interp_rots.as_euler('xyz', degrees=True)\n    array([[  14.31443779,  -27.50095894,   -3.7275787 ],\n           [   4.74588574,  -32.44683966,   81.25139984],\n           [  10.71094749,  -31.56690154,   38.06896408],\n           [  -1.79924227,  -24.69421529,  164.57701743],\n           [  11.72796022,   51.64207311, -171.7374683 ],\n           [ 146.15020772,   43.22849451,  -31.34891088],\n           [  68.10921869,   20.67625074,  -48.74886034],\n           [  46.39959442,   11.62126073,  -45.99719267],\n           [  12.35552615,    4.21525086,  -64.89288124],\n           [ -30.08117143,  -19.90769513,  -78.98121326],\n           [ -88.94647804,  -49.64400082,  -65.80546984]])\n\n    "
        ...
    
    @classmethod
    def __init_subclass__(cls) -> None:
        'This method is called when a class is subclassed.\n\nThe default implementation does nothing. It may be\noverridden to extend subclasses.\n'
        ...
    
    __module__: str
    @classmethod
    def __subclasshook__(cls, subclass: typing.Any) -> bool:
        'Abstract classes can override this to customize issubclass().\n\nThis is invoked early on by abc.ABCMeta.__subclasscheck__().\nIt should return True, False or NotImplemented.  If it returns\nNotImplemented, the normal algorithm is used.  Otherwise, it\noverrides the normal algorithm (and the outcome is cached).\n'
        ...
    
    @property
    def __weakref__(self) -> typing.Any:
        'list of weak references to the object (if defined)'
        ...
    
    def __getattr__(self, name) -> typing.Any:
        ...
    

__doc__: typing.Any
__file__: str
__name__: str
__package__: str
def __pyx_unpickle_Enum() -> typing.Any:
    ...

def __pyx_unpickle_Rotation() -> typing.Any:
    ...

__test__: dict
def check_random_state(seed) -> typing.Any:
    'Turn seed into a np.random.RandomState instance\n\n    If seed is None (or np.random), return the RandomState singleton used\n    by np.random.\n    If seed is an int, return a new RandomState instance seeded with seed.\n    If seed is already a RandomState instance, return it.\n    If seed is a new-style np.random.Generator, return it.\n    Otherwise, raise ValueError.\n    '
    ...

def create_group(cls, group, axis) -> typing.Any:
    ...

def __getattr__(name) -> typing.Any:
    ...

