# MatplotLibAPI

MatplotLibAPI is a Python library that simplifies the process of creating various types of plots from pandas DataFrames. It provides a high-level API for generating bubble charts, network graphs, pivot tables, tables, time series plots, treemaps, and sunburst charts.

## Installation

To install the library, you can use pip:

```bash
pip install MatplotLibAPI
```

## Generating Sample Data

The examples in this README use sample data that can be generated by running the `generate_sample_data.py` script:

```bash
python scripts/generate_sample_data.py
```

This will create a `data` directory with CSV files for each plot type.

## Usage

Here's a simple example of how to create a bubble chart using MatplotLibAPI with a sample CSV file:

```python
import pandas as pd
import matplotlib.pyplot as plt
from MatplotLibAPI import fplot_bubble

# Load the sample data
df = pd.read_csv('data/bubble.csv')

# Generate the bubble chart
fig = fplot_bubble(df, label='country', x='gdp_per_capita', y='population', z='population', title='Country Statistics')

# Display the plot
plt.show()
```

## Plot Types

The library supports the following plot types:

- **Bubble (Scatter plot)**
- **Bar / Stacked Bar**
- **Histogram + KDE**
- **Box / Violin**
- **Heatmap / Correlation Matrix**
- **Area**
- **Pie / Donut**
- **Waffle**
- **Sankey**
- **Network (Graph)**
- **Pivot**
- **Table**
- **Timeserie**
- **Treemap**
- **Sunburst**

## Examples with Sample Data

This repository includes a `data` directory with sample CSV files for each plot type. Here's how you can use them:

### Bubble Chart

```python
import pandas as pd
from MatplotLibAPI import fplot_bubble

df = pd.read_csv('data/bubble.csv')
fig = fplot_bubble(df, label='country', x='gdp_per_capita', y='life_expectancy', z='population')
fig.show()
```

### Network Graph

```python
import pandas as pd
from MatplotLibAPI import fplot_network

df = pd.read_csv('data/network.csv')
fig = fplot_network(df)
fig.show()
```

### Bar / Stacked Bar

```python
import pandas as pd
from MatplotLibAPI import fplot_bar

df = pd.read_csv('data/bar.csv')
fig = fplot_bar(df, category='product', value='revenue', group='region', stacked=True)
fig.show()
```

### Histogram + KDE

```python
import pandas as pd
from MatplotLibAPI import fplot_histogram_kde

df = pd.read_csv('data/histogram.csv')
fig = fplot_histogram_kde(df, column='waiting_time_minutes', bins=8, kde=True)
fig.show()
```

### Box / Violin

```python
import pandas as pd
from MatplotLibAPI import fplot_box_violin

df = pd.read_csv('data/box_violin.csv')
fig = fplot_box_violin(df, column='satisfaction_score', category='department', use_violin=True)
fig.show()
```

### Heatmap / Correlation Matrix

```python
import pandas as pd
from MatplotLibAPI import fplot_heatmap, fplot_correlation_matrix

heatmap_df = pd.read_csv('data/heatmap.csv')
correlation_df = pd.read_csv('data/correlation.csv')

fig_heatmap = fplot_heatmap(heatmap_df, index='month', columns='channel', values='engagements')
fig_corr = fplot_correlation_matrix(correlation_df)

fig_heatmap.show()
fig_corr.show()
```

### Area

```python
import pandas as pd
from MatplotLibAPI import fplot_area

df = pd.read_csv('data/area.csv')
fig = fplot_area(df, x='quarter', y='subscriptions', label='segment', stacked=True)
fig.show()
```

### Pie / Donut

```python
import pandas as pd
from MatplotLibAPI import fplot_pie_donut

df = pd.read_csv('data/pie.csv')
fig = fplot_pie_donut(df, category='device', value='sessions', donut=True)
fig.show()
```

### Waffle

```python
import pandas as pd
from MatplotLibAPI import fplot_waffle

df = pd.read_csv('data/waffle.csv')
fig = fplot_waffle(df, category='device', value='sessions')
fig.show()
```

### Sankey

```python
import pandas as pd
from MatplotLibAPI import fplot_sankey

df = pd.read_csv('data/sankey.csv')
fig = fplot_sankey(df, source='source', target='target', value='value')
fig.show()
```

### Pivot Table

```python
import pandas as pd
from MatplotLibAPI.Pivot import plot_pivoted_bars

df = pd.read_csv('data/pivot.csv')
ax = plot_pivoted_bars(data=df, label="category", x="date", y="value")
ax.figure.show()
```

### Table

```python
import pandas as pd
from MatplotLibAPI import fplot_table

df = pd.read_csv('data/table.csv')
fig = fplot_table(pd_df=df, cols=["col1", "col2"])
fig.show()
```

### Timeseries Plot

```python
import pandas as pd
from MatplotLibAPI import fplot_timeserie

df = pd.read_csv('data/timeserie.csv')
fig = fplot_timeserie(pd_df=df, label="group", x="date", y="value")
fig.show()
```

### Treemap

```python
import pandas as pd
from MatplotLibAPI import fplot_treemap

df = pd.read_csv('data/treemap.csv')
fig = fplot_treemap(pd_df=df, path="path", values="values")
fig.show()
```

### Sunburst Chart

```python
import pandas as pd
from MatplotLibAPI import fplot_sunburst

df = pd.read_csv('data/sunburst.csv')
fig = fplot_sunburst(df, labels="labels", parents="parents", values="values")
fig.show()
```

### Word Cloud

```python
import pandas as pd
from MatplotLibAPI import fplot_wordcloud

df = pd.read_csv('data/wordcloud.csv')
fig = fplot_wordcloud(df, text_column="word", weight_column="weight")
fig.show()
```
