# Методы и функции

## 📋 Содержание

- [Управление Чатами](#управление-чатами)
- [Управление Группами](#управление-группами)
- [Обработка Событий](#обработка-событий)
- [Управление Сообщениями](#управление-сообщениями)
- [Управление Профилем](#управление-профилем)
- [Управление Пользователями](#управление-пользователями)

## 💬 Управление чатами {#управление-чатами}

### Работа с Каналами

#### `resolve_channel_by_name(name: str) -> bool`

!!! info "Описание"
    Пытается найти канал по его имени.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `name` | `str` | Имя канала для поиска |

**Возвращает**

`bool`: `True` если канал найден, иначе вызывается исключение

**Пример использования**

```python
try:
    found = await client.resolve_channel_by_name("my_channel")
    if found:
        print("Канал найден!")
except Exception as e:
    print(f"Ошибка: {e}")
```

#### `join_channel(link: str) -> bool`

!!! info "Описание"
    Присоединяется к каналу по ссылке.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `link` | `str` | Ссылка на канал |

**Возвращает**

`bool`: `True` если присоединение прошло успешно, иначе будет выброшено исключение

**Пример использования**

```python
success = await client.join_channel("https://max.ru/test_channel")
if success:
    print("Присоединились к каналу")
```

#### `load_members(chat_id: int, marker: int = DEFAULT_MARKER_VALUE, count: int = DEFAULT_CHAT_MEMBERS_LIMIT) -> tuple[list[Member], int | None]`

!!! info "Описание"
    Загружает участников (членов) канала с поддержкой маркера пагинации.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID канала |
| `marker` | `int` | Маркер для пагинации (опционально) |
| `count` | `int` | Количество участников для загрузки (опционально) |

**Возвращает**

`tuple[list[Member], int | None]`: Кортеж со списком объектов `Member` и маркером (`int`) для следующей страницы или `None`.

**Пример использования**

```python
members, marker = await client.load_members(chat_id=123)
for m in members:
    print(m.user_id, m.username)
```

#### `find_members(chat_id: int, query: str) -> tuple[list[Member], int | None]`

!!! info "Описание"
    Поиск участников канала по строке. Заметьте, веб-клиент возвращает ограниченное число результатов — пагинация не всегда доступна.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID канала |
| `query` | `str` | Строка запроса для поиска участников |

**Возвращает**

`tuple[list[Member], int | None]`: Список найденных участников и маркер, если он был возвращен.

**Пример использования**

```python
members, marker = await client.find_members(chat_id=123, query="test")
```

#### `get_chats(chat_ids: list[int]) -> list[Chat]`

!!! info "Описание"
    Получает информацию о нескольких чатах/группах по их ID. Метод возвращает объекты `Chat` и обновляет локальный кэш клиента.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_ids` | `list[int]` | Список ID чатов для получения |

**Возвращает**

`list[Chat]`: Список объектов `Chat` с информацией о чатах

**Пример использования**
```python
chats = await client.get_chats([12345, 67890])
for chat in chats:
    print(chat.title, chat.id)
```

#### `get_chat(chat_id: int) -> Chat`

!!! info "Описание"
    Получает информацию об одном чате (группе/канале) по его ID. Выбрасывает исключение, если чат не найден.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |

**Возвращает**

`Chat`: Объект `Chat` с информацией о чате

**Пример использования**
```python
chat = await client.get_chat(12345)
print(chat.title)
```



## 👥 Управление группами {#управление-группами}

### Создание и Настройка Групп

#### `create_group(name: str, participant_ids: list[int] | None = None, notify: bool = True) -> tuple[Chat, Message] | None`

!!! info "Описание"
    Создает новую группу с указанным названием и начальным списком участников.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `name` | `str` | Название новой группы |
| `participant_ids` | `list[int] | None` | Список ID пользователей для добавления в группу (опционально) |
| `notify` | `bool` | Отправлять ли уведомление. По умолчанию `True` |

**Возвращает**

`tuple[Chat, Message] | None`: Кортеж с объектом созданной группы и системным сообщением, или `None` при ошибке

**Пример использования**
```python
result = await client.create_group(
    name="Моя группа",
    participant_ids=[123456, 789012],
    notify=True
)
if result:
    chat, message = result
    print(f"Создана группа: {chat.title}")
```

### Управление Участниками

#### `invite_users_to_group(chat_id: int, user_ids: list[int], show_history: bool = True) -> bool`

!!! info "Описание"
    Приглашает новых пользователей в существующую группу.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID группы |
| `user_ids` | `list[int]` | Список ID пользователей для приглашения |
| `show_history` | `bool` | Показывать ли историю сообщений приглашенным пользователям. По умолчанию `True` |

#### `remove_users_from_group(chat_id: int, user_ids: list[int], clean_msg_period: int) -> bool`

!!! info "Описание"
    Удаляет пользователей из группы.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID группы |
| `user_ids` | `list[int]` | Список ID пользователей для удаления |
| `clean_msg_period` | `int` | Период удаления сообщений (в днях или другие единицы) |

### Настройка Группы

#### `change_group_settings(chat_id: int, all_can_pin_message: bool | None = None, only_owner_can_change_icon_title: bool | None = None, only_admin_can_add_member: bool | None = None, only_admin_can_call: bool | None = None, members_can_see_private_link: bool | None = None) -> None`

!!! info "Описание"
    Изменяет настройки группы.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID группы |
| `all_can_pin_message` | `bool | None` | Все ли могут закреплять сообщения (опционально) |
| `only_owner_can_change_icon_title` | `bool | None` | Только владелец может менять иконку и название (опционально) |
| `only_admin_can_add_member` | `bool | None` | Только администраторы могут добавлять участников (опционально) |
| `only_admin_can_call` | `bool | None` | Только администраторы могут звонить (опционально) |
| `members_can_see_private_link` | `bool | None` | Участники видят приватную ссылку (опционально) |

**Возвращает**

`None`: Метод не возвращает значения, исключение выбрасывается при ошибке

#### `change_group_profile(chat_id: int, name: str | None, description: str | None = None) -> None`

!!! info "Описание"
    Изменяет название и/или описание группы.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID группы |
| `name` | `str | None` | Новое название группы |
| `description` | `str | None` | Новое описание группы (опционально) |

**Возвращает**

`None`: Метод не возвращает значения, исключение выбрасывается при ошибке

### Присоединение к Группе

#### `join_group(link: str) -> Chat | None`

!!! info "Описание"
    Присоединяется к группе по пригласительной ссылке.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `link` | `str` | Ссылка-приглашение в группу |

**Пример использования**
```python
chat = await client.join_group("https://max.ru/join/group123")
if chat:
    print(f"Присоединились к группе: {chat.title}")
```

#### `rework_invite_link(chat_id: int) -> Chat`

!!! info "Описание"
    Пересоздает ссылку для приглашения в группу (инвалидирует старую ссылку).

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID группы |

**Возвращает**

`Chat`: Обновленный объект чата с новой ссылкой приглашения

**Пример использования**
```python
chat = await client.rework_invite_link(chat_id=123456)
print(f"Новая ссылка создана для группы: {chat.title}")
```

## 🎯 Обработка событий {#обработка-событий}

### Обработчики Событий

#### `on_message(filter: Filter | None = None) -> Callable`

!!! info "Описание"
    Декоратор для обработки входящих сообщений. Позволяет установить фильтры для обработки только определенных типов сообщений.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `filter` | `Filter | None` | Фильтр для сообщений (опционально) |

**Примеры использования**

```python
# Обработка всех сообщений
@client.on_message()
async def handle_message(message: Message):
    print(f"Получено сообщение: {message.text}")

# Обработка только текстовых сообщений
@client.on_message(Filter(chat_id=0))
async def handle_text(message: Message):
    print(f"Получено текстовое сообщение из избранного: {message.text}")
```

#### `on_message_edit(filter: Filter | None = None) -> Callable`

!!! info "Описание"
    Декоратор для установки обработчика отредактированных сообщений.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `filter` | `Filter | None` | Фильтр для обработки сообщений (опционально) |

#### `on_message_delete(filter: Filter | None = None) -> Callable`

!!! info "Описание"
    Декоратор для установки обработчика удаленных сообщений.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `filter` | `Filter | None` | Фильтр для обработки сообщений (опционально) |

#### `on_reaction_change(handler: Callable[[str, int, ReactionInfo], Any | Awaitable[Any]]) -> Callable`

!!! info "Описание"
    Устанавливает обработчик изменения реакций на сообщения.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `handler` | `Callable` | Функция с аргументами `(message_id: str, chat_id: int, reaction_info: ReactionInfo)` |

#### `on_chat_update(handler: Callable[[Chat], Any | Awaitable[Any]]) -> Callable`

!!! info "Описание"
    Устанавливает обработчик обновления информации о чате.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `handler` | `Callable` | Функция с аргументом `(chat: Chat)` |

#### `on_start() -> Callable`

!!! info "Описание"
    Декоратор для обработки события запуска клиента. Выполняется один раз при успешном запуске и авторизации клиента.

**Пример использования**

```python
@client.on_start
async def handle_start():
    print("Клиент успешно запущен")
    # Инициализация при запуске
    await client.send_message(
        chat_id=admin_chat,
        text="Бот запущен и готов к работе!",
        notify=True
    )
```

!!! tip "Совет"
    Вы можете использовать `on_start` для выполнения начальной настройки, загрузки данных или отправки уведомлений при запуске бота.

## 📨 Управление сообщениями {#управление-сообщениями}

### Отправка и Редактирование

#### `send_message(text: str, chat_id: int, notify: bool, attachment: Photo | File | Video | None = None, attachments: list[Photo | File | Video] | None = None, reply_to: int | None = None, use_queue: bool = False) -> Message | None`

!!! info "Описание"
    Отправляет сообщение в чат с возможностью прикрепления медиафайлов и ответа на другие сообщения.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `text` | `str` | Текст сообщения |
| `chat_id` | `int` | ID чата для отправки |
| `notify` | `bool` | Отправлять ли уведомление |
| `attachment` | `Photo | File | Video | None` | Один файл/фото/видео для прикрепления (опционально) |
| `attachments` | `list[Photo | File | Video] | None` | Список файлов/фото/видео для прикрепления (опционально) |
| `reply_to` | `int | None` | ID сообщения, на которое отвечаем (опционально) |
| `use_queue` | `bool` | Добавить в очередь вместо немедленной отправки. По умолчанию `False` |

**Возвращает**

`Message | None`: Объект отправленного сообщения или `None` при ошибке

**Пример использования**
```python
# Простое сообщение
msg = await client.send_message(
    chat_id=123456,
    text="Привет!",
    notify=True
)

# Сообщение с фото
from pymax import Photo
photo = Photo("/photos/photo.png")
msg = await client.send_message(
    chat_id=123456,
    text="Смотри, какое фото!",
    attachment=photo,
    notify=True
)
```

#### `edit_message(chat_id: int, message_id: int, text: str, attachment: Photo | File | Video | None = None, attachments: list[Photo | File | Video] | None = None, use_queue: bool = False) -> Message | None`

!!! info "Описание"
    Редактирует текст и/или вложения существующего сообщения.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_id` | `int` | ID сообщения |
| `text` | `str` | Новый текст |
| `attachment` | `Photo | File | Video | None` | Одно фото/файл/видео для прикрепления (опционально) |
| `attachments` | `list[Photo | File | Video] | None` | Список фото/файлов/видео для прикрепления (опционально) |
| `use_queue` | `bool` | Добавить в очередь вместо немедленной отправки. По умолчанию `False` |

**Возвращает**

`Message | None`: Обновленный объект сообщения или `None` при ошибке

### Управление Историей

#### `fetch_history(chat_id: int, from_time: int | None = None, forward: int = 0, backward: int = 200) -> list[Message] | None`

!!! info "Описание"
    Получает историю сообщений чата.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `from_time` | `int | None` | Временная метка отсчета |
| `forward` | `int` | Кол-во сообщений после from_time |
| `backward` | `int` | Кол-во сообщений до from_time |

### Работа с Медиафайлами

#### `get_video_by_id(chat_id: int, message_id: int, video_id: int) -> VideoRequest | None`

!!! info "Описание"
    Получает информацию о видео и ссылку для скачивания.

**Возвращает**
- `VideoRequest` содержит:
  - `external` - API ссылка
  - `cache` - Статус кэширования
  - `url` - Ссылка на видео

#### `get_file_by_id(chat_id: int, message_id: int, file_id: int) -> FileRequest | None`

!!! info "Описание"
    Получает информацию о файле и ссылку для скачивания.

**Возвращает**
- `FileRequest` содержит:
  - `unsafe` - Статус проверки безопасности
  - `url` - Ссылка на файл

### Модерация Сообщений

#### `delete_message(chat_id: int, message_ids: list[int], for_me: bool, use_queue: bool = False) -> bool`

!!! info "Описание"
    Удаляет сообщения из чата.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_ids` | `list[int]` | Список ID сообщений |
| `for_me` | `bool` | Удалить только для себя |
| `use_queue` | `bool` | Добавить в очередь вместо немедленной отправки. По умолчанию `False` |

#### `pin_message(chat_id: int, message_id: int, notify_pin: bool) -> bool`

!!! info "Описание"
    Закрепляет сообщение в чате.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_id` | `int` | ID сообщения |
| `notify_pin` | `bool` | Уведомить о закреплении |

### Реакции на Сообщения

#### `add_reaction(chat_id: int, message_id: str, reaction: str) -> ReactionInfo | None`

!!! info "Описание"
    Добавляет эмодзи-реакцию к сообщению.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_id` | `str` | ID сообщения |
| `reaction` | `str` | Эмодзи реакции |

**Пример использования**
```python
result = await client.add_reaction(
    chat_id=123456,
    message_id="msg_123",
    reaction="👍"
)
if result:
    print(f"Всего реакций: {result.total_count}")
```

#### `get_reactions(chat_id: int, message_ids: list[str]) -> dict[str, ReactionInfo] | None`

!!! info "Описание"
    Получает информацию о реакциях на несколько сообщений.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_ids` | `list[str]` | Список ID сообщений |

#### `remove_reaction(chat_id: int, message_id: str) -> ReactionInfo | None`

!!! info "Описание"
    Удаляет реакцию текущего пользователя с сообщения.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `chat_id` | `int` | ID чата |
| `message_id` | `str` | ID сообщения |

## 👤 Управление профилем {#управление-профилем}

### Персональные Данные

#### `change_profile(first_name: str, last_name: str | None = None, description: str | None = None) -> bool`

!!! info "Описание"
    Изменяет информацию в профиле пользователя.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `first_name` | `str` | Имя (обязательно) |
| `last_name` | `str | None` | Фамилия (опционально) |
| `description` | `str | None` | Описание/Биография (опционально) |

!!! warning "Примечание"
    Метод **не поддерживает** изменение фотографии профиля (параметр `photo` не реализован)

**Пример использования**
```python
# Обновление имени и описания
success = await client.change_profile(
    first_name="Иван",
    description="Python разработчик"
)

# Обновление только фамилии
success = await client.change_profile(
    first_name="Иван",
    last_name="Петров"
)
```

### Управление папками

#### `create_folder(title: str, chat_include: list[int], filters: list[Any] | None = None) -> FolderUpdate`

!!! info "Описание"
    Создаёт новую пользовательскую папку для организации чатов по заданным правилам.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `title` | `str` | Название папки |
| `chat_include` | `list[int]` | Список ID чатов для включения в папку |
| `filters` | `list[Any] | None` | Список дополнительных фильтров (API-зависимый) |

**Возвращает**

`FolderUpdate`: Обновлённые данные о папке и порядке папок

**Пример использования**
```python
result = await client.create_folder(title="Friends", chat_include=[123, 456])
print(result.folder.id, result.folder.title)
```

#### `get_folders(folder_sync: int = 0) -> FolderList`

!!! info "Описание"
    Возвращает список папок пользователя и данные синхронизации.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `folder_sync` | `int` | Маркер синхронизации (опционально) |

**Возвращает**

`FolderList`: Список папок с порядком и маркером синхронизации

**Пример использования**
```python
folders = await client.get_folders()
for f in folders.folders:
    print(f.title, f.id)
```

#### `update_folder(folder_id: str, title: str, chat_include: list[int] | None = None, filters: list[Any] | None = None, options: list[Any] | None = None) -> FolderUpdate`

!!! info "Описание"
    Обновляет существующую папку: название, список включённых чатов, фильтры и опции.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `folder_id` | `str` | Уникальный ID папки |
| `title` | `str` | Новое название папки |
| `chat_include` | `list[int] | None` | Список ID чатов для папки (опционально) |
| `filters` | `list[Any] | None` | Обновлённые фильтры папки (опционально) |
| `options` | `list[Any] | None` | Параметры папки (опционально) |

**Возвращает**

`FolderUpdate`: Результат обновления папки

**Пример использования**
```python
updated = await client.update_folder(folder_id=result.folder.id, title="Best friends", chat_include=[123])
print(updated.folder.title)
```

#### `delete_folder(folder_id: str) -> FolderUpdate`

!!! info "Описание"
    Удаляет папку и возвращает обновление порядка папок.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `folder_id` | `str` | Уникальный ID папки |

**Возвращает**

`FolderUpdate`: Результат удаления

**Пример использования**
```python
await client.delete_folder(folder_id=result.folder.id)
```

## 👥 Управление пользователями {#управление-пользователями}

### Получение Информации о Пользователях

#### `get_cached_user(user_id: int) -> User | None`

!!! info "Описание"
    Быстро получает информацию о пользователе из локального кэша.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `user_id` | `int` | ID пользователя |

#### `get_user(user_id: int) -> User | None`

!!! info "Описание"
    Получает актуальную информацию о пользователе.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `user_id` | `int` | ID пользователя |

#### `get_users(user_ids: list[int]) -> list[User]`

!!! info "Описание"
    Получает информацию о нескольких пользователях за один запрос.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `user_ids` | `list[int]` | Список ID пользователей |

#### `get_chat_id(first_user_id: int, second_user_id: int) -> int`

!!! info "Описание"
    Возвращает идентификатор личного диалога между двумя пользователями.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `first_user_id` | `int` | ID первого пользователя |
| `second_user_id` | `int` | ID второго пользователя |

**Возвращает**

`int`: Вычисленный ID диалога (XOR от двух ID пользователей)

#### `fetch_users(user_ids: list[int]) -> list[User]`

!!! info "Описание"
    Принудительно загружает актуальную информацию о пользователях с сервера.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `user_ids` | `list[int]` | Список ID пользователей |

### Поиск Пользователей

#### `search_by_phone(phone: str) -> User | None`

!!! info "Описание"
    Ищет пользователя по номеру телефона.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `phone` | `str` | Номер телефона |

### Контакты

#### `add_contact(contact_id: int) -> Contact`

!!! info "Описание"
    Добавляет пользователя в список контактов по его ID.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `contact_id` | `int` | ID пользователя для добавления в контакты |

**Возвращает**

`Contact`: Объект контакта после успешного добавления

**Пример использования**

```python
contact = await client.add_contact(123456)
print(f"Добавлен контакт: {contact.id}")
```

#### `remove_contact(contact_id: int) -> bool`

!!! info "Описание"
    Удаляет пользователя из списка контактов.

**Параметры**

| Параметр | Тип | Описание |
|----------|-----|----------|
| `contact_id` | `int` | ID пользователя для удаления из контактов |

**Возвращает**

`bool`: `True`, если удаление прошло успешно

**Пример использования**

```python
success = await client.remove_contact(123456)
if success:
    print("Контакт удалён")
```

### Управление Сессиями

#### `get_sessions() -> list[Session]`

!!! info "Описание"
    Получает список всех активных сессий текущего пользователя.

**Пример использования**
```python
sessions = await client.get_sessions()
for session in sessions:
    print(f"Сессия: {session.client} из {session.location}")
```

## 📝 Рекомендации по использованию

### Обработка Ошибок

!!! tip "Проверка результатов"
    Всегда проверяйте возвращаемые значения на None:
    ```python
    result = await client.send_message(
        chat_id=123456,
        text="Тестовое сообщение",
        notify=True
    )
    if result is None:
        print("Не удалось отправить сообщение")
    ```

!!! warning "Обработка исключений"
    Используйте try-except для обработки возможных ошибок:
    ```python
    try:
        await client.pin_message(
            chat_id=123456,
            message_id=789,
            notify_pin=True
        )
    except Exception as e:
        print(f"Ошибка при закреплении: {e}")
    ```


### Оптимизация Запросов

!!! tip "Массовые операции"
    Используйте методы для работы с множеством объектов:
    - `get_users()` вместо множества вызовов `get_user()`
    - `delete_message()` с списком ID вместо удаления по одному
