#!/usr/bin/env python
"""
Color palettes from the NCAR Command Language (NCL)

Colormaps from PyNGL, the Python interface to the NCAR Command Language (NCL)
  http://www.pyngl.ucar.edu/Graphics/color_table_gallery.shtml

.. moduleauthor:: Stephan Thober and Matthias Cuntz

History
   * Written Mar 2014, Stephan Thober
   * Made dictionaries with color palettes,
     Mar 2015, Matthias Cuntz
   * Ported to pyjams, Nov 2021, Matthias Cuntz
   * flake8 compatible, Nov 2021, Matthias Cuntz
   * Removed double meteo swiss maps, Nov 2021, Matthias Cuntz
   * Removed black and white at beginning of meteo swiss maps,
     Nov 2021, Matthias Cuntz
   * Prepend ncl to all palette names, Nov 2021, Matthias Cuntz
   * Use : instead of _ in colormap names, Oct 2024, Matthias Cuntz

"""


__all__ = ['ncl_large', 'ncl_small', 'ncl_meteo_swiss']


# NCL color maps

# large color maps (more than 50 colors)
ncl_large = {
    'ncl:bkblaqgryeorreviwh200': [
        (1., 1., 1.), (0., 0., 0.), (0., 0., 0.), (0., 0., 0.0392),
        (0., 0., 0.0784), (0., 0., 0.1216), (0., 0., 0.1608),
        (0., 0., 0.2), (0., 0., 0.2392), (0., 0., 0.2784),
        (0., 0., 0.3216), (0., 0., 0.3608), (0., 0., 0.4),
        (0., 0., 0.4392), (0., 0., 0.4784), (0., 0., 0.5216),
        (0., 0., 0.5608), (0., 0., 0.6), (0., 0., 0.6392),
        (0., 0., 0.6784), (0., 0., 0.7216), (0., 0., 0.7608),
        (0., 0., 0.8), (0., 0., 0.8392), (0., 0., 0.8784),
        (0., 0., 0.9216), (0., 0., 0.9608), (0., 0., 1.),
        (0., 0.0392, 1.), (0., 0.0784, 1.), (0., 0.1216, 1.),
        (0., 0.1608, 1.), (0., 0.2, 1.), (0., 0.2392, 1.),
        (0., 0.2784, 1.), (0., 0.3216, 1.), (0., 0.3608, 1.),
        (0., 0.4, 1.), (0., 0.4392, 1.), (0., 0.4784, 1.),
        (0., 0.5216, 1.), (0., 0.5608, 1.), (0., 0.6, 1.),
        (0., 0.6392, 1.), (0., 0.6784, 1.), (0., 0.7216, 1.),
        (0., 0.7608, 1.), (0., 0.8, 1.), (0., 0.8392, 1.),
        (0., 0.8784, 1.), (0., 0.9216, 1.), (0., 0.9608, 1.),
        (0., 1., 1.), (0., 0.9882, 0.9608), (0.0039, 0.9765, 0.9255),
        (0.0039, 0.9647, 0.8863), (0.0039, 0.9529, 0.851),
        (0.0078, 0.9412, 0.8118), (0.0078, 0.9294, 0.7725),
        (0.0078, 0.9176, 0.7373), (0.0118, 0.9059, 0.698),
        (0.0118, 0.8941, 0.6627), (0.0118, 0.8824, 0.6235),
        (0.0157, 0.8706, 0.5843), (0.0157, 0.8588, 0.549),
        (0.0157, 0.8431, 0.5098), (0.0157, 0.8314, 0.4745),
        (0.0196, 0.8196, 0.4353), (0.0196, 0.8078, 0.3961),
        (0.0196, 0.7961, 0.3608), (0.0235, 0.7843, 0.3216),
        (0.0235, 0.7725, 0.2863), (0.0235, 0.7608, 0.2471),
        (0.0275, 0.749, 0.2078), (0.0275, 0.7373, 0.1725),
        (0.0275, 0.7255, 0.1333), (0.0314, 0.7137, 0.098),
        (0.0314, 0.702, 0.0588), (0.0706, 0.7137, 0.0549),
        (0.1098, 0.7255, 0.0549), (0.149, 0.7373, 0.051),
        (0.1882, 0.749, 0.051), (0.2235, 0.7608, 0.0471),
        (0.2627, 0.7725, 0.0431), (0.302, 0.7843, 0.0431),
        (0.3412, 0.7961, 0.0392), (0.3804, 0.8078, 0.0392),
        (0.4196, 0.8196, 0.0353), (0.4588, 0.8314, 0.0314),
        (0.498, 0.8431, 0.0314), (0.5333, 0.8588, 0.0275),
        (0.5725, 0.8706, 0.0275), (0.6118, 0.8824, 0.0235),
        (0.651, 0.8941, 0.0196), (0.6902, 0.9059, 0.0196),
        (0.7294, 0.9176, 0.0157), (0.7686, 0.9294, 0.0157),
        (0.8078, 0.9412, 0.0118), (0.8431, 0.9529, 0.0078),
        (0.8824, 0.9647, 0.0078), (0.9216, 0.9765, 0.0039),
        (0.9608, 0.9882, 0.0039), (1., 1., 0.), (1., 0.9843, 0.),
        (1., 0.9725, 0.), (1., 0.9569, 0.), (1., 0.9412, 0.),
        (1., 0.9255, 0.), (1., 0.9137, 0.), (1., 0.898, 0.),
        (1., 0.8824, 0.), (1., 0.8667, 0.), (1., 0.8549, 0.),
        (1., 0.8392, 0.), (1., 0.8235, 0.), (1., 0.8078, 0.),
        (1., 0.7961, 0.), (1., 0.7804, 0.), (1., 0.7647, 0.),
        (1., 0.749, 0.), (1., 0.7373, 0.), (1., 0.7216, 0.),
        (1., 0.7059, 0.), (1., 0.6902, 0.), (1., 0.6784, 0.),
        (1., 0.6627, 0.), (1., 0.6471, 0.), (1., 0.6196, 0.),
        (1., 0.5961, 0.), (1., 0.5686, 0.), (1., 0.5451, 0.),
        (1., 0.5176, 0.), (1., 0.4902, 0.), (1., 0.4667, 0.),
        (1., 0.4392, 0.), (1., 0.4157, 0.), (1., 0.3882, 0.),
        (1., 0.3608, 0.), (1., 0.3373, 0.), (1., 0.3098, 0.),
        (1., 0.2863, 0.), (1., 0.2588, 0.), (1., 0.2314, 0.),
        (1., 0.2078, 0.), (1., 0.1804, 0.), (1., 0.1569, 0.),
        (1., 0.1294, 0.), (1., 0.102, 0.), (1., 0.0784, 0.),
        (1., 0.051, 0.), (1., 0.0275, 0.), (1., 0., 0.),
        (0.9725, 0., 0.0196), (0.949, 0., 0.0431), (0.9216, 0., 0.0627),
        (0.8941, 0., 0.0863), (0.8667, 0., 0.1059), (0.8431, 0., 0.1294),
        (0.8157, 0., 0.149), (0.7882, 0., 0.1725), (0.7647, 0., 0.1922),
        (0.7373, 0., 0.2118), (0.7098, 0., 0.2353), (0.6824, 0., 0.2549),
        (0.6588, 0., 0.2784), (0.6314, 0., 0.298), (0.6039, 0., 0.3216),
        (0.5765, 0., 0.3412), (0.5529, 0., 0.3608), (0.5255, 0., 0.3843),
        (0.498, 0., 0.4039), (0.4745, 0., 0.4275), (0.4471, 0., 0.4471),
        (0.4196, 0., 0.4706), (0.3922, 0., 0.4902), (0.3686, 0., 0.5137),
        (0.3412, 0., 0.5333), (0.3686, 0.0392, 0.5529),
        (0.3922, 0.0784, 0.5725), (0.4196, 0.1216, 0.5882),
        (0.4471, 0.1608, 0.6078), (0.4745, 0.2, 0.6275),
        (0.498, 0.2392, 0.6471), (0.5255, 0.2784, 0.6627),
        (0.5529, 0.3216, 0.6824), (0.5765, 0.3608, 0.702),
        (0.6039, 0.4, 0.7216), (0.6314, 0.4392, 0.7373),
        (0.6588, 0.4784, 0.7569), (0.6824, 0.5216, 0.7765),
        (0.7098, 0.5608, 0.7961), (0.7373, 0.6, 0.8118),
        (0.7647, 0.6392, 0.8314), (0.7882, 0.6784, 0.851),
        (0.8157, 0.7216, 0.8706), (0.8431, 0.7608, 0.8863),
        (0.8667, 0.8, 0.9059), (0.8941, 0.8392, 0.9255),
        (0.9216, 0.8784, 0.9451), (0.949, 0.9216, 0.9608),
        (0.9725, 0.9608, 0.9804), (1., 1., 1.)],
    'ncl:blaqgryeorre': [
        (1., 1., 1.), (0., 0., 0.), (0., 0., 1.), (0., 0.0824, 1.),
        (0., 0.1686, 1.), (0., 0.251, 1.), (0., 0.3333, 1.),
        (0., 0.4157, 1.), (0., 0.502, 1.), (0., 0.5843, 1.),
        (0., 0.6667, 1.), (0., 0.749, 1.), (0., 0.8353, 1.),
        (0., 0.9176, 1.), (0., 1., 1.), (0.0471, 1., 1.),
        (0.0941, 1., 1.), (0.1373, 1., 1.), (0.1843, 1., 1.),
        (0.2314, 1., 1.), (0.2784, 1., 1.), (0.3216, 1., 1.),
        (0.3686, 1., 1.), (0.4157, 1., 1.), (0.4627, 1., 1.),
        (0.5059, 1., 1.), (0.5529, 1., 1.), (0.6, 1., 1.),
        (0.549, 1., 0.9176), (0.502, 1., 0.8353), (0.451, 1., 0.749),
        (0.4, 1., 0.6667), (0.349, 1., 0.5843), (0.302, 1., 0.502),
        (0.251, 1., 0.4157), (0.2, 1., 0.3333), (0.149, 1., 0.251),
        (0.102, 1., 0.1686), (0.051, 1., 0.0824), (0., 1., 0.),
        (0.0627, 1., 0.), (0.1216, 1., 0.), (0.1843, 1., 0.),
        (0.2471, 1., 0.), (0.3059, 1., 0.), (0.3686, 1., 0.),
        (0.4314, 1., 0.), (0.4941, 1., 0.), (0.5529, 1., 0.),
        (0.6157, 1., 0.), (0.6784, 1., 0.), (0.7373, 1., 0.),
        (0.8, 1., 0.), (0.8118, 1., 0.), (0.8235, 1., 0.),
        (0.8392, 1., 0.), (0.851, 1., 0.), (0.8627, 1., 0.),
        (0.8745, 1., 0.), (0.8863, 1., 0.), (0.902, 1., 0.),
        (0.9137, 1., 0.), (0.9255, 1., 0.), (0.9373, 1., 0.),
        (0.949, 1., 0.), (0.9608, 1., 0.), (0.9765, 1., 0.),
        (0.9882, 1., 0.), (1., 1., 0.), (1., 0.9725, 0.),
        (1., 0.9412, 0.), (1., 0.9137, 0.), (1., 0.8824, 0.),
        (1., 0.8549, 0.), (1., 0.8235, 0.), (1., 0.7961, 0.),
        (1., 0.7647, 0.), (1., 0.7373, 0.), (1., 0.7059, 0.),
        (1., 0.6784, 0.), (1., 0.6471, 0.), (1., 0.6196, 0.),
        (1., 0.5882, 0.), (1., 0.5608, 0.), (1., 0.5294, 0.),
        (1., 0.502, 0.), (1., 0.4706, 0.), (1., 0.4392, 0.),
        (1., 0.4078, 0.), (1., 0.3765, 0.), (1., 0.3451, 0.),
        (1., 0.3137, 0.), (1., 0.2824, 0.), (1., 0.251, 0.),
        (1., 0.2196, 0.), (1., 0.1882, 0.), (1., 0.1569, 0.),
        (1., 0.1255, 0.), (1., 0.0941, 0.), (1., 0.0627, 0.),
        (1., 0.0314, 0.), (1., 0., 0.)],
    'ncl:blaqgryeorrevi200': [
        (1., 1., 1.), (0., 0., 0.), (0., 0., 1.), (0., 0.0314, 1.),
        (0., 0.0588, 1.), (0., 0.0902, 1.), (0., 0.1216, 1.),
        (0., 0.1529, 1.), (0., 0.1804, 1.), (0., 0.2118, 1.),
        (0., 0.2431, 1.), (0., 0.2745, 1.), (0., 0.302, 1.),
        (0., 0.3333, 1.), (0., 0.3647, 1.), (0., 0.3922, 1.),
        (0., 0.4235, 1.), (0., 0.4549, 1.), (0., 0.4863, 1.),
        (0., 0.5137, 1.), (0., 0.5451, 1.), (0., 0.5765, 1.),
        (0., 0.6078, 1.), (0., 0.6353, 1.), (0., 0.6667, 1.),
        (0., 0.698, 1.), (0., 0.7255, 1.), (0., 0.7569, 1.),
        (0., 0.7882, 1.), (0., 0.8196, 1.), (0., 0.8471, 1.),
        (0., 0.8784, 1.), (0., 0.9098, 1.), (0., 0.9412, 1.),
        (0., 0.9686, 1.), (0., 1., 1.), (0., 0.9922, 0.9725),
        (0., 0.9804, 0.9412), (0.0039, 0.9725, 0.9137),
        (0.0039, 0.9647, 0.8863), (0.0039, 0.9529, 0.8588),
        (0.0039, 0.9451, 0.8275), (0.0078, 0.9373, 0.8),
        (0.0078, 0.9294, 0.7725), (0.0078, 0.9176, 0.7451),
        (0.0078, 0.9098, 0.7137), (0.0118, 0.902, 0.6863),
        (0.0118, 0.8902, 0.6588), (0.0118, 0.8824, 0.6275),
        (0.0118, 0.8745, 0.6), (0.0157, 0.8627, 0.5725),
        (0.0157, 0.8549, 0.5451), (0.0157, 0.8471, 0.5137),
        (0.0157, 0.8392, 0.4863), (0.0196, 0.8275, 0.4588),
        (0.0196, 0.8196, 0.4314), (0.0196, 0.8118, 0.4),
        (0.0196, 0.8, 0.3725), (0.0235, 0.7922, 0.3451),
        (0.0235, 0.7843, 0.3137), (0.0235, 0.7725, 0.2863),
        (0.0235, 0.7647, 0.2588), (0.0275, 0.7569, 0.2314),
        (0.0275, 0.749, 0.2), (0.0275, 0.7373, 0.1725),
        (0.0275, 0.7294, 0.1451), (0.0314, 0.7216, 0.1176),
        (0.0314, 0.7098, 0.0863), (0.0314, 0.702, 0.0588),
        (0.0588, 0.7098, 0.0588), (0.0902, 0.7176, 0.0549),
        (0.1176, 0.7294, 0.0549), (0.1451, 0.7373, 0.051),
        (0.1725, 0.7451, 0.051), (0.2039, 0.7529, 0.0471),
        (0.2314, 0.7647, 0.0471), (0.2588, 0.7725, 0.0431),
        (0.2863, 0.7804, 0.0431), (0.3176, 0.7882, 0.0431),
        (0.3451, 0.8, 0.0392), (0.3725, 0.8078, 0.0392),
        (0.4, 0.8157, 0.0353), (0.4314, 0.8235, 0.0353),
        (0.4588, 0.8353, 0.0314), (0.4863, 0.8431, 0.0314),
        (0.5176, 0.851, 0.0314), (0.5451, 0.8588, 0.0275),
        (0.5725, 0.8667, 0.0275), (0.6, 0.8784, 0.0235),
        (0.6314, 0.8863, 0.0235), (0.6588, 0.8941, 0.0196),
        (0.6863, 0.902, 0.0196), (0.7137, 0.9137, 0.0157),
        (0.7451, 0.9216, 0.0157), (0.7725, 0.9294, 0.0157),
        (0.8, 0.9373, 0.0118), (0.8275, 0.949, 0.0118),
        (0.8588, 0.9569, 0.0078), (0.8863, 0.9647, 0.0078),
        (0.9137, 0.9725, 0.0039), (0.9412, 0.9843, 0.0039),
        (0.9725, 0.9922, 0.), (1., 1., 0.), (1., 0.9882, 0.),
        (1., 0.9804, 0.), (1., 0.9686, 0.), (1., 0.9569, 0.),
        (1., 0.9451, 0.), (1., 0.9373, 0.), (1., 0.9255, 0.),
        (1., 0.9137, 0.), (1., 0.902, 0.), (1., 0.8941, 0.),
        (1., 0.8824, 0.), (1., 0.8706, 0.), (1., 0.8627, 0.),
        (1., 0.851, 0.), (1., 0.8392, 0.), (1., 0.8275, 0.),
        (1., 0.8196, 0.), (1., 0.8078, 0.), (1., 0.7961, 0.),
        (1., 0.7843, 0.), (1., 0.7765, 0.), (1., 0.7647, 0.),
        (1., 0.7529, 0.), (1., 0.7451, 0.), (1., 0.7333, 0.),
        (1., 0.7216, 0.), (1., 0.7098, 0.), (1., 0.702, 0.),
        (1., 0.6902, 0.), (1., 0.6784, 0.), (1., 0.6667, 0.),
        (1., 0.6588, 0.), (1., 0.6471, 0.), (1., 0.6275, 0.),
        (1., 0.6078, 0.), (1., 0.5882, 0.), (1., 0.5686, 0.),
        (1., 0.549, 0.), (1., 0.5294, 0.), (1., 0.5098, 0.),
        (1., 0.4902, 0.), (1., 0.4706, 0.), (1., 0.451, 0.),
        (1., 0.4314, 0.), (1., 0.4118, 0.), (1., 0.3922, 0.),
        (1., 0.3725, 0.), (1., 0.3529, 0.), (1., 0.3333, 0.),
        (1., 0.3137, 0.), (1., 0.2941, 0.), (1., 0.2745, 0.),
        (1., 0.2549, 0.), (1., 0.2353, 0.), (1., 0.2157, 0.),
        (1., 0.1961, 0.), (1., 0.1765, 0.), (1., 0.1569, 0.),
        (1., 0.1373, 0.), (1., 0.1176, 0.), (1., 0.098, 0.),
        (1., 0.0784, 0.), (1., 0.0588, 0.), (1., 0.0392, 0.),
        (1., 0.0196, 0.), (1., 0., 0.), (0.9804, 0., 0.0157),
        (0.9608, 0., 0.0314), (0.9412, 0., 0.0471), (0.9216, 0., 0.0627),
        (0.902, 0., 0.0824), (0.8784, 0., 0.098), (0.8588, 0., 0.1137),
        (0.8392, 0., 0.1294), (0.8196, 0., 0.1451), (0.8, 0., 0.1608),
        (0.7804, 0., 0.1765), (0.7608, 0., 0.1922), (0.7412, 0., 0.2118),
        (0.7216, 0., 0.2275), (0.702, 0., 0.2431), (0.6824, 0., 0.2588),
        (0.6588, 0., 0.2745), (0.6392, 0., 0.2902), (0.6196, 0., 0.3059),
        (0.6, 0., 0.3216), (0.5804, 0., 0.3412), (0.5608, 0., 0.3569),
        (0.5412, 0., 0.3725), (0.5216, 0., 0.3882), (0.502, 0., 0.4039),
        (0.4824, 0., 0.4196), (0.4627, 0., 0.4353), (0.4392, 0., 0.451),
        (0.4196, 0., 0.4706), (0.4, 0., 0.4863), (0.3804, 0., 0.502),
        (0.3608, 0., 0.5176), (0.3412, 0., 0.5333)],
    'ncl:blgryeorrevi200': [
        (1., 1., 1.), (0., 0., 0.), (0., 0., 1.), (0., 0.0157, 0.9765),
        (0., 0.0353, 0.9529), (0.0039, 0.051, 0.9294),
        (0.0039, 0.0706, 0.9059), (0.0039, 0.0863, 0.8824),
        (0.0039, 0.1059, 0.8588), (0.0039, 0.1216, 0.8353),
        (0.0078, 0.1412, 0.8118), (0.0078, 0.1569, 0.7882),
        (0.0078, 0.1765, 0.7647), (0.0078, 0.1922, 0.7412),
        (0.0078, 0.2118, 0.7176), (0.0118, 0.2275, 0.6941),
        (0.0118, 0.2471, 0.6706), (0.0118, 0.2627, 0.6471),
        (0.0118, 0.2824, 0.6235), (0.0118, 0.298, 0.6),
        (0.0157, 0.3176, 0.5765), (0.0157, 0.3333, 0.5529),
        (0.0157, 0.3529, 0.5294), (0.0157, 0.3686, 0.5059),
        (0.0157, 0.3843, 0.4824), (0.0196, 0.4039, 0.4588),
        (0.0196, 0.4196, 0.4353), (0.0196, 0.4392, 0.4118),
        (0.0196, 0.4549, 0.3882), (0.0196, 0.4745, 0.3647),
        (0.0235, 0.4902, 0.3412), (0.0235, 0.5098, 0.3176),
        (0.0235, 0.5255, 0.2941), (0.0235, 0.5451, 0.2706),
        (0.0235, 0.5608, 0.2471), (0.0275, 0.5804, 0.2235),
        (0.0275, 0.5961, 0.2), (0.0275, 0.6157, 0.1765),
        (0.0275, 0.6314, 0.1529), (0.0275, 0.651, 0.1294),
        (0.0314, 0.6667, 0.1059), (0.0314, 0.6863, 0.0824),
        (0.0314, 0.702, 0.0588), (0.0549, 0.7098, 0.0588),
        (0.0784, 0.7176, 0.0549), (0.1059, 0.7255, 0.0549),
        (0.1294, 0.7333, 0.0549), (0.1529, 0.7412, 0.051),
        (0.1765, 0.7451, 0.051), (0.2, 0.7529, 0.0471),
        (0.2235, 0.7608, 0.0471), (0.251, 0.7686, 0.0471),
        (0.2745, 0.7765, 0.0431), (0.298, 0.7843, 0.0431),
        (0.3216, 0.7922, 0.0431), (0.3451, 0.8, 0.0392),
        (0.3686, 0.8078, 0.0392), (0.3961, 0.8157, 0.0353),
        (0.4196, 0.8196, 0.0353), (0.4431, 0.8275, 0.0353),
        (0.4667, 0.8353, 0.0314), (0.4902, 0.8431, 0.0314),
        (0.5176, 0.851, 0.0314), (0.5412, 0.8588, 0.0275),
        (0.5647, 0.8667, 0.0275), (0.5882, 0.8745, 0.0235),
        (0.6118, 0.8824, 0.0235), (0.6353, 0.8902, 0.0235),
        (0.6627, 0.8941, 0.0196), (0.6863, 0.902, 0.0196),
        (0.7098, 0.9098, 0.0196), (0.7333, 0.9176, 0.0157),
        (0.7569, 0.9255, 0.0157), (0.7804, 0.9333, 0.0118),
        (0.8078, 0.9412, 0.0118), (0.8314, 0.949, 0.0118),
        (0.8549, 0.9569, 0.0078), (0.8784, 0.9647, 0.0078),
        (0.902, 0.9686, 0.0078), (0.9255, 0.9765, 0.0039),
        (0.9529, 0.9843, 0.0039), (0.9765, 0.9922, 0.),
        (1., 1., 0.), (1., 0.9922, 0.), (1., 0.9804, 0.),
        (1., 0.9725, 0.), (1., 0.9647, 0.), (1., 0.9529, 0.),
        (1., 0.9451, 0.), (1., 0.9373, 0.), (1., 0.9294, 0.),
        (1., 0.9176, 0.), (1., 0.9098, 0.), (1., 0.902, 0.),
        (1., 0.8902, 0.), (1., 0.8824, 0.), (1., 0.8745, 0.),
        (1., 0.8627, 0.), (1., 0.8549, 0.), (1., 0.8471, 0.),
        (1., 0.8353, 0.), (1., 0.8275, 0.), (1., 0.8196, 0.),
        (1., 0.8118, 0.), (1., 0.8, 0.), (1., 0.7922, 0.),
        (1., 0.7843, 0.), (1., 0.7725, 0.), (1., 0.7647, 0.),
        (1., 0.7569, 0.), (1., 0.7451, 0.), (1., 0.7373, 0.),
        (1., 0.7294, 0.), (1., 0.7176, 0.), (1., 0.7098, 0.),
        (1., 0.702, 0.), (1., 0.6941, 0.), (1., 0.6824, 0.),
        (1., 0.6745, 0.), (1., 0.6667, 0.), (1., 0.6549, 0.),
        (1., 0.6471, 0.), (1., 0.6314, 0.), (1., 0.6157, 0.),
        (1., 0.6, 0.), (1., 0.5843, 0.), (1., 0.5647, 0.),
        (1., 0.549, 0.), (1., 0.5333, 0.), (1., 0.5176, 0.),
        (1., 0.502, 0.), (1., 0.4863, 0.), (1., 0.4706, 0.),
        (1., 0.4549, 0.), (1., 0.4353, 0.), (1., 0.4196, 0.),
        (1., 0.4039, 0.), (1., 0.3882, 0.), (1., 0.3725, 0.),
        (1., 0.3569, 0.), (1., 0.3412, 0.), (1., 0.3255, 0.),
        (1., 0.3059, 0.), (1., 0.2902, 0.), (1., 0.2745, 0.),
        (1., 0.2588, 0.), (1., 0.2431, 0.), (1., 0.2275, 0.),
        (1., 0.2118, 0.), (1., 0.1961, 0.), (1., 0.1765, 0.),
        (1., 0.1608, 0.), (1., 0.1451, 0.), (1., 0.1294, 0.),
        (1., 0.1137, 0.), (1., 0.098, 0.), (1., 0.0824, 0.),
        (1., 0.0667, 0.), (1., 0.0471, 0.), (1., 0.0314, 0.),
        (1., 0.0157, 0.), (1., 0., 0.), (0.9843, 0., 0.0118),
        (0.9686, 0., 0.0275), (0.949, 0., 0.0392), (0.9333, 0., 0.0549),
        (0.9176, 0., 0.0667), (0.902, 0., 0.0784), (0.8863, 0., 0.0941),
        (0.8667, 0., 0.1059), (0.851, 0., 0.1216), (0.8353, 0., 0.1333),
        (0.8196, 0., 0.1451), (0.8039, 0., 0.1608), (0.7843, 0., 0.1725),
        (0.7686, 0., 0.1882), (0.7529, 0., 0.2), (0.7373, 0., 0.2118),
        (0.7216, 0., 0.2275), (0.702, 0., 0.2392), (0.6863, 0., 0.2549),
        (0.6706, 0., 0.2667), (0.6549, 0., 0.2784), (0.6392, 0., 0.2941),
        (0.6196, 0., 0.3059), (0.6039, 0., 0.3216), (0.5882, 0., 0.3333),
        (0.5725, 0., 0.3451), (0.5569, 0., 0.3608), (0.5373, 0., 0.3725),
        (0.5216, 0., 0.3882), (0.5059, 0., 0.4), (0.4902, 0., 0.4118),
        (0.4745, 0., 0.4275), (0.4549, 0., 0.4392), (0.4392, 0., 0.4549),
        (0.4235, 0., 0.4667), (0.4078, 0., 0.4784), (0.3922, 0., 0.4941),
        (0.3725, 0., 0.5059), (0.3569, 0., 0.5216), (0.3412, 0., 0.5333)],
    'ncl:blre': [
        (1., 1., 1.), (0., 0., 0.), (0., 0., 0.502), (0., 0., 0.5216),
        (0., 0., 0.5412), (0., 0., 0.5608), (0., 0., 0.5804),
        (0., 0., 0.6), (0., 0., 0.6196), (0., 0., 0.6431),
        (0., 0., 0.6627), (0., 0., 0.6824), (0., 0., 0.702),
        (0., 0., 0.7216), (0., 0., 0.7412), (0., 0., 0.7608),
        (0., 0., 0.7804), (0., 0., 0.8), (0., 0., 0.8196),
        (0., 0., 0.8392), (0., 0., 0.8588), (0., 0., 0.8824),
        (0., 0., 0.902), (0., 0., 0.9216), (0., 0., 0.9412),
        (0., 0., 0.9608), (0., 0., 0.9804), (0., 0., 1.),
        (0.0392, 0.0392, 1.), (0.0784, 0.0784, 1.), (0.1216, 0.1216, 1.),
        (0.1608, 0.1608, 1.), (0.2, 0.2, 1.), (0.2392, 0.2392, 1.),
        (0.2784, 0.2784, 1.), (0.3216, 0.3216, 1.),
        (0.3608, 0.3608, 1.), (0.4, 0.4, 1.), (0.4392, 0.4392, 1.),
        (0.4784, 0.4784, 1.), (0.5216, 0.5216, 1.),
        (0.5608, 0.5608, 1.), (0.6, 0.6, 1.), (0.6392, 0.6392, 1.),
        (0.6784, 0.6784, 1.), (0.7216, 0.7216, 1.),
        (0.7608, 0.7608, 1.), (0.8, 0.8, 1.), (0.8392, 0.8392, 1.),
        (0.8784, 0.8784, 1.), (1., 0.8784, 0.8784),
        (1., 0.8392, 0.8392), (1., 0.8, 0.8), (1., 0.7608, 0.7608),
        (1., 0.7216, 0.7216), (1., 0.6784, 0.6784),
        (1., 0.6392, 0.6392), (1., 0.6, 0.6), (1., 0.5608, 0.5608),
        (1., 0.5216, 0.5216), (1., 0.4784, 0.4784), (1., 0.4392, 0.4392),
        (1., 0.4, 0.4), (1., 0.3608, 0.3608), (1., 0.3216, 0.3216),
        (1., 0.2784, 0.2784), (1., 0.2392, 0.2392), (1., 0.2, 0.2),
        (1., 0.1608, 0.1608), (1., 0.1216, 0.1216), (1., 0.0784, 0.0784),
        (1., 0.0392, 0.0392), (1., 0., 0.), (0.9804, 0., 0.),
        (0.9608, 0., 0.), (0.9412, 0., 0.), (0.9216, 0., 0.),
        (0.902, 0., 0.), (0.8824, 0., 0.), (0.8588, 0., 0.),
        (0.8392, 0., 0.), (0.8196, 0., 0.), (0.8, 0., 0.),
        (0.7804, 0., 0.), (0.7608, 0., 0.), (0.7412, 0., 0.),
        (0.7216, 0., 0.), (0.702, 0., 0.), (0.6824, 0., 0.),
        (0.6627, 0., 0.), (0.6431, 0., 0.), (0.6196, 0., 0.),
        (0.6, 0., 0.), (0.5804, 0., 0.), (0.5608, 0., 0.),
        (0.5412, 0., 0.), (0.5216, 0., 0.), (0.502, 0., 0.)],
    'ncl:bluered': [
        (1., 1., 1.), (0., 0., 0.), (0.098, 0.3216, 1.),
        (0.1059, 0.3294, 1.), (0.1098, 0.3333, 1.), (0.1176, 0.3373, 1.),
        (0.1216, 0.3412, 1.), (0.1294, 0.3451, 1.), (0.1333, 0.349, 1.),
        (0.1412, 0.3529, 1.), (0.1451, 0.3608, 1.), (0.1529, 0.3647, 1.),
        (0.1569, 0.3686, 1.), (0.1647, 0.3725, 1.), (0.1686, 0.3765, 1.),
        (0.1765, 0.3804, 1.), (0.1804, 0.3843, 1.), (0.1882, 0.3922, 1.),
        (0.1922, 0.3961, 1.), (0.2, 0.4, 1.), (0.2078, 0.4039, 1.),
        (0.2118, 0.4078, 1.), (0.2196, 0.4118, 1.), (0.2235, 0.4157, 1.),
        (0.2314, 0.4235, 1.), (0.2353, 0.4275, 1.), (0.2431, 0.4314, 1.),
        (0.2471, 0.4353, 1.), (0.2549, 0.4392, 1.), (0.2588, 0.4431, 1.),
        (0.2667, 0.451, 1.), (0.2706, 0.4549, 1.), (0.2784, 0.4588, 1.),
        (0.2824, 0.4627, 1.), (0.2902, 0.4667, 1.), (0.2941, 0.4706, 1.),
        (0.302, 0.4745, 1.), (0.3098, 0.4824, 1.), (0.3137, 0.4863, 1.),
        (0.3216, 0.4902, 1.), (0.3255, 0.4941, 1.), (0.3333, 0.498, 1.),
        (0.3373, 0.502, 1.), (0.3451, 0.5059, 1.), (0.349, 0.5137, 1.),
        (0.3569, 0.5176, 1.), (0.3608, 0.5216, 1.), (0.3686, 0.5255, 1.),
        (0.3725, 0.5294, 1.), (0.3804, 0.5333, 1.), (0.3843, 0.5373, 1.),
        (0.3922, 0.5451, 1.), (0.4, 0.549, 1.), (0.4039, 0.5529, 1.),
        (0.4118, 0.5569, 1.), (0.4157, 0.5608, 1.), (0.4235, 0.5647, 1.),
        (0.4275, 0.5686, 1.), (0.4353, 0.5765, 1.), (0.4392, 0.5804, 1.),
        (0.4471, 0.5843, 1.), (0.451, 0.5882, 1.), (0.4588, 0.5922, 1.),
        (0.4627, 0.5961, 1.), (0.4706, 0.6039, 1.), (0.4745, 0.6078, 1.),
        (0.4824, 0.6118, 1.), (0.4863, 0.6157, 1.), (0.4941, 0.6196, 1.),
        (0.502, 0.6235, 1.), (0.5059, 0.6275, 1.), (0.5137, 0.6353, 1.),
        (0.5176, 0.6392, 1.), (0.5255, 0.6431, 1.), (0.5294, 0.6471, 1.),
        (0.5373, 0.651, 1.), (0.5412, 0.6549, 1.), (0.549, 0.6588, 1.),
        (0.5529, 0.6667, 1.), (0.5608, 0.6706, 1.), (0.5647, 0.6745, 1.),
        (0.5725, 0.6784, 1.), (0.5765, 0.6824, 1.), (0.5843, 0.6863, 1.),
        (0.5882, 0.6902, 1.), (0.5961, 0.698, 1.), (0.6039, 0.702, 1.),
        (0.6078, 0.7059, 1.), (0.6157, 0.7098, 1.), (0.6196, 0.7137, 1.),
        (0.6275, 0.7176, 1.), (0.6314, 0.7255, 1.), (0.6392, 0.7294, 1.),
        (0.6431, 0.7333, 1.), (0.651, 0.7373, 1.), (0.6549, 0.7412, 1.),
        (0.6627, 0.7451, 1.), (0.6667, 0.749, 1.), (0.6745, 0.7569, 1.),
        (0.6784, 0.7608, 1.), (0.6863, 0.7647, 1.), (0.6902, 0.7686, 1.),
        (0.698, 0.7725, 1.), (0.7059, 0.7765, 1.), (0.7098, 0.7804, 1.),
        (0.7176, 0.7882, 1.), (0.7216, 0.7922, 1.), (0.7294, 0.7961, 1.),
        (0.7333, 0.8, 1.), (0.7412, 0.8039, 1.), (0.7451, 0.8078, 1.),
        (0.7529, 0.8118, 1.), (0.7569, 0.8196, 1.), (0.7647, 0.8235, 1.),
        (0.7686, 0.8275, 1.), (0.7765, 0.8314, 1.), (0.7804, 0.8353, 1.),
        (0.7882, 0.8392, 1.), (0.7922, 0.8431, 1.), (0.8, 0.851, 1.),
        (0.8078, 0.8549, 1.), (0.8118, 0.8588, 1.), (0.8196, 0.8627, 1.),
        (0.8235, 0.8667, 1.), (0.8314, 0.8706, 1.), (0.8353, 0.8784, 1.),
        (0.8431, 0.8824, 1.), (0.8471, 0.8863, 1.), (1., 0.8471, 0.8471),
        (1., 0.8431, 0.8431), (1., 0.8353, 0.8353), (1., 0.8314, 0.8314),
        (1., 0.8235, 0.8235), (1., 0.8196, 0.8196), (1., 0.8118, 0.8118),
        (1., 0.8078, 0.8078), (1., 0.8, 0.8), (1., 0.7922, 0.7922),
        (1., 0.7882, 0.7882), (1., 0.7804, 0.7804), (1., 0.7765, 0.7765),
        (1., 0.7686, 0.7686), (1., 0.7647, 0.7647), (1., 0.7569, 0.7569),
        (1., 0.7529, 0.7529), (1., 0.7451, 0.7451), (1., 0.7412, 0.7412),
        (1., 0.7333, 0.7333), (1., 0.7294, 0.7294), (1., 0.7216, 0.7216),
        (1., 0.7176, 0.7176), (1., 0.7098, 0.7098), (1., 0.7059, 0.7059),
        (1., 0.698, 0.698), (1., 0.6902, 0.6902), (1., 0.6863, 0.6863),
        (1., 0.6784, 0.6784), (1., 0.6745, 0.6745), (1., 0.6667, 0.6667),
        (1., 0.6627, 0.6627), (1., 0.6549, 0.6549), (1., 0.651, 0.651),
        (1., 0.6431, 0.6431), (1., 0.6392, 0.6392), (1., 0.6314, 0.6314),
        (1., 0.6275, 0.6275), (1., 0.6196, 0.6196), (1., 0.6157, 0.6157),
        (1., 0.6078, 0.6078), (1., 0.6039, 0.6039), (1., 0.5961, 0.5961),
        (1., 0.5882, 0.5882), (1., 0.5843, 0.5843), (1., 0.5765, 0.5765),
        (1., 0.5725, 0.5725), (1., 0.5647, 0.5647), (1., 0.5608, 0.5608),
        (1., 0.5529, 0.5529), (1., 0.549, 0.549), (1., 0.5412, 0.5412),
        (1., 0.5373, 0.5373), (1., 0.5294, 0.5294), (1., 0.5255, 0.5255),
        (1., 0.5176, 0.5176), (1., 0.5137, 0.5137), (1., 0.5059, 0.5059),
        (1., 0.502, 0.502), (1., 0.4941, 0.4941), (1., 0.4863, 0.4863),
        (1., 0.4824, 0.4824), (1., 0.4745, 0.4745), (1., 0.4706, 0.4706),
        (1., 0.4627, 0.4627), (1., 0.4588, 0.4588), (1., 0.451, 0.451),
        (1., 0.4471, 0.4471), (1., 0.4392, 0.4392), (1., 0.4353, 0.4353),
        (1., 0.4275, 0.4275), (1., 0.4235, 0.4235), (1., 0.4157, 0.4157),
        (1., 0.4118, 0.4118), (1., 0.4039, 0.4039), (1., 0.3961, 0.3961),
        (1., 0.3922, 0.3922), (1., 0.3843, 0.3843), (1., 0.3804, 0.3804),
        (1., 0.3725, 0.3725), (1., 0.3686, 0.3686), (1., 0.3608, 0.3608),
        (1., 0.3569, 0.3569), (1., 0.349, 0.349), (1., 0.3451, 0.3451),
        (1., 0.3373, 0.3373), (1., 0.3333, 0.3333), (1., 0.3255, 0.3255),
        (1., 0.3216, 0.3216), (1., 0.3137, 0.3137), (1., 0.3098, 0.3098),
        (1., 0.302, 0.302), (1., 0.2941, 0.2941), (1., 0.2902, 0.2902),
        (1., 0.2824, 0.2824), (1., 0.2784, 0.2784), (1., 0.2706, 0.2706),
        (1., 0.2667, 0.2667), (1., 0.2588, 0.2588), (1., 0.2549, 0.2549),
        (1., 0.2471, 0.2471), (1., 0.2431, 0.2431), (1., 0.2353, 0.2353),
        (1., 0.2314, 0.2314), (1., 0.2235, 0.2235), (1., 0.2196, 0.2196),
        (1., 0.2118, 0.2118), (1., 0.2078, 0.2078), (1., 0.2, 0.2),
        (1., 0.1922, 0.1922), (1., 0.1882, 0.1882), (1., 0.1804, 0.1804),
        (1., 0.1765, 0.1765), (1., 0.1686, 0.1686), (1., 0.1647, 0.1647),
        (1., 0.1569, 0.1569), (1., 0.1529, 0.1529), (1., 0.1451, 0.1451),
        (1., 0.1412, 0.1412), (1., 0.1333, 0.1333), (1., 0.1294, 0.1294),
        (1., 0.1216, 0.1216), (1., 0.1176, 0.1176), (1., 0.1098, 0.1098),
        (1., 0.1059, 0.1059), (1., 0.098, 0.098)],
    'ncl:blueredgray': [
        (1., 1., 1.), (0., 0., 0.), (0.098, 0.3216, 1.),
        (0.1059, 0.3294, 1.), (0.1098, 0.3333, 1.), (0.1176, 0.3373, 1.),
        (0.1216, 0.3412, 1.), (0.1294, 0.3451, 1.), (0.1333, 0.349, 1.),
        (0.1412, 0.3529, 1.), (0.1451, 0.3608, 1.), (0.1529, 0.3647, 1.),
        (0.1569, 0.3686, 1.), (0.1647, 0.3725, 1.), (0.1686, 0.3765, 1.),
        (0.1765, 0.3804, 1.), (0.1804, 0.3843, 1.), (0.1882, 0.3922, 1.),
        (0.1922, 0.3961, 1.), (0.2, 0.4, 1.), (0.2078, 0.4039, 1.),
        (0.2118, 0.4078, 1.), (0.2196, 0.4118, 1.), (0.2235, 0.4157, 1.),
        (0.2314, 0.4235, 1.), (0.2353, 0.4275, 1.), (0.2431, 0.4314, 1.),
        (0.2471, 0.4353, 1.), (0.2549, 0.4392, 1.), (0.2588, 0.4431, 1.),
        (0.2667, 0.451, 1.), (0.2706, 0.4549, 1.), (0.2784, 0.4588, 1.),
        (0.2824, 0.4627, 1.), (0.2902, 0.4667, 1.), (0.2941, 0.4706, 1.),
        (0.302, 0.4745, 1.), (0.3098, 0.4824, 1.), (0.3137, 0.4863, 1.),
        (0.3216, 0.4902, 1.), (0.3255, 0.4941, 1.), (0.3333, 0.498, 1.),
        (0.3373, 0.502, 1.), (0.3451, 0.5059, 1.), (0.349, 0.5137, 1.),
        (0.3569, 0.5176, 1.), (0.3608, 0.5216, 1.), (0.3686, 0.5255, 1.),
        (0.3725, 0.5294, 1.), (0.3804, 0.5333, 1.), (0.3843, 0.5373, 1.),
        (0.3922, 0.5451, 1.), (0.4, 0.549, 1.), (0.4039, 0.5529, 1.),
        (0.4118, 0.5569, 1.), (0.4157, 0.5608, 1.), (0.4235, 0.5647, 1.),
        (0.4275, 0.5686, 1.), (0.4353, 0.5765, 1.), (0.4392, 0.5804, 1.),
        (0.4471, 0.5843, 1.), (0.451, 0.5882, 1.), (0.4588, 0.5922, 1.),
        (0.4627, 0.5961, 1.), (0.4706, 0.6039, 1.), (0.4745, 0.6078, 1.),
        (0.4824, 0.6118, 1.), (0.4863, 0.6157, 1.), (0.4941, 0.6196, 1.),
        (0.502, 0.6235, 1.), (0.5059, 0.6275, 1.), (0.5137, 0.6353, 1.),
        (0.5176, 0.6392, 1.), (0.5255, 0.6431, 1.), (0.5294, 0.6471, 1.),
        (0.5373, 0.651, 1.), (0.5412, 0.6549, 1.), (0.549, 0.6588, 1.),
        (0.5529, 0.6667, 1.), (0.5608, 0.6706, 1.), (0.5647, 0.6745, 1.),
        (0.5725, 0.6784, 1.), (0.5765, 0.6824, 1.), (0.5843, 0.6863, 1.),
        (0.5882, 0.6902, 1.), (0.5961, 0.698, 1.), (0.6039, 0.702, 1.),
        (0.6078, 0.7059, 1.), (0.6157, 0.7098, 1.), (0.6196, 0.7137, 1.),
        (0.6275, 0.7176, 1.), (0.6314, 0.7255, 1.), (0.6392, 0.7294, 1.),
        (0.6431, 0.7333, 1.), (0.651, 0.7373, 1.), (0.6549, 0.7412, 1.),
        (0.6627, 0.7451, 1.), (0.6667, 0.749, 1.), (0.6745, 0.7569, 1.),
        (0.6784, 0.7608, 1.), (0.6863, 0.7647, 1.), (0.6902, 0.7686, 1.),
        (0.698, 0.7725, 1.), (0.7059, 0.7765, 1.), (0.7098, 0.7804, 1.),
        (0.7176, 0.7882, 1.), (0.7216, 0.7922, 1.), (0.7294, 0.7961, 1.),
        (0.7333, 0.8, 1.), (0.7412, 0.8039, 1.), (0.7451, 0.8078, 1.),
        (0.7529, 0.8118, 1.), (0.7569, 0.8196, 1.), (0.7647, 0.8235, 1.),
        (0.7686, 0.8275, 1.), (0.7765, 0.8314, 1.), (0.7804, 0.8353, 1.),
        (0.7882, 0.8392, 1.), (0.7922, 0.8431, 1.), (0.8, 0.851, 1.),
        (0.8078, 0.8549, 1.), (0.8118, 0.8588, 1.), (0.8196, 0.8627, 1.),
        (0.8235, 0.8667, 1.), (0.8314, 0.8706, 1.), (0.8353, 0.8784, 1.),
        (0.8431, 0.8824, 1.), (0.8471, 0.8863, 1.), (1., 0.8471, 0.8471),
        (1., 0.8431, 0.8431), (1., 0.8353, 0.8353), (1., 0.8314, 0.8314),
        (1., 0.8235, 0.8235), (1., 0.8196, 0.8196), (1., 0.8118, 0.8118),
        (1., 0.8078, 0.8078), (1., 0.8, 0.8), (1., 0.7922, 0.7922),
        (1., 0.7882, 0.7882), (1., 0.7804, 0.7804), (1., 0.7765, 0.7765),
        (1., 0.7686, 0.7686), (1., 0.7647, 0.7647), (1., 0.7569, 0.7569),
        (1., 0.7529, 0.7529), (1., 0.7451, 0.7451), (1., 0.7412, 0.7412),
        (1., 0.7333, 0.7333), (1., 0.7294, 0.7294), (1., 0.7216, 0.7216),
        (1., 0.7176, 0.7176), (1., 0.7098, 0.7098), (1., 0.7059, 0.7059),
        (1., 0.698, 0.698), (1., 0.6902, 0.6902), (1., 0.6863, 0.6863),
        (1., 0.6784, 0.6784), (1., 0.6745, 0.6745), (1., 0.6667, 0.6667),
        (1., 0.6627, 0.6627), (1., 0.6549, 0.6549), (1., 0.651, 0.651),
        (1., 0.6431, 0.6431), (1., 0.6392, 0.6392), (1., 0.6314, 0.6314),
        (1., 0.6275, 0.6275), (1., 0.6196, 0.6196), (1., 0.6157, 0.6157),
        (1., 0.6078, 0.6078), (1., 0.6039, 0.6039), (1., 0.5961, 0.5961),
        (1., 0.5882, 0.5882), (1., 0.5843, 0.5843), (1., 0.5765, 0.5765),
        (1., 0.5725, 0.5725), (1., 0.5647, 0.5647), (1., 0.5608, 0.5608),
        (1., 0.5529, 0.5529), (1., 0.549, 0.549), (1., 0.5412, 0.5412),
        (1., 0.5373, 0.5373), (1., 0.5294, 0.5294), (1., 0.5255, 0.5255),
        (1., 0.5176, 0.5176), (1., 0.5137, 0.5137), (1., 0.5059, 0.5059),
        (1., 0.502, 0.502), (1., 0.4941, 0.4941), (1., 0.4863, 0.4863),
        (1., 0.4824, 0.4824), (1., 0.4745, 0.4745), (1., 0.4706, 0.4706),
        (1., 0.4627, 0.4627), (1., 0.4588, 0.4588), (1., 0.451, 0.451),
        (1., 0.4471, 0.4471), (1., 0.4392, 0.4392), (1., 0.4353, 0.4353),
        (1., 0.4275, 0.4275), (1., 0.4235, 0.4235), (1., 0.4157, 0.4157),
        (1., 0.4118, 0.4118), (1., 0.4039, 0.4039), (1., 0.3961, 0.3961),
        (1., 0.3922, 0.3922), (1., 0.3843, 0.3843), (1., 0.3804, 0.3804),
        (1., 0.3725, 0.3725), (1., 0.3686, 0.3686), (1., 0.3608, 0.3608),
        (1., 0.3569, 0.3569), (1., 0.349, 0.349), (1., 0.3451, 0.3451),
        (1., 0.3373, 0.3373), (1., 0.3333, 0.3333), (1., 0.3255, 0.3255),
        (1., 0.3216, 0.3216), (1., 0.3137, 0.3137), (1., 0.3098, 0.3098),
        (1., 0.302, 0.302), (1., 0.2941, 0.2941), (1., 0.2902, 0.2902),
        (1., 0.2824, 0.2824), (1., 0.2784, 0.2784), (1., 0.2706, 0.2706),
        (1., 0.2667, 0.2667), (1., 0.2588, 0.2588), (1., 0.2549, 0.2549),
        (1., 0.2471, 0.2471), (1., 0.2431, 0.2431), (1., 0.2353, 0.2353),
        (1., 0.2314, 0.2314), (1., 0.2235, 0.2235), (1., 0.2196, 0.2196),
        (1., 0.2118, 0.2118), (1., 0.2078, 0.2078), (1., 0.2, 0.2),
        (1., 0.1922, 0.1922), (1., 0.1882, 0.1882), (1., 0.1804, 0.1804),
        (1., 0.1765, 0.1765), (1., 0.1686, 0.1686), (1., 0.1647, 0.1647),
        (1., 0.1569, 0.1569), (1., 0.1529, 0.1529), (1., 0.1451, 0.1451),
        (1., 0.1412, 0.1412), (1., 0.1333, 0.1333), (1., 0.1294, 0.1294),
        (1., 0.1216, 0.1216), (1., 0.1176, 0.1176), (1., 0.1098, 0.1098),
        (1., 0.1059, 0.1059), (1., 0.098, 0.098), (0.8275, 0.8275, 0.8275)],
    'ncl:bluewhiteorangered': [
        (1., 1., 1.), (0., 0., 0.),
        (0.1059, 0.1725, 0.3843), (0.1059, 0.1804, 0.3961),
        (0.1098, 0.1882, 0.4078), (0.1098, 0.1961, 0.4196),
        (0.1137, 0.2039, 0.4314), (0.1137, 0.2078, 0.4392),
        (0.1137, 0.2157, 0.451), (0.1176, 0.2235, 0.4627),
        (0.1176, 0.2314, 0.4745), (0.1176, 0.2392, 0.4863),
        (0.1216, 0.2471, 0.498), (0.1216, 0.2549, 0.5098),
        (0.1255, 0.2627, 0.5216), (0.1255, 0.2667, 0.5294),
        (0.1255, 0.2745, 0.5412), (0.1294, 0.2824, 0.5529),
        (0.1294, 0.2902, 0.5647), (0.1294, 0.298, 0.5765),
        (0.1333, 0.3059, 0.5882), (0.1333, 0.3137, 0.6),
        (0.1373, 0.3216, 0.6118), (0.1373, 0.3255, 0.6196),
        (0.1373, 0.3333, 0.6314), (0.1412, 0.349, 0.6549),
        (0.149, 0.3608, 0.6627), (0.1569, 0.3725, 0.6706),
        (0.1647, 0.3843, 0.6784), (0.1765, 0.4, 0.6863),
        (0.1843, 0.4118, 0.6941), (0.1922, 0.4235, 0.702),
        (0.2, 0.4353, 0.7098), (0.2078, 0.4471, 0.7176),
        (0.2157, 0.4588, 0.7255), (0.2235, 0.4706, 0.7333),
        (0.2314, 0.4824, 0.7412), (0.2431, 0.498, 0.7529),
        (0.251, 0.5098, 0.7608), (0.2588, 0.5216, 0.7686),
        (0.2667, 0.5333, 0.7765), (0.2745, 0.5451, 0.7843),
        (0.2824, 0.5569, 0.7922), (0.2902, 0.5686, 0.8),
        (0.298, 0.5804, 0.8078), (0.3098, 0.5961, 0.8157),
        (0.3176, 0.6078, 0.8235), (0.3255, 0.6196, 0.8314),
        (0.3412, 0.6431, 0.8471), (0.3529, 0.651, 0.851),
        (0.3647, 0.6627, 0.8588), (0.3765, 0.6706, 0.8627),
        (0.3882, 0.6784, 0.8667), (0.4, 0.6863, 0.8706),
        (0.4118, 0.698, 0.8784), (0.4196, 0.7059, 0.8824),
        (0.4314, 0.7137, 0.8863), (0.4431, 0.7216, 0.8941),
        (0.4549, 0.7333, 0.898), (0.4667, 0.7412, 0.902),
        (0.4784, 0.749, 0.9098), (0.4902, 0.7569, 0.9137),
        (0.502, 0.7686, 0.9176), (0.5137, 0.7765, 0.9216),
        (0.5255, 0.7843, 0.9294), (0.5373, 0.7922, 0.9333),
        (0.549, 0.8039, 0.9373), (0.5569, 0.8118, 0.9451),
        (0.5686, 0.8196, 0.949), (0.5804, 0.8275, 0.9529),
        (0.5922, 0.8392, 0.9569), (0.6157, 0.8549, 0.9686),
        (0.6235, 0.8588, 0.9686), (0.6353, 0.8627, 0.9686),
        (0.6431, 0.8667, 0.9686), (0.6549, 0.8706, 0.9686),
        (0.6627, 0.8706, 0.9686), (0.6745, 0.8745, 0.9725),
        (0.6824, 0.8784, 0.9725), (0.6902, 0.8824, 0.9725),
        (0.702, 0.8863, 0.9725), (0.7098, 0.8902, 0.9725),
        (0.7216, 0.8941, 0.9725), (0.7294, 0.898, 0.9725),
        (0.7373, 0.898, 0.9725), (0.749, 0.902, 0.9725),
        (0.7569, 0.9059, 0.9725), (0.7686, 0.9098, 0.9725),
        (0.7765, 0.9137, 0.9725), (0.7882, 0.9176, 0.9765),
        (0.7961, 0.9216, 0.9765), (0.8039, 0.9255, 0.9765),
        (0.8157, 0.9255, 0.9765), (0.8235, 0.9294, 0.9765),
        (0.8431, 0.9373, 0.9765), (0.851, 0.9412, 0.9765),
        (0.8549, 0.9412, 0.9804), (0.8627, 0.9451, 0.9804),
        (0.8706, 0.949, 0.9804), (0.8745, 0.949, 0.9804),
        (0.8824, 0.9529, 0.9843), (0.8902, 0.9569, 0.9843),
        (0.8941, 0.9569, 0.9843), (0.902, 0.9608, 0.9843),
        (0.9098, 0.9647, 0.9882), (0.9137, 0.9647, 0.9882),
        (0.9216, 0.9686, 0.9882), (0.9294, 0.9725, 0.9882),
        (0.9333, 0.9725, 0.9922), (0.9412, 0.9765, 0.9922),
        (0.949, 0.9804, 0.9922), (0.9529, 0.9804, 0.9922),
        (0.9608, 0.9843, 0.9961), (0.9686, 0.9882, 0.9961),
        (0.9725, 0.9882, 0.9961), (0.9804, 0.9922, 0.9961),
        (0.9882, 0.9961, 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 0.9961, 0.9843),
        (1., 0.9922, 0.9686), (1., 0.9922, 0.9529),
        (1., 0.9882, 0.9333), (0.9961, 0.9843, 0.9176),
        (0.9961, 0.9804, 0.902), (0.9961, 0.9765, 0.8863),
        (0.9961, 0.9725, 0.8706), (0.9961, 0.9725, 0.8549),
        (0.9961, 0.9686, 0.8353), (0.9961, 0.9647, 0.8196),
        (0.9961, 0.9608, 0.8039), (0.9922, 0.9569, 0.7882),
        (0.9922, 0.9529, 0.7725), (0.9922, 0.9529, 0.7569),
        (0.9922, 0.949, 0.7373), (0.9922, 0.9451, 0.7216),
        (0.9922, 0.9412, 0.7059), (0.9922, 0.9373, 0.6902),
        (0.9922, 0.9333, 0.6745), (0.9882, 0.9333, 0.6588),
        (0.9882, 0.9294, 0.6392), (0.9882, 0.9216, 0.6078),
        (0.9882, 0.9137, 0.5922), (0.9882, 0.902, 0.5725),
        (0.9882, 0.8941, 0.5569), (0.9882, 0.8824, 0.5412),
        (0.9882, 0.8745, 0.5216), (0.9922, 0.8627, 0.5059),
        (0.9922, 0.8549, 0.4863), (0.9922, 0.8431, 0.4706),
        (0.9922, 0.8353, 0.4549), (0.9922, 0.8235, 0.4353),
        (0.9922, 0.8157, 0.4196), (0.9922, 0.8078, 0.4039),
        (0.9922, 0.7961, 0.3843), (0.9922, 0.7882, 0.3686),
        (0.9922, 0.7765, 0.349), (0.9922, 0.7686, 0.3333),
        (0.9922, 0.7569, 0.3176), (0.9961, 0.749, 0.298),
        (0.9961, 0.7373, 0.2824), (0.9961, 0.7294, 0.2667),
        (0.9961, 0.7176, 0.2471), (0.9961, 0.7098, 0.2314),
        (0.9961, 0.6902, 0.1961), (0.9961, 0.6784, 0.1961),
        (0.9922, 0.6667, 0.1922), (0.9922, 0.6549, 0.1922),
        (0.9922, 0.6431, 0.1922), (0.9882, 0.6314, 0.1882),
        (0.9882, 0.6235, 0.1882), (0.9843, 0.6118, 0.1843),
        (0.9843, 0.6, 0.1843), (0.9843, 0.5882, 0.1843),
        (0.9804, 0.5765, 0.1804), (0.9804, 0.5647, 0.1804),
        (0.9804, 0.5529, 0.1804), (0.9765, 0.5412, 0.1765),
        (0.9765, 0.5294, 0.1765), (0.9725, 0.5176, 0.1725),
        (0.9725, 0.5059, 0.1725), (0.9725, 0.4941, 0.1725),
        (0.9686, 0.4863, 0.1686), (0.9686, 0.4745, 0.1686),
        (0.9686, 0.4627, 0.1686), (0.9647, 0.451, 0.1647),
        (0.9647, 0.4392, 0.1647), (0.9608, 0.4157, 0.1608),
        (0.9569, 0.4039, 0.1608), (0.949, 0.3922, 0.1608),
        (0.9451, 0.3804, 0.1608), (0.9373, 0.3686, 0.1608),
        (0.9333, 0.3529, 0.1608), (0.9294, 0.3412, 0.1608),
        (0.9216, 0.3294, 0.1608), (0.9176, 0.3176, 0.1608),
        (0.9098, 0.3059, 0.1608), (0.9059, 0.2941, 0.1608),
        (0.898, 0.2824, 0.1608), (0.8941, 0.2706, 0.1608),
        (0.8902, 0.2549, 0.1569), (0.8824, 0.2431, 0.1569),
        (0.8784, 0.2314, 0.1569), (0.8706, 0.2196, 0.1569),
        (0.8667, 0.2078, 0.1569), (0.8627, 0.1961, 0.1569),
        (0.8549, 0.1843, 0.1569), (0.851, 0.1725, 0.1569),
        (0.8431, 0.1569, 0.1569), (0.8392, 0.1451, 0.1569),
        (0.8275, 0.1216, 0.1569), (0.8157, 0.1216, 0.1529),
        (0.8078, 0.1176, 0.1529), (0.7961, 0.1176, 0.149),
        (0.7843, 0.1137, 0.149), (0.7725, 0.1137, 0.1451),
        (0.7647, 0.1137, 0.1412), (0.7529, 0.1098, 0.1412),
        (0.7412, 0.1098, 0.1373), (0.7333, 0.1059, 0.1333),
        (0.7216, 0.1059, 0.1333), (0.7098, 0.102, 0.1294),
        (0.702, 0.102, 0.1294), (0.6902, 0.102, 0.1255),
        (0.6784, 0.098, 0.1216), (0.6667, 0.098, 0.1216),
        (0.6588, 0.0941, 0.1176), (0.6471, 0.0941, 0.1137),
        (0.6353, 0.0941, 0.1137), (0.6275, 0.0902, 0.1098),
        (0.6157, 0.0902, 0.1098), (0.6039, 0.0863, 0.1059),
        (0.5922, 0.0863, 0.102), (0.5725, 0.0824, 0.098)],
    'ncl:blueyellowred': [
        (1., 1., 1.), (0., 0., 0.),
        (0.0196, 0.1373, 0.3137), (0.0196, 0.1451, 0.3255),
        (0.0196, 0.1529, 0.3373), (0.0196, 0.1569, 0.349),
        (0.0196, 0.1647, 0.3608), (0.0235, 0.1725, 0.3725),
        (0.0235, 0.1804, 0.3843), (0.0235, 0.1882, 0.3922),
        (0.0235, 0.1922, 0.4039), (0.0235, 0.2, 0.4157),
        (0.0235, 0.2078, 0.4275), (0.0235, 0.2157, 0.4392),
        (0.0235, 0.2235, 0.451), (0.0275, 0.2314, 0.4627),
        (0.0275, 0.2353, 0.4745), (0.0275, 0.2431, 0.4863),
        (0.0275, 0.251, 0.498), (0.0275, 0.2588, 0.5098),
        (0.0275, 0.2667, 0.5216), (0.0275, 0.2706, 0.5333),
        (0.0275, 0.2784, 0.5412), (0.0275, 0.2863, 0.5529),
        (0.0314, 0.2941, 0.5647), (0.0314, 0.302, 0.5765),
        (0.0314, 0.3059, 0.5882), (0.0314, 0.3216, 0.6118),
        (0.0392, 0.3294, 0.6196), (0.0471, 0.3412, 0.6235),
        (0.0588, 0.349, 0.6314), (0.0667, 0.3608, 0.6392),
        (0.0745, 0.3686, 0.6431), (0.0824, 0.3804, 0.651),
        (0.0941, 0.3882, 0.6588), (0.102, 0.4, 0.6627),
        (0.1098, 0.4078, 0.6706), (0.1176, 0.4196, 0.6784),
        (0.1294, 0.4275, 0.6824), (0.1373, 0.4392, 0.6902),
        (0.1451, 0.4471, 0.698), (0.1529, 0.4549, 0.702),
        (0.1608, 0.4667, 0.7098), (0.1725, 0.4745, 0.7137),
        (0.1804, 0.4863, 0.7216), (0.1882, 0.4941, 0.7294),
        (0.1961, 0.5059, 0.7333), (0.2078, 0.5137, 0.7412),
        (0.2157, 0.5255, 0.749), (0.2235, 0.5333, 0.7529),
        (0.2314, 0.5451, 0.7608), (0.2431, 0.5529, 0.7686),
        (0.2588, 0.5725, 0.7804), (0.2667, 0.5765, 0.7843),
        (0.2706, 0.5843, 0.7843), (0.2784, 0.5882, 0.7882),
        (0.2824, 0.5961, 0.7882), (0.2902, 0.6, 0.7922),
        (0.298, 0.6078, 0.7922), (0.302, 0.6118, 0.7961),
        (0.3098, 0.6157, 0.8), (0.3176, 0.6235, 0.8),
        (0.3216, 0.6275, 0.8039), (0.3294, 0.6353, 0.8039),
        (0.3333, 0.6392, 0.8078), (0.3412, 0.6471, 0.8118),
        (0.349, 0.651, 0.8118), (0.3529, 0.6549, 0.8157),
        (0.3608, 0.6627, 0.8157), (0.3647, 0.6667, 0.8196),
        (0.3725, 0.6745, 0.8196), (0.3804, 0.6784, 0.8235),
        (0.3843, 0.6824, 0.8275), (0.3922, 0.6902, 0.8275),
        (0.4, 0.6941, 0.8314), (0.4039, 0.702, 0.8314),
        (0.4118, 0.7059, 0.8353), (0.4157, 0.7137, 0.8353),
        (0.4235, 0.7176, 0.8392), (0.4314, 0.7216, 0.8431),
        (0.4431, 0.7294, 0.8431), (0.451, 0.7333, 0.8471),
        (0.4627, 0.7412, 0.8471), (0.4706, 0.7451, 0.851),
        (0.4784, 0.7529, 0.8549), (0.4902, 0.7569, 0.8549),
        (0.498, 0.7608, 0.8588), (0.5059, 0.7686, 0.8627),
        (0.5176, 0.7725, 0.8627), (0.5255, 0.7804, 0.8667),
        (0.5373, 0.7843, 0.8667), (0.5451, 0.7922, 0.8706),
        (0.5529, 0.7961, 0.8745), (0.5647, 0.8, 0.8745),
        (0.5725, 0.8078, 0.8784), (0.5843, 0.8118, 0.8784),
        (0.5922, 0.8196, 0.8824), (0.6, 0.8235, 0.8863),
        (0.6118, 0.8275, 0.8863), (0.6196, 0.8353, 0.8902),
        (0.6275, 0.8392, 0.8941), (0.6392, 0.8471, 0.8941),
        (0.6471, 0.851, 0.898), (0.6667, 0.8627, 0.902),
        (0.6745, 0.8667, 0.9059), (0.6824, 0.8706, 0.9098),
        (0.6902, 0.8745, 0.9137), (0.698, 0.8784, 0.9176),
        (0.702, 0.8824, 0.9216), (0.7098, 0.8863, 0.9255),
        (0.7176, 0.8902, 0.9294), (0.7255, 0.8941, 0.9333),
        (0.7333, 0.898, 0.9373), (0.7412, 0.902, 0.9412),
        (0.749, 0.9059, 0.9451), (0.7569, 0.9098, 0.949),
        (0.7647, 0.9137, 0.9529), (0.7686, 0.9137, 0.9529),
        (0.7765, 0.9176, 0.9569), (0.7843, 0.9216, 0.9608),
        (0.7922, 0.9255, 0.9647), (0.8, 0.9294, 0.9686),
        (0.8078, 0.9333, 0.9725), (0.8157, 0.9373, 0.9765),
        (0.8235, 0.9412, 0.9804), (0.8275, 0.9451, 0.9843),
        (0.8353, 0.949, 0.9882), (0.8431, 0.9529, 0.9922),
        (0.8588, 0.9608, 1.), (1., 1., 0.7843),
        (1., 0.9961, 0.7725), (1., 0.9922, 0.7569),
        (1., 0.9882, 0.7451), (1., 0.9843, 0.7333),
        (1., 0.9804, 0.7216), (1., 0.9765, 0.7059),
        (1., 0.9725, 0.6941), (1., 0.9686, 0.6824),
        (1., 0.9647, 0.6706), (1., 0.9608, 0.6549),
        (1., 0.9569, 0.6431), (1., 0.9529, 0.6314),
        (1., 0.9529, 0.6196), (1., 0.949, 0.6039),
        (1., 0.9451, 0.5922), (1., 0.9412, 0.5804),
        (1., 0.9373, 0.5647), (1., 0.9333, 0.5529),
        (1., 0.9294, 0.5412), (1., 0.9255, 0.5294),
        (1., 0.9216, 0.5137), (1., 0.9176, 0.502),
        (1., 0.9137, 0.4902), (1., 0.9098, 0.4784),
        (1., 0.902, 0.451), (1., 0.898, 0.4431),
        (0.9961, 0.8902, 0.4353), (0.9961, 0.8863, 0.4275),
        (0.9922, 0.8824, 0.4235), (0.9922, 0.8784, 0.4157),
        (0.9882, 0.8706, 0.4078), (0.9882, 0.8667, 0.4),
        (0.9843, 0.8627, 0.3922), (0.9843, 0.8588, 0.3843),
        (0.9804, 0.851, 0.3804), (0.9804, 0.8471, 0.3725),
        (0.9765, 0.8431, 0.3647), (0.9765, 0.8392, 0.3569),
        (0.9725, 0.8314, 0.349), (0.9725, 0.8275, 0.3412),
        (0.9686, 0.8235, 0.3333), (0.9686, 0.8157, 0.3294),
        (0.9647, 0.8118, 0.3216), (0.9647, 0.8078, 0.3137),
        (0.9608, 0.8039, 0.3059), (0.9608, 0.7961, 0.298),
        (0.9569, 0.7922, 0.2902), (0.9569, 0.7882, 0.2863),
        (0.9529, 0.7843, 0.2784), (0.949, 0.7725, 0.2627),
        (0.949, 0.7608, 0.2549), (0.9451, 0.749, 0.2471),
        (0.9451, 0.7373, 0.2431), (0.9412, 0.7255, 0.2353),
        (0.9412, 0.7137, 0.2275), (0.9373, 0.702, 0.2196),
        (0.9373, 0.6902, 0.2118), (0.9333, 0.6784, 0.2078),
        (0.9333, 0.6667, 0.2), (0.9294, 0.6549, 0.1922),
        (0.9294, 0.6431, 0.1843), (0.9255, 0.6314, 0.1765),
        (0.9255, 0.6235, 0.1725), (0.9255, 0.6118, 0.1647),
        (0.9216, 0.6, 0.1569), (0.9216, 0.5882, 0.149),
        (0.9176, 0.5765, 0.1412), (0.9176, 0.5647, 0.1333),
        (0.9137, 0.5529, 0.1294), (0.9137, 0.5412, 0.1216),
        (0.9098, 0.5294, 0.1137), (0.9098, 0.5176, 0.1059),
        (0.9059, 0.5059, 0.098), (0.9059, 0.4941, 0.0941),
        (0.902, 0.4824, 0.0863), (0.902, 0.4706, 0.0784),
        (0.898, 0.4627, 0.0745), (0.8941, 0.4549, 0.0706),
        (0.8902, 0.4471, 0.0706), (0.8824, 0.4392, 0.0667),
        (0.8784, 0.4314, 0.0627), (0.8745, 0.4235, 0.0588),
        (0.8706, 0.4196, 0.0588), (0.8667, 0.4118, 0.0549),
        (0.8627, 0.4039, 0.051), (0.8549, 0.3961, 0.0471),
        (0.851, 0.3882, 0.0471), (0.8471, 0.3804, 0.0431),
        (0.8431, 0.3725, 0.0392), (0.8392, 0.3647, 0.0353),
        (0.8353, 0.3569, 0.0314), (0.8314, 0.349, 0.0314),
        (0.8235, 0.3412, 0.0275), (0.8196, 0.3333, 0.0235),
        (0.8157, 0.3255, 0.0196), (0.8118, 0.3216, 0.0196),
        (0.8078, 0.3137, 0.0157), (0.8039, 0.3059, 0.0118),
        (0.7961, 0.298, 0.0078), (0.7922, 0.2902, 0.0078),
        (0.7843, 0.2745, 0.), (0.7765, 0.2706, 0.0039),
        (0.7647, 0.2667, 0.0078), (0.7569, 0.2627, 0.0118),
        (0.7451, 0.2549, 0.0157), (0.7373, 0.251, 0.0157),
        (0.7255, 0.2471, 0.0196), (0.7176, 0.2431, 0.0235),
        (0.7059, 0.2392, 0.0275), (0.698, 0.2353, 0.0314),
        (0.6863, 0.2275, 0.0353), (0.6784, 0.2235, 0.0392),
        (0.6667, 0.2196, 0.0431), (0.6588, 0.2157, 0.0471),
        (0.6471, 0.2118, 0.0471), (0.6392, 0.2078, 0.051),
        (0.6275, 0.2039, 0.0549), (0.6196, 0.1961, 0.0588),
        (0.6078, 0.1922, 0.0627), (0.6, 0.1882, 0.0667),
        (0.5882, 0.1843, 0.0706), (0.5804, 0.1804, 0.0745),
        (0.5686, 0.1765, 0.0745), (0.5608, 0.1686, 0.0784),
        (0.549, 0.1647, 0.0824), (0.5294, 0.1569, 0.0902)],
    'ncl:blwhre': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.502), (0., 0., 0.5216),
        (0., 0., 0.5412), (0., 0., 0.5608),
        (0., 0., 0.5804), (0., 0., 0.6),
        (0., 0., 0.6196), (0., 0., 0.6431),
        (0., 0., 0.6627), (0., 0., 0.6824),
        (0., 0., 0.702), (0., 0., 0.7216),
        (0., 0., 0.7412), (0., 0., 0.7608),
        (0., 0., 0.7804), (0., 0., 0.8),
        (0., 0., 0.8196), (0., 0., 0.8392),
        (0., 0., 0.8588), (0., 0., 0.8824),
        (0., 0., 0.902), (0., 0., 0.9216),
        (0., 0., 0.9412), (0., 0., 0.9608),
        (0., 0., 0.9804), (0., 0., 1.),
        (0.0392, 0.0392, 1.), (0.0784, 0.0784, 1.),
        (0.1216, 0.1216, 1.), (0.1608, 0.1608, 1.),
        (0.2, 0.2, 1.), (0.2392, 0.2392, 1.),
        (0.2784, 0.2784, 1.), (0.3216, 0.3216, 1.),
        (0.3608, 0.3608, 1.), (0.4, 0.4, 1.),
        (0.4392, 0.4392, 1.), (0.4784, 0.4784, 1.),
        (0.5216, 0.5216, 1.), (0.5608, 0.5608, 1.),
        (0.6, 0.6, 1.), (0.6392, 0.6392, 1.),
        (0.6784, 0.6784, 1.), (0.7216, 0.7216, 1.),
        (0.7608, 0.7608, 1.), (0.8, 0.8, 1.),
        (0.8392, 0.8392, 1.), (0.8784, 0.8784, 1.),
        (0.9216, 0.9216, 1.), (0.9608, 0.9608, 1.),
        (1., 1., 1.), (1., 0.9608, 0.9608),
        (1., 0.9216, 0.9216), (1., 0.8784, 0.8784),
        (1., 0.8392, 0.8392), (1., 0.8, 0.8),
        (1., 0.7608, 0.7608), (1., 0.7216, 0.7216),
        (1., 0.6784, 0.6784), (1., 0.6392, 0.6392),
        (1., 0.6, 0.6), (1., 0.5608, 0.5608),
        (1., 0.5216, 0.5216), (1., 0.4784, 0.4784),
        (1., 0.4392, 0.4392), (1., 0.4, 0.4),
        (1., 0.3608, 0.3608), (1., 0.3216, 0.3216),
        (1., 0.2784, 0.2784), (1., 0.2392, 0.2392),
        (1., 0.2, 0.2), (1., 0.1608, 0.1608),
        (1., 0.1216, 0.1216), (1., 0.0784, 0.0784),
        (1., 0.0392, 0.0392), (1., 0., 0.),
        (0.9804, 0., 0.), (0.9608, 0., 0.),
        (0.9412, 0., 0.), (0.9216, 0., 0.),
        (0.902, 0., 0.), (0.8824, 0., 0.),
        (0.8588, 0., 0.), (0.8392, 0., 0.),
        (0.8196, 0., 0.), (0.8, 0., 0.),
        (0.7804, 0., 0.), (0.7608, 0., 0.),
        (0.7412, 0., 0.), (0.7216, 0., 0.),
        (0.702, 0., 0.), (0.6824, 0., 0.),
        (0.6627, 0., 0.), (0.6431, 0., 0.),
        (0.6196, 0., 0.), (0.6, 0., 0.),
        (0.5804, 0., 0.), (0.5608, 0., 0.),
        (0.5412, 0., 0.), (0.5216, 0., 0.),
        (0.502, 0., 0.)],
    'ncl:detail': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 1.), (0., 0.0392, 1.),
        (0.0039, 0.0784, 0.9961), (0.0078, 0.1569, 0.9922),
        (0.0118, 0.2353, 0.9882), (0.0157, 0.3137, 0.9843),
        (0.0196, 0.4471, 0.9804), (0.0235, 0.5686, 0.9765),
        (0.0275, 0.6863, 0.9725), (0.0314, 0.7922, 0.9686),
        (0.0353, 0.8824, 0.9647), (0.0392, 0.9451, 0.9608),
        (0.0431, 0.9843, 0.9569), (0.0471, 0.9922, 0.9529),
        (0.051, 0.9725, 0.949), (0.0549, 0.9216, 0.9451),
        (0.0588, 0.8471, 0.9412), (0.0627, 0.7529, 0.9373),
        (0.0667, 0.6392, 0.9333), (0.0706, 0.5216, 0.9294),
        (0.0745, 0.4, 0.9255), (0.0784, 0.2824, 0.9216),
        (0.0824, 0.1804, 0.9176), (0.0863, 0.0941, 0.9137),
        (0.0902, 0.0353, 0.9098), (0.0941, 0.0039, 0.9059),
        (0.098, 0., 0.902), (0.102, 0.0275, 0.898),
        (0.1059, 0.0824, 0.8941), (0.1098, 0.1608, 0.8902),
        (0.1137, 0.2627, 0.8863), (0.1176, 0.3765, 0.8824),
        (0.1216, 0.4941, 0.8784), (0.1255, 0.6157, 0.8745),
        (0.1294, 0.7294, 0.8706), (0.1333, 0.8314, 0.8667),
        (0.1373, 0.9098, 0.8627), (0.1412, 0.9647, 0.8588),
        (0.1451, 0.9922, 0.8549), (0.149, 0.9882, 0.851),
        (0.1529, 0.9569, 0.8471), (0.1569, 0.898, 0.8431),
        (0.1608, 0.8118, 0.8392), (0.1647, 0.7098, 0.8353),
        (0.1686, 0.5922, 0.8314), (0.1725, 0.4706, 0.8275),
        (0.1765, 0.3529, 0.8235), (0.1804, 0.2392, 0.8196),
        (0.1843, 0.1451, 0.8157), (0.1882, 0.0706, 0.8118),
        (0.1922, 0.0196, 0.8078), (0.1961, 0., 0.8039),
        (0.2, 0.0078, 0.8), (0.2039, 0.0471, 0.7961),
        (0.2078, 0.1098, 0.7922), (0.2118, 0.2, 0.7882),
        (0.2157, 0.3059, 0.7843), (0.2196, 0.4235, 0.7804),
        (0.2235, 0.5451, 0.7765), (0.2275, 0.6627, 0.7725),
        (0.2314, 0.7725, 0.7686), (0.2353, 0.8667, 0.7647),
        (0.2392, 0.9373, 0.7608), (0.2431, 0.9804, 0.7569),
        (0.2471, 0.9961, 0.7529), (0.251, 0.9804, 0.749),
        (0.2549, 0.9373, 0.7451), (0.2588, 0.8667, 0.7412),
        (0.2627, 0.7725, 0.7373), (0.2667, 0.6627, 0.7333),
        (0.2706, 0.5451, 0.7294), (0.2745, 0.4235, 0.7255),
        (0.2784, 0.3059, 0.7216), (0.2824, 0.2, 0.7176),
        (0.2863, 0.1098, 0.7137), (0.2902, 0.0471, 0.7098),
        (0.2941, 0.0078, 0.7059), (0.298, 0., 0.702),
        (0.302, 0.0196, 0.698), (0.3059, 0.0706, 0.6941),
        (0.3098, 0.1451, 0.6902), (0.3137, 0.2392, 0.6863),
        (0.3176, 0.3529, 0.6824), (0.3216, 0.4706, 0.6784),
        (0.3255, 0.5922, 0.6745), (0.3294, 0.7098, 0.6706),
        (0.3333, 0.8118, 0.6667), (0.3373, 0.898, 0.6627),
        (0.3412, 0.9569, 0.6588), (0.3451, 0.9882, 0.6549),
        (0.349, 0.9922, 0.651), (0.3529, 0.9647, 0.6471),
        (0.3569, 0.9098, 0.6431), (0.3608, 0.8314, 0.6392),
        (0.3647, 0.7294, 0.6353), (0.3686, 0.6157, 0.6314),
        (0.3725, 0.498, 0.6275), (0.3765, 0.3765, 0.6235),
        (0.3804, 0.2627, 0.6196), (0.3843, 0.1608, 0.6157),
        (0.3882, 0.0824, 0.6118), (0.3922, 0.0275, 0.6078),
        (0.3961, 0., 0.6039), (0.4, 0.0039, 0.6),
        (0.4039, 0.0353, 0.5961), (0.4078, 0.0941, 0.5922),
        (0.4118, 0.1804, 0.5882), (0.4157, 0.2824, 0.5843),
        (0.4196, 0.4, 0.5804), (0.4235, 0.5216, 0.5765),
        (0.4275, 0.6392, 0.5725), (0.4314, 0.7529, 0.5686),
        (0.4353, 0.8471, 0.5647), (0.4392, 0.9216, 0.5608),
        (0.4431, 0.9725, 0.5569), (0.4471, 0.9922, 0.5529),
        (0.451, 0.9843, 0.549), (0.4549, 0.9451, 0.5451),
        (0.4588, 0.8824, 0.5412), (0.4627, 0.7922, 0.5373),
        (0.4667, 0.6863, 0.5333), (0.4706, 0.5686, 0.5294),
        (0.4745, 0.4471, 0.5255), (0.4784, 0.3294, 0.5216),
        (0.4824, 0.2196, 0.5176), (0.4863, 0.1255, 0.5137),
        (0.4902, 0.0549, 0.5098), (0.4941, 0.0118, 0.5059),
        (0.498, 0., 0.502), (0.502, 0.0118, 0.498),
        (0.5059, 0.0549, 0.4941), (0.5098, 0.1255, 0.4902),
        (0.5137, 0.2196, 0.4863), (0.5176, 0.3294, 0.4824),
        (0.5216, 0.4471, 0.4784), (0.5255, 0.5686, 0.4745),
        (0.5294, 0.6863, 0.4706), (0.5333, 0.7922, 0.4667),
        (0.5373, 0.8824, 0.4627), (0.5412, 0.9451, 0.4588),
        (0.5451, 0.9843, 0.4549), (0.549, 0.9922, 0.451),
        (0.5529, 0.9725, 0.4471), (0.5569, 0.9216, 0.4431),
        (0.5608, 0.8471, 0.4392), (0.5647, 0.7529, 0.4353),
        (0.5686, 0.6392, 0.4314), (0.5725, 0.5216, 0.4275),
        (0.5765, 0.4, 0.4235), (0.5804, 0.2824, 0.4196),
        (0.5843, 0.1804, 0.4157), (0.5882, 0.0941, 0.4118),
        (0.5922, 0.0353, 0.4078), (0.5961, 0.0039, 0.4039),
        (0.6, 0., 0.4), (0.6039, 0.0275, 0.3961),
        (0.6078, 0.0824, 0.3922), (0.6118, 0.1608, 0.3882),
        (0.6157, 0.2627, 0.3843), (0.6196, 0.3765, 0.3804),
        (0.6235, 0.4941, 0.3765), (0.6275, 0.6157, 0.3725),
        (0.6314, 0.7294, 0.3686), (0.6353, 0.8314, 0.3647),
        (0.6392, 0.9098, 0.3608), (0.6431, 0.9647, 0.3569),
        (0.6471, 0.9922, 0.3529), (0.651, 0.9882, 0.349),
        (0.6549, 0.9569, 0.3451), (0.6588, 0.898, 0.3412),
        (0.6627, 0.8118, 0.3373), (0.6667, 0.7098, 0.3333),
        (0.6706, 0.5922, 0.3294), (0.6745, 0.4706, 0.3255),
        (0.6784, 0.3529, 0.3216), (0.6824, 0.2392, 0.3176),
        (0.6863, 0.1451, 0.3137), (0.6902, 0.0706, 0.3098),
        (0.6941, 0.0196, 0.3059), (0.698, 0., 0.302),
        (0.702, 0.0078, 0.298), (0.7059, 0.0471, 0.2941),
        (0.7098, 0.1098, 0.2902), (0.7137, 0.2, 0.2863),
        (0.7176, 0.3059, 0.2824), (0.7216, 0.4235, 0.2784),
        (0.7255, 0.5451, 0.2745), (0.7294, 0.6627, 0.2706),
        (0.7333, 0.7725, 0.2667), (0.7373, 0.8667, 0.2627),
        (0.7412, 0.9373, 0.2588), (0.7451, 0.9804, 0.2549),
        (0.749, 0.9961, 0.251), (0.7529, 0.9804, 0.2471),
        (0.7569, 0.9373, 0.2431), (0.7608, 0.8667, 0.2392),
        (0.7647, 0.7725, 0.2353), (0.7686, 0.6627, 0.2314),
        (0.7725, 0.5451, 0.2275), (0.7765, 0.4235, 0.2235),
        (0.7804, 0.3059, 0.2196), (0.7843, 0.2, 0.2157),
        (0.7882, 0.1098, 0.2118), (0.7922, 0.0471, 0.2078),
        (0.7961, 0.0078, 0.2039), (0.8, 0., 0.2),
        (0.8039, 0.0196, 0.1961), (0.8078, 0.0706, 0.1922),
        (0.8118, 0.1451, 0.1882), (0.8157, 0.2392, 0.1843),
        (0.8196, 0.3529, 0.1804), (0.8235, 0.4706, 0.1765),
        (0.8275, 0.5922, 0.1725), (0.8314, 0.7098, 0.1686),
        (0.8353, 0.8118, 0.1647), (0.8392, 0.898, 0.1608),
        (0.8431, 0.9569, 0.1569), (0.8471, 0.9882, 0.1529),
        (0.851, 0.9922, 0.149), (0.8549, 0.9647, 0.1451),
        (0.8588, 0.9098, 0.1412), (0.8627, 0.8314, 0.1373),
        (0.8667, 0.7294, 0.1333), (0.8706, 0.6157, 0.1294),
        (0.8745, 0.498, 0.1255), (0.8784, 0.3765, 0.1216),
        (0.8824, 0.2627, 0.1176), (0.8863, 0.1608, 0.1137),
        (0.8902, 0.0824, 0.1098), (0.8941, 0.0275, 0.1059),
        (0.898, 0., 0.102), (0.902, 0.0039, 0.098),
        (0.9059, 0.0353, 0.0941), (0.9098, 0.0941, 0.0902),
        (0.9137, 0.1804, 0.0863), (0.9176, 0.2824, 0.0824),
        (0.9216, 0.4, 0.0784), (0.9255, 0.5216, 0.0745),
        (0.9294, 0.6392, 0.0706), (0.9333, 0.7529, 0.0667),
        (0.9373, 0.8471, 0.0627), (0.9412, 0.9216, 0.0588),
        (0.9451, 0.9725, 0.0549), (0.949, 0.9922, 0.051),
        (0.9529, 0.9843, 0.0471), (0.9569, 0.9451, 0.0431),
        (0.9608, 0.8824, 0.0392), (0.9647, 0.7922, 0.0353),
        (0.9686, 0.6863, 0.0314), (0.9725, 0.5686, 0.0275),
        (0.9765, 0.4471, 0.0235), (0.9804, 0.3294, 0.0196),
        (0.9843, 0.2196, 0.0157), (0.9882, 0.1255, 0.0118)],
    'ncl:example': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 0.9686), (1., 1., 0.9059),
        (1., 1., 0.8431), (1., 1., 0.7804),
        (1., 1., 0.7176), (1., 1., 0.6549),
        (1., 1., 0.5922), (1., 1., 0.5294),
        (1., 1., 0.4706), (1., 1., 0.4078),
        (1., 1., 0.3451), (1., 1., 0.2824),
        (1., 1., 0.2196), (1., 1., 0.1569),
        (1., 1., 0.0941), (1., 1., 0.0314),
        (1., 0.9686, 0.0314), (1., 0.9059, 0.0941),
        (1., 0.8431, 0.1569), (1., 0.7804, 0.2196),
        (1., 0.7176, 0.2824), (1., 0.6549, 0.3451),
        (1., 0.5922, 0.4078), (1., 0.5294, 0.4706),
        (1., 0.4706, 0.5294), (1., 0.4078, 0.5922),
        (1., 0.3451, 0.6549), (1., 0.2824, 0.7176),
        (1., 0.2196, 0.7804), (1., 0.1569, 0.8431),
        (1., 0.0941, 0.9059), (1., 0.0314, 0.9686),
        (1., 0., 0.9686), (1., 0., 0.9059),
        (1., 0., 0.8431), (1., 0., 0.7804),
        (1., 0., 0.7176), (1., 0., 0.6549),
        (1., 0., 0.5922), (1., 0., 0.5294),
        (1., 0., 0.4706), (1., 0., 0.4078),
        (1., 0., 0.3451), (1., 0., 0.2824),
        (1., 0., 0.2196), (1., 0., 0.1569),
        (1., 0., 0.0941), (1., 0., 0.0314),
        (0.9686, 0.0314, 0.0314), (0.9059, 0.0941, 0.0941),
        (0.8431, 0.1569, 0.1569), (0.7804, 0.2196, 0.2196),
        (0.7176, 0.2824, 0.2824), (0.6549, 0.3451, 0.3451),
        (0.5922, 0.4078, 0.4078), (0.5294, 0.4706, 0.4706),
        (0.4706, 0.5294, 0.5294), (0.4078, 0.5922, 0.5922),
        (0.3451, 0.6549, 0.6549), (0.2824, 0.7176, 0.7176),
        (0.2196, 0.7804, 0.7804), (0.1569, 0.8431, 0.8431),
        (0.0941, 0.9059, 0.9059), (0.0314, 0.9686, 0.9686),
        (0., 1., 0.9686), (0., 1., 0.9373),
        (0., 1., 0.8745), (0., 1., 0.8118),
        (0., 1., 0.7804), (0., 1., 0.7176),
        (0., 1., 0.6549), (0., 1., 0.5922),
        (0., 1., 0.5294), (0., 1., 0.4706),
        (0., 1., 0.4078), (0., 1., 0.3451),
        (0., 1., 0.2824), (0., 1., 0.2196),
        (0., 1., 0.1569), (0., 1., 0.0941),
        (0., 1., 0.0314), (0., 0.9686, 0.0314),
        (0., 0.9059, 0.0941), (0., 0.8431, 0.1569),
        (0., 0.7804, 0.2196), (0., 0.7176, 0.2824),
        (0., 0.6549, 0.3451), (0., 0.5922, 0.4078),
        (0., 0.5294, 0.4706), (0., 0.4706, 0.5294),
        (0., 0.4078, 0.5922), (0., 0.3451, 0.6549),
        (0., 0.2824, 0.7176), (0., 0.2196, 0.7804),
        (0., 0.1569, 0.8431), (0., 0.0941, 0.9059),
        (0., 0.0314, 0.9686), (0., 0., 0.9686),
        (0., 0., 0.9059), (0., 0., 0.8431),
        (0., 0., 0.7804), (0., 0., 0.7176),
        (0., 0., 0.6549), (0., 0., 0.5922),
        (0., 0., 0.5294), (0., 0., 0.4706),
        (0., 0., 0.4078), (0., 0., 0.3451),
        (0., 0., 0.2824), (0., 0., 0.2196),
        (0., 0., 0.1569), (0., 0., 0.0941),
        (0., 0., 0.0314)],
    'ncl:extrema': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 1.), (0.0078, 0., 0.9882),
        (0.0196, 0., 0.9765), (0.0314, 0.0039, 0.9647),
        (0.0431, 0.0118, 0.9529), (0.0549, 0.0353, 0.9412),
        (0.0667, 0.0784, 0.9294), (0.0745, 0.1529, 0.9216),
        (0.0863, 0.2627, 0.9098), (0.098, 0.4118, 0.898),
        (0.1059, 0.5843, 0.8902), (0.1176, 0.7569, 0.8784),
        (0.1255, 0.898, 0.8706), (0.1373, 0.9843, 0.8588),
        (0.1451, 0.9922, 0.851), (0.1529, 0.9333, 0.8431),
        (0.1647, 0.8157, 0.8314), (0.1725, 0.6706, 0.8235),
        (0.1804, 0.5216, 0.8157), (0.1882, 0.3882, 0.8078),
        (0.1961, 0.2902, 0.8), (0.2078, 0.2314, 0.7882),
        (0.2157, 0.2118, 0.7804), (0.2235, 0.2392, 0.7725),
        (0.2314, 0.302, 0.7647), (0.2392, 0.3961, 0.7569),
        (0.2431, 0.5098, 0.7529), (0.251, 0.6353, 0.7451),
        (0.2588, 0.7608, 0.7373), (0.2667, 0.8667, 0.7294),
        (0.2745, 0.9451, 0.7216), (0.2824, 0.9882, 0.7137),
        (0.2863, 0.9961, 0.7098), (0.2941, 0.9647, 0.702),
        (0.302, 0.9059, 0.6941), (0.3059, 0.8196, 0.6902),
        (0.3137, 0.7216, 0.6824), (0.3176, 0.6196, 0.6784),
        (0.3255, 0.5176, 0.6706), (0.3294, 0.4275, 0.6667),
        (0.3373, 0.349, 0.6588), (0.3412, 0.2902, 0.6549),
        (0.3451, 0.2471, 0.651), (0.3529, 0.2196, 0.6431),
        (0.3569, 0.2118, 0.6392), (0.3608, 0.2235, 0.6353),
        (0.3647, 0.2471, 0.6314), (0.3725, 0.2824, 0.6235),
        (0.3765, 0.3294, 0.6196), (0.3804, 0.3882, 0.6157),
        (0.3843, 0.4549, 0.6118), (0.3882, 0.5216, 0.6078),
        (0.3922, 0.5922, 0.6039), (0.3961, 0.6627, 0.6),
        (0.4, 0.7333, 0.5961), (0.4039, 0.7961, 0.5922),
        (0.4078, 0.851, 0.5882), (0.4118, 0.898, 0.5843),
        (0.4157, 0.9373, 0.5804), (0.4196, 0.9686, 0.5765),
        (0.4235, 0.9882, 0.5725), (0.4275, 0.9961, 0.5686),
        (0.4314, 0.9961, 0.5647), (0.4314, 0.9922, 0.5647),
        (0.4353, 0.9765, 0.5608), (0.4392, 0.9569, 0.5569),
        (0.4431, 0.9294, 0.5529), (0.4431, 0.898, 0.5529),
        (0.4471, 0.8667, 0.549), (0.451, 0.8275, 0.5451),
        (0.451, 0.7922, 0.5451), (0.4549, 0.7529, 0.5412),
        (0.4549, 0.7137, 0.5412), (0.4588, 0.6784, 0.5373),
        (0.4588, 0.6392, 0.5373), (0.4627, 0.6039, 0.5333),
        (0.4627, 0.5725, 0.5333), (0.4667, 0.5373, 0.5294),
        (0.4667, 0.5098, 0.5294), (0.4706, 0.4784, 0.5255),
        (0.4706, 0.4549, 0.5255), (0.4745, 0.4275, 0.5216),
        (0.4745, 0.4078, 0.5216), (0.4745, 0.3843, 0.5216),
        (0.4784, 0.3647, 0.5176), (0.4784, 0.349, 0.5176),
        (0.4784, 0.3333, 0.5176), (0.4824, 0.3176, 0.5137),
        (0.4824, 0.3059, 0.5137), (0.4824, 0.2941, 0.5137),
        (0.4863, 0.2863, 0.5098), (0.4863, 0.2745, 0.5098),
        (0.4863, 0.2667, 0.5098), (0.4863, 0.2588, 0.5098),
        (0.4902, 0.2549, 0.5059), (0.4902, 0.2471, 0.5059),
        (0.4902, 0.2431, 0.5059), (0.4902, 0.2392, 0.5059),
        (0.4902, 0.2353, 0.5059), (0.4941, 0.2314, 0.502),
        (0.4941, 0.2314, 0.502), (0.4941, 0.2275, 0.502),
        (0.4941, 0.2235, 0.502), (0.4941, 0.2235, 0.502),
        (0.4941, 0.2235, 0.502), (0.4941, 0.2196, 0.502),
        (0.4941, 0.2196, 0.502), (0.4941, 0.2196, 0.502),
        (0.498, 0.2196, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2157, 0.498),
        (0.498, 0.2157, 0.498), (0.498, 0.2118, 0.498),
        (0.498, 0.2118, 0.498), (0.498, 0.2118, 0.498),
        (0.498, 0.2118, 0.498), (0.498, 0.2118, 0.498),
        (0.498, 0.2118, 0.498), (0.502, 0.2118, 0.4941),
        (0.502, 0.2118, 0.4941), (0.502, 0.2118, 0.4941),
        (0.502, 0.2157, 0.4941), (0.502, 0.2157, 0.4941),
        (0.502, 0.2157, 0.4941), (0.502, 0.2157, 0.4941),
        (0.502, 0.2157, 0.4941), (0.502, 0.2196, 0.4941),
        (0.5059, 0.2196, 0.4902), (0.5059, 0.2235, 0.4902),
        (0.5059, 0.2235, 0.4902), (0.5059, 0.2275, 0.4902),
        (0.5059, 0.2314, 0.4902), (0.5098, 0.2353, 0.4863),
        (0.5098, 0.2392, 0.4863), (0.5098, 0.2471, 0.4863),
        (0.5098, 0.251, 0.4863), (0.5137, 0.2588, 0.4824),
        (0.5137, 0.2706, 0.4824), (0.5137, 0.2784, 0.4824),
        (0.5176, 0.2902, 0.4784), (0.5176, 0.3059, 0.4784),
        (0.5176, 0.3176, 0.4784), (0.5216, 0.3333, 0.4745),
        (0.5216, 0.3529, 0.4745), (0.5216, 0.3725, 0.4745),
        (0.5255, 0.3961, 0.4706), (0.5255, 0.4196, 0.4706),
        (0.5294, 0.4471, 0.4667), (0.5294, 0.4745, 0.4667),
        (0.5333, 0.5059, 0.4627), (0.5333, 0.5373, 0.4627),
        (0.5373, 0.5725, 0.4588), (0.5373, 0.6078, 0.4588),
        (0.5412, 0.6471, 0.4549), (0.5412, 0.6863, 0.4549),
        (0.5451, 0.7255, 0.451), (0.5451, 0.7686, 0.451),
        (0.549, 0.8078, 0.4471), (0.5529, 0.8471, 0.4431),
        (0.5529, 0.8824, 0.4431), (0.5569, 0.9176, 0.4392),
        (0.5608, 0.9451, 0.4353), (0.5647, 0.9686, 0.4314),
        (0.5647, 0.9882, 0.4314), (0.5686, 0.9961, 0.4275),
        (0.5725, 0.9961, 0.4235), (0.5765, 0.9882, 0.4196),
        (0.5804, 0.9686, 0.4157), (0.5843, 0.9412, 0.4118),
        (0.5882, 0.902, 0.4078), (0.5922, 0.8549, 0.4039),
        (0.5961, 0.7961, 0.4), (0.6, 0.7294, 0.3961),
        (0.6039, 0.6627, 0.3922), (0.6078, 0.5882, 0.3882),
        (0.6118, 0.5176, 0.3843), (0.6157, 0.4471, 0.3804),
        (0.6196, 0.3804, 0.3765), (0.6235, 0.3216, 0.3725),
        (0.6314, 0.2745, 0.3647), (0.6353, 0.2392, 0.3608),
        (0.6392, 0.2196, 0.3569), (0.6431, 0.2118, 0.3529),
        (0.651, 0.2275, 0.3451), (0.6549, 0.2549, 0.3412),
        (0.6588, 0.3059, 0.3373), (0.6667, 0.3725, 0.3294),
        (0.6706, 0.4549, 0.3255), (0.6784, 0.549, 0.3176),
        (0.6824, 0.6549, 0.3137), (0.6902, 0.7569, 0.3059),
        (0.6941, 0.8549, 0.302), (0.702, 0.9333, 0.2941),
        (0.7098, 0.9804, 0.2863), (0.7137, 1., 0.2824),
        (0.7216, 0.9765, 0.2745), (0.7294, 0.9137, 0.2667),
        (0.7373, 0.8196, 0.2588), (0.7451, 0.702, 0.251),
        (0.7529, 0.5765, 0.2431), (0.7569, 0.451, 0.2392),
        (0.7647, 0.3451, 0.2314), (0.7725, 0.2667, 0.2235),
        (0.7804, 0.2196, 0.2157), (0.7882, 0.2157, 0.2078),
        (0.8, 0.2549, 0.1961), (0.8078, 0.3373, 0.1882),
        (0.8157, 0.4549, 0.1804), (0.8235, 0.6, 0.1725),
        (0.8314, 0.7529, 0.1647), (0.8431, 0.8863, 0.1529),
        (0.851, 0.9765, 0.1451), (0.8588, 0.9961, 0.1373),
        (0.8706, 0.9412, 0.1255), (0.8784, 0.8196, 0.1176),
        (0.8902, 0.6549, 0.1059), (0.898, 0.4745, 0.098),
        (0.9098, 0.3137, 0.0863), (0.9216, 0.1882, 0.0745),
        (0.9294, 0.102, 0.0667), (0.9412, 0.0471, 0.0549),
        (0.9529, 0.0196, 0.0431), (0.9647, 0.0078, 0.0314)],
    'ncl:graywhitegray': [
        (1., 1., 1.), (0., 0., 0.),
        (0.3725, 0.3725, 0.3725), (0.3804, 0.3804, 0.3804),
        (0.3882, 0.3882, 0.3882), (0.3961, 0.3961, 0.3961),
        (0.4039, 0.4039, 0.4039), (0.4078, 0.4078, 0.4078),
        (0.4157, 0.4157, 0.4157), (0.4235, 0.4235, 0.4235),
        (0.4314, 0.4314, 0.4314), (0.4392, 0.4392, 0.4392),
        (0.4471, 0.4471, 0.4471), (0.4549, 0.4549, 0.4549),
        (0.4627, 0.4627, 0.4627), (0.4667, 0.4667, 0.4667),
        (0.4745, 0.4745, 0.4745), (0.4824, 0.4824, 0.4824),
        (0.4902, 0.4902, 0.4902), (0.498, 0.498, 0.498),
        (0.5059, 0.5059, 0.5059), (0.5137, 0.5137, 0.5137),
        (0.5216, 0.5216, 0.5216), (0.5294, 0.5294, 0.5294),
        (0.5333, 0.5333, 0.5333), (0.5412, 0.5412, 0.5412),
        (0.549, 0.549, 0.549), (0.5569, 0.5569, 0.5569),
        (0.5647, 0.5647, 0.5647), (0.5725, 0.5725, 0.5725),
        (0.5804, 0.5804, 0.5804), (0.5882, 0.5882, 0.5882),
        (0.5922, 0.5922, 0.5922), (0.6, 0.6, 0.6),
        (0.6078, 0.6078, 0.6078), (0.6157, 0.6157, 0.6157),
        (0.6235, 0.6235, 0.6235), (0.6314, 0.6314, 0.6314),
        (0.6392, 0.6392, 0.6392), (0.6471, 0.6471, 0.6471),
        (0.6549, 0.6549, 0.6549), (0.6588, 0.6588, 0.6588),
        (0.6667, 0.6667, 0.6667), (0.6745, 0.6745, 0.6745),
        (0.6824, 0.6824, 0.6824), (0.6902, 0.6902, 0.6902),
        (0.698, 0.698, 0.698), (0.7059, 0.7059, 0.7059),
        (0.7137, 0.7137, 0.7137), (0.7176, 0.7176, 0.7176),
        (0.7255, 0.7255, 0.7255), (0.7333, 0.7333, 0.7333),
        (0.7412, 0.7412, 0.7412), (0.749, 0.749, 0.749),
        (0.7569, 0.7569, 0.7569), (0.7647, 0.7647, 0.7647),
        (0.7725, 0.7725, 0.7725), (0.7804, 0.7804, 0.7804),
        (0.7843, 0.7843, 0.7843), (0.7922, 0.7922, 0.7922),
        (0.8, 0.8, 0.8), (0.8078, 0.8078, 0.8078),
        (0.8157, 0.8157, 0.8157), (0.8235, 0.8235, 0.8235),
        (0.8314, 0.8314, 0.8314), (0.8392, 0.8392, 0.8392),
        (0.8431, 0.8431, 0.8431), (0.851, 0.851, 0.851),
        (0.8588, 0.8588, 0.8588), (0.8667, 0.8667, 0.8667),
        (0.8745, 0.8745, 0.8745), (0.8824, 0.8824, 0.8824),
        (0.8902, 0.8902, 0.8902), (0.898, 0.898, 0.898),
        (0.9059, 0.9059, 0.9059), (0.9098, 0.9098, 0.9098),
        (0.9176, 0.9176, 0.9176), (0.9255, 0.9255, 0.9255),
        (0.9333, 0.9333, 0.9333), (0.9412, 0.9412, 0.9412),
        (0.949, 0.949, 0.949), (0.9569, 0.9569, 0.9569),
        (0.9647, 0.9647, 0.9647), (0.9686, 0.9686, 0.9686),
        (0.9765, 0.9765, 0.9765), (0.9843, 0.9843, 0.9843),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (0.9922, 0.9922, 0.9922), (0.9843, 0.9843, 0.9843),
        (0.9765, 0.9765, 0.9765), (0.9686, 0.9686, 0.9686),
        (0.9647, 0.9647, 0.9647), (0.9569, 0.9569, 0.9569),
        (0.949, 0.949, 0.949), (0.9412, 0.9412, 0.9412),
        (0.9333, 0.9333, 0.9333), (0.9255, 0.9255, 0.9255),
        (0.9176, 0.9176, 0.9176), (0.9098, 0.9098, 0.9098),
        (0.9059, 0.9059, 0.9059), (0.898, 0.898, 0.898),
        (0.8902, 0.8902, 0.8902), (0.8824, 0.8824, 0.8824),
        (0.8745, 0.8745, 0.8745), (0.8667, 0.8667, 0.8667),
        (0.8588, 0.8588, 0.8588), (0.851, 0.851, 0.851),
        (0.8431, 0.8431, 0.8431), (0.8392, 0.8392, 0.8392),
        (0.8314, 0.8314, 0.8314), (0.8235, 0.8235, 0.8235),
        (0.8157, 0.8157, 0.8157), (0.8078, 0.8078, 0.8078),
        (0.8, 0.8, 0.8), (0.7922, 0.7922, 0.7922),
        (0.7843, 0.7843, 0.7843), (0.7804, 0.7804, 0.7804),
        (0.7725, 0.7725, 0.7725), (0.7647, 0.7647, 0.7647),
        (0.7569, 0.7569, 0.7569), (0.749, 0.749, 0.749),
        (0.7412, 0.7412, 0.7412), (0.7333, 0.7333, 0.7333),
        (0.7255, 0.7255, 0.7255), (0.7176, 0.7176, 0.7176),
        (0.7137, 0.7137, 0.7137), (0.7059, 0.7059, 0.7059),
        (0.698, 0.698, 0.698), (0.6902, 0.6902, 0.6902),
        (0.6824, 0.6824, 0.6824), (0.6745, 0.6745, 0.6745),
        (0.6667, 0.6667, 0.6667), (0.6588, 0.6588, 0.6588),
        (0.6549, 0.6549, 0.6549), (0.6471, 0.6471, 0.6471),
        (0.6392, 0.6392, 0.6392), (0.6314, 0.6314, 0.6314),
        (0.6235, 0.6235, 0.6235), (0.6157, 0.6157, 0.6157),
        (0.6078, 0.6078, 0.6078), (0.6, 0.6, 0.6),
        (0.5922, 0.5922, 0.5922), (0.5882, 0.5882, 0.5882),
        (0.5804, 0.5804, 0.5804), (0.5725, 0.5725, 0.5725),
        (0.5647, 0.5647, 0.5647), (0.5569, 0.5569, 0.5569),
        (0.549, 0.549, 0.549), (0.5412, 0.5412, 0.5412),
        (0.5333, 0.5333, 0.5333), (0.5294, 0.5294, 0.5294),
        (0.5216, 0.5216, 0.5216), (0.5137, 0.5137, 0.5137),
        (0.5059, 0.5059, 0.5059), (0.498, 0.498, 0.498),
        (0.4902, 0.4902, 0.4902), (0.4824, 0.4824, 0.4824),
        (0.4745, 0.4745, 0.4745), (0.4667, 0.4667, 0.4667),
        (0.4627, 0.4627, 0.4627), (0.4549, 0.4549, 0.4549),
        (0.4471, 0.4471, 0.4471), (0.4392, 0.4392, 0.4392),
        (0.4314, 0.4314, 0.4314), (0.4235, 0.4235, 0.4235),
        (0.4157, 0.4157, 0.4157), (0.4078, 0.4078, 0.4078),
        (0.4039, 0.4039, 0.4039), (0.3961, 0.3961, 0.3961),
        (0.3882, 0.3882, 0.3882), (0.3725, 0.3725, 0.3725)],
    'ncl:greenyellow': [
        (1., 1., 1.), (0., 0., 0.),
        (0.098, 1., 0.3216), (0.1059, 1., 0.3294),
        (0.1098, 1., 0.3333), (0.1176, 1., 0.3373),
        (0.1216, 1., 0.3412), (0.1294, 1., 0.3451),
        (0.1333, 1., 0.349), (0.1412, 1., 0.3529),
        (0.1451, 1., 0.3608), (0.1529, 1., 0.3647),
        (0.1569, 1., 0.3686), (0.1647, 1., 0.3725),
        (0.1686, 1., 0.3765), (0.1765, 1., 0.3804),
        (0.1804, 1., 0.3843), (0.1882, 1., 0.3922),
        (0.1922, 1., 0.3961), (0.2, 1., 0.4),
        (0.2078, 1., 0.4039), (0.2118, 1., 0.4078),
        (0.2196, 1., 0.4118), (0.2235, 1., 0.4157),
        (0.2314, 1., 0.4235), (0.2353, 1., 0.4275),
        (0.2431, 1., 0.4314), (0.2471, 1., 0.4353),
        (0.2549, 1., 0.4392), (0.2588, 1., 0.4431),
        (0.2667, 1., 0.451), (0.2706, 1., 0.4549),
        (0.2784, 1., 0.4588), (0.2824, 1., 0.4627),
        (0.2902, 1., 0.4667), (0.2941, 1., 0.4706),
        (0.302, 1., 0.4745), (0.3098, 1., 0.4824),
        (0.3137, 1., 0.4863), (0.3216, 1., 0.4902),
        (0.3255, 1., 0.4941), (0.3333, 1., 0.498),
        (0.3373, 1., 0.502), (0.3451, 1., 0.5059),
        (0.349, 1., 0.5137), (0.3569, 1., 0.5176),
        (0.3608, 1., 0.5216), (0.3686, 1., 0.5255),
        (0.3725, 1., 0.5294), (0.3804, 1., 0.5333),
        (0.3843, 1., 0.5373), (0.3922, 1., 0.5451),
        (0.4, 1., 0.549), (0.4039, 1., 0.5529),
        (0.4118, 1., 0.5569), (0.4157, 1., 0.5608),
        (0.4235, 1., 0.5647), (0.4275, 1., 0.5686),
        (0.4353, 1., 0.5765), (0.4392, 1., 0.5804),
        (0.4471, 1., 0.5843), (0.451, 1., 0.5882),
        (0.4588, 1., 0.5922), (0.4627, 1., 0.5961),
        (0.4706, 1., 0.6039), (0.4745, 1., 0.6078),
        (0.4824, 1., 0.6118), (0.4863, 1., 0.6157),
        (0.4941, 1., 0.6196), (0.502, 1., 0.6235),
        (0.5059, 1., 0.6275), (0.5137, 1., 0.6353),
        (0.5176, 1., 0.6392), (0.5255, 1., 0.6431),
        (0.5294, 1., 0.6471), (0.5373, 1., 0.651),
        (0.5412, 1., 0.6549), (0.549, 1., 0.6588),
        (0.5529, 1., 0.6667), (0.5608, 1., 0.6706),
        (0.5647, 1., 0.6745), (0.5725, 1., 0.6784),
        (0.5765, 1., 0.6824), (0.5843, 1., 0.6863),
        (0.5882, 1., 0.6902), (0.5961, 1., 0.698),
        (0.6039, 1., 0.702), (0.6078, 1., 0.7059),
        (0.6157, 1., 0.7098), (0.6196, 1., 0.7137),
        (0.6275, 1., 0.7176), (0.6314, 1., 0.7255),
        (0.6392, 1., 0.7294), (0.6431, 1., 0.7333),
        (0.651, 1., 0.7373), (0.6549, 1., 0.7412),
        (0.6627, 1., 0.7451), (0.6667, 1., 0.749),
        (0.6745, 1., 0.7569), (0.6784, 1., 0.7608),
        (0.6863, 1., 0.7647), (0.6902, 1., 0.7686),
        (0.698, 1., 0.7725), (0.7059, 1., 0.7765),
        (0.7098, 1., 0.7804), (0.7176, 1., 0.7882),
        (0.7216, 1., 0.7922), (0.7294, 1., 0.7961),
        (0.7333, 1., 0.8), (0.7412, 1., 0.8039),
        (0.7451, 1., 0.8078), (0.7529, 1., 0.8118),
        (0.7569, 1., 0.8196), (0.7647, 1., 0.8235),
        (0.7686, 1., 0.8275), (0.7765, 1., 0.8314),
        (0.7804, 1., 0.8353), (0.7882, 1., 0.8392),
        (0.7922, 1., 0.8431), (0.8, 1., 0.851),
        (0.8078, 1., 0.8549), (0.8118, 1., 0.8588),
        (0.8196, 1., 0.8627), (0.8235, 1., 0.8667),
        (0.8314, 1., 0.8706), (0.8353, 1., 0.8784),
        (0.8431, 1., 0.8824), (0.8471, 1., 0.8863),
        (0.9804, 1., 0.8471), (0.9804, 1., 0.8431),
        (0.9765, 1., 0.8353), (0.9765, 1., 0.8314),
        (0.9765, 1., 0.8235), (0.9765, 1., 0.8196),
        (0.9765, 1., 0.8118), (0.9725, 1., 0.8078),
        (0.9725, 1., 0.8), (0.9725, 1., 0.7922),
        (0.9725, 1., 0.7882), (0.9725, 1., 0.7804),
        (0.9686, 1., 0.7765), (0.9686, 1., 0.7686),
        (0.9686, 1., 0.7647), (0.9686, 1., 0.7569),
        (0.9686, 1., 0.7529), (0.9647, 1., 0.7451),
        (0.9647, 1., 0.7412), (0.9647, 1., 0.7333),
        (0.9647, 1., 0.7294), (0.9647, 1., 0.7216),
        (0.9647, 1., 0.7176), (0.9608, 1., 0.7098),
        (0.9608, 1., 0.7059), (0.9608, 1., 0.698),
        (0.9608, 1., 0.6902), (0.9608, 1., 0.6863),
        (0.9569, 1., 0.6784), (0.9569, 1., 0.6745),
        (0.9569, 1., 0.6667), (0.9569, 1., 0.6627),
        (0.9569, 1., 0.6549), (0.9529, 1., 0.651),
        (0.9529, 1., 0.6431), (0.9529, 1., 0.6392),
        (0.9529, 1., 0.6314), (0.9529, 1., 0.6275),
        (0.949, 1., 0.6196), (0.949, 1., 0.6157),
        (0.949, 1., 0.6078), (0.949, 1., 0.6039),
        (0.949, 1., 0.5961), (0.9451, 1., 0.5882),
        (0.9451, 1., 0.5843), (0.9451, 1., 0.5765),
        (0.9451, 1., 0.5725), (0.9451, 1., 0.5647),
        (0.9451, 1., 0.5608), (0.9412, 1., 0.5529),
        (0.9412, 1., 0.549), (0.9412, 1., 0.5412),
        (0.9412, 1., 0.5373), (0.9412, 1., 0.5294),
        (0.9373, 1., 0.5255), (0.9373, 1., 0.5176),
        (0.9373, 1., 0.5137), (0.9373, 1., 0.5059),
        (0.9373, 1., 0.502), (0.9333, 1., 0.4941),
        (0.9333, 1., 0.4863), (0.9333, 1., 0.4824),
        (0.9333, 1., 0.4745), (0.9333, 1., 0.4706),
        (0.9294, 1., 0.4627), (0.9294, 1., 0.4588),
        (0.9294, 1., 0.451), (0.9294, 1., 0.4471),
        (0.9294, 1., 0.4392), (0.9294, 1., 0.4353),
        (0.9255, 1., 0.4275), (0.9255, 1., 0.4235),
        (0.9255, 1., 0.4157), (0.9255, 1., 0.4118),
        (0.9255, 1., 0.4039), (0.9216, 1., 0.3961),
        (0.9216, 1., 0.3922), (0.9216, 1., 0.3843),
        (0.9216, 1., 0.3804), (0.9216, 1., 0.3725),
        (0.9176, 1., 0.3686), (0.9176, 1., 0.3608),
        (0.9176, 1., 0.3569), (0.9176, 1., 0.349),
        (0.9176, 1., 0.3451), (0.9137, 1., 0.3373),
        (0.9137, 1., 0.3333), (0.9137, 1., 0.3255),
        (0.9137, 1., 0.3216), (0.9137, 1., 0.3137),
        (0.9137, 1., 0.3098), (0.9098, 1., 0.302),
        (0.9098, 1., 0.2941), (0.9098, 1., 0.2902),
        (0.9098, 1., 0.2824), (0.9098, 1., 0.2784),
        (0.9059, 1., 0.2706), (0.9059, 1., 0.2667),
        (0.9059, 1., 0.2588), (0.9059, 1., 0.2549),
        (0.9059, 1., 0.2471), (0.902, 1., 0.2431),
        (0.902, 1., 0.2353), (0.902, 1., 0.2314),
        (0.902, 1., 0.2235), (0.902, 1., 0.2196),
        (0.898, 1., 0.2118), (0.898, 1., 0.2078),
        (0.898, 1., 0.2), (0.898, 1., 0.1922),
        (0.898, 1., 0.1882), (0.8941, 1., 0.1804),
        (0.8941, 1., 0.1765), (0.8941, 1., 0.1686),
        (0.8941, 1., 0.1647), (0.8941, 1., 0.1569),
        (0.8941, 1., 0.1529), (0.8902, 1., 0.1451),
        (0.8902, 1., 0.1412), (0.8902, 1., 0.1333),
        (0.8902, 1., 0.1294), (0.8902, 1., 0.1216),
        (0.8863, 1., 0.1176), (0.8863, 1., 0.1098),
        (0.8863, 1., 0.1059), (0.8863, 1., 0.098)],
    'ncl:helix': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0039, 0.0039, 0.), (0.0078, 0.0039, 0.),
        (0.0118, 0.0078, 0.), (0.0157, 0.0078, 0.),
        (0.0196, 0.0118, 0.), (0.0235, 0.0157, 0.),
        (0.0275, 0.0157, 0.), (0.0275, 0.0196, 0.),
        (0.0314, 0.0235, 0.), (0.0353, 0.0235, 0.),
        (0.0392, 0.0275, 0.), (0.0392, 0.0314, 0.),
        (0.0431, 0.0353, 0.), (0.0471, 0.0392, 0.),
        (0.0471, 0.0431, 0.), (0.051, 0.0431, 0.),
        (0.0549, 0.0471, 0.), (0.0549, 0.051, 0.),
        (0.0588, 0.0549, 0.), (0.0588, 0.0588, 0.),
        (0.0627, 0.0627, 0.), (0.0627, 0.0667, 0.),
        (0.0667, 0.0706, 0.), (0.0667, 0.0745, 0.),
        (0.0706, 0.0784, 0.), (0.0706, 0.0824, 0.),
        (0.0706, 0.0863, 0.), (0.0745, 0.0902, 0.),
        (0.0745, 0.098, 0.), (0.0745, 0.102, 0.),
        (0.0784, 0.1059, 0.), (0.0784, 0.1098, 0.),
        (0.0784, 0.1137, 0.), (0.0784, 0.1176, 0.),
        (0.0784, 0.1216, 0.0039), (0.0824, 0.1294, 0.0039),
        (0.0824, 0.1333, 0.0039), (0.0824, 0.1373, 0.0039),
        (0.0824, 0.1412, 0.0078), (0.0824, 0.1451, 0.0078),
        (0.0824, 0.1529, 0.0078), (0.0824, 0.1569, 0.0078),
        (0.0824, 0.1608, 0.0118), (0.0824, 0.1647, 0.0118),
        (0.0824, 0.1686, 0.0157), (0.0824, 0.1725, 0.0157),
        (0.0784, 0.1804, 0.0157), (0.0784, 0.1843, 0.0196),
        (0.0784, 0.1882, 0.0196), (0.0784, 0.1922, 0.0235),
        (0.0784, 0.1961, 0.0275), (0.0745, 0.2039, 0.0275),
        (0.0745, 0.2078, 0.0314), (0.0745, 0.2118, 0.0314),
        (0.0706, 0.2157, 0.0353), (0.0706, 0.2196, 0.0392),
        (0.0706, 0.2235, 0.0392), (0.0667, 0.2275, 0.0431),
        (0.0667, 0.2314, 0.0471), (0.0667, 0.2353, 0.051),
        (0.0627, 0.2392, 0.0549), (0.0627, 0.2431, 0.0588),
        (0.0627, 0.2471, 0.0588), (0.0588, 0.251, 0.0627),
        (0.0588, 0.2549, 0.0667), (0.0549, 0.2588, 0.0706),
        (0.0549, 0.2627, 0.0745), (0.051, 0.2667, 0.0824),
        (0.051, 0.2706, 0.0863), (0.0471, 0.2745, 0.0902),
        (0.0471, 0.2784, 0.0941), (0.0431, 0.2824, 0.098),
        (0.0431, 0.2824, 0.102), (0.0392, 0.2863, 0.1098),
        (0.0392, 0.2902, 0.1137), (0.0353, 0.2941, 0.1176),
        (0.0353, 0.2941, 0.1255), (0.0314, 0.298, 0.1294),
        (0.0314, 0.298, 0.1333), (0.0275, 0.302, 0.1412),
        (0.0275, 0.3059, 0.1451), (0.0235, 0.3059, 0.1529),
        (0.0235, 0.3098, 0.1569), (0.0196, 0.3098, 0.1647),
        (0.0196, 0.3098, 0.1686), (0.0157, 0.3137, 0.1765),
        (0.0157, 0.3137, 0.1804), (0.0157, 0.3137, 0.1882),
        (0.0118, 0.3176, 0.1961), (0.0118, 0.3176, 0.2),
        (0.0078, 0.3176, 0.2078), (0.0078, 0.3176, 0.2157),
        (0.0078, 0.3176, 0.2196), (0.0039, 0.3176, 0.2275),
        (0.0039, 0.3176, 0.2353), (0.0039, 0.3176, 0.2431),
        (0.0039, 0.3176, 0.2471), (0., 0.3176, 0.2549),
        (0., 0.3176, 0.2627), (0., 0.3176, 0.2706),
        (0., 0.3176, 0.2784), (0., 0.3176, 0.2863),
        (0., 0.3137, 0.2902), (0., 0.3137, 0.298),
        (0., 0.3137, 0.3059), (0., 0.3098, 0.3137),
        (0., 0.3098, 0.3216), (0., 0.3098, 0.3294),
        (0., 0.3059, 0.3373), (0., 0.302, 0.3451),
        (0., 0.302, 0.3529), (0., 0.298, 0.3569),
        (0., 0.298, 0.3647), (0.0039, 0.2941, 0.3725),
        (0.0039, 0.2902, 0.3804), (0.0039, 0.2902, 0.3882),
        (0.0078, 0.2863, 0.3961), (0.0078, 0.2824, 0.4039),
        (0.0118, 0.2784, 0.4118), (0.0118, 0.2745, 0.4157),
        (0.0157, 0.2706, 0.4235), (0.0157, 0.2706, 0.4314),
        (0.0196, 0.2667, 0.4392), (0.0235, 0.2627, 0.4471),
        (0.0235, 0.2588, 0.451), (0.0275, 0.2549, 0.4588),
        (0.0314, 0.251, 0.4667), (0.0353, 0.2431, 0.4745),
        (0.0392, 0.2392, 0.4784), (0.0431, 0.2353, 0.4863),
        (0.0471, 0.2314, 0.4941), (0.051, 0.2275, 0.498),
        (0.0549, 0.2235, 0.5059), (0.0588, 0.2157, 0.5098),
        (0.0627, 0.2118, 0.5176), (0.0706, 0.2078, 0.5216),
        (0.0745, 0.2039, 0.5294), (0.0784, 0.1961, 0.5333),
        (0.0863, 0.1922, 0.5412), (0.0902, 0.1882, 0.5451),
        (0.098, 0.1843, 0.549), (0.102, 0.1765, 0.5569),
        (0.1098, 0.1725, 0.5608), (0.1137, 0.1686, 0.5647),
        (0.1216, 0.1608, 0.5686), (0.1294, 0.1569, 0.5725),
        (0.1333, 0.1529, 0.5765), (0.1412, 0.1451, 0.5804),
        (0.149, 0.1412, 0.5843), (0.1569, 0.1373, 0.5882),
        (0.1647, 0.1294, 0.5922), (0.1725, 0.1255, 0.5961),
        (0.1804, 0.1216, 0.6), (0.1882, 0.1137, 0.6039),
        (0.1961, 0.1098, 0.6039), (0.2039, 0.1059, 0.6078),
        (0.2118, 0.098, 0.6118), (0.2235, 0.0941, 0.6118),
        (0.2314, 0.0902, 0.6157), (0.2392, 0.0863, 0.6157),
        (0.2471, 0.0784, 0.6196), (0.2588, 0.0745, 0.6196),
        (0.2667, 0.0706, 0.6196), (0.2784, 0.0667, 0.6196),
        (0.2863, 0.0627, 0.6235), (0.2941, 0.0588, 0.6235),
        (0.3059, 0.0549, 0.6235), (0.3176, 0.051, 0.6235),
        (0.3255, 0.0471, 0.6235), (0.3373, 0.0431, 0.6235),
        (0.3451, 0.0392, 0.6235), (0.3569, 0.0353, 0.6196),
        (0.3686, 0.0314, 0.6196), (0.3765, 0.0275, 0.6196),
        (0.3882, 0.0235, 0.6157), (0.4, 0.0196, 0.6157),
        (0.4078, 0.0196, 0.6118), (0.4196, 0.0157, 0.6118),
        (0.4314, 0.0118, 0.6078), (0.4431, 0.0118, 0.6078),
        (0.451, 0.0078, 0.6039), (0.4627, 0.0078, 0.6),
        (0.4745, 0.0039, 0.5961), (0.4863, 0.0039, 0.5922),
        (0.498, 0.0039, 0.5882), (0.5059, 0., 0.5843),
        (0.5176, 0., 0.5804), (0.5294, 0., 0.5765),
        (0.5412, 0., 0.5725), (0.5529, 0., 0.5686),
        (0.5608, 0., 0.5608), (0.5725, 0., 0.5569),
        (0.5843, 0., 0.5529), (0.5961, 0., 0.5451),
        (0.6078, 0., 0.5412), (0.6157, 0., 0.5333),
        (0.6275, 0.0039, 0.5294), (0.6392, 0.0039, 0.5216),
        (0.6471, 0.0039, 0.5176), (0.6588, 0.0078, 0.5098),
        (0.6706, 0.0118, 0.502), (0.6784, 0.0118, 0.4941),
        (0.6902, 0.0157, 0.4902), (0.702, 0.0196, 0.4824),
        (0.7098, 0.0235, 0.4745), (0.7216, 0.0235, 0.4667),
        (0.7294, 0.0275, 0.4588), (0.7373, 0.0314, 0.451),
        (0.749, 0.0392, 0.4431), (0.7569, 0.0431, 0.4353),
        (0.7686, 0.0471, 0.4275), (0.7765, 0.051, 0.4196),
        (0.7843, 0.0588, 0.4118), (0.7922, 0.0627, 0.4039),
        (0.8, 0.0706, 0.3922), (0.8078, 0.0745, 0.3843),
        (0.8196, 0.0824, 0.3765), (0.8275, 0.0902, 0.3686),
        (0.8314, 0.0941, 0.3608), (0.8392, 0.102, 0.349),
        (0.8471, 0.1098, 0.3412), (0.8549, 0.1176, 0.3333),
        (0.8627, 0.1255, 0.3255), (0.8667, 0.1333, 0.3137),
        (0.8745, 0.1451, 0.3059), (0.8784, 0.1529, 0.298),
        (0.8863, 0.1608, 0.2863), (0.8902, 0.1725, 0.2784),
        (0.898, 0.1804, 0.2706), (0.902, 0.1882, 0.2588),
        (0.9059, 0.2, 0.251), (0.9098, 0.2118, 0.2431),
        (0.9137, 0.2196, 0.2353), (0.9176, 0.2314, 0.2235),
        (0.9216, 0.2431, 0.2157), (0.9255, 0.2549, 0.2078),
        (0.9294, 0.2667, 0.2), (0.9333, 0.2784, 0.1922),
        (0.9333, 0.2902, 0.1804), (0.9373, 0.302, 0.1725),
        (0.9412, 0.3137, 0.1647), (0.9412, 0.3255, 0.1569),
        (0.9412, 0.3373, 0.149), (0.9451, 0.349, 0.1412),
        (0.9451, 0.3647, 0.1333), (0.9451, 0.3765, 0.1255),
        (0.9451, 0.3882, 0.1176), (0.9451, 0.4039, 0.1098),
        (0.9451, 0.4157, 0.1059), (0.9451, 0.4314, 0.098),
        (0.9412, 0.4431, 0.0902), (0.9412, 0.4588, 0.0824)],
    'ncl:helix1': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0039, 0.0039, 0.0039), (0.0078, 0.0039, 0.0039),
        (0.0118, 0.0078, 0.0078), (0.0157, 0.0078, 0.0078),
        (0.0196, 0.0118, 0.0118), (0.0235, 0.0157, 0.0157),
        (0.0275, 0.0157, 0.0157), (0.0275, 0.0196, 0.0196),
        (0.0314, 0.0235, 0.0235), (0.0353, 0.0235, 0.0235),
        (0.0392, 0.0275, 0.0275), (0.0392, 0.0314, 0.0314),
        (0.0431, 0.0353, 0.0353), (0.0471, 0.0392, 0.0392),
        (0.0471, 0.0431, 0.0431), (0.051, 0.0431, 0.0431),
        (0.0549, 0.0471, 0.0471), (0.0549, 0.051, 0.051),
        (0.0588, 0.0549, 0.0549), (0.0588, 0.0588, 0.0588),
        (0.0627, 0.0627, 0.0627), (0.0627, 0.0667, 0.0667),
        (0.0667, 0.0706, 0.0706), (0.0667, 0.0745, 0.0745),
        (0.0706, 0.0784, 0.0784), (0.0706, 0.0824, 0.0824),
        (0.0706, 0.0863, 0.0863), (0.0745, 0.0902, 0.0902),
        (0.0745, 0.098, 0.098), (0.0745, 0.102, 0.102),
        (0.0784, 0.1059, 0.1059), (0.0784, 0.1098, 0.1098),
        (0.0784, 0.1137, 0.1137), (0.0784, 0.1176, 0.1176),
        (0.0784, 0.1216, 0.1216), (0.0824, 0.1294, 0.1294),
        (0.0824, 0.1333, 0.1333), (0.0824, 0.1373, 0.1373),
        (0.0824, 0.1412, 0.1412), (0.0824, 0.1451, 0.1451),
        (0.0824, 0.1529, 0.1529), (0.0824, 0.1569, 0.1569),
        (0.0824, 0.1608, 0.1608), (0.0824, 0.1647, 0.1647),
        (0.0824, 0.1686, 0.1686), (0.0824, 0.1725, 0.1725),
        (0.0784, 0.1804, 0.1804), (0.0784, 0.1843, 0.1843),
        (0.0784, 0.1882, 0.1882), (0.0784, 0.1922, 0.1922),
        (0.0784, 0.1961, 0.1961), (0.0745, 0.2039, 0.2039),
        (0.0745, 0.2078, 0.2078), (0.0745, 0.2118, 0.2118),
        (0.0706, 0.2157, 0.2157), (0.0706, 0.2196, 0.2196),
        (0.0706, 0.2235, 0.2235), (0.0667, 0.2275, 0.2275),
        (0.0667, 0.2314, 0.2314), (0.0667, 0.2353, 0.2353),
        (0.0627, 0.2392, 0.2392), (0.0627, 0.2431, 0.2431),
        (0.0627, 0.2471, 0.2471), (0.0588, 0.251, 0.251),
        (0.0588, 0.2549, 0.2549), (0.0549, 0.2588, 0.2588),
        (0.0549, 0.2627, 0.2627), (0.051, 0.2667, 0.2667),
        (0.051, 0.2706, 0.2706), (0.0471, 0.2745, 0.2745),
        (0.0471, 0.2784, 0.2784), (0.0431, 0.2824, 0.2824),
        (0.0431, 0.2824, 0.2824), (0.0392, 0.2863, 0.2863),
        (0.0392, 0.2902, 0.2902), (0.0353, 0.2941, 0.2941),
        (0.0353, 0.2941, 0.2941), (0.0314, 0.298, 0.298),
        (0.0314, 0.298, 0.298), (0.0275, 0.302, 0.302),
        (0.0275, 0.3059, 0.3059), (0.0235, 0.3059, 0.3059),
        (0.0235, 0.3098, 0.3098), (0.0196, 0.3098, 0.3098),
        (0.0196, 0.3098, 0.3098), (0.0157, 0.3137, 0.3137),
        (0.0157, 0.3137, 0.3137), (0.0157, 0.3137, 0.3137),
        (0.0118, 0.3176, 0.3176), (0.0118, 0.3176, 0.3176),
        (0.0078, 0.3176, 0.3176), (0.0078, 0.3176, 0.3176),
        (0.0078, 0.3176, 0.3176), (0.0039, 0.3176, 0.3176),
        (0.0039, 0.3176, 0.3176), (0.0039, 0.3176, 0.3176),
        (0.0039, 0.3176, 0.3176), (0., 0.3176, 0.3176),
        (0., 0.3176, 0.3176), (0., 0.3176, 0.3176),
        (0., 0.3176, 0.3176), (0., 0.3176, 0.3176),
        (0., 0.3137, 0.3137), (0., 0.3137, 0.3137),
        (0., 0.3137, 0.3137), (0., 0.3098, 0.3098),
        (0., 0.3098, 0.3098), (0., 0.3098, 0.3098),
        (0., 0.3059, 0.3059), (0., 0.302, 0.302),
        (0., 0.302, 0.302), (0., 0.298, 0.298),
        (0., 0.298, 0.298), (0.0039, 0.2941, 0.2941),
        (0.0039, 0.2902, 0.2902), (0.0039, 0.2902, 0.2902),
        (0.0078, 0.2863, 0.2863), (0.0078, 0.2824, 0.2824),
        (0.0118, 0.2784, 0.2784), (0.0118, 0.2745, 0.2745),
        (0.0157, 0.2706, 0.2706), (0.0157, 0.2706, 0.2706),
        (0.0196, 0.2667, 0.2667), (0.0235, 0.2627, 0.2627),
        (0.0235, 0.2588, 0.2588), (0.0275, 0.2549, 0.2549),
        (0.0314, 0.251, 0.251), (0.0353, 0.2431, 0.2431),
        (0.0392, 0.2392, 0.2392), (0.0431, 0.2353, 0.2353),
        (0.0471, 0.2314, 0.2314), (0.051, 0.2275, 0.2275),
        (0.0549, 0.2235, 0.2235), (0.0588, 0.2157, 0.2157),
        (0.0627, 0.2118, 0.2118), (0.0706, 0.2078, 0.2078),
        (0.0745, 0.2039, 0.2039), (0.0784, 0.1961, 0.1961),
        (0.0863, 0.1922, 0.1922), (0.0902, 0.1882, 0.1882),
        (0.098, 0.1843, 0.1843), (0.102, 0.1765, 0.1765),
        (0.1098, 0.1725, 0.1725), (0.1137, 0.1686, 0.1686),
        (0.1216, 0.1608, 0.1608), (0.1294, 0.1569, 0.1569),
        (0.1333, 0.1529, 0.1529), (0.1412, 0.1451, 0.1451),
        (0.149, 0.1412, 0.1412), (0.1569, 0.1373, 0.1373),
        (0.1647, 0.1294, 0.1294), (0.1725, 0.1255, 0.1255),
        (0.1804, 0.1216, 0.1216), (0.1882, 0.1137, 0.1137),
        (0.1961, 0.1098, 0.1098), (0.2039, 0.1059, 0.1059),
        (0.2118, 0.098, 0.098), (0.2235, 0.0941, 0.0941),
        (0.2314, 0.0902, 0.0902), (0.2392, 0.0863, 0.0863),
        (0.2471, 0.0784, 0.0784), (0.2588, 0.0745, 0.0745),
        (0.2667, 0.0706, 0.0706), (0.2784, 0.0667, 0.0667),
        (0.2863, 0.0627, 0.0627), (0.2941, 0.0588, 0.0588),
        (0.3059, 0.0549, 0.0549), (0.3176, 0.051, 0.051),
        (0.3255, 0.0471, 0.0471), (0.3373, 0.0431, 0.0431),
        (0.3451, 0.0392, 0.0392), (0.3569, 0.0353, 0.0353),
        (0.3686, 0.0314, 0.0314), (0.3765, 0.0275, 0.0275),
        (0.3882, 0.0235, 0.0235), (0.4, 0.0196, 0.0196),
        (0.4078, 0.0196, 0.0196), (0.4196, 0.0157, 0.0157),
        (0.4314, 0.0118, 0.0118), (0.4431, 0.0118, 0.0118),
        (0.451, 0.0078, 0.0078), (0.4627, 0.0078, 0.0078),
        (0.4745, 0.0039, 0.0039), (0.4863, 0.0039, 0.0039),
        (0.498, 0.0039, 0.0039), (0.5059, 0., 0.),
        (0.5176, 0., 0.), (0.5294, 0., 0.),
        (0.5412, 0., 0.), (0.5529, 0., 0.),
        (0.5608, 0., 0.), (0.5725, 0., 0.),
        (0.5843, 0., 0.), (0.5961, 0., 0.),
        (0.6078, 0., 0.), (0.6157, 0., 0.),
        (0.6275, 0.0039, 0.0039), (0.6392, 0.0039, 0.0039),
        (0.6471, 0.0039, 0.0039), (0.6588, 0.0078, 0.0078),
        (0.6706, 0.0118, 0.0118), (0.6784, 0.0118, 0.0118),
        (0.6902, 0.0157, 0.0157), (0.702, 0.0196, 0.0196),
        (0.7098, 0.0235, 0.0235), (0.7216, 0.0235, 0.0235),
        (0.7294, 0.0275, 0.0275), (0.7373, 0.0314, 0.0314),
        (0.749, 0.0392, 0.0392), (0.7569, 0.0431, 0.0431),
        (0.7686, 0.0471, 0.0471), (0.7765, 0.051, 0.051),
        (0.7843, 0.0588, 0.0588), (0.7922, 0.0627, 0.0627),
        (0.8, 0.0706, 0.0706), (0.8078, 0.0745, 0.0745),
        (0.8196, 0.0824, 0.0824), (0.8275, 0.0902, 0.0902),
        (0.8314, 0.0941, 0.0941), (0.8392, 0.102, 0.102),
        (0.8471, 0.1098, 0.1098), (0.8549, 0.1176, 0.1176),
        (0.8627, 0.1255, 0.1255), (0.8667, 0.1333, 0.1333),
        (0.8745, 0.1451, 0.1451), (0.8784, 0.1529, 0.1529),
        (0.8863, 0.1608, 0.1608), (0.8902, 0.1725, 0.1725),
        (0.898, 0.1804, 0.1804), (0.902, 0.1882, 0.1882),
        (0.9059, 0.2, 0.2), (0.9098, 0.2118, 0.2118),
        (0.9137, 0.2196, 0.2196), (0.9176, 0.2314, 0.2314),
        (0.9216, 0.2431, 0.2431), (0.9255, 0.2549, 0.2549),
        (0.9294, 0.2667, 0.2667), (0.9333, 0.2784, 0.2784),
        (0.9333, 0.2902, 0.2902), (0.9373, 0.302, 0.302),
        (0.9412, 0.3137, 0.3137), (0.9412, 0.3255, 0.3255),
        (0.9412, 0.3373, 0.3373), (0.9451, 0.349, 0.349),
        (0.9451, 0.3647, 0.3647), (0.9451, 0.3765, 0.3765),
        (0.9451, 0.3882, 0.3882), (0.9451, 0.4039, 0.4039),
        (0.9451, 0.4157, 0.4157), (0.9451, 0.4314, 0.4314),
        (0.9412, 0.4431, 0.4431), (0.9412, 0.4588, 0.4588)],
    'ncl:hotres': [
        (1., 1., 1.), (0., 0., 0.),
        (0.0039, 0.0196, 0.2745), (0.0314, 0.098, 0.3922),
        (0.0588, 0.2078, 0.5098), (0.0902, 0.2667, 0.6275),
        (0.1216, 0.3255, 0.7451), (0.1529, 0.3843, 0.8627),
        (0.1843, 0.4392, 0.9804), (0.2157, 0.498, 0.9922),
        (0.2431, 0.549, 0.9804), (0.2706, 0.6, 0.9569),
        (0.2941, 0.6431, 0.9216), (0.3216, 0.6824, 0.8784),
        (0.3451, 0.7216, 0.8314), (0.3725, 0.7569, 0.7804),
        (0.3961, 0.7882, 0.7294), (0.4196, 0.8157, 0.6784),
        (0.4392, 0.8431, 0.6275), (0.4627, 0.8667, 0.5765),
        (0.4824, 0.8863, 0.5294), (0.502, 0.9059, 0.4824),
        (0.5255, 0.9216, 0.4392), (0.5451, 0.9373, 0.4),
        (0.5608, 0.949, 0.3647), (0.5804, 0.9608, 0.3294),
        (0.6, 0.9686, 0.298), (0.6157, 0.9765, 0.2706),
        (0.6314, 0.9804, 0.2431), (0.6471, 0.9843, 0.2196),
        (0.6627, 0.9882, 0.1961), (0.6784, 0.9922, 0.1765),
        (0.6941, 0.9922, 0.1569), (0.7098, 0.9922, 0.1412),
        (0.7216, 0.9882, 0.1255), (0.7373, 0.9882, 0.1137),
        (0.749, 0.9843, 0.098), (0.7608, 0.9804, 0.0902),
        (0.7725, 0.9725, 0.0784), (0.7843, 0.9686, 0.0706),
        (0.7961, 0.9608, 0.0627), (0.8078, 0.9529, 0.0549),
        (0.8157, 0.9451, 0.0471), (0.8275, 0.9373, 0.0431),
        (0.8353, 0.9294, 0.0392), (0.8471, 0.9216, 0.0353),
        (0.8549, 0.9098, 0.0314), (0.8627, 0.902, 0.0275),
        (0.8706, 0.8902, 0.0235), (0.8784, 0.8824, 0.0196),
        (0.8863, 0.8706, 0.0157), (0.8941, 0.8588, 0.0157),
        (0.902, 0.8471, 0.0118), (0.9098, 0.8353, 0.0118),
        (0.9137, 0.8235, 0.0118), (0.9216, 0.8118, 0.0078),
        (0.9255, 0.8, 0.0078), (0.9333, 0.7882, 0.0078),
        (0.9373, 0.7765, 0.0039), (0.9412, 0.7647, 0.0039),
        (0.9451, 0.7529, 0.0039), (0.949, 0.7412, 0.0039),
        (0.9529, 0.7294, 0.0039), (0.9569, 0.7176, 0.),
        (0.9608, 0.7059, 0.), (0.9647, 0.6941, 0.),
        (0.9686, 0.6784, 0.), (0.9725, 0.6667, 0.),
        (0.9725, 0.6549, 0.), (0.9765, 0.6431, 0.),
        (0.9804, 0.6314, 0.), (0.9804, 0.6196, 0.),
        (0.9843, 0.6078, 0.), (0.9843, 0.5961, 0.),
        (0.9843, 0.5843, 0.), (0.9882, 0.5765, 0.),
        (0.9882, 0.5647, 0.), (0.9882, 0.5529, 0.),
        (0.9882, 0.5412, 0.), (0.9922, 0.5294, 0.),
        (0.9922, 0.5176, 0.), (0.9922, 0.5098, 0.),
        (0.9922, 0.498, 0.), (0.9922, 0.4863, 0.),
        (0.9922, 0.4784, 0.), (0.9922, 0.4667, 0.),
        (0.9882, 0.4588, 0.), (0.9882, 0.4471, 0.),
        (0.9882, 0.4392, 0.), (0.9882, 0.4275, 0.),
        (0.9882, 0.4196, 0.), (0.9843, 0.4118, 0.),
        (0.9843, 0.4, 0.), (0.9843, 0.3922, 0.),
        (0.9804, 0.3843, 0.), (0.9804, 0.3765, 0.),
        (0.9765, 0.3647, 0.), (0.9765, 0.3569, 0.),
        (0.9725, 0.349, 0.), (0.9725, 0.3412, 0.),
        (0.9686, 0.3333, 0.), (0.9686, 0.3255, 0.),
        (0.9647, 0.3176, 0.), (0.9608, 0.3098, 0.),
        (0.9608, 0.3059, 0.), (0.9569, 0.298, 0.),
        (0.9529, 0.2902, 0.), (0.9529, 0.2824, 0.),
        (0.949, 0.2784, 0.), (0.9451, 0.2706, 0.),
        (0.9412, 0.2627, 0.), (0.9412, 0.2588, 0.),
        (0.9373, 0.251, 0.), (0.9333, 0.2431, 0.),
        (0.9294, 0.2392, 0.), (0.9255, 0.2314, 0.),
        (0.9216, 0.2275, 0.), (0.9216, 0.2235, 0.),
        (0.9176, 0.2157, 0.), (0.9137, 0.2118, 0.),
        (0.9098, 0.2078, 0.), (0.9059, 0.2, 0.),
        (0.902, 0.1961, 0.), (0.898, 0.1922, 0.),
        (0.8941, 0.1843, 0.), (0.8902, 0.1804, 0.),
        (0.8863, 0.1765, 0.), (0.8824, 0.1725, 0.),
        (0.8784, 0.1686, 0.), (0.8745, 0.1647, 0.),
        (0.8706, 0.1608, 0.), (0.8667, 0.1569, 0.),
        (0.8627, 0.1529, 0.), (0.8588, 0.149, 0.),
        (0.8549, 0.1451, 0.), (0.851, 0.1412, 0.),
        (0.8471, 0.1373, 0.), (0.8392, 0.1333, 0.),
        (0.8353, 0.1294, 0.), (0.8314, 0.1255, 0.),
        (0.8275, 0.1216, 0.), (0.8235, 0.1216, 0.),
        (0.8196, 0.1176, 0.), (0.8157, 0.1137, 0.),
        (0.8118, 0.1098, 0.), (0.8039, 0.1059, 0.),
        (0.8, 0.1059, 0.), (0.7961, 0.102, 0.),
        (0.7922, 0.098, 0.), (0.7882, 0.098, 0.),
        (0.7843, 0.0941, 0.), (0.7804, 0.0902, 0.),
        (0.7725, 0.0902, 0.), (0.7686, 0.0863, 0.),
        (0.7647, 0.0824, 0.), (0.7608, 0.0824, 0.),
        (0.7569, 0.0784, 0.), (0.7529, 0.0784, 0.),
        (0.749, 0.0745, 0.), (0.7412, 0.0745, 0.),
        (0.7373, 0.0706, 0.), (0.7333, 0.0706, 0.),
        (0.7294, 0.0667, 0.), (0.7255, 0.0667, 0.),
        (0.7216, 0.0627, 0.), (0.7137, 0.0627, 0.),
        (0.7098, 0.0588, 0.), (0.7059, 0.0588, 0.),
        (0.702, 0.0588, 0.), (0.698, 0.0549, 0.),
        (0.6941, 0.0549, 0.), (0.6863, 0.051, 0.),
        (0.6824, 0.051, 0.), (0.6784, 0.051, 0.),
        (0.6745, 0.0471, 0.), (0.6706, 0.0471, 0.),
        (0.6667, 0.0471, 0.), (0.6588, 0.0431, 0.),
        (0.6549, 0.0431, 0.), (0.651, 0.0431, 0.),
        (0.6471, 0.0392, 0.), (0.6431, 0.0392, 0.),
        (0.6392, 0.0392, 0.), (0.6353, 0.0353, 0.),
        (0.6314, 0.0353, 0.), (0.6235, 0.0353, 0.),
        (0.6196, 0.0353, 0.), (0.6157, 0.0314, 0.),
        (0.6118, 0.0314, 0.), (0.6078, 0.0314, 0.),
        (0.6039, 0.0314, 0.), (0.6, 0.0275, 0.),
        (0.5961, 0.0275, 0.), (0.5882, 0.0275, 0.),
        (0.5843, 0.0275, 0.), (0.5804, 0.0275, 0.),
        (0.5765, 0.0235, 0.), (0.5725, 0.0235, 0.),
        (0.5686, 0.0235, 0.), (0.5647, 0.0235, 0.),
        (0.5608, 0.0235, 0.), (0.5569, 0.0235, 0.),
        (0.5529, 0.0196, 0.), (0.549, 0.0196, 0.),
        (0.5451, 0.0196, 0.), (0.5412, 0.0196, 0.),
        (0.5373, 0.0196, 0.), (0.5333, 0.0196, 0.),
        (0.5294, 0.0157, 0.), (0.5216, 0.0157, 0.),
        (0.5176, 0.0157, 0.), (0.5137, 0.0157, 0.),
        (0.5098, 0.0157, 0.), (0.5059, 0.0157, 0.),
        (0.502, 0.0157, 0.), (0.498, 0.0157, 0.),
        (0.4941, 0.0118, 0.), (0.4902, 0.0118, 0.),
        (0.4863, 0.0118, 0.), (0.4824, 0.0118, 0.),
        (0.4784, 0.0118, 0.), (0.4784, 0.0118, 0.),
        (0.4745, 0.0118, 0.), (0.4706, 0.0118, 0.),
        (0.4667, 0.0118, 0.), (0.4627, 0.0118, 0.),
        (0.4588, 0.0078, 0.), (0.4549, 0.0078, 0.),
        (0.451, 0.0078, 0.), (0.4471, 0.0078, 0.),
        (0.4431, 0.0078, 0.), (0.4392, 0.0078, 0.),
        (0.4353, 0.0078, 0.), (0.4314, 0.0078, 0.),
        (0.4275, 0.0078, 0.), (0.4235, 0.0078, 0.),
        (0.4235, 0.0078, 0.), (0.4196, 0.0078, 0.),
        (0.4157, 0.0078, 0.), (0.4118, 0.0078, 0.),
        (0.4078, 0.0039, 0.), (0.4039, 0.0039, 0.),
        (0.4, 0.0039, 0.), (0.3961, 0.0039, 0.),
        (0.3961, 0.0039, 0.), (0.3922, 0.0039, 0.),
        (0.3882, 0.0039, 0.), (0.3843, 0.0039, 0.),
        (0.3804, 0.0039, 0.), (0.3765, 0.0039, 0.),
        (0.3765, 0.0039, 0.), (0.3725, 0.0039, 0.),
        (0.3686, 0.0039, 0.), (0.3647, 0.0039, 0.),
        (0.3608, 0.0039, 0.), (0.3608, 0.0039, 0.)],
    'ncl:ncview_default': [
        (1., 1., 1.), (0., 0., 0.),
        (0.5137, 0., 1.), (0.502, 0., 1.),
        (0.4902, 0., 0.9961), (0.4784, 0., 0.9961),
        (0.4667, 0., 0.9961), (0.4549, 0., 0.9922),
        (0.4431, 0.0039, 0.9922), (0.4314, 0.0078, 0.9882),
        (0.4235, 0.0118, 0.9882), (0.4157, 0.0118, 0.9882),
        (0.4039, 0.0157, 0.9843), (0.3961, 0.0196, 0.9843),
        (0.3882, 0.0235, 0.9804), (0.3765, 0.0275, 0.9804),
        (0.3686, 0.0314, 0.9765), (0.3608, 0.0353, 0.9765),
        (0.3529, 0.0392, 0.9725), (0.3412, 0.0471, 0.9686),
        (0.3333, 0.051, 0.9686), (0.3255, 0.0549, 0.9647),
        (0.3176, 0.0627, 0.9608), (0.3098, 0.0706, 0.9569),
        (0.298, 0.0745, 0.9529), (0.2902, 0.0824, 0.949),
        (0.2824, 0.0902, 0.9451), (0.2745, 0.0941, 0.9412),
        (0.2667, 0.102, 0.9373), (0.2588, 0.1098, 0.9333),
        (0.251, 0.1176, 0.9294), (0.2431, 0.1255, 0.9255),
        (0.2353, 0.1373, 0.9216), (0.2275, 0.1451, 0.9137),
        (0.2196, 0.1529, 0.9098), (0.2118, 0.1608, 0.9059),
        (0.2039, 0.1725, 0.898), (0.1961, 0.1804, 0.8941),
        (0.1882, 0.1882, 0.8902), (0.1843, 0.2, 0.8824),
        (0.1765, 0.2078, 0.8784), (0.1686, 0.2196, 0.8706),
        (0.1608, 0.2314, 0.8667), (0.1529, 0.2392, 0.8588),
        (0.149, 0.251, 0.851), (0.1412, 0.2627, 0.8471),
        (0.1373, 0.2706, 0.8392), (0.1294, 0.2824, 0.8314),
        (0.1216, 0.2941, 0.8275), (0.1176, 0.3059, 0.8196),
        (0.1098, 0.3176, 0.8118), (0.1059, 0.3294, 0.8039),
        (0.098, 0.3412, 0.7961), (0.0941, 0.3529, 0.7922),
        (0.0902, 0.3647, 0.7843), (0.0824, 0.3765, 0.7765),
        (0.0784, 0.3882, 0.7686), (0.0745, 0.4, 0.7608),
        (0.0706, 0.4118, 0.7529), (0.0627, 0.4235, 0.7451),
        (0.0588, 0.4353, 0.7373), (0.0549, 0.4471, 0.7294),
        (0.051, 0.4588, 0.7216), (0.0471, 0.4706, 0.7137),
        (0.0431, 0.4824, 0.702), (0.0392, 0.4941, 0.6941),
        (0.0353, 0.5098, 0.6863), (0.0314, 0.5216, 0.6784),
        (0.0275, 0.5333, 0.6706), (0.0275, 0.5451, 0.6627),
        (0.0235, 0.5569, 0.651), (0.0196, 0.5686, 0.6431),
        (0.0196, 0.5804, 0.6353), (0.0157, 0.5922, 0.6275),
        (0.0118, 0.6039, 0.6157), (0.0118, 0.6157, 0.6078),
        (0.0078, 0.6275, 0.6), (0.0078, 0.6392, 0.5922),
        (0.0039, 0.651, 0.5804), (0.0039, 0.6627, 0.5725),
        (0.0039, 0.6745, 0.5647), (0., 0.6863, 0.5529),
        (0., 0.698, 0.5451), (0., 0.7098, 0.5373),
        (0., 0.7216, 0.5255), (0., 0.7294, 0.5176),
        (0., 0.7412, 0.5098), (0., 0.7529, 0.498),
        (0., 0.7608, 0.4902), (0., 0.7725, 0.4824),
        (0., 0.7843, 0.4706), (0., 0.7922, 0.4627),
        (0., 0.8039, 0.4549), (0., 0.8118, 0.4431),
        (0., 0.8196, 0.4353), (0.0039, 0.8314, 0.4275),
        (0.0039, 0.8392, 0.4157), (0.0078, 0.8471, 0.4078),
        (0.0078, 0.8549, 0.4), (0.0078, 0.8667, 0.3882),
        (0.0118, 0.8745, 0.3804), (0.0118, 0.8824, 0.3725),
        (0.0157, 0.8902, 0.3647), (0.0196, 0.898, 0.3529),
        (0.0196, 0.902, 0.3451), (0.0235, 0.9098, 0.3373),
        (0.0275, 0.9176, 0.3294), (0.0314, 0.9216, 0.3216),
        (0.0353, 0.9294, 0.3098), (0.0353, 0.9373, 0.302),
        (0.0392, 0.9412, 0.2941), (0.0431, 0.9451, 0.2863),
        (0.0471, 0.9529, 0.2784), (0.051, 0.9569, 0.2706),
        (0.0549, 0.9608, 0.2627), (0.0627, 0.9647, 0.2549),
        (0.0667, 0.9686, 0.2471), (0.0706, 0.9725, 0.2392),
        (0.0745, 0.9765, 0.2314), (0.0784, 0.9804, 0.2235),
        (0.0863, 0.9804, 0.2157), (0.0902, 0.9843, 0.2078),
        (0.0941, 0.9882, 0.2), (0.102, 0.9882, 0.1922),
        (0.1059, 0.9922, 0.1843), (0.1137, 0.9922, 0.1765),
        (0.1176, 0.9922, 0.1725), (0.1255, 0.9922, 0.1647),
        (0.1294, 0.9922, 0.1569), (0.1373, 0.9961, 0.149),
        (0.1451, 0.9922, 0.1451), (0.149, 0.9922, 0.1373),
        (0.1569, 0.9922, 0.1294), (0.1647, 0.9922, 0.1255),
        (0.1725, 0.9922, 0.1176), (0.1765, 0.9882, 0.1137),
        (0.1843, 0.9882, 0.1059), (0.1922, 0.9843, 0.102),
        (0.2, 0.9804, 0.0941), (0.2078, 0.9804, 0.0902),
        (0.2157, 0.9765, 0.0863), (0.2235, 0.9725, 0.0784),
        (0.2314, 0.9686, 0.0745), (0.2392, 0.9647, 0.0706),
        (0.2471, 0.9608, 0.0667), (0.2549, 0.9569, 0.0627),
        (0.2627, 0.9529, 0.0549), (0.2706, 0.9451, 0.051),
        (0.2784, 0.9412, 0.0471), (0.2863, 0.9373, 0.0431),
        (0.2941, 0.9294, 0.0392), (0.302, 0.9216, 0.0353),
        (0.3098, 0.9176, 0.0353), (0.3216, 0.9098, 0.0314),
        (0.3294, 0.902, 0.0275), (0.3373, 0.898, 0.0235),
        (0.3451, 0.8902, 0.0196), (0.3529, 0.8824, 0.0196),
        (0.3647, 0.8745, 0.0157), (0.3725, 0.8667, 0.0118),
        (0.3804, 0.8549, 0.0118), (0.3882, 0.8471, 0.0078),
        (0.4, 0.8392, 0.0078), (0.4078, 0.8314, 0.0078),
        (0.4157, 0.8196, 0.0039), (0.4275, 0.8118, 0.0039),
        (0.4353, 0.8039, 0.), (0.4431, 0.7922, 0.),
        (0.4549, 0.7843, 0.), (0.4627, 0.7725, 0.),
        (0.4706, 0.7608, 0.), (0.4824, 0.7529, 0.),
        (0.4902, 0.7412, 0.), (0.498, 0.7294, 0.),
        (0.5098, 0.7216, 0.), (0.5176, 0.7098, 0.),
        (0.5255, 0.698, 0.), (0.5373, 0.6863, 0.),
        (0.5451, 0.6745, 0.), (0.5529, 0.6627, 0.),
        (0.5647, 0.651, 0.0039), (0.5725, 0.6392, 0.0039),
        (0.5804, 0.6275, 0.0039), (0.5922, 0.6157, 0.0078),
        (0.6, 0.6039, 0.0078), (0.6078, 0.5922, 0.0118),
        (0.6157, 0.5804, 0.0118), (0.6275, 0.5686, 0.0157),
        (0.6353, 0.5569, 0.0196), (0.6431, 0.5451, 0.0196),
        (0.651, 0.5333, 0.0235), (0.6627, 0.5216, 0.0275),
        (0.6706, 0.5098, 0.0275), (0.6784, 0.4941, 0.0314),
        (0.6863, 0.4824, 0.0353), (0.6941, 0.4706, 0.0392),
        (0.702, 0.4588, 0.0431), (0.7137, 0.4471, 0.0471),
        (0.7216, 0.4353, 0.051), (0.7294, 0.4235, 0.0549),
        (0.7373, 0.4118, 0.0588), (0.7451, 0.4, 0.0627),
        (0.7529, 0.3882, 0.0706), (0.7608, 0.3765, 0.0745),
        (0.7686, 0.3647, 0.0784), (0.7765, 0.3529, 0.0824),
        (0.7843, 0.3412, 0.0902), (0.7922, 0.3294, 0.0941),
        (0.7961, 0.3176, 0.098), (0.8039, 0.3059, 0.1059),
        (0.8118, 0.2941, 0.1098), (0.8196, 0.2824, 0.1176),
        (0.8275, 0.2706, 0.1216), (0.8314, 0.2627, 0.1294),
        (0.8392, 0.251, 0.1373), (0.8471, 0.2392, 0.1412),
        (0.851, 0.2314, 0.149), (0.8588, 0.2196, 0.1529),
        (0.8667, 0.2078, 0.1608), (0.8706, 0.2, 0.1686),
        (0.8784, 0.1882, 0.1765), (0.8824, 0.1804, 0.1843),
        (0.8902, 0.1725, 0.1882), (0.8941, 0.1608, 0.1961),
        (0.898, 0.1529, 0.2039), (0.9059, 0.1451, 0.2118),
        (0.9098, 0.1373, 0.2196), (0.9137, 0.1255, 0.2275),
        (0.9216, 0.1176, 0.2353), (0.9255, 0.1098, 0.2431),
        (0.9294, 0.102, 0.251), (0.9333, 0.0941, 0.2588),
        (0.9373, 0.0902, 0.2667), (0.9412, 0.0824, 0.2745),
        (0.9451, 0.0745, 0.2824), (0.949, 0.0706, 0.2902),
        (0.9529, 0.0627, 0.298), (0.9569, 0.0549, 0.3098),
        (0.9608, 0.051, 0.3176), (0.9647, 0.0471, 0.3255),
        (0.9686, 0.0392, 0.3333), (0.9686, 0.0353, 0.3412),
        (0.9725, 0.0314, 0.3529), (0.9765, 0.0275, 0.3608),
        (0.9765, 0.0235, 0.3686), (0.9804, 0.0196, 0.3765),
        (0.9804, 0.0157, 0.3882), (0.9843, 0.0118, 0.3961),
        (0.9843, 0.0118, 0.4039), (0.9882, 0.0078, 0.4157),
        (0.9882, 0.0039, 0.4235), (0.9882, 0.0039, 0.4314),
        (0.9922, 0., 0.4431), (0.9922, 0., 0.451),
        (0.9922, 0., 0.4588), (0.9922, 0., 0.4667)],
    'ncl:oceanlakelandsnow': [
        (1., 1., 1.), (0., 0., 0.),
        (0.0941, 0.4549, 0.8039), (0.5294, 0.8078, 1.),
        (0.149, 0.4039, 0.1569), (0.1647, 0.4196, 0.1569),
        (0.1804, 0.4314, 0.1608), (0.1961, 0.4471, 0.1608),
        (0.2118, 0.4588, 0.1608), (0.2275, 0.4745, 0.1647),
        (0.2431, 0.4863, 0.1647), (0.2588, 0.502, 0.1647),
        (0.2745, 0.5137, 0.1686), (0.2902, 0.5294, 0.1686),
        (0.3059, 0.5412, 0.1686), (0.3216, 0.5569, 0.1725),
        (0.3373, 0.5686, 0.1725), (0.3529, 0.5843, 0.1725),
        (0.3686, 0.5961, 0.1765), (0.3843, 0.6118, 0.1765),
        (0.4, 0.6235, 0.1804), (0.4157, 0.6392, 0.1804),
        (0.4314, 0.651, 0.1804), (0.4471, 0.6667, 0.1843),
        (0.4627, 0.6784, 0.1843), (0.4784, 0.6941, 0.1843),
        (0.4941, 0.7059, 0.1882), (0.5098, 0.7216, 0.1882),
        (0.5255, 0.7333, 0.1882), (0.5412, 0.749, 0.1922),
        (0.5569, 0.7608, 0.1922), (0.5725, 0.7765, 0.1922),
        (0.6039, 0.8039, 0.1961), (0.6078, 0.8078, 0.2039),
        (0.6118, 0.8118, 0.2118), (0.6196, 0.8157, 0.2196),
        (0.6235, 0.8235, 0.2235), (0.6275, 0.8275, 0.2314),
        (0.6314, 0.8314, 0.2392), (0.6353, 0.8353, 0.2471),
        (0.6392, 0.8392, 0.2549), (0.6471, 0.8431, 0.2627),
        (0.651, 0.8471, 0.2706), (0.6549, 0.8549, 0.2745),
        (0.6588, 0.8588, 0.2824), (0.6627, 0.8627, 0.2902),
        (0.6667, 0.8667, 0.298), (0.6745, 0.8706, 0.3059),
        (0.6784, 0.8745, 0.3137), (0.6824, 0.8784, 0.3216),
        (0.6863, 0.8824, 0.3294), (0.6902, 0.8902, 0.3333),
        (0.6941, 0.8941, 0.3412), (0.702, 0.898, 0.349),
        (0.7059, 0.902, 0.3569), (0.7098, 0.9059, 0.3647),
        (0.7137, 0.9098, 0.3725), (0.7176, 0.9137, 0.3804),
        (0.7216, 0.9216, 0.3843), (0.7294, 0.9255, 0.3922),
        (0.7373, 0.9333, 0.4078), (0.7451, 0.9333, 0.4196),
        (0.7529, 0.9333, 0.4353), (0.7608, 0.9373, 0.4471),
        (0.7725, 0.9373, 0.4588), (0.7804, 0.9373, 0.4745),
        (0.7882, 0.9373, 0.4863), (0.7961, 0.9412, 0.498),
        (0.8039, 0.9412, 0.5098), (0.8118, 0.9412, 0.5255),
        (0.8196, 0.9412, 0.5373), (0.8314, 0.9451, 0.549),
        (0.8392, 0.9451, 0.5647), (0.8471, 0.9451, 0.5765),
        (0.8549, 0.9451, 0.5882), (0.8627, 0.949, 0.6039),
        (0.8706, 0.949, 0.6157), (0.8784, 0.949, 0.6275),
        (0.8863, 0.949, 0.6431), (0.898, 0.9529, 0.6549),
        (0.9059, 0.9529, 0.6667), (0.9137, 0.9529, 0.6824),
        (0.9216, 0.9529, 0.6941), (0.9294, 0.9569, 0.7059),
        (0.9373, 0.9569, 0.7176), (0.9451, 0.9569, 0.7333),
        (0.9569, 0.9569, 0.7451), (0.9647, 0.9608, 0.7569),
        (0.9804, 0.9608, 0.7843), (0.9804, 0.9569, 0.7804),
        (0.9765, 0.9569, 0.7725), (0.9765, 0.9529, 0.7686),
        (0.9765, 0.9529, 0.7608), (0.9725, 0.949, 0.7569),
        (0.9725, 0.949, 0.749), (0.9725, 0.9451, 0.7451),
        (0.9686, 0.9451, 0.7373), (0.9686, 0.9412, 0.7333),
        (0.9686, 0.9373, 0.7294), (0.9647, 0.9373, 0.7216),
        (0.9647, 0.9333, 0.7176), (0.9647, 0.9333, 0.7098),
        (0.9608, 0.9294, 0.7059), (0.9608, 0.9294, 0.698),
        (0.9569, 0.9255, 0.6941), (0.9569, 0.9255, 0.6863),
        (0.9569, 0.9216, 0.6824), (0.9529, 0.9216, 0.6745),
        (0.9529, 0.9176, 0.6706), (0.9529, 0.9137, 0.6667),
        (0.949, 0.9137, 0.6588), (0.949, 0.9098, 0.6549),
        (0.949, 0.9098, 0.6471), (0.9451, 0.9059, 0.6431),
        (0.9451, 0.9059, 0.6353), (0.9451, 0.902, 0.6314),
        (0.9412, 0.898, 0.6196), (0.9373, 0.8941, 0.6078),
        (0.9333, 0.8863, 0.6), (0.9294, 0.8824, 0.5882),
        (0.9255, 0.8784, 0.5804), (0.9216, 0.8706, 0.5686),
        (0.9176, 0.8667, 0.5569), (0.9137, 0.8627, 0.549),
        (0.9137, 0.8588, 0.5373), (0.9098, 0.851, 0.5294),
        (0.9059, 0.8471, 0.5176), (0.902, 0.8431, 0.5098),
        (0.898, 0.8353, 0.498), (0.8941, 0.8314, 0.4863),
        (0.8902, 0.8275, 0.4784), (0.8863, 0.8196, 0.4667),
        (0.8824, 0.8157, 0.4588), (0.8784, 0.8118, 0.4471),
        (0.8745, 0.8039, 0.4353), (0.8706, 0.8, 0.4275),
        (0.8667, 0.7961, 0.4157), (0.8627, 0.7882, 0.4078),
        (0.8627, 0.7843, 0.3961), (0.8588, 0.7804, 0.3882),
        (0.8549, 0.7765, 0.3765), (0.851, 0.7686, 0.3647),
        (0.8471, 0.7647, 0.3569), (0.8353, 0.749, 0.3255),
        (0.8353, 0.7451, 0.3176), (0.8314, 0.7412, 0.3098),
        (0.8314, 0.7333, 0.302), (0.8314, 0.7294, 0.298),
        (0.8314, 0.7255, 0.2902), (0.8275, 0.7216, 0.2824),
        (0.8275, 0.7137, 0.2745), (0.8275, 0.7098, 0.2667),
        (0.8275, 0.7059, 0.2588), (0.8235, 0.702, 0.251),
        (0.8235, 0.6941, 0.2471), (0.8235, 0.6902, 0.2392),
        (0.8196, 0.6863, 0.2314), (0.8196, 0.6824, 0.2235),
        (0.8196, 0.6745, 0.2157), (0.8196, 0.6706, 0.2078),
        (0.8157, 0.6667, 0.2), (0.8157, 0.6627, 0.1922),
        (0.8157, 0.6549, 0.1882), (0.8118, 0.651, 0.1804),
        (0.8118, 0.6471, 0.1725), (0.8118, 0.6431, 0.1647),
        (0.8118, 0.6353, 0.1569), (0.8078, 0.6314, 0.149),
        (0.8078, 0.6275, 0.1412), (0.8078, 0.6235, 0.1373),
        (0.8078, 0.6157, 0.1294), (0.8039, 0.6078, 0.1137),
        (0.7961, 0.6, 0.1137), (0.7843, 0.5922, 0.1176),
        (0.7765, 0.5804, 0.1176), (0.7686, 0.5725, 0.1216),
        (0.7608, 0.5647, 0.1216), (0.749, 0.5569, 0.1255),
        (0.7412, 0.5451, 0.1255), (0.7333, 0.5373, 0.1294),
        (0.7255, 0.5294, 0.1294), (0.7137, 0.5216, 0.1333),
        (0.7059, 0.5098, 0.1333), (0.698, 0.502, 0.1373),
        (0.6863, 0.4941, 0.1373), (0.6784, 0.4863, 0.1412),
        (0.6706, 0.4745, 0.1412), (0.6627, 0.4667, 0.1451),
        (0.651, 0.4588, 0.1451), (0.6431, 0.451, 0.149),
        (0.6353, 0.4392, 0.149), (0.6235, 0.4314, 0.1529),
        (0.6157, 0.4235, 0.1529), (0.6078, 0.4157, 0.1569),
        (0.6, 0.4039, 0.1569), (0.5882, 0.3961, 0.1608),
        (0.5804, 0.3882, 0.1608), (0.5725, 0.3804, 0.1647),
        (0.5647, 0.3686, 0.1647), (0.5451, 0.3529, 0.1686),
        (0.5373, 0.349, 0.1647), (0.5294, 0.3451, 0.1647),
        (0.5216, 0.3373, 0.1608), (0.5137, 0.3333, 0.1608),
        (0.5059, 0.3294, 0.1569), (0.498, 0.3255, 0.1529),
        (0.4902, 0.3176, 0.1529), (0.4863, 0.3137, 0.149),
        (0.4784, 0.3098, 0.149), (0.4706, 0.3059, 0.1451),
        (0.4627, 0.298, 0.1451), (0.4549, 0.2941, 0.1412),
        (0.4471, 0.2902, 0.1373), (0.4392, 0.2863, 0.1373),
        (0.4314, 0.2784, 0.1333), (0.4235, 0.2745, 0.1333),
        (0.4157, 0.2706, 0.1294), (0.4078, 0.2667, 0.1255),
        (0.4, 0.2588, 0.1255), (0.3922, 0.2549, 0.1216),
        (0.3843, 0.251, 0.1216), (0.3804, 0.2471, 0.1176),
        (0.3725, 0.2392, 0.1176), (0.3647, 0.2353, 0.1137),
        (0.3569, 0.2314, 0.1098), (0.349, 0.2275, 0.1098),
        (0.3412, 0.2196, 0.1059), (0.3255, 0.2118, 0.102),
        (0.349, 0.2392, 0.1333), (0.3686, 0.2627, 0.1647),
        (0.3922, 0.2902, 0.1961), (0.4157, 0.3176, 0.2275),
        (0.4353, 0.3412, 0.2549), (0.4588, 0.3686, 0.2863),
        (0.4824, 0.3961, 0.3176), (0.5059, 0.4235, 0.349),
        (0.5255, 0.4471, 0.3804), (0.549, 0.4745, 0.4118),
        (0.5725, 0.502, 0.4431), (0.5922, 0.5255, 0.4745),
        (0.6157, 0.5529, 0.5059), (0.6392, 0.5804, 0.5373),
        (0.6588, 0.6039, 0.5647), (0.6824, 0.6314, 0.5961),
        (0.7059, 0.6588, 0.6275), (0.7255, 0.6824, 0.6588),
        (0.749, 0.7098, 0.6902), (0.7725, 0.7373, 0.7216),
        (0.7922, 0.7608, 0.7529), (0.8157, 0.7882, 0.7843),
        (0.8392, 0.8157, 0.8157), (0.8627, 0.8431, 0.8471),
        (0.8824, 0.8667, 0.8745), (0.9059, 0.8941, 0.9059),
        (0.9294, 0.9216, 0.9373), (0.9725, 0.9725, 1.)],
    'ncl:psgcap': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 0., 0.), (0., 1., 0.),
        (0., 0., 1.), (0., 1., 1.),
        (1., 0., 0.83), (1., 1., 0.),
        (1., 0.5, 0.), (0.6, 0.83, 0.),
        (0., 1., 0.6), (0., 0.5, 1.),
        (0.55, 0., 0.83), (1., 0., 0.55),
        (0.33, 0.33, 0.33), (0.67, 0.67, 0.67),
        (1., 1., 0.33), (0.75, 1., 0.45),
        (0.45, 1., 0.6), (0.17, 1., 0.75),
        (0.25, 0.83, 0.83), (0.5, 0.67, 0.83),
        (0.75, 0.55, 0.83), (1., 0.33, 0.9),
        (0.67, 0.9, 0.45), (0.4, 0.9, 0.55),
        (0.17, 0.9, 0.67), (0.17, 0.67, 0.9),
        (0.17, 0.5, 1.), (0.45, 0.33, 1.),
        (0.75, 0.17, 1.), (0.9, 0.09, 1.),
        (0.83, 1., 0.17), (0.67, 1., 0.25),
        (0.45, 1., 0.33), (0.17, 1., 0.5),
        (0.17, 0.83, 0.6), (0.17, 0.67, 0.75),
        (0.17, 0.55, 0.83), (0.25, 0.45, 0.9),
        (0.4, 0.33, 0.9), (0.67, 0.17, 0.9),
        (0.83, 0.17, 0.83), (0.9, 0.33, 0.67),
        (0.83, 0.45, 0.6), (0.83, 0.6, 0.5),
        (0.9, 0.67, 0.4), (0.9, 0.67, 0.25),
        (1., 0.9, 0.09), (0.83, 1., 0.09),
        (0.6, 1., 0.17), (0.45, 1., 0.25),
        (0.17, 0.9, 0.45), (0.17, 0.83, 0.55),
        (0.17, 0.67, 0.67), (0.17, 0.55, 0.75),
        (0.17, 0.4, 0.9), (0.4, 0.25, 0.9),
        (0.55, 0.17, 0.9), (0.83, 0.17, 0.75),
        (1., 0.17, 0.67), (1., 0.25, 0.6),
        (1., 0.4, 0.5), (1., 0.5, 0.4),
        (1., 0.83, 0.09), (0.75, 1., 0.),
        (0.6, 1., 0.13), (0.4, 1., 0.17),
        (0.13, 1., 0.33), (0.13, 0.83, 0.45),
        (0.17, 0.75, 0.5), (0.13, 0.6, 0.67),
        (0.13, 0.5, 0.75), (0.13, 0.4, 0.83),
        (0.17, 0.25, 0.9), (0.17, 0.17, 1.),
        (0.33, 0.09, 1.), (0.55, 0.09, 0.9),
        (0.75, 0., 0.83), (0.9, 0., 0.75),
        (0.9, 0.75, 0.09), (0.67, 0.9, 0.09),
        (0.55, 0.9, 0.13), (0.25, 0.9, 0.25),
        (0.13, 0.9, 0.33), (0.17, 0.75, 0.4),
        (0.13, 0.67, 0.5), (0.13, 0.55, 0.6),
        (0.13, 0.45, 0.67), (0.17, 0.33, 0.75),
        (0.17, 0.25, 0.83), (0.33, 0.17, 0.83),
        (0.55, 0.13, 0.75), (0.75, 0.13, 0.67),
        (0.9, 0.17, 0.55), (0.83, 0.25, 0.5),
        (0.83, 0.75, 0.), (0.6, 0.83, 0.09),
        (0.45, 0.9, 0.09), (0.33, 0.9, 0.13),
        (0.13, 0.9, 0.25), (0.17, 0.83, 0.25),
        (0.17, 0.75, 0.33), (0.17, 0.6, 0.45),
        (0.13, 0.5, 0.55), (0.13, 0.45, 0.6),
        (0.17, 0.33, 0.67), (0.33, 0.25, 0.67),
        (0.45, 0.25, 0.6), (0.67, 0.25, 0.5),
        (0.83, 0.25, 0.4), (1., 0.25, 0.33),
        (0.75, 0.67, 0.), (0.55, 0.75, 0.09),
        (0.4, 0.83, 0.09), (0.25, 0.9, 0.09),
        (0.17, 0.9, 0.13), (0.17, 0.83, 0.17),
        (0.13, 0.75, 0.25), (0.13, 0.67, 0.33),
        (0.13, 0.6, 0.4), (0.13, 0.45, 0.5),
        (0.13, 0.4, 0.55), (0.25, 0.33, 0.55),
        (0.45, 0.33, 0.45), (0.55, 0.33, 0.4),
        (0.67, 0.33, 0.33), (0.83, 0.33, 0.25),
        (0.67, 0.6, 0.), (0.5, 0.67, 0.09),
        (0.4, 0.67, 0.13), (0.25, 0.67, 0.17),
        (0.13, 0.67, 0.25), (0.09, 0.6, 0.33),
        (0.09, 0.5, 0.4), (0.13, 0.4, 0.45),
        (0.13, 0.33, 0.5), (0.09, 0.17, 0.67),
        (0.17, 0.13, 0.67), (0.4, 0.13, 0.55),
        (0.5, 0.13, 0.5), (0.6, 0.13, 0.45),
        (0.75, 0.13, 0.4), (0.9, 0.13, 0.33),
        (0.55, 0.5, 0.09), (0.45, 0.6, 0.09),
        (0.33, 0.6, 0.13), (0.17, 0.6, 0.17),
        (0.09, 0.55, 0.25), (0.13, 0.45, 0.33),
        (0.09, 0.4, 0.4), (0.17, 0.33, 0.4),
        (0.17, 0.25, 0.45), (0.25, 0.17, 0.5),
        (0.25, 0.13, 0.55), (0.33, 0.13, 0.5),
        (0.45, 0.13, 0.45), (0.55, 0.13, 0.4),
        (0.67, 0.13, 0.33), (0.83, 0.13, 0.25),
        (0.5, 0.45, 0.09), (0.4, 0.5, 0.09),
        (0.33, 0.6, 0.), (0.17, 0.6, 0.09),
        (0.13, 0.67, 0.09), (0.13, 0.6, 0.13),
        (0.13, 0.5, 0.17), (0.09, 0.45, 0.25),
        (0.13, 0.4, 0.25), (0.13, 0.33, 0.33),
        (0.09, 0.25, 0.4), (0.09, 0.17, 0.5),
        (0.09, 0.13, 0.55), (0.13, 0., 0.6),
        (0.33, 0., 0.5), (0.5, 0., 0.4),
        (0.75, 0.33, 0.), (0.67, 0.4, 0.),
        (0.55, 0.4, 0.09), (0.33, 0.4, 0.17),
        (0.25, 0.33, 0.25), (0.25, 0.25, 0.33),
        (0.25, 0.17, 0.4), (0.33, 0.13, 0.4),
        (0.5, 0.13, 0.33), (0.6, 0.13, 0.25),
        (0.75, 0.09, 0.25), (0.83, 0., 0.25),
        (1., 0., 0.17), (1., 0.13, 0.13),
        (0.9, 0.17, 0.09), (0.83, 0.25, 0.09),
        (0.6, 0.25, 0.), (0.45, 0.4, 0.),
        (0.25, 0.4, 0.13), (0.25, 0.33, 0.17),
        (0.17, 0.25, 0.25), (0.17, 0.17, 0.33),
        (0.25, 0.13, 0.33), (0.33, 0.09, 0.33),
        (0.5, 0.09, 0.25), (0.67, 0.09, 0.17),
        (0.83, 0., 0.17), (0.9, 0., 0.13),
        (1., 0., 0.09), (1., 0.13, 0.09),
        (0.83, 0.17, 0.), (0.75, 0.25, 0.),
        (0.5, 0.17, 0.), (0.4, 0.33, 0.),
        (0.13, 0.45, 0.09), (0.09, 0.4, 0.13),
        (0.09, 0.33, 0.17), (0.09, 0.25, 0.25),
        (0.13, 0.17, 0.33), (0.17, 0.09, 0.4),
        (0.25, 0., 0.33), (0.45, 0., 0.25),
        (0.6, 0., 0.17), (0.75, 0., 0.13),
        (0.83, 0., 0.09), (0.75, 0.09, 0.09),
        (0.67, 0.13, 0.09), (0.6, 0.17, 0.),
        (0.4, 0.17, 0.), (0.17, 0.33, 0.),
        (0.09, 0.33, 0.), (0.09, 0.25, 0.09),
        (0.09, 0.17, 0.13), (0.09, 0.13, 0.25),
        (0.09, 0.13, 0.33), (0.09, 0., 0.33),
        (0.13, 0., 0.25), (0.17, 0., 0.17),
        (0.33, 0., 0.17), (0.33, 0., 0.13), (0.4, 0., 0.09),
        (0.45, 0.09, 0.09), (0.45, 0.13, 0.), (0.17, 0.09, 0.)],
    'ncl:rainbow': [
        (1., 1., 1.), (0., 0., 0.),
        (0.3294, 0., 0.6235), (0.3294, 0., 0.6392),
        (0.3294, 0., 0.6588), (0.3098, 0., 0.6941),
        (0.3059, 0., 0.7137), (0.302, 0., 0.7294),
        (0.298, 0., 0.749), (0.2745, 0., 0.7843),
        (0.2667, 0., 0.8), (0.2588, 0., 0.8196),
        (0.2353, 0., 0.8392), (0.2275, 0., 0.8549),
        (0.2157, 0., 0.8745), (0.1804, 0., 0.9098),
        (0.1686, 0., 0.9255), (0.1569, 0., 0.9451),
        (0.1412, 0., 0.9608), (0.1294, 0., 0.9804),
        (0.098, 0., 1.), (0.0627, 0., 1.),
        (0.0471, 0., 1.), (0.0157, 0., 1.),
        (0., 0., 1.), (0., 0.0157, 1.),
        (0., 0.0627, 1.), (0., 0.0824, 1.),
        (0., 0.098, 1.), (0., 0.1137, 1.),
        (0., 0.149, 1.), (0., 0.1647, 1.),
        (0., 0.1804, 1.), (0., 0.2, 1.),
        (0., 0.2471, 1.), (0., 0.2627, 1.),
        (0., 0.2824, 1.), (0., 0.3294, 1.),
        (0., 0.349, 1.), (0., 0.3647, 1.),
        (0., 0.3804, 1.), (0., 0.4157, 1.),
        (0., 0.4314, 1.), (0., 0.4471, 1.),
        (0., 0.4667, 1.), (0., 0.498, 1.),
        (0., 0.5294, 1.), (0., 0.549, 1.),
        (0., 0.5961, 1.), (0., 0.6157, 1.),
        (0., 0.6314, 1.), (0., 0.6471, 1.),
        (0., 0.6824, 1.), (0., 0.698, 1.),
        (0., 0.7137, 1.), (0., 0.7333, 1.),
        (0., 0.7647, 1.), (0., 0.7804, 1.),
        (0., 0.8471, 1.), (0., 0.8627, 1.),
        (0., 0.8824, 1.), (0., 0.898, 1.),
        (0., 0.9137, 1.), (0., 0.949, 1.),
        (0., 0.9647, 1.), (0., 0.9804, 1.),
        (0., 1., 1.), (0., 1., 0.9647),
        (0., 1., 0.949), (0., 1., 0.9333),
        (0., 1., 0.8824), (0., 1., 0.8471),
        (0., 1., 0.8314), (0., 1., 0.7961),
        (0., 1., 0.7804), (0., 1., 0.7647),
        (0., 1., 0.749), (0., 1., 0.7333),
        (0., 1., 0.698), (0., 1., 0.6824),
        (0., 1., 0.6667), (0., 1., 0.6157),
        (0., 1., 0.5961), (0., 1., 0.5647),
        (0., 1., 0.5294), (0., 1., 0.5137),
        (0., 1., 0.498), (0., 1., 0.4824),
        (0., 1., 0.4471), (0., 1., 0.4314),
        (0., 1., 0.4157), (0., 1., 0.4),
        (0., 1., 0.349), (0., 1., 0.3294),
        (0., 1., 0.3137), (0., 1., 0.298),
        (0., 1., 0.2471), (0., 1., 0.2314),
        (0., 1., 0.2157), (0., 1., 0.1804),
        (0., 1., 0.1647), (0., 1., 0.149),
        (0., 1., 0.098), (0., 1., 0.0824),
        (0., 1., 0.0627), (0., 1., 0.0471),
        (0., 1., 0.0314), (0., 1., 0.),
        (0.0314, 1., 0.), (0.0471, 1., 0.),
        (0.0824, 1., 0.), (0.098, 1., 0.),
        (0.1137, 1., 0.), (0.1647, 1., 0.),
        (0.1804, 1., 0.), (0.2, 1., 0.),
        (0.2157, 1., 0.), (0.2471, 1., 0.),
        (0.2627, 1., 0.), (0.2824, 1., 0.),
        (0.298, 1., 0.), (0.349, 1., 0.),
        (0.3647, 1., 0.), (0.3804, 1., 0.),
        (0.4314, 1., 0.), (0.4471, 1., 0.),
        (0.4667, 1., 0.), (0.4824, 1., 0.),
        (0.5137, 1., 0.), (0.5294, 1., 0.),
        (0.549, 1., 0.), (0.5647, 1., 0.),
        (0.6, 1., 0.), (0.6314, 1., 0.),
        (0.6471, 1., 0.), (0.698, 1., 0.),
        (0.7137, 1., 0.), (0.7333, 1., 0.),
        (0.749, 1., 0.), (0.7804, 1., 0.),
        (0.7961, 1., 0.), (0.8157, 1., 0.),
        (0.8314, 1., 0.), (0.8667, 1., 0.),
        (0.8824, 1., 0.), (0.949, 1., 0.),
        (0.9647, 1., 0.), (0.9804, 1., 0.),
        (1., 1., 0.), (1., 0.9804, 0.),
        (1., 0.949, 0.), (1., 0.9333, 0.),
        (1., 0.9137, 0.), (1., 0.898, 0.),
        (1., 0.8667, 0.), (1., 0.8471, 0.),
        (1., 0.8314, 0.), (1., 0.7804, 0.),
        (1., 0.749, 0.), (1., 0.7333, 0.),
        (1., 0.698, 0.), (1., 0.6824, 0.),
        (1., 0.6667, 0.), (1., 0.6471, 0.),
        (1., 0.6314, 0.), (1., 0.6, 0.),
        (1., 0.5804, 0.), (1., 0.5647, 0.),
        (1., 0.5137, 0.), (1., 0.498, 0.),
        (1., 0.4667, 0.), (1., 0.4314, 0.),
        (1., 0.4157, 0.), (1., 0.4, 0.),
        (1., 0.3804, 0.), (1., 0.349, 0.),
        (1., 0.3333, 0.), (1., 0.3137, 0.),
        (1., 0.298, 0.), (1., 0.2471, 0.),
        (1., 0.2314, 0.), (1., 0.2157, 0.),
        (1., 0.2, 0.), (1., 0.149, 0.),
        (1., 0.1333, 0.), (1., 0.1137, 0.),
        (1., 0.0824, 0.), (1., 0.0667, 0.),
        (1., 0.0471, 0.), (1., 0., 0.)],
    'ncl:rainbowwhitegray': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0.0157, 0., 0.0118),
        (0.0353, 0., 0.0275), (0.051, 0., 0.0392),
        (0.0706, 0., 0.0549), (0.0863, 0., 0.0745),
        (0.1059, 0., 0.0902), (0.1216, 0., 0.1098),
        (0.1412, 0., 0.1255), (0.1569, 0., 0.149),
        (0.1765, 0., 0.1686), (0.1961, 0., 0.1882),
        (0.2275, 0., 0.2314), (0.251, 0., 0.2667),
        (0.2667, 0., 0.2824), (0.2706, 0., 0.302),
        (0.2824, 0., 0.3176), (0.2902, 0., 0.3373),
        (0.302, 0., 0.3569), (0.3098, 0., 0.3725),
        (0.3137, 0., 0.3922), (0.3216, 0., 0.4078),
        (0.3255, 0., 0.4275), (0.3294, 0., 0.4627),
        (0.3373, 0., 0.4784), (0.3451, 0., 0.5176),
        (0.3373, 0., 0.5333), (0.3412, 0., 0.5529),
        (0.3412, 0., 0.5686), (0.3412, 0., 0.5882),
        (0.3333, 0., 0.6039), (0.3294, 0., 0.6235),
        (0.3294, 0., 0.6392), (0.3294, 0., 0.6588),
        (0.3098, 0., 0.6941), (0.3059, 0., 0.7137),
        (0.302, 0., 0.7294), (0.298, 0., 0.749),
        (0.2745, 0., 0.7843), (0.2667, 0., 0.8),
        (0.2588, 0., 0.8196), (0.2353, 0., 0.8392),
        (0.2275, 0., 0.8549), (0.2157, 0., 0.8745),
        (0.1804, 0., 0.9098), (0.1686, 0., 0.9255),
        (0.1569, 0., 0.9451), (0.1412, 0., 0.9608),
        (0.1294, 0., 0.9804), (0.098, 0., 1.),
        (0.0627, 0., 1.), (0.0471, 0., 1.),
        (0.0157, 0., 1.), (0., 0., 1.),
        (0., 0.0157, 1.), (0., 0.0627, 1.),
        (0., 0.0824, 1.), (0., 0.098, 1.),
        (0., 0.1137, 1.), (0., 0.149, 1.),
        (0., 0.1647, 1.), (0., 0.1804, 1.),
        (0., 0.2, 1.), (0., 0.2471, 1.),
        (0., 0.2627, 1.), (0., 0.2824, 1.),
        (0., 0.3294, 1.), (0., 0.349, 1.),
        (0., 0.3647, 1.), (0., 0.3804, 1.),
        (0., 0.4157, 1.), (0., 0.4314, 1.),
        (0., 0.4471, 1.), (0., 0.4667, 1.),
        (0., 0.498, 1.), (0., 0.5294, 1.),
        (0., 0.549, 1.), (0., 0.5961, 1.),
        (0., 0.6157, 1.), (0., 0.6314, 1.),
        (0., 0.6471, 1.), (0., 0.6824, 1.),
        (0., 0.698, 1.), (0., 0.7137, 1.),
        (0., 0.7333, 1.), (0., 0.7647, 1.),
        (0., 0.7804, 1.), (0., 0.8471, 1.),
        (0., 0.8627, 1.), (0., 0.8824, 1.),
        (0., 0.898, 1.), (0., 0.9137, 1.),
        (0., 0.949, 1.), (0., 0.9647, 1.),
        (0., 0.9804, 1.), (0., 1., 1.),
        (0., 1., 0.9647), (0., 1., 0.949),
        (0., 1., 0.9333), (0., 1., 0.8824),
        (0., 1., 0.8471), (0., 1., 0.8314),
        (0., 1., 0.7961), (0., 1., 0.7804),
        (0., 1., 0.7647), (0., 1., 0.749),
        (0., 1., 0.7333), (0., 1., 0.698),
        (0., 1., 0.6824), (0., 1., 0.6667),
        (0., 1., 0.6157), (0., 1., 0.5961),
        (0., 1., 0.5647), (0., 1., 0.5294),
        (0., 1., 0.5137), (0., 1., 0.498),
        (0., 1., 0.4824), (0., 1., 0.4471),
        (0., 1., 0.4314), (0., 1., 0.4157),
        (0., 1., 0.4), (0., 1., 0.349),
        (0., 1., 0.3294), (0., 1., 0.3137),
        (0., 1., 0.298), (0., 1., 0.2471),
        (0., 1., 0.2314), (0., 1., 0.2157),
        (0., 1., 0.1804), (0., 1., 0.1647),
        (0., 1., 0.149), (0., 1., 0.098),
        (0., 1., 0.0824), (0., 1., 0.0627),
        (0., 1., 0.0471), (0., 1., 0.0314),
        (0., 1., 0.), (0.0314, 1., 0.),
        (0.0471, 1., 0.), (0.0824, 1., 0.),
        (0.098, 1., 0.), (0.1137, 1., 0.),
        (0.1647, 1., 0.), (0.1804, 1., 0.),
        (0.2, 1., 0.), (0.2157, 1., 0.),
        (0.2471, 1., 0.), (0.2627, 1., 0.),
        (0.2824, 1., 0.), (0.298, 1., 0.),
        (0.349, 1., 0.), (0.3647, 1., 0.),
        (0.3804, 1., 0.), (0.4314, 1., 0.),
        (0.4471, 1., 0.), (0.4667, 1., 0.),
        (0.4824, 1., 0.), (0.5137, 1., 0.),
        (0.5294, 1., 0.), (0.549, 1., 0.),
        (0.5647, 1., 0.), (0.6, 1., 0.),
        (0.6314, 1., 0.), (0.6471, 1., 0.),
        (0.698, 1., 0.), (0.7137, 1., 0.),
        (0.7333, 1., 0.), (0.749, 1., 0.),
        (0.7804, 1., 0.), (0.7961, 1., 0.),
        (0.8157, 1., 0.), (0.8314, 1., 0.),
        (0.8667, 1., 0.), (0.8824, 1., 0.),
        (0.949, 1., 0.), (0.9647, 1., 0.),
        (0.9804, 1., 0.), (1., 1., 0.),
        (1., 0.9804, 0.), (1., 0.949, 0.),
        (1., 0.9333, 0.), (1., 0.9137, 0.),
        (1., 0.898, 0.), (1., 0.8667, 0.),
        (1., 0.8471, 0.), (1., 0.8314, 0.),
        (1., 0.7804, 0.), (1., 0.749, 0.),
        (1., 0.7333, 0.), (1., 0.698, 0.),
        (1., 0.6824, 0.), (1., 0.6667, 0.),
        (1., 0.6471, 0.), (1., 0.6314, 0.),
        (1., 0.6, 0.), (1., 0.5804, 0.),
        (1., 0.5647, 0.), (1., 0.5137, 0.),
        (1., 0.498, 0.), (1., 0.4667, 0.),
        (1., 0.4314, 0.), (1., 0.4157, 0.),
        (1., 0.4, 0.), (1., 0.3804, 0.),
        (1., 0.349, 0.), (1., 0.3333, 0.),
        (1., 0.3137, 0.), (1., 0.298, 0.),
        (1., 0.2471, 0.), (1., 0.2314, 0.),
        (1., 0.2157, 0.), (1., 0.2, 0.),
        (1., 0.149, 0.), (1., 0.1333, 0.),
        (1., 0.1137, 0.), (1., 0.0824, 0.),
        (1., 0.0667, 0.), (1., 0.0471, 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 1., 1.), (0.8275, 0.8275, 0.8275)],
    'ncl:rainbowwhite': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0.0157, 0., 0.0118),
        (0.0353, 0., 0.0275), (0.051, 0., 0.0392),
        (0.0706, 0., 0.0549), (0.0863, 0., 0.0745),
        (0.1059, 0., 0.0902), (0.1216, 0., 0.1098),
        (0.1412, 0., 0.1255), (0.1569, 0., 0.149),
        (0.1765, 0., 0.1686), (0.1961, 0., 0.1882),
        (0.2275, 0., 0.2314), (0.251, 0., 0.2667),
        (0.2667, 0., 0.2824), (0.2706, 0., 0.302),
        (0.2824, 0., 0.3176), (0.2902, 0., 0.3373),
        (0.302, 0., 0.3569), (0.3098, 0., 0.3725),
        (0.3137, 0., 0.3922), (0.3216, 0., 0.4078),
        (0.3255, 0., 0.4275), (0.3294, 0., 0.4627),
        (0.3373, 0., 0.4784), (0.3451, 0., 0.5176),
        (0.3373, 0., 0.5333), (0.3412, 0., 0.5529),
        (0.3412, 0., 0.5686), (0.3412, 0., 0.5882),
        (0.3333, 0., 0.6039), (0.3294, 0., 0.6235),
        (0.3294, 0., 0.6392), (0.3294, 0., 0.6588),
        (0.3098, 0., 0.6941), (0.3059, 0., 0.7137),
        (0.302, 0., 0.7294), (0.298, 0., 0.749),
        (0.2745, 0., 0.7843), (0.2667, 0., 0.8),
        (0.2588, 0., 0.8196), (0.2353, 0., 0.8392),
        (0.2275, 0., 0.8549), (0.2157, 0., 0.8745),
        (0.1804, 0., 0.9098), (0.1686, 0., 0.9255),
        (0.1569, 0., 0.9451), (0.1412, 0., 0.9608),
        (0.1294, 0., 0.9804), (0.098, 0., 1.),
        (0.0627, 0., 1.), (0.0471, 0., 1.),
        (0.0157, 0., 1.), (0., 0., 1.),
        (0., 0.0157, 1.), (0., 0.0627, 1.),
        (0., 0.0824, 1.), (0., 0.098, 1.),
        (0., 0.1137, 1.), (0., 0.149, 1.),
        (0., 0.1647, 1.), (0., 0.1804, 1.),
        (0., 0.2, 1.), (0., 0.2471, 1.),
        (0., 0.2627, 1.), (0., 0.2824, 1.),
        (0., 0.3294, 1.), (0., 0.349, 1.),
        (0., 0.3647, 1.), (0., 0.3804, 1.),
        (0., 0.4157, 1.), (0., 0.4314, 1.),
        (0., 0.4471, 1.), (0., 0.4667, 1.),
        (0., 0.498, 1.), (0., 0.5294, 1.),
        (0., 0.549, 1.), (0., 0.5961, 1.),
        (0., 0.6157, 1.), (0., 0.6314, 1.),
        (0., 0.6471, 1.), (0., 0.6824, 1.),
        (0., 0.698, 1.), (0., 0.7137, 1.),
        (0., 0.7333, 1.), (0., 0.7647, 1.),
        (0., 0.7804, 1.), (0., 0.8471, 1.),
        (0., 0.8627, 1.), (0., 0.8824, 1.),
        (0., 0.898, 1.), (0., 0.9137, 1.),
        (0., 0.949, 1.), (0., 0.9647, 1.),
        (0., 0.9804, 1.), (0., 1., 1.),
        (0., 1., 0.9647), (0., 1., 0.949),
        (0., 1., 0.9333), (0., 1., 0.8824),
        (0., 1., 0.8471), (0., 1., 0.8314),
        (0., 1., 0.7961), (0., 1., 0.7804),
        (0., 1., 0.7647), (0., 1., 0.749),
        (0., 1., 0.7333), (0., 1., 0.698),
        (0., 1., 0.6824), (0., 1., 0.6667),
        (0., 1., 0.6157), (0., 1., 0.5961),
        (0., 1., 0.5647), (0., 1., 0.5294),
        (0., 1., 0.5137), (0., 1., 0.498),
        (0., 1., 0.4824), (0., 1., 0.4471),
        (0., 1., 0.4314), (0., 1., 0.4157),
        (0., 1., 0.4), (0., 1., 0.349),
        (0., 1., 0.3294), (0., 1., 0.3137),
        (0., 1., 0.298), (0., 1., 0.2471),
        (0., 1., 0.2314), (0., 1., 0.2157),
        (0., 1., 0.1804), (0., 1., 0.1647),
        (0., 1., 0.149), (0., 1., 0.098),
        (0., 1., 0.0824), (0., 1., 0.0627),
        (0., 1., 0.0471), (0., 1., 0.0314),
        (0., 1., 0.), (0.0314, 1., 0.),
        (0.0471, 1., 0.), (0.0824, 1., 0.),
        (0.098, 1., 0.), (0.1137, 1., 0.),
        (0.1647, 1., 0.), (0.1804, 1., 0.),
        (0.2, 1., 0.), (0.2157, 1., 0.),
        (0.2471, 1., 0.), (0.2627, 1., 0.),
        (0.2824, 1., 0.), (0.298, 1., 0.),
        (0.349, 1., 0.), (0.3647, 1., 0.),
        (0.3804, 1., 0.), (0.4314, 1., 0.),
        (0.4471, 1., 0.), (0.4667, 1., 0.),
        (0.4824, 1., 0.), (0.5137, 1., 0.),
        (0.5294, 1., 0.), (0.549, 1., 0.),
        (0.5647, 1., 0.), (0.6, 1., 0.),
        (0.6314, 1., 0.), (0.6471, 1., 0.),
        (0.698, 1., 0.), (0.7137, 1., 0.),
        (0.7333, 1., 0.), (0.749, 1., 0.),
        (0.7804, 1., 0.), (0.7961, 1., 0.),
        (0.8157, 1., 0.), (0.8314, 1., 0.),
        (0.8667, 1., 0.), (0.8824, 1., 0.),
        (0.949, 1., 0.), (0.9647, 1., 0.),
        (0.9804, 1., 0.), (1., 1., 0.),
        (1., 0.9804, 0.), (1., 0.949, 0.),
        (1., 0.9333, 0.), (1., 0.9137, 0.),
        (1., 0.898, 0.), (1., 0.8667, 0.),
        (1., 0.8471, 0.), (1., 0.8314, 0.),
        (1., 0.7804, 0.), (1., 0.749, 0.),
        (1., 0.7333, 0.), (1., 0.698, 0.),
        (1., 0.6824, 0.), (1., 0.6667, 0.),
        (1., 0.6471, 0.), (1., 0.6314, 0.),
        (1., 0.6, 0.), (1., 0.5804, 0.),
        (1., 0.5647, 0.), (1., 0.5137, 0.),
        (1., 0.498, 0.), (1., 0.4667, 0.),
        (1., 0.4314, 0.), (1., 0.4157, 0.),
        (1., 0.4, 0.), (1., 0.3804, 0.),
        (1., 0.349, 0.), (1., 0.3333, 0.),
        (1., 0.3137, 0.), (1., 0.298, 0.),
        (1., 0.2471, 0.), (1., 0.2314, 0.),
        (1., 0.2157, 0.), (1., 0.2, 0.),
        (1., 0.149, 0.), (1., 0.1333, 0.),
        (1., 0.1137, 0.), (1., 0.0824, 0.),
        (1., 0.0667, 0.), (1., 0.0471, 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 1., 1.)],
    'ncl:rainbowgray': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0.0157, 0., 0.0118),
        (0.0353, 0., 0.0275), (0.051, 0., 0.0392),
        (0.0706, 0., 0.0549), (0.0863, 0., 0.0745),
        (0.1059, 0., 0.0902), (0.1216, 0., 0.1098),
        (0.1412, 0., 0.1255), (0.1569, 0., 0.149),
        (0.1765, 0., 0.1686), (0.1961, 0., 0.1882),
        (0.2275, 0., 0.2314), (0.251, 0., 0.2667),
        (0.2667, 0., 0.2824), (0.2706, 0., 0.302),
        (0.2824, 0., 0.3176), (0.2902, 0., 0.3373),
        (0.302, 0., 0.3569), (0.3098, 0., 0.3725),
        (0.3137, 0., 0.3922), (0.3216, 0., 0.4078),
        (0.3255, 0., 0.4275), (0.3294, 0., 0.4627),
        (0.3373, 0., 0.4784), (0.3451, 0., 0.5176),
        (0.3373, 0., 0.5333), (0.3412, 0., 0.5529),
        (0.3412, 0., 0.5686), (0.3412, 0., 0.5882),
        (0.3333, 0., 0.6039), (0.3294, 0., 0.6235),
        (0.3294, 0., 0.6392), (0.3294, 0., 0.6588),
        (0.3098, 0., 0.6941), (0.3059, 0., 0.7137),
        (0.302, 0., 0.7294), (0.298, 0., 0.749),
        (0.2745, 0., 0.7843), (0.2667, 0., 0.8),
        (0.2588, 0., 0.8196), (0.2353, 0., 0.8392),
        (0.2275, 0., 0.8549), (0.2157, 0., 0.8745),
        (0.1804, 0., 0.9098), (0.1686, 0., 0.9255),
        (0.1569, 0., 0.9451), (0.1412, 0., 0.9608),
        (0.1294, 0., 0.9804), (0.098, 0., 1.),
        (0.0627, 0., 1.), (0.0471, 0., 1.),
        (0.0157, 0., 1.), (0., 0., 1.),
        (0., 0.0157, 1.), (0., 0.0627, 1.),
        (0., 0.0824, 1.), (0., 0.098, 1.),
        (0., 0.1137, 1.), (0., 0.149, 1.),
        (0., 0.1647, 1.), (0., 0.1804, 1.),
        (0., 0.2, 1.), (0., 0.2471, 1.),
        (0., 0.2627, 1.), (0., 0.2824, 1.),
        (0., 0.3294, 1.), (0., 0.349, 1.),
        (0., 0.3647, 1.), (0., 0.3804, 1.),
        (0., 0.4157, 1.), (0., 0.4314, 1.),
        (0., 0.4471, 1.), (0., 0.4667, 1.),
        (0., 0.498, 1.), (0., 0.5294, 1.),
        (0., 0.549, 1.), (0., 0.5961, 1.),
        (0., 0.6157, 1.), (0., 0.6314, 1.),
        (0., 0.6471, 1.), (0., 0.6824, 1.),
        (0., 0.698, 1.), (0., 0.7137, 1.),
        (0., 0.7333, 1.), (0., 0.7647, 1.),
        (0., 0.7804, 1.), (0., 0.8471, 1.),
        (0., 0.8627, 1.), (0., 0.8824, 1.),
        (0., 0.898, 1.), (0., 0.9137, 1.),
        (0., 0.949, 1.), (0., 0.9647, 1.),
        (0., 0.9804, 1.), (0., 1., 1.),
        (0., 1., 0.9647), (0., 1., 0.949),
        (0., 1., 0.9333), (0., 1., 0.8824),
        (0., 1., 0.8471), (0., 1., 0.8314),
        (0., 1., 0.7961), (0., 1., 0.7804),
        (0., 1., 0.7647), (0., 1., 0.749),
        (0., 1., 0.7333), (0., 1., 0.698),
        (0., 1., 0.6824), (0., 1., 0.6667),
        (0., 1., 0.6157), (0., 1., 0.5961),
        (0., 1., 0.5647), (0., 1., 0.5294),
        (0., 1., 0.5137), (0., 1., 0.498),
        (0., 1., 0.4824), (0., 1., 0.4471),
        (0., 1., 0.4314), (0., 1., 0.4157),
        (0., 1., 0.4), (0., 1., 0.349),
        (0., 1., 0.3294), (0., 1., 0.3137),
        (0., 1., 0.298), (0., 1., 0.2471),
        (0., 1., 0.2314), (0., 1., 0.2157),
        (0., 1., 0.1804), (0., 1., 0.1647),
        (0., 1., 0.149), (0., 1., 0.098),
        (0., 1., 0.0824), (0., 1., 0.0627),
        (0., 1., 0.0471), (0., 1., 0.0314),
        (0., 1., 0.), (0.0314, 1., 0.),
        (0.0471, 1., 0.), (0.0824, 1., 0.),
        (0.098, 1., 0.), (0.1137, 1., 0.),
        (0.1647, 1., 0.), (0.1804, 1., 0.),
        (0.2, 1., 0.), (0.2157, 1., 0.),
        (0.2471, 1., 0.), (0.2627, 1., 0.),
        (0.2824, 1., 0.), (0.298, 1., 0.),
        (0.349, 1., 0.), (0.3647, 1., 0.),
        (0.3804, 1., 0.), (0.4314, 1., 0.),
        (0.4471, 1., 0.), (0.4667, 1., 0.),
        (0.4824, 1., 0.), (0.5137, 1., 0.),
        (0.5294, 1., 0.), (0.549, 1., 0.),
        (0.5647, 1., 0.), (0.6, 1., 0.),
        (0.6314, 1., 0.), (0.6471, 1., 0.),
        (0.698, 1., 0.), (0.7137, 1., 0.),
        (0.7333, 1., 0.), (0.749, 1., 0.),
        (0.7804, 1., 0.), (0.7961, 1., 0.),
        (0.8157, 1., 0.), (0.8314, 1., 0.),
        (0.8667, 1., 0.), (0.8824, 1., 0.),
        (0.949, 1., 0.), (0.9647, 1., 0.),
        (0.9804, 1., 0.), (1., 1., 0.),
        (1., 0.9804, 0.), (1., 0.949, 0.),
        (1., 0.9333, 0.), (1., 0.9137, 0.),
        (1., 0.898, 0.), (1., 0.8667, 0.),
        (1., 0.8471, 0.), (1., 0.8314, 0.),
        (1., 0.7804, 0.), (1., 0.749, 0.),
        (1., 0.7333, 0.), (1., 0.698, 0.),
        (1., 0.6824, 0.), (1., 0.6667, 0.),
        (1., 0.6471, 0.), (1., 0.6314, 0.),
        (1., 0.6, 0.), (1., 0.5804, 0.),
        (1., 0.5647, 0.), (1., 0.5137, 0.),
        (1., 0.498, 0.), (1., 0.4667, 0.),
        (1., 0.4314, 0.), (1., 0.4157, 0.),
        (1., 0.4, 0.), (1., 0.3804, 0.),
        (1., 0.349, 0.), (1., 0.3333, 0.),
        (1., 0.3137, 0.), (1., 0.298, 0.),
        (1., 0.2471, 0.), (1., 0.2314, 0.),
        (1., 0.2157, 0.), (1., 0.2, 0.),
        (1., 0.149, 0.), (1., 0.1333, 0.),
        (1., 0.1137, 0.), (1., 0.0824, 0.),
        (1., 0.0667, 0.), (1., 0.0471, 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (0.8275, 0.8275, 0.8275)],
    'ncl:testcmap': [
        (1., 1., 1.), (0., 0., 0.),
        (0.3412, 0., 0.5333), (0.3294, 0., 0.549),
        (0.3216, 0., 0.5608), (0.3098, 0., 0.5765),
        (0.298, 0., 0.5882), (0.2902, 0., 0.6039),
        (0.2784, 0., 0.6196), (0.2706, 0., 0.6314),
        (0.2588, 0., 0.6471), (0.2471, 0., 0.6588),
        (0.2392, 0., 0.6745), (0.2275, 0., 0.6902),
        (0.2157, 0., 0.702), (0.2078, 0., 0.7176),
        (0.1961, 0., 0.7294), (0.1843, 0., 0.7451),
        (0.1765, 0., 0.7608), (0.1647, 0., 0.7725),
        (0.1569, 0., 0.7882), (0.1451, 0., 0.8039),
        (0.1333, 0., 0.8157), (0.1255, 0., 0.8314),
        (0.1137, 0., 0.8431), (0.102, 0., 0.8588),
        (0.0941, 0., 0.8745), (0.0824, 0., 0.8863),
        (0.0706, 0., 0.902), (0.0627, 0., 0.9137),
        (0.051, 0., 0.9294), (0.0431, 0., 0.9451),
        (0.0314, 0., 0.9569), (0.0196, 0., 0.9725),
        (0.0118, 0., 0.9843), (0., 0., 1.),
        (0., 0.0196, 0.9725), (0., 0.0431, 0.9412),
        (0.0039, 0.0627, 0.9137), (0.0039, 0.0863, 0.8863),
        (0.0039, 0.1059, 0.8588), (0.0039, 0.1294, 0.8275),
        (0.0078, 0.149, 0.8), (0.0078, 0.1686, 0.7725),
        (0.0078, 0.1922, 0.7451), (0.0078, 0.2118, 0.7137),
        (0.0118, 0.2353, 0.6863), (0.0118, 0.2549, 0.6588),
        (0.0118, 0.2784, 0.6275), (0.0118, 0.298, 0.6),
        (0.0157, 0.3176, 0.5725), (0.0157, 0.3412, 0.5451),
        (0.0157, 0.3608, 0.5137), (0.0157, 0.3843, 0.4863),
        (0.0196, 0.4039, 0.4588), (0.0196, 0.4235, 0.4314),
        (0.0196, 0.4471, 0.4), (0.0196, 0.4667, 0.3725),
        (0.0235, 0.4902, 0.3451), (0.0235, 0.5098, 0.3137),
        (0.0235, 0.5333, 0.2863), (0.0235, 0.5529, 0.2588),
        (0.0275, 0.5725, 0.2314), (0.0275, 0.5961, 0.2),
        (0.0275, 0.6157, 0.1725), (0.0275, 0.6392, 0.1451),
        (0.0314, 0.6588, 0.1176), (0.0314, 0.6824, 0.0863),
        (0.0314, 0.702, 0.0588), (0.0588, 0.7098, 0.0863),
        (0.0902, 0.7216, 0.1176), (0.1176, 0.7294, 0.1451),
        (0.149, 0.7373, 0.1725), (0.1765, 0.749, 0.2),
        (0.2078, 0.7569, 0.2314), (0.2353, 0.7647, 0.2588),
        (0.2667, 0.7725, 0.2863), (0.2941, 0.7843, 0.3137),
        (0.3255, 0.7922, 0.3451), (0.3529, 0.8, 0.3725),
        (0.3843, 0.8118, 0.4), (0.4118, 0.8196, 0.4314),
        (0.4431, 0.8275, 0.4588), (0.4706, 0.8392, 0.4863),
        (0.502, 0.8471, 0.5137), (0.5294, 0.8549, 0.5451),
        (0.5608, 0.8627, 0.5725), (0.5882, 0.8745, 0.6),
        (0.6196, 0.8824, 0.6275), (0.6471, 0.8902, 0.6588),
        (0.6784, 0.902, 0.6863), (0.7059, 0.9098, 0.7137),
        (0.7373, 0.9176, 0.7451), (0.7647, 0.9294, 0.7725),
        (0.7961, 0.9373, 0.8), (0.8235, 0.9451, 0.8275),
        (0.8549, 0.9529, 0.8588), (0.8824, 0.9647, 0.8863),
        (0.9137, 0.9725, 0.9137), (0.9412, 0.9804, 0.9412),
        (0.9725, 0.9922, 0.9725), (1., 1., 1.),
        (1., 1., 0.9686), (1., 1., 0.9412),
        (1., 1., 0.9098), (1., 1., 0.8784),
        (1., 1., 0.8471), (1., 1., 0.8196),
        (1., 1., 0.7882), (1., 1., 0.7569),
        (1., 1., 0.7255), (1., 1., 0.698),
        (1., 1., 0.6667), (1., 1., 0.6353),
        (1., 1., 0.6078), (1., 1., 0.5765),
        (1., 1., 0.5451), (1., 1., 0.5137),
        (1., 1., 0.4863), (1., 1., 0.4549),
        (1., 1., 0.4235), (1., 1., 0.3922),
        (1., 1., 0.3647), (1., 1., 0.3333),
        (1., 1., 0.302), (1., 1., 0.2745),
        (1., 1., 0.2431), (1., 1., 0.2118),
        (1., 1., 0.1804), (1., 1., 0.1529),
        (1., 1., 0.1216), (1., 1., 0.0902),
        (1., 1., 0.0588), (1., 1., 0.0314),
        (1., 1., 0.), (1., 0.9882, 0.),
        (1., 0.9804, 0.), (1., 0.9686, 0.),
        (1., 0.9569, 0.), (1., 0.9451, 0.),
        (1., 0.9373, 0.), (1., 0.9255, 0.),
        (1., 0.9137, 0.), (1., 0.902, 0.),
        (1., 0.8941, 0.), (1., 0.8824, 0.),
        (1., 0.8706, 0.), (1., 0.8627, 0.),
        (1., 0.851, 0.), (1., 0.8392, 0.),
        (1., 0.8275, 0.), (1., 0.8196, 0.),
        (1., 0.8078, 0.), (1., 0.7961, 0.),
        (1., 0.7843, 0.), (1., 0.7765, 0.),
        (1., 0.7647, 0.), (1., 0.7529, 0.),
        (1., 0.7451, 0.), (1., 0.7333, 0.),
        (1., 0.7216, 0.), (1., 0.7098, 0.),
        (1., 0.702, 0.), (1., 0.6902, 0.),
        (1., 0.6784, 0.), (1., 0.6667, 0.),
        (1., 0.6588, 0.), (1., 0.6471, 0.),
        (1., 0.6275, 0.), (1., 0.6078, 0.),
        (1., 0.5882, 0.), (1., 0.5686, 0.),
        (1., 0.549, 0.), (1., 0.5294, 0.),
        (1., 0.5098, 0.), (1., 0.4902, 0.),
        (1., 0.4706, 0.), (1., 0.451, 0.),
        (1., 0.4314, 0.), (1., 0.4118, 0.),
        (1., 0.3922, 0.), (1., 0.3725, 0.),
        (1., 0.3529, 0.), (1., 0.3333, 0.),
        (1., 0.3137, 0.), (1., 0.2941, 0.),
        (1., 0.2745, 0.), (1., 0.2549, 0.),
        (1., 0.2353, 0.), (1., 0.2157, 0.),
        (1., 0.1961, 0.), (1., 0.1765, 0.),
        (1., 0.1569, 0.), (1., 0.1373, 0.),
        (1., 0.1176, 0.), (1., 0.098, 0.),
        (1., 0.0784, 0.), (1., 0.0588, 0.),
        (1., 0.0392, 0.), (1., 0.0196, 0.),
        (1., 0., 0.)],
    'ncl:tbr_240_300': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0.0392, 0., 0.0588),
        (0.0745, 0., 0.1176), (0.1137, 0., 0.1765),
        (0.1529, 0., 0.2353), (0.1882, 0., 0.298),
        (0.2275, 0., 0.3569), (0.2667, 0., 0.4157),
        (0.302, 0., 0.4745), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3333, 0.0196, 0.5216), (0.3255, 0.0392, 0.5059),
        (0.3137, 0.0588, 0.4941), (0.3059, 0.0784, 0.4824),
        (0.298, 0.098, 0.4667), (0.2902, 0.1176, 0.4549),
        (0.2824, 0.1373, 0.4392), (0.2706, 0.1569, 0.4275),
        (0.2627, 0.1765, 0.4157), (0.2549, 0.1961, 0.4),
        (0.2471, 0.2157, 0.3882), (0.2392, 0.2353, 0.3765),
        (0.2275, 0.2549, 0.3608), (0.2196, 0.2745, 0.349),
        (0.2118, 0.2941, 0.3373), (0.2039, 0.3137, 0.3216),
        (0.1961, 0.3333, 0.3098), (0.1882, 0.3529, 0.298),
        (0.1765, 0.3686, 0.2824), (0.1686, 0.3882, 0.2706),
        (0.1608, 0.4078, 0.2549), (0.1529, 0.4275, 0.2431),
        (0.1451, 0.4471, 0.2314), (0.1333, 0.4667, 0.2157),
        (0.1255, 0.4863, 0.2039), (0.1176, 0.5059, 0.1922),
        (0.1098, 0.5255, 0.1765), (0.102, 0.5451, 0.1647),
        (0.0902, 0.5647, 0.1529), (0.0824, 0.5843, 0.1373),
        (0.0745, 0.6039, 0.1255), (0.0667, 0.6235, 0.1098),
        (0.0588, 0.6431, 0.098), (0.0471, 0.6627, 0.0863),
        (0.0392, 0.6824, 0.0706), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0588, 0.7098, 0.0588),
        (0.0863, 0.7176, 0.0549), (0.1137, 0.7255, 0.0549),
        (0.1373, 0.7333, 0.051), (0.1647, 0.7451, 0.051),
        (0.1922, 0.7529, 0.051), (0.2196, 0.7608, 0.0471),
        (0.2471, 0.7686, 0.0471), (0.2745, 0.7765, 0.0431),
        (0.302, 0.7843, 0.0431), (0.3255, 0.7922, 0.0392),
        (0.3529, 0.8, 0.0392), (0.3804, 0.8078, 0.0392),
        (0.4078, 0.8196, 0.0353), (0.4353, 0.8275, 0.0353),
        (0.4627, 0.8353, 0.0314), (0.4902, 0.8431, 0.0314),
        (0.5176, 0.851, 0.0314), (0.5412, 0.8588, 0.0275),
        (0.5686, 0.8667, 0.0275), (0.5961, 0.8745, 0.0235),
        (0.6235, 0.8824, 0.0235), (0.651, 0.8941, 0.0196),
        (0.6784, 0.902, 0.0196), (0.7059, 0.9098, 0.0196),
        (0.7294, 0.9176, 0.0157), (0.7569, 0.9255, 0.0157),
        (0.7843, 0.9333, 0.0118), (0.8118, 0.9412, 0.0118),
        (0.8392, 0.949, 0.0118), (0.8667, 0.9569, 0.0078),
        (0.8941, 0.9686, 0.0078), (0.9176, 0.9765, 0.0039),
        (0.9451, 0.9843, 0.0039), (0.9725, 0.9922, 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 0.9647, 0.),
        (1., 0.9294, 0.), (1., 0.898, 0.),
        (1., 0.8627, 0.), (1., 0.8275, 0.),
        (1., 0.7922, 0.), (1., 0.7569, 0.),
        (1., 0.7255, 0.), (1., 0.6902, 0.),
        (1., 0.6549, 0.), (1., 0.6196, 0.),
        (1., 0.5843, 0.), (1., 0.5529, 0.),
        (1., 0.5176, 0.), (1., 0.4824, 0.),
        (1., 0.4471, 0.), (1., 0.4157, 0.),
        (1., 0.3804, 0.), (1., 0.3451, 0.),
        (1., 0.3098, 0.), (1., 0.2745, 0.),
        (1., 0.2431, 0.), (1., 0.2078, 0.),
        (1., 0.1725, 0.), (1., 0.1373, 0.),
        (1., 0.102, 0.), (1., 0.0706, 0.),
        (1., 0.0353, 0.), (1., 0., 0.),
        (0.9961, 0.0118, 0.0392), (0.9961, 0.0275, 0.0824),
        (0.9922, 0.0392, 0.1216), (0.9882, 0.0549, 0.1647),
        (0.9882, 0.0667, 0.2039), (0.9843, 0.0824, 0.2471),
        (0.9804, 0.0941, 0.2863), (0.9804, 0.1098, 0.3255),
        (0.9765, 0.1216, 0.3686), (0.9725, 0.1373, 0.4078),
        (0.9725, 0.149, 0.451), (0.9686, 0.1647, 0.4902),
        (0.9686, 0.1765, 0.5333), (0.9647, 0.1922, 0.5725),
        (0.9608, 0.2039, 0.6157), (0.9608, 0.2196, 0.6549),
        (0.9569, 0.2314, 0.6941), (0.9529, 0.2471, 0.7373),
        (0.9529, 0.2588, 0.7765), (0.949, 0.2745, 0.8196),
        (0.9451, 0.2863, 0.8588), (0.9451, 0.302, 0.902),
        (0.9412, 0.3137, 0.9412), (0.9451, 0.3529, 0.9451),
        (0.949, 0.3961, 0.949), (0.9529, 0.4353, 0.9529),
        (0.9569, 0.4745, 0.9569), (0.9569, 0.5137, 0.9569),
        (0.9608, 0.5569, 0.9608), (0.9647, 0.5961, 0.9647),
        (0.9686, 0.6353, 0.9686), (0.9725, 0.6784, 0.9725),
        (0.9765, 0.7176, 0.9765), (0.9804, 0.7569, 0.9804),
        (0.9843, 0.8, 0.9843), (0.9843, 0.8392, 0.9843),
        (0.9882, 0.8784, 0.9882), (0.9922, 0.9176, 0.9922),
        (0.9961, 0.9608, 0.9961), (1., 1., 1.)],
    'ncl:tbr_stdev_0_30': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0196, 0., 0.0275), (0.0353, 0., 0.0549),
        (0.0549, 0., 0.0824), (0.0706, 0., 0.1137),
        (0.0902, 0., 0.1412), (0.1059, 0., 0.1686),
        (0.1255, 0., 0.1961), (0.1451, 0., 0.2235),
        (0.1608, 0., 0.251), (0.1804, 0., 0.2824),
        (0.1961, 0., 0.3098), (0.2157, 0., 0.3373),
        (0.2353, 0., 0.3647), (0.251, 0., 0.3922),
        (0.2706, 0., 0.4196), (0.2863, 0., 0.451),
        (0.3059, 0., 0.4784), (0.3216, 0., 0.5059),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3294, 0.0235, 0.5176), (0.3216, 0.0471, 0.502),
        (0.3098, 0.0667, 0.4863), (0.302, 0.0902, 0.4706),
        (0.2902, 0.1137, 0.4549), (0.2824, 0.1373, 0.4431),
        (0.2706, 0.1569, 0.4275), (0.2627, 0.1804, 0.4118),
        (0.251, 0.2039, 0.3961), (0.2431, 0.2275, 0.3804),
        (0.2314, 0.251, 0.3647), (0.2196, 0.2706, 0.349),
        (0.2118, 0.2941, 0.3333), (0.2, 0.3176, 0.3176),
        (0.1922, 0.3412, 0.302), (0.1804, 0.3608, 0.2902),
        (0.1725, 0.3843, 0.2745), (0.1608, 0.4078, 0.2588),
        (0.1529, 0.4314, 0.2431), (0.1412, 0.451, 0.2275),
        (0.1294, 0.4745, 0.2118), (0.1216, 0.498, 0.1961),
        (0.1098, 0.5216, 0.1804), (0.102, 0.5451, 0.1647),
        (0.0902, 0.5647, 0.149), (0.0824, 0.5882, 0.1373),
        (0.0706, 0.6118, 0.1216), (0.0627, 0.6353, 0.1059),
        (0.051, 0.6549, 0.0902), (0.0431, 0.6784, 0.0745),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0784, 0.7176, 0.0549),
        (0.1255, 0.7294, 0.0549), (0.1686, 0.7451, 0.051),
        (0.2157, 0.7569, 0.0471), (0.2627, 0.7725, 0.0431),
        (0.3098, 0.7882, 0.0431), (0.3529, 0.8, 0.0392),
        (0.4, 0.8157, 0.0353), (0.4471, 0.8314, 0.0353),
        (0.4941, 0.8431, 0.0314), (0.5373, 0.8588, 0.0275),
        (0.5843, 0.8706, 0.0235), (0.6314, 0.8863, 0.0235),
        (0.6784, 0.902, 0.0196), (0.7216, 0.9137, 0.0157),
        (0.7686, 0.9294, 0.0157), (0.8157, 0.9451, 0.0118),
        (0.8627, 0.9569, 0.0078), (0.9059, 0.9725, 0.0039),
        (0.9529, 0.9843, 0.0039), (1., 1., 0.),
        (1., 0.9647, 0.), (1., 0.9294, 0.),
        (1., 0.8941, 0.), (1., 0.8588, 0.),
        (1., 0.8196, 0.), (1., 0.7843, 0.),
        (1., 0.749, 0.), (1., 0.7137, 0.),
        (1., 0.6784, 0.), (1., 0.6431, 0.),
        (1., 0.6078, 0.), (1., 0.5725, 0.),
        (1., 0.5373, 0.), (1., 0.502, 0.),
        (1., 0.4627, 0.), (1., 0.4275, 0.),
        (1., 0.3922, 0.), (1., 0.3569, 0.),
        (1., 0.3216, 0.), (1., 0.2863, 0.),
        (1., 0.251, 0.), (1., 0.2157, 0.),
        (1., 0.1804, 0.), (1., 0.1412, 0.),
        (1., 0.1059, 0.), (1., 0.0706, 0.),
        (1., 0.0353, 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (0.9922, 0.051, 0.1569),
        (0.9804, 0.1059, 0.3137), (0.9725, 0.1569, 0.4706),
        (0.9608, 0.2078, 0.6275), (0.9529, 0.2627, 0.7843),
        (0.9412, 0.3137, 0.9412), (0.949, 0.4, 0.949),
        (0.9569, 0.4863, 0.9569), (0.9647, 0.5725, 0.9647),
        (0.9725, 0.6588, 0.9725), (0.9765, 0.7412, 0.9765),
        (0.9843, 0.8275, 0.9843), (0.9922, 0.9137, 0.9922),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.)],
    'ncl:tbr_var_0_500': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0353, 0., 0.0549), (0.0667, 0., 0.1059),
        (0.102, 0., 0.1608), (0.1373, 0., 0.2118),
        (0.1725, 0., 0.2667), (0.2039, 0., 0.3216),
        (0.2392, 0., 0.3725), (0.2745, 0., 0.4275),
        (0.3059, 0., 0.4784), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3255, 0.0353, 0.5098), (0.3098, 0.0667, 0.4863),
        (0.298, 0.102, 0.4667), (0.2824, 0.1333, 0.4431),
        (0.2667, 0.1686, 0.4196), (0.251, 0.2, 0.3961),
        (0.2392, 0.2353, 0.3765), (0.2235, 0.2667, 0.3529),
        (0.2078, 0.302, 0.3294), (0.1922, 0.3333, 0.3059),
        (0.1804, 0.3686, 0.2863), (0.1647, 0.4, 0.2627),
        (0.149, 0.4353, 0.2392), (0.1333, 0.4667, 0.2157),
        (0.1216, 0.502, 0.1961), (0.1059, 0.5333, 0.1725),
        (0.0902, 0.5686, 0.149), (0.0745, 0.6, 0.1255),
        (0.0627, 0.6353, 0.1059), (0.0471, 0.6667, 0.0824),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0824, 0.7176, 0.0549), (0.1333, 0.7333, 0.051),
        (0.1843, 0.749, 0.051), (0.2353, 0.7647, 0.0471),
        (0.2863, 0.7804, 0.0431), (0.3373, 0.7961, 0.0392),
        (0.3882, 0.8118, 0.0353), (0.4392, 0.8275, 0.0353),
        (0.4902, 0.8431, 0.0314), (0.5412, 0.8588, 0.0275),
        (0.5922, 0.8745, 0.0235), (0.6431, 0.8902, 0.0235),
        (0.6941, 0.9059, 0.0196), (0.7451, 0.9216, 0.0157),
        (0.7961, 0.9373, 0.0118), (0.8471, 0.9529, 0.0078),
        (0.898, 0.9686, 0.0078), (0.949, 0.9843, 0.0039),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 0.9765, 0.),
        (1., 0.9529, 0.), (1., 0.9294, 0.),
        (1., 0.9059, 0.), (1., 0.8824, 0.),
        (1., 0.8588, 0.), (1., 0.8353, 0.),
        (1., 0.8078, 0.), (1., 0.7843, 0.),
        (1., 0.7608, 0.), (1., 0.7373, 0.),
        (1., 0.7137, 0.), (1., 0.6902, 0.),
        (1., 0.6667, 0.), (1., 0.6431, 0.),
        (1., 0.6196, 0.), (1., 0.5961, 0.),
        (1., 0.5725, 0.), (1., 0.549, 0.),
        (1., 0.5255, 0.), (1., 0.502, 0.),
        (1., 0.4745, 0.), (1., 0.451, 0.),
        (1., 0.4275, 0.), (1., 0.4039, 0.),
        (1., 0.3804, 0.), (1., 0.3569, 0.),
        (1., 0.3333, 0.), (1., 0.3098, 0.),
        (1., 0.2863, 0.), (1., 0.2627, 0.),
        (1., 0.2392, 0.), (1., 0.2157, 0.),
        (1., 0.1922, 0.), (1., 0.1686, 0.),
        (1., 0.1412, 0.), (1., 0.1176, 0.),
        (1., 0.0941, 0.), (1., 0.0706, 0.),
        (1., 0.0471, 0.), (1., 0.0235, 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (0.9961, 0.0314, 0.0941),
        (0.9882, 0.0627, 0.1882), (0.9843, 0.0941, 0.2824),
        (0.9765, 0.1255, 0.3765), (0.9725, 0.1569, 0.4706),
        (0.9647, 0.1882, 0.5647), (0.9608, 0.2196, 0.6588),
        (0.9529, 0.251, 0.7529), (0.949, 0.2824, 0.8471),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9451, 0.3412, 0.9451), (0.9451, 0.3725, 0.9451),
        (0.949, 0.4, 0.949), (0.9529, 0.4275, 0.9529),
        (0.9529, 0.4549, 0.9529), (0.9569, 0.4863, 0.9569),
        (0.9569, 0.5137, 0.9569), (0.9608, 0.5412, 0.9608),
        (0.9647, 0.5725, 0.9647), (0.9647, 0.6, 0.9647),
        (0.9686, 0.6275, 0.9686), (0.9725, 0.6588, 0.9725),
        (0.9725, 0.6863, 0.9725), (0.9765, 0.7137, 0.9765),
        (0.9765, 0.7412, 0.9765), (0.9804, 0.7725, 0.9804),
        (0.9843, 0.8, 0.9843), (0.9843, 0.8275, 0.9843),
        (0.9882, 0.8588, 0.9882), (0.9922, 0.8863, 0.9922),
        (0.9922, 0.9137, 0.9922), (0.9961, 0.9412, 0.9961),
        (0.9961, 0.9725, 0.9961), (1., 1., 1.)],
    'ncl:tbravg1': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0.0667, 0., 0.1059),
        (0.1373, 0., 0.2118), (0.2039, 0., 0.3216),
        (0.2745, 0., 0.4275), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3255, 0.0392, 0.5059),
        (0.3059, 0.0784, 0.4824), (0.2902, 0.1176, 0.4549),
        (0.2706, 0.1569, 0.4275), (0.2549, 0.1961, 0.4),
        (0.2392, 0.2353, 0.3765), (0.2196, 0.2745, 0.349),
        (0.2039, 0.3137, 0.3216), (0.1882, 0.3529, 0.298),
        (0.1686, 0.3882, 0.2706), (0.1529, 0.4275, 0.2431),
        (0.1333, 0.4667, 0.2157), (0.1176, 0.5059, 0.1922),
        (0.102, 0.5451, 0.1647), (0.0824, 0.5843, 0.1373),
        (0.0667, 0.6235, 0.1098), (0.0471, 0.6627, 0.0863),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0863, 0.7176, 0.0549),
        (0.1373, 0.7333, 0.051), (0.1922, 0.7529, 0.051),
        (0.2471, 0.7686, 0.0471), (0.302, 0.7843, 0.0431),
        (0.3529, 0.8, 0.0392), (0.4078, 0.8196, 0.0353),
        (0.4627, 0.8353, 0.0314), (0.5176, 0.851, 0.0314),
        (0.5686, 0.8667, 0.0275), (0.6235, 0.8824, 0.0235),
        (0.6784, 0.902, 0.0196), (0.7294, 0.9176, 0.0157),
        (0.7843, 0.9333, 0.0118), (0.8392, 0.949, 0.0118),
        (0.8941, 0.9686, 0.0078), (0.9451, 0.9843, 0.0039),
        (1., 1., 0.), (1., 1., 0.),
        (1., 0.9294, 0.), (1., 0.8588, 0.),
        (1., 0.7843, 0.), (1., 0.7137, 0.),
        (1., 0.6431, 0.), (1., 0.5725, 0.),
        (1., 0.502, 0.), (1., 0.4275, 0.),
        (1., 0.3569, 0.), (1., 0.2863, 0.),
        (1., 0.2157, 0.), (1., 0.1412, 0.),
        (1., 0.0706, 0.), (1., 0., 0.),
        (0.9961, 0.0275, 0.0784), (0.9922, 0.051, 0.1569),
        (0.9843, 0.0784, 0.2353), (0.9804, 0.1059, 0.3137),
        (0.9765, 0.1294, 0.3922), (0.9725, 0.1569, 0.4706),
        (0.9647, 0.1843, 0.549), (0.9608, 0.2078, 0.6275),
        (0.9569, 0.2353, 0.7059), (0.9529, 0.2627, 0.7843),
        (0.9451, 0.2863, 0.8627), (0.9412, 0.3137, 0.9412),
        (0.949, 0.4, 0.949), (0.9569, 0.4863, 0.9569),
        (0.9647, 0.5725, 0.9647), (0.9725, 0.6588, 0.9725),
        (0.9765, 0.7412, 0.9765), (0.9843, 0.8275, 0.9843),
        (0.9922, 0.9137, 0.9922), (1., 1., 1.)],
    'ncl:tbrstd1': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0392, 0., 0.0588), (0.0745, 0., 0.1176),
        (0.1137, 0., 0.1765), (0.1529, 0., 0.2353),
        (0.1882, 0., 0.298), (0.2275, 0., 0.3569),
        (0.2667, 0., 0.4157), (0.302, 0., 0.4745),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3216, 0.0471, 0.502),
        (0.298, 0.0941, 0.4706), (0.2784, 0.1412, 0.4392),
        (0.2588, 0.1882, 0.4078), (0.2392, 0.2353, 0.3765),
        (0.2157, 0.2824, 0.3451), (0.1961, 0.3294, 0.3137),
        (0.1765, 0.3725, 0.2784), (0.1569, 0.4196, 0.2471),
        (0.1333, 0.4667, 0.2157), (0.1137, 0.5137, 0.1843),
        (0.0941, 0.5608, 0.1529), (0.0745, 0.6078, 0.1216),
        (0.051, 0.6549, 0.0902), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.1176, 0.7294, 0.0549), (0.2078, 0.7569, 0.0471),
        (0.2941, 0.7843, 0.0431), (0.3843, 0.8118, 0.0392),
        (0.4706, 0.8392, 0.0314), (0.5608, 0.8627, 0.0275),
        (0.6471, 0.8902, 0.0196), (0.7373, 0.9176, 0.0157),
        (0.8235, 0.9451, 0.0118), (0.9137, 0.9725, 0.0039),
        (1., 1., 0.), (1., 0.9294, 0.),
        (1., 0.8588, 0.), (1., 0.7843, 0.),
        (1., 0.7137, 0.), (1., 0.6431, 0.),
        (1., 0.5725, 0.), (1., 0.502, 0.),
        (1., 0.4275, 0.), (1., 0.3569, 0.),
        (1., 0.2863, 0.), (1., 0.2157, 0.),
        (1., 0.1412, 0.), (1., 0.0706, 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (0.9804, 0.1059, 0.3137),
        (0.9608, 0.2078, 0.6275), (0.9412, 0.3137, 0.9412),
        (0.9529, 0.451, 0.9529), (0.9647, 0.5882, 0.9647),
        (0.9765, 0.7255, 0.9765), (0.9882, 0.8627, 0.9882),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.)],
    'ncl:tbrvar1': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0., 0., 0.), (0., 0., 0.),
        (0.0667, 0., 0.1059), (0.1373, 0., 0.2118),
        (0.2039, 0., 0.3216), (0.2745, 0., 0.4275),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3412, 0., 0.5333), (0.3412, 0., 0.5333),
        (0.3137, 0.0627, 0.4902), (0.2863, 0.1294, 0.4471),
        (0.2549, 0.1922, 0.4039), (0.2275, 0.2549, 0.3608),
        (0.2, 0.3176, 0.3176), (0.1725, 0.3843, 0.2745),
        (0.1451, 0.4471, 0.2314), (0.1176, 0.5098, 0.1882),
        (0.0863, 0.5725, 0.1451), (0.0588, 0.6392, 0.102),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.0314, 0.702, 0.0588), (0.0314, 0.702, 0.0588),
        (0.1373, 0.7333, 0.051), (0.2471, 0.7686, 0.0471),
        (0.3529, 0.8, 0.0392), (0.4627, 0.8353, 0.0314),
        (0.5686, 0.8667, 0.0275), (0.6784, 0.902, 0.0196),
        (0.7843, 0.9333, 0.0118), (0.8941, 0.9686, 0.0078),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 0.9529, 0.),
        (1., 0.9059, 0.), (1., 0.8588, 0.),
        (1., 0.8078, 0.), (1., 0.7608, 0.),
        (1., 0.7137, 0.), (1., 0.6667, 0.),
        (1., 0.6196, 0.), (1., 0.5725, 0.),
        (1., 0.5255, 0.), (1., 0.4745, 0.),
        (1., 0.4275, 0.), (1., 0.3804, 0.),
        (1., 0.3333, 0.), (1., 0.2863, 0.),
        (1., 0.2392, 0.), (1., 0.1922, 0.),
        (1., 0.1412, 0.), (1., 0.0941, 0.),
        (1., 0.0471, 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (1., 0., 0.), (1., 0., 0.),
        (0.9843, 0.0784, 0.2353), (0.9725, 0.1569, 0.4706),
        (0.9569, 0.2353, 0.7059), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9412, 0.3137, 0.9412),
        (0.9412, 0.3137, 0.9412), (0.9451, 0.3725, 0.9451),
        (0.9529, 0.4275, 0.9529), (0.9569, 0.4863, 0.9569),
        (0.9608, 0.5412, 0.9608), (0.9647, 0.6, 0.9647),
        (0.9725, 0.6588, 0.9725), (0.9765, 0.7137, 0.9765),
        (0.9804, 0.7725, 0.9804), (0.9843, 0.8275, 0.9843),
        (0.9922, 0.8863, 0.9922), (0.9961, 0.9412, 0.9961),
        (1., 1., 1.)],
    'ncl:temp1': [
        (1., 1., 1.), (0., 0., 0.),
        (0.7, 0.7, 0.7), (0.65, 0.65, 0.7),
        (0.61, 0.6, 0.7), (0.55, 0.55, 0.7),
        (0.56, 0.5, 0.7), (0.45, 0.45, 0.7),
        (0.42, 0.4, 0.7), (0.35, 0.35, 0.7),
        (0.3, 0.3, 0.7), (0.25, 0.25, 0.7),
        (0.2, 0.2, 0.7), (0.15, 0.15, 0.7),
        (0.1, 0.1, 0.7), (0.05, 0.05, 0.7),
        (0., 0., 0.7), (0., 0.05, 0.7),
        (0., 0.1, 0.7), (0., 0.15, 0.7),
        (0., 0.2, 0.7), (0., 0.25, 0.7),
        (0., 0.3, 0.7), (0., 0.35, 0.7),
        (0., 0.4, 0.7), (0., 0.45, 0.6),
        (0., 0.5, 0.5), (0., 0.55, 0.4),
        (0., 0.6, 0.3), (0., 0.65, 0.2),
        (0., 0.7, 0.1), (0., 0.725, 0.),
        (0., 0.69, 0.), (0.03, 0.685, 0.),
        (0.06, 0.68, 0.), (0.1, 0.575, 0.),
        (0.13, 0.57, 0.), (0.16, 0.565, 0.),
        (0.55, 0.55, 0.), (0.555, 0.545, 0.),
        (0.56, 0.53, 0.), (0.565, 0.485, 0.),
        (0.57, 0.42, 0.), (0.675, 0.375, 0.),
        (0.68, 0.33, 0.), (0.69, 0.3, 0.),
        (0.7, 0.285, 0.), (0.7, 0.27, 0.),
        (0.7, 0.26, 0.), (0.7, 0.24, 0.),
        (0.7, 0.18, 0.), (0.7, 0.13, 0.),
        (0.7, 0.12, 0.), (0.7, 0.1, 0.),
        (0.7, 0.09, 0.), (0.75, 0.09, 0.),
        (0.8, 0.09, 0.), (0.83, 0.07, 0.),
        (0.87, 0.05, 0.), (0.9, 0.03, 0.),
        (0.95, 0.01, 0.), (0.99, 0., 0.), (1., 0., 0.)],
    'ncl:thelix': [
        (1., 1., 1.), (0., 0., 0.),
        (0.0039, 0.0039, 0.), (0.0078, 0.0039, 0.),
        (0.0118, 0.0078, 0.), (0.0157, 0.0078, 0.),
        (0.0196, 0.0118, 0.), (0.0235, 0.0157, 0.),
        (0.0275, 0.0157, 0.), (0.0275, 0.0196, 0.),
        (0.0314, 0.0235, 0.), (0.0353, 0.0235, 0.),
        (0.0392, 0.0275, 0.), (0.0392, 0.0314, 0.),
        (0.0431, 0.0353, 0.), (0.0471, 0.0392, 0.),
        (0.0471, 0.0431, 0.), (0.051, 0.0431, 0.),
        (0.0549, 0.0471, 0.), (0.0549, 0.051, 0.),
        (0.0588, 0.0549, 0.), (0.0588, 0.0588, 0.),
        (0.0627, 0.0627, 0.), (0.0627, 0.0667, 0.),
        (0.0667, 0.0706, 0.), (0.0667, 0.0745, 0.),
        (0.0706, 0.0784, 0.), (0.0706, 0.0824, 0.),
        (0.0706, 0.0863, 0.), (0.0745, 0.0902, 0.),
        (0.0745, 0.098, 0.), (0.0745, 0.102, 0.),
        (0.0784, 0.1059, 0.), (0.0784, 0.1098, 0.),
        (0.0784, 0.1137, 0.), (0.0784, 0.1176, 0.),
        (0.0784, 0.1216, 0.0039), (0.0824, 0.1294, 0.0039),
        (0.0824, 0.1333, 0.0039), (0.0824, 0.1373, 0.0039),
        (0.0824, 0.1412, 0.0078), (0.0824, 0.1451, 0.0078),
        (0.0824, 0.1529, 0.0078), (0.0824, 0.1569, 0.0078),
        (0.0824, 0.1608, 0.0118), (0.0824, 0.1647, 0.0118),
        (0.0824, 0.1686, 0.0157), (0.0824, 0.1725, 0.0157),
        (0.0784, 0.1804, 0.0157), (0.0784, 0.1843, 0.0196),
        (0.0784, 0.1882, 0.0196), (0.0784, 0.1922, 0.0235),
        (0.0784, 0.1961, 0.0275), (0.0745, 0.2039, 0.0275),
        (0.0745, 0.2078, 0.0314), (0.0745, 0.2118, 0.0314),
        (0.0706, 0.2157, 0.0353), (0.0706, 0.2196, 0.0392),
        (0.0706, 0.2235, 0.0392), (0.0667, 0.2275, 0.0431),
        (0.0667, 0.2314, 0.0471), (0.0667, 0.2353, 0.051),
        (0.0627, 0.2392, 0.0549), (0.0627, 0.2431, 0.0588),
        (0.0627, 0.2471, 0.0588), (0.0588, 0.251, 0.0627),
        (0.0588, 0.2549, 0.0667), (0.0549, 0.2588, 0.0706),
        (0.0549, 0.2627, 0.0745), (0.051, 0.2667, 0.0824),
        (0.051, 0.2706, 0.0863), (0.0471, 0.2745, 0.0902),
        (0.0471, 0.2784, 0.0941), (0.0431, 0.2824, 0.098),
        (0.0431, 0.2824, 0.102), (0.0392, 0.2863, 0.1098),
        (0.0392, 0.2902, 0.1137), (0.0353, 0.2941, 0.1176),
        (0.0353, 0.2941, 0.1255), (0.0314, 0.298, 0.1294),
        (0.0314, 0.298, 0.1333), (0.0275, 0.302, 0.1412),
        (0.0275, 0.3059, 0.1451), (0.0235, 0.3059, 0.1529),
        (0.0235, 0.3098, 0.1569), (0.0196, 0.3098, 0.1647),
        (0.0196, 0.3098, 0.1686), (0.0157, 0.3137, 0.1765),
        (0.0157, 0.3137, 0.1804), (0.0157, 0.3137, 0.1882),
        (0.0118, 0.3176, 0.1961), (0.0118, 0.3176, 0.2),
        (0.0078, 0.3176, 0.2078), (0.0078, 0.3176, 0.2157),
        (0.0078, 0.3176, 0.2196), (0.0039, 0.3176, 0.2275),
        (0.0039, 0.3176, 0.2353), (0.0039, 0.3176, 0.2431),
        (0.0039, 0.3176, 0.2471), (0., 0.3176, 0.2549),
        (0., 0.3176, 0.2627), (0., 0.3176, 0.2706),
        (0., 0.3176, 0.2784), (0., 0.3176, 0.2863),
        (0., 0.3137, 0.2902), (0., 0.3137, 0.298),
        (0., 0.3137, 0.3059), (0., 0.3098, 0.3137),
        (0., 0.3098, 0.3216), (0., 0.3098, 0.3294),
        (0., 0.3059, 0.3373), (0., 0.302, 0.3451),
        (0., 0.302, 0.3529), (0., 0.298, 0.3569),
        (0., 0.298, 0.3647), (0.0039, 0.2941, 0.3725),
        (0.0039, 0.2902, 0.3804), (0.0039, 0.2902, 0.3882),
        (0.0078, 0.2863, 0.3961), (0.0078, 0.2824, 0.4039),
        (0.0118, 0.2784, 0.4118), (0.0118, 0.2745, 0.4157),
        (0.0157, 0.2706, 0.4235), (0.0157, 0.2706, 0.4314),
        (0.0196, 0.2667, 0.4392), (0.0235, 0.2627, 0.4471),
        (0.0235, 0.2588, 0.451), (0.0275, 0.2549, 0.4588),
        (0.0314, 0.251, 0.4667), (0.0353, 0.2431, 0.4745),
        (0.0392, 0.2392, 0.4784), (0.0431, 0.2353, 0.4863),
        (0.0471, 0.2314, 0.4941), (0.051, 0.2275, 0.498),
        (0.0549, 0.2235, 0.5059), (0.0588, 0.2157, 0.5098),
        (0.0627, 0.2118, 0.5176), (0.0706, 0.2078, 0.5216),
        (0.0745, 0.2039, 0.5294), (0.0784, 0.1961, 0.5333),
        (0.0863, 0.1922, 0.5412), (0.0902, 0.1882, 0.5451),
        (0.098, 0.1843, 0.549), (0.102, 0.1765, 0.5569),
        (0.1098, 0.1725, 0.5608), (0.1137, 0.1686, 0.5647),
        (0.1216, 0.1608, 0.5686), (0.1294, 0.1569, 0.5725),
        (0.1333, 0.1529, 0.5765), (0.1412, 0.1451, 0.5804),
        (0.149, 0.1412, 0.5843), (0.1569, 0.1373, 0.5882),
        (0.1647, 0.1294, 0.5922), (0.1725, 0.1255, 0.5961),
        (0.1804, 0.1216, 0.6), (0.1882, 0.1137, 0.6039),
        (0.1961, 0.1098, 0.6039), (0.2039, 0.1059, 0.6078),
        (0.2118, 0.098, 0.6118), (0.2235, 0.0941, 0.6118),
        (0.2314, 0.0902, 0.6157), (0.2392, 0.0863, 0.6157),
        (0.2471, 0.0784, 0.6196), (0.2588, 0.0745, 0.6196),
        (0.2667, 0.0706, 0.6196), (0.2784, 0.0667, 0.6196),
        (0.2863, 0.0627, 0.6235), (0.2941, 0.0588, 0.6235),
        (0.3059, 0.0549, 0.6235), (0.3176, 0.051, 0.6235),
        (0.3255, 0.0471, 0.6235), (0.3373, 0.0431, 0.6235),
        (0.3451, 0.0392, 0.6235), (0.3569, 0.0353, 0.6196),
        (0.3686, 0.0314, 0.6196), (0.3765, 0.0275, 0.6196),
        (0.3882, 0.0235, 0.6157), (0.4, 0.0196, 0.6157),
        (0.4078, 0.0196, 0.6118), (0.4196, 0.0157, 0.6118),
        (0.4314, 0.0118, 0.6078), (0.4431, 0.0118, 0.6078),
        (0.451, 0.0078, 0.6039), (0.4627, 0.0078, 0.6),
        (0.4745, 0.0039, 0.5961), (0.4863, 0.0039, 0.5922),
        (0.498, 0.0039, 0.5882), (0.5059, 0., 0.5843),
        (0.5176, 0., 0.5804), (0.5294, 0., 0.5765),
        (0.5412, 0., 0.5725), (0.5529, 0., 0.5686),
        (0.5608, 0., 0.5608), (0.5725, 0., 0.5569),
        (0.5843, 0., 0.5529), (0.5961, 0., 0.5451),
        (0.6078, 0., 0.5412), (0.6157, 0., 0.5333),
        (0.6275, 0.0039, 0.5294), (0.6392, 0.0039, 0.5216),
        (0.6471, 0.0039, 0.5176), (0.6588, 0.0078, 0.5098),
        (0.6706, 0.0118, 0.502), (0.6784, 0.0118, 0.4941),
        (0.6902, 0.0157, 0.4902), (0.702, 0.0196, 0.4824),
        (0.7098, 0.0235, 0.4745), (0.7216, 0.0235, 0.4667),
        (0.7294, 0.0275, 0.4588), (0.7373, 0.0314, 0.451),
        (0.749, 0.0392, 0.4431), (0.7569, 0.0431, 0.4353),
        (0.7686, 0.0471, 0.4275), (0.7765, 0.051, 0.4196),
        (0.7843, 0.0588, 0.4118), (0.7922, 0.0627, 0.4039),
        (0.8, 0.0706, 0.3922), (0.8078, 0.0745, 0.3843),
        (0.8196, 0.0824, 0.3765), (0.8275, 0.0902, 0.3686),
        (0.8314, 0.0941, 0.3608), (0.8392, 0.102, 0.349),
        (0.8471, 0.1098, 0.3412), (0.8549, 0.1176, 0.3333),
        (0.8627, 0.1255, 0.3255), (0.8667, 0.1333, 0.3137),
        (0.8745, 0.1451, 0.3059), (0.8784, 0.1529, 0.298),
        (0.8863, 0.1608, 0.2863), (0.8902, 0.1725, 0.2784),
        (0.898, 0.1804, 0.2706), (0.902, 0.1882, 0.2588),
        (0.9059, 0.2, 0.251), (0.9098, 0.2118, 0.2431),
        (0.9137, 0.2196, 0.2353), (0.9176, 0.2314, 0.2235),
        (0.9216, 0.2431, 0.2157), (0.9255, 0.2549, 0.2078),
        (0.9294, 0.2667, 0.2), (0.9333, 0.2784, 0.1922),
        (0.9333, 0.2902, 0.1804), (0.9373, 0.302, 0.1725),
        (0.9412, 0.3137, 0.1647), (0.9412, 0.3255, 0.1569),
        (0.9412, 0.3373, 0.149), (0.9451, 0.349, 0.1412),
        (0.9451, 0.3647, 0.1333), (0.9451, 0.3765, 0.1255),
        (0.9451, 0.3882, 0.1176), (0.9451, 0.4039, 0.1098),
        (0.9451, 0.4157, 0.1059), (0.9451, 0.4314, 0.098),
        (0.9412, 0.4431, 0.0902), (0.9412, 0.4588, 0.0824),
        (0.9373, 0.4706, 0.0784), (0.9373, 0.4863, 0.0706)],
    'ncl:uniform': [
        (1., 1., 1.), (0., 0., 0.),
        (0.75, 1., 1.), (0.5, 1., 1.),
        (0.25, 1., 1.), (0., 1., 1.),
        (1., 0.8333, 1.), (0.75, 0.8333, 1.),
        (0.5, 0.8333, 1.), (0.25, 0.8333, 1.),
        (0., 0.8333, 1.), (1., 0.6667, 1.),
        (0.75, 0.6667, 1.), (0.5, 0.6667, 1.),
        (0.25, 0.6667, 1.), (0., 0.6667, 1.),
        (1., 0.5, 1.), (0.75, 0.5, 1.),
        (0.5, 0.5, 1.), (0.25, 0.5, 1.),
        (0., 0.5, 1.), (1., 0.3333, 1.),
        (0.75, 0.3333, 1.), (0.5, 0.3333, 1.),
        (0.25, 0.3333, 1.), (0., 0.3333, 1.),
        (1., 0.1667, 1.), (0.75, 0.1667, 1.),
        (0.5, 0.1667, 1.), (0.25, 0.1667, 1.),
        (0., 0.1667, 1.), (1., 0., 1.),
        (0.75, 0., 1.), (0.5, 0., 1.),
        (0.25, 0., 1.), (0., 0., 1.),
        (1., 1., 0.25), (0.75, 1., 0.25),
        (0.5, 1., 0.25), (0.25, 1., 0.25),
        (0., 1., 0.25), (1., 0.8333, 0.25),
        (0.75, 0.8333, 0.25), (0.5, 0.8333, 0.25),
        (0.25, 0.8333, 0.25), (0., 0.8333, 0.25),
        (1., 0.6667, 0.25), (0.75, 0.6667, 0.25),
        (0.5, 0.6667, 0.25), (0.25, 0.6667, 0.25),
        (0., 0.6667, 0.25), (1., 0.5, 0.25),
        (0.75, 0.5, 0.25), (0.5, 0.5, 0.25),
        (0.25, 0.5, 0.25), (0., 0.5, 0.25),
        (1., 0.3333, 0.25), (0.75, 0.3333, 0.25),
        (0.5, 0.3333, 0.25), (0.25, 0.3333, 0.25),
        (0., 0.3333, 0.25), (1., 0.1667, 0.25),
        (0.75, 0.1667, 0.25), (0.5, 0.1667, 0.25),
        (0.25, 0.1667, 0.25), (0., 0.1667, 0.25),
        (1., 0., 0.25), (0.75, 0., 0.25),
        (0.5, 0., 0.25), (0.25, 0., 0.25),
        (0., 0., 0.25), (1., 1., 0.5),
        (0.75, 1., 0.5), (0.5, 1., 0.5),
        (0.25, 1., 0.5), (0., 1., 0.5),
        (1., 0.8333, 0.5), (0.75, 0.8333, 0.5),
        (0.5, 0.8333, 0.5), (0.25, 0.8333, 0.5),
        (0., 0.8333, 0.5), (1., 0.6667, 0.5),
        (0.75, 0.6667, 0.5), (0.5, 0.6667, 0.5),
        (0.25, 0.6667, 0.5), (0., 0.6667, 0.5),
        (1., 0.5, 0.5), (0.75, 0.5, 0.5),
        (0.5, 0.5, 0.5), (0.25, 0.5, 0.5),
        (0., 0.5, 0.5), (1., 0.3333, 0.5),
        (0.75, 0.3333, 0.5), (0.5, 0.3333, 0.5),
        (0.25, 0.3333, 0.5), (0., 0.3333, 0.5),
        (1., 0.1667, 0.5), (0.75, 0.1667, 0.5),
        (0.5, 0.1667, 0.5), (0.25, 0.1667, 0.5),
        (0., 0.1667, 0.5), (1., 0., 0.5),
        (0.75, 0., 0.5), (0.5, 0., 0.5),
        (0.25, 0., 0.5), (0., 0., 0.5),
        (1., 1., 0.75), (0.75, 1., 0.75),
        (0.5, 1., 0.75), (0.25, 1., 0.75),
        (0., 1., 0.75), (1., 0.8333, 0.75),
        (0.75, 0.8333, 0.75), (0.5, 0.8333, 0.75),
        (0.25, 0.8333, 0.75), (0., 0.8333, 0.75),
        (1., 0.6667, 0.75), (0.75, 0.6667, 0.75),
        (0.5, 0.6667, 0.75), (0.25, 0.6667, 0.75),
        (0., 0.6667, 0.75), (1., 0.5, 0.75),
        (0.75, 0.5, 0.75), (0.5, 0.5, 0.75),
        (0.25, 0.5, 0.75), (0., 0.5, 0.75),
        (1., 0.3333, 0.75), (0.75, 0.3333, 0.75),
        (0.5, 0.3333, 0.75), (0.25, 0.3333, 0.75),
        (0., 0.3333, 0.75), (1., 0.1667, 0.75),
        (0.75, 0.1667, 0.75), (0.5, 0.1667, 0.75),
        (0.25, 0.1667, 0.75), (0., 0.1667, 0.75),
        (1., 0., 0.75), (0.75, 0., 0.75),
        (0.5, 0., 0.75), (0.25, 0., 0.75),
        (0., 0., 0.75), (1., 1., 0.),
        (0.75, 1., 0.), (0.5, 1., 0.),
        (0.25, 1., 0.), (0., 1., 0.),
        (1., 0.8333, 0.), (0.75, 0.8333, 0.),
        (0.5, 0.8333, 0.), (0.25, 0.8333, 0.),
        (0., 0.8333, 0.), (1., 0.6667, 0.),
        (0.75, 0.6667, 0.), (0.5, 0.6667, 0.),
        (0.25, 0.6667, 0.), (0., 0.6667, 0.),
        (1., 0.5, 0.), (0.75, 0.5, 0.),
        (0.5, 0.5, 0.), (0.25, 0.5, 0.),
        (0., 0.5, 0.), (1., 0.3333, 0.),
        (0.75, 0.3333, 0.), (0.5, 0.3333, 0.),
        (0.25, 0.3333, 0.), (0., 0.3333, 0.),
        (1., 0.1667, 0.), (0.75, 0.1667, 0.),
        (0.5, 0.1667, 0.), (0.25, 0.1667, 0.),
        (0., 0.1667, 0.), (1., 0., 0.),
        (0.75, 0., 0.), (0.5, 0., 0.),
        (0.25, 0., 0.)],
    'ncl:viblgrwhyeorre': [
        (1., 1., 1.), (0., 0., 0.),
        (0.3412, 0., 0.5333), (0.3176, 0., 0.5647),
        (0.2941, 0., 0.5961), (0.2745, 0., 0.6275),
        (0.251, 0., 0.6588), (0.2275, 0., 0.6902),
        (0.2039, 0., 0.7216), (0.1804, 0., 0.7529),
        (0.1608, 0., 0.7804), (0.1373, 0., 0.8118),
        (0.1137, 0., 0.8431), (0.0902, 0., 0.8745),
        (0.0667, 0., 0.9059), (0.0471, 0., 0.9373),
        (0.0235, 0., 0.9686), (0., 0., 1.),
        (0.0039, 0.0431, 0.9412), (0.0039, 0.0863, 0.8824),
        (0.0078, 0.1333, 0.8235), (0.0078, 0.1765, 0.7647),
        (0.0118, 0.2196, 0.7059), (0.0118, 0.2627, 0.6471),
        (0.0157, 0.3059, 0.5882), (0.0157, 0.3529, 0.5294),
        (0.0196, 0.3961, 0.4706), (0.0196, 0.4392, 0.4118),
        (0.0235, 0.4824, 0.3529), (0.0235, 0.5255, 0.2941),
        (0.0275, 0.5686, 0.2353), (0.0275, 0.6157, 0.1765),
        (0.0314, 0.6588, 0.1176), (0.0314, 0.702, 0.0588),
        (0.0941, 0.7216, 0.1216), (0.1608, 0.7412, 0.1843),
        (0.2235, 0.7608, 0.2471), (0.2902, 0.7804, 0.3098),
        (0.3529, 0.8, 0.3725), (0.4196, 0.8196, 0.4353),
        (0.4824, 0.8392, 0.498), (0.549, 0.8627, 0.5608),
        (0.6118, 0.8824, 0.6235), (0.6784, 0.902, 0.6863),
        (0.7412, 0.9216, 0.749), (0.8078, 0.9412, 0.8118),
        (0.8706, 0.9608, 0.8745), (0.9373, 0.9804, 0.9373),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 1.),
        (1., 1., 1.), (1., 1., 0.9333),
        (1., 1., 0.8667), (1., 1., 0.8),
        (1., 1., 0.7333), (1., 1., 0.6667),
        (1., 1., 0.6), (1., 1., 0.5333),
        (1., 1., 0.4667), (1., 1., 0.4),
        (1., 1., 0.3333), (1., 1., 0.2667),
        (1., 1., 0.2), (1., 1., 0.1333),
        (1., 1., 0.0667), (1., 1., 0.),
        (1., 0.9765, 0.), (1., 0.9569, 0.),
        (1., 0.9333, 0.), (1., 0.9137, 0.),
        (1., 0.8902, 0.), (1., 0.8667, 0.),
        (1., 0.8471, 0.), (1., 0.8235, 0.),
        (1., 0.8, 0.), (1., 0.7804, 0.),
        (1., 0.7569, 0.), (1., 0.7373, 0.),
        (1., 0.7137, 0.), (1., 0.6902, 0.),
        (1., 0.6706, 0.), (1., 0.6471, 0.),
        (1., 0.6039, 0.), (1., 0.5608, 0.),
        (1., 0.5176, 0.), (1., 0.4745, 0.),
        (1., 0.4314, 0.), (1., 0.3882, 0.),
        (1., 0.3451, 0.), (1., 0.302, 0.),
        (1., 0.2588, 0.), (1., 0.2157, 0.),
        (1., 0.1725, 0.), (1., 0.1294, 0.),
        (1., 0.0863, 0.), (1., 0.0431, 0.),
        (1., 0., 0.)],
    'ncl:wh_bl_gr_ye_re': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9804, 0.9804, 1.),
        (0.9608, 0.9608, 1.), (0.9412, 0.9412, 1.),
        (0.9216, 0.9216, 1.), (0.902, 0.902, 1.),
        (0.8784, 0.8784, 1.), (0.8588, 0.8588, 1.),
        (0.8392, 0.8392, 1.), (0.8196, 0.8196, 1.),
        (0.8, 0.8, 1.), (0.7804, 0.7804, 1.),
        (0.7608, 0.7608, 1.), (0.7412, 0.7412, 1.),
        (0.7216, 0.7216, 1.), (0.702, 0.702, 1.),
        (0.6784, 0.6784, 1.), (0.6588, 0.6588, 1.),
        (0.6392, 0.6392, 1.), (0.6196, 0.6196, 1.),
        (0.6, 0.6, 1.), (0.5804, 0.5804, 1.),
        (0.5608, 0.5608, 1.), (0.5412, 0.5412, 1.),
        (0.5216, 0.5216, 1.), (0.502, 0.502, 1.),
        (0.4784, 0.4784, 1.), (0.4588, 0.4588, 1.),
        (0.4392, 0.4392, 1.), (0.4196, 0.4196, 1.),
        (0.4, 0.4, 1.), (0.3804, 0.3804, 1.),
        (0.3608, 0.3608, 1.), (0.3412, 0.3412, 1.),
        (0.3216, 0.3216, 1.), (0.302, 0.302, 1.),
        (0.2784, 0.2784, 1.), (0.2588, 0.2588, 1.),
        (0.2392, 0.2392, 1.), (0.2196, 0.2196, 1.),
        (0.2, 0.2, 1.), (0.1804, 0.1804, 1.),
        (0.1608, 0.1608, 1.), (0.1412, 0.1412, 1.),
        (0.1216, 0.1216, 1.), (0.102, 0.102, 1.),
        (0.0784, 0.0784, 1.), (0.0588, 0.0588, 1.),
        (0.0392, 0.0392, 1.), (0.0196, 0.0196, 1.),
        (0., 0., 1.), (0., 0.0157, 0.9804),
        (0., 0.0275, 0.9608), (0., 0.0431, 0.9412),
        (0.0039, 0.0588, 0.9216), (0.0039, 0.0706, 0.9059),
        (0.0039, 0.0863, 0.8863), (0.0039, 0.102, 0.8667),
        (0.0039, 0.1137, 0.8471), (0.0039, 0.1294, 0.8275),
        (0.0078, 0.1451, 0.8078), (0.0078, 0.1569, 0.7882),
        (0.0078, 0.1725, 0.7686), (0.0078, 0.1843, 0.749),
        (0.0078, 0.2, 0.7294), (0.0078, 0.2157, 0.7137),
        (0.0118, 0.2275, 0.6941), (0.0118, 0.2431, 0.6745),
        (0.0118, 0.2588, 0.6549), (0.0118, 0.2706, 0.6353),
        (0.0118, 0.2863, 0.6157), (0.0118, 0.302, 0.5961),
        (0.0157, 0.3137, 0.5765), (0.0157, 0.3294, 0.5569),
        (0.0157, 0.3451, 0.5373), (0.0157, 0.3569, 0.5216),
        (0.0157, 0.3725, 0.502), (0.0157, 0.3882, 0.4824),
        (0.0196, 0.4, 0.4627), (0.0196, 0.4157, 0.4431),
        (0.0196, 0.4314, 0.4235), (0.0196, 0.4431, 0.4039),
        (0.0196, 0.4588, 0.3843), (0.0196, 0.4745, 0.3647),
        (0.0235, 0.4863, 0.3451), (0.0235, 0.502, 0.3294),
        (0.0235, 0.5176, 0.3098), (0.0235, 0.5294, 0.2902),
        (0.0235, 0.5451, 0.2706), (0.0235, 0.5569, 0.251),
        (0.0275, 0.5725, 0.2314), (0.0275, 0.5882, 0.2118),
        (0.0275, 0.6, 0.1922), (0.0275, 0.6157, 0.1725),
        (0.0275, 0.6314, 0.1529), (0.0275, 0.6431, 0.1373),
        (0.0314, 0.6588, 0.1176), (0.0314, 0.6745, 0.098),
        (0.0314, 0.6863, 0.0784), (0.0314, 0.702, 0.0588),
        (0.051, 0.7098, 0.0588), (0.0706, 0.7137, 0.0549),
        (0.0902, 0.7216, 0.0549), (0.1098, 0.7255, 0.0549),
        (0.1294, 0.7333, 0.0549), (0.149, 0.7373, 0.051),
        (0.1686, 0.7451, 0.051), (0.1882, 0.749, 0.051),
        (0.2039, 0.7569, 0.0471), (0.2235, 0.7608, 0.0471),
        (0.2431, 0.7686, 0.0471), (0.2627, 0.7725, 0.0431),
        (0.2824, 0.7804, 0.0431), (0.302, 0.7843, 0.0431),
        (0.3216, 0.7922, 0.0431), (0.3412, 0.7961, 0.0392),
        (0.3608, 0.8039, 0.0392), (0.3804, 0.8078, 0.0392),
        (0.4, 0.8157, 0.0353), (0.4196, 0.8196, 0.0353),
        (0.4392, 0.8275, 0.0353), (0.4588, 0.8314, 0.0314),
        (0.4784, 0.8392, 0.0314), (0.498, 0.8431, 0.0314),
        (0.5176, 0.851, 0.0314), (0.5333, 0.8588, 0.0275),
        (0.5529, 0.8627, 0.0275), (0.5725, 0.8706, 0.0275),
        (0.5922, 0.8745, 0.0235), (0.6118, 0.8824, 0.0235),
        (0.6314, 0.8863, 0.0235), (0.651, 0.8941, 0.0196),
        (0.6706, 0.898, 0.0196), (0.6902, 0.9059, 0.0196),
        (0.7098, 0.9098, 0.0196), (0.7294, 0.9176, 0.0157),
        (0.749, 0.9216, 0.0157), (0.7686, 0.9294, 0.0157),
        (0.7882, 0.9333, 0.0118), (0.8078, 0.9412, 0.0118),
        (0.8275, 0.9451, 0.0118), (0.8431, 0.9529, 0.0078),
        (0.8627, 0.9569, 0.0078), (0.8824, 0.9647, 0.0078),
        (0.902, 0.9686, 0.0078), (0.9216, 0.9765, 0.0039),
        (0.9412, 0.9804, 0.0039), (0.9608, 0.9882, 0.0039),
        (0.9804, 0.9922, 0.), (1., 1., 0.),
        (1., 0.9804, 0.), (1., 0.9608, 0.),
        (1., 0.9373, 0.), (1., 0.9176, 0.),
        (1., 0.898, 0.), (1., 0.8784, 0.),
        (1., 0.8588, 0.), (1., 0.8353, 0.),
        (1., 0.8157, 0.), (1., 0.7961, 0.),
        (1., 0.7765, 0.), (1., 0.7569, 0.),
        (1., 0.7333, 0.), (1., 0.7137, 0.),
        (1., 0.6941, 0.), (1., 0.6745, 0.),
        (1., 0.6549, 0.), (1., 0.6314, 0.),
        (1., 0.6118, 0.), (1., 0.5922, 0.),
        (1., 0.5725, 0.), (1., 0.5529, 0.),
        (1., 0.5294, 0.), (1., 0.5098, 0.),
        (1., 0.4902, 0.), (1., 0.4706, 0.),
        (1., 0.4471, 0.), (1., 0.4275, 0.),
        (1., 0.4078, 0.), (1., 0.3882, 0.),
        (1., 0.3686, 0.), (1., 0.3451, 0.),
        (1., 0.3255, 0.), (1., 0.3059, 0.),
        (1., 0.2863, 0.), (1., 0.2667, 0.),
        (1., 0.2431, 0.), (1., 0.2235, 0.),
        (1., 0.2039, 0.), (1., 0.1843, 0.),
        (1., 0.1647, 0.), (1., 0.1412, 0.),
        (1., 0.1216, 0.), (1., 0.102, 0.),
        (1., 0.0824, 0.), (1., 0.0627, 0.),
        (1., 0.0392, 0.), (1., 0.0196, 0.), (1., 0., 0.)],
    'ncl:whblgryere': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9608, 0.9608, 1.),
        (0.9216, 0.9216, 1.), (0.8784, 0.8784, 1.),
        (0.8392, 0.8392, 1.), (0.8, 0.8, 1.),
        (0.7608, 0.7608, 1.), (0.7216, 0.7216, 1.),
        (0.6784, 0.6784, 1.), (0.6392, 0.6392, 1.),
        (0.6, 0.6, 1.), (0.5608, 0.5608, 1.),
        (0.5216, 0.5216, 1.), (0.4784, 0.4784, 1.),
        (0.4392, 0.4392, 1.), (0.4, 0.4, 1.),
        (0.3608, 0.3608, 1.), (0.3216, 0.3216, 1.),
        (0.2784, 0.2784, 1.), (0.2392, 0.2392, 1.),
        (0.2, 0.2, 1.), (0.1608, 0.1608, 1.),
        (0.1216, 0.1216, 1.), (0.0784, 0.0784, 1.),
        (0.0392, 0.0392, 1.), (0., 0., 1.),
        (0., 0.0275, 0.9608), (0.0039, 0.0549, 0.9255),
        (0.0039, 0.0824, 0.8863), (0.0039, 0.1137, 0.851),
        (0.0078, 0.1412, 0.8118), (0.0078, 0.1686, 0.7725),
        (0.0078, 0.1961, 0.7373), (0.0118, 0.2235, 0.698),
        (0.0118, 0.251, 0.6627), (0.0118, 0.2824, 0.6235),
        (0.0157, 0.3098, 0.5843), (0.0157, 0.3373, 0.549),
        (0.0157, 0.3647, 0.5098), (0.0157, 0.3922, 0.4745),
        (0.0196, 0.4196, 0.4353), (0.0196, 0.451, 0.3961),
        (0.0196, 0.4784, 0.3608), (0.0235, 0.5059, 0.3216),
        (0.0235, 0.5333, 0.2863), (0.0235, 0.5608, 0.2471),
        (0.0275, 0.5882, 0.2078), (0.0275, 0.6196, 0.1725),
        (0.0275, 0.6471, 0.1333), (0.0314, 0.6745, 0.098),
        (0.0314, 0.702, 0.0588), (0.0706, 0.7137, 0.0549),
        (0.1137, 0.7255, 0.0549), (0.1529, 0.7412, 0.051),
        (0.1922, 0.7529, 0.051), (0.2314, 0.7647, 0.0471),
        (0.2745, 0.7765, 0.0431), (0.3137, 0.7882, 0.0431),
        (0.3529, 0.8, 0.0392), (0.3961, 0.8157, 0.0353),
        (0.4353, 0.8275, 0.0353), (0.4745, 0.8392, 0.0314),
        (0.5176, 0.851, 0.0314), (0.5569, 0.8627, 0.0275),
        (0.5961, 0.8745, 0.0235), (0.6353, 0.8902, 0.0235),
        (0.6784, 0.902, 0.0196), (0.7176, 0.9137, 0.0157),
        (0.7569, 0.9255, 0.0157), (0.8, 0.9373, 0.0118),
        (0.8392, 0.949, 0.0118), (0.8784, 0.9647, 0.0078),
        (0.9176, 0.9765, 0.0039), (0.9608, 0.9882, 0.0039),
        (1., 1., 0.), (1., 0.9608, 0.),
        (1., 0.9216, 0.), (1., 0.8784, 0.),
        (1., 0.8392, 0.), (1., 0.8, 0.),
        (1., 0.7608, 0.), (1., 0.7216, 0.),
        (1., 0.6784, 0.), (1., 0.6392, 0.),
        (1., 0.6, 0.), (1., 0.5608, 0.),
        (1., 0.5216, 0.), (1., 0.4784, 0.),
        (1., 0.4392, 0.), (1., 0.4, 0.),
        (1., 0.3608, 0.), (1., 0.3216, 0.),
        (1., 0.2784, 0.), (1., 0.2392, 0.),
        (1., 0.2, 0.), (1., 0.1608, 0.), (1., 0.1216, 0.),
        (1., 0.0784, 0.), (1., 0.0392, 0.), (1., 0., 0.)],
    'ncl:whblrewh': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9804, 0.9804, 1.),
        (0.9608, 0.9608, 1.), (0.9373, 0.9373, 1.),
        (0.9176, 0.9176, 1.), (0.898, 0.898, 1.),
        (0.8784, 0.8784, 1.), (0.8588, 0.8588, 1.),
        (0.8353, 0.8353, 1.), (0.8157, 0.8157, 1.),
        (0.7961, 0.7961, 1.), (0.7765, 0.7765, 1.),
        (0.7569, 0.7569, 1.), (0.7333, 0.7333, 1.),
        (0.7137, 0.7137, 1.), (0.6941, 0.6941, 1.),
        (0.6745, 0.6745, 1.), (0.6549, 0.6549, 1.),
        (0.6314, 0.6314, 1.), (0.6118, 0.6118, 1.),
        (0.5922, 0.5922, 1.), (0.5725, 0.5725, 1.),
        (0.5529, 0.5529, 1.), (0.5294, 0.5294, 1.),
        (0.5098, 0.5098, 1.), (0.4902, 0.4902, 1.),
        (0.4706, 0.4706, 1.), (0.4471, 0.4471, 1.),
        (0.4275, 0.4275, 1.), (0.4078, 0.4078, 1.),
        (0.3882, 0.3882, 1.), (0.3686, 0.3686, 1.),
        (0.3451, 0.3451, 1.), (0.3255, 0.3255, 1.),
        (0.3059, 0.3059, 1.), (0.2863, 0.2863, 1.),
        (0.2667, 0.2667, 1.), (0.2431, 0.2431, 1.),
        (0.2235, 0.2235, 1.), (0.2039, 0.2039, 1.),
        (0.1843, 0.1843, 1.), (0.1647, 0.1647, 1.),
        (0.1412, 0.1412, 1.), (0.1216, 0.1216, 1.),
        (0.102, 0.102, 1.), (0.0824, 0.0824, 1.),
        (0.0627, 0.0627, 1.), (0.0392, 0.0392, 1.),
        (0.0196, 0.0196, 1.), (0., 0., 1.),
        (1., 0., 0.), (1., 0.0196, 0.0196),
        (1., 0.0392, 0.0392), (1., 0.0627, 0.0627),
        (1., 0.0824, 0.0824), (1., 0.102, 0.102),
        (1., 0.1216, 0.1216), (1., 0.1412, 0.1412),
        (1., 0.1647, 0.1647), (1., 0.1843, 0.1843),
        (1., 0.2039, 0.2039), (1., 0.2235, 0.2235),
        (1., 0.2431, 0.2431), (1., 0.2667, 0.2667),
        (1., 0.2863, 0.2863), (1., 0.3059, 0.3059),
        (1., 0.3255, 0.3255), (1., 0.3451, 0.3451),
        (1., 0.3686, 0.3686), (1., 0.3882, 0.3882),
        (1., 0.4078, 0.4078), (1., 0.4275, 0.4275),
        (1., 0.4471, 0.4471), (1., 0.4706, 0.4706),
        (1., 0.4902, 0.4902), (1., 0.5098, 0.5098),
        (1., 0.5294, 0.5294), (1., 0.5529, 0.5529),
        (1., 0.5725, 0.5725), (1., 0.5922, 0.5922),
        (1., 0.6118, 0.6118), (1., 0.6314, 0.6314),
        (1., 0.6549, 0.6549), (1., 0.6745, 0.6745),
        (1., 0.6941, 0.6941), (1., 0.7137, 0.7137),
        (1., 0.7333, 0.7333), (1., 0.7569, 0.7569),
        (1., 0.7765, 0.7765), (1., 0.7961, 0.7961),
        (1., 0.8157, 0.8157), (1., 0.8353, 0.8353),
        (1., 0.8588, 0.8588), (1., 0.8784, 0.8784),
        (1., 0.898, 0.898), (1., 0.9176, 0.9176), (1., 0.9373, 0.9373),
        (1., 0.9608, 0.9608), (1., 0.9804, 0.9804), (1., 1., 1.)],
    'ncl:whviblgryeorre': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9608, 0.9412, 0.9725),
        (0.9216, 0.8824, 0.9451), (0.8824, 0.8235, 0.9176),
        (0.8431, 0.7647, 0.8902), (0.8078, 0.7059, 0.8627),
        (0.7686, 0.6471, 0.8353), (0.7294, 0.5882, 0.8078),
        (0.6902, 0.5294, 0.7804), (0.651, 0.4706, 0.7529),
        (0.6118, 0.4118, 0.7255), (0.5725, 0.3529, 0.698),
        (0.5333, 0.2941, 0.6706), (0.498, 0.2353, 0.6431),
        (0.4588, 0.1765, 0.6157), (0.4196, 0.1176, 0.5882),
        (0.3804, 0.0588, 0.5608), (0.3412, 0., 0.5333),
        (0.3216, 0., 0.5608), (0.298, 0., 0.5922),
        (0.2784, 0., 0.6196), (0.2549, 0., 0.651),
        (0.2353, 0., 0.6784), (0.2118, 0., 0.7098),
        (0.1922, 0., 0.7373), (0.1725, 0., 0.7686),
        (0.149, 0., 0.7961), (0.1294, 0., 0.8235),
        (0.1059, 0., 0.8549), (0.0863, 0., 0.8824),
        (0.0627, 0., 0.9137), (0.0431, 0., 0.9412),
        (0.0196, 0., 0.9725), (0., 0., 1.),
        (0., 0.0431, 0.9451), (0.0039, 0.0824, 0.8902),
        (0.0039, 0.1255, 0.8353), (0.0078, 0.1647, 0.7804),
        (0.0078, 0.2078, 0.7216), (0.0118, 0.2471, 0.6667),
        (0.0118, 0.2902, 0.6118), (0.0157, 0.3294, 0.5569),
        (0.0157, 0.3725, 0.502), (0.0196, 0.4118, 0.4471),
        (0.0196, 0.4549, 0.3922), (0.0235, 0.4941, 0.3373),
        (0.0235, 0.5373, 0.2784), (0.0275, 0.5765, 0.2235),
        (0.0275, 0.6196, 0.1686), (0.0314, 0.6588, 0.1137),
        (0.0314, 0.702, 0.0588), (0.0902, 0.7176, 0.0549),
        (0.1451, 0.7373, 0.051), (0.2039, 0.7529, 0.0471),
        (0.2588, 0.7725, 0.0431), (0.3176, 0.7882, 0.0431),
        (0.3725, 0.8078, 0.0392), (0.4314, 0.8235, 0.0353),
        (0.4863, 0.8431, 0.0314), (0.5451, 0.8588, 0.0275),
        (0.6, 0.8784, 0.0235), (0.6588, 0.8941, 0.0196),
        (0.7137, 0.9137, 0.0157), (0.7725, 0.9294, 0.0157),
        (0.8275, 0.949, 0.0118), (0.8863, 0.9647, 0.0078),
        (0.9412, 0.9843, 0.0039), (1., 1., 0.),
        (1., 0.9765, 0.), (1., 0.9569, 0.),
        (1., 0.9333, 0.), (1., 0.9137, 0.),
        (1., 0.8902, 0.), (1., 0.8667, 0.),
        (1., 0.8471, 0.), (1., 0.8235, 0.),
        (1., 0.8, 0.), (1., 0.7804, 0.),
        (1., 0.7569, 0.), (1., 0.7373, 0.),
        (1., 0.7137, 0.), (1., 0.6902, 0.),
        (1., 0.6706, 0.), (1., 0.6471, 0.),
        (1., 0.6078, 0.), (1., 0.5725, 0.),
        (1., 0.5333, 0.), (1., 0.4941, 0.),
        (1., 0.4549, 0.), (1., 0.4196, 0.),
        (1., 0.3804, 0.), (1., 0.3412, 0.),
        (1., 0.3059, 0.), (1., 0.2667, 0.),
        (1., 0.2275, 0.), (1., 0.1922, 0.),
        (1., 0.1529, 0.), (1., 0.1137, 0.),
        (1., 0.0745, 0.), (1., 0.0392, 0.),
        (1., 0., 0.)],
    'ncl:whviblgryeorrewh': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9412, 0.9098, 0.9569),
        (0.8784, 0.8196, 0.9137), (0.8196, 0.7255, 0.8745),
        (0.7608, 0.6353, 0.8314), (0.702, 0.5451, 0.7882),
        (0.6392, 0.4549, 0.7451), (0.5804, 0.3647, 0.702),
        (0.5216, 0.2745, 0.6588), (0.4627, 0.1804, 0.6196),
        (0.4, 0.0902, 0.5765), (0.3412, 0., 0.5333),
        (0.3098, 0., 0.5294), (0.2784, 0., 0.5294),
        (0.2471, 0., 0.5255), (0.2157, 0., 0.5216),
        (0.1843, 0., 0.5176), (0.1569, 0., 0.5176),
        (0.1255, 0., 0.5137), (0.0941, 0., 0.5098),
        (0.0627, 0., 0.5059), (0.0314, 0., 0.5059),
        (0., 0., 0.502), (0., 0., 0.549),
        (0., 0., 0.5922), (0., 0., 0.6392),
        (0., 0., 0.6824), (0., 0., 0.7294),
        (0., 0., 0.7725), (0., 0., 0.8196),
        (0., 0., 0.8627), (0., 0., 0.9098),
        (0., 0., 0.9529), (0., 0., 1.),
        (0., 0.0902, 0.9098), (0., 0.1804, 0.8196),
        (0., 0.2745, 0.7255), (0., 0.3647, 0.6353),
        (0., 0.4549, 0.5451), (0., 0.5451, 0.4549),
        (0., 0.6353, 0.3647), (0., 0.7255, 0.2745),
        (0., 0.8196, 0.1804), (0., 0.9098, 0.0902),
        (0., 1., 0.), (0.0824, 1., 0.),
        (0.1686, 1., 0.), (0.251, 1., 0.),
        (0.3333, 1., 0.), (0.4157, 1., 0.),
        (0.502, 1., 0.), (0.5843, 1., 0.),
        (0.6667, 1., 0.), (0.749, 1., 0.),
        (0.8353, 1., 0.), (0.9176, 1., 0.),
        (1., 1., 0.), (1., 0.9686, 0.),
        (1., 0.9373, 0.), (1., 0.902, 0.),
        (1., 0.8706, 0.), (1., 0.8392, 0.),
        (1., 0.8078, 0.), (1., 0.7765, 0.),
        (1., 0.7451, 0.), (1., 0.7098, 0.),
        (1., 0.6784, 0.), (1., 0.6471, 0.),
        (0.9529, 0.5882, 0.), (0.9098, 0.5294, 0.),
        (0.8627, 0.4706, 0.), (0.8196, 0.4118, 0.),
        (0.7725, 0.3529, 0.), (0.7294, 0.2941, 0.),
        (0.6824, 0.2353, 0.), (0.6392, 0.1765, 0.),
        (0.5922, 0.1176, 0.), (0.549, 0.0588, 0.),
        (0.502, 0., 0.), (0.549, 0., 0.),
        (0.5922, 0., 0.), (0.6392, 0., 0.),
        (0.6824, 0., 0.), (0.7294, 0., 0.),
        (0.7725, 0., 0.), (0.8196, 0., 0.),
        (0.8627, 0., 0.), (0.9098, 0., 0.),
        (0.9529, 0., 0.), (1., 0., 0.),
        (1., 0.0902, 0.0902), (1., 0.1804, 0.1804),
        (1., 0.2745, 0.2745), (1., 0.3647, 0.3647),
        (1., 0.4549, 0.4549), (1., 0.5451, 0.5451),
        (1., 0.6353, 0.6353), (1., 0.7255, 0.7255),
        (1., 0.8196, 0.8196), (1., 0.9098, 0.9098),
        (1., 1., 1.)],
    'ncl:whiteblue': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9961, 1., 1.),
        (0.9922, 0.9961, 1.), (0.9922, 0.9961, 1.),
        (0.9882, 0.9961, 1.), (0.9843, 0.9922, 0.9961),
        (0.9804, 0.9922, 0.9961), (0.9804, 0.9922, 0.9961),
        (0.9765, 0.9922, 0.9961), (0.9725, 0.9882, 0.9961),
        (0.9686, 0.9882, 0.9961), (0.9686, 0.9882, 0.9961),
        (0.9647, 0.9843, 0.9961), (0.9608, 0.9843, 0.9961),
        (0.9569, 0.9843, 0.9922), (0.9529, 0.9804, 0.9922),
        (0.9529, 0.9804, 0.9922), (0.949, 0.9804, 0.9922),
        (0.9451, 0.9765, 0.9922), (0.9412, 0.9765, 0.9922),
        (0.9412, 0.9765, 0.9922), (0.9373, 0.9765, 0.9922),
        (0.9333, 0.9725, 0.9882), (0.9294, 0.9725, 0.9882),
        (0.9294, 0.9725, 0.9882), (0.9255, 0.9686, 0.9882),
        (0.9216, 0.9686, 0.9882), (0.9176, 0.9686, 0.9882),
        (0.9137, 0.9647, 0.9882), (0.9137, 0.9647, 0.9882),
        (0.9098, 0.9647, 0.9882), (0.9059, 0.9608, 0.9843),
        (0.902, 0.9608, 0.9843), (0.902, 0.9608, 0.9843),
        (0.898, 0.9608, 0.9843), (0.8941, 0.9569, 0.9843),
        (0.8902, 0.9569, 0.9843), (0.8902, 0.9569, 0.9843),
        (0.8863, 0.9529, 0.9843), (0.8824, 0.9529, 0.9843),
        (0.8784, 0.9529, 0.9804), (0.8745, 0.949, 0.9804),
        (0.8745, 0.949, 0.9804), (0.8706, 0.949, 0.9804),
        (0.8667, 0.9451, 0.9804), (0.8627, 0.9451, 0.9804),
        (0.8627, 0.9451, 0.9804), (0.8588, 0.9451, 0.9804),
        (0.8549, 0.9412, 0.9765), (0.851, 0.9412, 0.9765),
        (0.851, 0.9412, 0.9765), (0.8431, 0.9373, 0.9765),
        (0.8392, 0.9373, 0.9765), (0.8353, 0.9333, 0.9765),
        (0.8314, 0.9333, 0.9765), (0.8275, 0.9294, 0.9765),
        (0.8196, 0.9294, 0.9765), (0.8157, 0.9294, 0.9765),
        (0.8118, 0.9255, 0.9765), (0.8078, 0.9255, 0.9765),
        (0.8039, 0.9216, 0.9765), (0.8, 0.9216, 0.9765),
        (0.7961, 0.9216, 0.9765), (0.7922, 0.9176, 0.9765),
        (0.7882, 0.9176, 0.9765), (0.7804, 0.9137, 0.9725),
        (0.7765, 0.9137, 0.9725), (0.7725, 0.9137, 0.9725),
        (0.7686, 0.9098, 0.9725), (0.7647, 0.9098, 0.9725),
        (0.7608, 0.9059, 0.9725), (0.7569, 0.9059, 0.9725),
        (0.7529, 0.9059, 0.9725), (0.7451, 0.902, 0.9725),
        (0.7412, 0.902, 0.9725), (0.7373, 0.898, 0.9725),
        (0.7333, 0.898, 0.9725), (0.7294, 0.898, 0.9725),
        (0.7255, 0.8941, 0.9725), (0.7216, 0.8941, 0.9725),
        (0.7176, 0.8902, 0.9725), (0.7137, 0.8902, 0.9725),
        (0.7059, 0.8863, 0.9725), (0.702, 0.8863, 0.9725),
        (0.698, 0.8863, 0.9725), (0.6941, 0.8824, 0.9725),
        (0.6902, 0.8824, 0.9725), (0.6863, 0.8784, 0.9725),
        (0.6824, 0.8784, 0.9725), (0.6784, 0.8784, 0.9725),
        (0.6745, 0.8745, 0.9725), (0.6667, 0.8745, 0.9686),
        (0.6627, 0.8706, 0.9686), (0.6588, 0.8706, 0.9686),
        (0.6549, 0.8706, 0.9686), (0.651, 0.8667, 0.9686),
        (0.6471, 0.8667, 0.9686), (0.6431, 0.8627, 0.9686),
        (0.6392, 0.8627, 0.9686), (0.6314, 0.8627, 0.9686),
        (0.6275, 0.8588, 0.9686), (0.6157, 0.8549, 0.9686),
        (0.6118, 0.851, 0.9647), (0.6039, 0.8471, 0.9647),
        (0.6, 0.8431, 0.9608), (0.5961, 0.8392, 0.9608),
        (0.5882, 0.8353, 0.9569), (0.5843, 0.8314, 0.9529),
        (0.5804, 0.8275, 0.9529), (0.5725, 0.8235, 0.949),
        (0.5686, 0.8196, 0.949), (0.5647, 0.8157, 0.9451),
        (0.5569, 0.8118, 0.9412), (0.5529, 0.8078, 0.9412),
        (0.549, 0.8039, 0.9373), (0.5412, 0.7961, 0.9373),
        (0.5373, 0.7922, 0.9333), (0.5294, 0.7882, 0.9294),
        (0.5255, 0.7843, 0.9294), (0.5216, 0.7804, 0.9255),
        (0.5137, 0.7765, 0.9255), (0.5098, 0.7725, 0.9216),
        (0.5059, 0.7686, 0.9176), (0.498, 0.7647, 0.9176),
        (0.4941, 0.7608, 0.9137), (0.4902, 0.7569, 0.9137),
        (0.4824, 0.7529, 0.9098), (0.4784, 0.749, 0.9098),
        (0.4745, 0.7451, 0.9059), (0.4667, 0.7412, 0.902),
        (0.4627, 0.7373, 0.902), (0.4588, 0.7333, 0.898),
        (0.451, 0.7294, 0.898), (0.4471, 0.7255, 0.8941),
        (0.4431, 0.7216, 0.8902), (0.4353, 0.7176, 0.8902),
        (0.4314, 0.7137, 0.8863), (0.4275, 0.7098, 0.8863),
        (0.4196, 0.7059, 0.8824), (0.4157, 0.702, 0.8784),
        (0.4118, 0.698, 0.8784), (0.4039, 0.6902, 0.8745),
        (0.4, 0.6863, 0.8745), (0.3922, 0.6824, 0.8706),
        (0.3882, 0.6784, 0.8667), (0.3843, 0.6745, 0.8667),
        (0.3765, 0.6706, 0.8627), (0.3725, 0.6667, 0.8627),
        (0.3686, 0.6627, 0.8588), (0.3608, 0.6588, 0.8549),
        (0.3569, 0.6549, 0.8549), (0.3529, 0.651, 0.851),
        (0.3412, 0.6431, 0.8471), (0.3373, 0.6392, 0.8431),
        (0.3333, 0.6314, 0.8392), (0.3294, 0.6275, 0.8353),
        (0.3255, 0.6196, 0.8314), (0.3216, 0.6157, 0.8275),
        (0.3176, 0.6078, 0.8235), (0.3137, 0.6039, 0.8196),
        (0.3098, 0.5961, 0.8157), (0.3059, 0.5922, 0.8157),
        (0.302, 0.5882, 0.8118), (0.298, 0.5804, 0.8078),
        (0.2941, 0.5765, 0.8039), (0.2902, 0.5686, 0.8),
        (0.2863, 0.5647, 0.7961), (0.2824, 0.5569, 0.7922),
        (0.2784, 0.5529, 0.7882), (0.2745, 0.5451, 0.7843),
        (0.2706, 0.5412, 0.7804), (0.2667, 0.5373, 0.7765),
        (0.2627, 0.5294, 0.7725), (0.2588, 0.5255, 0.7686),
        (0.2549, 0.5176, 0.7647), (0.251, 0.5137, 0.7608),
        (0.2471, 0.5059, 0.7569), (0.2431, 0.502, 0.7529),
        (0.2431, 0.498, 0.7529), (0.2392, 0.4902, 0.749),
        (0.2353, 0.4863, 0.7451), (0.2314, 0.4784, 0.7412),
        (0.2275, 0.4745, 0.7373), (0.2235, 0.4667, 0.7333),
        (0.2196, 0.4627, 0.7294), (0.2157, 0.4549, 0.7255),
        (0.2118, 0.451, 0.7216), (0.2078, 0.4471, 0.7176),
        (0.2039, 0.4392, 0.7137), (0.2, 0.4353, 0.7098),
        (0.1961, 0.4275, 0.7059), (0.1922, 0.4235, 0.702),
        (0.1882, 0.4157, 0.698), (0.1843, 0.4118, 0.6941),
        (0.1804, 0.4039, 0.6902), (0.1765, 0.4, 0.6863),
        (0.1725, 0.3961, 0.6863), (0.1686, 0.3882, 0.6824),
        (0.1647, 0.3843, 0.6784), (0.1608, 0.3765, 0.6745),
        (0.1569, 0.3725, 0.6706), (0.1529, 0.3647, 0.6667),
        (0.1412, 0.349, 0.6549), (0.1412, 0.3451, 0.651),
        (0.1412, 0.3412, 0.6431), (0.1373, 0.3373, 0.6392),
        (0.1373, 0.3373, 0.6353), (0.1373, 0.3333, 0.6275),
        (0.1373, 0.3294, 0.6235), (0.1373, 0.3255, 0.6196),
        (0.1373, 0.3216, 0.6118), (0.1333, 0.3176, 0.6078),
        (0.1333, 0.3137, 0.6039), (0.1333, 0.3098, 0.5961),
        (0.1333, 0.3098, 0.5922), (0.1333, 0.3059, 0.5882),
        (0.1333, 0.302, 0.5804), (0.1294, 0.298, 0.5765),
        (0.1294, 0.2941, 0.5725), (0.1294, 0.2902, 0.5647),
        (0.1294, 0.2863, 0.5608), (0.1294, 0.2863, 0.5569),
        (0.1294, 0.2824, 0.549), (0.1255, 0.2784, 0.5451),
        (0.1255, 0.2745, 0.5412), (0.1255, 0.2706, 0.5333),
        (0.1255, 0.2667, 0.5294), (0.1255, 0.2627, 0.5255),
        (0.1255, 0.2627, 0.5216), (0.1216, 0.2588, 0.5137),
        (0.1216, 0.2549, 0.5098), (0.1216, 0.251, 0.5059),
        (0.1216, 0.2471, 0.498), (0.1216, 0.2431, 0.4941),
        (0.1176, 0.2392, 0.4902), (0.1176, 0.2353, 0.4824),
        (0.1176, 0.2353, 0.4784), (0.1176, 0.2314, 0.4745),
        (0.1176, 0.2275, 0.4667), (0.1176, 0.2235, 0.4627),
        (0.1137, 0.2196, 0.4588), (0.1137, 0.2157, 0.451),
        (0.1137, 0.2118, 0.4471), (0.1137, 0.2118, 0.4431),
        (0.1137, 0.2078, 0.4353), (0.1137, 0.2039, 0.4314),
        (0.1098, 0.2, 0.4275), (0.1098, 0.1961, 0.4196),
        (0.1098, 0.1922, 0.4157), (0.1098, 0.1882, 0.4118),
        (0.1098, 0.1843, 0.4039), (0.1098, 0.1843, 0.4),
        (0.1059, 0.1804, 0.3961), (0.1059, 0.1725, 0.3843)],
    'ncl:whitebluegreenyellowred': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9882, 0.9961, 1.),
        (0.9804, 0.9922, 1.), (0.9686, 0.9882, 0.9961),
        (0.9569, 0.9843, 0.9961), (0.949, 0.9804, 0.9961),
        (0.9373, 0.9765, 0.9961), (0.9255, 0.9725, 0.9922),
        (0.9176, 0.9686, 0.9922), (0.9059, 0.9647, 0.9922),
        (0.898, 0.9608, 0.9922), (0.8863, 0.9569, 0.9922),
        (0.8745, 0.9529, 0.9882), (0.8667, 0.949, 0.9882),
        (0.8549, 0.9451, 0.9882), (0.8431, 0.9412, 0.9882),
        (0.8353, 0.9373, 0.9882), (0.8235, 0.9333, 0.9843),
        (0.8118, 0.9294, 0.9843), (0.8039, 0.9255, 0.9843),
        (0.7922, 0.9216, 0.9843), (0.7804, 0.9176, 0.9804),
        (0.7725, 0.9137, 0.9804), (0.7608, 0.9098, 0.9804),
        (0.749, 0.9059, 0.9804), (0.7412, 0.902, 0.9804),
        (0.7294, 0.898, 0.9765), (0.7176, 0.8941, 0.9765),
        (0.7098, 0.8902, 0.9765), (0.698, 0.8863, 0.9765),
        (0.6902, 0.8824, 0.9765), (0.6784, 0.8784, 0.9725),
        (0.6667, 0.8745, 0.9725), (0.6588, 0.8706, 0.9725),
        (0.6471, 0.8667, 0.9725), (0.6353, 0.8627, 0.9686),
        (0.6157, 0.8549, 0.9686), (0.6078, 0.8471, 0.9647),
        (0.5961, 0.8392, 0.9608), (0.5882, 0.8314, 0.9529),
        (0.5804, 0.8235, 0.949), (0.5725, 0.8157, 0.9451),
        (0.5608, 0.8078, 0.9412), (0.5529, 0.8, 0.9333),
        (0.5451, 0.7922, 0.9294), (0.5333, 0.7843, 0.9255),
        (0.5255, 0.7725, 0.9216), (0.5176, 0.7647, 0.9176),
        (0.5059, 0.7569, 0.9098), (0.498, 0.749, 0.9059),
        (0.4902, 0.7412, 0.902), (0.4824, 0.7333, 0.898),
        (0.4706, 0.7255, 0.8941), (0.4627, 0.7176, 0.8863),
        (0.4549, 0.7098, 0.8824), (0.4431, 0.702, 0.8784),
        (0.4353, 0.6941, 0.8745), (0.4275, 0.6863, 0.8667),
        (0.4157, 0.6784, 0.8627), (0.4078, 0.6706, 0.8588),
        (0.4, 0.6627, 0.8549), (0.3922, 0.6549, 0.851),
        (0.3804, 0.6471, 0.8431), (0.3725, 0.6392, 0.8392),
        (0.3647, 0.6275, 0.8353), (0.3529, 0.6196, 0.8314),
        (0.3451, 0.6118, 0.8275), (0.3373, 0.6039, 0.8196),
        (0.3255, 0.5961, 0.8157), (0.3176, 0.5882, 0.8118),
        (0.3098, 0.5804, 0.8078), (0.302, 0.5725, 0.8),
        (0.2824, 0.5569, 0.7922), (0.2824, 0.5608, 0.7765),
        (0.2824, 0.5647, 0.7647), (0.2824, 0.5686, 0.749),
        (0.2824, 0.5725, 0.7373), (0.2824, 0.5765, 0.7216),
        (0.2824, 0.5804, 0.7098), (0.2824, 0.5843, 0.6941),
        (0.2824, 0.5882, 0.6784), (0.2824, 0.5922, 0.6667),
        (0.2824, 0.6, 0.651), (0.2824, 0.6039, 0.6392),
        (0.2824, 0.6078, 0.6235), (0.2824, 0.6118, 0.6118),
        (0.2824, 0.6157, 0.5961), (0.2824, 0.6196, 0.5804),
        (0.2824, 0.6235, 0.5686), (0.2824, 0.6275, 0.5529),
        (0.2824, 0.6314, 0.5412), (0.2863, 0.6353, 0.5255),
        (0.2863, 0.6392, 0.5137), (0.2863, 0.6431, 0.498),
        (0.2863, 0.6471, 0.4863), (0.2863, 0.651, 0.4706),
        (0.2863, 0.6549, 0.4549), (0.2863, 0.6588, 0.4431),
        (0.2863, 0.6627, 0.4275), (0.2863, 0.6667, 0.4157),
        (0.2863, 0.6745, 0.4), (0.2863, 0.6784, 0.3882),
        (0.2863, 0.6824, 0.3725), (0.2863, 0.6863, 0.3569),
        (0.2863, 0.6902, 0.3451), (0.2863, 0.6941, 0.3294),
        (0.2863, 0.698, 0.3176), (0.2863, 0.702, 0.302),
        (0.2863, 0.7098, 0.2745), (0.3059, 0.7137, 0.2784),
        (0.3255, 0.7216, 0.2784), (0.3412, 0.7255, 0.2824),
        (0.3608, 0.7333, 0.2824), (0.3804, 0.7373, 0.2863),
        (0.4, 0.7412, 0.2902), (0.4157, 0.749, 0.2902),
        (0.4353, 0.7529, 0.2941), (0.4549, 0.7569, 0.2941),
        (0.4745, 0.7647, 0.298), (0.4941, 0.7686, 0.302),
        (0.5098, 0.7765, 0.302), (0.5294, 0.7804, 0.3059),
        (0.549, 0.7843, 0.3059), (0.5686, 0.7922, 0.3098),
        (0.5882, 0.7961, 0.3137), (0.6039, 0.8, 0.3137),
        (0.6235, 0.8078, 0.3176), (0.6431, 0.8118, 0.3176),
        (0.6627, 0.8196, 0.3216), (0.6784, 0.8235, 0.3216),
        (0.698, 0.8275, 0.3255), (0.7176, 0.8353, 0.3294),
        (0.7373, 0.8392, 0.3294), (0.7569, 0.8431, 0.3333),
        (0.7725, 0.851, 0.3333), (0.7922, 0.8549, 0.3373),
        (0.8118, 0.8627, 0.3412), (0.8314, 0.8667, 0.3412),
        (0.851, 0.8706, 0.3451), (0.8667, 0.8784, 0.3451),
        (0.8863, 0.8824, 0.349), (0.9059, 0.8863, 0.3529),
        (0.9255, 0.8941, 0.3529), (0.9412, 0.898, 0.3569),
        (0.9608, 0.9059, 0.3569), (0.9804, 0.9098, 0.3608),
        (0.9804, 0.898, 0.3569), (0.9804, 0.8824, 0.349),
        (0.9804, 0.8706, 0.3451), (0.9765, 0.8549, 0.3373),
        (0.9765, 0.8431, 0.3333), (0.9765, 0.8314, 0.3294),
        (0.9765, 0.8157, 0.3216), (0.9765, 0.8039, 0.3176),
        (0.9765, 0.7882, 0.3137), (0.9765, 0.7765, 0.3059),
        (0.9765, 0.7647, 0.302), (0.9725, 0.749, 0.2941),
        (0.9725, 0.7373, 0.2902), (0.9725, 0.7216, 0.2863),
        (0.9725, 0.7098, 0.2784), (0.9725, 0.698, 0.2745),
        (0.9725, 0.6824, 0.2706), (0.9725, 0.6706, 0.2627),
        (0.9686, 0.6549, 0.2588), (0.9686, 0.6431, 0.251),
        (0.9686, 0.6275, 0.2471), (0.9686, 0.6157, 0.2431),
        (0.9686, 0.6039, 0.2353), (0.9686, 0.5882, 0.2314),
        (0.9686, 0.5765, 0.2275), (0.9647, 0.5608, 0.2196),
        (0.9647, 0.549, 0.2157), (0.9647, 0.5373, 0.2078),
        (0.9647, 0.5216, 0.2039), (0.9647, 0.5098, 0.2),
        (0.9647, 0.4941, 0.1922), (0.9647, 0.4824, 0.1882),
        (0.9647, 0.4706, 0.1843), (0.9608, 0.4549, 0.1765),
        (0.9608, 0.4431, 0.1725), (0.9608, 0.4157, 0.1608),
        (0.9569, 0.4078, 0.1608), (0.9529, 0.4, 0.1608),
        (0.949, 0.3922, 0.1608), (0.9451, 0.3843, 0.1608),
        (0.9412, 0.3765, 0.1608), (0.9373, 0.3686, 0.1608),
        (0.9373, 0.3608, 0.1608), (0.9333, 0.3529, 0.1608),
        (0.9294, 0.3451, 0.1608), (0.9255, 0.3373, 0.1608),
        (0.9216, 0.3294, 0.1608), (0.9176, 0.3216, 0.1608),
        (0.9137, 0.3137, 0.1608), (0.9098, 0.3059, 0.1608),
        (0.9059, 0.298, 0.1608), (0.902, 0.2902, 0.1608),
        (0.898, 0.2824, 0.1608), (0.8941, 0.2745, 0.1608),
        (0.8941, 0.2627, 0.1569), (0.8902, 0.2549, 0.1569),
        (0.8863, 0.2471, 0.1569), (0.8824, 0.2392, 0.1569),
        (0.8784, 0.2314, 0.1569), (0.8745, 0.2235, 0.1569),
        (0.8706, 0.2157, 0.1569), (0.8667, 0.2078, 0.1569),
        (0.8627, 0.2, 0.1569), (0.8588, 0.1922, 0.1569),
        (0.8549, 0.1843, 0.1569), (0.851, 0.1765, 0.1569),
        (0.851, 0.1686, 0.1569), (0.8471, 0.1608, 0.1569),
        (0.8431, 0.1529, 0.1569), (0.8392, 0.1451, 0.1569),
        (0.8353, 0.1373, 0.1569), (0.8275, 0.1216, 0.1569),
        (0.8196, 0.1216, 0.1569), (0.8118, 0.1176, 0.1529),
        (0.8078, 0.1176, 0.1529), (0.8, 0.1176, 0.149),
        (0.7922, 0.1176, 0.149), (0.7843, 0.1137, 0.149),
        (0.7804, 0.1137, 0.1451), (0.7725, 0.1137, 0.1451),
        (0.7647, 0.1137, 0.1412), (0.7569, 0.1098, 0.1412),
        (0.7529, 0.1098, 0.1412), (0.7451, 0.1098, 0.1373),
        (0.7373, 0.1059, 0.1373), (0.7294, 0.1059, 0.1333),
        (0.7255, 0.1059, 0.1333), (0.7176, 0.1059, 0.1333),
        (0.7098, 0.102, 0.1294), (0.702, 0.102, 0.1294),
        (0.698, 0.102, 0.1255), (0.6902, 0.102, 0.1255),
        (0.6824, 0.098, 0.1216), (0.6745, 0.098, 0.1216),
        (0.6706, 0.098, 0.1216), (0.6627, 0.098, 0.1176),
        (0.6549, 0.0941, 0.1176), (0.6471, 0.0941, 0.1137),
        (0.6431, 0.0941, 0.1137), (0.6353, 0.0902, 0.1137),
        (0.6275, 0.0902, 0.1098), (0.6196, 0.0902, 0.1098),
        (0.6157, 0.0902, 0.1059), (0.6078, 0.0863, 0.1059),
        (0.6, 0.0863, 0.1059), (0.5922, 0.0863, 0.102),
        (0.5882, 0.0863, 0.102), (0.5725, 0.0824, 0.098)],
    'ncl:whitegreen': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (0.9922, 0.9961, 0.9922),
        (0.9882, 0.9961, 0.9882), (0.9804, 0.9922, 0.9804),
        (0.9765, 0.9882, 0.9725), (0.9686, 0.9882, 0.9647),
        (0.9647, 0.9843, 0.9608), (0.9569, 0.9843, 0.9529),
        (0.9529, 0.9804, 0.9451), (0.9451, 0.9765, 0.9412),
        (0.9412, 0.9765, 0.9333), (0.9333, 0.9725, 0.9255),
        (0.9294, 0.9686, 0.9176), (0.9216, 0.9686, 0.9137),
        (0.9176, 0.9647, 0.9059), (0.9098, 0.9608, 0.898),
        (0.9059, 0.9608, 0.8941), (0.898, 0.9569, 0.8863),
        (0.8902, 0.9569, 0.8784), (0.8863, 0.9529, 0.8706),
        (0.8784, 0.949, 0.8667), (0.8745, 0.949, 0.8588),
        (0.8667, 0.9451, 0.851), (0.8627, 0.9412, 0.8431),
        (0.8549, 0.9412, 0.8392), (0.851, 0.9373, 0.8314),
        (0.8431, 0.9333, 0.8235), (0.8392, 0.9333, 0.8196),
        (0.8314, 0.9294, 0.8118), (0.8275, 0.9294, 0.8039),
        (0.8196, 0.9255, 0.7961), (0.8157, 0.9216, 0.7922),
        (0.8078, 0.9216, 0.7843), (0.8039, 0.9176, 0.7765),
        (0.7961, 0.9137, 0.7725), (0.7882, 0.9137, 0.7647),
        (0.7843, 0.9098, 0.7569), (0.7765, 0.9059, 0.749),
        (0.7725, 0.9059, 0.7451), (0.7647, 0.902, 0.7373),
        (0.7608, 0.902, 0.7294), (0.7529, 0.898, 0.7255),
        (0.749, 0.8941, 0.7176), (0.7412, 0.8941, 0.7098),
        (0.7373, 0.8902, 0.702), (0.7294, 0.8863, 0.698),
        (0.7255, 0.8863, 0.6902), (0.7176, 0.8824, 0.6824),
        (0.7137, 0.8824, 0.6784), (0.7059, 0.8784, 0.6706),
        (0.702, 0.8745, 0.6627), (0.6941, 0.8745, 0.6549),
        (0.6863, 0.8706, 0.651), (0.6824, 0.8667, 0.6431),
        (0.6745, 0.8667, 0.6353), (0.6706, 0.8627, 0.6314),
        (0.6627, 0.8588, 0.6235), (0.6588, 0.8588, 0.6157),
        (0.651, 0.8549, 0.6078), (0.6471, 0.8549, 0.6039),
        (0.6392, 0.851, 0.5961), (0.6353, 0.8471, 0.5882),
        (0.6275, 0.8471, 0.5843), (0.6235, 0.8431, 0.5765),
        (0.6157, 0.8392, 0.5686), (0.6118, 0.8392, 0.5608),
        (0.6039, 0.8353, 0.5569), (0.6, 0.8314, 0.549),
        (0.5922, 0.8314, 0.5412), (0.5843, 0.8275, 0.5333),
        (0.5804, 0.8275, 0.5294), (0.5725, 0.8235, 0.5216),
        (0.5686, 0.8196, 0.5137), (0.5608, 0.8196, 0.5098),
        (0.5569, 0.8157, 0.502), (0.549, 0.8118, 0.4941),
        (0.5451, 0.8118, 0.4863), (0.5373, 0.8078, 0.4824),
        (0.5333, 0.8039, 0.4745), (0.5255, 0.8039, 0.4667),
        (0.5216, 0.8, 0.4627), (0.5137, 0.8, 0.4549),
        (0.5098, 0.7961, 0.4471), (0.502, 0.7922, 0.4392),
        (0.4902, 0.7882, 0.4275), (0.4863, 0.7843, 0.4235),
        (0.4784, 0.7843, 0.4196), (0.4745, 0.7804, 0.4157),
        (0.4667, 0.7804, 0.4118), (0.4627, 0.7765, 0.4078),
        (0.4588, 0.7765, 0.4039), (0.451, 0.7725, 0.4),
        (0.4471, 0.7725, 0.3961), (0.4431, 0.7686, 0.3922),
        (0.4353, 0.7686, 0.3882), (0.4314, 0.7647, 0.3843),
        (0.4235, 0.7647, 0.3804), (0.4196, 0.7608, 0.3765),
        (0.4157, 0.7608, 0.3725), (0.4078, 0.7569, 0.3647),
        (0.4039, 0.7569, 0.3608), (0.4, 0.7529, 0.3569),
        (0.3922, 0.749, 0.3529), (0.3882, 0.749, 0.349),
        (0.3804, 0.7451, 0.3451), (0.3765, 0.7451, 0.3412),
        (0.3725, 0.7412, 0.3373), (0.3647, 0.7412, 0.3333),
        (0.3608, 0.7373, 0.3294), (0.3569, 0.7373, 0.3255),
        (0.349, 0.7333, 0.3216), (0.3451, 0.7333, 0.3176),
        (0.3373, 0.7294, 0.3137), (0.3333, 0.7294, 0.3098),
        (0.3294, 0.7255, 0.3059), (0.3216, 0.7255, 0.302),
        (0.3176, 0.7216, 0.298), (0.3137, 0.7216, 0.2941),
        (0.3059, 0.7176, 0.2902), (0.302, 0.7137, 0.2863),
        (0.2941, 0.7137, 0.2824), (0.2902, 0.7098, 0.2784),
        (0.2863, 0.7098, 0.2745), (0.2784, 0.7059, 0.2706),
        (0.2745, 0.7059, 0.2667), (0.2706, 0.702, 0.2627),
        (0.2627, 0.702, 0.2588), (0.2588, 0.698, 0.251),
        (0.251, 0.698, 0.2471), (0.2471, 0.6941, 0.2431),
        (0.2431, 0.6941, 0.2392), (0.2353, 0.6902, 0.2353),
        (0.2314, 0.6902, 0.2314), (0.2275, 0.6863, 0.2275),
        (0.2196, 0.6863, 0.2235), (0.2157, 0.6824, 0.2196),
        (0.2078, 0.6784, 0.2157), (0.2039, 0.6784, 0.2118),
        (0.2, 0.6745, 0.2078), (0.1922, 0.6745, 0.2039),
        (0.1882, 0.6706, 0.2), (0.1843, 0.6706, 0.1961),
        (0.1765, 0.6667, 0.1922), (0.1725, 0.6667, 0.1882),
        (0.1647, 0.6627, 0.1843), (0.1608, 0.6627, 0.1804),
        (0.1569, 0.6588, 0.1765), (0.149, 0.6588, 0.1725),
        (0.1451, 0.6549, 0.1686), (0.1412, 0.6549, 0.1647),
        (0.1333, 0.651, 0.1608), (0.1294, 0.651, 0.1569),
        (0.1216, 0.6471, 0.1529), (0.1176, 0.6431, 0.149),
        (0.1137, 0.6431, 0.1451), (0.1059, 0.6392, 0.1373),
        (0.102, 0.6392, 0.1333), (0.098, 0.6353, 0.1294),
        (0.0902, 0.6353, 0.1255), (0.0863, 0.6314, 0.1216),
        (0.0784, 0.6314, 0.1176), (0.0745, 0.6275, 0.1137),
        (0.0706, 0.6275, 0.1098), (0.0627, 0.6235, 0.1059),
        (0.0588, 0.6235, 0.102), (0.0549, 0.6196, 0.098),
        (0.0471, 0.6196, 0.0941), (0.0431, 0.6157, 0.0902),
        (0.0353, 0.6157, 0.0863), (0.0314, 0.6118, 0.0824),
        (0.0314, 0.6078, 0.0824), (0.0314, 0.6039, 0.0824),
        (0.0314, 0.6, 0.0824), (0.0314, 0.5961, 0.0863),
        (0.0314, 0.5961, 0.0863), (0.0314, 0.5922, 0.0863),
        (0.0314, 0.5882, 0.0863), (0.0314, 0.5843, 0.0863),
        (0.0314, 0.5804, 0.0863), (0.0314, 0.5765, 0.0902),
        (0.0314, 0.5725, 0.0902), (0.0314, 0.5686, 0.0902),
        (0.0314, 0.5647, 0.0902), (0.0314, 0.5608, 0.0902),
        (0.0275, 0.5608, 0.0902), (0.0275, 0.5569, 0.0941),
        (0.0275, 0.5529, 0.0941), (0.0275, 0.549, 0.0941),
        (0.0275, 0.5451, 0.0941), (0.0275, 0.5412, 0.0941),
        (0.0275, 0.5373, 0.0941), (0.0275, 0.5333, 0.098),
        (0.0275, 0.5294, 0.098), (0.0275, 0.5294, 0.098),
        (0.0275, 0.5255, 0.098), (0.0275, 0.5216, 0.098),
        (0.0275, 0.5176, 0.098), (0.0275, 0.5137, 0.102),
        (0.0275, 0.5098, 0.102), (0.0275, 0.5059, 0.102),
        (0.0275, 0.502, 0.102), (0.0275, 0.498, 0.102),
        (0.0275, 0.4941, 0.102), (0.0275, 0.4941, 0.1059),
        (0.0275, 0.4902, 0.1059), (0.0275, 0.4863, 0.1059),
        (0.0275, 0.4824, 0.1059), (0.0275, 0.4784, 0.1059),
        (0.0275, 0.4745, 0.1059), (0.0275, 0.4706, 0.1098),
        (0.0275, 0.4667, 0.1098), (0.0275, 0.4627, 0.1098),
        (0.0235, 0.4627, 0.1098), (0.0235, 0.4588, 0.1098),
        (0.0235, 0.4549, 0.1098), (0.0235, 0.451, 0.1137),
        (0.0235, 0.4471, 0.1137), (0.0235, 0.4431, 0.1137),
        (0.0235, 0.4392, 0.1137), (0.0235, 0.4353, 0.1137),
        (0.0235, 0.4314, 0.1137), (0.0235, 0.4314, 0.1176),
        (0.0235, 0.4275, 0.1176), (0.0235, 0.4235, 0.1176),
        (0.0235, 0.4196, 0.1176), (0.0235, 0.4157, 0.1176),
        (0.0235, 0.4118, 0.1176), (0.0235, 0.4078, 0.1216),
        (0.0235, 0.4039, 0.1216), (0.0235, 0.4, 0.1216),
        (0.0235, 0.3961, 0.1216), (0.0235, 0.3961, 0.1216),
        (0.0235, 0.3922, 0.1216), (0.0235, 0.3882, 0.1255),
        (0.0235, 0.3843, 0.1255), (0.0235, 0.3804, 0.1255),
        (0.0235, 0.3765, 0.1255), (0.0235, 0.3725, 0.1255),
        (0.0235, 0.3686, 0.1255), (0.0235, 0.3647, 0.1294),
        (0.0196, 0.3647, 0.1294), (0.0196, 0.3608, 0.1294),
        (0.0196, 0.3569, 0.1294), (0.0196, 0.3529, 0.1294),
        (0.0196, 0.349, 0.1294), (0.0196, 0.3451, 0.1333),
        (0.0196, 0.3412, 0.1333), (0.0196, 0.3373, 0.1333),
        (0.0196, 0.3333, 0.1333), (0.0196, 0.3294, 0.1333),
        (0.0196, 0.3294, 0.1333), (0.0196, 0.3255, 0.1373),
        (0.0196, 0.3216, 0.1373), (0.0196, 0.3137, 0.1373)],
    'ncl:whiteyelloworangered': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 1.), (1., 1., 0.9922),
        (1., 0.9961, 0.9843), (1., 0.9961, 0.9765),
        (1., 0.9922, 0.9686), (1., 0.9922, 0.9608),
        (1., 0.9922, 0.9529), (1., 0.9882, 0.949),
        (1., 0.9882, 0.9412), (0.9961, 0.9882, 0.9333),
        (0.9961, 0.9843, 0.9255), (0.9961, 0.9843, 0.9176),
        (0.9961, 0.9804, 0.9098), (0.9961, 0.9804, 0.902),
        (0.9961, 0.9804, 0.8941), (0.9961, 0.9765, 0.8863),
        (0.9961, 0.9765, 0.8784), (0.9961, 0.9725, 0.8706),
        (0.9961, 0.9725, 0.8627), (0.9961, 0.9725, 0.8549),
        (0.9961, 0.9686, 0.851), (0.9961, 0.9686, 0.8431),
        (0.9961, 0.9686, 0.8353), (0.9961, 0.9647, 0.8275),
        (0.9961, 0.9647, 0.8196), (0.9961, 0.9608, 0.8118),
        (0.9961, 0.9608, 0.8039), (0.9922, 0.9608, 0.7961),
        (0.9922, 0.9569, 0.7882), (0.9922, 0.9569, 0.7804),
        (0.9922, 0.9529, 0.7725), (0.9922, 0.9529, 0.7647),
        (0.9922, 0.9529, 0.7569), (0.9922, 0.949, 0.7529),
        (0.9922, 0.949, 0.7451), (0.9922, 0.949, 0.7373),
        (0.9922, 0.9451, 0.7294), (0.9922, 0.9451, 0.7216),
        (0.9922, 0.9412, 0.7137), (0.9922, 0.9412, 0.7059),
        (0.9922, 0.9412, 0.698), (0.9922, 0.9373, 0.6902),
        (0.9922, 0.9373, 0.6824), (0.9922, 0.9333, 0.6745),
        (0.9882, 0.9333, 0.6667), (0.9882, 0.9333, 0.6588),
        (0.9882, 0.9294, 0.6549), (0.9882, 0.9294, 0.6471),
        (0.9882, 0.9294, 0.6392), (0.9882, 0.9255, 0.6314),
        (0.9882, 0.9255, 0.6235), (0.9882, 0.9216, 0.6078),
        (0.9882, 0.9176, 0.6), (0.9882, 0.9137, 0.5922),
        (0.9882, 0.9098, 0.5843), (0.9882, 0.902, 0.5765),
        (0.9882, 0.898, 0.5686), (0.9882, 0.8941, 0.5608),
        (0.9882, 0.8902, 0.5529), (0.9882, 0.8863, 0.5451),
        (0.9882, 0.8824, 0.5373), (0.9882, 0.8784, 0.5294),
        (0.9882, 0.8745, 0.5216), (0.9882, 0.8667, 0.5137),
        (0.9922, 0.8627, 0.5059), (0.9922, 0.8588, 0.498),
        (0.9922, 0.8549, 0.4902), (0.9922, 0.851, 0.4824),
        (0.9922, 0.8471, 0.4745), (0.9922, 0.8431, 0.4667),
        (0.9922, 0.8353, 0.4588), (0.9922, 0.8314, 0.451),
        (0.9922, 0.8275, 0.4431), (0.9922, 0.8235, 0.4353),
        (0.9922, 0.8196, 0.4275), (0.9922, 0.8157, 0.4196),
        (0.9922, 0.8118, 0.4118), (0.9922, 0.8078, 0.4039),
        (0.9922, 0.8, 0.3922), (0.9922, 0.7961, 0.3843),
        (0.9922, 0.7922, 0.3765), (0.9922, 0.7882, 0.3686),
        (0.9922, 0.7843, 0.3608), (0.9922, 0.7804, 0.3529),
        (0.9922, 0.7765, 0.3451), (0.9922, 0.7686, 0.3373),
        (0.9922, 0.7647, 0.3294), (0.9922, 0.7608, 0.3216),
        (0.9922, 0.7569, 0.3137), (0.9922, 0.7529, 0.3059),
        (0.9961, 0.749, 0.298), (0.9961, 0.7451, 0.2902),
        (0.9961, 0.7373, 0.2824), (0.9961, 0.7333, 0.2745),
        (0.9961, 0.7294, 0.2667), (0.9961, 0.7255, 0.2588),
        (0.9961, 0.7216, 0.251), (0.9961, 0.7176, 0.2431),
        (0.9961, 0.7137, 0.2353), (0.9961, 0.7098, 0.2275),
        (0.9961, 0.702, 0.2196), (0.9961, 0.6902, 0.1961),
        (0.9961, 0.6863, 0.1961), (0.9961, 0.6784, 0.1961),
        (0.9922, 0.6745, 0.1922), (0.9922, 0.6706, 0.1922),
        (0.9922, 0.6627, 0.1922), (0.9922, 0.6588, 0.1922),
        (0.9922, 0.6549, 0.1922), (0.9922, 0.6471, 0.1922),
        (0.9882, 0.6431, 0.1882), (0.9882, 0.6392, 0.1882),
        (0.9882, 0.6314, 0.1882), (0.9882, 0.6275, 0.1882),
        (0.9882, 0.6235, 0.1882), (0.9882, 0.6157, 0.1882),
        (0.9843, 0.6118, 0.1843), (0.9843, 0.6039, 0.1843),
        (0.9843, 0.6, 0.1843), (0.9843, 0.5961, 0.1843),
        (0.9843, 0.5882, 0.1843), (0.9843, 0.5843, 0.1843),
        (0.9804, 0.5804, 0.1804), (0.9804, 0.5725, 0.1804),
        (0.9804, 0.5686, 0.1804), (0.9804, 0.5647, 0.1804),
        (0.9804, 0.5569, 0.1804), (0.9804, 0.5529, 0.1804),
        (0.9765, 0.549, 0.1765), (0.9765, 0.5412, 0.1765),
        (0.9765, 0.5373, 0.1765), (0.9765, 0.5333, 0.1765),
        (0.9765, 0.5255, 0.1765), (0.9725, 0.5216, 0.1725),
        (0.9725, 0.5176, 0.1725), (0.9725, 0.5098, 0.1725),
        (0.9725, 0.5059, 0.1725), (0.9725, 0.502, 0.1725),
        (0.9725, 0.4941, 0.1725), (0.9686, 0.4902, 0.1686),
        (0.9686, 0.4863, 0.1686), (0.9686, 0.4784, 0.1686),
        (0.9686, 0.4745, 0.1686), (0.9686, 0.4667, 0.1686),
        (0.9686, 0.4627, 0.1686), (0.9647, 0.4588, 0.1647),
        (0.9647, 0.451, 0.1647), (0.9647, 0.4471, 0.1647),
        (0.9647, 0.4431, 0.1647), (0.9647, 0.4353, 0.1647),
        (0.9647, 0.4314, 0.1647), (0.9608, 0.4275, 0.1608),
        (0.9608, 0.4157, 0.1608), (0.9569, 0.4118, 0.1608),
        (0.9569, 0.4039, 0.1608), (0.9529, 0.4, 0.1608),
        (0.949, 0.3922, 0.1608), (0.949, 0.3882, 0.1608),
        (0.9451, 0.3804, 0.1608), (0.9412, 0.3765, 0.1608),
        (0.9412, 0.3686, 0.1608), (0.9373, 0.3647, 0.1608),
        (0.9333, 0.3608, 0.1608), (0.9333, 0.3529, 0.1608),
        (0.9294, 0.349, 0.1608), (0.9294, 0.3412, 0.1608),
        (0.9255, 0.3373, 0.1608), (0.9216, 0.3294, 0.1608),
        (0.9216, 0.3255, 0.1608), (0.9176, 0.3176, 0.1608),
        (0.9137, 0.3137, 0.1608), (0.9137, 0.3098, 0.1608),
        (0.9098, 0.302, 0.1608), (0.9059, 0.298, 0.1608),
        (0.9059, 0.2902, 0.1608), (0.902, 0.2863, 0.1608),
        (0.898, 0.2784, 0.1608), (0.898, 0.2745, 0.1608),
        (0.8941, 0.2706, 0.1608), (0.8902, 0.2627, 0.1569),
        (0.8902, 0.2588, 0.1569), (0.8863, 0.251, 0.1569),
        (0.8824, 0.2471, 0.1569), (0.8824, 0.2392, 0.1569),
        (0.8784, 0.2353, 0.1569), (0.8745, 0.2275, 0.1569),
        (0.8745, 0.2235, 0.1569), (0.8706, 0.2196, 0.1569),
        (0.8667, 0.2118, 0.1569), (0.8667, 0.2078, 0.1569),
        (0.8627, 0.2, 0.1569), (0.8627, 0.1961, 0.1569),
        (0.8588, 0.1882, 0.1569), (0.8549, 0.1843, 0.1569),
        (0.8549, 0.1765, 0.1569), (0.851, 0.1725, 0.1569),
        (0.8471, 0.1686, 0.1569), (0.8471, 0.1608, 0.1569),
        (0.8431, 0.1569, 0.1569), (0.8392, 0.149, 0.1569),
        (0.8392, 0.1451, 0.1569), (0.8353, 0.1373, 0.1569),
        (0.8275, 0.1216, 0.1569), (0.8235, 0.1216, 0.1569),
        (0.8196, 0.1216, 0.1529), (0.8118, 0.1176, 0.1529),
        (0.8078, 0.1176, 0.1529), (0.8039, 0.1176, 0.1529),
        (0.8, 0.1176, 0.149), (0.7922, 0.1176, 0.149),
        (0.7882, 0.1137, 0.149), (0.7843, 0.1137, 0.1451),
        (0.7804, 0.1137, 0.1451), (0.7725, 0.1137, 0.1451),
        (0.7686, 0.1137, 0.1451), (0.7647, 0.1137, 0.1412),
        (0.7608, 0.1098, 0.1412), (0.7529, 0.1098, 0.1412),
        (0.749, 0.1098, 0.1373), (0.7451, 0.1098, 0.1373),
        (0.7412, 0.1098, 0.1373), (0.7333, 0.1059, 0.1373),
        (0.7294, 0.1059, 0.1333), (0.7255, 0.1059, 0.1333),
        (0.7216, 0.1059, 0.1333), (0.7137, 0.1059, 0.1294),
        (0.7098, 0.102, 0.1294), (0.7059, 0.102, 0.1294),
        (0.702, 0.102, 0.1294), (0.6941, 0.102, 0.1255),
        (0.6902, 0.102, 0.1255), (0.6863, 0.098, 0.1255),
        (0.6824, 0.098, 0.1216), (0.6745, 0.098, 0.1216),
        (0.6706, 0.098, 0.1216), (0.6667, 0.098, 0.1176),
        (0.6627, 0.0941, 0.1176), (0.6549, 0.0941, 0.1176),
        (0.651, 0.0941, 0.1176), (0.6471, 0.0941, 0.1137),
        (0.6431, 0.0941, 0.1137), (0.6353, 0.0941, 0.1137),
        (0.6314, 0.0902, 0.1098), (0.6275, 0.0902, 0.1098),
        (0.6235, 0.0902, 0.1098), (0.6157, 0.0902, 0.1098),
        (0.6118, 0.0902, 0.1059), (0.6078, 0.0863, 0.1059),
        (0.6039, 0.0863, 0.1059), (0.5961, 0.0863, 0.102),
        (0.5922, 0.0863, 0.102), (0.5882, 0.0863, 0.102),
        (0.5843, 0.0824, 0.102), (0.5725, 0.0824, 0.098)],
    'ncl:wxpenir': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (1., 1., 0.), (1., 1., 0.),
        (0.3922, 0.1961, 0.098), (0.3922, 0.1961, 0.098),
        (0.3922, 0.1961, 0.098), (0.3922, 0.1961, 0.098),
        (0., 1., 0.), (0., 1., 0.),
        (0., 1., 0.), (0., 1., 0.),
        (0., 1., 0.), (0., 0.502, 0.),
        (0., 0.502, 0.), (0., 0.502, 0.),
        (0., 0.502, 0.), (0., 0.502, 0.),
        (0., 0., 0.502), (0., 0., 0.502),
        (0., 0., 0.502), (0., 0., 0.502),
        (0., 0., 1.), (0., 0., 1.),
        (0., 0., 1.), (0., 0., 1.),
        (0., 0., 1.), (0., 1., 1.),
        (0., 1., 1.), (0., 1., 1.),
        (0., 1., 1.), (0., 1., 1.),
        (0., 1., 1.), (0., 1., 1.),
        (0., 1., 1.), (0., 1., 1.),
        (0., 0.502, 0.502), (0., 0.502, 0.502),
        (0., 0.502, 0.502), (0., 0.502, 0.502),
        (0., 0.502, 0.502), (0., 0.502, 0.502),
        (0., 0.502, 0.502), (0., 0.502, 0.502),
        (0., 0.502, 0.502), (0.8039, 0.8039, 0.8039),
        (0.7882, 0.7882, 0.7882), (0.7725, 0.7725, 0.7725),
        (0.7569, 0.7569, 0.7569), (0.7412, 0.7412, 0.7412),
        (0.7255, 0.7255, 0.7255), (0.7059, 0.7059, 0.7059),
        (0.6902, 0.6902, 0.6902), (0.6745, 0.6745, 0.6745),
        (0.6588, 0.6588, 0.6588), (0.6431, 0.6431, 0.6431),
        (0.6275, 0.6275, 0.6275), (0.6118, 0.6118, 0.6118),
        (0.5961, 0.5961, 0.5961), (0.5804, 0.5804, 0.5804),
        (0.5647, 0.5647, 0.5647), (0.5451, 0.5451, 0.5451),
        (0.5294, 0.5294, 0.5294), (0.5137, 0.5137, 0.5137),
        (0.498, 0.498, 0.498), (0.4824, 0.4824, 0.4824),
        (0.4667, 0.4667, 0.4667), (0.451, 0.451, 0.451),
        (0.4353, 0.4353, 0.4353), (0.4196, 0.4196, 0.4196),
        (0.4039, 0.4039, 0.4039), (0.3843, 0.3843, 0.3843),
        (0.3686, 0.3686, 0.3686), (0.3529, 0.3529, 0.3529),
        (0.3373, 0.3373, 0.3373), (0.3216, 0.3216, 0.3216),
        (0.3059, 0.3059, 0.3059), (0.2902, 0.2902, 0.2902),
        (0.2745, 0.2745, 0.2745), (0.2588, 0.2588, 0.2588),
        (0.2431, 0.2431, 0.2431), (0.2235, 0.2235, 0.2235),
        (0.2078, 0.2078, 0.2078), (0.1922, 0.1922, 0.1922),
        (0.1765, 0.1765, 0.1765), (0.1608, 0.1608, 0.1608),
        (0.1451, 0.1451, 0.1451), (0.1294, 0.1294, 0.1294),
        (0.1137, 0.1137, 0.1137), (0.098, 0.098, 0.098),
        (0.0824, 0.0824, 0.0824), (0.0627, 0.0627, 0.0627),
        (0.0471, 0.0471, 0.0471), (0.0314, 0.0314, 0.0314),
        (0.0157, 0.0157, 0.0157), (0., 0., 0.)],
    'ncl:gauss': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 0., 1.), (0., 0., 1.),
        (0.0039, 0.0078, 0.9961), (0.0078, 0.0157, 0.9922),
        (0.0118, 0.0235, 0.9882), (0.0157, 0.0353, 0.9843),
        (0.0196, 0.0471, 0.9804), (0.0235, 0.0627, 0.9765),
        (0.0275, 0.0824, 0.9725), (0.0314, 0.102, 0.9686),
        (0.0353, 0.1255, 0.9647), (0.0392, 0.1529, 0.9608),
        (0.0431, 0.1804, 0.9569), (0.0471, 0.2078, 0.9529),
        (0.051, 0.2392, 0.949), (0.0549, 0.2706, 0.9451),
        (0.0588, 0.3059, 0.9412), (0.0627, 0.3412, 0.9373),
        (0.0667, 0.3765, 0.9333), (0.0706, 0.4118, 0.9294),
        (0.0745, 0.4471, 0.9255), (0.0784, 0.4824, 0.9216),
        (0.0824, 0.5216, 0.9176), (0.0863, 0.5569, 0.9137),
        (0.0902, 0.5922, 0.9098), (0.0941, 0.6275, 0.9059),
        (0.098, 0.6627, 0.902), (0.102, 0.698, 0.898),
        (0.1059, 0.7294, 0.8941), (0.1098, 0.7608, 0.8902),
        (0.1137, 0.7922, 0.8863), (0.1176, 0.8196, 0.8824),
        (0.1216, 0.8471, 0.8784), (0.1255, 0.8745, 0.8745),
        (0.1294, 0.898, 0.8706), (0.1333, 0.9176, 0.8667),
        (0.1373, 0.9373, 0.8627), (0.1412, 0.9529, 0.8588),
        (0.1451, 0.9647, 0.8549), (0.149, 0.9765, 0.851),
        (0.1529, 0.9843, 0.8471), (0.1569, 0.9922, 0.8431),
        (0.1608, 0.9922, 0.8392), (0.1647, 0.9922, 0.8353),
        (0.1686, 0.9922, 0.8314), (0.1725, 0.9882, 0.8275),
        (0.1765, 0.9804, 0.8235), (0.1804, 0.9686, 0.8196),
        (0.1843, 0.9569, 0.8157), (0.1882, 0.9412, 0.8118),
        (0.1922, 0.9216, 0.8078), (0.1961, 0.902, 0.8039),
        (0.2, 0.8824, 0.8), (0.2039, 0.8549, 0.7961),
        (0.2078, 0.8314, 0.7922), (0.2118, 0.8039, 0.7882),
        (0.2157, 0.7725, 0.7843), (0.2196, 0.7412, 0.7804),
        (0.2235, 0.7098, 0.7765), (0.2275, 0.6745, 0.7725),
        (0.2314, 0.6392, 0.7686), (0.2353, 0.6039, 0.7647),
        (0.2392, 0.5686, 0.7608), (0.2431, 0.5333, 0.7569),
        (0.2471, 0.498, 0.7529), (0.251, 0.4588, 0.749),
        (0.2549, 0.4235, 0.7451), (0.2588, 0.3882, 0.7412),
        (0.2627, 0.3529, 0.7373), (0.2667, 0.3176, 0.7333),
        (0.2706, 0.2824, 0.7294), (0.2745, 0.251, 0.7255),
        (0.2784, 0.2196, 0.7216), (0.2824, 0.1882, 0.7176),
        (0.2863, 0.1608, 0.7137), (0.2902, 0.1373, 0.7098),
        (0.2941, 0.1098, 0.7059), (0.298, 0.0902, 0.702),
        (0.302, 0.0706, 0.698), (0.3059, 0.051, 0.6941),
        (0.3098, 0.0353, 0.6902), (0.3137, 0.0235, 0.6863),
        (0.3176, 0.0118, 0.6824), (0.3216, 0.0039, 0.6784),
        (0.3255, 0., 0.6745), (0.3294, 0., 0.6706),
        (0.3333, 0., 0.6667), (0.3373, 0., 0.6627),
        (0.3412, 0.0078, 0.6588), (0.3451, 0.0157, 0.6549),
        (0.349, 0.0275, 0.651), (0.3529, 0.0392, 0.6471),
        (0.3569, 0.0549, 0.6431), (0.3608, 0.0745, 0.6392),
        (0.3647, 0.0941, 0.6353), (0.3686, 0.1176, 0.6314),
        (0.3725, 0.1451, 0.6275), (0.3765, 0.1725, 0.6235),
        (0.3804, 0.2, 0.6196), (0.3843, 0.2314, 0.6157),
        (0.3882, 0.2627, 0.6118), (0.3922, 0.2941, 0.6078),
        (0.3961, 0.3294, 0.6039), (0.4, 0.3647, 0.6),
        (0.4039, 0.4, 0.5961), (0.4078, 0.4353, 0.5922),
        (0.4118, 0.4706, 0.5882), (0.4157, 0.5098, 0.5843),
        (0.4196, 0.5451, 0.5804), (0.4235, 0.5804, 0.5765),
        (0.4275, 0.6157, 0.5725), (0.4314, 0.651, 0.5686),
        (0.4353, 0.6863, 0.5647), (0.4392, 0.7216, 0.5608),
        (0.4431, 0.7529, 0.5569), (0.4471, 0.7843, 0.5529),
        (0.451, 0.8118, 0.549), (0.4549, 0.8392, 0.5451),
        (0.4588, 0.8667, 0.5412), (0.4627, 0.8902, 0.5373),
        (0.4667, 0.9098, 0.5333), (0.4706, 0.9294, 0.5294),
        (0.4745, 0.9451, 0.5255), (0.4784, 0.9608, 0.5216),
        (0.4824, 0.9725, 0.5176), (0.4863, 0.9804, 0.5137),
        (0.4902, 0.9882, 0.5098), (0.4941, 0.9922, 0.5059),
        (0.498, 0.9961, 0.502), (0.502, 0.9922, 0.498),
        (0.5059, 0.9882, 0.4941), (0.5098, 0.9804, 0.4902),
        (0.5137, 0.9725, 0.4863), (0.5176, 0.9608, 0.4824),
        (0.5216, 0.9451, 0.4784), (0.5255, 0.9294, 0.4745),
        (0.5294, 0.9098, 0.4706), (0.5333, 0.8902, 0.4667),
        (0.5373, 0.8667, 0.4627), (0.5412, 0.8392, 0.4588),
        (0.5451, 0.8118, 0.4549), (0.549, 0.7843, 0.451),
        (0.5529, 0.7529, 0.4471), (0.5569, 0.7216, 0.4431),
        (0.5608, 0.6863, 0.4392), (0.5647, 0.651, 0.4353),
        (0.5686, 0.6157, 0.4314), (0.5725, 0.5804, 0.4275),
        (0.5765, 0.5451, 0.4235), (0.5804, 0.5098, 0.4196),
        (0.5843, 0.4706, 0.4157), (0.5882, 0.4353, 0.4118),
        (0.5922, 0.4, 0.4078), (0.5961, 0.3647, 0.4039),
        (0.6, 0.3294, 0.4), (0.6039, 0.2941, 0.3961),
        (0.6078, 0.2627, 0.3922), (0.6118, 0.2314, 0.3882),
        (0.6157, 0.2, 0.3843), (0.6196, 0.1725, 0.3804),
        (0.6235, 0.1451, 0.3765), (0.6275, 0.1176, 0.3725),
        (0.6314, 0.0941, 0.3686), (0.6353, 0.0745, 0.3647),
        (0.6392, 0.0549, 0.3608), (0.6431, 0.0392, 0.3569),
        (0.6471, 0.0275, 0.3529), (0.651, 0.0157, 0.349),
        (0.6549, 0.0078, 0.3451), (0.6588, 0., 0.3412),
        (0.6627, 0., 0.3373), (0.6667, 0., 0.3333),
        (0.6706, 0., 0.3294), (0.6745, 0.0039, 0.3255),
        (0.6784, 0.0118, 0.3216), (0.6824, 0.0235, 0.3176),
        (0.6863, 0.0353, 0.3137), (0.6902, 0.051, 0.3098),
        (0.6941, 0.0706, 0.3059), (0.698, 0.0902, 0.302),
        (0.702, 0.1098, 0.298), (0.7059, 0.1373, 0.2941),
        (0.7098, 0.1608, 0.2902), (0.7137, 0.1882, 0.2863),
        (0.7176, 0.2196, 0.2824), (0.7216, 0.251, 0.2784),
        (0.7255, 0.2824, 0.2745), (0.7294, 0.3176, 0.2706),
        (0.7333, 0.3529, 0.2667), (0.7373, 0.3882, 0.2627),
        (0.7412, 0.4235, 0.2588), (0.7451, 0.4588, 0.2549),
        (0.749, 0.4941, 0.251), (0.7529, 0.5333, 0.2471),
        (0.7569, 0.5686, 0.2431), (0.7608, 0.6039, 0.2392),
        (0.7647, 0.6392, 0.2353), (0.7686, 0.6745, 0.2314),
        (0.7725, 0.7098, 0.2275), (0.7765, 0.7412, 0.2235),
        (0.7804, 0.7725, 0.2196), (0.7843, 0.8039, 0.2157),
        (0.7882, 0.8314, 0.2118), (0.7922, 0.8549, 0.2078),
        (0.7961, 0.8824, 0.2039), (0.8, 0.902, 0.2),
        (0.8039, 0.9216, 0.1961), (0.8078, 0.9412, 0.1922),
        (0.8118, 0.9569, 0.1882), (0.8157, 0.9686, 0.1843),
        (0.8196, 0.9804, 0.1804), (0.8235, 0.9882, 0.1765),
        (0.8275, 0.9922, 0.1725), (0.8314, 0.9922, 0.1686),
        (0.8353, 0.9922, 0.1647), (0.8392, 0.9922, 0.1608),
        (0.8431, 0.9843, 0.1569), (0.8471, 0.9765, 0.1529),
        (0.851, 0.9647, 0.149), (0.8549, 0.9529, 0.1451),
        (0.8588, 0.9373, 0.1412), (0.8627, 0.9176, 0.1373),
        (0.8667, 0.898, 0.1333), (0.8706, 0.8745, 0.1294),
        (0.8745, 0.8471, 0.1255), (0.8784, 0.8196, 0.1216),
        (0.8824, 0.7922, 0.1176), (0.8863, 0.7608, 0.1137),
        (0.8902, 0.7294, 0.1098), (0.8941, 0.698, 0.1059),
        (0.898, 0.6627, 0.102), (0.902, 0.6275, 0.098),
        (0.9059, 0.5922, 0.0941), (0.9098, 0.5569, 0.0902),
        (0.9137, 0.5216, 0.0863), (0.9176, 0.4824, 0.0824),
        (0.9216, 0.4471, 0.0784), (0.9255, 0.4118, 0.0745),
        (0.9294, 0.3765, 0.0706), (0.9333, 0.3412, 0.0667),
        (0.9373, 0.3059, 0.0627), (0.9412, 0.2706, 0.0588),
        (0.9451, 0.2392, 0.0549), (0.949, 0.2078, 0.051),
        (0.9529, 0.1804, 0.0471), (0.9569, 0.1529, 0.0431),
        (0.9608, 0.1255, 0.0392), (0.9647, 0.102, 0.0353),
        (0.9686, 0.0824, 0.0314), (0.9725, 0.0627, 0.0275),
        (0.9765, 0.0471, 0.0235), (0.9804, 0.0314, 0.0196),
        (0.9843, 0.0196, 0.0157), (0.9882, 0.0118, 0.0118)],
    'ncl:saw': [
        (1., 1., 1.), (0., 0., 0.),
        (0., 1., 1.), (0.0039, 0.9843, 1.),
        (0.0078, 0.9647, 0.9961), (0.0118, 0.9451, 0.9922),
        (0.0157, 0.9255, 0.9882), (0.0196, 0.9059, 0.9843),
        (0.0235, 0.8863, 0.9804), (0.0275, 0.8667, 0.9765),
        (0.0314, 0.8471, 0.9725), (0.0353, 0.8275, 0.9686),
        (0.0392, 0.8078, 0.9647), (0.0431, 0.7882, 0.9608),
        (0.0471, 0.7686, 0.9569), (0.051, 0.749, 0.9529),
        (0.0549, 0.7294, 0.949), (0.0588, 0.7098, 0.9451),
        (0.0627, 0.6902, 0.9412), (0.0667, 0.6706, 0.9373),
        (0.0706, 0.651, 0.9333), (0.0745, 0.6314, 0.9294),
        (0.0784, 0.6118, 0.9255), (0.0824, 0.5922, 0.9216),
        (0.0863, 0.5725, 0.9176), (0.0902, 0.5529, 0.9137),
        (0.0941, 0.5333, 0.9098), (0.098, 0.5137, 0.9059),
        (0.102, 0.4941, 0.902), (0.1059, 0.4745, 0.898),
        (0.1098, 0.4549, 0.8941), (0.1137, 0.4353, 0.8902),
        (0.1176, 0.4157, 0.8863), (0.1216, 0.3961, 0.8824),
        (0.1255, 0.3765, 0.8784), (0.1294, 0.3569, 0.8745),
        (0.1333, 0.3373, 0.8706), (0.1373, 0.3176, 0.8667),
        (0.1412, 0.298, 0.8627), (0.1451, 0.2784, 0.8588),
        (0.149, 0.2588, 0.8549), (0.1529, 0.2392, 0.851),
        (0.1569, 0.2196, 0.8471), (0.1608, 0.2, 0.8431),
        (0.1647, 0.1804, 0.8392), (0.1686, 0.1608, 0.8353),
        (0.1725, 0.1412, 0.8314), (0.1765, 0.1216, 0.8275),
        (0.1804, 0.102, 0.8235), (0.1843, 0.0824, 0.8196),
        (0.1882, 0.0627, 0.8157), (0.1922, 0.0431, 0.8118),
        (0.1961, 0.0235, 0.8078), (0.2, 0.0039, 0.8039),
        (0.2039, 0.9882, 0.8), (0.2078, 0.9686, 0.7961),
        (0.2118, 0.949, 0.7922), (0.2157, 0.9294, 0.7882),
        (0.2196, 0.9098, 0.7843), (0.2235, 0.8902, 0.7804),
        (0.2275, 0.8706, 0.7765), (0.2314, 0.851, 0.7725),
        (0.2353, 0.8314, 0.7686), (0.2392, 0.8118, 0.7647),
        (0.2431, 0.7922, 0.7608), (0.2471, 0.7725, 0.7569),
        (0.251, 0.7529, 0.7529), (0.2549, 0.7333, 0.749),
        (0.2588, 0.7137, 0.7451), (0.2627, 0.6941, 0.7412),
        (0.2667, 0.6745, 0.7373), (0.2706, 0.6549, 0.7333),
        (0.2745, 0.6353, 0.7294), (0.2784, 0.6157, 0.7255),
        (0.2824, 0.5961, 0.7216), (0.2863, 0.5765, 0.7176),
        (0.2902, 0.5569, 0.7137), (0.2941, 0.5373, 0.7098),
        (0.298, 0.5176, 0.7059), (0.302, 0.498, 0.702),
        (0.3059, 0.4784, 0.698), (0.3098, 0.4588, 0.6941),
        (0.3137, 0.4392, 0.6902), (0.3176, 0.4196, 0.6863),
        (0.3216, 0.4, 0.6824), (0.3255, 0.3804, 0.6784),
        (0.3294, 0.3608, 0.6745), (0.3333, 0.3412, 0.6706),
        (0.3373, 0.3216, 0.6667), (0.3412, 0.302, 0.6627),
        (0.3451, 0.2824, 0.6588), (0.349, 0.2627, 0.6549),
        (0.3529, 0.2431, 0.651), (0.3569, 0.2235, 0.6471),
        (0.3608, 0.2039, 0.6431), (0.3647, 0.1843, 0.6392),
        (0.3686, 0.1647, 0.6353), (0.3725, 0.1451, 0.6314),
        (0.3765, 0.1255, 0.6275), (0.3804, 0.1059, 0.6235),
        (0.3843, 0.0863, 0.6196), (0.3882, 0.0667, 0.6157),
        (0.3922, 0.0471, 0.6118), (0.3961, 0.0275, 0.6078),
        (0.4, 0.0078, 0.6039), (0.4039, 0.9922, 0.6),
        (0.4078, 0.9725, 0.5961), (0.4118, 0.9529, 0.5922),
        (0.4157, 0.9333, 0.5882), (0.4196, 0.9137, 0.5843),
        (0.4235, 0.8941, 0.5804), (0.4275, 0.8745, 0.5765),
        (0.4314, 0.8549, 0.5725), (0.4353, 0.8353, 0.5686),
        (0.4392, 0.8157, 0.5647), (0.4431, 0.7961, 0.5608),
        (0.4471, 0.7765, 0.5569), (0.451, 0.7569, 0.5529),
        (0.4549, 0.7373, 0.549), (0.4588, 0.7176, 0.5451),
        (0.4627, 0.698, 0.5412), (0.4667, 0.6784, 0.5373),
        (0.4706, 0.6588, 0.5333), (0.4745, 0.6392, 0.5294),
        (0.4784, 0.6196, 0.5255), (0.4824, 0.6, 0.5216),
        (0.4863, 0.5804, 0.5176), (0.4902, 0.5608, 0.5137),
        (0.4941, 0.5412, 0.5098), (0.498, 0.5216, 0.5059),
        (0.502, 0.502, 0.502), (0.5059, 0.4824, 0.498),
        (0.5098, 0.4627, 0.4941), (0.5137, 0.4431, 0.4902),
        (0.5176, 0.4235, 0.4863), (0.5216, 0.4039, 0.4824),
        (0.5255, 0.3843, 0.4784), (0.5294, 0.3647, 0.4745),
        (0.5333, 0.3451, 0.4706), (0.5373, 0.3255, 0.4667),
        (0.5412, 0.3059, 0.4627), (0.5451, 0.2863, 0.4588),
        (0.549, 0.2667, 0.4549), (0.5529, 0.2471, 0.451),
        (0.5569, 0.2275, 0.4471), (0.5608, 0.2078, 0.4431),
        (0.5647, 0.1882, 0.4392), (0.5686, 0.1686, 0.4353),
        (0.5725, 0.149, 0.4314), (0.5765, 0.1294, 0.4275),
        (0.5804, 0.1098, 0.4235), (0.5843, 0.0902, 0.4196),
        (0.5882, 0.0706, 0.4157), (0.5922, 0.051, 0.4118),
        (0.5961, 0.0314, 0.4078), (0.6, 0.0118, 0.4039),
        (0.6039, 0.9961, 0.4), (0.6078, 0.9765, 0.3961),
        (0.6118, 0.9569, 0.3922), (0.6157, 0.9373, 0.3882),
        (0.6196, 0.9176, 0.3843), (0.6235, 0.898, 0.3804),
        (0.6275, 0.8784, 0.3765), (0.6314, 0.8588, 0.3725),
        (0.6353, 0.8392, 0.3686), (0.6392, 0.8196, 0.3647),
        (0.6431, 0.8, 0.3608), (0.6471, 0.7804, 0.3569),
        (0.651, 0.7608, 0.3529), (0.6549, 0.7412, 0.349),
        (0.6588, 0.7216, 0.3451), (0.6627, 0.702, 0.3412),
        (0.6667, 0.6824, 0.3373), (0.6706, 0.6627, 0.3333),
        (0.6745, 0.6431, 0.3294), (0.6784, 0.6235, 0.3255),
        (0.6824, 0.6039, 0.3216), (0.6863, 0.5843, 0.3176),
        (0.6902, 0.5647, 0.3137), (0.6941, 0.5451, 0.3098),
        (0.698, 0.5255, 0.3059), (0.702, 0.5059, 0.302),
        (0.7059, 0.4863, 0.298), (0.7098, 0.4667, 0.2941),
        (0.7137, 0.4471, 0.2902), (0.7176, 0.4275, 0.2863),
        (0.7216, 0.4078, 0.2824), (0.7255, 0.3882, 0.2784),
        (0.7294, 0.3686, 0.2745), (0.7333, 0.349, 0.2706),
        (0.7373, 0.3294, 0.2667), (0.7412, 0.3098, 0.2627),
        (0.7451, 0.2902, 0.2588), (0.749, 0.2706, 0.2549),
        (0.7529, 0.251, 0.251), (0.7569, 0.2314, 0.2471),
        (0.7608, 0.2118, 0.2431), (0.7647, 0.1922, 0.2392),
        (0.7686, 0.1725, 0.2353), (0.7725, 0.1529, 0.2314),
        (0.7765, 0.1333, 0.2275), (0.7804, 0.1137, 0.2235),
        (0.7843, 0.0941, 0.2196), (0.7882, 0.0745, 0.2157),
        (0.7922, 0.0549, 0.2118), (0.7961, 0.0353, 0.2078),
        (0.8, 0.0157, 0.2039), (0.8039, 1., 0.2),
        (0.8078, 0.9804, 0.1961), (0.8118, 0.9608, 0.1922),
        (0.8157, 0.9412, 0.1882), (0.8196, 0.9216, 0.1843),
        (0.8235, 0.902, 0.1804), (0.8275, 0.8824, 0.1765),
        (0.8314, 0.8627, 0.1725), (0.8353, 0.8431, 0.1686),
        (0.8392, 0.8235, 0.1647), (0.8431, 0.8039, 0.1608),
        (0.8471, 0.7843, 0.1569), (0.851, 0.7647, 0.1529),
        (0.8549, 0.7451, 0.149), (0.8588, 0.7255, 0.1451),
        (0.8627, 0.7059, 0.1412), (0.8667, 0.6863, 0.1373),
        (0.8706, 0.6667, 0.1333), (0.8745, 0.6471, 0.1294),
        (0.8784, 0.6275, 0.1255), (0.8824, 0.6078, 0.1216),
        (0.8863, 0.5882, 0.1176), (0.8902, 0.5686, 0.1137),
        (0.8941, 0.549, 0.1098), (0.898, 0.5294, 0.1059),
        (0.902, 0.5098, 0.102), (0.9059, 0.4902, 0.098),
        (0.9098, 0.4706, 0.0941), (0.9137, 0.451, 0.0902),
        (0.9176, 0.4314, 0.0863), (0.9216, 0.4118, 0.0824),
        (0.9255, 0.3922, 0.0784), (0.9294, 0.3725, 0.0745),
        (0.9333, 0.3529, 0.0706), (0.9373, 0.3333, 0.0667),
        (0.9412, 0.3137, 0.0627), (0.9451, 0.2941, 0.0588),
        (0.949, 0.2745, 0.0549), (0.9529, 0.2549, 0.051),
        (0.9569, 0.2353, 0.0471), (0.9608, 0.2157, 0.0431),
        (0.9647, 0.1961, 0.0392), (0.9686, 0.1765, 0.0353),
        (0.9725, 0.1569, 0.0314), (0.9765, 0.1373, 0.0275),
        (0.9804, 0.1176, 0.0235), (0.9843, 0.098, 0.0196),
        (0.9882, 0.0784, 0.0157), (0.9922, 0.0588, 0.0118)]
}


# small color maps (less than 50 colors)
ncl_small = {
    'ncl:amwg': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.576, 0.439, 0.859), (0., 0., 0.784),
        (0.235, 0.392, 0.902), (0.471, 0.608, 0.949),
        (0.69, 0.878, 0.902), (0.125, 0.698, 0.667),
        (0.604, 0.804, 0.196), (0.18, 0.545, 0.341),
        (0.961, 0.902, 0.745), (0.871, 0.722, 0.529),
        (1., 0.882, 0. ), (1., 0.647, 0. ), (1., 0.271, 0. ),
        (0.698, 0.133, 0.133), (1., 0.714, 0.757), (1., 0.078, 0.576)],
    'ncl:amwg_blueyellowred': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.51, 0.125, 0.941), (0., 0., 0.588),
        (0., 0., 0.804), (0.255, 0.412, 0.882),
        (0.118, 0.565, 1. ), (0., 0.749, 1. ),
        (0.627, 0.824, 1. ), (0.824, 0.961, 1. ),
        (1., 1., 0.784), (1., 0.882, 0.196),
        (1., 0.667, 0. ), (1., 0.431, 0. ), (1., 0., 0. ),
        (0.784, 0., 0. ), (0.627, 0.137, 0.137), (1., 0.412, 0.706)],
    'ncl:bluedarkred18': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.141, 0., 0.847), (0.094, 0.11, 0.969),
        (0.157, 0.341, 1. ), (0.239, 0.529, 1. ),
        (0.337, 0.69, 1. ), (0.459, 0.827, 1. ),
        (0.6, 0.918, 1. ), (0.737, 0.976, 1. ),
        (0.918, 1., 1. ), (1., 1., 0.918),
        (1., 0.945, 0.737), (1., 0.839, 0.6 ),
        (1., 0.675, 0.459), (1., 0.471, 0.337),
        (1., 0.239, 0.239), (0.969, 0.153, 0.208),
        (0.847, 0.082, 0.184), (0.647, 0., 0.129)],
    'ncl:bluedarkorange18': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0., 0.4, 0.4 ), (0., 0.6, 0.6 ),
        (0., 0.8, 0.8 ), (0., 1., 1. ),
        (0.2, 1., 1. ), (0.396, 1., 1. ),
        (0.6, 1., 1. ), (0.698, 1., 1. ),
        (0.796, 1., 1. ), (0.898, 1., 1. ),
        (1., 0.898, 0.796), (1., 0.792, 0.6 ),
        (1., 0.678, 0.396), (1., 0.557, 0.2 ),
        (1., 0.431, 0. ), (0.8, 0.333, 0. ),
        (0.6, 0.239, 0. ), (0.4, 0.153, 0. )],
    'ncl:bluegreen14': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0., 0., 1. ), (0.2, 0.2, 1. ),
        (0.396, 0.396, 1. ), (0.6, 0.6, 1. ),
        (0.698, 0.698, 1. ), (0.796, 0.796, 1. ),
        (0.898, 0.898, 1. ), (0.898, 1., 0.898),
        (0.796, 1., 0.796), (0.698, 1., 0.698),
        (0.6, 1., 0.6 ), (0.396, 1., 0.396), (0.2, 1., 0.2 ),
        (0., 1., 0. )],
    'ncl:brownblue12': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.2, 0.098, 0. ), (0.4, 0.184, 0. ),
        (0.6, 0.376, 0.208), (0.8, 0.608, 0.478),
        (0.847, 0.686, 0.592), (0.949, 0.855, 0.804),
        (0.8, 0.992, 1. ), (0.6, 0.973, 1. ),
        (0.396, 0.937, 1. ), (0.196, 0.89, 1. ),
        (0., 0.663, 0.8 ), (0., 0.478, 0.6 )],
    'ncl:cat12': [
        (1., 1., 1. ), (0., 0., 0. ),
        (1., 0.749, 0.498), (1., 0.498, 0. ),
        (1., 1., 0.6 ), (1., 1., 0.196),
        (0.698, 1., 0.549), (0.196, 1., 0. ),
        (0.647, 0.929, 1. ), (0.098, 0.698, 1. ),
        (0.8, 0.749, 1. ), (0.396, 0.298, 1. ),
        (1., 0.6, 0.749), (0.898, 0.098, 0.196)],
    'ncl:cosam12': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.392, 0., 0.455), (0.471, 0., 0.533),
        (0.353, 0., 0.722), (0.275, 0., 0.961),
        (0., 0.667, 0.882), (0., 0.784, 0.784),
        (0., 0.784, 0.49 ), (0.765, 1., 0. ),
        (1., 1., 0. ), (1., 0.392, 0. ),
        (1., 0.608, 0. ), (1., 0., 0. )],
    'ncl:cosam': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.471, 0., 0.533), (0.353, 0., 0.722),
        (0.275, 0., 0.961), (0., 0.667, 0.882),
        (0., 0.784, 0.784), (0., 0.784, 0.49 ), (0.765, 1., 0. ),
        (1., 1., 0. ), (1., 0.608, 0. ), (1., 0., 0. )],
    'ncl:cyclic': [
        (1., 1., 1.), (0., 0., 0.),
        (1., 0., 0.), (0., 1., 0.),
        (0., 0., 1.), (1., 1., 0.),
        (0., 1., 1.), (1., 0., 1.)],
    'ncl:default': [
        (1., 1., 1. ), (0., 0., 0. ),
        (1., 0., 0. ), (0., 1., 0. ),
        (0., 0., 1. ), (1., 1., 0. ),
        (0., 1., 1. ), (1., 0., 1. ),
        (0.5, 0., 0. ), (0.5, 1., 1. ),
        (0., 0., 0.5 ), (1., 1., 0.5 ),
        (0.5, 0., 1. ), (1., 0.5, 0. ),
        (0., 0.5, 1. ), (0.5, 1., 0. ),
        (0.5, 0., 0.5 ), (0.5, 1., 0.5 ),
        (1., 0.5, 1. ), (0., 0.5, 0. ),
        (0.5, 0.5, 1. ), (1., 0., 0.5 ),
        (0.5, 0.5, 0. ), (0., 0.5, 0.5 ),
        (1., 0.5, 0.5 ), (0., 1., 0.5 ),
        (0.5, 0.5, 0.5 ), (0.125, 0.125, 0.125),
        (0.75, 0.75, 0.75 ), (0.25, 0.25, 0.25 ),
        (0.625, 0.625, 0.625), (0.375, 0.375, 0.375)],
    'ncl:greenmagenta16': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0., 0.314, 0. ), (0., 0.525, 0. ),
        (0., 0.733, 0. ), (0., 0.945, 0. ),
        (0.314, 1., 0.314), (0.525, 1., 0.525),
        (0.733, 1., 0.733), (1., 1., 1. ),
        (1., 0.945, 1. ), (1., 0.733, 1. ),
        (1., 0.525, 1. ), (1., 0.314, 1. ), (0.945, 0., 0.945),
        (0.733, 0., 0.733), (0.525, 0., 0.525), (0.314, 0., 0.314)],
    'ncl:gscyclic': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.5, 0.5, 0.5 ), (0.125, 0.125, 0.125),
        (0.75, 0.75, 0.75 ), (0.25, 0.25, 0.25 ),
        (0.625, 0.625, 0.625), (0.375, 0.375, 0.375)],
    'ncl:gsdtol': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.031, 0.031, 0.031), (0.062, 0.062, 0.062),
        (0.094, 0.094, 0.094), (0.125, 0.125, 0.125),
        (0.156, 0.156, 0.156), (0.188, 0.188, 0.188),
        (0.219, 0.219, 0.219), (0.25, 0.25, 0.25 ),
        (0.281, 0.281, 0.281), (0.312, 0.312, 0.312),
        (0.344, 0.344, 0.344), (0.375, 0.375, 0.375),
        (0.406, 0.406, 0.406), (0.438, 0.438, 0.438),
        (0.469, 0.469, 0.469), (0.5, 0.5, 0.5 ),
        (0.531, 0.531, 0.531), (0.562, 0.562, 0.562),
        (0.594, 0.594, 0.594), (0.625, 0.625, 0.625),
        (0.656, 0.656, 0.656), (0.688, 0.688, 0.688),
        (0.719, 0.719, 0.719), (0.75, 0.75, 0.75 ),
        (0.781, 0.781, 0.781), (0.812, 0.812, 0.812),
        (0.844, 0.844, 0.844), (0.875, 0.875, 0.875),
        (0.906, 0.906, 0.906), (0.938, 0.938, 0.938),
        (0.969, 0.969, 0.969)],
    'ncl:gsltod': [
        (0., 0., 0. ), (1., 1., 1. ),
        (0.969, 0.969, 0.969), (0.938, 0.938, 0.938),
        (0.906, 0.906, 0.906), (0.875, 0.875, 0.875),
        (0.844, 0.844, 0.844), (0.812, 0.812, 0.812),
        (0.781, 0.781, 0.781), (0.75, 0.75, 0.75 ),
        (0.719, 0.719, 0.719), (0.688, 0.688, 0.688),
        (0.656, 0.656, 0.656), (0.625, 0.625, 0.625),
        (0.594, 0.594, 0.594), (0.562, 0.562, 0.562),
        (0.531, 0.531, 0.531), (0.5, 0.5, 0.5 ),
        (0.469, 0.469, 0.469), (0.438, 0.438, 0.438),
        (0.406, 0.406, 0.406), (0.375, 0.375, 0.375),
        (0.344, 0.344, 0.344), (0.312, 0.312, 0.312),
        (0.281, 0.281, 0.281), (0.25, 0.25, 0.25 ),
        (0.219, 0.219, 0.219), (0.188, 0.188, 0.188),
        (0.156, 0.156, 0.156), (0.125, 0.125, 0.125),
        (0.094, 0.094, 0.094), (0.062, 0.062, 0.062),
        (0.031, 0.031, 0.031)],
    'ncl:gui_default': [
        (0., 0., 0. ), (1., 1., 1. ),
        (0., 0., 1. ), (0., 0.4, 1. ),
        (0., 0.835, 1. ), (0.094, 1., 1. ),
        (0.4, 1., 1. ), (0.6, 1., 1. ),
        (0.4, 1., 0.6 ), (0.102, 1., 0.169),
        (0.2, 1., 0. ), (0.494, 1., 0. ),
        (0.8, 1., 0. ), (0.878, 0.965, 0.008),
        (1., 1., 0. ), (1., 1., 0. ),
        (1., 0.91, 0. ), (1., 0.8, 0. ),
        (1., 0.6, 0. ), (1., 0.5, 0. ), (1., 0.4, 0. ),
        (1., 0.2, 0. ), (1., 0.161, 0. ), (1., 0., 0. )],
    'ncl:hlu_default': [
        (0., 0., 0. ), (1., 1., 1. ),
        (1., 0., 0. ), (0., 1., 0. ),
        (0., 0., 1. ), (1., 1., 0. ),
        (0., 1., 1. ), (1., 0., 1. ),
        (0.5, 0., 0. ), (0.5, 1., 1. ),
        (0., 0., 0.5 ), (1., 1., 0.5 ),
        (0.5, 0., 1. ), (1., 0.5, 0. ),
        (0., 0.5, 1. ), (0.5, 1., 0. ),
        (0.5, 0., 0.5 ), (0.5, 1., 0.5 ),
        (1., 0.5, 1. ), (0., 0.5, 0. ),
        (0.5, 0.5, 1. ), (1., 0., 0.5 ),
        (0.5, 0.5, 0. ), (0., 0.5, 0.5 ),
        (1., 0.5, 0.5 ), (0., 1., 0.5 ),
        (0.5, 0.5, 0.5 ), (0.125, 0.125, 0.125),
        (0.75, 0.75, 0.75 ), (0.25, 0.25, 0.25 ),
        (0.625, 0.625, 0.625), (0.375, 0.375, 0.375)]
}


# meteo swiss maps
ncl_meteo_swiss = {
    'ncl:hotcold_18lev': [
        (0.094, 0.094, 0.439), (0.063, 0.306, 0.545),
        (0.09, 0.455, 0.804), (0.282, 0.463, 1. ),
        (0.357, 0.675, 0.929), (0.678, 0.843, 0.902),
        (0.82, 0.929, 0.929), (0.898, 0.937, 0.976),
        (0.949, 1., 1. ), (1., 1., 1. ),
        (0.992, 0.961, 0.902), (1., 0.894, 0.706),
        (0.953, 0.643, 0.376), (0.929, 0.463, 0. ),
        (0.804, 0.4, 0.114), (0.878, 0.192, 0.059),
        (0.929, 0., 0. ), (0.804, 0., 0. ), (0.545, 0., 0. )],
    'ncl:hotcolr_19lev': [
        (0., 0., 0.196), (0.094, 0.094, 0.439),
        (0.063, 0.306, 0.545), (0.09, 0.455, 0.804),
        (0.282, 0.463, 1. ), (0.357, 0.675, 0.929),
        (0.678, 0.843, 0.902), (0.82, 0.929, 0.929),
        (0.898, 0.937, 0.976), (0.949, 1., 1. ),
        (0.992, 0.961, 0.902), (1., 0.894, 0.706),
        (0.953, 0.643, 0.376), (0.929, 0.463, 0. ),
        (0.804, 0.4, 0.114), (0.878, 0.192, 0.059),
        (0.929, 0., 0. ), (0.804, 0., 0. ), (0.545, 0., 0. ),
        (0.196, 0., 0. )],
    'ncl:mch_default': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.125, 0.125, 0.125), (0.251, 0.251, 0.251),
        (0.376, 0.376, 0.376), (0.502, 0.502, 0.502),
        (0.627, 0.627, 0.627), (0.753, 0.753, 0.753),
        (0.878, 0.878, 0.878), (1., 0., 0. ),
        (0., 1., 0. ), (0., 0., 1. ), (1., 1., 0. ),
        (0., 1., 1. ), (1., 0., 1. )],
    'ncl:nrl_sirkes': [
        (0., 0.38, 0.502), (0., 0.502, 0.631),
        (0., 0.631, 0.749), (0., 0.749, 0.878),
        (0., 0.878, 1. ), (0., 1., 1. ),
        (0.2, 0.988, 0.988), (0.4, 0.988, 0.988),
        (0.6, 0.988, 0.988), (0.8, 0.988, 0.988),
        (1., 1., 1. ), (0.988, 0.988, 0. ),
        (0.988, 0.878, 0. ), (0.988, 0.749, 0. ),
        (0.988, 0.631, 0. ), (0.988, 0.502, 0. ),
        (0.988, 0.38, 0. ), (0.988, 0.251, 0. ), (0.988, 0.129, 0. ),
        (0.749, 0., 0. ), (0.502, 0., 0. )],
    'ncl:perc2_9lev': [
        (0.843, 0.89, 0.933), (0.71, 0.792, 1. ),
        (0.561, 0.702, 1. ), (0.498, 0.592, 1. ),
        (0.671, 0.812, 0.388), (0.91, 0.961, 0.62 ), (1., 0.98, 0.078),
        (1., 0.82, 0.129), (1., 0.639, 0.039), (1., 0.298, 0. )],
    'ncl:percent_11lev': [
        (0.392, 0., 0.588), (0.153, 0.188, 0.843),
        (0.349, 0.553, 0.988), (0.545, 0.937, 0.851),
        (0.376, 0.812, 0.569), (0.102, 0.596, 0.314),
        (0.569, 0.812, 0.376), (0.851, 0.937, 0.545),
        (0.996, 0.878, 0.545), (0.988, 0.553, 0.349),
        (0.843, 0.188, 0.153), (0.588, 0., 0.392)],
    'ncl:posneg_1': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0., 0., 0. ), (0.094, 0.094, 0.439),
        (0.063, 0.306, 0.545), (0.09, 0.455, 0.804),
        (0.282, 0.463, 1. ), (0.357, 0.675, 0.929),
        (0.678, 0.843, 0.902), (0.82, 0.929, 0.929),
        (0.902, 0.902, 0.98 ), (1., 0.894, 0.706),
        (0.953, 0.643, 0.376), (0.929, 0.463, 0. ),
        (0.824, 0.412, 0.118), (1., 0., 0. ),
        (0.929, 0., 0. ), (0.804, 0., 0. ), (0.545, 0., 0. )],
    'ncl:posneg_2': [
        (1., 1., 1. ), (0., 0., 0. ),
        (0.094, 0.094, 0.439), (0.063, 0.306, 0.545),
        (0.09, 0.455, 0.804), (0.282, 0.463, 1. ),
        (0.357, 0.675, 0.929), (0.678, 0.843, 0.902),
        (0.82, 0.929, 0.929), (0.898, 0.937, 0.976),
        (0.949, 1., 1. ), (0.992, 0.961, 0.902),
        (1., 0.894, 0.706), (0.953, 0.643, 0.376),
        (0.929, 0.463, 0. ), (0.804, 0.4, 0.114),
        (0.878, 0.192, 0.059), (0.929, 0., 0. ), (0.804, 0., 0. ),
        (0.545, 0., 0. )],
    'ncl:prcp_1': [
        (1., 1., 1. ), (0.667, 1., 1. ),
        (0.333, 0.627, 1. ), (0.114, 0., 1. ),
        (0.494, 0.898, 0.357), (0.306, 0.8, 0.263),
        (0.18, 0.698, 0.224), (0.118, 0.6, 0.239),
        (1., 1., 0.4 ), (1., 0.8, 0.4 ),
        (1., 0.533, 0.298), (1., 0.098, 0.098),
        (0.8, 0.239, 0.239), (0.647, 0.192, 0.192),
        (0.929, 0., 0.929), (0.537, 0.404, 0.804),
        (0.98, 0.941, 0.902)],
    'ncl:prcp_2': [
        (0.961, 0.961, 0.961), (0.686, 0.929, 0.929),
        (0.596, 0.984, 0.596), (0.263, 0.804, 0.502),
        (0.231, 0.702, 0.443), (0.98, 0.98, 0.824),
        (1., 1., 0. ), (1., 0.643, 0. ),
        (1., 0., 0. ), (0.804, 0.216, 0. ),
        (0.78, 0.078, 0.522), (0.929, 0.51, 0.929)],
    'ncl:prcp_3': [
        (1., 1., 1. ), (0., 0., 0. ),
        (1., 1., 1. ), (0.914, 0.8, 0.976),
        (0.812, 0.502, 0.875), (0.514, 0.2, 0.576),
        (0.227, 0., 0.69 ), (0.114, 0., 0.843),
        (0., 0., 1. ), (0.012, 0.235, 0.686),
        (0.02, 0.467, 0.373), (0.031, 0.702, 0.059),
        (0.518, 0.851, 0.031), (1., 1., 0. ),
        (1., 0.667, 0. ), (1., 0.333, 0. ),
        (1., 0., 0. ), (0.702, 0., 0. ),
        (0.4, 0., 0. ), (0.2, 0., 0. ),
        (0., 0., 0. ), (0.98, 0.773, 0.98 ),
        (1., 1., 1. )],
    'ncl:precip_11lev': [
        (1., 1., 1. ), (0.929, 0.98, 0.761),
        (0.804, 1., 0.804), (0.6, 0.941, 0.698),
        (0.325, 0.741, 0.624), (0.196, 0.651, 0.588),
        (0.196, 0.588, 0.706), (0.02, 0.439, 0.69 ),
        (0.02, 0.314, 0.549), (0.039, 0.122, 0.588),
        (0.173, 0.008, 0.275), (0.416, 0.173, 0.353)],
    'ncl:precip_diff_12lev': [
        (0.714, 0.416, 0.157), (0.804, 0.522, 0.247),
        (0.882, 0.647, 0.392), (0.961, 0.804, 0.518),
        (0.961, 0.878, 0.62 ), (1., 0.961, 0.729),
        (1., 1., 1. ), (0.804, 1., 0.804),
        (0.6, 0.941, 0.698), (0.325, 0.741, 0.624),
        (0.431, 0.667, 0.784), (0.02, 0.439, 0.69 ),
        (0.008, 0.22, 0.345)],
    'ncl:precip_diff_1lev': [
        (0.325, 0.741, 0.624), (0.882, 0.647, 0.392)],
    'ncl:precip2_15lev': [
        (1., 1., 1. ), (0.392, 0., 0.392),
        (0.686, 0., 0.686), (0.863, 0., 0.863),
        (0.196, 0.196, 0.784), (0., 0.392, 1. ),
        (0., 0.588, 0.588), (0., 0.784, 0.196),
        (0.392, 1., 0. ), (0.588, 1., 0. ),
        (0.784, 1., 0.196), (1., 1., 0. ), (1., 0.784, 0. ),
        (1., 0.627, 0. ), (1., 0.49, 0. ), (0.882, 0.098, 0. )],
    'ncl:precip2_17lev': [
        (1., 1., 1. ), (0.922, 0.965, 1. ),
        (0.839, 0.886, 1. ), (0.71, 0.788, 1. ),
        (0.557, 0.698, 1. ), (0.498, 0.588, 1. ),
        (0.447, 0.522, 0.973), (0.388, 0.439, 0.973),
        (0., 0.62, 0.118), (0.235, 0.737, 0.239),
        (0.702, 0.82, 0.431), (0.725, 0.976, 0.431),
        (1., 0.976, 0.075), (1., 0.639, 0.035),
        (0.898, 0., 0. ), (0.741, 0., 0. ),
        (0.506, 0., 0. ), (0., 0., 0. )],
    'ncl:precip3_16lev': [
        (1., 1., 1. ), (0.839, 0.886, 1. ),
        (0.71, 0.788, 1. ), (0.557, 0.698, 1. ),
        (0.498, 0.588, 1. ), (0.388, 0.439, 0.969),
        (0., 0.388, 1. ), (0., 0.588, 0.588),
        (0., 0.776, 0.2 ), (0.388, 1., 0. ),
        (0.588, 1., 0. ), (0.776, 1., 0.2 ),
        (1., 1., 0. ), (1., 0.776, 0. ),
        (1., 0.627, 0. ), (1., 0.486, 0. ),
        (1., 0.098, 0. )],
    'ncl:precip4_11lev': [
        (0.996, 0.996, 0.996), (0.875, 1., 0.976),
        (0.604, 0.851, 0.792), (0.404, 0.761, 0.639),
        (0.251, 0.678, 0.459), (0.196, 0.651, 0.588),
        (0.353, 0.627, 0.804), (0.259, 0.573, 0.78 ),
        (0.298, 0.553, 0.769), (0.027, 0.184, 0.42 ),
        (0.027, 0.118, 0.275), (0.298, 0., 0.451)],
    'ncl:precip4_diff_19lev': [
        (0.42, 0.129, 0.027), (0.71, 0.231, 0.129),
        (0.78, 0.333, 0.259), (0.804, 0.412, 0.353),
        (0.651, 0.196, 0.216), (0.678, 0.251, 0.345),
        (0.761, 0.404, 0.447), (0.851, 0.604, 0.624),
        (1., 0.875, 0.878), (0.996, 0.996, 0.996),
        (0.996, 0.996, 0.996), (0.875, 1., 0.976),
        (0.604, 0.851, 0.792), (0.404, 0.761, 0.639),
        (0.251, 0.678, 0.459), (0.196, 0.651, 0.588),
        (0.353, 0.627, 0.804), (0.259, 0.573, 0.78 ),
        (0.298, 0.553, 0.769), (0.027, 0.184, 0.42 )],
    'ncl:radar': [
        (0., 1., 1. ), (0., 0.616, 1. ),
        (0., 0., 1. ), (0.035, 0.51, 0.686),
        (0., 1., 0. ), (0.031, 0.686, 0.078),
        (1., 0.839, 0. ), (1., 0.596, 0. ),
        (1., 0., 0. ), (0.867, 0., 0.106),
        (0.737, 0., 0.212), (0.475, 0., 0.427), (0.475, 0.2, 0.627),
        (0.765, 0.639, 0.831), (1., 1., 1. )],
    'ncl:radar_1': [
        (0.698, 0.973, 1. ), (0.698, 0.722, 1. ),
        (0.49, 0.145, 0.804), (0.329, 0.102, 0.545),
        (0.929, 0.902, 0.522), (0.804, 0.776, 0.451),
        (0.588, 0.588, 0.588), (1., 1., 1. ),
        (0.667, 1., 1. ), (0.333, 0.627, 1. ),
        (0.114, 0., 1. ), (0.494, 0.898, 0.357),
        (0.306, 0.8, 0.263), (0.18, 0.698, 0.224),
        (0.118, 0.6, 0.239), (1., 1., 0.4 ),
        (1., 0.8, 0.4 ), (1., 0.533, 0.298),
        (1., 0.098, 0.098), (0.8, 0.239, 0.239),
        (0.647, 0.192, 0.192), (0.929, 0., 0.929),
        (0.537, 0.404, 0.804), (0.98, 0.941, 0.902)],
    'ncl:rh_19lev': [
        (0.996, 0.996, 0.996), (0.996, 0.996, 0.627),
        (0.996, 0.996, 0.388), (0.957, 0.957, 0.431),
        (1., 0.824, 0.137), (1., 0.639, 0.098),
        (1., 0.349, 0.098), (0.902, 0.478, 0.396),
        (0.929, 0.569, 0.486), (0.937, 0.698, 0.573),
        (0.969, 0.78, 0.698), (1., 0.902, 0.902),
        (0.843, 0.882, 1. ), (0.588, 0.824, 1. ),
        (0.118, 0.741, 1. ), (0.078, 0.624, 1. ),
        (0.039, 0.424, 0.941), (0.043, 0.455, 1. ),
        (0.039, 0.408, 0.784), (0., 0.349, 0.624)],
    'ncl:so4_21': [
        (0., 0., 0. ), (0.114, 0., 0.176),
        (0.227, 0., 0.357), (0.341, 0., 0.533),
        (0.227, 0., 0.69 ), (0.114, 0., 0.843),
        (0., 0., 1. ), (0.012, 0.235, 0.686),
        (0.02, 0.467, 0.373), (0.031, 0.702, 0.059),
        (0.518, 0.851, 0.031), (1., 1., 0. ),
        (1., 0.667, 0. ), (1., 0.333, 0. ),
        (1., 0., 0. ), (0.98, 0.106, 0.314),
        (0.961, 0.208, 0.627), (0.941, 0.314, 0.941), (0.961, 0.541, 0.961),
        (0.98, 0.773, 0.98 ), (1., 1., 1. )],
    'ncl:so4_23': [
        (0., 0., 0. ), (0.341, 0., 0.533),
        (0.227, 0., 0.69 ), (0.114, 0., 0.843),
        (0., 0., 1. ), (0.012, 0.235, 0.686),
        (0.02, 0.467, 0.373), (0.031, 0.702, 0.059),
        (0.275, 0.776, 0.043), (0.518, 0.851, 0.031),
        (0.757, 0.925, 0.016), (1., 1., 0. ),
        (1., 0.749, 0. ), (1., 0.502, 0. ),
        (1., 0.251, 0. ), (1., 0., 0. ),
        (0.98, 0.106, 0.314), (0.961, 0.208, 0.627),
        (0.941, 0.314, 0.941), (0.957, 0.486, 0.957),
        (0.973, 0.659, 0.973), (0.984, 0.827, 0.984),
        (1., 1., 1. )],
    'ncl:spread_15lev': [
        (1., 1., 1. ), (1., 0.882, 0.882),
        (1., 0.824, 0.824), (1., 0.647, 0.647),
        (1., 0.471, 0.471), (1., 0.294, 0.294),
        (1., 0., 0. ), (1., 0.392, 0. ),
        (1., 0.588, 0. ), (1., 0.784, 0. ),
        (1., 1., 0. ), (0.549, 1., 0. ), (0., 1., 0. ),
        (0., 0.804, 0.373), (0., 0.569, 0.784), (0., 0., 1. )],
    'ncl:stepseq25': [
        (0.6, 0.059, 0.059), (0.698, 0.173, 0.173),
        (0.8, 0.318, 0.318), (0.898, 0.494, 0.494),
        (1., 0.698, 0.698), (0.6, 0.329, 0.059),
        (0.698, 0.435, 0.173), (0.8, 0.557, 0.318),
        (0.898, 0.694, 0.494), (1., 0.847, 0.698),
        (0.42, 0.6, 0.059), (0.522, 0.698, 0.173),
        (0.639, 0.8, 0.318), (0.765, 0.898, 0.494),
        (0.898, 1., 0.698), (0.059, 0.42, 0.6 ),
        (0.173, 0.522, 0.698), (0.318, 0.639, 0.8 ),
        (0.494, 0.765, 0.898), (0.698, 0.898, 1. ),
        (0.149, 0.059, 0.6 ), (0.259, 0.173, 0.698),
        (0.396, 0.318, 0.8 ), (0.561, 0.494, 0.898),
        (0.749, 0.698, 1. )],
    'ncl:sunshine_9lev': [
        (1., 1., 1. ), (1., 0.961, 0.8 ),
        (1., 0.902, 0.439), (1., 0.8, 0.2 ),
        (1., 0.686, 0.2 ), (1., 0.6, 0.2 ), (1., 0.435, 0.2 ),
        (1., 0.333, 0. ), (0.902, 0.157, 0.118),
        (0.784, 0.118, 0.078)],
    'ncl:sunshine_diff_12lev': [
        (0.318, 0.035, 0.475), (0.584, 0.059, 0.875),
        (0.718, 0.294, 0.953), (0.796, 0.494, 0.965),
        (0.882, 0.706, 0.98 ), (0.925, 0.816, 0.988),
        (1., 1., 1. ), (1., 0.961, 0.8 ),
        (1., 0.902, 0.439), (1., 0.8, 0.2 ),
        (1., 0.686, 0.2 ), (1., 0.435, 0. ), (0.902, 0.157, 0.118)],
    'ncl:temp_19lev': [
        (0.027, 0.118, 0.275), (0.027, 0.184, 0.42 ),
        (0.031, 0.322, 0.612), (0.129, 0.443, 0.71 ),
        (0.259, 0.573, 0.78 ), (0.353, 0.627, 0.804),
        (0.471, 0.749, 0.839), (0.667, 0.863, 0.902),
        (0.859, 0.961, 1. ), (0.941, 0.988, 1. ),
        (1., 0.941, 0.961), (1., 0.878, 0.878),
        (0.988, 0.733, 0.667), (0.988, 0.573, 0.447),
        (0.984, 0.416, 0.29 ), (0.941, 0.235, 0.169),
        (0.8, 0.094, 0.118), (0.651, 0.059, 0.078),
        (0.471, 0.039, 0.059),
        (0.373, 0., 0. )],
    'ncl:temp_diff_18lev': [
        (0.027, 0.118, 0.275), (0.027, 0.184, 0.42 ),
        (0.031, 0.341, 0.612), (0.129, 0.443, 0.71 ),
        (0.259, 0.573, 0.78 ), (0.353, 0.627, 0.804),
        (0.471, 0.749, 0.839), (0.667, 0.863, 0.902),
        (0.859, 0.961, 1. ), (1., 1., 1. ),
        (1., 0.878, 0.878), (0.988, 0.733, 0.667),
        (0.988, 0.573, 0.447), (0.984, 0.416, 0.29 ),
        (0.941, 0.235, 0.169), (0.8, 0.094, 0.118),
        (0.651, 0.059, 0.078), (0.471, 0.039, 0.059),
        (0.373, 0., 0. )],
    'ncl:temp_diff_1lev': [
        (0.941, 0.235, 0.169), (0.129, 0.443, 0.71 )],
    'ncl:topo_15lev': [
        (0.157, 0.212, 0.604), (0., 0.788, 0.196),
        (0.118, 0.827, 0.408), (0.369, 0.878, 0.455),
        (0.635, 0.922, 0.51 ), (0.875, 0.973, 0.573),
        (0.965, 0.898, 0.584), (0.784, 0.698, 0.463),
        (0.635, 0.494, 0.369), (0.561, 0.38, 0.329),
        (0.635, 0.49, 0.455), (0.698, 0.588, 0.545), (0.78, 0.69, 0.667),
        (0.859, 0.804, 0.792), (0.925, 0.894, 0.886), (1., 1., 1. )],
    'ncl:wgne15': [
        (0.627, 0.125, 0.941), (0., 0., 0.706),
        (0.235, 0.392, 0.902), (0.471, 0.608, 0.949),
        (0.69, 0.878, 0.902), (0.18, 0.545, 0.341),
        (0.392, 0.882, 0. ), (0.824, 1., 0.184),
        (0.961, 0.902, 0.745), (0.871, 0.722, 0.529),
        (1., 0.882, 0. ), (1., 0.647, 0. ), (1., 0.271, 0. ),
        (0.588, 0.133, 0.133), (1., 0.412, 0.706)],
    'ncl:wind_17lev': [
        (1., 1., 1. ), (0.937, 0.957, 0.82 ),
        (0.91, 0.957, 0.62 ), (0.667, 0.808, 0.388),
        (0.886, 0.929, 0.086), (1., 0.929, 0. ),
        (1., 0.929, 0.51 ), (0.957, 0.82, 0.498),
        (0.929, 0.647, 0.286), (0.898, 0.549, 0.239),
        (0.859, 0.486, 0.239), (0.937, 0.027, 0.239),
        (0.91, 0.337, 0.639), (0.608, 0.439, 0.659),
        (0.388, 0.439, 0.969), (0.498, 0.588, 1. ),
        (0.557, 0.698, 1. ), (0.71, 0.788, 1. )],
}


if __name__ == '__main__':
    import doctest
    doctest.testmod(optionflags=doctest.NORMALIZE_WHITESPACE)
